/*
 * tty.c --
 *	Looking up terminals and access times.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: tty.c,v 1.28 94/12/07 12:57:03 stolcke Exp $ ICSI (Berkeley)";
#endif

#include "../config.h"			/* for USE_TTYENT */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <sys/param.h>
#ifdef USE_TTYENT
#include <ttyent.h>
#endif

#include "general.h"
#include "os.h"
#include "error.h"
#include "fingerpaths.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* The directory containing the tty devices. */
#ifndef TTYPATH
#define TTYPATH "/dev"
#endif

/* The console device */
#ifndef CONSOLE
#define CONSOLE "console"
#endif

/*
 * NOTE: For cluster access to non-local ttys, the following functions
 * take a hostname argument.
 *
 *	get_device_name (hostname, tty)
 *	get_last_access (hostname, tty)
 *	is_bogus_tty (hostname, tty)
 *
 * Cluster access is currently implemented only for HP-UX.
 */

/* Get the device file name for a tty line */
/*ARGSUSED*/
char *
get_device_name (hostname, tty)
    char *hostname;
    char *tty;
{
    static char device_name[sizeof(TTYPATH) + 2 + 16 + 20];
#ifdef USE_HPCLUSTER
    if (hostname && !same_hostname(hostname, get_full_hostname ()))
        sprintf (device_name, "%s+/%s/%s", TTYPATH, hostname, tty);
    else
#endif /* USR_HPCLUSTER */
        sprintf (device_name, "%s/%s", TTYPATH, tty);

    return device_name;
}

/* **************************************************************** */
/*                                                                  */
/*                  TTY Location and Access Time                    */
/*                                                                  */
/* **************************************************************** */

/* Return the time that TTY was last accessed.  TTY is the name of the
   terminal device, as in "ttyp0", or "console". */
time_t
get_last_access (hostname, tty)
     char *hostname;
     char *tty;
{
  struct stat stats;

  if (! *tty)
    return (time ((time_t *)0));

  /* If we can't stat the tty, then if the user is on the console,
     make this last access time be large.  Otherwise, assume active. */
  if (stat (get_device_name (hostname, tty), &stats) == -1)
    if (strcmp (tty, CONSOLE) == 0)
      stats.st_atime = (time_t)0;
    else
      return (time ((time_t *)0));

  /* If the user is on the console, then return the read time of
     the console input device(s) if that is newer. */

#ifdef KEYBOARD
  if (strcmp (tty, CONSOLE) == 0)
    {
      /* handle special case to force zero idle time */
      if (! KEYBOARD)
        return (time ((time_t *)0));
      else
	{
	  time_t atime = get_last_access(hostname, KEYBOARD);

	  if (atime > stats.st_atime)
	    stats.st_atime = atime;

#ifdef KEYBOARD2
	  atime = get_last_access(hostname, KEYBOARD2);
	  if (atime > stats.st_atime)
	    stats.st_atime = atime;
#endif /* KEYBOARD2 */
	}
    }
#endif /* KEYBOARD */

  return (stats.st_atime);
}

static char **tty_locs = NULL;

/* Lookup the associated location of TTY.  Return NULL if not found. */
char *
tty_lookup (host, tty)
     char *host, *tty;
{
  int i;

  if (!host && !tty)
    return (NULL);

  /* the first time we read the ttylocs file */
  if (tty_locs == NULL)
    {
      FILE *stream;
      char *entry;
      int num_ttys = 0; 
      int max_ttys = 2;

      tty_locs = (char **)xmalloc (max_ttys * sizeof(char *));

      if (!(stream = fopen (TTYLOCFILE, "r")))
	{
	  tty_locs[0] = NULL;
	  return (NULL);
	}

      while (entry = get_config_entry (NULL, stream))
	{
	  /* Allocate more space if needed */
	  if (num_ttys + 1 >= max_ttys )
	    {
	      max_ttys *=2;
	      tty_locs =
		(char **)xrealloc (tty_locs, max_ttys * sizeof(char *));
	    }
	  tty_locs[num_ttys++] = entry;
	}
      tty_locs[num_ttys] = NULL;
      fclose (stream);
    }

  /* Now search for the tty */
  for (i = 0 ; tty_locs[i]; i++)
    {
      char *colon = strchr (tty_locs[i], ':');

      if (!colon)
	{
	  if (!host && strcmp (tty_locs[i], tty) == 0 ||
	      !tty && same_hostname (tty_locs[i], host))
	    break;
	}
      else
	{
	  *colon = '\0';
	  if (host && tty && strcmp (colon + 1, tty) == 0 &&
	      same_hostname (tty_locs[i], host))
	    {
	      *colon = ':';
	      break;
	    }
	  *colon = ':';
	}
    }
  
  if (tty_locs[i])
    return (tty_locs[i] + strlen(tty_locs[i]) + 1);

#ifdef USE_TTYENT
  /* don't give up quite yet. Look if the entry in /etc/ttytab for this line
     has something to say */
  if (!host)
    {
      struct ttyent *ty;

      if ((ty = getttynam (tty)) &&
	  ty->ty_comment && *ty->ty_comment != '\0')
	{
	  char *loc;

	  /*
	   * Some systems aren't kind enough to strip the initial comment
	   * character and white space, so we have to do it ourselves...
	   */
	  loc = ty->ty_comment;
	  if (*loc == '#')
	    {
	      do loc++;
	      while (*loc && isspace(*loc));
	    }
	  return (loc);
	}
    }
#endif /* USE_TTYENT */

  return (NULL);
}

/* Invalidate tty_locs array. */
void
reset_ttylocs ()
{
  if (tty_locs)
    {
      free_array (tty_locs);
      tty_locs = NULL;
    }
}

/* Check out whether this tty is actually being used (like fixutmp)
   To check whether a pty is still in use, we try to open the associated
   master device.  If that succeeds, the pty is no longer in use and the
   utmp entry is probably a bogus left-over.  Opening the master may fail
   because the pty is busy, doesn't exist, or is non-local (in the HP cluster).
   In any case the utmp entry cannot positively be deemed bogus. */
int
is_bogus_tty (hostname, tty)
     char *hostname;
     char *tty;
{
  char pty[12];
  char *device;
  int fd;

  if (! *tty) 
    return 0;

  if (strncmp (tty, "tty", 3) == 0)
    {
      /* BSD style ptys: slave /dev/ttyxx, master /dev/ptyxx */
      strncpy (pty, tty, sizeof(pty) - 1);
      pty[0] = 'p';
    }
  else if (strncmp (tty, "pty/tty", 7) == 0)
    {
      /* HP-UX style ptys: slave /dev/pty/ttyxx, master /dev/ptym/ptyxx */
      strcpy (pty, "ptym/");
      strncat (pty, tty + 4, sizeof(pty) - 6);
      pty[5] = 'p';
    }
  else
    /* not a pty -- no need to check */
    return (0);
   
  /* try to open master device to check if pty is still in use */
  device = get_device_name (hostname, pty);
  if ((fd = open (device, O_RDWR | O_NDELAY)) >= 0)
    {
      close (fd);

      /* Ha! A bogus login, left over from some terminal emulator.
	 Log the fact for sysadmin's benefit. */
      warning ("%s is probably a bogus utmp entry", tty);
      return (1);
    }
  else
    return (0);
}

/* Strip ttyname to compact format */
char *
strip_ttyname (ttyname)
  char *ttyname;
{
  static char stripped[10];
  char *start;

  if (ttyname[strlen(ttyname) - 1] == '*')
    stripped[0] = '*';
  else
    stripped[0] = ' ';

  /* first, remove everything except for the final pathname component */
  if (start = strrchr (ttyname, '/'))
    start += 1;
  else
    start = ttyname;

  /* now strip "tty" if the tty name starts thusly */
  if (strncmp (start, "tty", 3) == 0)
    start += 3;

  /* copy what remains to the static buffer */
  strncpy (stripped + 1, start, sizeof(stripped) - 1);
  stripped[sizeof(stripped) - 1] = '\0';
  if (stripped[strlen(stripped) - 1] == '*')
    stripped[strlen(stripped) - 1] = '\0';

  return stripped;
}
