/*
 * tcp.c --
 *	Functions for talking to other machines via TCP services.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: tcp.c,v 1.27 1994/09/15 06:25:53 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>

#include "../config.h"

#include "general.h"
#include "fingerpaths.h"
#include "tcp.h"
#include "os.h"
#include "error.h"
#include "authuser.h"
#include "util.h"

#ifndef SIGRET
#define SIGRET void
#endif

#ifndef LOCAL_NETS 
#define LOCAL_NETS	0
#endif

#define	USPS	1000000		/* # of microseconds in a second */

/* Because Unix is too stupid to make this a define.  This is
   worse than the ptrace stuff.  Value signifies Internet Protocol. */
#define IP 0

/* **************************************************************** */
/*								    */
/*			TCP Stream Functions			    */
/*								    */
/* **************************************************************** */

/* How come these don't come in a library somewhere?  Everyone could
   use one.  Like to open a SMTP connection, or talk, or anything. */

/* Default number of micro-seconds before timing out on connect call. */
#define TIME_OUT 750000

/* Number of micro-seconds before timing out on connect call.
   Zero value means no timeouts.  */
int connect_timeout = TIME_OUT;

/* Non-zero means only allow TIME_OUT micro-seconds for a connect () to
   succeed, instead of whatever the infernal network code allows. */
int allow_time_outs = 0;

/* Default number of micro-seconds before timing out on a read call. */
#define READ_TIME_OUT	(2*TIME_OUT)

/* Number of micro-seconds before timing out on a read call. */
int read_timeout = READ_TIME_OUT;

/* connect timeout handler */
/*ARGSUSED*/
static SIGRET
connect_timed_out (signo)
    int signo;
{
  ualarm (0, 0);
}

/* Open a filedes to SERVICE at ADDRESS.  If SERVICE is the name of a
   service, then it must exist on the local machine.  SERVICE can also
   be the ASCII representation of a decimal number, in which case it is
   interpreted as the port number to connect to.  Returns a valid file
   descriptor if successful, or -1 if not. */
int
tcp_to_service (service, address)
     char *service;
     struct in_addr *address;
{
  struct servent *server;
  struct sockaddr_in name;
  int connection;

  /* Prepare the socket name for binding. */
  bzero (&name, sizeof (name));

  name.sin_family = AF_INET;
  name.sin_addr = *address;

  /* Find the port to use for the requested service. */
  if (strcmp (service, CFINGER_SERVICE) == 0)
    name.sin_port = htons((u_short) CFINGER_PORT);
  else if (isdigit (*service))
    name.sin_port = htons((u_short) atoi (service));
  else
    {
      server = getservbyname (service, "tcp");
      if (!server)
	return (-1);
      name.sin_port = server->s_port;
    }

  /* Make a new socket. */
  connection = socket (PF_INET, SOCK_STREAM, IP);

  if (connection < 0)
    return (-1);

  /* Connect to the desired port.  We have a shorter timeout than
     the connect call uses by default. */
  {
    int error;

    if (allow_time_outs && connect_timeout > 0)
      {
	signal (SIGALRM, connect_timed_out);
	ualarm (connect_timeout, 0);
	error = connect (connection, (struct sockaddr *)&name, sizeof (name));
	ualarm (0, 0);
	signal (SIGALRM, SIG_DFL);
	if ( error < 0 && errno == EINTR )	/* timeout expired */
	  {
	    errno = ETIMEDOUT;	/* so file_error() gives a sensible message */
	  }
      }
    else
      error = connect (connection, (struct sockaddr *)&name, sizeof (name));

    if (error < 0)
      {
	close (connection);
	return (-1);
      }
  }
  
  return (connection);
}

/* Reliable read from a socket.
   Guaranteed to fill a buffer unless timed out. */
int
sock_read(f, buf, n)
     int f;
     char *buf;
     int n;
{
     char *bufptr = buf;
     int got = 0;
     fd_set fds;
     struct timeval tv;

     FD_ZERO(&fds);
     FD_SET(f, &fds);

     if (read_timeout > 0)
       {
         tv.tv_usec = read_timeout % USPS;
         tv.tv_sec = read_timeout / USPS;
       }

     while (got < n) {
	int res;

	/* wait for data */
	res = select(f+1, &fds, NULL, NULL, read_timeout > 0 ? &tv : NULL);

	if (res < 0) {
	    got = res;
	    break;
	}
	else if (res == 0) {
	    break;
	}

	/* read it */
	res = read(f, bufptr, n - got);

	if (res < 0) {
	    got = res;
	    break;
	}
	else if (res == 0) {
	    break;
	}
	else {
	    got += res;
	    bufptr += res;
	}
     }
     return got;
}

/* Check the socket s for local connected-ness.
   Also, return the peer address and user name. User name lookup is
   skipped if NULL a buffer pointer is given. In that case the
   peer address is also not converted into domain name format
   for efficiency. */
int
local_socket (s, peer_name, peer_namelen, user_name, user_namelen)
    int s;
    char *peer_name, *user_name;
    int peer_namelen, user_namelen;
{
  struct in_addr local_addr, remote_addr;
  unsigned short local_port, remote_port;
  char *peer_ip, *peer_user = NULL;
  int local;

  /* make sure we can get our peer's address */
  remote_addr.s_addr = 0;
  if (auth_fd2 (s, &local_addr, &remote_addr, &local_port, &remote_port) < 0)
    {
      if (errno != ENOTSOCK && errno != EINVAL &&
	  errno != EPIPE && errno != EAFNOSUPPORT)
	{
	  file_error (FATAL, "auth_fd");
	}
      else
	{
	  local = 1;
	  peer_ip = "[NO_SOCKET]";
	  peer_user = "";
	}
    }
  else if (peer_ip = inet_ntoa (remote_addr))
    local = match_host_config ("localnet", peer_ip, NULL);
  else
    {
      peer_ip = "(none)";
      peer_user = "";
      local = 0;
    }

  if (remote_addr.s_addr)
    {
      /* try to get the domain host name, but only if the user name is
	 also requested. If not, we figure user-friendliness is not
	 an issue and save a reverse name lookup.  */
      if (user_name)
	{
          struct hostent *he;
      
          if (he = gethostbyaddr ((char *)&remote_addr, sizeof(remote_addr),
                                  AF_INET))
            peer_ip = he->h_name;

	  /* try to get the peer user name */
	  if (!match_host_config ("authenticate", peer_ip, NULL))
	    peer_user = "";
	  else
	    if (!(peer_user = auth_tcpuser3(local_addr, remote_addr,
				    local_port, remote_port, AUTH_TIMEOUT)))
	      {
		if (errno == ETIMEDOUT)
		  warning ("authentication at %s timed out after %d seconds",
			   peer_ip, AUTH_TIMEOUT);
		peer_user = "";
	      }
	}
    }

  strncpy (peer_name, peer_ip, peer_namelen);
  peer_name[peer_namelen-1] = '\0';

  /* tack on the local domain if hostname has none
     (note that IP numbers always have a dot in them) */
  if (!strchr (peer_name, '.'))
    {
      strcat (peer_name, ".");
      strncat (peer_name, get_domain_name (),
	       peer_namelen - strlen (peer_name) - 1);
    }

  if (user_name && peer_user)
    {
      strncpy (user_name, peer_user, user_namelen);
      user_name[user_namelen-1] = '\0';
    }

  return local;
}

