/*
 * packet.c --
 *	Functions for manipulating packets.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: packet.c,v 1.39 1994/01/11 02:14:00 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <signal.h>
#include <ctype.h>

#include "general.h"
#include "packet.h"
#include "os.h"
#include "error.h"
#include "tcp.h"
#include "util.h"

/* **************************************************************** */
/*								    */
/*			Sorting Packets				    */
/*								    */
/* **************************************************************** */
/* Function for qsort to compare names. */
static int
name_sort_comp (p1, p2)
     FINGER_PACKET **p1, **p2;
{
  return (strcmp ((*p1)->name, (*p2)->name));
}

/* Sort the the packets in LIST by side-effecting them.  Use whatever
   the user default for sorting is.  (By username.) */
void
sort_packets (list)
     FINGER_PACKET **list;
{
  qsort (list, array_len (list), sizeof (FINGER_PACKET *), name_sort_comp);
}

/* Erase duplicate packets for the same user. Return number of duplicates */
int
uniq_packets (list)
     FINGER_PACKET **list;
{
    int i;
    char *prevname = NULL;
    int deleted = 0;

    for ( i = 0; list[i]; i++ )
      {
	if ( prevname && strcmp(prevname, list[i]->name) == 0 )
	  {
	    int j;
	    free(list[i]);
	    for ( j = i; list[j]; j++ )
	      list[j] = list[j+1];
	    i--;
	    deleted++;
	  }
	else
	  prevname = list[i]->name;
      }
    return (deleted);
}

/* **************************************************************** */
/*								    */
/*			Print a Packet				    */
/*								    */
/* **************************************************************** */

#define WIDTH_CNT	4
#define WIDTH_SEPERATOR	','
#ifndef DEFAULT_WIDTHS
#define DEFAULT_WIDTHS 16, 7, 8, 17
#endif

static int widths[WIDTH_CNT] = { DEFAULT_WIDTHS };

/* Parse and set field print widths from a string */
void
set_print_widths (s)
     char *s;
{
  int n, i;
  char inc, dec, val;

  inc = dec = val = n = i = 0;
  while (*s)
    {
      if (*s == '+')
	inc++, s++;
      else if (*s == '-')
	dec++, s++;
      if (isdigit (*s))
	{
	  do
	    {
	      n = n * 10 + (*s - '0');
	      s++;
	    }
	  while (isdigit (*s));
	  val++;
	}
      if (val)
	{
	  if (inc)
	    widths[i] += n;
	  else if (dec)
	    {
	      widths[i] -= n;
	      if (widths[i] < 0)
		widths[i] = 0;
	    }
	  else
	    widths[i] = n;
	}
      if (*s == WIDTH_SEPERATOR)
	{
	  s++;
	  inc = dec = val = n = 0;
	  i++;
	  if (i < WIDTH_CNT)
	    continue;
	  else
	    break;
	}
      else
	break;
    }
}

/* Print PACKET on STREAM.  HEADERP is non-zero if we should print
   a header first.
bfox     Brian Fox          fingerd        *p0 vision   (cs100-mux dialup from
bfox     Brian Fox                   0:38  *p0 fen      (vision  Eng I rm 3118
bfox     Brian Fox          rm       0:06  *p0 quagmire (vision  Eng I rm 3118
 */
void
ascii_packet (packet, stream, headerp)
     FINGER_PACKET *packet;
     FILE *stream;
     int headerp;
{
  static char *header = NULL;

/* Generate arguments for a %*s%-*.*s printf format so as to center string */
#define CENTER_FIELD(string, width) \
	 (width) < sizeof(string)-1 ? 0 : ((width)-(sizeof(string)-1)+1)/2,\
	 "", \
	 (width) < sizeof(string)-1 ? (width) : ((width)+(sizeof(string)-1))/2,\
	 (width) < sizeof(string)-1 ? (width) : ((width)+(sizeof(string)-1))/2,\
	 (string)
#define MAYBE_SPACE(wid) \
	((wid) > 0 ? " " : "")

   if (headerp)
     {
       if (!header)
	 {
	   header = xmalloc (8 + 1 + widths[0] + 1 + widths[1] + 1 + 5 + 1 + 3
                             + 1 + widths[2] + 1 + 9 + 1 + widths[3] + 1);
	   sprintf (header,
	            "%-8s%s%*s%-*.*s%s%*s%-*.*s %5s%4s%s%*s%-*.*s %*s%-*.*s%s%.*s\r\n",
		    "Login",
	            MAYBE_SPACE (widths[0]),
		    CENTER_FIELD ("Name", widths[0]),
	            MAYBE_SPACE (widths[1]),
		    CENTER_FIELD ("What", widths[1]),
		    "Idle",
		    "TTY",
	            MAYBE_SPACE (widths[2]),
		    CENTER_FIELD ("Host", widths[2]),
		    CENTER_FIELD ("When", 9),
	            MAYBE_SPACE (widths[3]),
		    widths[3], "Where");
	 }
       fprintf (stream, header);
       fflush (stream);
     }
  print_packet (packet, stream);
}

/* Print the contents of PACKET prettily.  The output goes to STREAM. */
void
print_packet (packet, stream)
     FINGER_PACKET *packet;
     FILE *stream;
{
  char *host = strip_hostname (packet->host);
  char *tty = strip_ttyname (packet->ttyname);
  char *idle = idle_time_string (packet->idle_time, 1);
#define DATE_OFFSET	4
#define TIME_OFFSET	11
  char *when = ctime (&packet->login_time);
#define WEEKS_TIME (7 * 24 * 60 * 60)
  int old_login = time(NULL) -  packet->login_time > WEEKS_TIME;

  if (old_login)
      fprintf (stream,
           "%-8s%s%-*.*s%s%-*.*s %5s%-4.4s%s%-*.*s %6.6s   %s%.*s\r\n",
	   packet->name, 
	   MAYBE_SPACE(widths[0]),
	   widths[0], widths[0], packet->real_name,
	   MAYBE_SPACE(widths[1]),
	   widths[1], widths[1], packet->what,
	   idle,
	   tty,
	   MAYBE_SPACE(widths[2]),
	   widths[2], widths[2], host,
	   when + DATE_OFFSET,
	   MAYBE_SPACE(widths[3]),
	   widths[3], packet->ttyloc);
  else
      fprintf (stream,
           "%-8s%s%-*.*s%s%-*.*s %5s%-4.4s%s%-*.*s %3.3s %-5.5s%s%.*s\r\n",
	   packet->name,
	   MAYBE_SPACE(widths[0]),
	   widths[0], widths[0], packet->real_name,
	   MAYBE_SPACE(widths[1]),
	   widths[1], widths[1], packet->what,
	   idle,
	   tty,
	   MAYBE_SPACE(widths[2]),
	   widths[2], widths[2], host,
	   when,
	   when + TIME_OFFSET,
	   MAYBE_SPACE(widths[3]),
	   widths[3], packet->ttyloc);

  free (idle);
  fflush (stream);
}

#ifdef notdef
/* Print out everything in a packet to stderr. */
void
debug_packet (packet)
     FINGER_PACKET *packet;
{
  char *idle = idle_time_string (packet->idle_time, 0);

  fprintf (stderr, "FINGER_PACKET *0x%8x {\n\
        name: \"%s\"\n\
   real_name: \"%s\"\n\
        host: \"%s\"\n\
  login_time: \"%s\"\n\
   idle_time: \"%s\"\n\
     ttyname: \"%s\"\n\
      ttyloc: \"%s\"\n\
        what: \"%s\"\n\
}", packet,
	   packet->name, packet->real_name, packet->host,
	   ctime (&(packet->login_time)), idle,
	   packet->ttyname, packet->ttyloc, packet->what);
  free (idle);
  fflush (stderr);
}
#endif /* notdef */


/* **************************************************************** */
/*								    */
/*		  Building and Procuring Packets		    */
/*								    */
/* **************************************************************** */

/* Return a NULL terminated array of (FINGER_PACKET *) which represents
   the list of users at ADDRESS, an inet address.  If there
   are no users on that machine, an array of zero length is returned.
   If we cannot talk to that machine, a NULL pointer is returned instead.
   USER is an argument to be passed to our server.  NULL or empty is the
   same as ".all". */
FINGER_PACKET **
finger_at_address (address, user)
     struct in_addr *address;
     char *user;
{
  FINGER_PACKET **packets, **read_packets ();
  int connection;

  if (!user)
    user = "";

  connection = tcp_to_service (CFINGER_SERVICE, address);

  if (connection < 0)
    return ((FINGER_PACKET **)NULL);

  /* Ask the server to return all of the users in binary form, please. */
  write (connection, user, strlen (user));
  write (connection, "\n", 1);

  packets = read_packets (connection, 1);

  /* packets came over the network, so fix byte order */
  if (packets)
    {
      int i;

      for ( i = 0; packets[i]; i++ )
        {
          packets[i]->login_time = ntohl(packets[i]->login_time);
          packets[i]->idle_time = ntohl(packets[i]->idle_time);
        }
    }
	
  return (packets);
}
  

/* **************************************************************** */
/*								    */
/*			Files and Packets			    */
/*								    */
/* **************************************************************** */

/* Remove packets from LIST, side-effecting LIST, a NULL terminated array
   of FINGER_PACKET *.  Remove the entries for which FUNCTION returns
   non-zero when called with the current packet and ARG, as in
   function (list[i], arg).  Returns number of packets deleted. */
int
remove_packets (list, function, arg)
     FINGER_PACKET **list;
     Function *function;
     char *arg;
{
  int i, j;
  int length;
  int count = 0;

  /* The first pass over the list replaces all removed packets with
     NULL pointers and records the list length */
  for (i = 0; list[i]; i++)
    {
      if ((*function)(list[i], arg))
	{
	  free (list[i]);
 	  list[i] = (FINGER_PACKET *)NULL;
	  count++;
	}
    }
  length = i;

  if (!count)
    return (0);

  /* The second pass copies the remaining pointers into new consecutive
     locations. */
  for (i = 0, j = 0; i < length; i++)
    {
      if (list[i])
        list[j++] = list[i];
    }
  list[j] = (FINGER_PACKET *)NULL;

  return (count);
}

/* Write out the contents of LIST (some FINGER_PACKET *'s) to file.
   -1 is returned if an I/O error occurred. */
int
write_packets (list, filedes)
     FINGER_PACKET **list;
     int filedes;
{
  register int i;

  /* write file from start */
  if (lseek(filedes, 0L, 0) != 0)
      return -1;

  if (list)
    {
      for (i = 0; list[i]; i++)
	if (write (filedes, list[i], sizeof (FINGER_PACKET))
		!= sizeof (FINGER_PACKET))
	  return -1;
    }

  /* through away any old data */
  if (ftruncate(filedes, (long)(i * sizeof (FINGER_PACKET))) < 0)
      return -1;

  return 0;
}

/* Read a list of packets from FILEDES.  This always returns a pointer
   to a NULL terminated array of FINGER_PACKET *, even if no packets were
   read. A NULL pointer is returned if an I/O error occurred. */
FINGER_PACKET **
read_packets (filedes, is_sock)
     int filedes;
     int is_sock;	/* is this a socket ? */
{
  int packet_index = 0, packet_size = 10;
  FINGER_PACKET **packets =
    (FINGER_PACKET **)xmalloc (packet_size * sizeof (FINGER_PACKET *));
  FINGER_PACKET packet;
  int res;
  
  packets[0] = (FINGER_PACKET *)NULL;

  while ((res = (is_sock ? sock_read (filedes, &packet, sizeof (packet))
		  : read (filedes, &packet, sizeof (packet))))
	  == sizeof (packet))
    {
      if (packet_index + 2 >= packet_size)
	{
	  packets = (FINGER_PACKET **)
	    xrealloc (packets,
		      (1 + (packet_size *= 2)) * sizeof (FINGER_PACKET *));
	}

      packets[packet_index] =
	(FINGER_PACKET *)xmalloc (sizeof (FINGER_PACKET));

      bcopy (&packet, packets[packet_index], sizeof (FINGER_PACKET));
      packets[++packet_index] = (FINGER_PACKET *)NULL;
    }

  /* close here since we have read to EOF anyway */
  close (filedes);

  if (res < 0)
    {
      free_array (packets);
      return (FINGER_PACKET **)NULL;
    }
  else if (res > 0)
    {
      warning("Incomplete packet (%d bytes) read from %s",
              res, is_sock ? "socket" : "file");
      return (packets);
    }
  else
      return (packets);
}

