/*
 * os.c --
 *	Operating system specific functions.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: os.c,v 1.61 1994/12/24 19:25:27 stolcke Exp $ ICSI (Berkeley)";
#endif

#include "../config.h"

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#ifdef WANT_ACCT
#include <sys/acct.h>
#endif
#include <time.h>
#include <errno.h>

#ifdef HAVE_XUTMP
# include <xutmp.h>
#define UTMP xutmp
#else /* !HAVE_XUTMP */
#ifdef HAVE_UTMPX
# include <utmpx.h>
#define UTMP utmpx
#else /* !HAVE_UTMPX */
# include <utmp.h>
#define UTMP utmp
#endif /* HAVE_UTMPX */
#endif /* HAVE_XUTMP */

#ifdef USE_LASTLOG
# ifndef _PATH_LASTLOG
#  include <lastlog.h>
# endif
#endif
#include <pwd.h>
#include <ctype.h>
#include <sys/param.h>

#include <netdb.h>
#include <netinet/in.h>
#if !defined (hpux)
#include <arpa/inet.h>
#endif

#ifdef USE_HPCLUSTER
#include <cluster.h>
#endif

#include "os.h"
#include "packet.h"
#include "util.h"
#include "error.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* Location of the accounting file.  This is for filling in the WHAT field. */
#ifndef ACCT_FILE
# if defined(sun) || defined(SYSV)
#  define ACCT_FILE	"/usr/adm/pacct"
# else
#  define ACCT_FILE	"/usr/adm/acct"
# endif
#endif

/* Where the utmp file is located and what its fields are called  */
#ifdef HAVE_XUTMP

#define UTNAME(uptr)		((uptr)->utx_name)
#define UTLINE(uptr)		((uptr)->utx_line)
#define UTTYPE(uptr)		((uptr)->utx_type)
#define UTPID(uptr)		((uptr)->utx_pid)
#define UTHOST(uptr)		((uptr)->utx_host)
#define UTTIME(uptr)		((uptr)->utx_time)

# undef UTMP_FILE
# ifdef _XUTMP_FILE
#  define UTMP_FILE	_XUTMP_FILE
# else
#  define UTMP_FILE	"/etc/xutmp"
# endif
#else /* !HAVE_XUTMP */

#define UTNAME(uptr)		((uptr)->ut_name)
#define UTLINE(uptr)		((uptr)->ut_line)
#define UTTYPE(uptr)		((uptr)->ut_type)
#define UTPID(uptr)		((uptr)->ut_pid)
#define UTHOST(uptr)		((uptr)->ut_host)

#ifdef HAVE_UTMPX
# undef UTMP_FILE
# ifdef UTMPX_FILE
#  define UTMP_FILE	UTMPX_FILE
# else
#  define UTMP_FILE	"/etc/utmpx"
# endif

#define UTTIME(uptr)		((uptr)->ut_xtime)

#else /* !HAVE_UTMPX */
# ifndef UTMP_FILE
#  ifdef _PATH_UTMP
#   define UTMP_FILE	_PATH_UTMP
#  else
#   define UTMP_FILE	"/etc/utmp"
#  endif
# endif

#define UTTIME(uptr)		((uptr)->ut_time)

#endif /* HAVE_UTMPX */
#endif /* HAVE_XUTMP */

/* Where the lastlog file is located. */
#ifdef USE_LASTLOG
# ifndef LASTLOG_FILE
#  ifdef _PATH_LASTLOG
#   define LASTLOG_FILE	_PATH_LASTLOG
#  else
#   define LASTLOG_FILE	"/usr/adm/lastlog"
#  endif
# endif
#endif

/* Where the resolver config file resides. */
#ifndef RESOLV_CONF
#define RESOLV_CONF	"/etc/resolv.conf"
#endif

/* The directory containing the tty devices. */
#ifndef TTYPATH
#define TTYPATH "/dev"
#endif

/* The console device */
#ifndef CONSOLE
#define CONSOLE "console"
#endif

#define TALKABLE 0220		/* mode of a talkable line */

#define NOTTY	"?"		/* string to use for unknown line */

static struct UTMP *dummy_utmp;	/* dummy to define size macros */

#define UTNAMESIZE		sizeof(UTNAME(dummy_utmp))
#define UTLINESIZE		sizeof(UTLINE(dummy_utmp))

/*
 * Some machines don't have a ut_host field in their utmp structure.
 * We need this however, if only to record remote host info from the
 * lastlog records (which get converted to utmp records).
 * In an excercise in blatantly bad programming style, we fake a ut_host
 * field by allocating some extra space after each utmp struct.
 */
#ifdef HAVE_UTHOST
#define UTHOSTSIZE		sizeof(UTHOST(dummy_utmp))
#define UTSIZE			sizeof(struct UTMP)
#else
#undef UTHOST
#define UTHOST(uptr)		(((char *)(uptr))+sizeof(struct UTMP))
#define UTHOSTSIZE		(16)
#define UTSIZE			(sizeof(struct UTMP) + UTHOSTSIZE)
#endif

/* Copy a ut_* field to memory and nul-terminate it */
#define COPY_NUL(target, source) \
	{ strncpy(target, source, sizeof(target) - 1); \
	  (target)[sizeof(target) - 1] = '\0'; \
	}

static FINGER_PACKET **utmp_to_packets ();
static struct UTMP **make_utmp_array ();
static struct UTMP **make_lastlog_array ();

/*
 * NOTE: For cluster access to non-local login records, the functions here
 * take a hostname argument.
 * Cluster access is currently implemented only for HP-UX.
 */

/* **************************************************************** */
/*                                                                  */
/*              Getting the Original Finger Data                    */
/*                                                                  */
/* **************************************************************** */

/* Return a NULL terminated array of (FINGER_PACKET *) which represents
   the list of interesting users on a given machine.  Single argument
   ALL_USERS when zero means don't place any user in the output list
   twice. */
FINGER_PACKET **
get_finger_data (hostname, all_users)
     char *hostname;
     int all_users;
{
  FINGER_PACKET **result;
  struct UTMP **users;

  if (!hostname)
    hostname = get_full_hostname ();

  users = make_utmp_array (hostname, !all_users);
  result = utmp_to_packets (hostname, NULL, users, !all_users, 0);

  if (users) free_array (users);
  return (result);
}

/* Return a NULL terminated array of (FINGER_PACKET *) which represents
   the list of last logins on a given machine. */
FINGER_PACKET **
get_lastlog_data (hostname)
  char *hostname;
{
  FINGER_PACKET **result;
  struct UTMP **users;

  if (!hostname)
    hostname = get_full_hostname ();

  users = make_lastlog_array (hostname);
  result = utmp_to_packets (hostname, NULL, users, 1, 1);

  if (users) free_array (users);
  return (result);
}

#ifdef WANT_ACCT
static acct_file = -1;			/* file descriptor to acct file */
static acct_nentries;			/* number of acct entries */
static acct_ncached = 0;		/* number of entries cached */
static struct acct *acct_cache = NULL;	/* cached entries */
static acct_nalloc = 0;			/* number of allocated cache entries */

/* Initialize the acct file cache */
static void
acct_open ()
{
  char *acct_name;

  (void)open_host_config ();
  acct_name = get_config_entry ("acctfile", NULL);

  /* empty acctfile disables acct lookup */
  if (acct_name && !*acct_name)
    {
      free (acct_name);
      return;
    }
  /* Open accounting file only once */
  else if (acct_name)
    {
      acct_file = open (acct_name, O_RDONLY);
      free (acct_name);
    }
  else
    acct_file = open (ACCT_FILE, O_RDONLY);

  if (acct_file < 0)
    {
      file_error (WARNING, acct_name);
      return;
    }
  else
    {
      struct stat sbuf;

      if (fstat (acct_file, &sbuf) < 0)
        {
	  close (acct_file);
	  acct_file = -1;
	}
      else
        {
	  acct_nentries = sbuf.st_size / sizeof(struct acct);
	  if (acct_nentries == 0)
	    {
	      close (acct_file);
	      acct_file = -1;
	    }
	}
    }

  if (acct_file >= 0)
    {
      acct_ncached = 0;
      acct_nalloc = 100;
      acct_cache = (struct acct *)malloc(acct_nalloc * sizeof(struct acct));
    }
}

/* search for an acct entry by uid and tty. */
static struct acct *
acct_search (uid, tty)
     int uid;
     dev_t tty;
{
  int i;
  long offset;

  /* fprintf(stderr, "acct_ncached = %d\n", acct_ncached); */

  /* First search the cached entries */
  for (i = 0; i < acct_ncached; i++ )
    {
      if (acct_cache[i].ac_tty == tty && acct_cache[i].ac_uid == uid)
	return &(acct_cache[i]);
    }

  /* Read more acct entries from file -- in reverse order */

  offset = (acct_nentries - acct_ncached) * sizeof(struct acct);
  
  while (acct_file >= 0 && offset >= 0)
    {
      /* make sure there is enough cache allocated */
      if (acct_ncached == acct_nalloc)
        {
	  acct_nalloc *= 2;
	  acct_cache = (struct acct *)xrealloc(acct_cache,
	 	 	                       acct_nalloc*sizeof(struct acct));
        }

      offset -= sizeof(struct acct);

      if (lseek (acct_file, offset, 0) < 0 ||
	  read (acct_file, &acct_cache[acct_ncached], sizeof (struct acct))
							!= sizeof(struct acct))
	{
	  close (acct_file);
	  acct_file = -1;
	}
      else
	{
	  i = (acct_ncached++);

	  if (acct_cache[i].ac_tty == tty && acct_cache[i].ac_uid == uid)
	    return &(acct_cache[i]);
	}
    }

  return (struct acct *)NULL;
}

/* free acct cache */
static void
acct_close ()
{
  if (acct_file >= 0)
    {
     close (acct_file);
     acct_file = -1;
    }

  if (acct_cache)
    {
      free (acct_cache);
      acct_cache = NULL;
      acct_nalloc = 0;
      acct_ncached = 0;
    }
}
#endif  /* WANT_ACCT */

/* Return a NULL terminated array of (FINGER PACKET *) manufactured
   from the members of UTMPLIST (a NULL terminated array of struct UTMP *). */
static FINGER_PACKET **
utmp_to_packets (hostname, packets, utmplist, unique, lastlog)
     char *hostname;
     FINGER_PACKET **packets;		/* list to append to */
     struct UTMP **utmplist;
     int unique;			/* report only unique entries */
     int lastlog;			/* don't worry about idle time */
{
  int i;
  time_t idle, current_time;
  char *ttyloc;

  FINGER_PACKET **result;
  int result_size, old_size;

  current_time = time ((time_t *)0);

  if (!packets)
    {
      /* Make the output array. */
      result_size = array_len (utmplist) + 1;

      if (result_size == 1)
	result_size++;		/* room for dummy packet */

      packets = result =
	(FINGER_PACKET **)xmalloc (result_size * sizeof (FINGER_PACKET *));
    }
  else
    {
      /* Append to the output array. */
      old_size = array_len (packets) + 1;
      result_size = old_size + array_len (utmplist);

      if (result_size == old_size + 1)
	result_size++;		/* room for dummy packet */

      packets = (FINGER_PACKET **)xrealloc (packets,
			result_size * sizeof (FINGER_PACKET *));
      result = packets + old_size;
    }

#ifdef WANT_ACCT
  acct_open ();
#endif
   
  /* Fill it in. */
  for (i = 0; utmplist && utmplist[i]; i++)
    {
      int tty_stat = -1;
      struct stat finfo;
      char line[UTLINESIZE + 1];

      COPY_NUL(line, UTLINE(utmplist[i]));

      if (lastlog)
	idle = 0;
      else
        idle = current_time - get_last_access (hostname, line);

      if (idle < 0)
	idle = 0;

      result[i] = (FINGER_PACKET *)xmalloc (sizeof (FINGER_PACKET));
      bzero (result[i], sizeof (FINGER_PACKET));

      bcopy (UTNAME(utmplist[i]), result[i]->name, UTNAMESIZE);
      result[i]->login_time = UTTIME(utmplist[i]);
      result[i]->idle_time = idle;
      strcpy (result[i]->host, hostname);
      strcpy (result[i]->ttyname, line);

      if (*line)
	{
          char *device_name = get_device_name (hostname, line);

          if (!lastlog && (tty_stat = stat (device_name, &finfo)) != -1)
	    if ((finfo.st_mode & TALKABLE) != TALKABLE)
	      strcat (result[i]->ttyname, "*");
	}

      ttyloc = tty_location (hostname, utmplist[i], unique);
      strncpy (result[i]->ttyloc, ttyloc, TTYLOC_LEN - 1);

      {
	struct passwd *passwd_entry = getpwnam (result[i]->name);

	if (passwd_entry && passwd_entry->pw_gecos)
	  {
#ifdef WANT_ACCT
	    struct acct *entry;
	    int uid = passwd_entry->pw_uid;
#define ACCOMMSIZE	sizeof(entry->ac_comm)

	    if (!lastlog &&
	        tty_stat != -1 &&
		(finfo.st_mode & S_IFCHR) &&
		(entry = acct_search (uid, finfo.st_rdev)))
	      {
		if (sizeof(result[i]->what) < ACCOMMSIZE + 1)
		  COPY_NUL(result[i]->what, entry->ac_comm)
		else
		  {
		    strncpy (result[i]->what, entry->ac_comm,
			     ACCOMMSIZE);
		    result[i]->what[ACCOMMSIZE] = '\0';
		  }
	      }
#endif /* WANT_ACCT */

	    COPY_NUL (result[i]->real_name, pw_real_name(passwd_entry));
      	  }
	else
	  strcpy (result[i]->real_name, "");
      }
    }

#ifdef WANT_ACCT
  acct_close ();
#endif

  /* add a dummy packet if no users were found */
  if (i == 0)
    {
      result[i] = (FINGER_PACKET *)xmalloc (sizeof (FINGER_PACKET));

      result[i]->name[0] = '\0';
      strncpy(result[i]->real_name, utmplist ? HOST_NOUSERS : HOST_DOWN,
						sizeof(result[i]->real_name));
      strncpy(result[i]->host, hostname, sizeof(result[i]->host));
      result[i]->login_time = 0;
      result[i]->idle_time = 0;
      result[i]->ttyname[0] = '\0';
      result[i]->ttyloc[0] = '\0';
      result[i]->what[0] = '\0';

      i++;
    }

  result[i] = (FINGER_PACKET *)NULL;
  return (packets);
}

#ifdef USE_LASTLOG
/* Return utmp struct corresponding to lastlog for user */
/*ARGSUSED*/
static struct UTMP *
get_lastlog_entry (hostname, pw)
    char *hostname;
    struct passwd *pw;
{
  struct UTMP *result;
  struct lastlog entry;
  struct stat statbuf;
  static int file = -1;	/* -1: not open yet; -2 = lastlog is dir */

  if (file == -1)
    {
      if (stat(LASTLOG_FILE, &statbuf) < 0)
	{
	  file_error (WARNING, LASTLOG_FILE);
	  return (struct UTMP *)NULL;
	}

      if (statbuf.st_mode & S_IFDIR)
        /* lastlog is a directory --
           assume this is an IRIX-style lastlog with one file per user. */
	file = -2;
      else
	{
	  file = open (LASTLOG_FILE, O_RDONLY);
	  if (file < 0)
	    {
	      file_error (WARNING, LASTLOG_FILE);
	      return (struct UTMP *)NULL;
	    }
	}
    }

  /* Now we know what we're dealing with, so do it */
  if (file == -2)
    {
      int ufile;
      char path[sizeof(LASTLOG_FILE) + 80]; 

      sprintf (path, "%s/%.79s", LASTLOG_FILE, pw->pw_name);

      ufile = open (path, O_RDONLY);
      if (ufile < 0)
	return (struct UTMP *)NULL;

      if (read(ufile, (char *)&entry, sizeof(entry)) != sizeof(entry)
	  || entry.ll_time == 0)
	{
	  close (ufile);
	  return (struct UTMP *)NULL;
	}
      close (ufile);
    
      result = (struct UTMP *)xmalloc (UTSIZE);

      /*
       * fake utmp entry with info from lastlog
       * NOTE: IRIX has the full tty path in ll_line, so make sure /dev
       * prefix is removed. Also, IRIX records remote user names in the
       * tty field, so don't copy those.
       */
      strncpy (UTNAME(result), pw->pw_name, UTNAMESIZE);

      if (strncmp (entry.ll_line, TTYPATH, sizeof(TTYPATH) - 1) == 0
	  && entry.ll_line[sizeof(TTYPATH) - 1] == '/')
	strncpy (UTLINE(result), entry.ll_line + sizeof(TTYPATH), UTLINESIZE);
      else
	strcpy (UTLINE(result), NOTTY);

      strncpy (UTHOST(result), entry.ll_host, UTHOSTSIZE);
      UTTIME(result) = entry.ll_time;

      return result;
    }
  else
    {
      if (lseek (file, (long) pw->pw_uid * sizeof(entry), 0) == -1 ||
	  read (file, (char *)&entry, sizeof(entry)) != sizeof(entry) ||
          entry.ll_time == 0)
	return (struct UTMP *)NULL;
      
      /* we have a valid entry */
      result = (struct UTMP *)xmalloc (UTSIZE);

      /*
       * fake utmp entry with info from lastlog
       */
      strncpy (UTNAME(result), pw->pw_name, UTNAMESIZE);
      strncpy (UTLINE(result), entry.ll_line, UTLINESIZE);
      strncpy (UTHOST(result), entry.ll_host, UTHOSTSIZE);
      UTTIME(result) = entry.ll_time;

      return result;
    }
}
#endif /* USE_LASTLOG */

/* Return an array of the users which are using this machine.  If UNIQUE
   is non-zero, we only return one entry per each user; the one with the
   least idle time. */
static struct UTMP **
make_utmp_array (hostname, unique)
     char *hostname;
     int unique;
{
  int i, file;
  struct UTMP entry;
  struct UTMP **result;
  int result_size = 0;
  int localhost = 1;

#ifdef USE_HPCLUSTER
  if (hostname && !same_hostname (hostname, get_full_hostname ()))
    {
      struct cct_entry *ccent;
      cnode_t cids[MAX_CNODE];
      int i;
      char utmp_file[sizeof(UTMP_FILE) + 2 + 16];

      localhost = 0;

      /* Get cluster info */
      if (cnodes(&cids) < 0 ||
          !(ccent = getccnam(hostname)))
	return (NULL);

      /* Make sure the cluster node is up */
      for (i = 0; cids[i]; i++)
	if (cids[i] == (ccent->cnode_id)) break;
      
      if (!cids[i])
	return (NULL);

      sprintf(utmp_file, "%s+/%s", UTMP_FILE, hostname);
      file = open (utmp_file, O_RDONLY);
    }
  else
#endif /* USE_HPCLUSTER */
    file = open (UTMP_FILE, O_RDONLY);

  if (file < 0)
    {
      file_error (FATAL, UTMP_FILE);
    }

  result_size = 20;
  result =
    (struct UTMP **)xmalloc (result_size * sizeof(struct UTMP *));

  result[0] = (struct UTMP *)NULL;

  while (read (file, &entry, sizeof (entry)) == sizeof (entry))
    {
      char line[UTLINESIZE + 1];

      COPY_NUL(line, UTLINE(&entry));

      /* Hopefully the nonuser macro will eliminate the lossage we had with
	 all of the non-existant users laying around in the utmp file. */
      if (UTNAME(&entry)[0] != '\0' && 
#if defined(sun) && defined(nonuser)
          !nonuser(entry) && 
#endif
#ifdef USER_PROCESS /* SYSV */
          UTTYPE(&entry) == USER_PROCESS && 	/* process is current */
          (!localhost ||
	   kill(UTPID(&entry), 0) == 0 ||
	   errno != ESRCH) &&			/* and exits */
#endif
          !is_bogus_tty(hostname, line))
	{
	  int new_entry = 0;

	  for (i = 0; result[i]; i++)
	    {
	      if (unique && same_person (result[i], &entry))
		{
		  if (idle_less_than (hostname, &entry, result[i]))
		    {
		      bcopy (&entry, result[i], UTSIZE);
		      new_entry = 1;
		    }
		  break;
		}
	    }

	  if (!result[i])
	    {
	      result[i] = (struct UTMP *)xmalloc (UTSIZE);
	      bcopy (&entry, result[i], sizeof(entry));
	      new_entry = 1;

	      if (i + 2 > result_size)
		{
		  result = (struct UTMP **)
		    xrealloc (result,
			      (result_size += 10) * sizeof(struct UTMP *));
		}

	      result[i + 1] = (struct UTMP *)NULL;
	    }

	  /* If this user is using the console device remember that for
	     finding the tty location.  `v0' is what X window terminals
	     use for the console login window. */
	  if (new_entry)
	    {
#ifndef HAVE_UTHOST
# ifdef USE_LASTLOG
	      /* if this machine has no ut_host, but supports lastlog records,
	         try to determined the remote hostname from there. */
	      char login[UTNAMESIZE + 1];
	      struct passwd *pw;
	      struct UTMP *lastlog;

	      COPY_NUL(login, UTNAME(result[i]));

	      if ((pw = getpwnam (login)) &&
	          (lastlog = get_lastlog_entry (hostname, pw)))
		{
		  /* look for matching login times and lines */
		  if ((UTTIME(lastlog) - UTTIME(result[i]) == 0||
		       UTTIME(lastlog) - UTTIME(result[i]) == 1) &&
		      (strncmp(UTLINE(lastlog), NOTTY, UTLINESIZE) == 0 ||
		       strncmp(UTLINE(lastlog), UTLINE(result[i]),
				UTLINESIZE) == 0))
		    bcopy(UTHOST(lastlog), UTHOST(result[i]), UTHOSTSIZE);
		  else
	            UTHOST(result[i])[0] = '\0';

		  free (lastlog);
		}
	      else
	        UTHOST(result[i])[0] = '\0';
# else /* !USE_LASTLOG */
	      UTHOST(result[i])[0] = '\0';
# endif /* USE_LASTLOG */
#endif /* !HAVE_UTHOST */
	     }

	}
    }
  close (file);
  endpwent ();
  return (result);
}

/* Return an array of lastlog entries in utmp format. */
/*ARGSUSED*/
static struct UTMP **
make_lastlog_array (hostname)
    char *hostname;
{
#ifdef USE_LASTLOG
  struct passwd *pw;
  struct UTMP **result;
  int result_size = 20, i = 0;

  result = (struct UTMP **)xmalloc (result_size * sizeof (struct UTMP *));

  /* for each valid user, look for lastlog info */
  while (pw = getpwent ())
    {
#ifdef SKIP_STAR_PASSWD
      /* no pseudo accounts */
      if (strcmp (pw->pw_passwd, "*") == 0)
	  continue;
#endif

      result[i] = get_lastlog_entry (hostname, pw);
      if (!result[i])
        continue;
      
      /* extend array if necessary */
      if (i + 2 > result_size)
	result = (struct UTMP **)
	    xrealloc (result, (result_size += 10) * UTSIZE);

      i++;
    }
  result[i] = (struct UTMP *)NULL;
  endpwent ();

#else /* ! USE_LASTLOG */

  struct UTMP **result;

  result = (struct UTMP **)xmalloc (sizeof (struct UTMP *));
  result[0] = (struct UTMP *)NULL;

#endif /* USE_LASTLOG */

  return (result);
}

/* Returns non-zero if the utmp entry pointed to by A is being used
   by the same person as then one pointed to by B. */
int
same_person (a, b)
     struct UTMP *a, *b;
{
  return (strncmp(UTNAME(a), UTNAME(b), UTNAMESIZE) == 0);
}

/* Returns non-zero if the utmp entry pointed to by A has been idle less
   time than the utmp entry pointed to by B. */
int
idle_less_than (hostname, a, b)
     char *hostname;
     struct UTMP *a, *b;
{
  char linea[UTLINESIZE + 1],
       lineb[UTLINESIZE + 1];
  
  COPY_NUL(linea, UTLINE(a));
  COPY_NUL(lineb, UTLINE(b));

  return (get_last_access (hostname, linea) >
          get_last_access (hostname, lineb));
}

/* Return a string which is the location of the terminal in utmp ENTRY. */
char *
tty_location (hostname, entry, unique)
     char *hostname;
     struct UTMP *entry;
     int unique;
{
  char *result, line[UTLINESIZE + 1];

  COPY_NUL(line, UTLINE(entry));

  /* If there is an entry in the ttylocs file corresponding to this terminal,
     then use that one, regardless of where the user is actually logged in
     from. */
  if (*hostname && (result = tty_lookup (hostname, line)) ||
      (result = tty_lookup (NULL, line)))
    return (result);

  /* If this user is remotely logged in, then the tty location is whatever
     that remote host is.  But, if there is a local entry in the table for
     that remote host, then use that entry. */
  if (UTHOST(entry)[0])
    {
      char host[UTHOSTSIZE + 1], *tty;
      static char remote_host[UTHOSTSIZE + 2]; /* extra space for leading '>' */

      COPY_NUL(host, UTHOST(entry));

      /* If we are displaying unique entries for each user,
	 check if the login is from an X display or screen-managed terminal.
	 This would give us definite proof of location.
	 In non-unique mode we give the original host field for completeness. */
      if (unique && (tty = strchr (host, ':')))
	{
	  char *screen;

	  /* We used to allow any display number here (isdigit (tty[1])),
	     but htis would include Xterminals which can be located anywhere.
	     Only assume display 0 is at the console location. */
	  if (tty[1] == '0')
	    {
	      /* entry by X terminal emulator: [HOST]:DISPLAY	*/
	      *tty = '\0';

	      /* if the display name is empty or "unix", must be the console */
	      if (!*host || strcmp (host, "unix") == 0)
		{
		  if ((result = tty_lookup (hostname, CONSOLE)) ||
		      (result = tty_lookup (hostname, NULL)) ||
		      (result = tty_lookup (NULL, CONSOLE)))
		    return (result);
		}
	      else if ((result = tty_lookup (host, CONSOLE)) ||
		       (result = tty_lookup (host, NULL)))
		return (result);

	      *tty = ':';
	    }
	  else if ((screen = strchr (tty + 1, ':')) && screen[1] == 'S')
	    {
	      /* entry by screen terminal emulator: [HOST]:TTY:S.#	*/
	      *tty = '\0';
	      *screen = '\0';

	      if (host == tty)	/* empty hostname */
		{
		  if ((result = tty_lookup (hostname, tty + 1)) ||
		      strcmp (tty + 1, CONSOLE) == 0 &&
		        (result = tty_lookup (hostname, NULL)) ||
		      (result = tty_lookup (NULL, tty + 1)))
		    return (result);
		}
	      else if ((result = tty_lookup (host, tty + 1)) ||
		       strcmp (tty + 1, CONSOLE) == 0 &&
		         (result = tty_lookup (host, NULL)))
		return (result);

	      *tty = ':';
	      *screen = ':';
	    }
	}

      /* Don't confuse the caller.  Make sure they can tell the location of
	 the host is referencing the host that is rlogging into this host.
	 A '>' character as the first character of the hostname is a reasonable
	 solution (and takes up only one extra column). */
      sprintf(remote_host, ">%s", host);
      return (remote_host);
    }

  /* If all else fails, try to look up the host itself.  Unlike in previous
     version we do look up only "host", not "host:console". This way the
     contents of the ttylocs file decides whether we want the host location
     to be assumed as the default location for all terminals or not. */
  if (*hostname && (result = tty_lookup (hostname, NULL)))
    return (result);

  return ("");
}

/* Find local host's DNS domain */
char *get_domain_name ()
{
  static char *domainname;
  FILE *rconf;

  /* saved from previous call */
  if (domainname)
    return (domainname);

  /* first, try override in hostconfig file */
  (void)open_host_config ();
  domainname = get_config_entry ("domainname", NULL);
  if (domainname)
    return (domainname);
    
  domainname = xmalloc(MAXHOSTNAMELEN + 1);
  *domainname = '\0';

  /* next, try /etc/resolv.conf */
  if (rconf = fopen (RESOLV_CONF, "r"))
    {
      char linebuf[265];

      while (fgets (linebuf, sizeof(linebuf), rconf))
	{
	  /* XXX: make sure field width is at least MAXHOSTNAMELEN */
	  if (sscanf (linebuf," domain %64s", domainname) == 1)
	    break;
	}
      fclose (rconf);
    }

  /* Check the local host name for a domain part. */
  if (! *domainname)
    {
      if (gethostname (domainname, MAXHOSTNAMELEN + 1) < 0 ||
 	  !strchr (domainname, '.'))
	*domainname = '\0';
      else
        domainname = strchr (domainname, '.') + 1;
    }

#ifdef HAVE_GETDOMAINNAME
  /* Some people like to set getdomainname(3) to the DNS domain. */
  if (! *domainname)
    {
      if (getdomainname (domainname, MAXHOSTNAMELEN + 1) < 0)
	*domainname = '\0';
    }
#endif /* HAVE_GETDOMAINNAME */

  return domainname;
}

/* Get fully qualified local hostname */
char *get_full_hostname ()
{
  static char *localname;
  char *domainname;

  if (localname)
    return (localname);

  localname = xmalloc(MAXHOSTNAMELEN + 1);
  *localname = '\0';

  if (gethostname(localname, MAXHOSTNAMELEN) < 0)
    file_error (FATAL, "gethostname");
  
  if (strchr (localname, '.'))
    return (localname);

  /* Hostname has no domain, so tack it on ourselves */
  domainname = get_domain_name();
  if (*domainname)
    {
      strcat (localname, ".");
      strncat (localname, domainname, MAXHOSTNAMELEN - strlen(localname));
    }

  localname[MAXHOSTNAMELEN] = '\0';
  return (localname);
}

/* Strip local domain off hostname */
/* strips off a full or partial domain name suffix that matches the
   local domain, e.g., assuming the local domain is "icsi.berkeley.edu":

	foo.icsi.berkeley.edu	==>	foo
	foo.bar.icsi.berkeley.edu ==>	foo.bar
	foo.cs.berkeley.edu	==>	foo.cs
	foo.icsi		==>	foo
	foo.stanford.edu	==>	foo.stanford
	foo.bar.mil		==>	foo.bar.mil
	foo			==>	foo
 */
char *strip_hostname (hostname)
    char *hostname;
{
  static char stripped[MAXHOSTNAMELEN + 1];
  char *domainname = get_domain_name();
  char *p, *q;
  int hostname_len = strlen(hostname);

  strncpy (stripped, hostname, MAXHOSTNAMELEN); 
  stripped[MAXHOSTNAMELEN] = '\0';

  if (! *domainname)
    return (stripped);

  for ((p = strchr(stripped, '.')) && p++; p; (p = strchr(p, '.')) && p++)
    {
      int len = hostname_len - (p - stripped);	/* length of remaining suffix */

      for (q = domainname; q; (q = strchr(q, '.')) && q++)
	{
	  /* check whether the hostname suffix p is a prefix of the domainname
	     suffix q */
	  if (toupper(*p) == toupper(*q) &&
	      strnicmp (p, q, len) == 0 &&
	      (q[len] == '\0' || q[len] == '.'))
	    {
	      *(p - 1) = '\0';
	      break;
	    }
	}
      if (q) break;
    }

  return (stripped);
}

/* compare two hostnames for equality */
int
same_hostname (name1, name2)
    char *name1, *name2;
{
    char stripped1[MAXHOSTNAMELEN];

    /* do a quick check before we go through all this trouble */
    if (toupper (*name1) != toupper (*name2))
      return (0);

    strncpy(stripped1, strip_hostname(name1), sizeof(stripped1));
    return (strnicmp(stripped1, strip_hostname(name2), sizeof(stripped1)) == 0);
}

/* compare two hosts for identity by comparing their IP addresses
   arguments can be either domain names or dotted octets */
int
same_hostip (host1, host2)
    char *host1, *host2;
{
    struct hostent *hostent;
    unsigned long addr1, addr2;

    if (isdigit(*host1))
      addr1 = inet_addr (host1);
    else if (hostent = gethostbyname (host1))
      bcopy (hostent->h_addr, &addr1, 4);
    else
      return 0;

    if (isdigit(*host2))
      addr2 = inet_addr (host2);
    else if (hostent = gethostbyname (host2))
      bcopy (hostent->h_addr, &addr2, 4);
    else
      return 0;

    return (addr1 == addr2);
}

