#define RK_C TRUE
/*========================================================================
 *
 * Name - %M%
 *
 * Version:    %I%
 *
 * ccsid:      %W% - %G% %U%
 * from:       %F%
 * date:       %H% %T%
 *
 * %Q%
 *
 * Description: Initial and Main Control for The Reactive Keyboard
 *
 *========================================================================
 */

#include <stdio.h>

#include "rkglobal.h"    /* JJD 3-89 */
#include "rk.h"
#include "defaults.h"
#include "system.h"
#include "reactive.h"    /* JJD 9-86 */
#include "functions.h"    /* MLJ 5-89 */
#include "display.h"
#include "kbdbind.h"
#include "kbdsys.h"
#include "utl.h"

#define BUFF_SIZE 1024
#define XOFF_OVERRUN 30  /* max overrun on ^S */

/* Global Variables */

static TermInfoSType pty_sgtty;
static int pty_slave, tty;
static BoolType canonical_kbd;

int     finished=0;
int    childid;

static void childdied();
static int cleanup();
static int do_file();
static int edit_line();
static void execShell();
static int handleFinishedEditing();
static void init_slave();
static void init_terminal();
int read_pty();
static set_up_buffs();
static void waitActivity();
int main();

#define LETTERS "pqr"

/* =======================================================================
 * Name - childdied
 *
 * Purpose - Field dying children (usually ignore)
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void childdied()
{
#if defined(FLG_HPUX_UNIX)
int w;
#else
union wait      w;
#endif

register int    childPid;

while ((childPid = wait3(&w, WUNTRACED | WNOHANG,
    (struct rusage *)NULL)) > 0) {
  if (WIFSTOPPED(w)) {
    fprintf(stderr,"%s: Child %d Stopped.\r\n", ProgramName, childPid);
    }
  else
  if (WIFEXITED(w)) {
    }
  else
  if (WIFSIGNALED(w)) {
    fprintf(stderr,"%s: Child %d Killed.\r\n", ProgramName, childPid);
    }

/* if this is our shell who died, shut down */
  if (childPid == childid)
    abortit("", 0);
  }
}


/* =======================================================================
 * Name - cleanup
 *
 * Purpose - General-purpose exceptional signal handler
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int cleanup(sig)
unsigned sig;
{

fflush(stdout);
fprintf(stderr, "\r\n\r\n%s:   Abnormal signal %d received \r\n", ProgramName,
    sig);
psignal(sig, "Signal is");
fprintf(stderr, "\r");
perror("Last error is");
fprintf(stderr, "\r");
fflush(stderr);
abortit("", -1);

return 1; /* for lint's sake */
}

/* =======================================================================
 * Name - do_file
 *
 * Purpose - Main control loop
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int do_file()
{
int num;
char buf[BUFF_SIZE];
BoolType kbdActivity, ptyActivity;

signal(SIGQUIT, cleanup); /* quit from terminal */
signal(SIGILL, cleanup); /* illegal instruction */
signal(SIGFPE, cleanup); /* floating point exception */

#if defined(FLG_AIX_UNIX)
signal(SIGDANGER, cleanup);
#endif

signal(SIGBUS, cleanup); /* bus error */
signal(SIGSEGV, cleanup); /* segmentation violation */
signal(SIGSYS, cleanup); /* bad arg to system call */
signal(SIGTERM, cleanup); /* software terminate - no core */

#if !defined(FLG_AIX_UNIX)
signal(SIGABRT, cleanup); /* abort */

#if !defined(FLG_HPUX_UNIX)
signal(SIGXFSZ, cleanup); /* file size limit exceeded */
signal(SIGXCPU, cleanup); /* exceeded CPU time limit */
#endif

signal(SIGEMT, cleanup); /* emulator trap */
signal(SIGLOST, cleanup); /* resource lost */
#endif


kbdActivity = FALSE;
ptyActivity = FALSE;
make_a_prediction();
while (TRUE) {
  if (canonical_kbd) {/* Make sure a prediction is displayed */
    draw_current_edit_line();
    if(!kbdActivity)
	display_pred_buffer();
    }
  else {
    erase_pred_buffer();
    erase_current_edit_line();
    }
  /* Wait until something happens */
  waitActivity(&kbdActivity, &ptyActivity, TRUE);
  /* Keystrokes get priority over terminal output */
  if (kbdActivity) {
    if (canonical_kbd) /* run through our line editor */
      num = edit_line(buf); /* get some back? */
    else /* read them raw */
      num = read_kbd(buf, sizeof(buf));
    if (num != 0) /* pass on to slave */
      write(pty_master, buf, num);
    }
  if (ptyActivity && (outputEnabled || !canonical_kbd)) { /* data for display */
    /* Make sure edit line and prediction disappear */
    erase_pred_buffer();
    erase_current_edit_line();
    /* Display, in fairly small increments, until done or user interrupts */
    do {
      num = read_pty(buf, (int)(canonical_kbd ? XOFF_OVERRUN : sizeof(buf)));
      waitActivity(&kbdActivity, &ptyActivity, FALSE); /* poll */
      if (canonical_kbd)
        flush_pty(buf, num); /* Try to track prompt */
      else { /* raw mode */
        PromptStrSize = 0;
       write_display(buf, num);
        }
      } while (canonical_kbd && ptyActivity && !kbdActivity);
    waitActivity(&kbdActivity, &ptyActivity, FALSE); /* race condition? */
    }
  }
}

/* =======================================================================
 * Name - edit_line
 *
 * Purpose - Control loop for line editing
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int edit_line(c)
char           *c;
{
char            ch;
int             status, num;

FILE           *popen();
BoolType kbdActivity, ptyActivity;

status = OK;

kbdActivity = FALSE;
ptyActivity = FALSE;
while (status == OK || status == HAVE_CHAR) {
  /* Make sure everything shows */
  draw_current_edit_line();
  if(!kbdActivity)
	display_pred_buffer();
  waitActivity(&kbdActivity, &ptyActivity, TRUE); /* wait for activity */
  if (!canonical_kbd) { /* No longer canonical; defer to other control loop */
    strcpy(c, "");
    return 0;
    }
  if (kbdActivity) { /* keyboard events get priority over display */
    read_kbd(&ch, 1);
    if (!eight_bit_mode)
      ch = STRIP_PARITY(ch);
    erase_pred_buffer(); /* functions get edit line but no prediction */
    status = dispatch_char(ch);
    flush_termcap(); /* flush results */
    if (pred_mode && (status == OK || status == HAVE_CHAR)) {
      make_a_prediction();
      waitActivity(&kbdActivity, &ptyActivity, FALSE); /*poll */
      if (!kbdActivity)
      display_pred_buffer();
      }
    }
  else
  if (outputEnabled && ptyActivity) { /* output, but no keyboard */
    /* Make everything vanish */
    erase_pred_buffer();
    erase_current_edit_line();
    /* Fairly tight loop, outputing data until done or user interrupt */
    do {
      EditStrType temp_str;

      num = read_pty(temp_str,
         (int)(canonical_kbd ? XOFF_OVERRUN : sizeof(temp_str)));
      waitActivity(&kbdActivity, &ptyActivity, FALSE); /* poll */
      if (canonical_kbd)
         flush_pty(temp_str, num); /* track prompt */
      else { /* very weird.  Flush bytes and get out of here */
        PromptStrSize = 0;
       write_display(temp_str, num);
        }
      } while (canonical_kbd && ptyActivity && !kbdActivity);
    if (!canonical_kbd) { /* went non-canonical.  Weird */
      strcpy(c, "");
      return 0;
      }
    }
  }

switch (status) {
case FINISHED_EDITING:
  return handleFinishedEditing(c);
  /*NOTREACHED*/
  break;

case FINISHED_BUT_DONT_ADD_CTRL_M:
  strcpy(c, TheEditData.current_buffer);
  TheEditData.dot = TheEditData.current_buffer;
  TheEditData.mark = TheEditData.current_buffer;
  *TheEditData.dot = '\0';
  return strlen(c);
  /*NOTREACHED*/
  break;

default:;
  }

c[0] = '\0';
return 0;
}

/* =======================================================================
 * Name - execShell
 *
 * Purpose - exec a shell
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void execShell()
{
  char *shellname;
  char **env;
  char **envptr;
  extern char **environ;
  char **ep;
  char *argv[2];
  char arg0[MAXPATHLEN];
  char *basename;
  char *p;

shellname = (char *)getenv ("SHELL");
if(shellname == NULL)
  shellname="/bin/csh";

/* get size of current environment table */
  for (ep = environ; *ep != NULL; ep ++);
/* make new environment */
  if ((env = (char **)MALLOC((ep + 2) - environ)) == NULL) {
    EditStrType junkStr;

    sprintf(junkStr, "%s: execShell, malloc() error (no memory)", ProgramName);
    perror(junkStr);
    abortit("", -1);
    }
/* copy new environment, avoiding RK environment variable */
  envptr = environ;
  for (ep = env; *envptr != NULL; envptr ++) {
    if (strncmp(*envptr, "RK=", 3) != 0)
      *ep ++ = *envptr;
    }
/* complete environment list appropriately */
  *ep ++ = "RK=";
  *ep = NULL;
/* do the invocation */
  basename = shellname;
  for (p = shellname; *p != '\0'; p++) {
    if (*p == '/')
      basename = p + 1;
    }
  if (login) {
    arg0[0] = '-';
    for (p = arg0 + 1;
#if defined(FLG_HPUX_UNIX) /* brain damaged optimizer */
        p <= arg0 + MAXPATHLEN - 1
#else
        p < arg0 + MAXPATHLEN
#endif
          && *basename != '\0';
        *p ++ = *basename ++)
      ;
    *p = '\0';
    basename = arg0;
    }
  argv[0] = basename;
  argv[1]= NULL;

  if (execve(shellname, argv, env) < 0) {
    EditStrType junkStr;

    sprintf(junkStr, "%s: execve() error", ProgramName);
    perror(junkStr);
    abortit("", -1);
    }
}

/* =======================================================================
 * Name - handleFinishedEditing
 *
 * Purpose - Do cleanup before returning string
 *
 * Arguments:  c -- the string
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int handleFinishedEditing(c)
char c[];
{
char *homeEnv = getenv("HOME");
EditStrType tstring;
char *cptr;
int count;
EditChunkSType *buf_to_use, *cur_line;

#if 0
struct stat     buf;  /* JJD 3-89 was 512 */
#endif

strcpy(c, TheEditData.current_buffer);
buf_to_use = TheEditData.current_ed_buff_ptr;

#if 0 /* following makes it impossible to enter a file name as data */
if (strlen(c) != 0) {
  if (stat(c, &buf) != -1) {  /* if just a directory
                              * name on command line,
                              * assume cd  */

      /*
       * JJD: this is kind of dicey as is, assumes
       * a bit too much about dir names
       */
      /*
       * it should check that it is the only thing
       * on the line for one thing
       */

    if ((buf.st_mode & S_IFMT) == S_IFDIR) {
      if (strcmp(".", c) != 0) {  /* JJD 3-89 del . case
                                  * for ed... */
       strcpy(tstring, "cd ");
       strcat(tstring, c);
       strcpy(c, tstring);
       }
      }
    }
  }
#endif

  /*
   * JJD: to be really useful, this should also know about
   * pushd and popd
   */
if (strncmp(c, "cd ", 3) == 0 ||
    strncmp(c, "chdir ", 6) == 0 ||
    strcmp(c, "cd") == 0 ||
    strcmp(c, "chdir") == 0) {
  if (strncmp(c, "cd ", 3) == 0)
    cptr = &c[2];  /* JJD 3-89, c[-1]  */
  else
    cptr = &c[5];
  while (*cptr == ' ' || *cptr == '\t')
    cptr++;
  tstring[0] = '\0';
  if (*cptr == '\0' ||
      strcmp(c, "chdir") == 0 ||
      strcmp(c, "cd") == 0) {
    if (homeEnv != NULL)
      strcpy(tstring, homeEnv);
    }
  else {
    if (*cptr == '~' || *cptr == '$') {  /* JJD 3-89 added myabspath, rewrote */
      char *optr = cptr;

      if (*(++cptr) != '\0') {
       while (*cptr != '\0' && *cptr != ';')
         cptr++;
       *cptr = '\0';
       if (myabspath(optr, tstring) != 0)
         strcpy(tstring,"."); /* we have an error so stay in the current dir */
       }
      else
      if (homeEnv != NULL)
       strcpy(tstring, homeEnv);
      count = strlen(tstring);
      }
    else
      count = 0;
    for (; *cptr != '\0' && *cptr != ';'; cptr++, count++)
      tstring[count] = *cptr;
    tstring[count] = '\0';
    }
  if (tstring[0] != '\0')
    chdir(tstring);
  }


strcat(c, "\n");/* JJD 2-89 changed ^M to \n */
cur_line = TheEditData.current_ed_buff_ptr;
cur_line->dot = TheEditData.dot;
cur_line->mark = TheEditData.mark;
if (cur_line != buf_to_use) {
  strcpy(buf_to_use->string, cur_line->string);
  buf_to_use->dot = &buf_to_use->string[cur_line->dot - cur_line->string];
  buf_to_use->mark = &buf_to_use->string[cur_line->mark - cur_line->string];
  }
if (*(buf_to_use->string) != '\0') {
  TheEditData.current_ed_buff_ptr = buf_to_use->next_ptr;
  TheEditData.current_ed_buff_ptr->dot =
      TheEditData.current_ed_buff_ptr->string;
  TheEditData.dot = TheEditData.current_ed_buff_ptr->dot;
  /*
   * TheEditData.current_ed_buff_ptr->mark =
   * TheEditData.current_ed_buff_ptr->string;
   */
  TheEditData.mark = TheEditData.current_ed_buff_ptr->mark;
  *TheEditData.dot = '\0';
  TheEditData.current_buffer = TheEditData.dot;
  }
else {
  TheEditData.current_ed_buff_ptr = buf_to_use;
  TheEditData.current_buffer = buf_to_use->string;
  TheEditData.dot = buf_to_use->dot;
  TheEditData.mark = buf_to_use->mark;
  }

/*  strcat(c, ""); */
update_the_model(c);  /* JJD 9-86 */
make_a_prediction();  /* JJD 9-86 */
return strlen(c);
}

/* =======================================================================
 * Name - init_slave
 *
 * Purpose - Init slave tty
 *
 * Arguments:  buf -- Name of pty
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void init_slave(buf)
char buf[];
{
TermInfoSType slave_sgttyb;
int pid;
int zero = 0;

#if defined(FLG_HPUX_UNIX)
setsid(); /* create new session */
#else
if ((tty = open("/dev/tty", O_RDWR)) < 0) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: open(\"/dev/tty\", O_RDWR) error", ProgramName);
  perror(junkStr);
  abortit("", -1);
  }
ioctl(tty, TIOCNOTTY, (int)&zero); /* disassociate from terminal */
close(tty);
#endif

close(pty_master);
close(0);

#if defined(FLG_AIX_UNIX)
    setpgrp(1);
#endif

if ((pty_slave = open(buf, (O_RDWR | O_EXCL))) < 0) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: open(\"%s\",) error", ProgramName, buf);
  perror(junkStr);
  abortit("", -1);
  }
dup2(0, 1);
dup2(0, 2);
getTty(KEYBOARD, &slave_sgttyb);

#if defined(FLG_AIX_UNIX) || defined(FLG_HPUX_UNIX)
slave_sgttyb.c_lflag |= ICANON | ECHO | ISIG;
slave_sgttyb.c_cflag &= ~CBAUD;
slave_sgttyb.c_cflag |= cooked_stdin.c_cflag & CBAUD;
slave_sgttyb.c_oflag &= ~OPOST;
#else
slave_sgttyb.sg_flags &= ~(RAW | CBREAK| LCASE);
slave_sgttyb.sg_flags |= (ECHO | FF0 | CRMOD );
slave_sgttyb.sg_ospeed = cooked_stdin.sg_ospeed;
slave_sgttyb.sg_ispeed = cooked_stdin.sg_ispeed;
#endif
setTty(KEYBOARD, &slave_sgttyb);

#if !defined(FLG_AIX_UNIX)
pid = getpid();
if (setpgrp(pid, pid) < 0) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: setpgrp() error", ProgramName);
  perror(junkStr);
  abortit("", -1);
  }

#if defined(FLG_HPUX_UNIX)
tcsetpgrp(KEYBOARD, getpid());
#else
ioctl(KEYBOARD, TIOCSPGRP, (int)&pid);
#endif

#endif
}

/* =======================================================================
 * Name - init_terminal
 *
 * Purpose - Initialize display and master pty
 *
 * Arguments:  buf -- name of pty, to be filled in
 *
 * Returns     function return -- None
 *
 *========================================================================
 */

static void init_terminal(buf)
char buf[];
{
int             i = 48, c = 0;
TermInfoSType  master_sgttyb;

while (--i >= 0) {
  sprintf(buf, "/dev/pty%c%x", LETTERS[c >> 4], c & 15);
  if ((pty_master = open(buf, (O_RDWR | O_EXCL))) != -1)
    break;
  c++;
  }
if (i <= 0) {
  printf("%s: Unable to open a pty, aborting...\r\n", ProgramName);
  abortit("", -1);
  }
sprintf(buf, "/dev/tty%c%x", LETTERS[c >> 4], c & 15);
kbd_init();
kbd_raw(KEYBOARD);

getTty(pty_master, &master_sgttyb);

#if defined(FLG_AIX_UNIX) || defined(FLG_HPUX_UNIX)
  master_sgttyb.c_lflag |= ICANON | ECHO | ISIG;
  master_sgttyb.c_cflag &= ~CBAUD;
  master_sgttyb.c_cflag |= cooked_stdin.c_cflag & CBAUD;
  master_sgttyb.c_oflag |= OPOST | ONLCR;

#if defined(FLG_HPUX_UNIX)
  /* hpux doesn't init these values properly... */
  master_sgttyb.c_cc[VINTR]  = '\003';
  /* no flush */
  master_sgttyb.c_cc[VSTART]  = '\025';
  master_sgttyb.c_cc[VSTOP]  = '\021';
  master_sgttyb.c_cc[VEOF]  = '\004';
  master_sgttyb.c_cc[VQUIT]  = '\034';
  master_sgttyb.c_cc[VSUSP]  = '\032';
#endif

#else
  master_sgttyb.sg_flags &= ~(RAW | CBREAK| LCASE);
  master_sgttyb.sg_flags |= (ECHO | CRMOD | FF0);
  master_sgttyb.sg_ospeed = cooked_stdin.sg_ospeed;
  master_sgttyb.sg_ispeed = cooked_stdin.sg_ispeed;
#endif
  setTty(pty_master, &master_sgttyb);
  canonical_kbd = TRUE;
}

/* =======================================================================
 * Name - read_pty
 *
 * Purpose - Read bytes from slave process
 *
 * Arguments:
 *
 * Returns     function return -- number of bytes read
 *
 *========================================================================
 */
int read_pty(buf, nbytes)
char *buf;
int nbytes;
{
int ret;
static int count=0;
int realfcntl;
int junk;

realfcntl = fcntl(pty_master, F_GETFL, 0);
junk = realfcntl | FNDELAY;
fcntl(pty_master, F_SETFL, junk);
ret = read(pty_master, buf, nbytes);
if (ret > -1)
  goto ALL_DONE;
if (errno == EWOULDBLOCK)
  return 0;
if (count != 0) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: read(pty_master,,) error", ProgramName);
  perror(junkStr);
  abortit("", -1);
  }
count ++;
sleep(1);  /* wait for the signal */

ALL_DONE:
fcntl(pty_master, F_SETFL, realfcntl);
return ret;
}

/* =======================================================================
 * Name - set_up_buffers
 *
 * Purpose - Initialize circular list of edit buffers
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int set_up_buffers()
{
int             num_buffs;
EditChunkSType *prev_ptr;
EditChunkSType *new_chunk;

new_chunk = (EditChunkSType *)MALLOC(sizeof(EditChunkSType));
if (new_chunk == NULL)
  abortit("Can't even allocate one edit buffer!", -1);
TheEditData.current_ed_buff_ptr = new_chunk;

prev_ptr = TheEditData.current_ed_buff_ptr;
prev_ptr->dot = prev_ptr->string;
prev_ptr->mark = prev_ptr->string;  /* JJD 3-89 added */

for (num_buffs = num_buffers - 1; num_buffs != 0; --num_buffs) {
  new_chunk = (EditChunkSType *)MALLOC(sizeof(EditChunkSType));
  if (new_chunk == NULL) {
    EditStrType junkStr;

    sprintf(junkStr, "Can't allocate %d buffers of %d bytes (%d total)",
        num_buffers - num_buffs, sizeof(EditChunkSType),
       (num_buffers - num_buffs) * sizeof(EditChunkSType));
    abortit(junkStr, -1);
    }
  prev_ptr->next_ptr = (EditChunkSType *)MALLOC(sizeof(EditChunkSType));
  prev_ptr->next_ptr->prev_ptr = prev_ptr;
  prev_ptr = prev_ptr->next_ptr;
  prev_ptr->dot = prev_ptr->string;
  prev_ptr->mark = prev_ptr->string;  /* JJD 3-89 added */
  *(prev_ptr->dot) = '\0';
  }

prev_ptr->next_ptr = TheEditData.current_ed_buff_ptr;
TheEditData.current_ed_buff_ptr->prev_ptr = prev_ptr;

TheEditData.current_input_char = '\0';
TheEditData.factor = 1;
TheEditData.current_buffer = TheEditData.current_ed_buff_ptr->string;
TheEditData.dot = TheEditData.current_buffer;
/* JJD 3-89 added */
TheEditData.mark = TheEditData.current_ed_buff_ptr->mark;
TheEditData.current_buffer[0] = '\0';
}

/* =======================================================================
 * Name - waitActivity
 *
 * Purpose - Poll or wait for activity from keyboard or slave pty
 *
 * Arguments:  waitMode -- TRUE if we should sleep if no activity,
 *                              FALSE if this is just a poll
 *
 * Returns     function return --
 *                 *kbd -- whether keyboard wants service
 *                 *pty -- whether slave job wants service (is writing)
 *
 *========================================================================
 */
#if defined(FLG_AIX_UNIX) || defined(FLG_HPUX_UNIX)
#define IS_CANONICAL ((pty_sgtty.c_lflag & ECHO) && \
       (pty_sgtty.c_lflag & ICANON))
#else
#define IS_CANONICAL ((pty_sgtty.sg_flags & ECHO) && \
        !(pty_sgtty.sg_flags & (RAW | CBREAK)))
#endif

static struct timeval zeroWait = {0, 0};

static void waitActivity(kbd, pty, waitMode)
BoolType * kbd;
BoolType * pty;
BoolType waitMode;
{
fd_set readfd;
fd_set exceptfd;
struct timeval * theWait;

if (waitMode)
  theWait = NULL;
else
  theWait = &zeroWait;
*kbd = FALSE;
*pty = FALSE;
FD_ZERO(&readfd);
FD_SET(KEYBOARD, &readfd);
FD_SET(pty_master, &readfd);
FD_ZERO(&exceptfd);
FD_SET(KEYBOARD, &exceptfd);
FD_SET(pty_master, &exceptfd);
if (select(getdtablesize(), &readfd, (fd_set *)NULL, &exceptfd, theWait) < 0) {
  if (errno == EINTR) {
    errno = 0;
    return;
    }
  else {
    EditStrType junkStr;

    sprintf(junkStr, "%s: select() error", ProgramName);
    perror(junkStr);
    }
  }
*kbd = FD_ISSET(KEYBOARD, &readfd) || FD_ISSET(KEYBOARD, &exceptfd);
*pty = FD_ISSET(pty_master, &readfd) || FD_ISSET(pty_master, &exceptfd);
getTty(pty_master, &pty_sgtty);
canonical_kbd = IS_CANONICAL;
resizeWindow();
}

/* =======================================================================
 * Name - main -- where it all starts
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int main(argc, argv)
int             argc;
char          *argv[];
{
EditStrType buf;
EditStrType junkStr;
double loadAverage;

get_command_line_arguments(argc,argv);
if (!silent) {
  printf(
"Welcome to the Reactive Keyboard, written at the University of Calgary.\n");
  printf("Version: %s\r\n", RK_VERSION);
  }
loadAverage = getLoadAverage();

init_terminal(buf); /* init display and master pty */
childid = fork();
switch (childid) {
case -1: /* error on fork */
  sprintf(junkStr, "%s: fork() error", ProgramName);
  perror(junkStr);
  abortit("", -1); /* no return */
  break;

case 0: /* Child process: go directly to the shell */
  signal(SIGINT, (UnixSignalFuncType *)SIG_DFL);
  signal(SIGQUIT, (UnixSignalFuncType *)SIG_DFL);
  init_slave(buf); /* slave pty */
  execShell(); /* no return */
  break;

default: /* Master process.  Go on to line editor */
  signal(SIGCHLD, childdied);
#if defined(FLG_AIX_UNIX) || defined(FLG_HPUX_UNIX)
  signal(SIGHUP, (UnixSignalFuncType *)SIG_IGN);
#endif
  break;
  }

get_termcap_stuff(); /* only after creating pty */
set_up_keymap();     /* only after init'ing termcap */
get_key_bindings(key_file); /* only after default bindings from termcap */
if (!silent) {
  char *theBind = getFirstBinding(show_bindings);

  printf( "Please wait until your shell prompt appears ");
  if (theBind != NULL)
    printf( "(\"%s\" for help).\r\n", theBind);
  printf("\r\n");
  fflush(stdout);
  }
init_reactive(loadAverage);
set_up_buffers();
do_file();
/*NOTREACHED*/

abortit("", 1);    /* JJD 3-89 added 1 */
}
