#define KBDBIND_C TRUE
/*========================================================================
 *
 * Name - %M%
 *
 * Version:    %I%
 *
 * ccsid:      %W% - %G% %U%
 * from:       %F%
 * date:       %H% %T%
 *
 * %Q%
 *
 * Description: Process the key binding commands at initialization
 *
 *========================================================================
 */

#include <stdio.h>
#include <ctype.h>

#include "rkglobal.h"
#include "defaults.h"
#include "system.h"
#include "rk.h"
#include "display.h"
#include "kbdsys.h"

#include "kbdbind.h"
#include "functions.h"

#define MAXKEYLEN 19
#define MAXTOKEN  30
#define MAX_KBD_INPUT 255

#define UPCASE(c) (islower(c)?toupper(c):c)
#define KEYWORDS (sizeof(functions)/sizeof(struct func))
#define isodigit(c) ((isdigit(c))&&(c<'8'))

FILE *filedesc;
#define INVALID_FUNCTION -1

static char *CurrentLine; /* Line currently being parsed */
static char *CurrentLinePtr; /* Next character to be read from that line */

/* Following two values for error reporting */
static int CurrentLineNumber = 0;
static char * theFileName = (char *)NULL;

/* Following for managing multiple-key (meta) character sequences */
static char next_free_map = 1;

static BoolType initialized = FALSE;

static char TheMetaPrefixes[MAXEXTENSIONS][MAXEXTENSIONS];
static int  TheMetaMap[MAXEXTENSIONS][MAXEXTENSIONS];

/* The Dispatch Table*/
static int  (*KbdCallTable[MAX_KBD_INPUT][MAXEXTENSIONS])();

typedef int KbdFuncType();
static KbdFuncType *lastFunc = (KbdFuncType *)NULL;

static struct func {
  char *name;
  KbdFuncType *address;
  char *description;
} functions[] =

/*********
  ATTENTION -- following names must be in ASCII sorted order...
*********/

  {
    {"BOGUS",
      BOGUS,
     "Null routine, beeps terminal bell"},

    {"accept_forward_char",
      accept_forward_char,
     "Accept the next predicted character"},

    {"accept_forward_word",
      accept_forward_word,
     "Accept the next predicted word"},

    {"accept_to_end_of_line",
      accept_to_end_of_line,
     "Accept the whole predicted line"},

    {"accept_to_eol",
      accept_to_eol,
     "Accept the predicted line up to ^J"},

    {"backspace_char",
      backspace_char,
     "Backspace (delete) a single character"},

    {"backspace_word",
      backspace_word,
     "Backspace (delete) a single word"},

    {"backward_char",
      backward_char,
     "Go backwards a single character"},

    {"backward_paren",
      backward_paren,
     "Go backwards to matching parenthesis \"(\""},

    {"backward_word",
      backward_word,
     "Go backwards a single word"},

    {"beginning_of_line",
      beginning_of_line,
     "Move to the beginning of the line"},

    {"bogus",
      BOGUS,
     "Null routine, beeps terminal bell"},

    {"capitalize_word",
      capitalize_word,
     "Capitalize this word"},

    {"clear_display",
      clear_display,
     "Clear the screen and redraw the line"},

    {"close_paren",
      close_paren,
     "Close and show matching parenthesis"},

    {"command_completion",
      command_completion,
     "Expand a command using $PATH"},

    {"dash_to_ul_word",
      dash_to_ul_word,
     "Convert -'s to _'s in this word"},

    {"delete_char",
      delete_char,
     "Delete a single character"},

    {"delete_char_or_insert_eof",
      delete_char_or_insert_eof,
     "Delete char or if at EOL send EOF"},

    {"delete_region_to_killbuffer",
      delete_region_to_killbuffer,
     "Delete marked region to killbuffer"},

    {"delete_word",
      delete_word,
     "Delete a single word"},

    {"discard_current_edit_line",
      discard_current_edit_line,
     "Delete this line and forget it"},

    {"discard_rest_of_line",
      discard_rest_of_line,
     "Delete rest of line to killbuffer"},

    {"end_of_line",
      end_of_line,
     "Move to the end of the line"},

    {"exchange_mark_and_set",
      exchange_mark_and_set,
     "Move cursor to mark & mark prev. cursor"},

    {"file_completion",
      file_completion,
     "Expand pathname from the current prefix"},

    {"finish_editing_line",
      finish_editing_line,
     "Enter this line"},

    {"forward_char",
      forward_char,
     "Go forward a single character"},

    {"forward_paren",
      forward_paren,
     "Move to matching close parenthesis \")\""},

    {"forward_word",
      forward_word,
     "Go forward a single word"},

    {"increment_factor",
      increment_factor,
     "Do the next command 4^(presses) times"},

    {"input_key_bindings",
      input_key_bindings,
     "Bind keystrokes from stdin or file"},

    {"insert_eof_char",
      insert_eof_char,
     "Send an EOF character"},

    {"insert_flush_char",
      insert_flush_char,
     "Send a flush character"},

    {"insert_interrupt_char",
      insert_interrupt_char,
     "Send an interrupt character"},

    {"insert_quit_char",
      insert_quit_char,
     "Send a quit character"},

    {"insert_start_char",
      insert_start_char,
     "Send a start character"},

    {"insert_stop_char",
      insert_stop_char,
     "Send a stop character"},

    {"insert_suspend_char",
      insert_suspend_char,
     "Send a suspend character"},

    {"lowercase_word",
      lowercase_word,
     "Lowercase this word"},

    {"new_working_directory",
      new_working_directory,
     "Fix RK\'s idea of your current directory"},

    {"next_line",
      next_line,
     "Show the next line buffer"},

    {"next_pred",
      next_pred,
     "Show next alternative prediction"},

    {"null",
      BOGUS,
     "Null routine, beeps terminal bell"},

    {"open_paren",
      open_paren,
     "Open and show matching parenthesis"},

    {"previous_line",
      previous_line,
     "Show the previous line buffer"},

    {"previous_pred",
      previous_pred,
     "Show previous alternative prediction"},

    {"prime_from_file",
      prime_from_file,
     "Prime the predictions from a file"},

    {"quote_char",
      quote_char,
     "Literally insert the next character"},

    {"repeater",
      repeater,
     "Repeat the previous completion command"},

    {"run_mesg",
      run_mesg,
     "Run the mesg command"},

    {"run_ruptime",
      run_ruptime,
     "Run the ruptime command"},

    {"run_talk",
      run_talk,
     "Run the talk command"},

    {"run_tty_program",
      run_tty_program,
     "Run a program with rk turned off"},

    {"run_write",
      run_write,
     "Run the write command"},

    {"self_insert",
      self_insert,
     "Literally insert the current character"},

    {"set_mark",
      set_mark,
     "Set mark at the current cursor position"},

    {"show_arguments",
      show_arguments,
     "Show the current command line arguments"},

    {"show_bindings",
      show_bindings,
     "Show the current key bindings"},

    {"show_mark",
      show_mark,
     "Show the position of the current mark"},

    {"show_system_toggles",
      show_system_toggles,
     "Show current system toggle setting."},

    {"show_termcap_info",
      show_termcap_info,
     "Show the user\'s termcap settings."},

    {"show_used_nodes",
      show_used_nodes,
     "Show memory usage info for debugging"},

    {"show_version",
      show_version,
     "Show the current version number \& date"},

    {"toggle_add_space_mode",
      toggle_add_space_mode,
     "Toggle add_space_mode (see manual)"},

    {"toggle_eol_longer_mode",
      toggle_eol_longer_mode,
     "Toggle eol_longer_mode (see manual)"},

    {"toggle_include_stuff_mode",
      toggle_include_stuff_mode,
     "Toggle skipping of nonword chars"},

    {"toggle_lisp_mode",
      toggle_lisp_mode,
     "Toggle lisp_mode (see manual)"},

    {"toggle_nl_truncate_mode",
      toggle_nl_truncate_mode,
     "Toggle nl_truncate_mode (see manual)"},

    {"toggle_only_at_eol_mode",
      toggle_only_at_eol_mode,
     "Toggle only_at_eol_mode (see manual)"},

    {"toggle_pred_mode",
      toggle_pred_mode,
     "Toggle prediction display off or on"},

    {"toggle_record_all_mode",
      toggle_record_all_mode,
     "Toggle record_all_mode (see manual)"},

    {"toggle_show_eol_mode",
      toggle_show_eol_mode,
     "Toggle display of ^J at end of line"},

    {"toggle_wrap_mode",
      toggle_wrap_mode,
      "Toggle display predictions over EOL"},

    {"twiddle_chars",
      twiddle_chars,
     "Exchange previous two characters"},

    {"ul_to_dash_word",
      ul_to_dash_word,
     "Convert _'s to -'s in this word"},

    {"unbind",
      BOGUS,					/* or perhaps self_insert? */
     "Null routine, beeps terminal bell"},

    {"uppercase_word",
      uppercase_word,
     "Uppercase this word"},

    {"yank_from_kill_buffer",
      yank_from_kill_buffer,
     "Insert text stored in killbuffer"}
  };


struct list {
  char *key;
  struct list *next;
} *bindings[KEYWORDS];

static int add_to_bindings();
static void do_bind_to_key();
static int get_cntl();
static char *get_key();
static int get_meta();
static int get_slash();
static int my_bsearch();
static int my_strcpy();
static int parse_line();
static void reportErr();
static int show_binds();

/* =======================================================================
 * Name - add_to_bindings
 *
 * Purpose - Maintains map from function to list of input strings
 *           that map to it.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int add_to_bindings(key,function)
char *key;
int (*function)();
{
int x;
BoolType found_it = FALSE;
struct list *current,*previous;
char good_key[3 * MAXKEYLEN + 1];

my_strcpy(good_key, key);
for (x = 0; x < KEYWORDS; x++) {
  current = bindings[x];
  while (bindings[x] != NULL && strcmp(bindings[x]->key, good_key) == 0) {
    current = bindings[x]->next;
    FREE(bindings[x]);
    bindings[x] = current;
    }
  previous = current; /* for lint */
  while (current != NULL) {
    if (strcmp(current->key, good_key) == 0) {
      previous->next = current->next;
      FREE(current);
      }
    previous = current;
    current = current->next;
    }
  }
for (x = 0; x < KEYWORDS; x++) {
  if (function == functions[x].address) {
    found_it = TRUE;
    break;
    }
  }
if (found_it) {
  if (bindings[x] == NULL) {
    bindings[x] = (struct list *)MALLOC(sizeof(struct list));
    bindings[x]->key = MALLOC(get_display_length(key) + 1);
    my_strcpy(bindings[x]->key, key);
    bindings[x]->next = NULL;
    }
  else {
    current = bindings[x];
    while (current->next != NULL)
      current = current->next;
    current->next = (struct list *)MALLOC(sizeof(struct list));
    current->next->key = MALLOC(get_display_length(key) + 1);
    my_strcpy(current->next->key, key);
    current->next->next = NULL;
    }
  }
}

/* =======================================================================
 * Name - input_key_bindings
 *
 * Purpose - Read key bindings, from file or stdin
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int input_key_bindings()
{
EditStrType cmd;
int i, c;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
printf("\r\nRead bindings from file? (y/n): ");
fflush(stdout);
gets(cmd);
if (cmd[0] == 'Y' || cmd[0] == 'y') {
  EditStrType fname;

  printf("\nFilename: ");
  fflush(stdout);
  gets(fname);
  get_key_bindings(fname);
  goto ALL_DONE;
  }

/* Read line at a time and feed to do_bind */
printf("Enter an empty line to end...\n");
CurrentLine = &cmd[0];
do {
  CurrentLinePtr = &cmd[0];
  i = 0;
  while ((c = fgetc(stdin)) != '\n' && c != EOF && i < sizeof(EditStrType)) {
    *CurrentLinePtr ++ = c;
    i ++;
    }
  if (i == sizeof(EditStrType)) { /* line overflow */
    reportErr("Input line is too long");
    while ((c = fgetc(stdin)) != '\n' && c != EOF);
    continue;
    }
  *CurrentLinePtr = '\0'; /* null-terminate */
  do_bind(cmd);
  if (c == EOF)
    break;
} while (cmd[0] != '\0');

ALL_DONE:
kbd_raw(KEYBOARD);
TheEditData.factor = 1; /* CLEAR_FACTOR; */
display_continue();
return OK;
}

/* =======================================================================
 * Name - show_bindings
 *
 * Purpose - Describe key bindings, possibly to a file
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *						Dejan Apr 90.
 *========================================================================
 */
int show_bindings()
{
UnixSignalFuncType *sig;
EditStrType fname;

erase_current_edit_line();
printf("\r\nSend bindings to a file? (y/n): ");
fflush(stdout);
kbd_canonical(KEYBOARD);
gets(fname);
if (fname[0] == 'Y' || fname[0] == 'y') {
  sig = signal(SIGCHLD, (UnixSignalFuncType *)SIG_DFL);
  printf("\nFilename: ");
  gets(fname);
  show_binds(fname);
  }
else {
  sig = signal(SIGCHLD, (UnixSignalFuncType *)SIG_DFL);
  fname[0] = '\0';
  show_binds((char*)NULL);
  }
signal(SIGCHLD, sig);
kbd_raw(KEYBOARD);
TheEditData.factor = 1; /* CLEAR_FACTOR; */
display_continue();
}

/* =======================================================================
 * Name - dispatch_char
 *
 * Purpose - Dispatch based on given character
 *
 * Arguments:  ch -- character to dispatch on (kludge)
 *
 * Returns     function return --
 *
 *========================================================================
 */
int dispatch_char(ch)
char ch;
{
int result;
KbdFuncType *thisFunc = KbdCallTable[(unsigned char)ch][TheKeyMap];

if (thisFunc != meta_prefix) { /* about to do real work */
  if (lastFunc == file_completion || lastFunc == command_completion) {
    if (thisFunc == repeater)
      thisFunc = lastFunc; /* repeat last completion */
    else { /* clear current completion */
      lastFunc = BOGUS;
      clearCompletion();
      if (thisFunc == file_completion || thisFunc == command_completion)
        lastFunc = thisFunc; /* starting new completion */
      }
    }
  else { /* last function not completion */
    if (thisFunc == repeater)
      thisFunc = BOGUS;
    lastFunc = thisFunc; /* and save */
    }
  }
TheEditData.current_input_char = ch;
result = (*thisFunc)();
TheKeyMap = 0;
return result;
}

/* =======================================================================
 * Name - do_bind
 *
 * Purpose - Perform key binding
 *
 * Arguments:  cmd -- command to be executed, in keybinding format
 *
 * Returns     function return --
 *
 *========================================================================
 */
void do_bind(cmd)
char cmd[];
{
char key_string[MAXKEYLEN + 1];
int function_num;

  CurrentLine = &cmd[0];
  CurrentLinePtr = CurrentLine;
  function_num = parse_line(key_string);
  if (function_num == INVALID_FUNCTION)
    return;
  add_to_bindings(key_string, functions[function_num].address);
  do_bind_to_key(key_string, 0, functions[function_num].address);
}

/* =======================================================================
 * Name - do_bind_to_key
 *
 * Purpose - Keystroke translation table update (keystrokes to functions)
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void do_bind_to_key(key, thisKeyMap, function)
char *key;
char  thisKeyMap;
int (*function)();
{
int x;

if (strlen(key) < 2) {
  KbdCallTable[(unsigned char)key[0]][thisKeyMap] = function;
  return;
  }
for (x = 0; x < MAXEXTENSIONS &&
    TheMetaPrefixes[x][thisKeyMap] != '\0' &&
    key[0] != TheMetaPrefixes[x][thisKeyMap]; x++)
  ;
if (x == MAXEXTENSIONS)
  abortit("Too many keymaps, aborting.", -1);
if (TheMetaPrefixes[x][thisKeyMap] != '\0') {
  do_bind_to_key(key + 1, (char)TheMetaMap[x][thisKeyMap], function);
  return;
  }
TheMetaPrefixes[x][thisKeyMap] = key[0];
TheMetaMap[x][thisKeyMap] = next_free_map;
KbdCallTable[(unsigned char)key[0]][thisKeyMap] = meta_prefix;
next_free_map++;
if (next_free_map >= MAXEXTENSIONS)
  abortit("Too many keymaps, aborting.", -1);
do_bind_to_key(key + 1, next_free_map - 1, function);
}

/* =======================================================================
 * Name - get_cntl
 *
 * Purpose - Parse control meta-syntax in binding file
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int get_cntl()
{
int upc;

upc = *CurrentLinePtr ++;
upc = UPCASE(upc);
if (upc == '?')
  return ASCII_RUBOUT;
if (upc == '\0') {
  reportErr("Unexpected end of string in get_cntl");
  return EOF;
  }
if (upc < '@' || upc > '_') {
  EditStrType junk;

  sprintf(junk, "Peculiar control character \"\^%c\"", upc);
  reportErr(junk);
  return EOF;
  }
return upc - '@';
}

/* =======================================================================
 * Name - get_meta
 *
 * Purpose - Parse meta chars (>= 128) in binding file
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int get_meta()
{
int upc;

upc = *CurrentLinePtr ++;
if (upc == '^') {
  int result = get_cntl();

  if (result == EOF)
    return EOF;
  return result + MAX_SET;
  }
if (upc == '\\') {
  int result = get_slash();

  if (result == EOF)
    return EOF;
  if (result >= MAX_SET) {
    reportErr("Peculiar meta-character in get_meta");
    return EOF;
    }
  return result + MAX_SET;
  }
if (upc == '\0') {
  reportErr("Unexpected end of string in get_meta");
  return EOF;
  }
return upc + MAX_SET;
}

/* =======================================================================
 * Name - getFirstBinding
 *
 * Purpose - Return first string representing binding for given function
 *
 * Arguments:
 *
 * Returns     function return -- New string, NULL if none exists
 *
 *========================================================================
 */
char * getFirstBinding(function)
KbdFuncType *function;
{
int x;

if (!initialized) /* kludge for rk -h */
  return NULL;
for (x = 0; x < KEYWORDS; x++)
  if (function == functions[x].address)
    return bindings[x]->key;
return NULL;
}

/* =======================================================================
 * Name - get_key
 *
 * Purpose - Parse input mapping sequence of a key
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static char *get_key(buffer)
char *buffer;
{
char term_cap[3];
int  x = 0;
int  c;
int junk;

if (*CurrentLinePtr != '"') {
  char *key_ptr;
  char *message;

  if (theFileName == NULL)
    message = "Default binding";
  else
    message = "Requested binding";
  term_cap[0] = *CurrentLinePtr ++;
  term_cap[1] = *CurrentLinePtr ++;
  term_cap[2] = '\0';
  key_ptr = xlat_termcap(term_cap, message, "Binding is omitted");
  if (key_ptr == NULL) {
    if (theFileName != NULL) /* kludge to get line number out */
      reportErr("Error in input file");
    return NULL;
    }
  strcpy(buffer, key_ptr);
  return &buffer[0];
  }
else
  CurrentLinePtr ++;


while ((c = *CurrentLinePtr ++) != '\"' && c != '\0'
    && x <= MAXKEYLEN) {
  if (c == '\\')
    junk = get_slash();
  else
  if (c == '^')
    junk = get_cntl();
  else
  if (c == '!')
    junk = get_meta();
  else
    junk = c;
  if (junk == EOF)
    return NULL; /* error already reported */
  buffer[x++] = junk;
  }
if (c != '\"') {
  reportErr("Quoted string not terminated");
  return NULL;
  }
buffer[x] = '\0';

if (x > MAXKEYLEN) {
  reportErr("Key too long.");
  return NULL;
  }
return &buffer[0];
}

/* =======================================================================
 * Name - get_key_bindings
 *
 * Purpose - Load key bindings from key_file
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
int get_key_bindings(f)
char f[];
{
int c;
int i;
EditStrType thisLine;

if ((filedesc = fopen(f, "r")) == NULL) {
  return;
  }

CurrentLineNumber = 0;
CurrentLine = &thisLine[0];
theFileName = &f[0];
while (TRUE) {
  CurrentLineNumber ++;
  CurrentLinePtr = &thisLine[0];
  i = 0;
  while ((c = fgetc(filedesc)) != '\n' && c != EOF && i < sizeof(EditStrType)) {
    *CurrentLinePtr ++ = c;
    i ++;
    }
  if (i == sizeof(EditStrType)) { /* line overflow */
    reportErr("Input line is too long");
    while ((c = fgetc(filedesc)) != '\n' && c != EOF);
    continue;
    }
  *CurrentLinePtr = '\0'; /* null-terminate */
  do_bind(thisLine);
  if (c == EOF)
    break;
  }
fclose(filedesc);
theFileName = NULL;
CurrentLineNumber = 0;
}

/* =======================================================================
 * Name - get_slash
 *
 * Purpose - Parse octal or termcap syntax in key string
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int get_slash()
{
int c = *CurrentLinePtr ++;
int code = 0;

if (isodigit(c)) {
  while (isodigit(c)) {
    code = (code << 3) + (c - '0');
    if (code > FACTOR_MAXIMUM) {
      reportErr("get_slash: Character Overflow");
      return EOF;
      }
    c = *CurrentLinePtr ++;
    }
  CurrentLinePtr --; /* back up */
  return code;
  }
if (c == 'E')
  return 27; /* ascii ESCAPE */
if (c == 'b') /* backspace */
  return '\b';
if (c == 'f') /* formfeed */
  return '\f';
if (c == 'n') /* newline */
  return '\n';
if (c == 'r') /* carriage return */
  return '\r';
if (c == 't') /* tab */
  return '\t';
return c;
}

/* =======================================================================
 * Name - meta_prefix
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int meta_prefix()
{
char            metad_char;
int    i=0;
int    status;

while (i < MAXEXTENSIONS && TheMetaPrefixes[i][TheKeyMap] != '\0' &&
        TheEditData.current_input_char != TheMetaPrefixes[i][TheKeyMap])
  i++;
if (i == MAXEXTENSIONS) {
  TheKeyMap = 0; /* reset */
  return BOGUS();
  }
TheKeyMap = TheMetaMap[i][TheKeyMap]; /* chain to next keymap */

#if 0 /* nice to know we're between "words" */
display_pred_buffer();  /* JJD 9-86 */
#endif
read_kbd(&metad_char, 1);
erase_pred_buffer();  /* JJD 3-89 added */
if (!eight_bit_mode)
  metad_char = STRIP_PARITY(metad_char);
TheEditData.current_input_char = metad_char;
status = dispatch_char(metad_char); /* can recurse... */
flush_termcap();
return status;
}

/* =======================================================================
 * Name - my_bsearch
 *
 * Purpose - binary search from K&R "C" book
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int my_bsearch(s)
char *s;
{
int low, high, mid, cond;

low = 0;
high = KEYWORDS - 1;
while (low <= high) {
  mid = (low + high) / 2;
  if ((cond = strcmp(s,functions[mid].name)) < 0)
    high = mid - 1;
  else
  if (cond > 0)
    low = mid + 1;
  else
    return mid;
  }
return INVALID_FUNCTION;
}


/* =======================================================================
 * Name - my_strcpy
 *
 * Purpose - Copy from b to a but replace control chars with ^-char like ^C
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int my_strcpy(a,b)
char *a,*b;
{
while (*b != '\0') {
  unsigned char temp = (unsigned char)*b;

  if (temp >= MAX_SET) { /* meta char */
    *a++ = '!';
    temp -= MAX_SET;
    }
  if (temp < ' ' || temp == ASCII_RUBOUT) {
    *a++ = '^';
    *a++ = ((temp == ASCII_RUBOUT) ? '?' : temp + '@');
    }
  else
    *a++ = temp;
  b ++;
  }
*a = '\0';
}

/* =======================================================================
 * Name - parse_line
 *
 * Purpose - Parse a line for do_bind, return function number
 *
 * Arguments:
 *
 * Returns     function return -- function number
 *             buf -- input key binding for this function
 *
 *========================================================================
 */
static int parse_line(buf)
char buf[];
/* parse a given line */
{
char token[MAXTOKEN + 1];
int c;
int number;
int x;

/* Skip leading white space */
while (isspace(c = *CurrentLinePtr ++));

if (c == '\0')
  return INVALID_FUNCTION; /* end of file, or blank line */

/* read alphabetic token */
memset(token, 0, MAXTOKEN + 1);
token[0] = c;
x = 1;
if (token[0] == '#') /* comment */
  return INVALID_FUNCTION;

while (x <= MAXTOKEN && (isalpha(c = *CurrentLinePtr ++) || c == '_'))
  token[x ++] = c;

if (isalpha(c) && x == MAXTOKEN) {
  EditStrType junkStr;

  sprintf(junkStr, "Command %s too long", token);
  reportErr(junkStr);
  return INVALID_FUNCTION;
  }
if (!isspace(c)) {
  EditStrType junkStr;

  sprintf(junkStr, "Unexpected character %d, expecting whitespace or eof", c);
  reportErr(junkStr);
  return INVALID_FUNCTION;
  }
number = my_bsearch(token);
if (number == INVALID_FUNCTION) {
  EditStrType junkStr;

  sprintf(junkStr, "Function not found: %s", token);
  reportErr(junkStr);
  return INVALID_FUNCTION;
  }
/* skip white space */
while (isspace(c = *CurrentLinePtr++));
CurrentLinePtr --;
if (get_key(buf) == NULL) /* error already reported */
  return INVALID_FUNCTION;

/* Extra check for junk at end of line */
if (*CurrentLinePtr != '\0')
  while (isspace(c = *CurrentLinePtr++));
else
  c = '\0';
if (c != '\0' && c != '#') {
  reportErr("Trailing garbage on line");
  return INVALID_FUNCTION;
  }
return number;
}

/* =======================================================================
 * Name - read_kbd
 *
 * Purpose - Read bytes from user's keyboard
 *
 * Arguments:
 *
 * Returns     function return -- number of bytes read
 *
 *========================================================================
 */
int read_kbd(buf, nbytes)
char *buf;
int nbytes;
{
int ret;
static int count=0;

flush_termcap();
ret = read(KEYBOARD, buf, nbytes);
if (ret > -1)
  goto ALL_DONE;
if (
  errno != EBADF
#if defined(FLG_NEWS_UNIX) || defined(sun3)
  && errno != EIO
#endif
   ) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: read(KEYBOARD,,) error", ProgramName);
  perror(junkStr);
  abortit("", -1);
  }
if (count != 0) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: read(KEYBOARD,,) error", ProgramName);
  perror(junkStr);
  abortit("", -1);
  }
count ++;
sleep(1);  /* wait for the signal */

ALL_DONE:
return ret;
}

/* =======================================================================
 * Name - reportErr
 *
 * Purpose - Report errors, esp. when reading key binding file
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void reportErr(msg)
char msg[];
{
if (theFileName != NULL)
  printf("%s, line %d: ", theFileName, CurrentLineNumber);
printf("%s\r\n", msg);
fflush(stdout);
}

/* =======================================================================
 * Name - set_up_keymap
 *
 * Purpose - Install default key bindings
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
int set_up_keymap()
{
int x,y;
char buf[4];
EditStrType junkStr;

buf[1]=NULL;

for (y = 0; y < MAXEXTENSIONS; y++) {
    for (x = 0; x < MAXEXTENSIONS; x++) {
      TheMetaPrefixes[x][y] = NULL;
      TheMetaMap[x][y] = NULL;
      }
    }

for (y = 0; y < MAXEXTENSIONS; y++) {
  for (x = 0; x < MAX_KBD_INPUT; x++) {
    KbdCallTable[x][y] = BOGUS;
    }
  }

do_bind("accept_forward_char   \"^A\"");
do_bind("previous_line         \"^B\"");
do_bind("insert_interrupt_char \"^C\"");
do_bind("insert_eof_char       \"^D\"");
do_bind("accept_to_end_of_line \"^E\"");
do_bind("next_line             \"^F\"");
do_bind("accept_forward_word   \"^G\"");
do_bind("backward_char         \"^H\"");
do_bind("self_insert           \"^I\"");
do_bind("finish_editing_line   \"^J\"");
do_bind("repeater              \"^K\"");
do_bind("forward_char          \"^L\"");
do_bind("finish_editing_line   \"^M\"");
			      /* ^N */
do_bind("insert_flush_char     \"^O\"");
do_bind("accept_to_eol         \"^P\"");
do_bind("insert_start_char     \"^Q\"");
do_bind("clear_display         \"^R\"");
do_bind("insert_stop_char      \"^S\"");
do_bind("next_pred             \"^T\"");
do_bind("discard_current_edit_line \"^U\"");
do_bind("quote_char            \"^V\"");
do_bind("backspace_word        \"^W\"");
do_bind("delete_char           \"^X\"");
do_bind("previous_pred         \"^Y\"");
do_bind("insert_suspend_char   \"^Z\"");

do_bind("insert_quit_char      \"^\\\"");
do_bind("backspace_char        \"^?\"");
do_bind("BOGUS                 \"^^\"");
do_bind("BOGUS                 \"^_\"");
do_bind("file_completion       \"^]\"");
do_bind("self_insert           \" \""); /* space */

do_bind("self_insert           \"\\!\"");
do_bind("self_insert           \"\\\"\""); /* double quote */
for (x = '#'; x < '('; x++) {
  sprintf(junkStr, "self_insert        \"%c\"", x);
  do_bind(junkStr);
  }
do_bind("open_paren            \"(\"");
do_bind("close_paren           \")\"");
for (x = '*'; x < '\\'; x++) {
  sprintf(junkStr, "self_insert        \"%c\"", x);
  do_bind(junkStr);
  }
do_bind("self_insert           \"\\\\\"");  /* \ -- need 4 \'s! */
do_bind("self_insert           \"]\"");
do_bind("self_insert           \"\\^\"");  /* literal carat */
for (x = '_'; x <= '~'; x++) {
  sprintf(junkStr, "self_insert        \"%c\"", x);
  do_bind(junkStr);
  }

do_bind("delete_char_or_insert_eof       \"^[^X\"");	/* Dejan, see ^X */
do_bind("command_completion    \"^[^[\"");
do_bind("backward_paren        \"^[(\"");
do_bind("forward_paren         \"^[)\"");
do_bind("beginning_of_line     \"^[\\^\"");
do_bind("ul_to_dash_word       \"^[-\"");
do_bind("dash_to_ul_word       \"^[_\"");
do_bind("show_bindings         \"^[?\"");
do_bind("end_of_line           \"^[$\"");
do_bind("increment_factor      \"^[*\"");
do_bind("show_mark             \"^['\"");

do_bind("toggle_add_space_mode      \"^[A\"");
do_bind("show_arguments             \"^[C\"");
do_bind("discard_rest_of_line       \"^[D\"");
do_bind("toggle_eol_longer_mode     \"^[E\"");
do_bind("toggle_include_stuff_mode  \"^[I\"");
do_bind("toggle_lisp_mode           \"^[L\"");
do_bind("run_mesg                   \"^[M\"");
do_bind("toggle_nl_truncate_mode    \"^[N\"");
do_bind("toggle_only_at_eol_mode    \"^[O\"");
do_bind("toggle_pred_mode           \"^[P\"");
do_bind("toggle_record_all_mode     \"^[R\"");
do_bind("toggle_show_eol_mode       \"^[S\"");
do_bind("toggle_wrap_mode           \"^[W\"");

			      /* ^[a */
do_bind("backward_word         \"^[b\"");
do_bind("capitalize_word       \"^[c\"");
do_bind("delete_word           \"^[d\"");
			      /* ^[e */	
			      /* ^[f */
do_bind("prime_from_file       \"^[g\"");
do_bind("show_termcap_info     \"^[h\"");
do_bind("show_system_toggles   \"^[i\"");
do_bind("input_key_bindings    \"^[j\"");
do_bind("delete_region_to_killbuffer \"^[k\"");
do_bind("lowercase_word        \"^[l\"");
do_bind("set_mark              \"^[m\"");
do_bind("new_working_directory \"^[n\"");

#if 0
do_bind("run_pp                \"^[o\"");
#endif

do_bind("yank_from_kill_buffer \"^[p\"");
do_bind("show_used_nodes       \"^[q\"");
do_bind("run_tty_program       \"^[r\"");
do_bind("exchange_mark_and_set \"^[s\"");
do_bind("run_talk              \"^[t\"");
do_bind("uppercase_word        \"^[u\"");
do_bind("show_version          \"^[v\"");
do_bind("forward_word          \"^[w\"");
do_bind("twiddle_chars         \"^[x\"");
do_bind("run_write             \"^[y\"");
do_bind("run_ruptime           \"^[z\"");

/* Termcap bindings... */
#if 0
do_bind("previous_line ku");
do_bind("next_line kd");
do_bind("backward_char kl");
do_bind("forward_char kr");
#endif

initialized = TRUE;
}


/* =======================================================================
 * Name - show_binds
 *
 * Purpose - Actual guts of show_bindings
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int show_binds(fname)
char *fname;
{
int x;
struct list *current;
char *more;
FILE *pipe;

if (fname == NULL)
    if ((more = getenv("PAGER")) == NULL) {
      if ((pipe = popen("more", "w")) == NULL) {
        EditStrType junkStr;

        sprintf(junkStr, "%s: popen(\"more\", \"w\") error", ProgramName);
        perror(junkStr);
        return;
       }
      }
    else {
      if ((pipe = popen(more, "w")) == NULL) {
        EditStrType junkStr;

        sprintf(junkStr, "%s: popen(\"%s\", \"w\") error", ProgramName, more);
        perror(junkStr);
        return;
       }
      }
else
if ((pipe = fopen(fname, "w")) == NULL) {
  EditStrType junkStr;

  sprintf(junkStr, "%s: fopen(\"%s\", \"w\") error", ProgramName, fname);
  perror(junkStr);
  return;
  }



fprintf(pipe, "\n               FUNCTION NAME  BOUND TO \tDESCRIPTION\n");
fprintf(pipe,   "               -------- ----  ----- -- \t-----------\n");

for (x = 0; x < KEYWORDS; x++) {
  if (functions[x].address == BOGUS || functions[x].address == self_insert)
      continue;
  fprintf(pipe, "%28s",functions[x].name);
  current = bindings[x];
  if (current != NULL) {
    EditStrType junkStr;

    sprintf(junkStr, "\"%s\"", *(current->key) ? current->key : "\^\@");
    fprintf(pipe, "%10s\t", junkStr);
    current = current->next;
    }
  else {
    fprintf(pipe,"          \t");
    }
  fprintf(pipe, "%s", functions[x].description);
  while (current != NULL) {
    EditStrType junkStr;

    sprintf(junkStr, "\"%s\"", *(current->key) ? current->key : "\^\@");
    fprintf(pipe, "\n%38s", junkStr);
    current = current->next;
    }
  fprintf(pipe, "\n");
  }
pclose(pipe);
}
