#define FUNCTIONS_C TRUE
/*========================================================================
 *
 * Name - %M%
 *
 * Version:    %I%
 *
 * ccsid:      %W% - %G% %U%
 * from:       %F%
 * date:       %H% %T%
 *
 * %Q%
 *
 * Description: Most line editing functions
 *
 *========================================================================
 */

#include <ctype.h>
#include <stdio.h>

#include "rkglobal.h"
#include "rk.h"
#include "defaults.h"
#include "utl.h"
#include "functions.h"
#include "reactive.h"
#include "kbdbind.h"
#include "display.h"
#include "system.h"
#include "kbdsys.h"
#include "utl.h"

#if defined(FLG_AIX_UNIX)
#include <signal.h>
#define SIGCHLD SIGCLD
#endif

#define NOT_WORD_CHAR(x) (!isalnum(x) && (x) != '-' && (x) != '_')
#define IS_WORD_CHAR(x) (isalnum(x) || (x) == '-' || (x) == '_')

static BoolType get_display_size_left();
static int perform_line_char();
static void perform_word_edit();
static BoolType word_left();
static BoolType word_right();

typedef char EditFuncType();

/* =======================================================================
 * Name - accept_forward_char
 *
 * Purpose - Accept n characters from prediction
 *
 * Arguments:  TheEditData.factor -- number of characters to accept
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int accept_forward_char()
{        /* JJD 9-86 */
int             num_to_go;
BoolType finished = FALSE;
BoolType needs_beep;

if (!pred_mode)
  return forward_char();
if (pred_buff[0] == '\n')
  return finish_editing_line();

num_to_go = strlen(pred_buff);
if (TheEditData.factor > num_to_go)  /* to end of prediction */
  needs_beep = TRUE;
else {
  num_to_go = TheEditData.factor;
  needs_beep = FALSE;
  }
if (!CHECK_INSERT(num_to_go)) {
  bell();
  CLEAR_FACTOR;
  return OK;
  }

/* there is at most 1 NL at the end of the prediction */
if (pred_buff[num_to_go - 1] == '\n') {
  num_to_go--;
  finished = TRUE;
  }
pred_buff[num_to_go] = '\0';
overprint_str(pred_buff, num_to_go); /* overwrite */
edit_insert(pred_buff, num_to_go);
edit_on_display = TRUE;
TheEditData.dot += num_to_go;

if (needs_beep)
  bell();
CLEAR_FACTOR;
if (finished)
  return finish_editing_line();
else
  return OK;
}

/* =======================================================================
 * Name - accept_forward_word
 *
 * Purpose - Accept n words of current prediction
 *
 * Arguments:  TheEditData.factor -- number of words to accept
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int accept_forward_word()
{        /* JJD 9-86 */
char           *ch;
int             num_to_advance;
BoolType needs_beep;
int result;

if (!pred_mode)
  return forward_word();

/* truncate prediction to after first NL */
if (ch = index(pred_buff, '\n'))
    *(++ch) = '\0';

needs_beep = word_right(&pred_buff[0], &num_to_advance, TheEditData.factor);

/* Special case -- if last character is \n, only accept if it's
   also the ONLY character... */
if (num_to_advance > 1 && pred_buff[num_to_advance - 1] == '\n')
  num_to_advance --;

TheEditData.factor = num_to_advance;
result = accept_forward_char();
if (needs_beep)
  bell();
return result;
}

/* =======================================================================
 * Name - accept_to_eol
 *
 * Purpose - Accept all characters displayed in predictions up to first
 *           newline (excluding eol), if any.
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *							Dejan M. Apr 90
 *========================================================================
 */
int accept_to_eol()
{
char *ptr;

if(!pred_mode)
  return end_of_line();

/* truncate prediction on first NL */
if ((ptr = index(pred_buff, '\n')) != NULL)
  *ptr = '\0';

/* only go to end of prediction */
TheEditData.factor = strlen(pred_buff);
return accept_forward_char();
}

/* =======================================================================
 * Name - accept_to_end_of_line
 *
 * Purpose - Accept all characters displayed in prediction, up to and
 *           including the first newline, if any.
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int accept_to_end_of_line()
{        /* JJD 9-86 */
char           *ptr;

if (!pred_mode)
  return end_of_line();

/* truncate prediction to after first NL */
if ((ptr = index(pred_buff, '\n')) != NULL)
  *(++ptr) = '\0';

/* only go to end of prediction */
TheEditData.factor = strlen(pred_buff);
return accept_forward_char();
}

/* =======================================================================
 * Name - backspace_char
 *
 * Purpose - Delete to left of cursor
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int backspace_char()
{
int             num_to_delete;
BoolType needs_beep = FALSE;
int count;

needs_beep = get_display_size_left(TheEditData.factor, &count, &num_to_delete);
TheEditData.dot -= num_to_delete;
move_left(count);
erase_right(count, TheEditData.dot);
if (needs_beep)
  bell();
edit_remove(num_to_delete);
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - backspace_word
 *
 * Purpose - Delete word(s) to left of cursor
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int backspace_word()
{
int             num_to_go;
BoolType  needs_beep;
int result;

needs_beep = word_left(TheEditData.dot, &num_to_go, TheEditData.factor);
TheEditData.factor = num_to_go;
result = backspace_char();
if (needs_beep)
  bell();
return result;
}

/* =======================================================================
 * Name - backward_char
 *
 * Purpose - Move cursor left
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int backward_char()
{
int             num_to_go, count;
BoolType  needs_beep;

needs_beep = get_display_size_left(TheEditData.factor, &count, &num_to_go);
move_left(count);
TheEditData.dot -= num_to_go;
if (needs_beep)
  bell();
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - backward_paren
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int backward_paren()
{
char           *ch, *cb;
int             paren_count = 1;
int             num_to_go = 0;

ch = TheEditData.dot;
cb = TheEditData.current_buffer;

if (ch != cb)    /* not already at the beginning */
  while (--ch >= cb && paren_count != 0) {
    num_to_go++;
    if (*ch == '(')
      paren_count--;  /* a '(' with no intervening * ()'s */
    else
    if (*ch == ')')  /* search for ( matching this * ) */
      while (--ch >= cb && paren_count != 0) {
       num_to_go++;
       if (*ch == '(')
         paren_count--;
       else
        if (*ch == ')')
         paren_count++;
        }
    }

TheEditData.factor = num_to_go;
return backward_char();
}


/* =======================================================================
 * Name - backward_word
 *
 * Purpose - Move left word(s)
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int backward_word()
{
int             num_to_go;
BoolType  needs_beep;
int result;

needs_beep = word_left(TheEditData.dot, &num_to_go, TheEditData.factor);
TheEditData.factor = num_to_go;
result = backward_char();
if (needs_beep)
  bell();
return result;
}

/* =======================================================================
 * Name - beginning_of_line
 *
 * Purpose - Position cursor after prompt
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int beginning_of_line()
{
CLEAR_FACTOR;
move_left(CurrentCursor - PromptStrSize);
TheEditData.dot = &TheEditData.current_buffer[0];
return OK;
}

/* =======================================================================
 * Name - BOGUS
 *
 * Purpose - Null (invalid) key
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int BOGUS()
{
CLEAR_FACTOR;
bell();
return OK;
}


/* =======================================================================
 * Name - capitalize_word
 *
 * Purpose - Capitalize first alphabetic character after each run of
 * non alphabetic characters, and of course the first character
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

static BoolType       subword_flag;

static char word_edit_capitalize(ch)
char ch;
{
if (!isalnum(ch))
  subword_flag = TRUE;
else
if (subword_flag) { /* upper case first letter in word */
  subword_flag = FALSE;
  if (ch >= 'a' && ch <= 'z')
    ch -= 'a' - 'A';
  }
else  /* lowercase subsequent letters */
if (ch >= 'A' && ch <= 'Z')
  ch += 'a' - 'A';
return ch;
}

int capitalize_word()
{
subword_flag = TRUE;
perform_word_edit(word_edit_capitalize);
return OK;
}

/* =======================================================================
 * Name - clear_display
 *
 * Purpose - Blast display massively.  Think Tektronix 19" DVST...
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int clear_display()
{
CLEAR_FACTOR;
erase_current_edit_line();
clear_terminal();
redraw_prompt();
draw_current_edit_line();
return OK;
}

/* =======================================================================
 * Name - close_paren
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int close_paren()
{
char           *old_dot;

if (!lisp_mode)
  return self_insert();

old_dot = TheEditData.dot;
if (*old_dot != ')')  /* if not on a ')', insert one */
  self_insert();

forward_char();  /* skip past ')' so bkd-prn works */
backward_paren();  /* flash back to matching '(' */
flush_termcap();
sleep(1);

TheEditData.factor = 1 + old_dot - TheEditData.dot;
return forward_char();  /* skip ahead to old dot + 1 */
}

/* =======================================================================
 * Name - dash_to_ul_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

static char word_edit_dash_to_ul(ch)
char ch;
{
if (ch == '-')  /* replace hyphens with underscores */
  ch = '_';
return ch;
}

int dash_to_ul_word()
{
perform_word_edit(word_edit_dash_to_ul);
return OK;
}

/* =======================================================================
 * Name - delete_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int delete_char()
{
int             num_to_delete;
char           *c, *new_dot;
BoolType needs_beep = FALSE;

if (TheEditData.factor > strlen(TheEditData.dot)) {
  TheEditData.factor = strlen(TheEditData.dot);
  needs_beep = TRUE;
  }
new_dot = TheEditData.dot + TheEditData.factor;

num_to_delete = 0;
for (c = TheEditData.dot; c < new_dot; ++c)
  num_to_delete += get_char_display_length(*c);
erase_right(num_to_delete, TheEditData.dot);

if (needs_beep)
  bell();
edit_remove(TheEditData.factor);
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - delete_char_or_insert_eof
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *				Dejan Apr 90. 
 *========================================================================
 */
int delete_char_or_insert_eof()
{        /* JJD 9-86 */
if( *(TheEditData.dot) == '\0')  return perform_line_char(LINE_EOF);
else return delete_char();
}

/* =======================================================================
 * Name - delete_region_to_killbuffer
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
delete_region_to_killbuffer()
{
int length;

if (TheEditData.dot > TheEditData.mark) {
  length = TheEditData.dot - TheEditData.mark;
  strncpy(TheEditData.kill_buffer, TheEditData.mark, length);
  TheEditData.kill_buffer[length] = '\0';
  TheEditData.factor = length;
  backward_char();
  TheEditData.factor = length;
  TheEditData.mark = TheEditData.dot;
  return delete_char();
  }
if (TheEditData.mark > TheEditData.dot) {
  length = TheEditData.mark - TheEditData.dot;
  strncpy(TheEditData.kill_buffer, TheEditData.dot, length);
  TheEditData.kill_buffer[length] = '\0';
  TheEditData.factor = length;
  TheEditData.mark = TheEditData.dot;
  return delete_char();
  }
return OK;              /* JJD 3-89 added */
}

/* =======================================================================
 * Name - delete_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int delete_word()
{
int            num_to_advance;
BoolType       needs_beep;
int result;

num_to_advance = 0;
needs_beep = word_right(TheEditData.dot, &num_to_advance, TheEditData.factor);
TheEditData.factor = num_to_advance;
result = delete_char();
if (needs_beep)
  bell();
return result;
}


/* =======================================================================
 * Name - show_arguments
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

int show_arguments()  /* changes Dejan, and by John April 1990 */
{
if (!called_from_cmd_line) {
  erase_current_edit_line();
  printf("\r\n");
  /* printf("Current Command Line Argument Values:\r\n"); */
  }
/* 1. The command args that expect a filename. */
printf("Flag <Argument>            Filename Arguments\r\n");
printf("  -k <Key bindings file>   = \"%s\"\r\n", key_file);
printf("  -p <Prime file>          = \"%s\"\r\n", prime_file);
printf("  -q <(Quaint) Log file>   = \"%s\"\r\n", log_file);
printf("  -z <Zero frequency file> = \"%s\"\r\n", zero_freq_file);

/* 2. The command args that take a numeric argument. */
printf("\r\nFlag <Argument>            Numeric Arguments\r\n");
printf("  -b <Buffers to save>     = %d\r\n", num_buffers);
printf("  -e <End of line length>  = %d\r\n", max_eol);
printf("  -f <Frequency count max> = %d\r\n", max_freq);
printf("  -i <Inline length>       = %d\r\n", max_len);
printf("  -m <Memory maximum>      = %d (Kbytes)\r\n", mem_limit);
printf("  -n <Number prime chars>  = %d (characters)\r\n", maxprime);
printf("  -o <Order of model>      = %d\r\n", maxk);

/* 3. The non-bindable command line booleans (all lowercase). */
printf("\r\nFlag  Description          Lower-case Toggles\r\n");
printf("  -c  Cut prime file.      = %s\r\n", SAY_BOOL(login));
printf("  -d  Default values read. = %s\r\n", SAY_BOOL(login));
printf("  -l  Login shell mode.    = %s\r\n", SAY_BOOL(login));
printf("  -h  Help: prints this.   = %s\r\n", SAY_BOOL(login));
printf("  -s  Silent startup mode. = %s\r\n", SAY_BOOL(silent));
printf("  -u  Underline mode.      = %s\r\n", SAY_BOOL(underlineMode));
printf("  -v  Version is printed.  = %s\r\n", SAY_BOOL(underlineMode));
printf("  -8  8 (eight) bit mode.  = %s\r\n", SAY_BOOL(eight_bit_mode));

/* 4. The command line boolean toggles that are key bindable (UPPERCASE). */
show_system_toggles();
return OK;
}

/* =======================================================================
 * Name - show_system_toggles
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

static void dumpMessage(s1, bool, f)	/* changed by John April 1990 */
char s1[];
BoolType bool;
int (*f)();
{
char *theBind = getFirstBinding(f);

printf("  %-23s  = %-5s         \"%s\"\r\n",
     s1, SAY_BOOL(bool), (*theBind) ? theBind : "**not bound**");
}

int show_system_toggles()  /* New JJD */
{
/* 4. The command line boolean toggles that are key bindable (UPPERCASE). */
printf("\r\nFlag  Toggle_<name>        Upper-case Tog. Key Binding\r\n");
dumpMessage("-A  Add_space_mode.",   add_space_mode, toggle_add_space_mode);
dumpMessage("-E  Eol_longer_mode.",  eol_longer_mode, toggle_eol_longer_mode);
dumpMessage("-I  Include_stuff_mode.", skip_nonword_mode,
				     toggle_include_stuff_mode);
dumpMessage("-L  Lisp_mode.",        lisp_mode, toggle_lisp_mode);
dumpMessage("-N  Nl_truncate_mode.", nl_truncate_mode,toggle_nl_truncate_mode);
dumpMessage("-O  Only_at_eol_mode.", eol_only_mode, toggle_only_at_eol_mode);
dumpMessage("-P  Prediction_mode.",  pred_mode, toggle_pred_mode);
dumpMessage("-R  Record_all_mode.",  log_all_mode, toggle_record_all_mode);
dumpMessage("-S  Show_eol_mode.",    show_eol_mode, toggle_show_eol_mode);
dumpMessage("-W  Wrap_mode.",        wrap_mode, toggle_wrap_mode);

if (!called_from_cmd_line) {
  display_continue();
  CLEAR_FACTOR;
  }
return OK;
}

/* =======================================================================
 * Name - discard_current_edit_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int discard_current_edit_line()
{
strcpy(TheEditData.kill_buffer, TheEditData.current_buffer);
erase_current_edit_line();
ZERO_EDIT_LINE;
return OK;
}


/* =======================================================================
 * Name - discard_rest_of_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int discard_rest_of_line()
{
strcpy(TheEditData.kill_buffer, TheEditData.dot);
TheEditData.factor = strlen(TheEditData.dot);

if (TheEditData.factor == 0) {
  CLEAR_FACTOR;
  return OK;
  }

if (TheEditData.mark > TheEditData.dot)
  TheEditData.mark = TheEditData.dot;
return delete_char();
}

/* =======================================================================
 * Name - end_of_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int end_of_line()
{
CLEAR_FACTOR;
overprint_str(TheEditData.dot, strlen(TheEditData.dot));
TheEditData.dot = &(TheEditData.current_buffer[
    strlen(TheEditData.current_buffer)]);
return OK;
}

/* =======================================================================
 * Name - finish_editing_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int finish_editing_line()
{
char           *old_dot;

CLEAR_FACTOR;
old_dot = TheEditData.dot;
beginning_of_line();
TheEditData.dot = old_dot;  /* necessary to preserve cursor position
                   * across newlines                      */
return FINISHED_EDITING;
}

/* =======================================================================
 * Name - forward_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int forward_char()
{
int             num_to_go;
BoolType needs_beep;

if (TheEditData.factor > strlen(TheEditData.dot)) {
  num_to_go = strlen(TheEditData.dot);
  needs_beep = TRUE;
  }
else {
  num_to_go = TheEditData.factor;
  needs_beep = FALSE;
  }

overprint_str(TheEditData.dot, num_to_go);
TheEditData.dot += num_to_go;

if (needs_beep)
  bell();
edit_on_display = TRUE;
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - get_display_size_left
 *
 * Purpose - Return display size for moving n characters left from current
 *         position (TheEditData.dot)
 *
 * Arguments:  n -- number of characters to move left
 *
 * Returns     function return -- BoolType TRUE if beyond end of buffer
 *             result -- Number of display positions
 *             num_to_go -- Number of character positions (might be
 *                  less than n)
 *
 *========================================================================
 */
static BoolType get_display_size_left(n, result, num_to_go)
int n;
int *result;
int *num_to_go;
{
int count = n;
char *cptr;
BoolType needs_beep = FALSE;

*result = 0;
*num_to_go = 0;
cptr = TheEditData.dot - 1;
if (count > 0 && cptr < &TheEditData.current_buffer[0])
  return TRUE;
while (count > 0) {
  (*num_to_go) ++;
  *result += get_char_display_length(*cptr);
  if (cptr -- == &TheEditData.current_buffer[0]) {
    needs_beep = count > 1;
    break;
    }
  count --;
  }
return needs_beep;
}

/* =======================================================================
 * Name - forward_paren
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int forward_paren()
{
char           *ch = TheEditData.dot - 1;  /* pretend to go back one */
int             paren_count = 1;
int             num_to_advance = 0;

while (*++ch != '\0' && paren_count != 0) {
  num_to_advance++;
  if (*ch == ')')
    paren_count--;  /* a ')' with no intervening ()'s */
  else
  if (*ch == '(')  /* search for ) matching this ( */
    while (*++ch != '\0' && paren_count != 0) {
      num_to_advance++;
      if (*ch == ')')
       paren_count--;
      else
      if (*ch == '(')
       paren_count++;
      }
  }

TheEditData.factor = num_to_advance;
return forward_char();
}

/* =======================================================================
 * Name - forward_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int forward_word()
{
int num_to_advance;
BoolType needs_beep;
int result;

needs_beep = word_right(TheEditData.dot, &num_to_advance, TheEditData.factor);
TheEditData.factor = num_to_advance;
result = forward_char();
if (needs_beep)
  bell();
return result;
}

/* =======================================================================
 * Name - increment_factor
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int increment_factor()
{
if (TheEditData.factor > FACTOR_MAXIMUM)
  bell();
else
  TheEditData.factor *= 4;
return OK;
}

/* =======================================================================
 * Name - insert_eof_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_eof_char()
{        /* JJD 9-86 */
return perform_line_char(LINE_EOF);
}

/* =======================================================================
 * Name - insert_flush_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_flush_char()
{
return perform_line_char(LINE_FLUSH);
}

/* =======================================================================
 * Name - insert_interrupt_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_interrupt_char()
{
return perform_line_char(LINE_INTR);
}

/* =======================================================================
 * Name - insert_quit_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_quit_char()
{
return perform_line_char(LINE_QUIT);
}


/* =======================================================================
 * Name - insert_start_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_start_char()
{        /* JJD 9-86 */
return perform_line_char(LINE_START);
}

/* =======================================================================
 * Name - insert_stop_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_stop_char()
{        /* JJD 9-86 */
return perform_line_char(LINE_STOP);
}

/* =======================================================================
 * Name - insert_suspend_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int insert_suspend_char()
{
return perform_line_char(LINE_SUSPEND);
}

/* =======================================================================
 * Name - lowercase_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

static char word_edit_lowercase(ch)
char ch;
{
if (ch >= 'A' && ch <= 'Z')
  ch += 'a' - 'A';  /* lowercase all uppers */
return ch;
}

int lowercase_word()
{
perform_word_edit(word_edit_lowercase);
return OK;
}

/* =======================================================================
 * Name - new_working_directory
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int new_working_directory()
{
FILE *from, *popen();
char            cmd[MAXPATHLEN];
char tbuf[MAXPATHLEN];
UnixSignalFuncType *sig;

erase_current_edit_line();
kbd_canonical(KEYBOARD);
strcpy(cmd, "/bin/echo ");
printf("\nNew directory ([RETURN] to cancel): ");
fflush(stdout);
gets(&cmd[10]);
if (cmd[10] == '\0') {
  printf("new_working_directory cancelled.\r\n");
  kbd_raw(KEYBOARD);
  }
else {
  kbd_raw(KEYBOARD);
  sig=signal(SIGCHLD, (UnixSignalFuncType *)SIG_DFL);
  from = popen(cmd, "r");
  if (from == NULL) {
    EditStrType junk;

    sprintf(junk, "%s popen() error", ProgramName);
    perror(junk);
    }
  else {
    fgets(tbuf, sizeof(tbuf) - 1, from);
    pclose(from);
    if (tbuf[0] != '\0')
      tbuf[strlen(tbuf) - 1] = '\0'; /* remove trailing newline */
    if (chdir(tbuf) != 0) {
      EditStrType junk;

      sprintf(junk, "%s chdir() error", ProgramName);
      perror(junk);
      }
    getwd(tbuf);
    printf("current directory is now %s", tbuf);
    }
  }
fflush(stdout);
signal(SIGCHLD, sig);
display_continue();
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - next_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int next_line()
{
int             how_many;

how_many = TheEditData.factor;

erase_current_edit_line();
/* Definitely do not call ZERO_EDIT_LINE here */
for (; how_many != 0; --how_many)
  TheEditData.current_ed_buff_ptr = TheEditData.current_ed_buff_ptr->next_ptr;
TheEditData.current_buffer = TheEditData.current_ed_buff_ptr->string;
TheEditData.dot = TheEditData.current_ed_buff_ptr->dot;

TheEditData.mark = TheEditData.current_ed_buff_ptr->mark;

draw_current_edit_line();
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - open_paren
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int open_paren()
{
char           *old_dot;

self_insert();    /* stick in an open_paren */
if (!lisp_mode)
  return OK;

old_dot = TheEditData.dot;
if (*old_dot == ')' || *old_dot == 0) {
  TheEditData.current_input_char = ')';  /* if on a ')' or at the end */
  self_insert();  /* then insert matching ')' */
  backward_char();  /* and go back inside () */
  }
return OK;
}

/* =======================================================================
 * Name - perform_line_char
 *
 * Purpose -
 *
 * Arguments:  what -- line char to do
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
static int perform_line_char(what)
LineCharEType what;
{
int c = get_line_char(pty_master, what);

if (c < 0)
  return BOGUS();

erase_current_edit_line();
ZERO_EDIT_LINE;
*(TheEditData.dot++) = c;
*(TheEditData.dot) = '\0';
switch (what) {
case LINE_START:
  outputEnabled = TRUE;
  break;

case LINE_STOP:
  outputEnabled = FALSE;
  break;

default:
  break;
}
return FINISHED_BUT_DONT_ADD_CTRL_M;
}

/* =======================================================================
 * Name - perform_word_edit -- compress magic to edit a word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
static void perform_word_edit(f)
EditFuncType * f;
{
char           *ch, *cb;
int toBeginning, wholeWord, backToDot;
char * startWord;

/* get beginning of current word */
cb = &TheEditData.current_buffer[0];
ch = TheEditData.dot;
toBeginning = 0;
if (ch > cb) {
  while (ch >= cb && NOT_WORD_CHAR(*ch)) {/* search backwards until in a word*/
    ch--;
    toBeginning += get_char_display_length(*ch);
    }
  while (ch >= cb && IS_WORD_CHAR(*ch)) { /* find first letter in word */
    ch--;
    toBeginning += get_char_display_length(*ch);
    }
  toBeginning -= get_char_display_length(*ch);
  ch ++; /* now pointing at first character */
  }
move_left(toBeginning);

startWord = ch;
wholeWord = 0;
backToDot = 0;
do {
  *ch = (*f)(*ch);
  if (ch == TheEditData.dot)
    backToDot = get_char_display_length(*ch);
  else
    backToDot += get_char_display_length(*ch);
  ch ++;
  wholeWord ++;
  } while (IS_WORD_CHAR(*ch));
if (ch <= TheEditData.dot) { /* special case -- at end of word */
  backToDot = 0;
  while (ch < TheEditData.dot) { /* just overprint to get back to mark */
    ch ++;
    wholeWord ++;
    }
  }

overprint_str(startWord, wholeWord);
move_left(backToDot);
CLEAR_FACTOR;
}

/* =======================================================================
 * Name - previous_line
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int previous_line()
{
int             how_many;

how_many = TheEditData.factor;
erase_current_edit_line();
/* Definitely do not call ZERO_EDIT_LINE here */
for (; how_many != 0; --how_many)
  TheEditData.current_ed_buff_ptr = TheEditData.current_ed_buff_ptr->prev_ptr;
TheEditData.current_buffer = TheEditData.current_ed_buff_ptr->string;
TheEditData.dot = TheEditData.current_ed_buff_ptr->dot;
TheEditData.mark = TheEditData.current_ed_buff_ptr->mark;
draw_current_edit_line();
CLEAR_FACTOR;
return OK;
}


/* =======================================================================
 * Name - quote_char
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int quote_char()
{
char            quoted_char;

read_kbd(&quoted_char, 1);
if (!eight_bit_mode)
  quoted_char = STRIP_PARITY(quoted_char);
if (quoted_char == '\0') {
  bell();
  return OK;
  }
TheEditData.current_input_char = quoted_char;
return self_insert();
}

/* =======================================================================
 * Name - run_mesg
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int run_mesg()
{
EditStrType cmd;
char *on_or_off;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
strcpy(cmd, "mesg ");
on_or_off = &cmd[5];
printf("\nDo you want \"mesg y\" or \"mesg n\" ");
printf("(type y or n and press [RETURN]): ");
fflush(stdout);
gets(on_or_off);
if (strncmp(on_or_off, "on", 2) == 0)
  strcpy(on_or_off, "y");
else
if (strncmp(on_or_off, "off", 3) == 0)
  strcpy(on_or_off, "n");
else
if (*on_or_off == 'y' || *on_or_off == 'n')
  on_or_off[1] = '\0';
else
if (on_or_off[0] == '\0') {
  printf("Mesg cancelled.\r\n");
  kbd_raw(KEYBOARD);
  display_continue();
  return OK;
  }
else {
  printf("Mesg cancelled because you didn't respond with y or n.\r\n");
  kbd_raw(KEYBOARD);
  display_continue();
  return OK;
  }

kbd_raw(KEYBOARD);
run_program_connected_to_standard_tty(cmd);
return OK;
}

#if 0
int run_pp()
{        /* JJD 9-86 */
EditStrType            cmd;
char *who_to;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
strcpy(cmd, "pp ");
who_to = &cmd[3];
printf("\npp what file ([RETURN] to cancel): ");
fflush(stdout);
gets(who_to);
if (who_to[0] == '\0') {
  printf("pp cancelled.\r\n");
  display_continue();
  kbd_raw(KEYBOARD);
  }
else {
  kbd_raw(KEYBOARD);
  run_program_connected_to_standard_tty(cmd);
  }
return OK;
}
#endif

/* =======================================================================
 * Name - run_ruptime
 *
 * Purpose - Run the "uptime" command
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int run_ruptime()
{
erase_current_edit_line();
write_display("\r\n", 2);          /* JJD 3-89*/
quietly_run_program_connected_to_standard_tty ("uptime"); /* JJD 3-89*/
display_continue();
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - run_talk
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int run_talk()
{
EditStrType cmd;
char *who_to;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
strcpy(cmd, "talk ");
who_to = &cmd[5];
printf("\nTalk to whom ([RETURN] to cancel): ");
fflush(stdout);
gets(who_to);
if (who_to[0] == '\0') {
  printf("Talk cancelled.\r\n");
  kbd_raw(KEYBOARD);
  display_continue();
  }
else {
  kbd_raw(KEYBOARD);
  run_program_connected_to_standard_tty(cmd);
  /* NO display_continue() */
  }

return OK;
}

/* =======================================================================
 * Name - run_tty_program
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int run_tty_program()
{
EditStrType cmd;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
strcpy(cmd, "");
printf("\nCommand to run ([RETURN] to cancel): ");
fflush(stdout);
gets(cmd);
if (cmd[0] == '\0') {
  printf("Command cancelled.\r\n");
  kbd_raw(KEYBOARD);
  display_continue();
  }
else {
  kbd_raw(KEYBOARD);
  run_program_connected_to_standard_tty(cmd);
/* NO display_continue() */
  }

return OK;
}

/* =======================================================================
 * Name - run_write
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int run_write()
{
EditStrType  cmd;
char *who_to;

erase_current_edit_line();
/* why ZERO_EDIT_LINE; ? */
kbd_canonical(KEYBOARD);
strcpy(cmd, "write ");
who_to = &cmd[6];
printf("\nwrite to whom ([RETURN] to cancel): ");
fflush(stdout);
gets(who_to);
if (who_to[0] == '\0') {
  printf("Write cancelled.\r\n");
  kbd_raw(KEYBOARD);
  display_continue();
  }
else {
  kbd_raw(KEYBOARD);
  run_program_connected_to_standard_tty(cmd);
  }

return OK;
}

/* =======================================================================
 * Name - self_insert
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int self_insert()
{
EditStrType    junk;

memset(junk, TheEditData.current_input_char, TheEditData.factor);
if (!CHECK_INSERT(TheEditData.factor))
  bell();
else {
  insert_str(junk, TheEditData.factor, TheEditData.dot);
  edit_insert(junk, TheEditData.factor);
  TheEditData.dot += TheEditData.factor;
  edit_on_display = TRUE;
  }
CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - set_mark
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
set_mark()
{
  TheEditData.mark = TheEditData.dot;
           /* JJD 3-89 mark intialized in set_up_buffers()*/
  TheEditData.current_ed_buff_ptr->mark = TheEditData.dot; /* JJD 3-89 added */
  CLEAR_FACTOR;
}

/* =======================================================================
 * Name - exchange_mark_and_set
 *
 * Purpose - reposition cursor to mark and then set mark to old cursor pos.
 *
 * Arguments:
 *
 * Returns    function return -- Keyboard status
 *
 *						Dejan M. Apr 90
 *========================================================================
 */
exchange_mark_and_set()
{
char *pom;

pom = TheEditData.dot;
if(TheEditData.mark > TheEditData.dot) {
    TheEditData.factor = TheEditData.mark - TheEditData.dot;
    forward_char();
    flush_termcap();
    }
else if(TheEditData.mark < TheEditData.dot) {
    TheEditData.factor = TheEditData.dot - TheEditData.mark;
    backward_char();
    flush_termcap();
    }
TheEditData.dot = TheEditData.mark;
TheEditData.mark = pom;
}

/* =======================================================================
 * Name - show_mark
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 * Idea: You could use factor to adjust number of seconds it sleeps...
 *
 *========================================================================
 */
show_mark()
{
int num_to_go;

if (TheEditData.mark > TheEditData.dot) {
  TheEditData.factor = num_to_go = TheEditData.mark - TheEditData.dot;
  forward_char();
  flush_termcap();
  sleep(1);            /* JJD 3-89 change */
  TheEditData.factor = num_to_go;
  backward_char();
  }
else
if (TheEditData.mark < TheEditData.dot) {
  TheEditData.factor = num_to_go = TheEditData.dot - TheEditData.mark;
  backward_char();
  flush_termcap();
  sleep(1);            /* JJD 3-89 change */
  TheEditData.factor = num_to_go;
  forward_char();
  }
}

/* =======================================================================
 * Name - show_version
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int show_version()
{
if (!called_from_cmd_line) {
  erase_current_edit_line();
  printf("\r\n");
  }
printf("%s Version: %s\n", ProgramName, RK_VERSION);
if (!called_from_cmd_line) {
  display_continue();
  CLEAR_FACTOR;
  }
return OK;
}


/* =======================================================================
 * Name - toggle_add_space_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_add_space_mode()
{
add_space_mode = !add_space_mode;
return OK;
}

/* =======================================================================
 * Name - toggle_eol_longer_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_eol_longer_mode()
{        /* JJD 9-86 */
eol_longer_mode = !eol_longer_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_include_stuff_mode
 *
 * Purpose -
 * 
 * Arguments:
 *
 * Returns     function returns -- Keyboard status
 *                                                  Dejan Apr 90.
 *=========================================================================
 */
int toggle_include_stuff_mode()
{
skip_nonword_mode = ! skip_nonword_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_lisp_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_lisp_mode()
{
lisp_mode = !lisp_mode;
return OK;
}

/* =======================================================================
 * Name - toggle_nl_truncate_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_nl_truncate_mode()
{        /* JJD 9-86 */
nl_truncate_mode = !nl_truncate_mode;
if(!nl_truncate_mode && show_eol_mode) show_eol_mode = FALSE;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_only_at_eol_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_only_at_eol_mode()
{        /* JJD 9-86 */
eol_only_mode = !eol_only_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_pred_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_pred_mode()
{        /* JJD 9-86 */
pred_mode = !pred_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_record_all_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status    Dejan M. March 1990
 *
 *========================================================================
 */
int toggle_record_all_mode()
{
log_all_mode = !log_all_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_show_eol_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int toggle_show_eol_mode()
{
show_eol_mode = !show_eol_mode;
if(!nl_truncate_mode && show_eol_mode) show_eol_mode = FALSE;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - toggle_wrap_mode
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     functions return -- Keyboard status      Dejan 1990
 *
 *=======================================================================
 */
int toggle_wrap_mode()
{
wrap_mode = !wrap_mode;
make_a_prediction();
return OK;
}

/* =======================================================================
 * Name - twiddle_chars
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int twiddle_chars()
{
char            temp_char;
int             num_to_back_up;

#if 0
/* this seems to literally insert a ^T if twiddle_chars is
   invoked at beginning of line.  Not necessary if literal_next
   function is available.  Also anomalous if twiddle_next has
   been remapped to something besides ^T.
*/
if (strlen(TheEditData.current_buffer) == 0) {
  *TheEditData.dot++ = '\024';
  *TheEditData.dot = '\0';
  return FINISHED_BUT_DONT_ADD_CTRL_M;
  }
else
#endif

if ((TheEditData.dot - TheEditData.current_buffer) < 2)
  bell();
else {
  temp_char = *(TheEditData.dot - 1);
  *(TheEditData.dot - 1) = *(TheEditData.dot - 2);
  *(TheEditData.dot - 2) = temp_char;
  num_to_back_up = get_char_display_length(*(TheEditData.dot - 2));
  num_to_back_up += get_char_display_length(*(TheEditData.dot - 1));
  move_left(num_to_back_up);
  overprint_chr(*(TheEditData.dot - 2));
  overprint_chr(*(TheEditData.dot - 1));
  }

CLEAR_FACTOR;
return OK;
}

/* =======================================================================
 * Name - ul_to_dash_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
static char word_edit_ul_to_dash(ch)
char ch;
{
if (ch == '_')  /* replace underscores with hyphens */
  ch = '-';
return ch;
}

int ul_to_dash_word()
{
perform_word_edit(word_edit_ul_to_dash);
return OK;
}

/* =======================================================================
 * Name - uppercase_word
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */

static char word_edit_uppercase(ch)
char ch;
{
if (ch >= 'a' && ch <= 'z')
  ch -= 'a' - 'A';  /* uppercase all lowers */
return ch;
}

int uppercase_word()
{
perform_word_edit(word_edit_uppercase);
return OK;
}

/* =======================================================================
 * Name - word_left
 *
 * Purpose - Scan left in the edit buffer.
 *
 * Arguments:  s -- start point in TheEditData.current_buffer before which
 *                  to scan
 *             n -- how many words left to go
 *
 * Returns     function return -- TRUE if out of range
 *             num_to_backup -- number to back up
 *                                        Dejan
 *========================================================================
 */
static BoolType word_left(s, num_to_backup, n)
char *s;
int *num_to_backup;
int n;
{
BoolType needs_beep = FALSE;
int i;
char * cb = &TheEditData.current_buffer[0];
char *ptr;

ptr = &s[0];
*num_to_backup = 0;

for (i = n; i > 0; i --) {
  if (ptr == cb) {/* at beginning? */
    needs_beep = TRUE;
    break;
    }

  if (skip_nonword_mode) {               /* Dejan */
    while (ptr > cb && NOT_WORD_CHAR(*(ptr - 1))) { /* skip trailing spaces */
      ptr --;
      (*num_to_backup) ++;
      }
    if (ptr == cb) {
      needs_beep = i > 1;
      break;
      }
    }

 /*  if (NOT_WORD_CHAR(*(ptr - 1))) {
    ptr --;
    (*num_to_backup) ++;
    }
  else */
    { /* skip the word */
    while (ptr > cb && IS_WORD_CHAR(*(ptr - 1))) {
      ptr --;
      (*num_to_backup) ++;
      }
    }
/* Now pointing at first character of skipped word */

  if (ptr == cb) {
    needs_beep = i > 1;
    break;
    }
/* lets try this without the extra spaces
  if (add_space_mode && isspace(*(ptr - 1))) {
    while (ptr > cb && isspace(*(ptr - 1))) {
      ptr --;
      (*num_to_backup) ++;
      }
    }
    */
  }
return needs_beep;
}

/* =======================================================================
 * Name - word_right
 *
 * Purpose - Scan right in given string.  Return size of word,
 *           and pointer to first character following word
 *
 * Arguments:  s -- string to scan through
 *             n -- number of words
 *
 * Returns     function return -- TRUE if out of range
 *             num_to_advance -- number of characters to move
 *
 *========================================================================
 */
static BoolType word_right(s, num_to_advance, n)
char s[];
int * num_to_advance;
int n;
{
BoolType needs_beep = FALSE;
int i;
char * ptr;

ptr = &s[0];
*num_to_advance = 0;
for (i = n; i > 0; i --) {
  if (*ptr == '\0') {/* return on empty string */
    needs_beep = TRUE;
    break;
    }

  /* Skip beginning space */
  if (skip_nonword_mode) {
/*** change Dejan Apr 90.  ***/
    while (*ptr != '\0' && NOT_WORD_CHAR(*ptr)) {
      (*num_to_advance) ++;
      ptr++;
      }
    if (*ptr == '\0') {
      needs_beep = i > 1;
      break;
      }
    }

  if (NOT_WORD_CHAR(*ptr)) {
    /* Just return single strange punctuation */
    (*num_to_advance) ++;
    ptr ++;
    }
  else { /* skip the word */
    while (*ptr != '\0' && IS_WORD_CHAR(*ptr)) {
      (*num_to_advance) ++;
      ptr++;
      }
    }
  /* Now pointing to first character following skipped word */

  if (add_space_mode) { /* skip trailing space */
    while (*ptr != '\0' && isspace(*ptr)) {
      (*num_to_advance) ++;
      ptr++;
      }
    }
  }
return needs_beep;
}

/* =======================================================================
 * Name - yank_from_kill_buffer
 *
 * Purpose -
 *
 * Arguments:
 *
 * Returns     function return -- Keyboard status
 *
 *========================================================================
 */
int yank_from_kill_buffer()
{
int bufLen = strlen(TheEditData.kill_buffer);

if (!CHECK_INSERT(bufLen))
  bell();
else {
  insert_str(TheEditData.kill_buffer, bufLen, TheEditData.dot);
  edit_insert(TheEditData.kill_buffer, bufLen);
  TheEditData.dot = TheEditData.dot + bufLen;
  }
CLEAR_FACTOR;
return OK;
}
