#define DISPLAY_C TRUE
/*========================================================================
 *
 * Name - %M%
 *
 * Version:    %I%
 *
 * ccsid:      %W% - %G% %U%
 * from:       %F%
 * date:       %H% %T%
 *
 * %Q%
 *
 * Description: Implement display access to the edit line.
 *
 * RK 1.0 had a charming and naive approach to the edit line.  It used
 * 1-dimensional cursor-relative motion to effect overwrites, insertions,
 * and deletions.
 *
 * This behaves badly in the event of line-wrap.  This happens
 * most often with a prediction.  Since predictions are potentially
 * shown at the end of EVERY line, this is more serious than you might
 * first expect.
 *
 * This module provides an abstraction for access to the current edit
 * line.  It can be erased, overwritten, inserted, and deleted.
 * In order to maintain cursor wrap information, output to the display
 * from user programs must also be tracked here.
 *
 * The line editor functions still regard the edit line as being one-dimensional
 * -- this greatly simplifies the writing of that layer.
 *
 * Input is usually in terms of regular characters, which may have to be
 * expanded to 2 (or even more?) "display positions".  RK 1.0 always updated
 * the display in terms of the semantic form (i.e., the edit buffers).
 * This has been retained here.
 *
 * This has the advantage that display errors versus semantic errors are
 * kept distinct.  In particular, clear_screen does a VERY good job of
 * printing a fresh, clean line representing the current prompt, user
 * command, prediction, and current cursor position.
 *
 *========================================================================
 */

#include <stdio.h>

#include "rkglobal.h"
#include "rk.h"
#include "defaults.h"
#include "display.h"
#include "system.h"
#include "utl.h"

#define TERMCAP_SLOP 300  /* free space lower bound causes flush */

static char tc_ent[1024];
static char tc_seq_buf[1024];
static char *tc_ptr;
static char output_string[1024];
static int  output_string_length;
static char *cursor_left, *cursor_right;
static char *cursor_down, *cursor_up;
static char *audible_bell;
static char *clear_to_eol;
static char *clear_screen;
static char *enter_insert_mode, *exit_insert_mode;
static char *pre_insert_char, *post_insert_char;
static char *enter_delete_mode, *exit_delete_mode, *delete_a_char;
static char *enter_standout_mode, *exit_standout_mode; /* JJD 9-86 */
static char *pred_enter_standout_mode, *pred_exit_standout_mode;
static char *enter_ul_mode, *exit_ul_mode;
static char *cursor_left_margin;
static char *termname;
static char *multi_delete_char;
static BoolType automatic_margins;

static BoolType usingStandout; /* true if termcap currently enabled */

#ifdef TIOCGWINSZ
  struct winsize ptyWin;
#endif

/* =======================================================================
 * Name - ADD_TC_NUM
 *
 * Purpose - Add a termcap string with one argument to the output buffer
 *
 * Arguments:  cptr -- termcap string to add
 *
 * Returns     function return --
 *
 * The use of tgoto() here is somewhat suspicious, but it seems to work
 * with all of the curses libraries I've tried.  There's got to be a
 * legit way...
 *
 *========================================================================
 */

#define ADD_TC_NUM(cptr, n) { \
  char *tc_x; \
  if (output_string_length >= TERMCAP_SLOP) \
    FLUSH_TERMCAP; \
  if ((cptr) != NULL) {\
    tc_x = tgoto((cptr), (n), (n)); \
    if (tc_x != NULL) {\
      int tc_len = strlen(tc_x); \
      memcpy(&output_string[output_string_length], tc_x, tc_len); \
      output_string_length += tc_len; \
      } \
    } \
  }

/* =======================================================================
 * Name - ADD_TC_STR
 *
 * Purpose - Add a termcap string to the output buffer
 *
 * Arguments:  cptr -- termcap string to add
 *
 * Returns     function return --
 *
 *========================================================================
 */
#define ADD_TC_STR(cptr) { \
  if (output_string_length >= TERMCAP_SLOP) \
    FLUSH_TERMCAP; \
  if ((cptr) != NULL) \
    tputs((cptr), 1, append_to_output_string); \
  }

/* =======================================================================
 * Name - ADD_CHAR
 *
 * Purpose - Add a single character to the output buffer
 *
 * Arguments:  x -- char to append to output stream
 *
 * Returns     function return --
 *
 *========================================================================
 */
#define ADD_CHAR(x) { \
  if (output_string_length >= TERMCAP_SLOP) \
    FLUSH_TERMCAP; \
  output_string[output_string_length ++] = (x); \
  }

/* =======================================================================
 * Name - FLUSH_TERMCAP
 *
 * Purpose - Flush termcap -- here for efficiency
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
#define FLUSH_TERMCAP { \
  if (output_string_length > 0) \
    write_display(output_string, output_string_length); \
  output_string_length = 0; \
  }

/* =======================================================================
 * Name - TO_NEXT_LINE
 *
 * Purpose - Reposition cursor at end of display line condition
 *           Avoid last column on display due to weirdness on
 *           different terminals
 *
 * Arguments:
 *
 * Returns     function return --
 *
 * A note about the double-space with automatic_margin: vt100's need
 * one character to properly position cursor, and Heath-19's need 2!
 *
 *========================================================================
 */
#define TO_NEXT_LINE { \
  if (CurrentCursor != 0 && CurrentCursor % USED_COLUMNS == 0) { \
    if (usingStandout) \
      ADD_TC_STR(pred_exit_standout_mode); \
    if (automatic_margins) { \
      ADD_TC_STR("  "); /* make doubly sure  cursor is on next line */ \
      ADD_TC_STR(cursor_left_margin); /* and move to left margin */ \
      } \
    else { \
      ADD_TC_STR(cursor_left_margin); \
      ADD_TC_STR(cursor_down); \
      } \
    if (usingStandout) \
      ADD_TC_STR(pred_enter_standout_mode); \
    } \
  }

/* Static routines defined in this module */

static int append_to_output_string();
static void move_horizontal();
static void move_vertical();
static void simple_erase();
static int simple_insert();
static void stupid_erase();
static int stupid_insert();

/* =======================================================================
 * Name - append_to_output_string
 *
 * Purpose - Append function passed to curses routines (internal)
 *
 * Arguments:  c -- char to append to output stream
 *
 * Returns     function return --
 *
 *========================================================================
 */
static int append_to_output_string(c)
char            c;
{
ADD_CHAR(c);
}

/* =======================================================================
 * Name - bell
 *
 * Purpose - Sound the bell
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void bell()
/* Sound the bell */
{
ADD_TC_STR(audible_bell);
}

/* =======================================================================
 * Name - clear_terminal
 *
 * Purpose - Clear the terminal, make sure cursor is in first column.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void clear_terminal()
{
int i;

if (clear_screen != NULL) {
  ADD_TC_STR(clear_screen);
  }
else
  for (i = displayRows; i > 0; i --)
    ADD_TC_STR("\r\n");
FLUSH_TERMCAP;
}

/* =======================================================================
 * Name - display_continue
 *
 * Purpose - Refresh display after RK message
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
#define CONTINUE_MESSAGE   "\r\n[%s: operation complete...]\r\n"

void display_continue()
{
EditStrType junk;

if (PromptStrSize == 0) {
  sprintf(junk, CONTINUE_MESSAGE, ProgramName);
  write(DISPLAY, junk, strlen(junk));
  }
else
  write(DISPLAY, "\r\n", 2);
edit_on_display = FALSE;
CurrentCursor = 0;
redraw_prompt();
draw_current_edit_line();
}

/* =======================================================================
 * Name - display_pred_buffer
 *
 * Purpose - Make prediction appear, starting at current cursor
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void display_pred_buffer()
{        /* JJD 9-86 ???? */
int             count;

if (!pred_mode || pred_on_display || pred_buff[0] == '\0')
  return;
if (eol_only_mode && *TheEditData.dot != '\0')
  return;
ADD_TC_STR(pred_enter_standout_mode);
usingStandout = TRUE;
/* simulated insert might go in standout if insert_str is used */
/* count = insert_str(pred_buff, strlen(pred_buff), whatsLeft); */
count = overprint_str(pred_buff, strlen(pred_buff));
ADD_TC_STR(pred_exit_standout_mode);
usingStandout = FALSE;
count += overprint_str(TheEditData.dot, strlen(TheEditData.dot));
move_left(count);

FLUSH_TERMCAP;
pred_on_display = TRUE;
}

/* =======================================================================
 * Name - draw_current_edit_line
 *
 * Purpose - Make the user's accepted characters appear on display.
 *           Assumes cursor is currently located after the prompt.
 *           Leaves cursor positioned appropriately for the user.
 *
 *           Will not ensure that prediction is displayed.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void draw_current_edit_line()
/* Make the user's accepted characters appear on display */
{
int count;

if (edit_on_display)
  return;
overprint_str(TheEditData.current_buffer,
    TheEditData.dot - &TheEditData.current_buffer[0]);
count = overprint_str(TheEditData.dot, strlen(TheEditData.dot));
move_left(count);

edit_on_display = TRUE;
FLUSH_TERMCAP;
}

/* =======================================================================
 * Name - erase_current_edit_line
 *
 * Purpose - Make the user's current set of accepted characters disappear.
 *           Note: This will make sure the prediction disappears as well.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void erase_current_edit_line()
{
int             num_to_erase;
char           *old_dot;

/*
 * added to deal with META-prefixes which turn pred display on...may
 * not work for all cases
 */
if (!edit_on_display)
  return;
erase_pred_buffer();  /* JJD 9-86 */
old_dot = TheEditData.dot;
beginning_of_line();
TheEditData.dot = old_dot;
num_to_erase = get_display_length(TheEditData.current_buffer);
erase_right(num_to_erase, TheEditData.current_buffer);
edit_on_display = FALSE;
FLUSH_TERMCAP;
}

/* =======================================================================
 * Name - erase_pred_buffer
 *
 * Purpose - Make prediction disappear from display.  Assumed to start
 *           at current cursor.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void erase_pred_buffer()
{        /* JJD 9-86 */
int num_to_erase;
char junk[sizeof(EditStrType) + sizeof(pred_buff)]; /* to avoid possible overflow */

if (!pred_on_display)
  return;
ADD_TC_STR(pred_exit_standout_mode);
usingStandout = FALSE;
num_to_erase = get_display_length(pred_buff);
/* Kludge to make sure overprint logic has full picture */
strcpy(junk, pred_buff);
strcat(junk, TheEditData.dot);
/* Do abstract erasure */
erase_right(num_to_erase, junk);

#if 0
/* following more elegant for single-line case... */
count = overprint_str(TheEditData.dot, strlen(TheEditData.dot));
if (num_to_erase > count)
  erase_right(num_to_erase - count, NULL);
move_left(count);
#endif

pred_on_display = FALSE;
FLUSH_TERMCAP;
}

/* =======================================================================
 * Name - erase_right
 *
 * Purpose - Erase display cells to right of current cursor
 *
 * Arguments:  n -- number of display positions to erase
 *             pat -- current contents of display at current cursor, not
 *                 display expanded.  If strlen(pat) < n, the erasure
 *                 is assumed to be "complete", i.e., the rest of the
 *                 logical line.
 *
 *                 This argument is necessary for dumb termcaps, which may
 *                 have to do their work by redrawing the remainder of the
 *                 line.
 *
 * Returns     function return --
 *
 *========================================================================
 */
void erase_right(n, pat)
int n;
char pat[];
{
int patLength = get_display_length(pat);
int patSize = strlen(pat);

if (CurrentCursor / USED_COLUMNS ==
    (CurrentCursor + patLength + 1) / USED_COLUMNS) {
  /* All on same line */
  /* The +1 above is because the mere fact of writing in column 80 means
     that you might be in trouble already. */
  simple_erase(n, pat, patSize);
  return;
  }
/* Modification will affect a subsequent row of the display */
/* do it the dumb way for now.  You could get really tricky here... */

stupid_erase(n, pat, patSize);
}

/* =======================================================================
 * Name - flush_pty
 *
 * Purpose - Flush bytes from pty to the display while keyboard is
 *           in canonical mode.  Capture the pty's prompt and cursor
 *           position to properly do line wrap.
 *
 * Arguments:  buf -- bytes to write to display
 *             num -- number of bytes to write
 *
 * Returns     function return --
 *
 *========================================================================
 */
void flush_pty(buf, num)
char buf[];
int num;
{
  int i;
  char *cptr = &buf[0];
  char *pptr = &PromptStr[PromptStrSize];

  for (i = 0; i < num; i ++, cptr ++)
    /* Idea: any control character except bell resets prompt? */
    if (*cptr == '\n' || *cptr == '\r') { /* newline, resets prompt */
      PromptStrSize = 0;
      pptr = &PromptStr[0];
      CurrentCursor = 0;
      }
    else
    if (*cptr == '\007') { /* non-displaying */
      }
    else
    if (PromptStrSize != 0 &&
        PromptStrSize % USED_COLUMNS == 0) { /* assume wrap */
      pptr = &PromptStr[0];
      PromptStr[0] = *cptr;
      PromptStrSize = 1;
      CurrentCursor = 1;
      }
    else { /* add character to prompt */
      *pptr ++ = *cptr;
      PromptStrSize ++;
      CurrentCursor ++;
      }
  write_display(buf, num);
}

/* =======================================================================
 * Name - flush_termcap
 *
 * Purpose - Flush accumulated termcap commands and zero the length of
 *           the static termcap output buffer
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void flush_termcap()
/* Flush termcap buffer */
{
FLUSH_TERMCAP;
}

/* =======================================================================
 * Name - get_char_display_length
 *
 * Purpose - Return display length of a single character
 *
 * Arguments:  c -- the character to display
 *
 * Returns     function return -- The number of display positions required
 *                  to display this single character
 *
 *========================================================================
 */
int get_char_display_length(c)
/* return length of single character after control expansion */
char            c;
{
int             len;
unsigned char temp = (unsigned char)c;

len = 0;
if (temp >= MAX_SET) {
  len ++;
  temp -= MAX_SET;
  }
if (temp < ' ' || temp == ASCII_RUBOUT)
  len += 2;
else
  len++;
return len;
}

/* =======================================================================
 * Name - get_display_length
 *
 * Purpose - Determine length in display positions of an (unencoded) null-
 *           terminated string.
 *
 * Arguments:  s -- The null-terminated string to encode
 *
 * Returns     function return -- The number of display positions required
 *                  to display s
 *
 *========================================================================
 */
int get_display_length(s)
char           s[];
{
char           *cptr;
int             len;

cptr = &s[0];
len = 0;
while (*cptr != '\0') {
  unsigned char temp = (unsigned char)*cptr;

  if (temp >= MAX_SET) {
    len ++;
    temp -= MAX_SET;
    }
  if (temp < ' ' || temp == ASCII_RUBOUT)
    len += 2;
  else
    len++;
  ++cptr;
  }
return len;
}

/* =======================================================================
 * Name - get_termcap_stuff
 *
 * Purpose - Initialization routine for this module
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
#define PECULIAR "Expect peculiar results"

int get_termcap_stuff()
{
char * junkPtr;
int junkInt;

/* Set current cursor position */
CurrentCursor = 0;
output_string_length = 0;
usingStandout = FALSE;

/* Do the first-level totally desperation initialization */
cursor_left = "\010"; /* ^H */
cursor_right = "\033[C"; /* ANSI */
cursor_down = "\012"; /* ^J */
cursor_up = "\013"; /* ^K */
audible_bell = "\007"; /* ^G */
enter_insert_mode = exit_insert_mode = NULL;
pre_insert_char = post_insert_char = NULL;
enter_delete_mode = exit_delete_mode = NULL;
delete_a_char = NULL;
multi_delete_char = NULL;
clear_screen = NULL; /* we'll fake it */
enter_standout_mode = exit_standout_mode = NULL;
enter_ul_mode = exit_ul_mode = NULL;
cursor_left_margin = "\015"; /* ^M */
automatic_margins = TRUE;
displayRows = 24;
displayColumns = 80;
set_underline_mode();

/* Go after terminfo database entry */
termname = getenv("TERM");
if (termname == NULL)
  return;
tc_ptr = &tc_seq_buf[0];
if (tgetent(tc_ent, termname) != 1)
  return;

/* Now try to use that information to refine our termcap capabilitites */
/* Window Size */
junkInt = tgetnum("li");
if (junkInt != 0)
  displayRows = junkInt;
junkInt = tgetnum("co");
if (junkInt != 0)
  displayColumns = junkInt;

#ifdef TIOCGWINSZ
/* read initial states of user window and set pty window */
ioctl(DISPLAY, TIOCGWINSZ, (int)&ptyWin);
if (ptyWin.ws_row != 0 && ptyWin.ws_col != 0) /* not garbage */ {
  displayRows = ptyWin.ws_row;
  displayColumns = ptyWin.ws_col;
  ioctl(pty_master, TIOCSWINSZ, (int)&ptyWin);
  }
#endif

/* Cursor Motions */
if (!tgetflag("bs")) {
  junkPtr = xlat_termcap("bc", "cursor left", "Assuming ^H");
  if (junkPtr != NULL)
    cursor_left = junkPtr;
  }

junkPtr = xlat_termcap("nd", "cursor right", "Assuming ANSI (^[[C)");
if (junkPtr != NULL)
  cursor_right = junkPtr;

/* Cursor down is not usually included in most termcaps, so we silence
   the warning message */
junkPtr = xlat_termcap("do", "cursor down", (char *)NULL);
if (junkPtr != NULL)
  cursor_down = junkPtr;

junkPtr = xlat_termcap("up", "cursor up", "Assuming ^K");
if (junkPtr != NULL)
  cursor_up = junkPtr;

/* Carriage return is so basic most termcaps don't have this one either */
junkPtr = xlat_termcap("cr", "left margin", (char *)NULL);
if (junkPtr != NULL)
  cursor_left_margin = junkPtr;

/* Bell */
/* Bell is also not included in most termcaps. */
junkPtr = xlat_termcap("bl", "audible bell", (char *)NULL);
if (junkPtr != NULL)
  audible_bell = junkPtr;
else {
  junkPtr = xlat_termcap("vb", "visible bell", (char *)NULL);
  if (junkPtr != NULL)
    audible_bell = junkPtr;
  }

/* Non-critical but useful termcaps */
automatic_margins = tgetflag("am");
clear_to_eol = xlat_termcap("ce", "clear to eol", (char *)NULL);
enter_insert_mode = xlat_termcap("im", "enter insert mode", (char *)NULL);
exit_insert_mode = xlat_termcap("ei", "exit insert mode", (char *)NULL);
pre_insert_char = xlat_termcap("ic", "pre insert char", (char *)NULL);
post_insert_char = xlat_termcap("ip", "post insert char", (char *)NULL);
enter_delete_mode = xlat_termcap("dm", "enter delete mode", (char *)NULL);
exit_delete_mode = xlat_termcap("ed", "exit delete mode", (char *)NULL);
delete_a_char = xlat_termcap("dc", "delete character", (char *)NULL);
multi_delete_char = xlat_termcap("DC", "multi delete char",
    (char *)NULL);
clear_screen = xlat_termcap("cl", "clear screen", (char *)NULL);

/* Special Display Modes */
enter_standout_mode = xlat_termcap("so", "enter standout",
    "Expect no standout");
exit_standout_mode = xlat_termcap("se", "exit standout",
    "Expect no standout");
enter_ul_mode = xlat_termcap("us", "enter underscore",
    "Expect no underlining");
exit_ul_mode = xlat_termcap("ue", "exit underscore", "Expect no underlining");

/* Adjust conflicting termcaps */
if (enter_standout_mode == NULL || exit_standout_mode == NULL) /* JJD 9-86 */
    enter_standout_mode = exit_standout_mode = NULL;

if (enter_ul_mode == NULL || exit_ul_mode == NULL)
    enter_ul_mode = exit_ul_mode = NULL;

if (enter_delete_mode == NULL || exit_delete_mode == NULL)
    enter_delete_mode = exit_delete_mode = NULL;

if (enter_insert_mode == NULL|| exit_insert_mode == NULL)
    enter_insert_mode = exit_insert_mode = NULL;

/* Set underline mode */
set_underline_mode();
}

/* =======================================================================
 * Name - insert_str
 *
 * Purpose - Insert characters at current cursor on line.  Leaves
 *           cursor after inserted characters.
 *
 * Arguments:  s -- string to insert
 *             num -- number of characters of s to insert
 *             buf -- current contents of display line after current
 *                    cursor, not display expanded.  This is to help
 *                    dumb termcaps, which must do their work by reprinting
 *                    the line
 *
 * Returns     function return -- number of display positions used
 *
 *========================================================================
 */
int insert_str(s, num, buf)
char s[];
int num;
char buf[];
{
int count;
char *cptr = &s[0];
int result = 0;
int bufSize;
int bufLen;

if (buf == NULL || buf[0] == '\0') { /* special case -- at end */

  for (count = 0; count < num; count ++)
    result += overprint_chr(*cptr++);
  return result;
  }

/* Another special case -- all on same line */
bufSize = get_display_length(buf);
bufLen = strlen(buf);
if (CurrentCursor / USED_COLUMNS ==
    (CurrentCursor + bufSize + 1 + get_display_length(s)) / USED_COLUMNS) {
  /* All on same line */
  return simple_insert(s, num, buf, bufLen);
  }

/* Do it the dumb way.  You could get really tricky here... */
return stupid_insert(s, num, buf, bufLen);
}

/* =======================================================================
 * Name - move_horizontal
 *
 * Purpose - Move left n display positions (right if negative).  Only
 *           operates on a single line.
 *
 * Arguments:  n -- number of display positions to move
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void move_horizontal(n)
{
int count = n;

for (; count > 0; --count)
  ADD_TC_STR(cursor_left);
for (; count < 0; count ++)
  ADD_TC_STR(cursor_right);
CurrentCursor -= n;
}

/* =======================================================================
 * Name - move_left
 *
 * Purpose - Move cursor left n display positions.  The complication of
 *           multiple display lines is hidden by this interface.
 *
 * Arguments:  n -- number of display positions to move
 *
 * Returns     function return --
 *
 *========================================================================
 */
int move_left(n)
int n;
{
/* positions are zero-based */
int currentRow = CurrentCursor / USED_COLUMNS;
int currentColumn = CurrentCursor % USED_COLUMNS;
int targetCursor = CurrentCursor - n;
int targetRow;
int targetColumn;

if (targetCursor < PromptStrSize) /* limit to right of prompt */
  targetCursor = PromptStrSize;
targetRow = targetCursor / USED_COLUMNS;
targetColumn = targetCursor % USED_COLUMNS;
/* You know, if we kept track of the current physical row the cursor is
   on the display, we could use tgoto() instead of the following.
   Unfortunately, we don't have any good way of knowing where your cursor
   is (which row) during the time the terminal is in raw mode, so we
   would have to do something drastic after returning from raw mode in
   order to function properly... The more you think about it, the less
   attractive it sounds.  But I hold it out as a possibility... --DJP */
move_horizontal(currentColumn - targetColumn); /* should be before vertical */
move_vertical(currentRow - targetRow);
}

/* =======================================================================
 * Name - move_vertical
 *
 * Purpose - Move cursor vertically on display
 *
 * Arguments:  n -- number of positions to move UP (down if negative)
 *
 * Returns     function return --
 *
 * Note -- downward motion probably not used anywhere, but could be
 *         sometime in the future.
 *
 *========================================================================
 */
static void move_vertical(n)
{
int count = n;

for (; count > 0; --count)
  ADD_TC_STR(cursor_up);
for (; count < 0; count ++)
  ADD_TC_STR(cursor_down);
CurrentCursor -= n * USED_COLUMNS;
}

/* =======================================================================
 * Name - overprint_chr
 *
 * Purpose - Overprint the display starting at the current cursor.  Write
 *           one character.  More than one display position may be used
 *           due to control expansion.  The cursor is left after the last
 *           display position used (where next insertion will occur).
 *
 * Arguments:  c -- the single character to overwrite
 *
 * Returns     function return -- the number of display positions written
 *
 *========================================================================
 */
int overprint_chr(c)
char            c;
{
int result = 0;
unsigned char temp = (unsigned char)c;

if (temp >= MAX_SET) {
  ADD_CHAR('!');
  CurrentCursor ++;
  TO_NEXT_LINE;

  temp -= MAX_SET;
  result = 1;
  }
if (temp < ' ') {
  ADD_CHAR('^');
  CurrentCursor ++;
  TO_NEXT_LINE;

  ADD_CHAR(temp + '@');
  CurrentCursor ++;
  TO_NEXT_LINE;

  return result + 2;
  }
if (temp == ASCII_RUBOUT) {
  ADD_CHAR('^');
  CurrentCursor ++;
  TO_NEXT_LINE;

  ADD_CHAR('?');
  CurrentCursor ++;
  TO_NEXT_LINE;

  return result + 2;
  }

ADD_CHAR(temp);
CurrentCursor ++;
TO_NEXT_LINE;

return result + 1;
}

/* =======================================================================
 * Name - overprint_str
 *
 * Purpose - To overwrite the display at current cursor with zero or more
 *           characters.  The cursor is left after the last display position
 *           written.
 *
 * Arguments:  s -- The string to overwrite with.  The string is NOT display
 *           expanded.
 *             num -- The number of characters of s to write to the display.
 *           The actual number of display positions may be higher due to
 *           display expansion.
 *
 * Returns     function return --  The number of display positions written
 *
 *========================================================================
 */
int overprint_str(s, num)
char           s[];
int num;
{
int             len;
char           *cptr;

cptr = &s[0];
len = 0;
while (cptr != &s[num]) {
  len += overprint_chr(*cptr);
  cptr ++;
  }
return len;
}

/* =======================================================================
 * Name - redraw_prompt
 *
 * Purpose - To refresh the pty's prompt to the user's display.  Assumes
 *           cursor is currently in the first column.
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void redraw_prompt()
{
write_display(PromptStr, PromptStrSize);
CurrentCursor = PromptStrSize;
}

/* =======================================================================
 * Name - show_termcap_info
 *
 * Purpose - Report the termcap used by the user
 *
 * Arguments: 
 *
 * Returns     function return --
 *
 *========================================================================
 */
int show_termcap_info()  /* was report_termcap, now bindable John Apr 90 */
{
FILE *f = stdout;
fprintf(f, "\r\n");
fprintf(f, "         terminal: ");
if (termname == NULL)
  fprintf(f, "NONE");
else
  fprintf(f, "%s", termname);
fprintf(f, "; %d rows, %d columns\r\n", displayRows, displayColumns);
fprintf(f, "     clear_to_eol: %s\r\n", SAY_BOOL(clear_to_eol != NULL));
fprintf(f, "     clear_screen: %s\r\n", SAY_BOOL(clear_screen != NULL));
fprintf(f, "enter_insert_mode: %s\r\n", SAY_BOOL(enter_insert_mode != NULL));
fprintf(f, "  pre_insert_char: %s\r\n", SAY_BOOL(pre_insert_char != NULL));
fprintf(f, "     enter_delete: %s\r\n", SAY_BOOL(enter_delete_mode != NULL));
fprintf(f, "    delete_a_char: %s\r\n", SAY_BOOL(delete_a_char != NULL));
fprintf(f, "   enter_standout: %s\r\n", SAY_BOOL(enter_standout_mode != NULL));
fprintf(f, "         enter_ul: %s\r\n", SAY_BOOL(enter_ul_mode != NULL));
fprintf(f, "     auto_margins: %s\r\n", SAY_BOOL(automatic_margins));
fprintf(f, "multi delete char: %s\r\n", SAY_BOOL(multi_delete_char != NULL));
display_continue();
CLEAR_FACTOR;
return OK;
}


/* =======================================================================
 * Name - resizeWindow
 *
 * Purpose - Check to see if slave or display have changed window size,
 *           and if so, propogate properly
 *
 * Arguments:
 *
 * Returns     function return --
 *
 *========================================================================
 */
void resizeWindow()
{
#ifdef TIOCGWINSZ
 struct winsize theSize;

/* if user resizes sunview window, propogate to pty */
  if (ioctl(DISPLAY, TIOCGWINSZ, (int)&theSize) >= 0) {
    if ((theSize.ws_row != displayRows || theSize.ws_col != displayColumns)
        && theSize.ws_row != 0 && theSize.ws_col != 0 /* not garbage */) {
      displayRows = theSize.ws_row;
      displayColumns = theSize.ws_col;
      ioctl(pty_master, TIOCSWINSZ, (int)&theSize);
      }
    }

/* if pty resizes itself (like an "stty"), resize the master window */
  if (ioctl(pty_master, TIOCGWINSZ, (int)&theSize) >= 0) {
    if ((theSize.ws_row != ptyWin.ws_row || theSize.ws_col != ptyWin.ws_col) &&
        theSize.ws_row != 0 && theSize.ws_col != 0 /* not garbage */) {
      ptyWin = theSize;
      ioctl(DISPLAY, TIOCSWINSZ, (int)&theSize);
      displayRows = theSize.ws_row;
      displayColumns = theSize.ws_col;
      }
    }
#else
/* System V -- read lines and columns properly for this driving program, let
   pty fend for itself (yucch)
*/
int junkInt;

junkInt = tgetnum("li");
if (junkInt != 0)
  displayRows = junkInt;
junkInt = tgetnum("co");
if (junkInt != 0)
  displayColumns = junkInt;
#endif
}

/* =======================================================================
 * Name - set_underline_mode
 *
 * Purpose - Determine standout termcap based on requested underline mode
 *           and available termcaps
 *
 * Arguments:
 *
 * Returns     function return --
 *
 * Allows termcap decisions to be invoked from the command line semantics
 * without exposing this termcap layer's data structures or implementation.
 *
 *========================================================================
 */
void set_underline_mode()
/* Initialize underline termcap */
{
if (underlineMode) {
  pred_enter_standout_mode = enter_ul_mode;
  pred_exit_standout_mode = exit_ul_mode;
  }
else {
  pred_enter_standout_mode = enter_standout_mode;
  pred_exit_standout_mode = exit_standout_mode;
  }
}

/* =======================================================================
 * Name - simple_erase
 *
 * Purpose - Erase characters to right on single line, starting at
 *           current cursor position
 *
 * Arguments:  n -- Number of display positions to erase
 *             pat -- Contents of display at current cursor position, not
 *                 display expanded
 *             patSize -- size of pat, in chars
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void simple_erase(n, pat, patSize)
int n;
char pat[];
int patSize;
{
int count;
int patLength = get_display_length(pat);

/* change as in patch3   Dejan Apr 90 */
/* if(n == 0) return;     no-op */

if (clear_to_eol != NULL && patLength <= n) {
  ADD_TC_STR(clear_to_eol);
  return;
  }

if (multi_delete_char != NULL && n > strlen(multi_delete_char)) {
  ADD_TC_NUM(multi_delete_char, n);
  return;
  }

if (delete_a_char != NULL) {
  ADD_TC_STR(enter_delete_mode);

  for (count = n; count > 0; --count)
    ADD_TC_STR(delete_a_char);

  ADD_TC_STR(exit_delete_mode);
  return;
  }

/* no smart termcap */
stupid_erase(n, pat, patSize);
}

/* =======================================================================
 * Name - simple_insert
 *
 * Purpose - Do an insertion at current cursor.  Assumed to fit on a
 *           single line.  Cursor is left after insertion.
 *
 * Arguments:  s -- characters to insert, not display expanded
 *             num -- number of characters of s to insert
 *             buf -- current contents of line at cursor, not display expanded
 *             bufSize -- length of buf, in characters
 *
 * Returns     function return -- number of display positions used
 *
 *========================================================================
 */
static int simple_insert(s, num, buf, bufSize)
/* do an insertion, single line */
char s[];
int num;
char buf[];
int bufSize;
{
int count;
char *cptr = &s[0];
int result = 0;

if (enter_insert_mode  != NULL || pre_insert_char != NULL) {
  ADD_TC_STR(enter_insert_mode);

  for (count = 0; count < num; count ++) {
    unsigned char temp = (unsigned char)*cptr;

    ADD_TC_STR(pre_insert_char);
    if (temp >= MAX_SET) {
      ADD_CHAR('!');
      ADD_TC_STR(post_insert_char);
      ADD_TC_STR(pre_insert_char);
      CurrentCursor ++;
      result ++;
      temp -= MAX_SET;
      }
    if (temp < ' ') {
      ADD_CHAR('^');
      ADD_TC_STR(post_insert_char);
      ADD_TC_STR(pre_insert_char);
      ADD_CHAR(temp + '@');
      CurrentCursor += 2;
      result += 2;
      }
    else
    if (temp == ASCII_RUBOUT) {
      ADD_CHAR('^');
      ADD_TC_STR(post_insert_char);
      ADD_TC_STR(pre_insert_char);
      ADD_CHAR('?');
      CurrentCursor += 2;
      result += 2;
      }
    else {
      ADD_CHAR(temp);
      CurrentCursor ++;
      result ++;
      }
    cptr ++;
    ADD_TC_STR(post_insert_char);
    }

  ADD_TC_STR(exit_insert_mode);
  return result;
  }
/* dumb termcap */
return stupid_insert(s, num, buf, bufSize);
}

/* =======================================================================
 * Name - stupid_erase
 *
 * Purpose - Erase bytes, stupidly
 *
 * Arguments:  n -- number of display positions to erase
 *             pat -- current contents of display at current cursor (not
 *                 display expanded)
 *             patSize -- number of bytes of pat
 *
 * Returns     function return --
 *
 *========================================================================
 */
static void stupid_erase(n, pat, patSize)
int n;
char pat[];
int patSize;
{
int count;
int rest_of_line_length;
char * cptr;
int patLength = get_display_length(pat);

if (patLength <= n) {/* full blanking, or full line */
  for (count = n; count > 0; --count)
    overprint_chr(' ');
  move_left(n);
  return;
  }

cptr = &pat[0];
count = 0;
while (count < n) { /* figure how much to retain */
   count += get_char_display_length(*cptr);
   cptr ++;
   }
rest_of_line_length =  overprint_str(cptr, &pat[patSize] - cptr);
for (count = patLength - rest_of_line_length; count > 0; --count)
  overprint_chr(' ');
move_left(patLength);
}

/* =======================================================================
 * Name - stupid_insert
 *
 * Purpose - Insert characters by overwriting display.  Leaves cursor
 *           after inserted characters
 *
 * Arguments:  s -- characters to insert
 *             num -- number of characters of s to insert
 *            buf -- current contents at display cursor, not display
 *                 expanded
 *             bufSize -- size of buf, in chars
 *
 * Returns     function return -- Number of display positions written
 *
 *========================================================================
 */
static int stupid_insert(s, num, buf, bufSize)
char s[];
int num;
char buf[];
int bufSize;
{
int result = overprint_str(s, num);
int num_to_back_up = overprint_str(buf, bufSize);

move_left(num_to_back_up);
return result;
}

/* =======================================================================
 * Name - write_display
 *
 * Purpose - Unbuffered write to display
 *
 * Arguments:  buf -- Address of bytes to write
 *             num -- Number of bytes to write
 *
 * Returns     function return --
 *
 * Note:  This routine should eventually be rewritten to "schedule" bytes
 *        to be written in a non-blocking way.  This would allow all reads
 *        and writes to be done non-blocking, preventing certain race
 *        conditions involving overruns from hanging the program.
 *
 *========================================================================
 */
void write_display(buf, num)
char buf[];
int num;
{
write(DISPLAY, buf, num);
}

/* =======================================================================
 * Name - xlat_termcap
 *
 * Purpose - Translate a termcap code to an asciz string
 *
 * Arguments:  tc -- the termcap capability to translate, i.e., "ku"
 *             name -- a description of the purpose of this termcap,
 *                     used in error messages
 *             prognosis -- Description of what the failure to bind
 *                     this termcap means.  If prognosis is NULL,
 *                     failure is regarded as non-fatal (i.e., clear-to-eol
 *                     for example), and no message is printed
 *
 * Returns     function return -- Address of the termcap capability, NULL
 *                     if not available
 *
 *========================================================================
 */
char * xlat_termcap(tc, name, prognosis)
char tc[];
char name[];
char prognosis[];
{
char *result = tgetstr(tc, &tc_ptr);

if (result != NULL)
  return result;

if (prognosis != NULL)
  printf("%s: Termcap capability \"%s\" (%s) is missing.\r\n    %s\r\n",
      ProgramName, tc, name, prognosis);
return NULL;
}
