/* File: /u1/oystr/HRPC/RpcProtos/rpcSunServ.c  Date: 28-Jun-1986  */

/*
 * $Header$
 * INTERFACE:	Implements the server-side routines of the
 *		SUN RPC protocol.  Companion file rpcSunClnt.c
 *		contains client side routines.
 *
 * FUNCTION:	See above.
 *
 * IMPORTS:	basicRpc.h + SUN files
 *		+ rpcSunHdrs.c
 *
 * EXPORTS:	SunInitIncoming, SunPacketIncoming, SunFinishIncoming,
 *		SunInitReply,	 SunReplyPacket,    SunFinishReply
 *
 * DESIGN:	See comments in rpcSunClnt.c.
 *
 * $Log$
 * 28-Jun-1986:	Initial implementation, Jan Sanislo
 */

/*
 * SUN specific stuff.
 */
#include <sys/types.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <rpc/auth.h>
#include <rpc/rpc_msg.h>
#include <sys/time.h>
#include <HRPC/basicRpc.h>
#include <HRPC/hrpcErrCodes.h>
#include <HRPC/bufferOps.h>
#include <netinet/in.h>
#include "../Transports/connDefs.h"
#include "../RpcProtos/rpcSun.h"
#include <setjmp.h>

/* "forward" definitions */
extern SunPacketIncoming();
extern SunReplyPacket();
extern int AllocateBuffer();

/*
 * Wait for an incoming request message.  When it
 * arrives pull apart the header and return the
 * program, version and procedure number.  The assumption
 * is that the associated socket has already been opened
 * by the export routine.
 */
static struct opaque_auth sunAuthNULL; /* must be 0 */

/*
 * We generally unwind on a closed connection,
 * but in this InitIncoming, we will return the status
 * through fStatus.
 */

SunInitIncoming( fBptr, fProgNum, fVersNum, fProcNum,
		 fStatus, fSpare2 )
    HRPCBinding *fBptr;
    long *fProgNum;
    long *fVersNum;
    long *fProcNum;
    int  *fStatus;
    char *fSpare2;
{
    register struct rpc_msg *callHdr;
    register OtwControl   *otwptr = &(fBptr->otwDescr);
    register TransControl *tcptr  = &(fBptr->transDescr);
    SunCallInfo		  *scptr;
    memory		   saveUnwind;
    HRPCErrRec		  *errRes;
    jmp_buf		   siiUnwind;

    /*
     * Allocate/reuse callinfo structure.
     */
    if ( errRes = (HRPCErrRec *) setjmp( siiUnwind ) ) {
	*fStatus = (int) errRes->hrpcErr;
	HRPCFreeErr( errRes );
	fBptr->unwindTo = saveUnwind;
	return;
    }
    saveUnwind = fBptr->unwindTo;
    fBptr->unwindTo = (memory) siiUnwind;
    
    if ( (scptr = (SunCallInfo *) fBptr->rpcDescr.callState) == 
	    (SunCallInfo *) NULL ) {
	scptr = (SunCallInfo *) calloc( 1, sizeof(SunCallInfo) );
	if ( scptr == (SunCallInfo *) NULL ) {
	    fatalerr("SunInitIncoming: can't alloc call info.\n");
	}
	fBptr->rpcDescr.callState = (memory) scptr;
    }
    callHdr = &(scptr->callHdr);
    /* Some place for auth info - currently thrown away */
    callHdr->rm_call.cb_cred = callHdr->rm_call.cb_verf = sunAuthNULL;
    
    /*
     * Set OTW direction and snarf a buffer.
     */
    fBptr->otwOp = OTW_DECODE;

    *fStatus = (*tcptr->InitRecv)(fBptr, CONN_SRVCALL);
    
    /*
     * Try to read a packet - loop on timeout.
     */
    while( *fStatus == HRPC_REPLYTMO ) {
	*fStatus = (*tcptr->RecvPacket)(fBptr);
    };
    fBptr->unwindTo = saveUnwind;
    if ( *fStatus ) return;

    /*
     * If using TCP handler record marking.
     */
    if ( fBptr->transType != UDPTRANSP ) {
	(*fBptr->otwDescr.LongCardinal)(fBptr,&scptr->segSize);
	scptr->bufSize = fBptr->curBufSize;
	scptr->segSize &= ~LASTSEGTAG;
	SetFragFlags( scptr, fBptr );
    }
    
    /*
     * Decode call header.
     */
    SunCallHdr( fBptr, callHdr );

    /*
     * Some sanity checks.
     */
    if ( (callHdr->rm_direction != CALL) ||
	 (callHdr->rm_call.cb_rpcvers != RPC_MSG_VERSION) ) {
	fatalerr("SunInitIncoming: bad call header.\n");
    }

    /*
     * Looks OK, send back results.
     */
    *fProgNum = (long) callHdr->rm_call.cb_prog;
    *fVersNum = (long) callHdr->rm_call.cb_vers;
    *fProcNum = (long) callHdr->rm_call.cb_proc;

    /*
     * Stuff packet input routine.
     */
    fBptr->rpcDescr.GetPacket = SunPacketIncoming;

    return;
}

/*
 * Handle one incoming packet.  The RPC level routines
 * for this are actually the same for both server and client
 * side - they should actually be factored out and placed
 * in a separate file.  Later for this optimization.
 */

SunPacketIncoming( fBptr )
    HRPCBinding *fBptr;
{
    SunCallInfo *scptr;
    
    if ( fBptr->transType == UDPTRANSP ) {
	fatalerr("SunPacketIncoming: can't happen with UDP.\n");
    }
    scptr = (SunCallInfo *) fBptr->rpcDescr.callState;

    /*
     * Check to see if next segment in buffer.
     */
    if ( scptr->fragFlags & NEXTSEGINBUF ) {
	/* (re)set bytes remaining */
	fBptr->curBufSize = scptr->bufSize - scptr->fragSize;
	scptr->fragFlags &= ~ NEXTSEGINBUF;
        /* In case of fragmented count field */
	scptr->segSize = scptr->fragSize = 4;
	(*fBptr->otwDescr.LongCardinal)(fBptr,&scptr->segSize);
	scptr->segSize &= ~LASTSEGTAG;
    }
    else {
	/*
	 * Must read more data.
	 * First fake segment size down to bytes expected - bytes consumed
	 */
	if ( scptr->fragFlags & FRAGINBUF ) {
	    scptr->segSize -= scptr->fragSize;
	    (*fBptr->transDescr.RecvPacket)( fBptr );
	}
	else {
            /*
             * it is time to read a whole new segment into the buffer.
             * the last segment was neither fragmented, nor accompanied
             * by another segment in curbuf (NEXTSEGINBUF not there)
             */
            (*fBptr->transDescr.RecvPacket)(fBptr);
	    
            /*
             * since this a whole new segment, we have to recompute
             * its size and (of course) readjust fBptr->curBufSTUFF
             * to reflect our knowledge of the new segment
             *  (stolen fom SunInitAnswer)
             */
            (*fBptr->otwDescr.LongCardinal)(fBptr, &scptr->segSize);
            scptr->bufSize = fBptr->curBufSize;
            scptr->segSize &= ~ LASTSEGTAG;   
	}
    }
    /*
     * Sometimes we can get out of synch. The usual cases are
     * 1) segment header is *last* four bytes in the buffer,
     * and 2) when we do a read, we only get 4 bytes of a segment
     * header, leaving an empty data buffer.  Therefore, make sure
     * we leave here with the buffer containing some data to be
     * consumed.
     */
    if ( fBptr->curBufSize == 0 ) {
	(*fBptr->transDescr.RecvPacket)(fBptr);
	scptr->bufSize = fBptr->curBufSize;
    }

    SetFragFlags( scptr, fBptr );
}

/*
 * This routine does absolutely nothing for now.
 */
SunFinishIncoming( fBptr )
    HRPCBinding *fBptr;
{
    (*fBptr->transDescr.FinishRecv)(fBptr, CONN_SRVCALL);
}

/*
 * Routines for sending replies to incoming requests.
 */

SunInitReply( fBptr )
    HRPCBinding *fBptr;
{
    register struct rpc_msg *callHdr;
    register OtwControl   *otwptr = &(fBptr->otwDescr);
    register TransControl *tcptr  = &(fBptr->transDescr);
    SunCallInfo		  *scptr;
    struct accepted_reply *acptr;
    
    /*
     * Set direction.
     */
    fBptr->otwOp = OTW_ENCODE;
    AllocateBuffer(fBptr);
    (*tcptr->InitSend)(fBptr, CONN_SRVREPLY);
    
    /*
     * Reuse the original call header associated
     * with the request.
     */
    scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
    callHdr = &(scptr->callHdr);
    callHdr->rm_direction = REPLY;
    callHdr->rm_reply.rp_stat = MSG_ACCEPTED;
    acptr = &(scptr->callHdr.acpted_rply);

    /*
     * Watch out for this verf business --
     * two pointers to same memory location.
     */
    acptr->ar_verf = callHdr->rm_call.cb_verf;
    acptr->ar_stat = SUCCESS;

    if ( fBptr->transType != UDPTRANSP ) {
	GetBufPos( fBptr, &(scptr->countPos) );
	fBptr->curBufMark += sizeof(LongCardinal);
	fBptr->curBufSize -= sizeof(LongCardinal);
    }

    /* Stuff output routine */
    fBptr->rpcDescr.PutPacket = SunReplyPacket;

    /* Do header and return */
    (void ) SunReplyHdr( fBptr, callHdr );

    return;
}

SunReplyPacket( fBptr )
    HRPCBinding *fBptr;
{
    BufPos oldPos;
    int    byteCount;
    SunCallInfo *scptr;
    
    if ( fBptr->transType != UDPTRANSP ) {
	/* Stuff segment byte count and send it off */
	scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
	SetBufPos( fBptr, &(scptr->countPos), &oldPos );
	byteCount = oldPos.bufMark - scptr->countPos.bufMark
		      - sizeof(LongCardinal);
	(*fBptr->otwDescr.LongCardinal)(fBptr, &byteCount);
	SetBufPos( fBptr, &oldPos, 0);
    }
    else {
	/* should not happen if using UDP */
	fatalerr("SunPacketOutgoing: should not happen.\n");
    }
    (*fBptr->transDescr.SendPacket)(fBptr);
    if ( fBptr->transType != UDPTRANSP ) {
	/* record buffer position for next packet and save space */
	GetBufPos( fBptr, &(scptr->countPos) );
	fBptr->curBufMark += sizeof(LongCardinal);
	fBptr->curBufSize -= sizeof(LongCardinal);
    }
}

SunFinishReply( fBptr )
    register HRPCBinding *fBptr;
{
    BufPos oldPos;
    int    byteCount;
    SunCallInfo *scptr;
    
    if ( fBptr->transType != UDPTRANSP ) {
	/* Stuff segment byte count and send it off */
	scptr = (SunCallInfo *) fBptr->rpcDescr.callState;
	SetBufPos( fBptr, &(scptr->countPos), &oldPos );
	byteCount = oldPos.bufMark - scptr->countPos.bufMark
		       - sizeof(LongCardinal);
	/* last segment, set indicator */
	byteCount |= LASTSEGTAG;
	(*fBptr->otwDescr.LongCardinal)(fBptr, &byteCount);
	SetBufPos( fBptr, &oldPos, 0);
    }

    (*fBptr->transDescr.FinishSend)(fBptr, CONN_SRVREPLY);
}
