/* File: /u1/oystr/HRPC/HRpcRTS/otwSun.c  Date:  2-Mar-1986  */

/*
 * $Header$
 * INTERFACE:	Several - see procedure definitions.
 *
 * FUNCTION:	Marshalling routines XDR style.
 *
 * IMPORTS:	Basic rpc definitions.
 *
 * EXPORTS:	Many routines.
 *
 * DESIGN:	Direct rip off from SUN XDR routines.
 *		There probably should be a copyright notice
 *		from them in here.
 *
 *		There are two cases which deserve special mention.
 *		First, if we are trying to ENCODE an atomic value
 *		and there is no room in the buffer, we call the
 *		RPC level send routine to transmit the packet.  In other
 *		words, we do not fragment atomic data items over packet
 *		boundaries.  (For purposes of this discussion, individual
 *		characters are atomic - strings are composite types
 *		and hence may be fragmented across packet boundaries.)
 *		Second, if we are trying to DECODE an atomic data item
 *		and there is not enough data, the routine NeedBytes is
 *		called.  It is the responsibility of the caller to
 *		supply NeedBytes with a temporary buffer.  Remaining
 *		bytes in the current buffer are copied to the temp.,
 *		the RPC level read routine is called (so it can strip
 *		headers, etc.)  Enough bytes from the new buffer are
 *		appended to the temp. to complete the data item.  It
 *		is then the responsibility of the caller to complete
 *		the decoding using the data in the temporary buffer.
 *		There are alternatives to this scheme (e.g., save some
 *		ghost space at the start of the buffer, read in the
 *		new buffer, prepend the old bytes, return), but they
 *		require some nastiness with buffer allocation that
 *		amounts to defeating the purpose of having transport
 *		routines allocate the buffers (i.e., potential performance
 *		gains by direct memory mapping of network buffers).
 *
 * DEFECTS:	Many - mainly related to assumptions made about
 *		the storage allocation for Courier types in
 *		the application language.
 * $Log$
 *  2-Mar-1986:	Initial implementation, Jan Sanislo
 */

#include <HRPC/basicRpc.h>

/*
 * Macros for moving to/from network standard.
 * On the SUNs, these do nothing.  On the VAXen
 * they do byte by byte assignments.
 * These were ripped off from the Cornell Courier
 * implementation. blah(a, b) means "move a -> b",
 * the reverse of e.g. strcpy(a,b).  Notice also
 * that short integers occupy 4 bytes OTW, so we
 * we have to provide for that by having separate
 * macros for moving h[sl]<->xdr[sl].
 */
/*
 * Should be "defined(sun) || defined(ibm032)"
 * but the 'defined' contstruct is not recognized on
 * Pegusi.
 */
#ifdef ibm032
#define sun
#endif

#ifdef sun
#define htoxdrb(a,b)  (*((char *)(b)+3) = *(char *)(a))

#define xdrtohb(a,b)  (*(char *)(b) = *((char *)(a)+3))

#define htoxdrs(a,b)  (*((short *)(b)+1) = *(short *)(a))

#define xdrtohs(a,b)  (*(short *)(b) = *((short *)(a)+1))

#define htoxdrl(a,b)  (*(long *)(b) = *(long *)(a))

#define xdrtohl(a,b)  (*(long *)(b) = *(long *)(a))
#endif

#ifdef ibm032
#undef sun
#endif

#ifdef vax
#define htoxdrb(a,b)  ( *((char *)(b)+3) = *(char *)(a) )

#define xdrtohb(a,b)  ( *(char *)(b) = *((char *)(a)+3) )

#define htoxdrs(a,b)  ( *((char *)(b)+2) = *((char *)(a)+1),\
			*((char *)(b)+3) = *(char *)(a))

#define xdrtohs(a,b)  ( *((char *)(b)+1) = *((char *)(a)+2),\
			*(char *)(b) = *((char *)(a)+3) )

#define htoxdrl(a,b)  ( *(char *)(b) = *((char *)(a)+3),\
			*((char *)(b)+1) = *((char *)(a)+2),\
			*((char *)(b)+2) = *((char *)(a)+1),\
			*((char *)(b)+3) = *(char *)(a))

#define xdrtohl(a,b)  ( *((char *)(b)+3) = *(char *)(a),\
			*((char *)(b)+2) = *((char *)(a)+1),\
			*((char *)(b)+1) = *((char *)(a)+2),\
			*(char *)(b) = *((char *)(a)+3) )
#endif

/*
 * Define routine to read/write a count field.
 * For XDR rep, this is a 4-byte, unsigned integer.
 */
#define XdrCountField XdrLongCardinal

/*
 * Alignment control, for strings, etc.
 * This is 4 for XDR and 2 for COURIER.
 */
#define ALIGNSIZE 4

XdrDeallocate( fBptr )
    register HRPCBinding *fBptr;
{
    /* Do nothing for now */
}

XdrLongCardinal( fBptr, lp)
    register HRPCBinding *fBptr;
    register LongCardinal *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;
    
    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    xdrtohl(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	xdrtohl(ptr,lp);
    }
    else {
	htoxdrl(lp,ptr);
    }
    fBptr->curBufMark += sizeof(LongCardinal);
    fBptr->curBufSize -= sizeof(LongCardinal);
}

XdrCardinal( fBptr, lp)
    register HRPCBinding *fBptr;
    register Cardinal *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    xdrtohl(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	xdrtohs(ptr,lp);
    }
    else {
	htoxdrs(lp,ptr);
    }
    fBptr->curBufMark += sizeof(LongCardinal);
    fBptr->curBufSize -= sizeof(LongCardinal);
}

XdrInteger( fBptr, lp)
    register HRPCBinding *fBptr;
    register Integer *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    xdrtohs(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	xdrtohs(ptr,lp);
    }
    else {
	htoxdrs(lp,ptr);
    }
    fBptr->curBufMark += sizeof(LongInteger);
    fBptr->curBufSize -= sizeof(LongInteger);
}

XdrLongInteger( fBptr, lp)
    register HRPCBinding *fBptr;
    register LongInteger *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    xdrtohl(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	xdrtohl(ptr,lp);
    }
    else {
	htoxdrl(lp,ptr);
    }
    fBptr->curBufMark += sizeof(LongInteger);
    fBptr->curBufSize -= sizeof(LongInteger);
}

XdrUnspecified( fBptr, lp)
    register HRPCBinding *fBptr;
    register Unspecified *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    *lp = *((Unspecified *)(&tmp)+1);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	*lp = *((Unspecified *)(ptr)+1);
    }
    else {
	*((Unspecified *) (ptr)+1) = *lp;
    }
    fBptr->curBufMark += sizeof(LongUnspecified);
    fBptr->curBufSize -= sizeof(LongUnspecified);
}

XdrLongUnspecified( fBptr, lp)
    register HRPCBinding *fBptr;
    register LongUnspecified *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongUnspecified)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongUnspecified tmp;

	    OtwNeedBytes( fBptr, sizeof(LongUnspecified) - fBptr->curBufSize,
			  &tmp );
	    *lp = tmp;
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	*lp = *(LongUnspecified *)ptr;
    }
    else {
	*(LongUnspecified *) ptr = *lp;
    }
    fBptr->curBufMark += sizeof(LongUnspecified);
    fBptr->curBufSize -= sizeof(LongUnspecified);
}

XdrBoolean( fBptr, lp)
    register HRPCBinding *fBptr;
    register Boolean *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE ) return;

    if ((fBptr->curBufSize - sizeof(LongCardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    LongCardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(LongCardinal) - fBptr->curBufSize,
			  &tmp );
	    xdrtohb(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	xdrtohb(ptr, lp);
    }
    else {
	htoxdrb(lp,ptr);
    }
    fBptr->curBufMark += sizeof(LongUnspecified);
    fBptr->curBufSize -= sizeof(LongUnspecified);
}

XdrEnumeration( fBinding, fValue )
    HRPCBinding *fBinding;
    LongCardinal *fValue;
{
    XdrLongCardinal( fBinding, fValue );
}

XdrString( fBptr, fString )
    register HRPCBinding *fBptr;
    String	*fString;
{
    register char *sp = *fString;	
    int size, nodesize;

    if ( fBptr->otwOp == OTW_FREE ) {
	free( *fString );
	return;
    }
    
    if ( fBptr->otwOp != OTW_DECODE )
	size = strlen( sp );
    XdrCountField( fBptr, &size ); /* read/write size */
    nodesize = size + 1; /* for 0-byte terminator */
    
    switch( fBptr->otwOp ) {
	case OTW_DECODE:
	    *fString = sp = (char *) malloc( nodesize );
	    if ( sp == (char *) 0 )
		fatalerr("XdrString: can't alloc string space.\n");
	    sp[size] = (char) 0;
	    /* fall into ... */

	case OTW_ENCODE:
	    /*
	     * Should be XdrStringCopy in order to do
	     * character translation.
	     */
	    if ( size > 0 )
	        OtwByteCopy( fBptr, (memory) sp, size, ALIGNSIZE );
	    return;

	case OTW_FREE:
	    if ( sp != (char *) NULL ) {
		free( sp );
		*fString = (char *) NULL;
	    }
    }
}


XdrArray( fBinding, fAddr, fArrSize, fEltSize, fComp )
    register HRPCBinding *fBinding;
    memory fAddr;
    int fArrSize;
    int fEltSize;
    int (*fComp)();
{
    int realSize = 0;
    register memory target = fAddr;
    register int i;

    realSize = fArrSize;    
    if ( fBinding->otwOp == OTW_FREE ) {
	goto churn;
    }

    XdrCountField( fBinding, &realSize );
    if ( (realSize != fArrSize) || (realSize <= 0) ) {
	fatalerr("XdrArray: array size mismatch: %d != %d.\n",
		 realSize, fArrSize);
    }

    /* XXX deprecated: never allocate anything of fixed size.
     * To reinstall change definition of "memory fAddr" to "memory *fAddr".
     * But first change the stub compiler so it will pass you the
     * right thing.
     * Allocate memory based on fAddr != NULL
     *
    if ( target == (memory) NULL ) {
	if ( fBinding->otwOp == OTW_FREE ) return;
	if ( fBinding->otwOp == OTW_DECODE ) {
	    *fAddr = target = (memory) calloc( realSize * fEltSize, 1 );
	    if ( target == (memory) NULL ) {
		fatalerr("XdrArray: can't get memory for %d * %d bytes.\n",
			 realSize, fEltSize);
	    }
	}
    }
     * XXX */

    /*
     * Churn through, one element at a time.
     */
churn:
    for ( i = 0; i < realSize; i++ ) {
	(*fComp)(fBinding, target );
	target += fEltSize;
    }
}

XdrSequence( fBinding, fAddr, fArrSize, fMaxSize, fEltSize, fComp )
    register HRPCBinding *fBinding;
    memory *fAddr;
    Cardinal *fArrSize;
    int    fMaxSize;
    int    fEltSize;
    int    (*fComp)();
{
    int realSize;
    register memory target = *fAddr;
    register int i;

    if ( fBinding->otwOp == OTW_FREE ) {
	if ( *fArrSize <= 0 ) return;
	realSize = (int) *fArrSize;
	goto churn;
    }

    realSize = (int) *fArrSize;
    XdrCountField( fBinding, &realSize );
    *fArrSize = (Cardinal) realSize;
    if ( *fArrSize < 0 ) {
	fatalerr("XdrSequence: negative size: %d.\n", *fArrSize);
    }
    if ( *fArrSize == 0 ) return; /* guessing 0 length is ok */
    if ( *fArrSize > fMaxSize ) {
	fatalerr("XdrSequence: max size exceeded: %d > %d.\n",*fArrSize,
		  fMaxSize);
    }
	      
    /*
     * Always allocate on DECODE, never allocate on ENCODE.
     */
    if ( fBinding->otwOp == OTW_DECODE ) {
	*fAddr = target = (memory) calloc( realSize * fEltSize, 1 );
	if ( target == (memory) NULL ) {
	    fatalerr("XdrSequence: can't get memory for %d * %d bytes.\n",
		 realSize, fEltSize);
	}
    }

    /*
     * Churn through, one element at a time.
     */
churn:
    for ( i = 0; i < realSize; i++ ) {
	(*fComp)(fBinding, target );
	target += fEltSize;
    }
    if ( fBinding->otwOp == OTW_FREE ) {
	free(*fAddr);
    }
}

XdrError()
{
    abort();
}

XdrNilRecord()
{
    /* do absolutely nothing */
}

/*
 * Semi-dummy for now.
 */
/*ARGSUSED*/
XdrCloseOtw( fBptr )
    HRPCBinding *fBptr;
{
}

/*
 * 		>>>>> WARNING <<<<
 * Following three procedures are never called
 * directly.  Their function is performed by routines
 * generated by the stub compiler.
 */

XdrRecord()
{
    abort();
}

XdrProcedure()
{
    abort();
}

XdrChoice()
{
    abort();
}
