/* File: /u1/schwartz/HCS/Impl/HRPC/MyProtos/OtwReps/otwRaw.c  1-Sep-1986  */

/*
 * $Header$
 * INTERFACE:	See procedure definition.
 *
 * FUNCTION:	String marshalling routine.
 *
 * IMPORTS:	Basic rpc definitions.
 *
 * EXPORTS:	RawXdrString, RawXdrInteger, RawXdrCardinal, RawXdrArray,
 *		RawXdrUnspecified, RawXdrBoolean.
 *
 * DESIGN:	Use the same OTW rep. as Sun XDR, except: (1) strings aren't
 *		byte-counted (to have RAW OTW format match up with the common
 *		null-byte delimited C strings) and are short-aligned;
 *		(2) Integers are unaligned instead of long-word aligned, to
 *		allow for C message structures which need this granularity;
 *		similarly for Cardinals; (3) Array's aren't byte-counted (they
 *		are of fixed length anyway); (4) Unspecified's are short-word
 *		aligned; (5) Booleans are byte-aligned.
 *
 * $Log$
 *  1-Sep-1986:	Initial implementation, Mike Schwartz.
 */

#include <HRPC/basicRpc.h>

/*
 * Alignment control, for strings, etc.
 * This is 4 for XDR, 2 for COURIER, and 1 for RAW.
 */
#define ALIGNSIZE 1

RawXdrString( fBptr, fString )
    register HRPCBinding *fBptr;
    String	*fString;
{
    register char *sp = *fString;
    int size, nodesize;

    if ( fBptr->otwOp == OTW_DECODE )
	size = strlen(fBptr->curBufMark);
    else
	size = strlen( sp );
    nodesize = size + 1; /* for 0-byte terminator */
    
    switch( fBptr->otwOp ) {
	case OTW_DECODE:
	    *fString = sp = (char *) malloc( nodesize );
	    if ( sp == (char *) 0 )
		fatalerr("RawXdrString: can't alloc string space.\n");
	    sp[size] = (char) 0;
	    /* fall into ... */

	case OTW_ENCODE:
	    /*
	     * Should be XdrStringCopy in order to do
	     * character translation.
	     */
	    if ( size > 0 )
	        OtwByteCopy( fBptr, (memory) sp, nodesize, ALIGNSIZE );
	    return;

	case OTW_FREE:
	    if ( sp != (char *) NULL ) {
		free( sp );
		*fString = (char *) NULL;
	    }
    }
}


/* Data representation conversion macros.  Use these instead of ntohs and htons
   because they are 5-6x faster.  "htoRxdrs" stands for "host to Raw XDR
   standard"; similarly for other macro names */

#if defined(sun) || defined(ibm032)
#define htoRxdrs(a,b)  (*(short *)(b) = *(short *)(a))  
#define Rxdrtohs(a,b)  (*(short *)(b) = *(short *)(a))
#endif

#if defined(vax)
#define htoRxdrs(a,b)  ( *((unsigned char *)(b)+1) = ((unsigned char)(*a)),\
                        *((unsigned char *)(b)) = *((unsigned char *)(a)+1))
#define Rxdrtohs(a,b)  ( *((unsigned char *)(b)+1) = ((unsigned char)(*a)),\
                        *((unsigned char *)(b)) = *((unsigned char *)(a)+1))
#endif

RawXdrInteger( fBptr, lp)
    register HRPCBinding *fBptr;
    register Integer *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE )
	return;

    if ((fBptr->curBufSize - sizeof(Integer)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    Integer tmp;

	    OtwNeedBytes( fBptr, sizeof(Integer) - fBptr->curBufSize,
			  &tmp );
	    Rxdrtohs(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	Rxdrtohs(ptr,lp);
    }
    else {
	htoRxdrs(lp,ptr);
    }
    fBptr->curBufMark += sizeof(Integer);
    fBptr->curBufSize -= sizeof(Integer);
}

RawXdrCardinal( fBptr, lp)
    register HRPCBinding *fBptr;
    register Cardinal *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE )
	return;

    if ((fBptr->curBufSize - sizeof(Cardinal)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    Cardinal tmp;

	    OtwNeedBytes( fBptr, sizeof(Cardinal) - fBptr->curBufSize,
			  &tmp );
	    Rxdrtohs(&tmp,lp);
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	Rxdrtohs(ptr,lp);
    }
    else {
	htoRxdrs(lp,ptr);
    }
    fBptr->curBufMark += sizeof(Cardinal);
    fBptr->curBufSize -= sizeof(Cardinal);
}

RawXdrArray( fBinding, fAddr, fArrSize, fEltSize, fComp )
    register HRPCBinding *fBinding;
    memory fAddr;
    int fArrSize;
    int fEltSize;
    int (*fComp)();
{
    int realSize = 0;
    register memory target = fAddr;
    register int i;

    realSize = fArrSize;    
    if ( (realSize != fArrSize) || (realSize <= 0) ) {
	fatalerr("RawXdrArray: array size mismatch: %d != %d.\n",
		 realSize, fArrSize);
    }

    /* XXX deprecated: never allocate anything of fixed size.
     * To reinstall change definition of "memory fAddr" to "memory *fAddr".
     * But first change the stub compiler so it will pass you the
     * right thing.
     * Allocate memory based on fAddr != NULL
     *
    if ( target == (memory) NULL ) {
	if ( fBinding->otwOp == OTW_FREE ) return;
	if ( fBinding->otwOp == OTW_DECODE ) {
	    *fAddr = target = (memory) calloc( realSize * fEltSize, 1 );
	    if ( target == (memory) NULL ) {
		fatalerr("RawXdrArray: can't get memory for %d * %d bytes.\n",
			 realSize, fEltSize);
	    }
	}
    }
     * XXX */

    /*
     * Churn through, one element at a time.
     */
    for ( i = 0; i < realSize; i++ ) {
	(*fComp)(fBinding, target );
	target += fEltSize;
    }
}

RawXdrUnspecified( fBptr, lp)
    register HRPCBinding *fBptr;
    register Unspecified *lp;
{
    register memory ptr;

    if ( fBptr->otwOp == OTW_FREE )
	return;

    if ((fBptr->curBufSize - sizeof(Unspecified)) < 0) {
	if ( fBptr->otwOp == OTW_ENCODE ) {
	    (*fBptr->rpcDescr.PutPacket)( fBptr );
	}
	else {
	    Unspecified tmp;

	    OtwNeedBytes( fBptr, sizeof(Unspecified) - fBptr->curBufSize,
			  &tmp );
	    *lp = tmp;
	    return;
	}
    }

    ptr = fBptr->curBufMark;
    if (fBptr->otwOp == OTW_DECODE) {
	*lp = *(Unspecified *) (ptr);
    }
    else {
	*(Unspecified *) (ptr) = *lp;
    }
    fBptr->curBufMark += sizeof(Unspecified);
    fBptr->curBufSize -= sizeof(Unspecified);
}

RawXdrBoolean( fBptr, lp)
    register HRPCBinding *fBptr;
    Boolean *lp;
{
    if ( fBptr->otwOp == OTW_FREE )
	return;

    if ((fBptr->curBufSize -= sizeof(Boolean)) < 0 )
        fatalerr("No buffer space.\n");
    if (fBptr->otwOp == OTW_DECODE)
        *lp = *((Boolean *)(fBptr->curBufMark));
    else
        *((Boolean *)(fBptr->curBufMark)) = *lp;
    fBptr->curBufMark += sizeof(Boolean);
}
