/* File: /u1/oystr/HRPC/LWP/lwpSystemQs.c  Date:  18-Aug-1987  */

/*
 * $Header$
 * FUNCTION: These procedures maintain the system queues (namely,
 *		BindingQ, TaskQ, BufferQ and ArrivedPacketQs)
 *		introduced by the LWP control subsystem.
 * 
 * $Log$
 *
 *  18-Aug-1987: Initial implementation, Mark Squillante
 */

#include <stdio.h>
#include "kEvents.h"
#include <HRPC/LWP/LWPdefs.h>
#include <HRPC/LWP/LWPtypes.h>
#include "lwpSystemQs.h"
#ifdef HAS_XNS43
#include <netns/sp.h>
#endif HAS_XNS43

/*
 * AddToBindingQ adds the specified binding to the binding queue whose head
 * is stored in STMconnStates[spktyp].BindingQ (if it isn't already there - see
 * CleanUpBindingQ comments).  In any event, the IdTable gets set to what's
 * specified in idTable[] (we do this so the higher HRPC levels can set the
 * IdTable appropriately).
 * This procedure is called by the RecvPacket routines just before they call
 * HRPC_SelectAndReadPacket.  The SpeakTypeMgr uses this information to awaken
 * the appropriate LWP once its packet has arrived.  Since the signal handler
 * does not access this queue, there is no need for mutual exclusion.
 */
int	AddToBindingQ(fBptr,idTable,idTableSize,in_dispatcher)
    HRPCBinding			*fBptr;
    struct BindingQIdTableEntry	idTable[];
    int				idTableSize,in_dispatcher;
{
	struct BindingQelement	*new;
	struct BindingQelement  *CleanUpBindingQ();
	void			insertBQ();
	int			DeleteFromBindingQ();
	int			spktyp = (int) fBptr->speaking;
	int			i;

    if ( HRPC_lwpType == SingleThread )
	fatalerr("AddToBindingQ: called in SingleThread mode\n");

    if ( idTableSize > BindingQIdTableMax )
	fatalerr("AddToBindingQ: BindingQIdTableMax exceeded\n");

    if ( (new=CleanUpBindingQ(fBptr,0)) == NULL ) {
	    /* element not already on queue so add it */
	new = (struct BindingQelement *) malloc(sizeof(struct BindingQelement));
	if ( new == NULL )
	    fatalerr("AddToBindingQ: cannot alloc BindingQelement\n");
	new -> fBptr = fBptr;
	new->In_Dispatcher = in_dispatcher;
        new->NumIdValues = idTableSize;
	insertBQ(spktyp,new);
	    /* set IdTable */
    	for ( i=0; i < idTableSize; i++ ) {
		/* structure assignment */
	    new->IdTable[i] = idTable[i];
	}
    } else {
	    /* element already on queue */
	if ( new->NumIdValues == idTableSize ) {
		/* then already in correct position, just set IdTable */
	    new->In_Dispatcher = in_dispatcher;
    	    for ( i=0; i < idTableSize; i++ ) {
		    /* structure assignment */
		new->IdTable[i] = idTable[i];
	    }
	} else {
		/* put element in correct position */
	    DeleteFromBindingQ(spktyp,new,0);
	    new->In_Dispatcher = in_dispatcher;
	    new->NumIdValues = idTableSize;
	    insertBQ(spktyp,new);
		/* set IdTable */
    	    for ( i=0; i < idTableSize; i++ ) {
		    /* structure assignment */
		new->IdTable[i] = idTable[i];
	    }
	}
    }
    return(1);
}
/* insertBQ determines correct location for new in BindingQ (ordered on
 * nonincreasing NumIdValues) & inserts it there */
void	insertBQ(spktyp,new)
    int spktyp;
    struct BindingQelement *new;
{
	STMconnDescr		*stmconnptr = &STMconnStates[spktyp];
	struct BindingQelement	*pred, *succ;
	struct BindingQelement	*undefined = (struct BindingQelement *) -1;

	pred = undefined;
	succ = stmconnptr->BindingQ;
	while ( succ != NULL ) {
	    if ( succ->NumIdValues < new->NumIdValues )
		break;
	    pred = succ;
	    succ = succ->Next;
	}
	new->Next = succ;
	if ( succ != NULL )
	    succ->Prev = new;
	if ( pred == undefined ) {
	    new->Prev = NULL;
	    stmconnptr->BindingQ = new;
	} else {
	    new->Prev = pred;
	    pred->Next = new;
	}
}

/*
 * DeleteFromBindingQ removes the specified element from its binding queue.
 * This procedure is called by SearchBindingQ to remove an element from the
 * corresponding binding queue on a successful search.
 */
int	DeleteFromBindingQ(spktyp,element,freeSpace)
    int				spktyp, freeSpace;
    struct BindingQelement 	*element;
{
	if ( HRPC_lwpType == SingleThread )
	    fatalerr("DeleteFromBindingQ: called in SingleThread mode\n");

	if ( element == NULL )
	    fatalerr("DeleteFromBindingQ: called with NULL element\n");

	if ( element -> Prev != NULL )
	    (element -> Prev) -> Next = element -> Next;
	else
	    STMconnStates[spktyp].BindingQ = element -> Next;
	if ( element -> Next != NULL )
	    (element -> Next) -> Prev = element -> Prev;
	if ( freeSpace )
	    free(element);
	return(1);
}

/*
 * SearchBindingQ searches through the binding queue (ordered on nonincreasing
 * NumIdValues), whose head is stored in STMconnStates[spktyp].BindingQ,
 * looking for the first binding whose Id Values (specified by each LWP when
 * it adds a BindingQelement for itself onto its corresponding BindingQ) match
 * corresponding values obtained from the arrived packet.  The binding
 * corresponding to this element is returned and the element is removed from
 * the queue.  If Bptr is NULL, however, the first element on the binding
 * queue is chosen and the binding corresponding to this element is returned
 * (the element is also removed from the queue).
 * SearchBindingQ is called by the SpeakTypeMgr to determine which LWP an
 * arrived packet is destined for.  It is assumed that a previous call to
 * HRPC_SelectAndReadPacket added the appropriate element to the binding queue.
 * There should not be a race condition since the LWP package uses a
 * non-premptive scheduling alg and since we assume applications do not
 * relinquish the CPU in between sending a packet & calling
 * HRPC_SelectAndReadPacket.  In any event, 0 is returned if an appropriate
 * binding does not exist on the BindingQ.
 * NOTE: It is initially believed that the overhead of having a data structure
 * to store & retrieve values obtained from the arrived packet will be greater
 * than the overhead of calling procedures (redundantly) for each value test.
 * This assumption should at least be reconsidered.
 */
HRPCBinding	*SearchBindingQ(spktyp,recv_fd,Bptr)
    int 	recv_fd, spktyp;
    HRPCBinding	*Bptr;
{
	struct BindingQelement 	*temp;
	HRPCBinding		*tempfBptr;
	u_long			packetIdValue;
	int			i, IdMatchCount;

    if ( HRPC_lwpType == SingleThread )
	fatalerr("SearchBindingQ: called in SingleThread mode\n");

    temp = STMconnStates[spktyp].BindingQ;
    if ( Bptr == NULL ) {
	while ( TRUE ) {
	    if ( temp == NULL )
		return(0);
	    tempfBptr = temp->fBptr;
	    if ( tempfBptr->transDescr.sockfd == recv_fd  ||
						temp->In_Dispatcher )
		break;
	    temp = temp -> Next;
	}
    } else {
	while ( TRUE ) {
	    if ( temp == NULL )
		return(0);
	    tempfBptr = temp->fBptr;
	    if ( tempfBptr->transDescr.sockfd != recv_fd  &&
						!temp->In_Dispatcher ) {
		temp = temp -> Next;
		continue;
	    }
	    if ( tempfBptr->transType == TCPTRANSP  && !temp->In_Dispatcher  &&
			tempfBptr->connState != CONN_WREPLY  &&
			(recv_fd == connStates[recv_fd].masterfd) ) {
		    /* This is an ugly hack that should eventually be done
		     * right.  The problem is that a TCP server thread - not
		     * blocked in HRPCDispatcher - shouldn't receive a pkt
		     * that arrived on masterfd. */
		temp = temp -> Next;
		continue;
	    }
	    IdMatchCount = 0;
	    for ( i=0; i < temp->NumIdValues; i++ ) {
		(*temp->IdTable[i].getPacketIdValue)(Bptr,&packetIdValue);
		if ( packetIdValue != temp->IdTable[i].IdValue )
		    break;
		IdMatchCount++;
	    }
	    if ( IdMatchCount == temp->NumIdValues )
		break;
	    temp = temp -> Next;
	}
    }
    DeleteFromBindingQ(spktyp,temp,1);
    return(tempfBptr);
}

/*
 * CleanUpBindingQ searches through the binding queue, whose head is stored in
 * STMconnStates[spktyp].BindingQ, looking for an element whose associated
 * binding matches the specified binding.  If such an element exists on the
 * queue, it may be  removed (when remove is not 0).  The return value
 * specifies whether the binding argument was found - if found, the ptr to the
 * corresponding BindingQelement is returned; otherwise, 0 is returned.
 * This procedure is called by AddToBindingQ and the CloseLink
 * procedures.  It is needed by AddToBindingQ to prevent a binding from
 * appearing more than once in the binding queue.  This is a potential problem
 * since a binding element is not removed from the queue on a timeout.  The
 * binding element is not removed in this situation because the lwpSelectManager
 * (which awakens LWPs that have timed out) cannot correctly interpret the
 * "meaning" of a timeout (e.g., an LWP waiting for a reply on a tcp connection
 * specifies a timeout and on the occurrence of a timeout recalls
 * TcpRecvPacket - see SunInitAnswer).  Since it is possible for an LWP to
 * continue to wait after a timeout and for the signal handler to process a
 * packet arriving for a timed out LWP, the binding element is left on the
 * binding queue in case it is needed.  The CloseLink procedures call this
 * routine after closing a socket since a corresponding binding element on the
 * binding queue (if one exists) will never be needed.
 */
struct BindingQelement	*CleanUpBindingQ(fBptr,remove)
    HRPCBinding	*fBptr;
    int 	remove;
{
	int			spktyp = (int) fBptr->speaking;
	struct BindingQelement 	*temp;

	if ( HRPC_lwpType == SingleThread )
	    fatalerr("CleanUpBindingQ: called in SingleThread mode\n");

	temp = STMconnStates[spktyp].BindingQ;
	while ( TRUE ) {
	    if ( temp == NULL )
		return(temp);
	    if ( temp->fBptr == fBptr ) break;
	    temp = temp -> Next;
	}
	if ( remove )
	    DeleteFromBindingQ(spktyp,temp,1);
	return(temp);
}

/*
 * AddToTaskQ adds packet information for an arrived packet to the task queue
 * whose head is stored in STMconnStates[spktyp].TaskQ.  This procedure is
 * called by the signal handler to note the arrival of a packet and to store
 * info about this packet.  The SpeakTypeMgr walks through its corresponding
 * task queue processing (i.e., determining whom the corresponding packet is
 * for, setting its binding ptrs & awakening it) all the elements in the queue.
 * Essentially, this queue provides a means for the signal handler & the
 * SpeakTypeMgr to communicate about tasks the SpeakTypeMgr must carry out
 * when awakened.
 */
int	AddToTaskQ(spktyp,recv_fd,size,buffer,bufmark,from)
    int 		spktyp, recv_fd, size;
    memory		bufmark;
    memory		buffer;
    struct sockaddr 	from;
{
	struct PacketQelement	*new;
	STMconnDescr		*stmconnptr;
	int			sigsheldprevious;

	if ( HRPC_lwpType == SingleThread )
	    fatalerr("AddToTaskQ: called in SingleThread mode\n");

	new = (struct PacketQelement *) malloc(sizeof(struct PacketQelement));
	if ( new == NULL )
	    fatalerr("AddToTaskQ: cannot alloc PacketQelement\n");
	new -> Next = NULL;
	new -> BufSize = size;
	new -> BufMark = bufmark;
	new -> CurrentBuffer = buffer;
	new -> fd = recv_fd;
	new -> From = from;
	DISABLE_SignalHandler
	stmconnptr = &STMconnStates[spktyp];
	if ( stmconnptr->TaskQ == NULL ) {
		/* TaskQ has been emptied, so TAIL value not used */
	    new -> Prev = NULL;
	    stmconnptr->TaskQ = new;
	} else {
	    new -> Prev = TAIL_TaskQ[spktyp];
	    TAIL_TaskQ[spktyp] -> Next = new;
	}
	TAIL_TaskQ[spktyp] = new;
	ENABLE_SignalHandler
	return(1);
}

/*
 * DeleteFromTaskQ removes the specified element from its task queue.
 * This procedure is called by the SpeakTypeMgr to remove an element from the
 * corresponding task queue after it has processed the task.  Thus it is
 * assumed that signals are already held!
 */
int	DeleteFromTaskQ(spktyp,element,release)
    int				spktyp, release;
    struct PacketQelement 	*element;
{
	if ( HRPC_lwpType == SingleThread )
	    fatalerr("DeleteFromTaskQ: called in SingleThread mode\n");

	if ( element == NULL )
	    fatalerr("DeleteFromTaskQ: called with NULL element\n");

	if ( element -> Prev != NULL )
	    (element -> Prev) -> Next = element -> Next;
	else
	    STMconnStates[spktyp].TaskQ = element -> Next;
	if ( element -> Next != NULL )
	    (element -> Next) -> Prev = element -> Prev;
	if ( release )
	    free(element);
	return(1);
}

/*
 * AddToArrivedPacketQ adds packet information for an arrived packet to the
 * corresponding queue of the LWP that the packet is for.  This procedure is
 * called by a STM once it determines whom the packet is for.  Essentially,
 * this queue provides a means for the STM to pass multiple packets to a LWP
 * before they can be processed by the LWP.
 */
int	AddToArrivedPacketQ(fBptr,packetInfo)
    HRPCBinding             *fBptr;
    struct PacketQelement   *packetInfo;
{
	TransControl *tcptr = &fBptr->transDescr;

	if ( HRPC_lwpType == SingleThread )
	    fatalerr("AddToArrivedPacketQ: called in SingleThread mode\n");

	packetInfo -> Next = NULL;
	if ( tcptr->arrivedPacketQ == NULL ) {
		/* arrivedPacketQ has been emptied, so TAIL value not used */
	    packetInfo -> Prev = NULL;
	    tcptr->arrivedPacketQ = packetInfo;
	} else {
	    packetInfo -> Prev = tcptr->TAIL_arrivedPacketQ;
	    (tcptr->TAIL_arrivedPacketQ) -> Next = packetInfo;
	}
	tcptr->TAIL_arrivedPacketQ = packetInfo;
	return(1);
}

/*
 * GetFromArrivedPacketQ gets the next packet buffer from the arrivedPacketQ
 * and sets fBptr's packet pointers appropriately.  If discard is set, the
 * packet is essentially thrown away.
 */
int	GetFromArrivedPacketQ(fBptr, discard)
    HRPCBinding	*fBptr;
    int		discard;
{
	struct PacketQelement	*head;

	if ( HRPC_lwpType == SingleThread )
	    fatalerr("GetFromArrivedPacketQ: called in SingleThread mode\n");

	head = fBptr->transDescr.arrivedPacketQ;
	if ( head == NULL )
	    fatalerr("GetFromArrivedPacketQ: called with NULL Queue\n");

	if ( discard ) {
	    AddToBufferQ(fBptr->speaking,head->CurrentBuffer);
	} else {
	    fBptr->curBufSize = head->BufSize;
	    fBptr->curBufMark = head->BufMark;
	    fBptr->currentBuffer = head->CurrentBuffer;
	}
	fBptr->transDescr.arrivedPacketQ = head->Next;
	if ( head->Next != NULL )
	    (head->Next)->Prev = NULL;
	free(head);
}

/*
 * AddToBufferQ adds a preallocated buffer (of the correct size) to the buffer
 * queue whose head is stored in STMconnStates[spktyp].BufferQ, where spktyp
 * is the speaktype associated with fd.  This procedure is called by the
 * RecvPacket routines & several Rpc routines (eg, InitAnswer) to release a
 * packet buffer that is no longer needed.  If the # of buffers already on the
 * Q exceeds a threshold value, then the specified buffer is freed.  Otherwise,
 * it is added to the buffer pool associated with spktyp.  The signal handler
 * obtains a buffer from the pool each time a packet arrives on the
 * corresponding fd.
 */
int	AddToBufferQ(spktyp,buffer)
    int 	spktyp;
    memory	buffer;
{
	STMconnDescr		*stmconnptr;
	struct BufferQelement	*new;
	HRPCBinding		*temptr;
	int			threshold, sigsheldprevious;

	if ( (int) buffer <= 0 ) {
	    /* fatalerr("AddToBufferQ: called with NULL buffer\n"); */
	    return;
	}

	DISABLE_SignalHandler
	stmconnptr = &STMconnStates[spktyp];
	threshold = stmconnptr->connCount * 2;
		/* check if enough packet buffers on Q */
	if ( stmconnptr->BufferQsize >= threshold ) {
	    temptr = stmconnptr->STMbptr;
	    (*temptr->transDescr.BufDealloc)( buffer );
	    ENABLE_SignalHandler
	    return(1);
	}

		/* add packet buffer to Q */
	new = (struct BufferQelement *) malloc(sizeof(struct BufferQelement));
	if ( new == NULL )
	    fatalerr("AddToBufferQ: cannot alloc BufferQelement\n");
	new -> Prev = NULL;
	new -> Bufptr = buffer;
	new -> Next = stmconnptr->BufferQ;
	if ( stmconnptr->BufferQ != NULL )
		stmconnptr->BufferQ -> Prev = new;
	stmconnptr->BufferQ = new;
	(stmconnptr->BufferQsize)++;
	ENABLE_SignalHandler
	return(1);
}

/*
 * DeleteFromBufferQ removes the specified element from its buffer queue.
 * This procedure is called by GetFromBufferQ to remove an element from the
 * corresponding buffer queue after it has assigned a buffer for use.  Thus
 * it is assumed that signals are already held!
 */
int	DeleteFromBufferQ(spktyp,element)
    int				spktyp;
    struct BufferQelement 	*element;
{
	if ( element ==  NULL )
	    fatalerr("DeleteFromBufferQ: called with NULL element\n");

	if ( element -> Prev != NULL )
	    (element -> Prev) -> Next = element -> Next;
	else
	    STMconnStates[spktyp].BufferQ = element -> Next;
	if ( element -> Next != NULL )
	    (element -> Next) -> Prev = element -> Prev;
	free(element);
	return(1);
}

/*
 * GetFromBufferQ obtains a buffer from the buffer pool whose head is stored
 * in STMconnStates[spktyp].BufferQ, where spktyp is the speaktype associated
 * with fd.  It uses DeleteFromBufferQ to remove the buffer from its buffer
 * pool.  If there are no buffers on the BufferQ, a buffer (of the appropriate
 * size) is allocated.  This procedure is called by the signal handler to
 * obtain a buffer for an arrived packet.
 */
memory	GetFromBufferQ(spktyp)
    int	spktyp;
{
	struct BufferQelement	*head;
	memory			retBufptr;
	STMconnDescr		*stmconnptr;
	HRPCBinding		*temptr;
	int			sigsheldprevious;

	DISABLE_SignalHandler
	stmconnptr = &STMconnStates[spktyp];
	head = stmconnptr->BufferQ;
	if ( head == (struct BufferQelement *) 0 ) {
		/* out of buffers - generate a buffer of the right size */
	    temptr = stmconnptr->STMbptr;
	    retBufptr = (*temptr->transDescr.BufAlloc)();
	} else {
	    retBufptr = head -> Bufptr;
	    (stmconnptr->BufferQsize)--;
	    DeleteFromBufferQ(spktyp,head);
	}
	ENABLE_SignalHandler
	return(retBufptr);
}

/*
 * FreeConnStateQs empties all queues (BindingQ, TaskQ, BufferQ) associated
 * with STMconnStates[spktyp].  All elements on these queues are removed and
 * their allocated memory is freed.
 */
int	FreeConnStateQs(spktyp)
    int	spktyp;
{
	STMconnDescr		*stmconnptr = &STMconnStates[spktyp];
	struct BindingQelement	*BndQ_Next;
	struct PacketQelement	*TQ_Next;
	struct BufferQelement	*BufQ_Next;
	int                     sigsheldprevious;

		/* handle binding Q */
	while ( stmconnptr->BindingQ != NULL ) {
	    BndQ_Next = (stmconnptr->BindingQ)->Next;
	    free(stmconnptr->BindingQ);
	    stmconnptr->BindingQ = BndQ_Next;
	}

		/* handle task Q */
	DISABLE_SignalHandler
	while ( stmconnptr->TaskQ != NULL ) {
	    TQ_Next = (stmconnptr->TaskQ)->Next;
	    if ( (stmconnptr->TaskQ)->CurrentBuffer == NULL )
		fatalerr("FreeConnStateQs: NULL buffer on TaskQ\n");
	    free((stmconnptr->TaskQ)->CurrentBuffer); /* free buffer */
	    free(stmconnptr->TaskQ);
	    stmconnptr->TaskQ = TQ_Next;
	}
	ENABLE_SignalHandler

		/* handle buffer Q */
	DISABLE_SignalHandler
	while ( stmconnptr->BufferQ != NULL ) {
	    BufQ_Next = (stmconnptr->BufferQ)->Next;
	    if ( (stmconnptr->BufferQ)->Bufptr == NULL )
		fatalerr("FreeConnStateQs: NULL buffer on BufferQ\n");
	    free((stmconnptr->BufferQ)->Bufptr); /* free buffer */
	    free(stmconnptr->BufferQ);
	    stmconnptr->BufferQ = BufQ_Next;
	}
	ENABLE_SignalHandler
}
