/* File: /u1/oystr/HRPC/LWP/lwpSignalHandler.c  Date:  18-Aug-1987  */

/*
 * $Header$
 * FUNCTION: These procedures implement and maintain the signal
 *		handler needed by the LWP control subsystem.  In
 *		addition, routines are provided to set sockets
 *		appropriately (eg, cause the SH to awaken when a
 *		pkt arrives at the socket).
 * 
 * $Log$
 *
 *  18-Aug-1987: Initial implementation, Mark Squillante
 */

#include <stdio.h>
#include <signal.h>
#include "kEvents.h"
#include <HRPC/LWP/LWPdefs.h>
#include <HRPC/LWP/LWPtypes.h>
#include "../Transports/connDefs.h"
#include <netinet/in.h>
#include <fcntl.h>
#include <errno.h>

extern int errno;
extern char *sys_errlist[];

static int	HRPC_SignalMask = (1<<SIGIO); /* mask composed of signals */
					/* HRPC_SignalHandler responds to */

int	HRPC_LWPawakenSTMs()
{
	int	spktyp, stmask, sigsheldprevious;

	DISABLE_SignalHandler
	for ( spktyp=0; HRPC_STMsToAwakenMask; spktyp++ ) {
	    stmask = (1<<spktyp) & HRPC_STMsToAwakenMask;
	    if ( stmask == 0 ) continue;
	    (*HRPC_lwpDescr.Signal)(spktyp);
	    HRPC_STMsToAwakenMask &= ~stmask;
	}
	HRPC_SIGNALdelivered = 0;
	ENABLE_SignalHandler
	return(0);
}

int	HRPC_AddToREADFDS(fd)
    int	fd;
{
	if ( HRPC_lwpType != SingleThread )
	    HRPC_READFDS |= (1 << fd);
	else
	    fatalerr("HRPC_AddToREADFDS: called in SingleThread mode\n");
}

int	HRPC_RmFromREADFDS(fd)
    int	fd;
{
	if ( HRPC_lwpType != SingleThread )
	    HRPC_READFDS &= ~(1 << fd);
	else
	    fatalerr("HRPC_RmFromREADFDS: called in SingleThread mode\n");
}

/*
 * HRPC_SignalHandler determines which ports have received packets and copies
 * the contents into the pre-allocated memory locations.  The appropriate
 * SpeakTypeMgr (one per file descriptor - well almost) is awakened (via
 * HRPC_STM_Signal).  This procedure is (indirectly) invoked upon the arrival
 * of a packet.
 */
HRPC_SignalHandler(signo)
    int	signo;
{
	int			nfds, fd, cnt, tempfd, readfds, zero = 0;
	static struct timeval	Poll = { 0, 0 };
	memory			bufptr, bufmark;
	struct sockaddr		from;
	int			spktyp;
	extern memory		GetFromBufferQ();
	extern int		AddToTaskQ();

    if ( HRPC_lwpType == SingleThread )
	fatalerr("HRPC_SignalHandler: called in SingleThread mode\n");

    HRPC_SIGNALdelivered = 1;  /* Note that HRPC signal handler has executed */
    lwpSelectManager_timeout.tv_sec = 0;  /* Cause select in lwpSelectManager process to */
    lwpSelectManager_timeout.tv_usec = 0; /* return immediately */

    readfds = HRPC_READFDS;
    nfds = select(MAX_FDS, &readfds, &zero, &zero, &Poll);
    if( nfds < 0 ) { perror("\tHRPC_SignalHandler");
	printf("\tHRPC_SignalHandler: return val = %d\n",nfds);
	printf("\tHRPC_SignalHandler: returned rmask = %d\n",readfds);
	}
    for ( fd=0; nfds > 0 && fd < MAX_FDS; fd++ ) {
	tempfd = (1<<fd) & readfds;
	if ( tempfd == 0 ) continue;
	    /* sanity check stolen from RecvPacket routines */
	if ( (tempfd&HRPC_READFDS) == 0 )
	    fatalerr("HRPC_SignalHandler: select mask mismatch.\n");
	nfds--;
	spktyp = (int) (connStates[fd].connbptr)->speaking;
	do {
		/* get next buffer from connStates[spktyp].BufferQ */
	    bufptr = GetFromBufferQ(spktyp);
		/* do actual read from socket, return curBufSize */
	    cnt = (*connStates[fd].ReadProc) (fd,bufptr,&bufmark,&from);
		/* add this packet task to TaskQ of appropriate STM */
	    AddToTaskQ(spktyp,fd,cnt,bufptr,bufmark,from);
	} while ( select(fd,&tempfd,&zero,&zero,&Poll) > 0 );
	    /* cause appropriate SpeakTypeMgr to be awakened */
	if ( connStates[fd].masterfd == fd ) {
		/* cause STM to be awakened */
	    HRPC_STMsToAwakenMask |= (1 << spktyp);
	} else {
		/* make sure we have a TCP connection - otherwise have error */
	    if ( (connStates[fd].connbptr)->transType == TCPTRANSP ) {
		    /* cause STM to be awakened */
		HRPC_STMsToAwakenMask |= (1 << spktyp);
	    } else {
		/* log error */
                fatalerr("HRPC_SignalHandler: socket mismatch.\n");
	    }
	}
    }
}

/*
 * HRPC_SetUpSignalHandler performs the necessary operations to cause
 * HRPC_SignalHandler to be (indirectly) invoked upon the arrival of a packet.
 * HRPC_RealSigHandler is installed to facilitate the use of HoldSigs() &
 * ReleaseSigs().
 */
int	HRPC_SetUpSignalHandler()
{
	int		i;
	struct sigvec	sv;

    sv.sv_handler = HRPC_RealSigHandler;
    sv.sv_mask = -1;		/* mask all signals */
    sv.sv_onstack = 0;
    for ( i=0; i < NSIG; i++ ) {
	if ( (1<<i) & HRPC_SignalMask )
	    sigvec(i, &sv, 0);
    }
    return 0;
}

/*
 * HRPC_CancelSignalHandler performs the necessary operations to turn off 
 * the invocation of HRPC_SignalHandler upon the arrival of a packet.
 */
int	HRPC_CancelSignalHandler()
{
	int		i;
	struct sigvec	sv;

    sv.sv_handler = SIG_DFL;
    for ( i=0; i < NSIG; i++ ) {
	if ( (1<<i) & HRPC_SignalMask )
	    sigvec(i, &sv, 0);
    }
    return 0;
}

/*
 * Set up socket for asynchronous operation and possibly non-blocking I/O
 */
int	HRPC_SetSocketAsync(fd,blockflag)
    int	fd, blockflag;
{
	int	arg;

	if ( HRPC_lwpType == SingleThread )
	    fatalerr("HRPC_SetSocketAsync: called in SingleThread mode\n");

	if ( blockflag == 0 ) /* do not set for non-blocking I/O */
	    arg = FASYNC;
	else /* set for non-blocking I/O */
	    arg = FASYNC | FNDELAY;

	if ( fcntl(fd, F_SETFL, arg) < 0 )
	    fatalerr("HRPC_SetSocketAsync: error setting descr status flags.\n");

	if ( fcntl(fd, F_SETOWN, getpid()) < 0 )
	    fatalerr("HRPC_SetSocketAsync: error setting receiver of SIGIO.\n");

	return(0);
}

/*
 * Set up socket for non-blocking I/O
 */
int	HRPC_SetSocketNdelay(fd)
    int	fd;
{
	if ( HRPC_lwpType == SingleThread )
	    fatalerr("HRPC_SetSocketNdelay: called in SingleThread mode\n");

	if ( fcntl(fd, F_SETFL, FASYNC | FNDELAY) < 0 )
	    fatalerr("HRPC_SetSocketNdelay: error setting descr status flags.\n");

	return(0);
}
