/* File: /u1/oystr/HRPC/LWP/lwpSelect.c  Date:  18-Aug-1987  */

/*
 * $Header$
 * FUNCTION: These procedures provide a general select mechanism
 *		needed by the LWP control subsystem.  For the time
 *		being, there are two versions of HRPC_SelectAndReadPacket
 *		and HRPC_Select; one for lwpType=SingleThread, and one
 *		for multiple threads (regardless of LWP package used).
 *		The proper version is maintained in the lwpControl block.
 * 
 * $Log$
 *
 *  18-Aug-1987: Initial implementation, Mark Squillante
 */

#include <stdio.h>
#include <signal.h>
#include "kEvents.h"
#include <HRPC/LWP/LWPdefs.h>
#include <HRPC/LWP/LWPtypes.h>
#include "../Transports/connDefs.h"
#include <netinet/in.h>

	/* Multi Thread versions of HRPC_SelectAndReadPacket & HRPC_Select */
/*
 * Logically, MT_SelectAndReadPacket causes the calling lightweight process
 * to block until a packet arrives for it (based on IdTable contents) or until
 * the timeout period expires, whichever occurs first.  If a corresponding
 * packet arrives (w/o error), MT_SelectAndReadPacket passes the packet to
 * fBptr by setting it's buffer ptrs.  To provide this functionality, 
 * MT_SelectAndReadPacket constructs an lwpIoRequest block, inserts it into a
 * timer list and waits on fBptr.  (NOTE: it is assumed that the RecvPacket
 * procedures call AddToBindingQ & set the corresponding IdTable before
 * calling MT_SelectAndReadPacket)  If a packet (or a read error) arrives
 * for the calling process before timing out, the corresponding SpeakTypeMgr
 * will awaken the calling process.  Otherwise (a timeout has occurred), the
 * lwpSelectManager will awaken the calling process.  Upon being awakened, the
 * cause of being awakened is determined - if a timeout occurred, selectStatus
 * is set to 0 and MT_SelectAndReadPacket returns (note that selectStatus will
 * never be negative in the multiple thread case); otherwise,
 * MT_SelectAndReadPacket grabs the next packet on fBptr's arrivedPacketQ.  If
 * an error was encountered upon reading the packet, this is reflected in
 * readStatus and MT_SelectAndReadPacket returns.  Otherwise,
 * MT_SelectAndReadPacket sets fBptr's buffer ptrs to the information obtained
 * from the arrivedPacketQ and then returns.
 * NOTE: The data structures & routines provided in the ITC LWP package are
 *   used, even though some of the structures are only partially applicable.
 * NOTE: MT_Select is used to provide the "select" specific functions
 *   described above.
 */
void	MT_SelectAndReadPacket(fBptr, timeout, readmask, selectStatus, readStatus)
    HRPCBinding		*fBptr;
    struct timeval	*timeout;
    int			*readmask, *selectStatus, *readStatus;
{
    TransControl		*tcptr = &fBptr->transDescr;
    int				readCnt, recv_fd, readables;
    void			MT_Select();
    HRPCBinding			*fBptrArr[2];

    readables = *readmask;
    fBptrArr[0] = fBptr;
    fBptrArr[1] = (HRPCBinding *) NULL;
    MT_Select(fBptrArr, timeout, readmask, selectStatus);

	/* return on timeout */
    if ( *selectStatus <= 0 ) return;

    recv_fd = (tcptr->arrivedPacketQ)->fd;
    readCnt = (tcptr->arrivedPacketQ)->BufSize;

	/* carry out the appropriate "read" semantics */
    tcptr->tmp_sockfd = -1;
    if ( fBptr->transType == TCPTRANSP  &&  fBptr->connState != CONN_WREPLY  &&
		connStates[recv_fd].masterfd == connStates[recv_fd].connfd  &&
		readCnt <= CONN_CNTLPACK ) {
	    /* the packet arrived on a master connection & is a control packet,
	     * i.e., nothing was actually read.  This shouldn't happen here,
	     * so we abort */
	fatalerr("MT_SelectAndReadPacket: received control packet.\n");
    } else {
	    /* release current packet buffer & get packet from arrivedPacketQ */
        tcptr->packetArrived--;
        AddToBufferQ(fBptr->speaking,fBptr->currentBuffer);
        GetFromArrivedPacketQ(fBptr,0);

	    /* check if error occurred when SignalHandler did read */
        if ( fBptr->curBufSize <= 0 ) {
            *readStatus = fBptr->curBufSize;
            fBptr->curBufSize = 0;
        } else {
            *readStatus = 1;	/* no read error */
        }
    }
}
/*
 * See above comments for MT_SelectAndReadPacket.  MT_Select is called
 * directly by HRPCDispatcher & is used by MT_SelectAndReadPacket.
 */
void    MT_Select(fBptrArr, timeout, readmask, selectStatus)
    HRPCBinding		*fBptrArr[];
    struct timeval	*timeout;
    int			*readmask, *selectStatus;
{
    register struct lwpIoRequest	*request;
    register int			idxArr;
    TransControl			*tcptr;
    int					recv_fd;
    struct PacketQelement		*PQ_head;

	/* check specified readmask */
    if ( (*readmask&HRPC_READFDS) != *readmask ) {
	/* fatalerr("MT_Select: HRPC_READFDS doesn't include all of specified readmask.\n"); */
    }

	/* check to see if packet already arrived - if so, grab FIRST fBptr
	 * and "process" it */
    for ( idxArr=0; fBptrArr[idxArr] != (HRPCBinding *) NULL; idxArr++ ) {
	tcptr = &(fBptrArr[idxArr]->transDescr);
	if ( tcptr->packetArrived != 0 )
	    break;
    }
    if ( fBptrArr[idxArr] != (HRPCBinding *) NULL ) {
	if ( tcptr->arrivedPacketQ == (struct PacketQelement *) 0 )
	    fatalerr("MT_Select: arrivedPacketQ is empty\n");
	recv_fd = (tcptr->arrivedPacketQ)->fd;
	*readmask = 1 << recv_fd;
	*selectStatus = 1;	/* no select error */
	return;
    }

    	/* Construct request block & insert */
    request = lwpNewRequest();
    if (timeout == NIL) {
	request -> timeout.TotalTime.tv_sec = -1;
	request -> timeout.TotalTime.tv_usec = -1;
    } else
	request -> timeout.TotalTime = *timeout;
    request -> timeout.BackPointer = (char *) request;
    request -> HRPC_WaitCondition = fBptrArr[0];
#ifdef DEBUG
    request -> timeout.Next = (struct timerElem *) 1;
    request -> timeout.Prev = (struct timerElem *) 1;
#endif DEBUG
    timerInsert(lwpRequests, &request->timeout);

    HRPC_LWPawakenSTMs();

    	/* wait until a packet arrives for ANY fBptr or until timeout */
    (*HRPC_lwpDescr.Mwait)(1, fBptrArr);

	/* determine cause of being awakened, free request block, & return */
	/* if packet arrived, grab FIRST fBptr and "process" it */
    for ( idxArr=0; fBptrArr[idxArr] != (HRPCBinding *) NULL; idxArr++ ) {
	tcptr = &(fBptrArr[idxArr]->transDescr);
	if ( tcptr->packetArrived != 0 )
	    break;
    }
    if ( fBptrArr[idxArr] != (HRPCBinding *) NULL ) {
	    /* remove request block - if timeout occurred then entry was
	     * removed by lwpSelectManager, however, no harm done if
	     * lwpSelectManager already removed element (could happen if STM
	     * reads a packet for LWP & lwpSelectManager determines LWP had
	     * timedout before LWP gets CPU) */
	timerRemove(lwpRequests, &request->timeout);
	if ( (PQ_head=tcptr->arrivedPacketQ) == (struct PacketQelement *) 0 )
	    fatalerr("MT_Select: arrivedPacketQ is empty\n");
	recv_fd = PQ_head->fd;
	*readmask = 1 << recv_fd;
	*selectStatus = 1;	/* no select error */
    } else {
	*selectStatus = 0;	/* timeout */
    }
    free(request);
}

	/* Single Thread versions of HRPC_SelectAndReadPacket & HRPC_Select */
/*
 * Logically, ST_SelectAndReadPacket causes the calling process to block
 * until a packet arrives for it or until the timeout period expires,
 * whichever occurs first.  If a corresponding packet arrives (w/o error),
 * ST_SelectAndReadPacket passes the packet to fBptr by setting it's buffer
 * ptrs.  To provide this functionality, ST_SelectAndReadPacket performs a
 * UNIX select (using the socket associated with the specified binding as a
 * read mask) and returns on a timeout or error, after setting selectStatus
 * to reflect the return status of the UNIX select.  If a timeout or error is
 * not encountered, ST_SelectAndReadPacket then uses ReadProc (associated
 * with the specified binding) to read the packet.  If a read error is
 * encountered, this is reflected in readStatus and ST_SelectAndReadPacket
 * returns.  Otherwise, ST_SelectAndReadPacket sets the binding's buffer
 * ptrs before returning.
 * NOTE: ST_Select is used to provide the "select" specific functions
 *   described above.
 */
void	ST_SelectAndReadPacket(fBptr, timeout, readmask, selectStatus, readStatus)
    HRPCBinding		*fBptr;
    struct timeval	*timeout;
    int			*readmask, *selectStatus, *readStatus;
{
	int	fd, byteCnt, readables;
	memory	bufptr, bufmark;
	struct sockaddr		sourceAddr;
	struct sockaddr_in	*from, *peer;
	void			ST_Select();
	HRPCBinding		*fBptrArr[2];

	fd = fBptr->transDescr.sockfd;
	readables = *readmask;
	fBptrArr[0] = fBptr;
	fBptrArr[1] = (HRPCBinding *) NULL;
	ST_Select(fBptrArr, timeout, readmask, selectStatus);

	    /* return on error or timeout */
	if ( *selectStatus <= 0 ) return;

	    /* sanity check stolen from RecvPacket routines */
	if ( readables != *readmask ) {
	    fatalperr("ST_SelectAndReadPacket: select mask mismatch.\n");
	}

		/* read packet */
	bufptr = fBptr->currentBuffer;
	byteCnt = (*connStates[fd].ReadProc) (fd,bufptr,&bufmark,&sourceAddr);

		/* return on read error */
	if ( byteCnt <= 0 ) {
	    *readStatus = byteCnt;
	    return;
	} else
	    *readStatus = 1;	/* no read error */

		/* set binding ptrs */
	fBptr->curBufSize = byteCnt;
	fBptr->curBufMark = bufmark;
	if ( fBptr->transType == UDPTRANSP ) {
            if ( fBptr->connState == CONN_WREPLY ) {
		from = (struct sockaddr_in *) &sourceAddr;
                    /* sanity check stolen from UdpRecvPacket */
		peer = (struct sockaddr_in *) (&fBptr->transDescr.netAddr);
		if ( (from->sin_addr.s_addr != peer->sin_addr.s_addr) ||
                                (from->sin_port != peer->sin_port) )
                    fatalerr("Udp STM: address mismatch.\n");
            }
            else {
		fBptr->transDescr.replyTo = sourceAddr;
            }     
	}
}
/*
 * See above comments for ST_SelectAndReadPacket.  ST_Select is called
 * directly by HRPCDispatcher & is used by ST_SelectAndReadPacket.
 */
void	ST_Select(fBptrArr, timeout, readmask, selectStatus)
    HRPCBinding		*fBptrArr[];
    struct timeval	*timeout;
    int			*readmask, *selectStatus;
{
	int	cnt, zero = 0;

	    /* perform select */
	cnt = select(MAX_FDS, readmask, &zero, &zero, timeout);

	if ( cnt <= 0 )
	    *selectStatus = cnt;
	else
	    *selectStatus = 1;	/* no select error */
}
