--
-- Courier description for BIND message structures.  These definitions don't
-- include the TCP-specific byte-count that preceeds BIND TCP messages; that is
-- found in the TCP BIND Courier description file.
--

BIND_MsgStructs : PROGRAM 20190 VERSION 1 =

BEGIN
	HdrBitFields : TYPE = USERPROC HdrBitFieldsMarshaller
	-- The fields of this record get bit packed, so must be marshaled
	-- by a USERPROC
		RECORD [
			-- Field			# bits	BIND name
			ResponseFlag	: INTEGER,	-- 1	qr
			OpCode 		: INTEGER,	-- 4	opcode
			AuthAnsFlag	: INTEGER,	-- 1	aa
			TruncMsgFlag	: INTEGER,	-- 1	tc
			RecurDesFlag	: INTEGER,	-- 1	rd
			RecurAvailFlag	: INTEGER,	-- 1	ra
			PrimSvrRqdFlag	: INTEGER,	-- 1	pr
			Unused		: INTEGER,	-- 2	unused
			ResponseCode	: INTEGER	-- 4	rcode
		];
	
	-- Domain name; marshalling = compression/expansion
	DName : TYPE = USERPROC DnameMarshaller
		STRING;

	-- Header (corresponds to what BIND calls a HEADER)
	MsgHeader : TYPE = RECORD [
		QueryId : INTEGER,
		BitFieldsPart : HdrBitFields,
		qdcount	: INTEGER,	-- Number of question entries
		ancount	: INTEGER,	-- Number of answer entries
		nscount	: INTEGER,	-- Number of authority entries
		arcount	: INTEGER	-- Number of resource entries
	];
	
	-- Resource Record identifier: name, class, type
	RRecIdent : TYPE = RECORD [
		dname	: DName,
		type	: INTEGER,
		class	: INTEGER
	];

	-- Resource Record: Identifier followed by ttl, nbytes, and data.
	-- Need my own USERPROC for 2 reasons: (1) The data part is generic;
	-- it is marshalled by client-level routines (which are called by
	-- RRecMarshaller).  (2) Nbytes is skipped and then filled in by
	-- "backing up" in the USERPROC, relieving the client from filling
	-- in this byte count.
	RRec : TYPE = USERPROC RRecMarshaller
	RECORD [
		RR_Ident : RRecIdent,
		ttl : LONG INTEGER,	-- Time to live
		nbytes : INTEGER,	-- Size (in bytes) of Data
		-- Note: the Data field is a generic buffer for a user-defined
		-- structure, the marshalling routine for which gets determined
		-- at call time, and called by LookupOutMsgMarshaller.  We call
		-- it an ARRAY x OF UNSPECIFIED so that space is allocated for
		-- it in the message structure definitions, where x is the
		-- largest number of 16 bit words we could expect.  Use 256
		-- bytes (BIND's PACKETSZ) (= 128 UNSPECIFIEDs), since that's
		-- the most data BIND is willing to store.  Note that the
		-- space for this data area is preallocated, and hence does
		-- not need to be free'd after each lookup
		Data : ARRAY 128 OF UNSPECIFIED
	];




	-- Note: The name components "In" and "Out" below are relative to
	-- BIND, i.e., they are inputs and outputs to BIND



	LookupNameInMsg : TYPE = RECORD [
		Header : MsgHeader,
		Q_Ident : RRecIdent
	];

	-- Need my own USERPROC for the LookupNameOutMsg record type because
	-- the number of RRecs to be demarshaled for the result is dependent
	-- on Header.ancount (which doesn't fall in the right place to use
	-- the SEQUENCE definition).
	LookupNameOutMsg : TYPE = USERPROC LookupNameOutMsgMarshaller
		RECORD [
			Header : MsgHeader,
			Q_Ident : RRecIdent,
			AnsRecs : ARRAY 100 OF RRec	-- Actually, any number
							-- (up to 100) of
							-- RRecs can come back;
							-- we declare it as an
							-- ARRAY so it will be
							-- a pointer.  In
							-- reality, it's a
							-- sequence with ancount
							-- for the length field.
							-- The 100 also causes
							-- enough space to be
							-- allocated.
		];

	-- Definition of structure that gets stored in the cache.  This is
	-- different than above because the cache is preloaded with zone
	-- transfer data, and zone transfers bring back the data in this format
	CacheLookupNameOutMsg : TYPE = USERPROC CacheLookupNameOutMsgMarshaller
		RECORD [
			Header : MsgHeader,
			AnsRecs : ARRAY 100 OF RRec
		];

	AddRRInMsg : TYPE = RECORD [
			Header : MsgHeader,
			RR : RRec
		];

	AddRROutMsg : TYPE = RECORD [
			Header : MsgHeader
		];

	DeleteRRInMsg : TYPE = RECORD [
			Header : MsgHeader,
			RR : RRec
		];

	DeleteRROutMsg : TYPE = RECORD [
			Header : MsgHeader
		];

	ModifyInMsg : TYPE = RECORD [
			Header : MsgHeader,
			OldRR : RRec,
			NewRR : RRec
		];

	ModifyOutMsg : TYPE = RECORD [
			Header : MsgHeader
		];
END.
