/* listfuncs.c */

#include "structs.h"

#define MALLOC(ptr, type, num, errmsg) \
                if ((ptr = (type *) malloc( num * sizeof(type))) == NULL) {  \
                        fprintf(stderr,"malloc error: %s\N",errmsg);    \
                    }

#define ELAPSED(start,cur,elapsed) \
  elapsed.tv_sec = cur.tv_sec - start.tv_sec; \
  elapsed.tv_usec = cur.tv_usec - start.tv_usec; \
  if (elapsed.tv_usec < 0) { \
      printf ("."); \
      elapsed.tv_usec += 1000000; \
      elapsed.tv_sec--; }

#define MAX(a,b) ((a) > (b) ? (a) : (b))

#define PREROLL    4
#define POSTROLL   1
#define PRESWITCH  1
#define SETUP      150

#define FR_USEC (1000000.0 / 30.0)

/* "actions", "events", and "commands" explained:
**
** The server receives a list of "actions" from the client. 
** Each action contains a list of "events" which contain 
** specific insturctions.
**
** The first "event" in an action is always an instruction to
** run a disc segment starting with a specific frame.  There then 
** follow any number of instructions to switch to the disc,
** and the last event contains a "RELEASE" instruction.
**
** When the action list is executed, each event is translated to
** one or more "commands."
**
** hair-raisingly-complicated-but-efficient data structure:
**
**  action-script-head
**    |
**    |
**    V
**   action --------> event
**    |	               |
**    |                V
**    |               event
**    |                 |
**    |                 V
**    |               event
**    |                |
**    |                V
**    |               NULL
**    |
**    V
**  action -----> event
**    |             |
**    |             V
**    |            event
**    |              |
**    |              V
**    |            event
**    |             |
**    |             V
**    |            NULL
**    V
**  NULL
**
*/

int
runlist(tlist,mgsock)
command *tlist;
int mgsock;
{
  register struct timeval starttime;
  register struct timeval elapsedtime;
  register struct timeval curtime;
  register struct timeval waittime;
  register fd_set readfds;
  register fd_set temp;
  extern short DEBUG;

  FD_ZERO(&readfds);
  FD_SET(mgsock,&readfds);

  /* set elapsed time to 0 */
  gettimeofday(&starttime,0);
  
  while (tlist->next != NULL) {
    tlist = tlist->next;
    /* copy the descriptor set so we don't trash it */
    bcopy( (char *) &readfds, (char *) &temp, sizeof(fd_set) );
    gettimeofday(&curtime,0);
    ELAPSED(starttime,curtime,elapsedtime);  
    ELAPSED(elapsedtime,tlist->time,waittime);
    if (waittime.tv_sec < 0) {
      waittime.tv_sec = 0;
      waittime.tv_usec = 0;
    }

    if (DEBUG) {
      printf ("starting at %d.%d seconds -- ",
	      tlist->time.tv_sec, tlist->time.tv_usec);
      printf ("waiting for %d.%d seconds -- ",
	      waittime.tv_sec, waittime.tv_usec);
      fflush(stdout);
    }

    if(select(mgsock+1,&temp,NULL,NULL,&waittime)) {
      /* something is on the socket,
      ** so stop everything and return */
      if (DEBUG) printf("runlist: interrupted by client\n");
      all_stop();
      return(NO_ERROR);
    }
    else {
      /* do the action */
      switch (tlist->action) {
        case SEARCH:
          if ((*tlist->rpd->search)(tlist->rpd,tlist->argument,0) == -1)
            return(sendint(mgsock,COULDNT_SEARCH));
	  if (DEBUG) printf ("searching %d",tlist->argument);
	  break;
	case ROLL:
	  (*tlist->rpd->varspeed)(tlist->rpd,tlist->argument2); 
	  if (DEBUG) printf ("rolling %d speed %d",
			     tlist->argument,tlist->argument2);
	  break;
	case RELEASE:
	  (*tlist->rpd->varspeed)(tlist->rpd,STOP);
	  if (DEBUG) printf ("releasing %d",tlist->argument);
	  break;
        case SWITCH:
	  doswitchtree(tlist->out,tlist->argument);
	  if (DEBUG) printf ("switching %d",tlist->argument);
	  break;
	default:
	  if (DEBUG) printf("bad action in timing list: %d\n",tlist->action);
	  break;
      }
      if (DEBUG) {
	gettimeofday (&curtime,0); 
	ELAPSED (starttime,curtime,curtime);
	printf (" at %d %d\n", curtime.tv_sec,curtime.tv_usec );
      }
    }
  }
  return(NO_ERROR);
}  

command *
compile_script(script)
action *script;
{
  command *tlist; 	/* timing list */
  command *tcom;	/* a command in the timing list */
  script_event  *ev;
  int vol, res;
  struct timeval zerotime;
  action *act;
  extern int num_volumes;
  extern DeviceList *volumes;

  MALLOC(tlist,command,1,"in compile_script");
  tlist->next = NULL;
  tcom = tlist;

  frame_to_time(0,&zerotime);

  for (vol=0; vol < num_volumes; vol++) {
    act = script->next;
    while (act != NULL) {
      if (act->source_vol == vol) {
        for(res=0; res<volumes[vol].num_resources; res++) {
      	  assign_free_player(tlist, script, vol, 
                       res, &zerotime);
        }
        break;
      }
      act = act->next;
    }
  }

  while(tcom->next != NULL) {
    tcom = tcom->next;
    if (tcom->action == RELEASE) {
      assign_free_player(tlist, script, tcom->volume, 
                       tcom->resource, &tcom->time);
    }
  }

  return(tlist);
}

int
assign_free_player(tlist,script,vol,res,time)
command *tlist;
action *script;
int vol, res;
struct timeval *time;
{
  command *com;
  script_event *ev;
  action *act;
  action *last;
  int found_one;
  extern int num_volumes;
  extern DeviceList *volumes;
  extern Output rpd_out[];
  extern RPD_ptr players[];

  found_one = 0;
  act = script;

  /* find the next action which can be assigned to the player */
  while(act->next != NULL) {
    last = act;
    act = act->next;
    if (act->source_vol == vol) {
      /* pull act out of the list */
      found_one = 1;
      last->next = act->next;
      break;
    }
  }

  if (found_one == 0)
    return(0);  /* no actions could be assigned to this player */

  ev = act->events;
  while (ev != NULL) {
    switch(ev->action) {
      case ROLL:
        /* add search to commands */
        MALLOC(com,command,1,"assign_free_players");
	com->action = SEARCH;
	com->argument = ev->arg - PREROLL*ev->arg2/30;
        bcopy(time,&com->time,sizeof(struct timeval));
	com->rpd = players[volumes[vol].resource[res]];
	com->out = rpd_out[volumes[vol].resource[res]];
  	com->volume = vol;
 	com->resource = res;
	insert_in_tlist(tlist,com);

        /* now calculate preroll and add to commands */  
        MALLOC(com,command,1,"assign_free_players");
	com->action = ROLL;
        frame_to_time((ev->target_frame + SETUP - PREROLL*ev->arg2/30),
		      &com->time);
	com->rpd = players[volumes[vol].resource[res]];
	com->out = rpd_out[volumes[vol].resource[res]];
  	com->volume = vol;
 	com->resource = res;
	com->argument2 = ev->arg2;
	insert_in_tlist(tlist,com);
        break;
      case SWITCH:
        MALLOC(com,command,1,"assign_free_players");
        com->action = SWITCH;
        com->argument = ev->arg;
        frame_to_time((ev->target_frame + SETUP - PRESWITCH),&com->time);
	com->rpd = players[volumes[vol].resource[res]];
	com->out = rpd_out[volumes[vol].resource[res]];
  	com->volume = vol;
 	com->resource = res;
	insert_in_tlist(tlist,com);
        break;
      case RELEASE:
        MALLOC(com,command,1,"assign_free_players");
        com->action = RELEASE;
        frame_to_time((ev->target_frame + SETUP + POSTROLL),&com->time);
	com->rpd = players[volumes[vol].resource[res]];
	com->out = rpd_out[volumes[vol].resource[res]];
  	com->volume = vol;
 	com->resource = res;
	insert_in_tlist(tlist,com);
        break; 
      default:
	fprintf(stderr,"logic error: unknown command in assign_free_players: %d\n",ev->action);
	return(-1);
        break;
    }
    ev = ev->next;
  }
  return(1);
}  

int
insert_in_tlist(tlist,com)
command *tlist, *com;
{
  command *cur;
  command *prev;
  command dummy;
  extern short DEBUG;
  extern void print_timing_list();

  /*
  if (DEBUG) {
	printf("insert_in_tlist: inserting this item in list:\n");
	com->next = NULL;
	dummy.next = com;
	print_timing_list(&dummy);
	printf("--------------------------------------------------\n");
  }
  */

  cur = tlist;

  while(cur->next != NULL) {
    prev = cur;
    cur = cur->next;
    if (        ((com->time.tv_sec == cur->time.tv_sec) && 
		 (com->time.tv_usec < cur->time.tv_usec)) 
		|| (com->time.tv_sec < cur->time.tv_sec)) {
      com->next = cur;
      prev->next = com; 
      /* 
      if (DEBUG) print_timing_list(tlist);
      */
      return(0);
    } 
  }
  /* cur->next == NULL */
  cur->next = com;
  com->next = NULL;

  /* 
  if (DEBUG) print_timing_list(tlist);
  */
  return(0);
}

int
frame_to_time(frame,time)
int frame;
struct timeval *time;
{
  time->tv_sec = frame/30;
  time->tv_usec = (int) (((float)frame/30.0 - (float)time->tv_sec)
			 * 30 * FR_USEC);
}


action *
create_action_script(msgsock)
int msgsock;
{
	action *act, *script;
	script_event *ev, *prev;
	int target_fr;
	int vol, source_fr;
	int ints[4], netints[4];

	MALLOC(script,action,1,"read_actions");
	act = script;
	act->next = NULL;

	read(msgsock,netints,sizeof(int));
	nettohostcom(netints,ints,1);
	while (ints[0] != END_OF_TRANSMISSION) {
		MALLOC(act->next,action,1,"read_action");
		act = act->next;
		act->next = NULL;
		act->source_vol = ints[0];
		MALLOC(act->events,script_event,1,"read_action");
		ev = act->events;
		do {
			read(msgsock,netints,4*sizeof(int));
			nettohostcom(netints,ints,4);
			ev->target_frame = ints[0];
			ev->action = ints[1];
			ev->arg = ints[2];
			ev->arg2 = ints[3];
			MALLOC(ev->next,script_event,1,"read_action");
			prev = ev;
			ev = ev->next;
		} while (ints[1] != RELEASE);
		free(ev->next);
		prev->next = NULL;
		read(msgsock,netints,sizeof(int));
		nettohostcom(netints,ints,1);
	}
	return(script);
}


void
free_script(script)
action *script;
{
	action *act, *prevact;
	script_event *ev, *prevev;
	char str[10];
	
	act = script;
	
	while (act->next != NULL) {
		prevact = act;
		act = act->next;
		free(prevact);
		ev = act->events;
		while(ev != NULL) {
			prevev = ev;
			ev = ev->next;
			free(prevev);
		}
	}
	free(act);
}
						
void
print_script(script)
action *script;
{
	action *act;
	script_event *ev;
	char str[10];
	
	act = script;
	
	while (act->next != NULL) {
		act = act->next;
		ev = act->events;
		while(ev != NULL) {
			switch(ev->action) {
				case ROLL:
					printf("%-8d ROLL    %-8d %-8d %-8d\n",
						ev->target_frame,
						act->source_vol,
						ev->arg,
					       ev->arg2);
					break;
				case SWITCH:
					str[0] = '\0';
					if (ev->arg & RIGHT_CHAN)
						strcat(str,"L");
					if (ev->arg & LEFT_CHAN)
						strcat(str,"R");
					if (ev->arg & VIDEO_CHAN)
						strcat(str,"V");
					printf("%-8d SWITCH   %-8s\n",
						ev->target_frame,str);
					break;
				case RELEASE:
					printf("%-8d RELEASE\n",ev->target_frame);
					break;
				default:
					printf("bad data in script\n");
					break;
			}
			ev = ev->next;
		}
	}
}
						
void
free_timing_list(tlist)
command *tlist;
{
	char str[10];
	command *com, *prevcom;

	if (tlist == NULL)
	  return;

	com = tlist;

	while(com->next != NULL) {
		prevcom = com;
		com = com->next;
		free(prevcom);
	}
	free(com);
}

void
print_timing_list(tlist)
command *tlist;
{
	char str[10];
	command *com;

	com = tlist;

	while(com->next != NULL) {
		com = com->next;
		switch (com->action) {
			case SEARCH:
				printf("%3d.%06d    SEARCH    vol: %-3d res: %-3d frame: %d\n",
					(int)com->time.tv_sec, (int)com->time.tv_usec,
					com->volume, com->resource, com->argument);
				break;
			case ROLL:
				printf("%3d.%06d    ROLL    speed: %-8d  vol: %-3d res: %-3d\n",
				       (int)com->time.tv_sec,
				       (int)com->time.tv_usec,
				       com->arg2,
				       com->volume, com->resource);
				break;
			case SWITCH:
				str[0] = '\0';
				if (com->argument & RIGHT_CHAN)
					strcat(str,"L");
				if (com->argument & LEFT_CHAN)
					strcat(str,"R");
				if (com->argument & VIDEO_CHAN)
					strcat(str,"V");
				printf("%3d.%06d    SWITCH    vol: %-3d res: %-3d mask: %s\n",
					(int)com->time.tv_sec, (int)com->time.tv_usec,
					com->volume, com->resource, str);
				break;
			case RELEASE:
				printf("%3d.%06d    RELEASE   vol: %-3d res: %-3d\n",
					(int)com->time.tv_sec, (int)com->time.tv_usec,
					com->volume, com->resource);
				break;
			default:
				printf("bad data in timing list\n");
				break;
		}
	}
}

all_stop()

{
  extern RPD_ptr players[];
  extern int num_players;
  int i;

  for (i = 0; i < num_players; i++) 
    if (players[i] != NULL) {
      (*players[i]->varspeed)(players[i],STOP);
    }
}
