/* Copyright (c) 1989 by
 * Daniel I. Applebaum, Cambridge, MA.
 * All rights reserved
 *
 * This software is furnished on an as-is basis, and may be used and copied
 * only with the inclusion of the above copyright notice.
 *
 * The infomation in this software is subject to change without notice.
 * No committment is made as to the usability or reliability of this
 * software.
 *
 * Daniel I. Applebaum
 * MIT Media Laboratory, E15-443
 * 20 Ames Street
 * Cambridge, MA 02139
*/

/* Additional software by Ben Rubin and Hal Birkeland
   Date: September 88
*/

/****************************************************************
  This is the subroutine library for Galatea, the video control
  system.
****************************************************************/

/*********************** Version 2.00 **************************/

#include"Glib.h"

static short DEBUG = 0;
static int sync = 0;
static struct timeval timeout, stattime, *timer = NULL;
static int (*proc)() = NULL;

/* returns Server or NULL if failure */
Server *GOpenServer(host, output)
     char *host;
     int output;
{
  Server *tempserv;
  char *finalhost;

  tempserv = (Server *)calloc(1, sizeof(Server));
  if (tempserv == NULL) return(NULL);

  tempserv->output = output;
  tempserv->fd = _gconnserv(host, &finalhost);
  if (tempserv->fd == -1) {
    _freeserv(tempserv);
    return(NULL);
  }
  tempserv->server = (char *)malloc((strlen(finalhost) + 1) * sizeof(char));
  if (tempserv->server == NULL) return(NULL);
  if (finalhost) strcpy(tempserv->server, finalhost);
  if (_gopenserv(tempserv) == -1) {
    GCloseServer(tempserv);
    return(NULL);
  }
  return(tempserv);
}

/* returns Server or NULL if failure */
Server *GReopenServer(msgsock)
     Server *msgsock;
{
  int i;
  if (msgsock->volumes) {
    for (i = 0; i < msgsock->num_volumes; i++) {
      free(msgsock->volumes[i].volume);
    }
    free(msgsock->volumes);
  }
  msgsock->volumes = NULL;
  msgsock->num_volumes = 0;
  if (msgsock->extensions) {
    for (i = 0; i < msgsock->num_extensions; i++) {
      free(msgsock->extensions[i]);
    }
    free(msgsock->extensions);
  }
  msgsock->num_extensions = 0;
  _gopenserv(msgsock);
  return(msgsock);
}

_freeserv(msgsock)
     Server *msgsock;
{
  int i;

  if (msgsock->server) free(msgsock->server);
  if (msgsock->volumes) {
    for (i = 0; i < msgsock->num_volumes; i++) {
      if (msgsock->volumes[i].volume) 
	free(msgsock->volumes[i].volume);
    }
    free(msgsock->volumes);
  }
  if (msgsock->extensions) {
    for (i = 0; i < msgsock->num_extensions; i++) {
      free(msgsock->extensions[i]);
    }
    free(msgsock->extensions);
  }
  free(msgsock);
}

_gopenserv(sock)
     Server *sock;
{
  int cur_type, cur_index, i;
  char tempstring[30];

  if (SendCommand(sock->fd,OpenServer,sock->output,0,0,0,0,0,0,0,0) == -1)
    return(GIO_ERROR);
  sock->num_volumes = acceptint(sock->fd);
  if (sock->num_volumes == GIO_ERROR) return(-1);
  sock->volumes=(VolumeList *)malloc(sizeof(VolumeList)*(sock->num_volumes));
  if (sock->volumes == NULL) return(-1);
  for (i = 0; i < sock->num_volumes; i++) {
    sock->volumes[i].index = acceptint(sock->fd);
    if (sock->volumes[i].index == GIO_ERROR) return(-1);
    sock->volumes[i].type = acceptint(sock->fd);
    if (sock->volumes[i].type == GIO_ERROR) return(-1);
    sock->volumes[i].volume = acceptstring(sock->fd);
    if (sock->volumes[i].volume == NULL) {
      if (DEBUG) printf("Error reading Volume_List (volume)\n");
      return(-1);
    }
    if (DEBUG) printf ("volume %d: %s\n",i,sock->volumes[i].volume);
  }
  sock->switching_mask = acceptint(sock->fd);
  if (sock->switching_mask == GIO_ERROR) return(-1);
  sock->num_extensions = acceptint(sock->fd);
  if (sock->num_extensions == GIO_ERROR) return(-1);
  sock->extensions = (char **)malloc(sock->num_extensions*sizeof(char *));
  if (sock->extensions == NULL) return(-1);
  for (i = 0; i < sock->num_extensions; i++) {
    sock->extensions[i] = acceptstring(sock->fd);
    if (sock->extensions[i] == NULL) return(-1);
    if (DEBUG) printf("Extension: %s\n", sock->extensions[i]);
  }
  sock->max_lock_time = acceptint(sock->fd);
  if (sock->max_lock_time == GIO_ERROR) return(-1);

  /* these are left here for extensions to the structure */

  sock->ex1 = acceptint(sock->fd);
  if (sock->ex1 == GIO_ERROR) return(-1);
  sock->ex2 = acceptint(sock->fd);
  if (sock->ex2 == GIO_ERROR) return(-1);
  sock->ex3 = acceptint(sock->fd);
  if (sock->ex3 == GIO_ERROR) return(-1);
  sock->ex3 = acceptint(sock->fd);
  if (sock->ex4 == GIO_ERROR) return(-1);
  sock->ex5 = acceptint(sock->fd);
  if (sock->ex5 == GIO_ERROR) return(-1);
  sock->ex6 = acceptint(sock->fd);
  if (sock->ex6 == GIO_ERROR) return(-1);
  sock->ex7 = acceptint(sock->fd);
  if (sock->ex7 == GIO_ERROR) return(-1);
  sock->ex8 = acceptint(sock->fd);
  if (sock->ex8 == GIO_ERROR) return(-1);
  return(0);
}

/* returns {NO_ERROR} */
GCloseServer(msgsock)
Server *msgsock;
{
  close(msgsock->fd);
  _freeserv(msgsock);
  return(NO_ERROR);
}

/* returns {BAD_VOLUME | NO_ERROR} */
/* channel corresponds to volume */
GSwitch(msgsock,channel,mask)
     Server *msgsock;
     int channel, mask;
{
  if (SendCommand(msgsock->fd,Switch,channel,mask,sync,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return (NO_ERROR);
}
 
/* returns {actual current speed | BAD_VOLUME | FUNC_NOT_SUPPORTED} */
GRun(msgsock,volume, speed, mask)
     Server *msgsock;
     int speed, volume, mask;
{
  if (SendCommand(msgsock->fd,Run,volume,speed,mask,sync,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(speed);
}

/* direction is one of FORWARD or REVERSE */
/* returns {actual current frame | BAD_VOLUME |
   FUNC_NOT_SUPPORTED | NO_ERROR} */
GJog(msgsock, volume, direction, mask)
     Server *msgsock;
     int direction, volume, mask;
{
  if (SendCommand(msgsock->fd,Jog,volume,direction,mask,sync,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {actual current frame | BAD_VOLUME | 
   FUNC_NOT_SUPPORTED | NO_ERROR} */
GStill(msgsock,volume, mask)
     Server *msgsock;
     int volume, mask;
{
  if (SendCommand(msgsock->fd,Still,volume,mask,sync,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {NO_ERROR | BAD_VOLUME | BAD_ARGUMENT | FUNC_NOT_SUPPORTED} */
/* on_off is INDEX_ON or INDEX_OFF or INDEX_TOGGLE */
GIndex(msgsock, volume, on_off)
     Server *msgsock;
     int volume, on_off;
{
  if (SendCommand(msgsock->fd,Index,volume,on_off,sync,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {NO_ERROR | BAD_VOLUME | BAD_ARGUMENT | FUNC_NOT_SUPPORTED} */
/* chan is {LEFT_CHAN | RIGHT_CHAN | INDEX_CTRL} */
/* on_off is TURN_ON or TURN_OFF */
GConfigure(msgsock, volume, chan, on_off)
     Server *msgsock;
     int volume, chan, on_off;
{
  if(SendCommand(msgsock->fd,Configure,volume,chan,on_off,
		 sync,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {NO_ERROR | BAD_VOLUME | BAD_ARGUMENT | FUNC_NOT_SUPPORTED} */
/* on_off is LOAD or UNLOAD */
GLoad(msgsock, volume, on_off)
     Server *msgsock;
     int volume, on_off;
{
  if (SendCommand(msgsock->fd,Load,volume,on_off,sync,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {actual current frame | BAD_VOLUME | FUNC_NOT_SUPPORTED | 
 COULDNT_SEARCH}*/
GSearch(msgsock,volume,frame, mask)
     Server *msgsock;
     int volume, frame, mask;
{
  if (SendCommand(msgsock->fd,Search,volume,frame,mask,sync,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(frame);
}

/* returns {actual current frame | BAD_VOLUME | FUNC_NOT_SUPPORTED} */
GGetFrame(msgsock,volume)
     Server *msgsock;
     int volume;
{
  if (SendCommand(msgsock->fd,GetFrame,volume,sync,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

/* returns {actual current frame | BAD_VOLUME | FUNC_NOT_SUPPORTED} */
GPlaySeg(msgsock,volume,start,end,speed,mask,return_flag)
     Server *msgsock;
     int start, end, speed, mask, return_flag;
{
  if (SendCommand(msgsock->fd,PlaySeg,volume,start,end,
		  speed,mask,return_flag,sync,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

#ifdef SEAMLESS
GDownload(msgsock,script)
     Server *msgsock;
     action *script;
{
  action *act;
  script_event *ev;
  int ev_ints[4];
  int ev_netform[4];
  
  act = script;
	
  if (SendCommand(msgsock->fd,Download,0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  while (act != NULL) {
    ev_ints[0] = act->source_vol;
    hosttonetcom(ev_ints,ev_netform,1);
    full_write(msgsock->fd,ev_netform,sizeof(int));
    ev = act->events;
    while(ev != NULL) {
      ev_ints[0] = ev->target_frame;
      ev_ints[1] = ev->action;
      ev_ints[2] = ev->arg;
      ev_ints[3] = ev->arg2;
      hosttonetcom(ev_ints,ev_netform,4);
      full_write(msgsock->fd,ev_netform,4*sizeof(int));
      ev = ev->next;
    }
    act = act->next;
  }
  ev_ints[0] = END_OF_TRANSMISSION;
  hosttonetcom(ev_ints,ev_netform,1);
  full_write(msgsock->fd,ev_netform,sizeof(int));
  
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return (NO_ERROR);
}

GPlayList(msgsock)
     Server *msgsock;
{
  	if (SendCommand(msgsock->fd,PlayList,0,0,0,0,0,0,0,0,0) == -1)
	  return(_handleerror(msgsock,GIO_ERROR));
	return(NO_ERROR);
}
#endif SEAMLESS

/* returns {NO_ERROR | BAD_VOLUME | FUNC_NOT_SUPPORTED} */
GReset(msgsock,volume)
     Server *msgsock;
     int volume;
{
  if (SendCommand(msgsock->fd,Reset,volume,sync,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  if (!sync) return(_handleerror(msgsock,acceptint(msgsock->fd)));
  return(NO_ERROR);
}

/* returns {NO_ERROR | BAD_ARGUMENT} */
GShutOff(msgsock,shutoffmode)
     Server *msgsock;
     int shutoffmode;
{
  if (SendCommand(msgsock->fd,ShutOff,shutoffmode,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

/* returns {NO_ERROR | VOLUME_LIST_OLD} */
GCheckRevision(msgsock)
     Server *msgsock;
{
  if (SendCommand(msgsock->fd,CheckRevision,0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(msgsock,acceptint(msgsock->fd));
}

/* returns {NO_ERROR} */
GLock(msgsock)
     Server *msgsock;
{
  if (SendCommand(msgsock->fd,Lock,0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

/* returns {NO_ERROR} */
GUnLock(msgsock)
     Server *msgsock;
{
  if (SendCommand(msgsock->fd,UnLock,0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

/* return current server time in seconds since midnight, Jan 1, 1970 */
long GGetServerTime(msgsock)
     Server *msgsock;
{
  long nettime;

  if (SendCommand(msgsock->fd, GetServerTime, 0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return((long)(acceptint(msgsock->fd)));
}

#ifdef SIGGRAPH
/* GReserve is used to reserve a volume for a time period, an ID
   is returned which is used to authenticate a client.  This routine
   should only be used by an external scheduler */
int GMakeReservation(msgsock, volume, begin, end, output)
     Server *msgsock;
     int volume, output;
     long begin, end;
{
  if (SendCommand(msgsock->fd,MakeReservation,volume,(int)begin,
		  (int)end,output,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

GUseReservation(msgsock, id)
     Server *msgsock;
     int id;
{
  if (SendCommand(msgsock->fd,UseReservation,id,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(_handleerror(msgsock,acceptint(msgsock->fd)));
}

#endif SIGGRAPH

/* returns {NO_ERROR | BAD_ARGUMENT} */
GSetState(setDEBUG, settime, setsync)
     int setDEBUG, setsync;
     struct timeval *settime;
{
  DEBUG = (short)setDEBUG;
  sync = setsync;
  if (settime == NULL) timer = NULL;
  else {
    if (settime->tv_sec < 0 || settime->tv_usec < 0) return(BAD_ARGUMENT);
    else {
      stattime = *settime;
      timer = &timeout;
    }
  }
  return(NO_ERROR);
}


/* handler take the arguments (Server, ErrorCode) */
/* returns {NO_ERROR} */
GSetErrorHandler( handler )
     int (*handler)();  
{
  proc = handler;
  return(NO_ERROR);
}

/* returns success */
GRelease(msgsock)
     Server *msgsock;
{
  if (SendCommand(msgsock->fd,Release,0,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  return(NO_ERROR);
}

/* yesno is {Notify | NoNotify} */
/* returns {NO_ERROR} */
GRequestNotification(msgsock, yesno)
     Server *msgsock;
     int yesno;
{
  if (SendCommand(msgsock->fd,RequestNotification,yesno,0,0,0,0,0,0,0,0) == -1)
    return(_handleerror(msgsock,GIO_ERROR));
  setsockopt( msgsock->fd, SOL_SOCKET, SO_KEEPALIVE, &yesno, sizeof(int));
  return(NO_ERROR);
}

/* returns {index number of volume | BAD_VOLUME} */
int GWhichVolumeIndex(serv, volume)
     Server *serv;
     char *volume;
{
  int i;
  for ( i = 0 ; i < serv->num_volumes; i++)
    if (!strcmp(serv->volumes[i].volume, volume))
      return(serv->volumes[i].index);
  return(_handleerror(serv,BAD_VOLUME));
}

/* returns {name for requested index # | NULL} */
char *GWhichVolumeName(serv, volume)
     Server *serv;
     int volume;
{
  int i;
  for ( i = 0 ; i < serv->num_volumes; i++)
    if (serv->volumes[i].index == volume)
      return (serv->volumes[i].volume);
  return(NULL);
}

int GVolumeIndex(serv, which)
     Server *serv;
     int which;
{
  if (which > serv->num_volumes) return(_handleerror(serv,BAD_VOLUME));
  return(serv->volumes[which].index);
}

int GConnection(serv)
     Server *serv;
{
  return(serv->fd);
}

int GMaxLockTime(serv)
     Server *serv;
{
  return(serv->max_lock_time);
}

int GSwitchingMask(serv)
     Server *serv;
{
  return(serv->switching_mask);
}

int GVolumeType(serv, which)
     Server *serv;
     int which;
{
  if (which > serv->num_volumes) return(_handleerror(serv,BAD_VOLUME));
  return(serv->volumes[which].type);
}

char *GVolumeName(serv, which)
     Server *serv;
     int which;
{
  if (which > serv->num_volumes) return(NULL);
  return(serv->volumes[which].volume);
}

int GNumberVolumes(serv)
     Server *serv;
{
  return(serv->num_volumes);
}

int GNumberExtensions(serv)
     Server *serv;
{
  return(serv->num_extensions);
}

char *GExtensionName(serv, which)
     Server *serv;
     int which;
{
  if (which > serv->num_volumes) return(NULL);
  return(serv->extensions[which]);
}

/* the following are support routines for the above calls and
   are NOT guaranteed to exist or remain the same between versions */

_handleerror( serv, ErrorCode )
     Server *serv;
     int ErrorCode;
{
  if (ErrorCode < LOWEST_ERROR || !proc) return(ErrorCode);
  proc(serv, ErrorCode);
  return(ErrorCode);
}

fixtime()
{
  timeout = stattime;
  if (timer == NULL) return(0);
}

acceptint(msgsock)
int msgsock;
{
  int i;
  fixtime();
  if (IsOKToRead(msgsock,timer) < 0) return(GIO_ERROR);
  if (full_read(msgsock,&i,sizeof(int)) < sizeof(int)) {
    if (DEBUG) printf("Bad return code.\n");
    return(GIO_ERROR);
  }
  return(ntohl(i));
}

IsOKToRead(msgsock,time)
int msgsock;
struct timeval time;
{
  fd_set ready;
  int tm;
  FD_ZERO(&ready);
  FD_SET(msgsock,&ready);
  return(select(MAX_FDS,&ready,0,0,time));
}

SendCommand(msgsock,parm0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8,
	    parm9)
int msgsock, parm0,parm1,parm2,parm3,parm4,parm5,parm6,parm7,parm8, parm9;
{
  Galatea_packet hostcommand, commandline;
  hostcommand.parm[0] = parm0;
  hostcommand.parm[1] = parm1;
  hostcommand.parm[2] = parm2;
  hostcommand.parm[3] = parm3;
  hostcommand.parm[4] = parm4;
  hostcommand.parm[5] = parm5;
  hostcommand.parm[6] = parm6;
  hostcommand.parm[7] = parm7;
  hostcommand.parm[8] = parm8;
  hostcommand.parm[9] = parm9;
  hosttonetcom(&hostcommand,&commandline,10);
  return(full_write(msgsock,&commandline,sizeof(commandline)));
}

hosttonetcom(hostints,networkints,number)
int *networkints, *hostints, number;
{
  int i;
  for ( i = 0; i < number; i++)
    networkints[i] = htonl(hostints[i]);
}

int full_read( d, buf, nbytes)
     int d;
     char *buf;
     int nbytes;
{
  int i = 0, actual = 0;
  
  while ( actual < nbytes) {
    i = read(d, buf+actual, nbytes - actual);
    if ( i == -1) return(-1);
    if ( i == 0) return(actual);
    actual += i;
  }
  return(nbytes);
}

int full_write( d, buf, nbytes)
     int d;
     char *buf;
     int nbytes;
{
  int i = 0, actual = 0;
  
  while ( actual < nbytes) {
    i = write(d, buf+actual, nbytes - actual);
    if ( i == -1) return(-1);
    actual += i;
  }
  return(nbytes);
}

int _gconnserv(host, finalhost)
     char *host, **finalhost;
{
  int sock, port, mi;
  struct sockaddr_in server;
#ifdef UNIXCONN
  struct sockaddr_un unix_server;
#endif UNIXCONN
  struct sockaddr *addr;
  int addrlen;
  struct hostent *hp, *gethostbyname();
  struct servent *service;
  struct timeval to;
  fd_set ready;
  char realhost[60];
#ifdef UNIXCONN
  int isunix = 0;
#endif UNIXCONN

  *finalhost = NULL;

#ifdef UNIXCONN
  if (host[0] == '\0' || !strcmp("unix",host) || !host) {
    isunix = 1;
    strcpy(realhost,"unix");
  }
  else strcpy(realhost,host);
  if (isunix) sock = socket(AF_UNIX, SOCK_STREAM, 0);
  else {
#else
    if (host[0] == '\0' || !strcmp("unix",host) || !host)
      gethostname(realhost,60);
    else strcpy(realhost,host);
#endif UNIXCONN
    sock = socket(AF_INET,SOCK_STREAM,0);
#ifdef UNIXCONN
  }
#endif
  if (sock < 0) {
    if (DEBUG) printf("Error creating stream socket.\n");
    return(-1);
  }
#ifdef UNIXCONN
  if (isunix) {
    unix_server.sun_family = AF_UNIX;
    strcpy(unix_server.sun_path, G_UNIX_PATH);
    addr = (struct sockaddr *) &unix_server;
    addrlen = strlen(unix_server.sun_path) + 2;
    if (connect(sock, addr, addrlen) < 0) {
      if (DEBUG) printf("No server available on UNIX connection.\n");
      return(-1);
    }
  } else {
#endif UNIXCONN
    server.sin_family = AF_INET;
    hp = gethostbyname(host);
    if (hp == 0) {
      if (DEBUG) printf("Can't find host %s.\n",host);
      return(-1);
    }
    
    service = getservbyname("galatea","tcp");
    if (service == NULL) {
      server.sin_port = htons(4001);
    }
    else {
      server.sin_port = service->s_port;
    }
    bcopy(hp->h_addr,&server.sin_addr,hp->h_length);
    if (connect(sock,&server,sizeof(server)) < 0) {
      if (DEBUG) printf("No server available on %s at port %d.\n",
			host, ntohl(server.sin_port));
      return(-1);
    }
#ifdef TCP_NODELAY
    mi = 1;
    setsockopt( sock, IPPROTO_TCP, TCP_NODELAY, &mi, sizeof(int));
#endif

#ifdef UNIXCONN
  }
#endif UNIXCONN
  *finalhost = (char *)malloc((strlen(realhost)+1) * sizeof(char));
  if (*finalhost)
    strcpy(*finalhost, realhost);
  return(sock);
}

char *acceptstring(msgsock)
     int msgsock;
{
  int i;
  char *temp;
  i = acceptint(msgsock);
  if (i == -1) return(NULL);
  temp = (char *)malloc(i * sizeof(char));
  if (temp == NULL) return(NULL);
  if (full_read(msgsock, temp, i) != i) return(NULL);
  return(temp);
}
