/***************************************************************/
/*							       */
/*	       TANGO ANIMATION PACKAGE (transition)	       */
/*							       */
/***************************************************************/
/*    Copyright 1989 Brown University -- John T. Stasko        */

#include "tangolocal.h"

/***************************************************************/
/**********	    local functions		      **********/
/***************************************************************/

TANGO_TRANS	get_transition();
ACTION_PTR	get_action();
FRAME_PTR	get_frame();
void		copy_frames_shift();
ACTION_PTR	copy_action_shift();
void		transition_free();
void		bring_front_config();
void		push_tail_config();

#ifndef sun
#define usleep BPIOwait
#endif

/***************************************************************/
/*							       */
/*   TANGOtrans_create - return a transition of the given type,*/
/*	that modifies the given image along the given path.    */
/*							       */
/*	move -- The image is moved along the path.	       */
/*							       */
/*	visibility -- If the image is visible,		       */
/*	it is made invisible.  If it is invisible, it is made  */
/*	visible.  For each offset in the path, the visibility  */
/*	is toggled.					       */
/*							       */
/*	fill -- The image will alternate between unfilled      */
/*	(just an outline) and filled.	For each offset in the */
/*	path, the visibility is toggled.		       */
/*							       */
/*	resize -- The image is resized along the given path.   */
/*	Each type of image will have a "method" for resizing.  */
/*							       */
/*	raise -- The image is brought to the viewing plane     */
/*	closest to the user.				       */
/*							       */
/*	lower -- The image is lowered to the viewing farthest  */
/*	from the user (bottom of all images.)		       */
/*							       */
/*	delay -- do nothing in this frame.		       */
/*							       */
/***************************************************************/

TANGO_TRANS
TANGOtrans_create(type,image,path)
   TANGO_TRANS_TYPE    type;
   TANGO_IMAGE	       image;
   TANGO_PATH	       path;
{
   TANGO_TRANS	trans;
   FRAME_PTR	frame,fhead,ftail;
   int		count;
   OFFSET_PTR	op;
   ACTION_PTR	action;

   if (!TANGO__init) TANGOinit();

   trans = get_transition();

   count = 0;
   fhead = ftail = NULL;
   for (op=path->head; op; op=op->nexto)
      { count++;
	frame = get_frame(type);
	frame->frame_num = count;
	frame->dx = op->dx;
	frame->dy = op->dy;

	if (!fhead)
	   fhead = frame;
	else
	   ftail->nextf = frame;
	frame->prevf = ftail;
	frame->nextf = NULL;
	ftail = frame;
      }
   action = get_action(type);
   action->fhead = fhead;
   action->ftail = ftail;
   action->image = image;
   action->nexta = NULL;

   trans->num_frames = count;
   trans->actions = action;

   return(trans);
}




/***************************************************************/
/*							       */
/*   TANGOtrans_iterate - return a transition which is the     */
/*	given transition repeated num times.		       */
/*							       */
/***************************************************************/

TANGO_TRANS
TANGOtrans_iterate(trans,num)
   TANGO_TRANS trans;
   int num;
{
   TANGO_TRANS	ittrans;
   ACTION_PTR	act,last_action,new_action;
   int		it;

   if (!TANGO__init) TANGOinit();

   ittrans = get_transition();

   last_action = NULL;
   for (it=0; it<num; ++it)
      for (act=trans->actions; act; act=act->nexta)
	 { new_action = copy_action_shift(act,it*trans->num_frames);
	   if (last_action)
	      last_action->nexta = new_action;
	   else
	      ittrans->actions = new_action;
	   last_action = new_action;
	 }
   ittrans->num_frames = num * trans->num_frames;

   return(ittrans);
}



/***************************************************************/
/*							       */
/*   TANGOtrans_concatenate - return a transition that is the  */
/*	concatenation of given transitions one after the other.*/
/*							       */
/***************************************************************/

TANGO_TRANS
TANGOtrans_concatenate(num,t1,t2,t3,t4,t5,t6,t7,t8,t9,t10)
   int num;
   TANGO_TRANS t1,t2,t3,t4,t5,t6,t7,t8,t9,t10;
{
   TANGO_TRANS	contrans,transitions[11];
   ACTION_PTR	act,last_action,new_action;
   int		frame_count,tnum;

   if (!TANGO__init) TANGOinit();

   transitions[0] = t1;
   transitions[1] = t2;
   transitions[2] = t3;
   transitions[3] = t4;
   transitions[4] = t5;
   transitions[5] = t6;
   transitions[6] = t7;
   transitions[7] = t8;
   transitions[8] = t9;
   transitions[9] = t10;
   contrans = get_transition();

   last_action = NULL;
   frame_count = 0;
   for (tnum=0; tnum<num; ++tnum)
      { for (act=transitions[tnum]->actions; act; act=act->nexta)
	   { new_action = copy_action_shift(act,frame_count);
	     if (last_action)
		last_action->nexta = new_action;
	     else
		contrans->actions = new_action;
	     last_action = new_action;
	   }
	frame_count += transitions[tnum]->num_frames;
      }

   contrans->num_frames = frame_count;

   return(contrans);
}



/***************************************************************/
/*							       */
/*   TANGOtrans_compose - return a transition which is the     */
/*	composition of given transitions.  By composition, we  */
/*	mean the concurrent execution of all the transitions.  */
/*							       */
/***************************************************************/

TANGO_TRANS
TANGOtrans_compose(num,t1,t2,t3,t4,t5,t6,t7,t8,t9,t10)
   int num;
   TANGO_TRANS t1,t2,t3,t4,t5,t6,t7,t8,t9,t10;
{
   int		  max,tnum;
   TANGO_TRANS	  comtrans,transitions[10];
   ACTION_PTR	  act,last_action,new_action;

   if (!TANGO__init) TANGOinit();

   transitions[0] = t1;
   transitions[1] = t2;
   transitions[2] = t3;
   transitions[3] = t4;
   transitions[4] = t5;
   transitions[5] = t6;
   transitions[6] = t7;
   transitions[7] = t8;
   transitions[8] = t9;
   transitions[9] = t10;
   max = 0;
   for (tnum=0; tnum<num; ++tnum)
      if (transitions[tnum]->num_frames > max)
	 max = transitions[tnum]->num_frames;

   comtrans = get_transition();

   last_action = NULL;
   for (tnum=0; tnum<num; ++tnum)
      { for (act=transitions[tnum]->actions; act; act=act->nexta)
	   { new_action = copy_action_shift(act,0);
	     if (new_action->fhead)  /* if no frames added, bag this action */
		{ if (last_action)
		     last_action->nexta = new_action;
		  else
		     comtrans->actions = new_action;
		  last_action = new_action;
		}
	     else
		free(new_action);
	   }
      }

   comtrans->num_frames = max;

   return(comtrans);
}



/***************************************************************/
/*							       */
/*   TANGOtrans_perform - execute the given transition.        */
/*							       */
/***************************************************************/

void
TANGOtrans_perform(trans)
   TANGO_TRANS trans;
{
   register IMAGE_PTR	    imlist;
   register ACTION_PTR	    act;
   register ASH_WINDOW	    parent;
	    int 	    fnum;
	    TANGO_IMAGE_PTR iptr;

   if (!TANGO__init) TANGOinit();
   parent = TANGO__data->parentwindow;

   for (act=trans->actions; act; act=act->nexta)
      act->doing = act->fhead;	/* reset pointers to execute frames */

   ASHbatch_mode(1);
   for (fnum=1; fnum<=trans->num_frames; ++fnum)
      { for (act=trans->actions; act; act=act->nexta)
	   if ((act->doing) && (act->doing->frame_num == fnum))
	      { if (act->type == TANGO_TRANS_TYPE_MOVE)
		   { iptr = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,act->image->worm_obj);
		     iptr->loc[0] += act->doing->dx;
		     iptr->loc[1] += act->doing->dy;
		   }
		else if (act->type == TANGO_TRANS_TYPE_VISIBLE)
		   { iptr = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,act->image->worm_obj);
		     iptr->visible = !(iptr->visible);
		   }
		else if (act->type == TANGO_TRANS_TYPE_DELAY)
		     ; /* do nothing */
		else if (act->type == TANGO_TRANS_TYPE_RAISE)
		     bring_front_config(act->image);
		else if (act->type == TANGO_TRANS_TYPE_LOWER)
		     push_tail_config(act->image);
		else if (act->type == TANGO_TRANS_TYPE_REFRESH)
		     { if (TANGO__data->mode)
			  for (imlist=TANGO__data->configtail; imlist; imlist=imlist->previ)
			    imlist->image->status = (TANGO__data->blting ? 1 : 2);
		     }
		else /* non-standard transition, call specific image handler */
		     WORMapply(TANGO__opid_trans,act->image->worm_obj,act->type,
						   act->doing->dx,act->doing->dy);
		act->doing = act->doing->nextf;
		/* set it as changed for fast mode */
		if ((TANGO__data->mode) && (act->type != TANGO_TRANS_TYPE_DELAY) && (act->type != TANGO_TRANS_TYPE_REFRESH))
		   act->image->status = (TANGO__data->blting ? 1 : 2);
	      }

	TANGO_clear_screen();

	/* now draw all the images and go to the next frame */
	for (imlist=TANGO__data->configtail; imlist; imlist=imlist->previ)
	   if (TANGO__data->mode == 0) /* normal */
	      WORMapply(TANGO__opid_draw,imlist->image->worm_obj,imlist->image,0.0,0.0);
	   else /* fast mode */
	      { if (imlist->image->status > 0)
		   { WORMapply(TANGO__opid_draw,imlist->image->worm_obj,imlist->image,0.0,0.0);
		     imlist->image->status--;
		   }
	      }

	TANGO__data->window = ASHanim_next_frame(TANGO__data->window);
	if (!TANGO__data->blting)
	   ASHset_color_table(parent,ASHinq_color_table(TANGO__data->window));
	if (TANGO__data->delay > 0)
	   { ASHbatch_mode(0);
	     usleep(TANGO__data->delay);
	     ASHbatch_mode(1);
	   }
      }
   ASHbatch_mode(0);
}



/***************************************************************/
/*							       */
/*   TANGOtrans_free - free up the allocated structures        */
/*	associated with a transition.			       */
/*							       */
/***************************************************************/

void
TANGOtrans_free(num,t1,t2,t3,t4,t5,t6,t7,t8,t9,t10)
   int num;
   TANGO_TRANS t1,t2,t3,t4,t5,t6,t7,t8,t9,t10;
{
   if (!TANGO__init) TANGOinit();

   if (num>0)
      transition_free(t1);
   if (num>1)
      transition_free(t2);
   if (num>2)
      transition_free(t3);
   if (num>3)
      transition_free(t4);
   if (num>4)
      transition_free(t5);
   if (num>5)
      transition_free(t6);
   if (num>6)
      transition_free(t7);
   if (num>7)
      transition_free(t8);
   if (num>8)
      transition_free(t9);
   if (num>9)
      transition_free(t10);
}




/***************************************************************/
/*							       */
/*   get_transition - allocate and return a transition	       */
/*	structure.  All the fields are set to their default    */
/*	initial values. 				       */
/*							       */
/***************************************************************/

TANGO_TRANS
get_transition()
{
   TANGO_TRANS trans;

   trans = (TANGO_TRANS) malloc( sizeof( struct _TRANS));
   trans->num_frames = 1;
   trans->actions = NULL;
   trans->next = NULL;
   return(trans);
}



/***************************************************************/
/*							       */
/*   get_action - allocate and return an action structure.     */
/*	All the fields are set to their default initial        */
/*	values.  An action contains frames which affect an     */
/*	object in a transition.  (Because of		       */
/*	composition, one transition may affect many objects.)  */
/*							       */
/***************************************************************/

ACTION_PTR
get_action(type)
   TANGO_TRANS_TYPE type;
{
   ACTION_PTR act;

   act = (ACTION_PTR) malloc( sizeof( struct ACTION));
   act->type = type;
   act->image = NULL;
   act->doing = NULL;
   act->fhead = act->ftail = NULL;
   act->nexta = NULL;
   return(act);
}




/***************************************************************/
/*							       */
/*   get_frame - allocate and return a frame structure.        */
/*	All the fields are set to their default initial        */
/*	values.  A frame is one logical time unit of action    */
/*	in a transition.  The frame_num field is very important*/
/*	for the execution of transitions.		       */
/*							       */
/***************************************************************/

FRAME_PTR
get_frame(type)
   TANGO_TRANS_TYPE type;
{
   FRAME_PTR frame;

   frame = (FRAME_PTR) malloc( sizeof( struct FRAME));
   frame->type = type;
   frame->frame_num = 1;
   frame->dx = ZERO;
   frame->dy = ZERO;
   frame->nextf = frame->prevf = NULL;
   return(frame);
}



/***************************************************************/
/*							       */
/*   copy_frames_shift - copy all the frames from the given    */
/*	list, and return the appropriate head and tail ptrs.   */
/*	The frame_numbers are supplemented by the additional   */
/*	start_num.					       */
/*							       */
/***************************************************************/

void
copy_frames_shift(frames,start_num,head,tail)
   FRAME_PTR  frames;
   int	      start_num;
   FRAME_PTR *head;
   FRAME_PTR *tail;
{
   FRAME_PTR fra,new_frame;
   FRAME_PTR h,t;

   h = t = NULL;
   for (fra=frames; fra; fra=fra->nextf)
      { new_frame = (FRAME_PTR) malloc( sizeof( struct FRAME));
	new_frame->type = fra->type;
	new_frame->dx = fra->dx;
	new_frame->dy = fra ->dy;
	new_frame->frame_num = fra->frame_num + start_num;
	if (h)
	   t->nextf = new_frame;
	else
	   h = new_frame;
	new_frame->prevf = t;
	new_frame->nextf = NULL;
	t = new_frame;
      }
   *head = h;
   *tail = t;
}




/***************************************************************/
/*							       */
/*   copy_action_shift - copy the given action (and its sub-   */
/*	frames) into a new action which has the frame_numbers  */
/*	supplemented by start_frame.  Return the ACTION_PTR.   */
/*							       */
/***************************************************************/

ACTION_PTR
copy_action_shift(action,start_frame)
   ACTION_PTR action;
   int	      start_frame;
{
   ACTION_PTR new_action;
   FRAME_PTR  head,tail;

   new_action = (ACTION_PTR) malloc( sizeof( struct ACTION));
   new_action->type = action->type;
   new_action->image = action->image;
   new_action->doing = NULL;
   new_action->nexta = NULL;

   copy_frames_shift(action->fhead,start_frame,&head,&tail);
   new_action->fhead = head;
   new_action->ftail = tail;
   return(new_action);
}



/***************************************************************/
/*							       */
/*   transition_free - free up the allocated structures        */
/*	associated with a transition.			       */
/*							       */
/***************************************************************/

void
transition_free(trans)
   TANGO_TRANS trans;
{
   FRAME_PTR  f,old_f;
   ACTION_PTR act,old_act;

   if (!trans) return;
   act = trans->actions;
   while (act)
      { f = act->fhead;
	while (f)
	   { old_f = f;
	     f = f->nextf;
	     free(old_f);
	   }
	old_act = act;
	act = act->nexta;
	free(old_act);
      }
   free(trans);
}



/***************************************************************/
/*							       */
/*   bring_front_config - bring the given image to the front   */
/*	of the configuration list.  (it will be drawn last)    */
/*							       */
/***************************************************************/

void
bring_front_config(image)
   TANGO_IMAGE image;
{
   IMAGE_PTR im;

   im = image->inconfig;

   if (im == TANGO__data->confighead) return;

   if (im == TANGO__data->configtail)
      TANGO__data->configtail = im->previ;
   else
      im->nexti->previ = im->previ;

   im->previ->nexti = im->nexti;
   im->nexti = TANGO__data->confighead;
   TANGO__data->confighead->previ = im;
   TANGO__data->confighead = im;
   im->previ = NULL;
}



/***************************************************************/
/*							       */
/*   push_tail_config - put the given image at the tail        */
/*	of the configuration list.  (it will be drawn first)   */
/*							       */
/***************************************************************/

void
push_tail_config(image)
   TANGO_IMAGE image;
{
   IMAGE_PTR im;

   im = image->inconfig;

   if (im == TANGO__data->configtail) return;

   if (im == TANGO__data->confighead)
      TANGO__data->confighead = im->nexti;
   else
      im->previ->nexti = im->nexti;

   im->nexti->previ = im->previ;
   im->previ = TANGO__data->configtail;
   TANGO__data->configtail->nexti = im;
   TANGO__data->configtail = im;
   im->nexti = NULL;
}




/***************************************************************/
/*							       */
/*   TANGO_refresh - redraw all the active images to clean up  */
/*	the screen.					       */
/*							       */
/***************************************************************/

void
TANGO_refresh()
{
   register IMAGE_PTR	    imlist;
   register ASH_WINDOW	    parent;
	    int 	    times,i;

   parent = TANGO__data->parentwindow;

   if (TANGO__data->blting == 0)
      times = 2;
   else
      times = 1;

   for (imlist=TANGO__data->configtail; imlist; imlist=imlist->previ)
      imlist->image->status = times;

   ASHbatch_mode(1);
   for (i=times; i>0; --i)
      {
	TANGO_clear_screen();

	/* now draw all the images and go to the next frame */
	for (imlist=TANGO__data->configtail; imlist; imlist=imlist->previ)
	    { if (imlist->image->status > 0)
		 { WORMapply(TANGO__opid_draw,imlist->image->worm_obj,imlist->image,0.0,0.0);
		   imlist->image->status--;
		 }
	    }

	TANGO__data->window = ASHanim_next_frame(TANGO__data->window);
	if (!TANGO__data->blting)
	   ASHset_color_table(parent,ASHinq_color_table(TANGO__data->window));
	if (TANGO__data->delay > 0)
	   { ASHbatch_mode(0);
	     usleep(TANGO__data->delay);
	     ASHbatch_mode(1);
	   }
      }
   ASHbatch_mode(0);
}




