/***************************************************************/
/*							       */
/*	       TANGO ANIMATION PACKAGE (path)		       */
/*							       */
/***************************************************************/
/*    Copyright 1989 Brown University -- John T. Stasko        */

#include "tangolocal.h"

/***************************************************************/
/**********	    local functions		      **********/
/***************************************************************/

TANGO_PATH	save_path();
void		path_free();


/***************************************************************/
/*							       */
/*   TANGOpath_load - load a path from the given file name.    */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_load(file_name)
   char *file_name;
{
   FILE *fp;
   int count;
   WIN_COORD dx,dy;
   OFFSET_PTR off,head,tail;

   if (!TANGO__init) TANGOinit();

   if ((fp = fopen(file_name,"r")) == NULL)
      { fprintf(stderr,"Unable to open file %s for path read\n",file_name);
	return(NULL);
      }

   head = tail = NULL;
   count = 0;

   while (fscanf(fp,"%lf %lf\n",&dx,&dy) != EOF)
      { off = (OFFSET_PTR) malloc( sizeof (struct OFFSET));
	off->dx = dx;
	off->dy = dy;
	if (!head)
	   head = off;
	else
	   tail->nexto = off;
	off->nexto = NULL;
	off->prevo = tail;
	tail = off;
	++count;
      }
   fclose(fp);
   return( save_path(head,tail,count) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_store - allow the user to store a path in some  */
/*	file.  Returns NULL if the save failed. 	       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_store(path,file_name)
   TANGO_PATH path;
   char *file_name;
{
  FILE *fp;
  OFFSET_PTR off;

   if (!TANGO__init) TANGOinit();

   if ((fp = fopen(file_name,"w")) == NULL)
      { fprintf(stderr,"Unable to open file %s for path write\n",file_name);
	return(NULL);
      }

   for (off=path->head; off; off=off->nexto)
      fprintf(fp,"%lf %lf\n",off->dx,off->dy);
   fclose(fp);

   return(path);
}




/***************************************************************/
/*							       */
/*   TANGOpath_create - create a path given a sequence of      */
/*	offsets.					       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_create(num,x,y)
   int num;
   WIN_COORD x[],y[];
{
   OFFSET_PTR offset,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   for (i=0; i<num; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = x[i];
	offset->dy = y[i];
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,num) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_length - return the number of offsets in the    */
/*	given path.					       */
/*							       */
/***************************************************************/

int
TANGOpath_length(path)
   TANGO_PATH path;
{
   if (!TANGO__init) TANGOinit();

   return( path->count );
}



/***************************************************************/
/*							       */
/*   TANGOpath_dx - return the distance the path covers in     */
/*	x animation coordiante system units.		       */
/*							       */
/***************************************************************/

double
TANGOpath_dx(path)
   TANGO_PATH path;
{
   double dx=0.0;
   OFFSET_PTR off;

   for (off=path->head; off; off=off->nexto)
      dx += off->dx;

   return(dx);
}




/***************************************************************/
/*							       */
/*   TANGOpath_dy - return the distance the path covers in     */
/*	y animation coordiante system units.		       */
/*							       */
/***************************************************************/

double
TANGOpath_dy(path)
   TANGO_PATH path;
{
   double dy=0.0;
   OFFSET_PTR off;

   for (off=path->head; off; off=off->nexto)
      dy += off->dy;

   return(dy);
}



/***************************************************************/
/*							       */
/*   TANGOpath_null - return a null path (all <0,0> offsets)   */
/*	which is the given number of offsets long.	       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_null(count)
   int count;
{
   OFFSET_PTR offset,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   for (i=1; i<=count; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = offset->dy = ZERO;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,count) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_type - create a path which corresponds to one   */
/*	of the basic path types.			       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_type(path_type)
   TANGO_PATH_TYPE path_type;
{
   OFFSET_PTR offset,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   for (i=0; i<=TANGO_PATH_POINTS-1; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = TANGO__path_type[path_type][i][0];
	offset->dy = TANGO__path_type[path_type][i][1];
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,TANGO_PATH_POINTS) );
}





/***************************************************************/
/*							       */
/*   TANGOpath_color - create a path that corresponds to       */
/*	changing an image to the given color.		       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_color(color)
   int color;
{
   OFFSET_PTR offset;

   if (!TANGO__init) TANGOinit();

   offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
   offset->prevo = NULL;
   offset->nexto = NULL;

   switch (color)
      { case TANGO_COLOR_WHITE :
	   offset->dx = 0.05;
	   offset->dy = -0.01;
	   break;
	case TANGO_COLOR_BLACK :
	   offset->dx = 0.01;
	   offset->dy = -0.05;
	   break;
	case TANGO_COLOR_RED :
	   offset->dx = -0.01;
	   offset->dy = -0.05;
	   break;
	case TANGO_COLOR_ORANGE :
	   offset->dx = -0.05;
	   offset->dy = -0.01;
	   break;
	case TANGO_COLOR_YELLOW :
	   offset->dx = -0.05;
	   offset->dy = 0.01;
	   break;
	case TANGO_COLOR_GREEN :
	   offset->dx = -0.01;
	   offset->dy = 0.05;
	   break;
	case TANGO_COLOR_BLUE :
	   offset->dx = 0.01;
	   offset->dy = 0.05;
	   break;
	case TANGO_COLOR_MAROON :
	   offset->dx = 0.05;
	   offset->dy = 0.01;
	   break;
	default :
	   offset->dx = offset->dy = 0.0;
      }

   return( save_path(offset,offset,1) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_add_head - return a path which has the given    */
/*	number of null offsets added to the head of the given  */
/*	path.						       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_add_head(path,num)
   TANGO_PATH path;
   int num;
{
   OFFSET_PTR offset,walk,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   if (num < 0) num=0;

   for (i=1; i<=num; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = offset->dy = ZERO;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   for (walk=path->head; walk; walk=walk->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = walk->dx;
	offset->dy = walk->dy;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   return( save_path(head,tail,path->count+num) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_add_tail - return a path which has the given    */
/*	number of null offsets added to the tail of the given  */
/*	path.						       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_add_tail(path,num)
   TANGO_PATH path;
   int num;
{
   OFFSET_PTR offset,walk,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   if (num < 0) num=0;

   for (walk=path->head; walk; walk=walk->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = walk->dx;
	offset->dy = walk->dy;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   for (i=1; i<=num; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = offset->dy = ZERO;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   return( save_path(head,tail,path->count+num) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_delete_head - return a path which corresponds   */
/*	to the given path with the given number of offsets     */
/*	removed from its head.				       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_delete_head(path,num)
   TANGO_PATH path;
   int num;
{
   OFFSET_PTR offset,walk,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   if (num < 0) num=0;

   for (i=1,walk=path->head; i<=num; ++i)
      { if (!walk) break;
	walk = walk->nexto;
      }

   for ( ; walk; walk=walk->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = walk->dx;
	offset->dy = walk->dy;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   return( save_path(head,tail,((path->count-num < 0) ? 0 : path->count-num) ) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_delete_tail - return a path which corresponds   */
/*	to the given path with the given number of offsets     */
/*	removed from itsa tail. 			       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_delete_tail(path,num)
   TANGO_PATH path;
   int num;
{
   OFFSET_PTR offset,walk,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   if (num < 0) num=0;

   for (i=1,walk=path->head; (i<=path->count-num)&&walk; walk=walk->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = walk->dx;
	offset->dy = walk->dy;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }

   return( save_path(head,tail,((path->count-num < 0) ? 0 : path->count-num) ) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_copy - return a new path which is a duplicate   */
/*	of the given path.				       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_copy(path)
   TANGO_PATH path;
{
   OFFSET_PTR ptr,offset,head=NULL,tail=NULL;
   int i;

   if (!TANGO__init) TANGOinit();

   for (ptr=path->head; ptr; ptr=ptr->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = ptr->dx;
	offset->dy = ptr->dy;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_smooth - return a path that is a "smoother"     */
/*	version of the given path (avg. in nearby offsets)     */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_smooth(path)
   TANGO_PATH path;
{
   OFFSET_PTR ptr,offset,head,tail;
   WIN_COORD  oldx,oldy,newx,newy,xshould,yshould;

   if (!TANGO__init) TANGOinit();

   if (TANGOpath_length(path) == 1) return( TANGOpath_copy(path) );

   ptr = path->head;
   oldx = ptr->dx;
   oldy = ptr->dy;
   newx = (3.0 * ptr->dx + ptr->nexto->dx) / 4.0;
   newy = (3.0 * ptr->dy + ptr->nexto->dy) / 4.0;

   offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
   offset->dx = newx;
   offset->dy = newy;
   head = tail = offset;
   offset->prevo = offset->nexto = NULL;

   for (ptr=ptr->nexto; ptr->nexto; ptr=ptr->nexto)
      { oldx += ptr->dx;
	oldy += ptr->dy;
	xshould = (4.0 * oldx - ptr->prevo->dx + ptr->nexto->dx) / 4.0;
	yshould = (4.0 * oldy - ptr->prevo->dy + ptr->nexto->dy) / 4.0;

	offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = ptr->dx + xshould - oldx;
	offset->dy = ptr->dy + yshould - oldy;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail->nexto = offset;
	tail = offset;

	newx += offset->dx;
	newy += offset->dy;
      }
   offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
   offset->dx = TANGOpath_dx(path) - newx;
   offset->dy = TANGOpath_dy(path) - newy;
   offset->prevo = tail;
   offset->nexto = NULL;
   tail->nexto = offset;
   tail = offset;

   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_rotate - create a path which is the rotation    */
/*	of the given path by the given number of degrees       */
/*	(only positive degrees and in a counterclockwise       */
/*	rotation)					       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_rotate(path,deg)
   TANGO_PATH path;
   int deg;
{
   double rads, cosine, sine;
   OFFSET_PTR ptr, offset, head=NULL, tail=NULL;

   if (!TANGO__init) TANGOinit();

   rads = 0.017453 * deg;
   cosine = cos(rads);
   sine = sin(rads);

   for (ptr=path->head; ptr; ptr=ptr->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = (ptr->dx * cosine) + (ptr->dy * sine);
	offset->dy = -(ptr->dx * sine) + (ptr->dy * cosine);
	offset->prevo = tail;
	offset->nexto = NULL;  /* sine signs reversed because our system has */
	if (head)	       /* negatives at top and positives at window bottom */
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_scale - scale the path by the given factors in  */
/*	x and y.  (Scale each offset.)			       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_scale(path,xscale,yscale)
   TANGO_PATH path;
   double xscale,yscale;
{
   OFFSET_PTR ptr, offset, head=NULL, tail=NULL;

   if (!TANGO__init) TANGOinit();

   for (ptr=path->head; ptr; ptr=ptr->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = ptr->dx * xscale;
	offset->dy = ptr->dy * yscale;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_extend - add x and y factors to each offset     */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_extend(path,x,y)
   TANGO_PATH path;
   double x,y;
{
   OFFSET_PTR ptr, offset, head=NULL, tail=NULL;

   if (!TANGO__init) TANGOinit();

   for (ptr=path->head; ptr; ptr=ptr->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = ptr->dx + x;
	offset->dy = ptr->dy + y;
	offset->prevo = tail;
	offset->nexto = NULL;
	if (head)
	   tail->nexto = offset;
	else
	   head = offset;
	tail = offset;
      }
   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_interpolate - create a path which contains      */
/*	an interpolated number of offsets from the given path  */
/*	as indicated by the parameter factor.		       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_interpolate(path,factor)
   TANGO_PATH path;
   double factor;
{
   int count,found;
   double x,y,runx,runy,remainder,total,percent;
   OFFSET_PTR off,offset,head=NULL,tail=NULL;

   count = 0;
   remainder = 0.0;
   runx = runy = 0.0;
   total = factor;
   found = 0;
   off = path->head;
   x = off->dx;
   y = off->dy;
   while (!found)
      { if ((remainder+factor) < 1.0)
	   { runx += x;
	     runy += y;
	     off = off->nexto;
	     if (off)
		{ x = off->dx;
		  y = off->dy;
		  total += factor;
		  remainder += factor;
		}
	     else
		found = 1;
	   }
	else
	   { while (total >= 1.0)
		{ percent = (1.0 - remainder) / factor;
		  offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
		  count++;
		  offset->dx = runx + (percent * x);
		  offset->dy = runy + (percent * y);
		  offset->prevo = tail;
		  offset->nexto = NULL;
		  if (head)
		     tail->nexto = offset;
		  else
		     head = offset;
		  tail = offset;
		  runx = runy = 0.0;
		  remainder = 0.0;
		  total -= 1.0;
		}
	     remainder = total;
	     runx = (remainder / factor) * x;
	     runy = (remainder / factor) * y;
	     off = off->nexto;
	     if (off)
		{ x = off->dx;
		  y = off->dy;
		  total += factor;
		}
	     else
		found = 1;
	   }
      }
   offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
   count++;
   offset->dx = runx;
   offset->dy = runy;
   offset->prevo = tail;
   offset->nexto = NULL;
   if (head)
      tail->nexto = offset;
   else
      head = offset;
   tail = offset;

   return( save_path(head,tail,count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_example - create and return a path which        */
/*	looks like the given path, only it begins at fromloc   */
/*	and runs to the toloc.				       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_example(fromloc,toloc,path)
   TANGO_LOC  fromloc;
   TANGO_LOC  toloc;
   TANGO_PATH path;
{
   WIN_COORD	loc_dx,loc_dy;	      /* total x and y changes in given locs   */
   WIN_COORD	path_dx,path_dy;    /* total x and y changes in given path   */
   OFFSET_PTR	op,offset,head,tail;
   WIN_COORD	dx_ratio,dy_ratio;  /* loc--path differences over # points   */
   int xscale,yscale;

   if (!TANGO__init) TANGOinit();

   if (!path)
      return( TANGOpath_example(fromloc,toloc,TANGOpath_type(TANGO_PATH_TYPE_STRAIGHT)) );

   loc_dx = toloc->x - fromloc->x;
   loc_dy = toloc->y - fromloc->y;
   path_dx = TANGOpath_dx(path);
   path_dy = TANGOpath_dy(path);

   if ( EQUAL(path_dx,0.0) )
      { dx_ratio = loc_dx / (double)(path->count);
	xscale = 0;
      }
   else
      { dx_ratio = loc_dx / path_dx;
	xscale = 1;
      }

   if ( EQUAL(path_dy,0.0) )
      { dy_ratio = loc_dy / (double)(path->count);
	yscale = 0;
      }
   else
      { dy_ratio = loc_dy / path_dy;
	yscale = 1;
      }

   head = tail = NULL;

   for (op=path->head; op; op=op->nexto)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));

	if (xscale)
	   offset->dx = op->dx * dx_ratio;
	else
	   offset->dx = op->dx + dx_ratio;

	if (yscale)
	   offset->dy = op->dy * dy_ratio;
	else
	   offset->dy = op->dy + dy_ratio;

	if (!head)
	   head = offset;
	else
	   tail->nexto = offset;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail = offset;
      }

   return( save_path(head,tail,path->count) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_motion - create a path of the given type that   */
/*	moves between the two locations.		       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_motion(fromloc,toloc,type)
   TANGO_LOC  fromloc;
   TANGO_LOC  toloc;
   TANGO_PATH_TYPE type;
{
   TANGO_PATH path,ret_path;

   path = TANGOpath_type(type);
   ret_path = TANGOpath_example(fromloc,toloc,path);
   path_free(path);
   return(ret_path);
}



/***************************************************************/
/*							       */
/*   TANGOpath_distance - create and return a path that moves  */
/*	from the fromloc to the toloc, and that has an offset  */
/*	every time the given distance has been covered.        */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_distance(fromloc,toloc,distance)
   TANGO_LOC fromloc,toloc;
   WIN_COORD distance;
{
   int	       steps,i;
   WIN_COORD   fromx,fromy,tox,toy,
	       dx,dy,totdist,ratio,
	       xstep,ystep,runx,runy;
   OFFSET_PTR  offset,head,tail;

   if (!TANGO__init) TANGOinit();

   TANGOloc_inquire(fromloc,&fromx,&fromy);
   TANGOloc_inquire(toloc,&tox,&toy);
   dx = tox- fromx;
   dy = toy - fromy;

   totdist = sqrt( (dx*dx) + (dy*dy) );

   ratio = distance / totdist;

   steps = (int) (1.0 / ratio);

   xstep = dx * ratio;
   ystep = dy * ratio;
   runx = fromx;
   runy = fromy;

   head = tail = NULL;
   for (i=1; i<=steps; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = xstep;
	offset->dy = ystep;
	if (!head)
	   head = offset;
	else
	   tail->nexto = offset;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail = offset;

	runx += xstep;
	runy += ystep;
      }

   if ((runx != dx) || (runy != dy) || (steps == 0))
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = tox - runx;
	offset->dy = toy - runy;
	if (!head)
	   head = offset;
	else
	   tail->nexto = offset;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail = offset;
	++steps;
      }

   return( save_path(head,tail,steps) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_iterate - return a path which is "num"          */
/*	iterations of the path "path" concatenated after each  */
/*	other.						       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_iterate(path,num)
   TANGO_PATH path;
   double num;
{
   OFFSET_PTR offset,head,tail,op;
   int whole,num_offsets,parts,i;
   double decimal;

   if (!TANGO__init) TANGOinit();

   whole = (int) num;

   head = tail = NULL;
   num_offsets = 0;

   for (i=1; i<=whole; ++i)
      { for (op=path->head; op; op=op->nexto)
	   { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	     offset->dx = op->dx;
	     offset->dy = op->dy;
	     if (!head)
		head = offset;
	     else
		tail->nexto = offset;
	     offset->prevo = tail;
	     offset->nexto = NULL;
	     tail = offset;
	   }
	num_offsets += path->count;
      }

   decimal = num - (double) whole;
   parts = (int) (decimal * (double)(path->count));

   op = path->head;
   for (i=1; i<=parts; ++i)
      { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = op->dx;
	offset->dy = op->dy;
	if (!head)
	   head = offset;
	else
	   tail->nexto = offset;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail = offset;
	op = op->nexto;
      }
   num_offsets += parts;

   return( save_path(head,tail,num_offsets) );
}



/***************************************************************/
/*							       */
/*   TANGOpath_concatenate - concatenate "num" paths together  */
/*	(they're stored in the array "paths") into one new     */
/*	path.						       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_concatenate(num,p0,p1,p2,p3,p4,p5,p6,p7,p8,p9)
   int num;
   TANGO_PATH p0,p1,p2,p3,p4,p5,p6,p7,p8,p9;
{
   OFFSET_PTR offset,head,tail,op;
   int count,i;
   TANGO_PATH paths[10];

   if (!TANGO__init) TANGOinit();

   paths[0]=p0; paths[1]=p1; paths[2]=p2; paths[3]=p3; paths[4]=p4;
   paths[5]=p5; paths[6]=p6; paths[7]=p7; paths[8]=p8; paths[9]=p9;

   head = tail = NULL;
   count = 1;

   for (i=0; i<=num-1; ++i)
      { for (op=paths[i]->head; op; op=op->nexto)
	   { offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	     offset->dx = op->dx;
	     offset->dy = op->dy;
	     if (!head)
		head = offset;
	     else
		tail->nexto = offset;
	     offset->prevo = tail;
	     offset->nexto = NULL;
	     tail = offset;
	   }
	count += paths[i]->count;
      }
   return( save_path(head,tail,count) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_compose - return a path which is the composition*/
/*	of all the paths in "paths".  The paths must have      */
/*	the same number of offsets.			       */
/*							       */
/***************************************************************/

TANGO_PATH
TANGOpath_compose(num,p0,p1,p2,p3,p4,p5,p6,p7,p8,p9)
   int num;
   TANGO_PATH p0,p1,p2,p3,p4,p5,p6,p7,p8,p9;
{
   int i,j,count;
   WIN_COORD dx,dy;
   OFFSET_PTR offset,head,tail;
   OFFSET_PTR ptr[20];
   TANGO_PATH paths[10];

   if (!TANGO__init) TANGOinit();

   paths[0]=p0; paths[1]=p1; paths[2]=p2; paths[3]=p3; paths[4]=p4;
   paths[5]=p5; paths[6]=p6; paths[7]=p7; paths[8]=p8; paths[9]=p9;

   count = paths[0]->count;
   for (i=1; i<num; ++i)
      if (paths[i]->count != count)
	 return(NULL);

   for (i=0; i<num; ++i)
      { ptr[i] = paths[i]->head;
      }
   head = tail = NULL;
   for (i=1; i<=count; ++i)
      { dx = dy = ZERO;
	for (j=0; j<num; ++j)
	   { dx += ptr[j]->dx;
	     dy += ptr[j]->dy;
	     ptr[j] = ptr[j]->nexto;
	   }
	offset = (OFFSET_PTR) malloc( sizeof( struct OFFSET));
	offset->dx = dx;
	offset->dy = dy;
	if (!head)
	   head = offset;
	else
	   tail->nexto = offset;
	offset->prevo = tail;
	offset->nexto = NULL;
	tail = offset;
      }
   return( save_path(head,tail,count) );
}


/***************************************************************/
/*							       */
/*   TANGOpath_free - free up the malloced space occupied by   */
/*	"path."                                                */
/*							       */
/***************************************************************/

void
TANGOpath_free(num,p1,p2,p3,p4,p5,p6,p7,p8,p9,p10)
   int num;
   TANGO_PATH p1,p2,p3,p4,p5,p6,p7,p8,p9,p10;
{
   if (!TANGO__init) TANGOinit();

   if (num>0)
      path_free(p1);
   if (num>1)
      path_free(p2);
   if (num>2)
      path_free(p3);
   if (num>3)
      path_free(p4);
   if (num>4)
      path_free(p5);
   if (num>5)
      path_free(p6);
   if (num>6)
      path_free(p7);
   if (num>7)
      path_free(p8);
   if (num>8)
      path_free(p9);
   if (num>9)
      path_free(p10);
}



/***************************************************************/
/*							       */
/*   save_path - receive a list of offsets and save them as a  */
/*	new path.					       */
/*							       */
/***************************************************************/

TANGO_PATH
save_path(head,tail,count)
   OFFSET_PTR head;
   OFFSET_PTR tail;
   int count;
{
   TANGO_PATH new_path;

   new_path = (TANGO_PATH) malloc( sizeof (struct _PATH));
   new_path->head = head;
   new_path->tail = tail;
   new_path->count = count;
   new_path->nextp = NULL;
   new_path->prevp = NULL;
   return(new_path);
}



/***************************************************************/
/*							       */
/*   path_free - free up the malloced space occupied by path.  */
/*							       */
/***************************************************************/

void
path_free(path)
   TANGO_PATH path;
{
   OFFSET_PTR op,old;

   if (!path) return;
   op = path->head;
   while (op)
      { old = op;
	op = op->nexto;
	free(old);
      }
   free(path);
}

