/*************************************************************/
/*							     */
/*							     */
/*			TANGOPARSE.C			     */
/*							     */
/*							     */
/*************************************************************/
/*    Copyright 1989 Brown University -- John T. Stasko      */

#include  "tangolocal.h"
#include  "tangofsa.h"

int	      get_object_file();
int	      get_algoevents();
int	      get_animprocs();
int	      get_mappings();

ALGOEVT 	 Algoevts[MAXSIZE];
int		 Num_algoevts = 0;
ANIMPROC	 Animprocs[MAXSIZE];
int		 Num_animprocs = 0;



/***************************************************************/
/*							       */
/*  read_anim - this routine receives the animation name, a    */
/*	flag indicating whether to load the scene library,     */
/*	and return parameters for the window's real coords.    */
/*	It appends .anim to the incoming anim name to look     */
/*	for the control file.  From this file, it first gets   */
/*	the user-specified real coordinates for the animation  */
/*	window of the form {lx ty rx by}.  Next it gets        */
/*	any animation scene object files to dynamically load.  */
/*	Then it successively reads lines that define algorithm */
/*	events and their parameters.  Each line contains the   */
/*	event name and the types of its parameters.  Then it   */
/*	gets the names of animation scene functions created    */
/*	by the user.  They should be defined in the object     */
/*	files or in the library.  Finally, it reads the        */
/*	mappings from algorithm events to animation scenes.    */
/*							       */
/***************************************************************/

int
read_anim(name,loadlib,lx,by,rx,ty)
   char *name;
   int loadlib;
   WIN_COORD *lx,*by,*rx,*ty;
{
   FILE *fp;
   char anim_file[STRINGLENGTH],scenelib[STRINGLENGTH];

   sprintf(anim_file,"%s.anim",name);
   if ((fp = fopen(anim_file,"r")) == NULL)
      { fprintf(stderr,"tango: unable to open file %s\n",anim_file);
	return(0);
      }
   if (Tango_debug)
      printf("Reading file %s\n",anim_file);

   if (loadlib)
      { scenelib[0] = '\0';
	strcat(scenelib,BWEproject());
	strcat(scenelib,TANGO_SCENE_LIB);
	if (!DLload(scenelib))
	   { fprintf(stderr,"tango: unable to access TANGO scene lib %s\n",scenelib);
	     fclose(fp);
	     return(0);
	   }
      }

   if (!get_wincoords(fp,lx,by,rx,ty)) return(0);
   if (!get_object_file(fp)) return(0);
   if (!get_algoevents(fp)) return(0);
   if (!get_animprocs(fp)) return(0);
   if (!get_mappings(fp)) return(0);

   return(1);
}






/***************************************************************/
/*							       */
/*  get_wincoords -  read the control file; get the real-value */
/*	user-defined window coordinates.		       */
/*							       */
/***************************************************************/

int
get_wincoords(fp,lx,by,rx,ty)
   FILE *fp;
   WIN_COORD *lx,*by,*rx,*ty;
{
   char sep[STRINGLENGTH];

   if (Tango_debug)
      printf("Getting window coordinates  ");
   if (fscanf(fp,"%lf %lf %lf %lf",lx,ty,rx,by) != 4)
      { fprintf(stderr,"tango: missing or invalid window coords in .anim file\n");
	fclose(fp);
	return(0);
      }
   if ((*lx >= *rx) || (*ty >= *by))
      { fprintf(stderr,"tango: window coordinates are invalid.\n");
	fprintf("  They must be of form ->lx ty rx by<-  with rx>lx, by>ty\n");
	fclose(fp);
	return(0);
      }
   if (fscanf(fp,"%s",sep) == EOF)
      { fprintf(stderr,"tango: premature EOF in .anim file\n");
	fclose(fp);
	return(0);
      }
   if (strcmp(sep,"%%") != 0)
      { fprintf(stderr,"tango: missing %%%% separator after window coords\n");
	fclose(fp);
	return(0);
      }
   if (Tango_debug)
      printf("(%lf, %lf)   (%lf, %lf)\n",*lx,*ty,*rx,*by);
   return(1);
}







/***************************************************************/
/*							       */
/*  get_object_file - read the control file; get the names of  */
/*	.o files that have to be dynamically loaded; load them.*/
/*							       */
/***************************************************************/

int
get_object_file(fp)
   FILE *fp;
{
   char obj_file[STRINGLENGTH];

   if (Tango_debug)
      printf("Getting object file(s) ");
   while(1)    /* Get .o object files of compiled animation scenes */
      { if (fscanf(fp,"%s",obj_file) == EOF)
	   { fprintf(stderr,"tango: premature EOF in .anim file\n");
	     fclose(fp);
	     return(0);
	   }
	if (strcmp(obj_file,"%%") == 0)
	   break;
	else
	   if (!(DLload(obj_file)))
	      { fprintf(stderr,"tango: unable to load object file %s\n",obj_file);
		fclose(fp);
		return(0);
	      }
	if (Tango_debug)
	   printf("%s ",obj_file);
      }
   if (Tango_debug)
      printf("\n");
   return(1);
}







/***************************************************************/
/*							       */
/*  get_algoevents - get the names of the individual algorithm */
/*	events and their parameter types.  Save this info      */
/*	for use in the finite state machine.  Register these   */
/*	event types with the MSG package of FIELD.	       */
/*							       */
/***************************************************************/

int
get_algoevents(fp)
   FILE *fp;
{
   int	 num,i,pnum;
   char  name[STRINGLENGTH];
   char  pat[STRINGLENGTH];
   char  tail[STRINGLENGTH];
   char  line[STRINGLENGTH];
   int	 dfault[2];

   if (Tango_debug)
      printf("Getting algo operations(s)\n");
   num = 0;
   while(1)   /* Get algo events' names and parameters */
      { if (fscanf(fp,"%s",name) == EOF)
	   { fprintf(stderr,"tango: premature EOF in .anim file\n");
	     fclose(fp);
	     return(0);
	   }
	if (strcmp(name,"%%") == 0)
	   break;
	else
	   { if (*name == '*')  /* this algo_op expects a return string */
		{ if (*(name+1) == '\0')
		    { fprintf(stderr,"tango:  '*' found without immediate trailing algo operation\n");
		      fclose(fp);
		      return(0);
		    }
		  else
		     { Algoevts[num].callback = 1;
		       strcpy(Algoevts[num].name,name+1);
		     }
		}
	     else  /* plain algo_op expecting no return string */
		{ Algoevts[num].callback = 0;
		  strcpy(Algoevts[num].name,name);
		}
	     pnum = 1;
	     Algoevts[num].pat[0] = '\0';
	     if (fgets(line,STRINGLENGTH,fp) != NULL)
		{ i = 0;
		  while (i < strlen(line))
		     { switch(line[i]) {
			  case '%':
			     ++i;
			     if (line[i] == 'd')
				{ if (pnum == 1)
				     sprintf(tail,"%%1d"); /* get rid of space to start */
				  else
				     sprintf(tail," %%%dd",pnum);
				  strcat(Algoevts[num].pat,tail);
				  pnum++;
				}
			     else if (line[i] == 'f')
				{ if (pnum == 1)
				     sprintf(tail,"%%1f");
				  else
				     sprintf(tail," %%%df",pnum);
				  strcat(Algoevts[num].pat,tail);
				  pnum += 2;
				}
			     else if (line[i] == 's')
				{ if (pnum == 1)
				     sprintf(tail,"%%1s");
				  else
				     sprintf(tail," %%%ds",pnum);
				  strcat(Algoevts[num].pat,tail);
				  pnum += 1;
				}
			     else
				{ fprintf(stderr,"tango:  illegal algo operation parameter type %%%c\n",line[i]);
				  fclose(fp);
				  return(0);
				}
			     break;
			  case '\n': /* newline */
			  case '\t':  /* tab */
			  case ' ':  /* blank */
			     break;
			  default:
			     fprintf(stderr,"tango:  illegal algo operation parameter %%%c\n",line[i]);
			     fprintf(stderr,"operation name is %s\n",Algoevts[num].name);
			     fclose(fp);
			     return(0);
			     break;
		       } /* switch */
		       i++;
		     } /* while */
		  Algoevts[num].params = pnum-1;
		  if (Algoevts[num].params != 0)
		     sprintf(pat,"IE %s %%s %%s %%d %%1r",Algoevts[num].name);
		  else
		     sprintf(pat,"IE %s %%s %%s %%d",Algoevts[num].name);
		  dfault[0] = 0;
		  dfault[1] = num;  /* default value for algo evt number */
		  MSGregister(pat,fct_receive_event,2,dfault);
		  if (Tango_debug)
		     printf("     %s %s\n",Algoevts[num].name,Algoevts[num].pat);
		  num++;
		} /* if */
	     else
		{ fprintf(stderr,"tango: premature EOF in .anim file\n");
		  fclose(fp);
		  return(0);
		}
	   } /* else */
      } /* while */
   Num_algoevts = num;
   return(1);
}






/***************************************************************/
/*							       */
/*  get_animprocs - get the names of C functions that are the  */
/*	animation scenes.  Find their addresses via a DLlookup.*/
/*	Save this info. 				       */
/*							       */
/***************************************************************/

int
get_animprocs(fp)
   FILE *fp;
{
   int num;
   char fct_name[STRINGLENGTH];
   int	(*fct_addr)();

   if (Tango_debug)
      printf("Getting anim procedure(s) ");
   num = 0;
   while(1)
      { if (fscanf(fp,"%s",fct_name) == EOF)
	   { fprintf(stderr,"tango: premature EOF in .anim file\n");
	     fclose(fp);
	     return(0);
	   }
	if (strcmp(fct_name,"%%") == 0)
	   break;
	else
	   { if (!(fct_addr = DLlookup(fct_name)))
		{ fprintf(stderr,"tango: unable to access animation scene %s\n",fct_name);
		  fclose(fp);
		  return(0);
		}
	     else
		{ strcpy(Animprocs[num].name,fct_name);
		  Animprocs[num].addr = fct_addr;
		  num++;
		  if (Tango_debug)
		     printf("%s ",fct_name);
		}
	   }
      }
   Num_animprocs = num;
   if (Tango_debug)
      printf("\n");
   return(1);
}







/***************************************************************/
/*							       */
/*  get_mappings - read mappings from algorithm events to      */
/*	animation scenes, and store these mappings in the      */
/*	finite state machine.				       */
/*							       */
/***************************************************************/

int
get_mappings(fp)
   FILE *fp;
{
   TRANSITION_PTR t;
   INTNODE_PTR i;

   if (!read_mappings(fp)) return(0);
   return(1);
}




/***************************************************************/
/*							       */
/*  get_next_token - get the next string token from the given  */
/*	string, return it through the token parameter, then    */
/*	return a pointer to the subsequent starting location   */
/*	in the string after the token.			       */
/*							       */
/***************************************************************/

char *
get_next_token(str,token)
   char *str,*token;
{
   char *ptr,*inptr;
   char c;

   ptr = str;
   while (((c = *ptr)==' ') || (c=='\t') || (c=='\n'))
      ++ptr;
   if (c=='\0')
      return(NULL);

   inptr = token;
   *inptr = c;
   ++inptr;
   ++ptr;
   while (((c = *ptr)!=' ') && (c!='\0') && (c!='\n') && (c!='\t'))
      *(inptr++) = *(ptr++);
   *inptr = '\0';
   return(ptr);
}




