/***************************************************************/
/*							       */
/*	       TANGO ANIMATION PACKAGE (image)		       */
/*							       */
/***************************************************************/
/*    Copyright 1989 Brown University -- John T. Stasko        */

#include "tangolocal.h"

TANGO_IMAGE_PTR 	TANGO__image;
TANGO_LINE_PTR		TANGO__line;
TANGO_RECTANGLE_PTR	TANGO__rectangle;
TANGO_CIRCLE_PTR	TANGO__circle;
TANGO_TEXT_PTR		TANGO__text;
TANGO_ELLIPSE_PTR	TANGO__ellipse;
TANGO_POLYLINE_PTR	TANGO__polyline;
TANGO_POLYGON_PTR	TANGO__polygon;
TANGO_SPLINE_PTR	TANGO__spline;
TANGO_COMPOSITE_PTR	TANGO__composite;

WORM_TYPE		TANGO__type_image;
WORM_TYPE		TANGO__type_line;
WORM_TYPE		TANGO__type_rectangle;
WORM_TYPE		TANGO__type_circle;
WORM_TYPE		TANGO__type_text;
WORM_TYPE		TANGO__type_ellipse;
WORM_TYPE		TANGO__type_polyline;
WORM_TYPE		TANGO__type_polygon;
WORM_TYPE		TANGO__type_spline;
WORM_TYPE		TANGO__type_composite;

WORM_OPID		TANGO__opid_create;
WORM_OPID		TANGO__opid_draw;
WORM_OPID		TANGO__opid_trans;
WORM_OPID		TANGO__opid_loc;

/***************************************************************/
/**********	    local functions		      **********/
/***************************************************************/

TANGO_IMAGE	image_create();
void		compute_arrow_off();
void		arrow_poly_direction();
TANGO_IMAGE	read_composite();
int		decode_trans();
int		inquire_pathcolor();
int		get_color();
int		get_fillstyle();
ASH_LINE_STYLE	get_linestyle();

void	 add_erase();


void		line_create();
void		line_draw();
void		line_trans();
TANGO_LOC	line_loc();
void		line_bb();

void		rectangle_create();
void		rectangle_draw();
void		rectangle_trans();
TANGO_LOC	rectangle_loc();
void		rectangle_bb();

void		circle_create();
void		circle_draw();
void		circle_trans();
TANGO_LOC	circle_loc();
void		circle_bb();

void		text_create();
void		text_draw();
void		text_trans();
TANGO_LOC	text_loc();
void		text_bb();

void		ellipse_create();
void		ellipse_draw();
void		ellipse_trans();
TANGO_LOC	ellipse_loc();
void		ellipse_bb();

void		polyline_create();
void		polyline_draw();
void		polyline_trans();
TANGO_LOC	polyline_loc();
void		polyline_bb();

void		polygon_create();
void		polygon_draw();
void		polygon_trans();
TANGO_LOC	polygon_loc();
void		polygon_bb();

void		spline_create();
void		spline_draw();
void		spline_trans();
TANGO_LOC	spline_loc();
void		spline_bb();

void		composite_create();
void		composite_draw();
void		composite_trans();
TANGO_LOC	composite_loc();
void		composite_bb();


/***************************************************************/
/*							       */
/*   TANGOimage_create - create an TANGO_IMAGE with the given  */
/*	starting parameters and return it to the user.	       */
/*							       */
/***************************************************************/

TANGO_IMAGE
TANGOimage_create(type,p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15,p16,
		     p17,p18,p19,p20)
   TANGO_IMAGE_TYPE  type;
   int p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15,p16,p17,p18,p19,p20;
{
   TANGO_IMAGE	   new_image;
   TANGO_IMAGE_PTR imagedata;
   IMAGE_PTR	   im,imlist;

   if (!TANGO__init) TANGOinit();

   new_image = image_create(type,p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15,
				   p16,p17,p18,p19,p20);
   /* add this image onto the configuration draw list */
   im = (IMAGE_PTR) malloc( sizeof( struct IMAGE) );
   im->image = new_image;
   im->previ = NULL;
   im->nexti = TANGO__data->confighead;
   if (!TANGO__data->confighead)
      TANGO__data->configtail = im;
   else
      TANGO__data->confighead->previ = im;
   TANGO__data->confighead = im;
   new_image->inconfig = im;
   new_image->status = (TANGO__data->blting ? 1 : 2);

   imagedata = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,new_image->worm_obj);
   return(new_image);
}




/***************************************************************/
/*							       */
/*   TANGOimage_loc - return an TANGO_LOC corresponding to the */
/*	location of the given part of the given image.	       */
/*							       */
/***************************************************************/

TANGO_LOC
TANGOimage_loc(image,part)
   TANGO_IMAGE image;
   TANGO_PART_TYPE part;
{
   WIN_COORD x0,y0,x1,y1;
   int	     xsize,ysize;
   WIN_COORD corner;

   if (!TANGO__init) TANGOinit();

   return( (TANGO_LOC) WORMapply(TANGO__opid_loc,image->worm_obj,part) );
}




/***************************************************************/
/*							       */
/*   TANGO_image_init - make all the initial WORM calls to     */
/*	set up the types, supertypes, instances, and	       */
/*	methods necessary for the various types of images.     */
/*							       */
/***************************************************************/

void
TANGO_image_init()
{
   int i;

   TANGO__image = (TANGO_IMAGE_PTR) malloc( sizeof( struct _TANGO_IMAGE ) );
   TANGO__image->loc[0] = 0.0;
   TANGO__image->loc[1] = 0.0;
   TANGO__image->visible = 1;
   TANGO__type_image = WORMtype_define("WORM_TYPE_IMAGE",sizeof(struct _TANGO_IMAGE),
					 TANGO__image,0);

   TANGO__line = (TANGO_LINE_PTR) malloc( sizeof( struct _TANGO_LINE ) );
   TANGO__line->color = TANGO_COLOR_BLACK;
   TANGO__line->size[0] = 0.0;
   TANGO__line->size[1] = 0.0;
   TANGO__line->width = 0.0;
   TANGO__line->style = 1.0;
   TANGO__line->arrow = 0;
   for (i=0; i<=3; ++i)
      TANGO__line->arrowloc[i][0] = TANGO__line->arrowloc[i][1] = 0.0;
   TANGO__type_line = WORMtype_define("WORM_TYPE_LINE",sizeof(struct _TANGO_LINE),
					 TANGO__line,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_line,line_create,16,0);
   /* The param "16" above is the only one that changes from image to image.
      It is the num of parameters to the xxx_create function where doubles
      count as two.							    */
   WORMop_define("WORM_OP_DRAW",TANGO__type_line,line_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_line,line_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_line,line_loc,3,1);

   TANGO__rectangle = (TANGO_RECTANGLE_PTR) malloc( sizeof( struct _TANGO_RECTANGLE ) );
   TANGO__rectangle->color = TANGO_COLOR_BLACK;
   TANGO__rectangle->size[0] = 0.0;
   TANGO__rectangle->size[1] = 0.0;
   TANGO__rectangle->fill = 0.0;
   TANGO__type_rectangle = WORMtype_define("WORM_TYPE_RECTANGLE",sizeof(struct _TANGO_RECTANGLE),
					 TANGO__rectangle,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_rectangle,rectangle_create,13,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_rectangle,rectangle_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_rectangle,rectangle_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_rectangle,rectangle_loc,3,1);

   TANGO__circle = (TANGO_CIRCLE_PTR) malloc( sizeof( struct _TANGO_CIRCLE ) );
   TANGO__circle->color = TANGO_COLOR_BLACK;
   TANGO__circle->radius = 0.0;
   TANGO__circle->fill = 0.0;
   TANGO__type_circle = WORMtype_define("WORM_TYPE_CIRCLE",sizeof(struct _TANGO_CIRCLE),
					 TANGO__circle,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_circle,circle_create,11,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_circle,circle_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_circle,circle_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_circle,circle_loc,3,1);

   TANGO__ellipse = (TANGO_ELLIPSE_PTR) malloc( sizeof( struct _TANGO_ELLIPSE ) );
   TANGO__ellipse->color = TANGO_COLOR_BLACK;
   TANGO__ellipse->size[0] = 0.0;
   TANGO__ellipse->size[1] = 0.0;
   TANGO__ellipse->fill = 0.0;
   TANGO__type_ellipse = WORMtype_define("WORM_TYPE_ELLIPSE",sizeof(struct _TANGO_ELLIPSE),
					 TANGO__ellipse,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_ellipse,ellipse_create,13,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_ellipse,ellipse_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_ellipse,ellipse_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_ellipse,ellipse_loc,3,1);

   TANGO__polyline = (TANGO_POLYLINE_PTR) malloc( sizeof( struct _TANGO_POLYLINE ) );
   TANGO__polyline->color = TANGO_COLOR_BLACK;
   TANGO__polyline->vertices = 2;
   for (i=0; i<7; ++i)
     TANGO__polyline->vertex[i][0] = TANGO__polyline->vertex[i][1] = 0.0;
   TANGO__polyline->width = 0.0;
   TANGO__polyline->style = 1.0;
   TANGO__polyline->arrow = 0;
   for (i=0; i<=3; ++i)
      TANGO__polyline->arrowloc[i][0] = TANGO__polyline->arrowloc[i][1] = 0.0;
   TANGO__type_polyline = WORMtype_define("WORM_TYPE_POLYLINE",sizeof(struct _TANGO_POLYLINE),
					 TANGO__polyline,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_polyline,polyline_create,15,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_polyline,polyline_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_polyline,polyline_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_polyline,polyline_loc,3,1);

   TANGO__polygon = (TANGO_POLYGON_PTR) malloc( sizeof( struct _TANGO_POLYGON ) );
   TANGO__polygon->color = TANGO_COLOR_BLACK;
   TANGO__polygon->sides = 2;
   for (i=0; i<7; ++i)
     TANGO__polygon->vertex[i][0] = TANGO__polygon->vertex[i][1] = 0.0;
   TANGO__polygon->fill = 0.0;
   TANGO__type_polygon = WORMtype_define("WORM_TYPE_POLYGON",sizeof(struct _TANGO_POLYGON),
					 TANGO__polygon,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_polygon,polygon_create,12,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_polygon,polygon_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_polygon,polygon_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_polygon,polygon_loc,3,1);

   TANGO__spline = (TANGO_SPLINE_PTR) malloc( sizeof( struct _TANGO_SPLINE ) );
   TANGO__spline->color = TANGO_COLOR_BLACK;
   TANGO__spline->vertices = 2;
   for (i=0; i<7; ++i)
     TANGO__spline->vertex[i][0] = TANGO__spline->vertex[i][1] = 0.0;
   TANGO__spline->width = 0.0;
   TANGO__spline->style = 1.0;
   TANGO__type_spline = WORMtype_define("WORM_TYPE_SPLINE",sizeof(struct _TANGO_SPLINE),
					 TANGO__spline,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_spline,spline_create,14,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_spline,spline_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_spline,spline_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_spline,spline_loc,3,1);

   TANGO__text = (TANGO_TEXT_PTR) malloc( sizeof( struct _TANGO_TEXT ) );
   TANGO__text->color = TANGO_COLOR_BLACK;
   TANGO__text->font_name[0] = 0;
   TANGO__text->text[0] = 0;
   TANGO__text->fontid = 0;
   TANGO__text->orient = 1;
   TANGO__text->xext = TANGO__text->xoff = 0;
   TANGO__text->yext = TANGO__text->yoff = 0;
   TANGO__type_text = WORMtype_define("WORM_TYPE_TEXT",sizeof(struct _TANGO_TEXT),
					 TANGO__text,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_text,text_create,10,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_text,text_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_text,text_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_text,text_loc,3,1);

   TANGO__composite = (TANGO_COMPOSITE_PTR) malloc( sizeof( struct _TANGO_COMPOSITE ) );
   TANGO__composite->image_list = NULL;
   TANGO__type_composite = WORMtype_define("WORM_TYPE_COMPOSITE",sizeof(struct _TANGO_COMPOSITE),
					 TANGO__composite,1,TANGO__type_image);
   WORMop_define("WORM_OP_CREATE",TANGO__type_composite,composite_create,7,0);
   WORMop_define("WORM_OP_DRAW",TANGO__type_composite,composite_draw,7,1);
   WORMop_define("WORM_OP_TRANS",TANGO__type_composite,composite_trans,7,1);
   WORMop_define("WORM_OP_LOC",TANGO__type_composite,composite_loc,3,1);

   TANGO__opid_create = WORMop_find("WORM_OP_CREATE");
   TANGO__opid_draw = WORMop_find("WORM_OP_DRAW");
   TANGO__opid_trans = WORMop_find("WORM_OP_TRANS");
   TANGO__opid_loc = WORMop_find("WORM_OP_LOC");
}



/***************************************************************/
/*							       */
/*   image_create - set up the TANGO_IMAGE data structure	*/
/*	for a given type of image.			       */
/*							       */
/***************************************************************/

TANGO_IMAGE
image_create(type,p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15,p16,p17,p18,p19,p20)
   TANGO_IMAGE_TYPE  type;
   int p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,p13,p14,p15,p16,p17,p18,p19,p20;
{
   TANGO_IMAGE new_image;
   WORM_TYPE   worm_type;
   WORM_OBJ    worm_obj;

   new_image = (TANGO_IMAGE) malloc( sizeof( struct _IMAGE));
   new_image->type = type;
   switch (type)
   {
      case TANGO_IMAGE_TYPE_LINE:
	 worm_type = TANGO__type_line;
	 break;
      case TANGO_IMAGE_TYPE_RECTANGLE:
	 worm_type = TANGO__type_rectangle;
	 break;
      case TANGO_IMAGE_TYPE_CIRCLE:
	 worm_type = TANGO__type_circle;
	 break;
      case TANGO_IMAGE_TYPE_ELLIPSE:
	 worm_type = TANGO__type_ellipse;
	 break;
      case TANGO_IMAGE_TYPE_POLYLINE:
	 worm_type = TANGO__type_polyline;
	 break;
      case TANGO_IMAGE_TYPE_POLYGON:
	 worm_type = TANGO__type_polygon;
	 break;
      case TANGO_IMAGE_TYPE_SPLINE:
	 worm_type = TANGO__type_spline;
	 break;
      case TANGO_IMAGE_TYPE_TEXT:
	 worm_type = TANGO__type_text;
	 break;
      case TANGO_IMAGE_TYPE_COMPOSITE:
	 worm_type = TANGO__type_composite;
	 break;
      default:
	 fprintf(stderr,"Illegal TANGO_IMAGE_TYPE received by TANGOimage_create\n");
	 break;
   }

		/* worm_type is a valid WORM_OBJ */
   worm_obj = (WORM_OBJ) WORMapply(WORMop_find("Instance"),worm_type);
   WORMapply(TANGO__opid_create,worm_obj,p1,p2,p3,p4,p5,p6,p7,p8,p9,p10,p11,p12,
			 p13,p14,p15,p16,p17,p18,p19,p20);

   new_image->worm_obj = worm_obj;
   new_image->nexti = NULL;

   TANGO__data->numimages++;  /* used for determining type of clear_screen */
			      /* performed */
   return( new_image );
}




/***************************************************************/
/*							       */
/*  Method routines for the individual image types.  When a    */
/*	new image type xxx is created, it must supply the      */
/*	following four routines.			       */
/*							       */
/*	xxx_create - create a WORM_OBJ of this type, and set   */
/*	    its values to the initial parameters	       */
/*	xxx_draw - draw the image on the screen 	       */
/*	xxx_trans - update the image's data structure with     */
/*	    a response to a non-standard transition unit       */
/*	xxx_loc - return an TANGO_LOC corresponding to the     */
/*	    location of the given part of the image	       */
/*	xxx_bb - return the bounding box that surrounds the    */
/*	    particular image				       */
/*							       */
/***************************************************************/

/*  #######################	LINE	  ###################  */

void
line_create(obj,lx,ly,vis,color,sx,sy,width,style,arrow)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   WIN_COORD sx,sy;
   double    width;
   double    style;
   int	     arrow;
{
   TANGO_LINE_PTR   line;
   TANGO_IMAGE_PTR  image;

   line = (TANGO_LINE_PTR) WORMaccess(TANGO__type_line,obj);
   line->color = color;
   line->size[0] = sx;
   line->size[1] = sy;
   line->width = width;
   line->style = style;
   line->arrow = arrow;
   if ((line->arrow == 1) || (line->arrow == 3))
   compute_arrow_off(sx,sy,&(line->arrowloc[0][0]),&(line->arrowloc[0][1]),
			&(line->arrowloc[1][0]),&(line->arrowloc[1][1]));
   if ((line->arrow == 2) || (line->arrow == 3))
   compute_arrow_off(-sx,-sy,&(line->arrowloc[2][0]),&(line->arrowloc[2][1]),
			&(line->arrowloc[3][0]),&(line->arrowloc[3][1]));

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
line_draw(line,obj,timage,dx,dy)
   TANGO_LINE_PTR line;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD x0,y0,x1,y1, xr,yr;
   int ax[4],ay[4];
   ASH_LINE_STYLE old,lstyle;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;
   x1 = x0 + line->size[0];
   y1 = y0 + line->size[1];

   ASHanim_color(TANGO__data->window,line->color);
   lstyle = get_linestyle(line->width,line->style);

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   if (lstyle != ASH_STYLE_SOLID)
      old = ASHline_style(TANGO__data->window,lstyle);
   ASHline(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1));
   if ((line->arrow & 0x1) != 0)
      { ax[0] = XPIXEL(xr, x1 + line->arrowloc[0][0]);
	ay[0] = YPIXEL(yr, y1 + line->arrowloc[0][1]);
	ax[1] = XPIXEL(xr, x1 + line->arrowloc[1][0]);
	ay[1] = YPIXEL(yr, y1 + line->arrowloc[1][1]);
	ASHline(TANGO__data->window,XPIXEL(xr,x1),YPIXEL(yr,y1),ax[0],ay[0]);
	ASHline(TANGO__data->window,XPIXEL(xr,x1),YPIXEL(yr,y1),ax[1],ay[1]);
      }
   if ((line->arrow & 0x2) != 0)
      { ax[2] = XPIXEL(xr, x0 + line->arrowloc[2][0]);
	ay[2] = YPIXEL(yr, y0 + line->arrowloc[2][1]);
	ax[3] = XPIXEL(xr, x0 + line->arrowloc[3][0]);
	ay[3] = YPIXEL(yr, y0 + line->arrowloc[3][1]);
	ASHline(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),ax[2],ay[2]);
	ASHline(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),ax[3],ay[3]);
      }

   if (lstyle != ASH_STYLE_SOLID)
      ASHline_style(TANGO__data->window,old);

   add_erase(LINE,timage,lstyle,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1),0,NULL,
		0,NULL,NULL,line->arrow,ax,ay);
}



void
line_trans(line,obj,trans_type,dx,dy)
   TANGO_LINE_PTR     line;
   WORM_OBJ	      obj;
   TANGO_TRANS_TYPE   trans_type;
   WIN_COORD	      dx,dy;
{
   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 line->width += dx;
	 if (line->width < 0.0)
	    line->width = 0.0;
	 else if (line->width > 1.0)
	    line->width = 1.0;
	 line->style += dy;
	 if (line->style < 0.0)
	    line->style = 0.0;
	 else if (line->style > 1.0)
	    line->style = 1.0;
	 break;
      case TANGO_TRANS_TYPE_RESIZE:
	 line->size[0] += dx;
	 line->size[1] += dy;
	 if ((line->arrow == 1) || (line->arrow == 3))
	    compute_arrow_off(line->size[0],line->size[1],
			&(line->arrowloc[0][0]),&(line->arrowloc[0][1]),
			&(line->arrowloc[1][0]),&(line->arrowloc[1][1]));
	 if ((line->arrow == 2) || (line->arrow == 3))
	    compute_arrow_off(-line->size[0],-line->size[1],
			&(line->arrowloc[2][0]),&(line->arrowloc[2][1]),
			&(line->arrowloc[3][0]),&(line->arrowloc[3][1]));
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 line->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
}


TANGO_LOC
line_loc(line,obj,part)
   TANGO_LINE_PTR   line;
   WORM_OBJ	    obj;
   TANGO_PART_TYPE  part;
{
   WIN_COORD	   xmin,xmax,ymin,ymax;

   line_bb(obj,&xmin,&ymax,&xmax,&ymin);

   switch (part)
   {
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(xmin,ymin) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((xmin+xmax)/2.0,ymin) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(xmax,ymin) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(xmax,(ymin+ymax)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(xmax,ymax) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((xmin+xmax)/2.0,ymax) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(xmin,ymax) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(xmin,(ymin+ymax)/2.0) );
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((xmin+xmax)/2.0,(ymin+ymax)/2.0) );
   }
}



void
line_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_LINE_PTR line;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   line = (TANGO_LINE_PTR) WORMaccess(TANGO__type_line,obj);

   *lx = (line->size[0] > ZERO ? image->loc[0] : image->loc[0]+line->size[0]);
   *rx = (line->size[0] > ZERO ? image->loc[0]+line->size[0] : image->loc[0]);
   *ty = (line->size[1] > ZERO ? image->loc[1] : image->loc[1]+line->size[1]);
   *by = (line->size[1] > ZERO ? image->loc[1]+line->size[1] : image->loc[1]);
}



/*  #######################   RECTANGLE   ###################  */

void
rectangle_create(obj,lx,ly,vis,color,sx,sy,fill)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   WIN_COORD sx,sy;
   double    fill;
{
   TANGO_RECTANGLE_PTR	rectangle;
   TANGO_IMAGE_PTR	image;

   rectangle = (TANGO_RECTANGLE_PTR) WORMaccess(TANGO__type_rectangle,obj);
   rectangle->color = color;
   rectangle->size[0] = sx;
   rectangle->size[1] = sy;
   rectangle->fill = fill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
rectangle_draw(rectangle,obj,timage,dx,dy)
   TANGO_RECTANGLE_PTR rectangle;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD x0,y0,x1,y1, xr,yr;
   int fill_style,oldfill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;
   x1 = x0 + rectangle->size[0];
   y1 = y0 + rectangle->size[1];

   ASHanim_color(TANGO__data->window,rectangle->color);
   fill_style = get_fillstyle(rectangle->fill);

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   if (fill_style == TANGO_FILL_OUTLINE)
      { ASHbox(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1));
	add_erase(RECT,timage,ASH_STYLE_SOLID,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1));
      }
   else
      { oldfill = ASHfill(TANGO__data->window,fill_style);
	ASHrectangle(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1));
	add_erase(FILLRECT,timage,ASH_STYLE_SOLID,XPIXEL(xr,x0),YPIXEL(yr,y0),XPIXEL(xr,x1),YPIXEL(yr,y1));
	ASHfill(TANGO__data->window,oldfill);
      }
}



void
rectangle_trans(rectangle,obj,trans_type,dx,dy)
   TANGO_RECTANGLE_PTR	rectangle;
   WORM_OBJ		obj;
   TANGO_TRANS_TYPE	trans_type;
   WIN_COORD		dx,dy;
{
   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 rectangle->fill +=  dx;
	 if (rectangle->fill < 0.0)
	    rectangle->fill = 0.0;
	 else if (rectangle->fill > 1.0)
	    rectangle->fill = 1.0;
	 break;
      case TANGO_TRANS_TYPE_RESIZE:
	 rectangle->size[0] += dx;
	 rectangle->size[1] += dy;
	 if (rectangle->size[0] < ZERO)
	    rectangle->size[0] = ZERO;
	 if (rectangle->size[1] < ZERO)
	    rectangle->size[1] = ZERO;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 rectangle->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
}


TANGO_LOC
rectangle_loc(rectangle,obj,part)
   TANGO_RECTANGLE_PTR	rectangle;
   WORM_OBJ		obj;
   TANGO_PART_TYPE	part;
{
   WIN_COORD	   x0,y0,x1,y1;

   rectangle_bb(obj,&x0,&y1,&x1,&y0);

   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((x0 + x1)/2.0,(y0 + y1)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(x0,y0) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((x0+ x1)/2.0,y0) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(x1,y0) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(x1,(y0 + y1)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(x1,y1) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((x0 + x1)/2.0,y1) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(x0,y1) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(x0,(y0 + y1)/2.0) );
   }
}




void
rectangle_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_RECTANGLE_PTR rectangle;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   rectangle = (TANGO_RECTANGLE_PTR) WORMaccess(TANGO__type_rectangle,obj);
   *lx= image->loc[0];
   *ty= image->loc[1];
   *rx= *lx + rectangle->size[0];
   *by= *ty + rectangle->size[1];
}





/*  #######################    CIRCLE	  ###################  */

void
circle_create(obj,lx,ly,vis,color,radius,fill)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   WIN_COORD radius;
   double    fill;
{
   TANGO_CIRCLE_PTR  circle;
   TANGO_IMAGE_PTR   image;

   circle = (TANGO_CIRCLE_PTR) WORMaccess(TANGO__type_circle,obj);
   circle->color = color;
   circle->radius = radius;
   circle->fill = fill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
circle_draw(circle,obj,timage,dx,dy)
   TANGO_CIRCLE_PTR circle;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD x0,y0,x1,y1, xr,yr;
   int fill_style,oldfill;
   int rad;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;

   ASHanim_color(TANGO__data->window,circle->color);
   fill_style = get_fillstyle(circle->fill);

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);
   rad = (int) ROUND( circle->radius * (double) TANGO__data->width /
			  (TANGO__data->rx-TANGO__data->lx) );

   if (fill_style == TANGO_FILL_OUTLINE)
      { ASHcircle(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),rad);
	add_erase(CIRCLE,timage,ASH_STYLE_SOLID,XPIXEL(xr,x0),YPIXEL(yr,y0),rad);
      }
   else
      { oldfill = ASHfill(TANGO__data->window,fill_style);
	ASHfilled_circle(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),rad);
	add_erase(FILLCIRCLE,timage,ASH_STYLE_SOLID,XPIXEL(xr,x0),YPIXEL(yr,y0),rad);
	ASHfill(TANGO__data->window,oldfill);
      }
}



void
circle_trans(circle,obj,trans_type,dx,dy)
   TANGO_CIRCLE_PTR    circle;
   WORM_OBJ	       obj;
   TANGO_TRANS_TYPE    trans_type;
   WIN_COORD	      dx,dy;
{
   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 circle->fill += dx;
	 if (circle->fill < 0.0)
	    circle->fill = 0.0;
	 else if (circle->fill > 1.0)
	    circle->fill = 1.0;
	 break;
      case TANGO_TRANS_TYPE_RESIZE:
	 circle->radius += dx;
	 if (circle->radius < ZERO)
	    circle->radius = ZERO;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 circle->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
}

TANGO_LOC
circle_loc(circle,obj,part)
   TANGO_CIRCLE_PTR  circle;
   WORM_OBJ	     obj;
   TANGO_PART_TYPE   part;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD	   x0,y0,corner;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   x0 = image->loc[0];
   y0 = image->loc[1];
   corner = sqrt( (circle->radius * circle->radius) / 2.0 );
   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create(x0,y0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(x0-corner,y0-corner) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create(x0,y0-circle->radius) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(x0+corner,y0-corner) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(x0+circle->radius,y0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(x0+corner,y0+corner) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create(x0,y0+circle->radius) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(x0-corner,y0+corner) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(x0-circle->radius,y0) );
   }
}





void
circle_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_CIRCLE_PTR circle;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   circle = (TANGO_CIRCLE_PTR) WORMaccess(TANGO__type_circle,obj);

   *lx = image->loc[0] - circle->radius;
   *rx = image->loc[0] + circle->radius;
   *ty = image->loc[1] - circle->radius;
   *by = image->loc[1] + circle->radius;
}





/*  #######################   ELLIPSE	#####################  */

void
ellipse_create(obj,lx,ly,vis,color,rx,ry,fill)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   WIN_COORD rx,ry;
   double    fill;
{
   TANGO_ELLIPSE_PTR  ellipse;
   TANGO_IMAGE_PTR	image;

   ellipse = (TANGO_ELLIPSE_PTR) WORMaccess(TANGO__type_ellipse,obj);
   ellipse->color = color;
   ellipse->size[0] = rx;
   ellipse->size[1] = ry;
   ellipse->fill = fill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
ellipse_draw(ellipse,obj,timage,dx,dy)
   TANGO_ELLIPSE_PTR ellipse;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD cx,cy,rx,ry, xr,yr;
   int fill_style,oldfill;
   int radx,rady;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   cx = image->loc[0] + dx;
   cy = image->loc[1] + dy;
   rx = ellipse->size[0];
   ry = ellipse->size[1];

   ASHanim_color(TANGO__data->window,ellipse->color);
   fill_style = get_fillstyle(ellipse->fill);

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);
   radx = (int) ROUND( ellipse->size[0] * (double) TANGO__data->width /
			  (TANGO__data->rx-TANGO__data->lx) );
   rady = (int) ROUND( ellipse->size[1] * (double) TANGO__data->height /
			  (TANGO__data->by-TANGO__data->ty) );

   if (fill_style == TANGO_FILL_OUTLINE)
      { ASHellipse(TANGO__data->window,XPIXEL(xr,cx),YPIXEL(yr,cy),radx,rady);
	add_erase(ELLIPSE,timage,ASH_STYLE_SOLID,XPIXEL(xr,cx),YPIXEL(yr,cy),radx,rady);
      }
   else
      { oldfill = ASHfill(TANGO__data->window,fill_style);
	ASHfilled_ellipse(TANGO__data->window,XPIXEL(xr,cx),YPIXEL(yr,cy),radx,rady);
	add_erase(FILLELLIPSE,timage,ASH_STYLE_SOLID,XPIXEL(xr,cx),YPIXEL(yr,cy),radx,rady);
	ASHfill(TANGO__data->window,oldfill);
      }
}



void
ellipse_trans(ellipse,obj,trans_type,dx,dy)
   TANGO_ELLIPSE_PTR	ellipse;
   WORM_OBJ		obj;
   TANGO_TRANS_TYPE	trans_type;
   WIN_COORD		dx,dy;
{
   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 ellipse->fill +=  dx;
	 if (ellipse->fill < 0.0)
	    ellipse->fill = 0.0;
	 else if (ellipse->fill > 1.0)
	    ellipse->fill = 1.0;
	 break;
      case TANGO_TRANS_TYPE_RESIZE:
	 ellipse->size[0] += dx;
	 ellipse->size[1] += dy;
	 if (ellipse->size[0] < ZERO)
	    ellipse->size[0] = ZERO;
	 if (ellipse->size[1] < ZERO)
	    ellipse->size[1] = ZERO;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 ellipse->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
}


TANGO_LOC
ellipse_loc(ellipse,obj,part)
   TANGO_ELLIPSE_PTR	ellipse;
   WORM_OBJ		obj;
   TANGO_PART_TYPE	part;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD	   cx,cy,rx,ry;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   cx = image->loc[0];
   cy = image->loc[1];
   rx = ellipse->size[0];
   ry = ellipse->size[1];
   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create(cx,cy) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(cx-rx,cy-ry) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create(cx,cy-ry) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(cx+rx,cy-ry) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(cx+rx,cy) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(cx+rx,cy+ry) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create(cx,cy+ry) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(cx-rx,cy+ry) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(cx-rx,cy) );
   }
}




void
ellipse_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_ELLIPSE_PTR ellipse;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   ellipse = (TANGO_ELLIPSE_PTR) WORMaccess(TANGO__type_ellipse,obj);

   *lx = image->loc[0] - ellipse->size[0];
   *rx = image->loc[0] + ellipse->size[0];
   *ty = image->loc[1] - ellipse->size[1];
   *by = image->loc[1] + ellipse->size[1];
}





/*  #######################   POLYLINE	 ###################  */
/*  at most 8 vertices */

void
polyline_create(obj,lx,ly,vis,color,vertices,vertexX,vertexY,width,style,arrow)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   int	     vertices;
   double    vertexX[];
   double    vertexY[];
   double    width;
   double    style;
   int	     arrow;
{
   TANGO_POLYLINE_PTR polyline;
   TANGO_IMAGE_PTR    image;
   int		      i;
   double	      fx,fy,bx,by;

   polyline = (TANGO_POLYLINE_PTR) WORMaccess(TANGO__type_polyline,obj);
   polyline->color = color;
   polyline->vertices = vertices;
   for (i=0; i<vertices-1; ++i)
     { polyline->vertex[i][0] = vertexX[i];
       polyline->vertex[i][1] = vertexY[i];
     }
   polyline->width = width;
   polyline->style = style;
   polyline->arrow = arrow;
   if (polyline->arrow != 0)
      arrow_poly_direction(vertices,polyline->vertex,&fx,&fy,&bx,&by);
   if ((polyline->arrow == 1) || (polyline->arrow == 3))
      compute_arrow_off(fx,fy,&(polyline->arrowloc[0][0]),&(polyline->arrowloc[0][1]),
			&(polyline->arrowloc[1][0]),&(polyline->arrowloc[1][1]));
   if ((polyline->arrow == 2) || (polyline->arrow == 3))
      compute_arrow_off(bx,by,&(polyline->arrowloc[2][0]),&(polyline->arrowloc[2][1]),
			&(polyline->arrowloc[3][0]),&(polyline->arrowloc[3][1]));

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
polyline_draw(polyline,obj,timage,dx,dy)
   TANGO_POLYLINE_PTR polyline;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR    image;
   WIN_COORD	      x0, y0, xr, yr, x1, y1;
   int		      i, pixX[9], pixY[9], ax[4], ay[4];
   ASH_LINE_STYLE     lstyle,old;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;
   pixX[0] = XPIXEL(xr,x0);
   pixY[0] = YPIXEL(yr,y0);
   for (i=1; i<polyline->vertices; ++i)
      { pixX[i] = XPIXEL( xr,x0 + polyline->vertex[i-1][0] );
	pixY[i] = YPIXEL( yr,y0 + polyline->vertex[i-1][1] );
      }

   ASHanim_color(TANGO__data->window,polyline->color);
   lstyle = get_linestyle(polyline->width,polyline->style);

   if (lstyle != ASH_STYLE_SOLID)
      old = ASHline_style(TANGO__data->window,lstyle);
   ASHpolyline(TANGO__data->window,polyline->vertices,pixX,pixY);
   if ((polyline->arrow & 0x1) != 0)
      { x1 = x0 + polyline->vertex[polyline->vertices-2][0];
	y1 = y0 + polyline->vertex[polyline->vertices-2][1];
	ax[0] = XPIXEL(xr, x1 + polyline->arrowloc[0][0]);
	ay[0] = YPIXEL(yr, y1 + polyline->arrowloc[0][1]);
	ax[1] = XPIXEL(xr, x1 + polyline->arrowloc[1][0]);
	ay[1] = YPIXEL(yr, y1 + polyline->arrowloc[1][1]);
	ASHline(TANGO__data->window,XPIXEL(xr,x1),YPIXEL(yr,y1),ax[0],ay[0]);
	ASHline(TANGO__data->window,XPIXEL(xr,x1),YPIXEL(yr,y1),ax[1],ay[1]);
      }
   if ((polyline->arrow & 0x2) != 0)
      { ax[2] = XPIXEL(xr, x0 + polyline->arrowloc[2][0]);
	ay[2] = YPIXEL(yr, y0 + polyline->arrowloc[2][1]);
	ax[3] = XPIXEL(xr, x0 + polyline->arrowloc[3][0]);
	ay[3] = YPIXEL(yr, y0 + polyline->arrowloc[3][1]);
	ASHline(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),ax[2],ay[2]);
	ASHline(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),ax[3],ay[3]);
      }
   if (lstyle != ASH_STYLE_SOLID)
      ASHline_style(TANGO__data->window,old);

   add_erase(POLYLINE,timage,lstyle,0,0,0,0,0,NULL,polyline->vertices,pixX,pixY,polyline->arrow,ax,ay);
}



void
polyline_trans(polyline,obj,trans_type,dx,dy)
   TANGO_POLYLINE_PTR  polyline;
   WORM_OBJ		obj;
   TANGO_TRANS_TYPE	trans_type;
   WIN_COORD		dx,dy;
{
   int i,d;
   double fx,fy,bx,by;

   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 polyline->width += dx;
	 if (polyline->width < 0.0)
	    polyline->width = 0.0;
	 else if (polyline->width > 1.0)
	    polyline->width = 1.0;
	 polyline->style += dy;
	 if (polyline->style < 0.0)
	    polyline->style = 0.0;
	 else if (polyline->style > 1.0)
	    polyline->style = 1.0;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 polyline->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
   if ((d=decode_trans(trans_type)) > 0)
      { if (d >= 11)  /* GRABx */
	   { d = d%10;
	     polyline->vertex[d-1][0] += dx;
	     polyline->vertex[d-1][1] += dy;
	   }
	else if (d >= 1)  /* RESIZEx */
	   { for (i=d; i<polyline->vertices; ++i)
		{ polyline->vertex[i-1][0] += dx;
		  polyline->vertex[i-1][1] += dy;
		}
	   }
	if (polyline->arrow != 0)
	   arrow_poly_direction(polyline->vertices,polyline->vertex,&fx,&fy,&bx,&by);
	if ((polyline->arrow == 1) || (polyline->arrow == 3))
	   compute_arrow_off(fx,fy,&(polyline->arrowloc[0][0]),&(polyline->arrowloc[0][1]),
			     &(polyline->arrowloc[1][0]),&(polyline->arrowloc[1][1]));
	if ((polyline->arrow == 2) || (polyline->arrow == 3))
	   compute_arrow_off(bx,by,&(polyline->arrowloc[2][0]),&(polyline->arrowloc[2][1]),
			&(polyline->arrowloc[3][0]),&(polyline->arrowloc[3][1]));
      }
}


TANGO_LOC
polyline_loc(polyline,obj,part)
   TANGO_POLYLINE_PTR  polyline;
   WORM_OBJ		obj;
   TANGO_PART_TYPE	part;
{
   WIN_COORD	   xmin,ymin,xmax,ymax;

   polyline_bb(obj,&xmin,&ymax,&xmax,&ymin);

   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((xmin + xmax)/2.0,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(xmin,ymin) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((xmin+ xmax)/2.0,ymin) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(xmax,ymin) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(xmax,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(xmax,ymax) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((xmin + xmax)/2.0,ymax) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(xmin,ymax) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(xmin,(ymin + ymax)/2.0) );
   }
}





void
polyline_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_POLYLINE_PTR polyline;
   WIN_COORD x,y,xmin,xmax,ymin,ymax;
   int i;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   polyline = (TANGO_POLYLINE_PTR) WORMaccess(TANGO__type_polyline,obj);

   xmin = xmax = image->loc[0];
   ymin = ymax = image->loc[1];   /* Find the bounding box */
   for (i=0; i<(polyline->vertices - 1); ++i)
      { if ((x = (image->loc[0]+polyline->vertex[i][0])) > xmax) xmax = x;
	if ((x = (image->loc[0]+polyline->vertex[i][0])) < xmin) xmin = x;
	if ((y = (image->loc[1]+polyline->vertex[i][1])) > ymax) ymax = y;
	if ((y = (image->loc[1]+polyline->vertex[i][1])) < ymin) ymin = y;
      }
   *lx = xmin; *rx = xmax; *ty = ymin; *by = ymax;
}




/*  #######################   POLYGON	###################  */
/*  at most 8 sides */

void
polygon_create(obj,lx,ly,vis,color,sides,vertexX,vertexY,fill)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   int	     sides;
   double    vertexX[];
   double    vertexY[];
   double    fill;
{
   TANGO_POLYGON_PTR  polygon;
   TANGO_IMAGE_PTR    image;
   int		      i;

   polygon = (TANGO_POLYGON_PTR) WORMaccess(TANGO__type_polygon,obj);
   polygon->color = color;
   polygon->sides = sides;
   for (i=0; i<7; ++i)
     { polygon->vertex[i][0] = vertexX[i];
       polygon->vertex[i][1] = vertexY[i];
     }
   polygon->fill = fill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
polygon_draw(polygon,obj,timage,dx,dy)
   TANGO_POLYGON_PTR polygon;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   int i;
   TANGO_IMAGE_PTR image;
   WIN_COORD x0, y0, xr,yr;
   int pixX[9], pixY[9];
   int fill_style,oldfill;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;
   pixX[0] = XPIXEL(xr,x0);
   pixY[0] = YPIXEL(yr,y0);
   for (i=1; i<polygon->sides; ++i)
      { pixX[i] = XPIXEL( xr,x0 + polygon->vertex[i-1][0] );
	pixY[i] = YPIXEL( yr,y0 + polygon->vertex[i-1][1] );
      }

   ASHanim_color(TANGO__data->window,polygon->color);
   fill_style = get_fillstyle(polygon->fill);

   if (fill_style == TANGO_FILL_OUTLINE)
      { /* kludge, since polyline doesn't complete the polygon */
	pixX[i] = pixX[0];
	pixY[i] = pixY[0];
	ASHpolyline(TANGO__data->window,polygon->sides+1,pixX,pixY);
	add_erase(POLYGON,timage,ASH_STYLE_SOLID,0,0,0,0,0,NULL,polygon->sides+1,pixX,pixY);
      }
   else
      { oldfill = ASHfill(TANGO__data->window,fill_style);
	ASHpolygon(TANGO__data->window,polygon->sides,pixX,pixY);
	add_erase(FILLPOLYGON,timage,ASH_STYLE_SOLID,0,0,0,0,0,NULL,polygon->sides,pixX,pixY);
	ASHfill(TANGO__data->window,oldfill);
      }
}



void
polygon_trans(polygon,obj,trans_type,dx,dy)
   TANGO_POLYGON_PTR	polygon;
   WORM_OBJ		obj;
   TANGO_TRANS_TYPE	trans_type;
   WIN_COORD		dx,dy;
{
   int i,d;

   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 polygon->fill +=  dx;
	 if (polygon->fill < 0.0)
	    polygon->fill = 0.0;
	 else if (polygon->fill > 1.0)
	    polygon->fill = 1.0;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 polygon->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
   if ((d=decode_trans(trans_type)) >= 0) /* RESIZE or GRAB */
      { if (d >= 11)  /* GRABx */
	   { d = d%10;
	     polygon->vertex[d-1][0] += dx;
	     polygon->vertex[d-1][1] += dy;
	   }
	else if (d >= 1)  /* RESIZEx */
	   { for (i=d; i<polygon->sides; ++i)
		{ polygon->vertex[i-1][0] += dx;
		  polygon->vertex[i-1][1] += dy;
		}
	   }
      }
}


TANGO_LOC
polygon_loc(polygon,obj,part)
   TANGO_POLYGON_PTR  polygon;
   WORM_OBJ		obj;
   TANGO_PART_TYPE	part;
{
   WIN_COORD	   xmin,ymin,xmax,ymax;

   polygon_bb(obj,&xmin,&ymax,&xmax,&ymin);

   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((xmin + xmax)/2.0,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(xmin,ymin) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((xmin+ xmax)/2.0,ymin) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(xmax,ymin) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(xmax,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(xmax,ymax) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((xmin + xmax)/2.0,ymax) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(xmin,ymax) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(xmin,(ymin + ymax)/2.0) );
   }
}





void
polygon_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_POLYGON_PTR polygon;
   WIN_COORD x,y,xmin,xmax,ymin,ymax;
   int i;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   polygon = (TANGO_POLYGON_PTR) WORMaccess(TANGO__type_polygon,obj);

   xmin = xmax = image->loc[0];
   ymin = ymax = image->loc[1];   /* Find the bounding box */
   for (i=0; i<(polygon->sides - 1); ++i)
      { if ((x = (image->loc[0]+polygon->vertex[i][0])) > xmax) xmax = x;
	if ((x = (image->loc[0]+polygon->vertex[i][0])) < xmin) xmin = x;
	if ((y = (image->loc[1]+polygon->vertex[i][1])) > ymax) ymax = y;
	if ((y = (image->loc[1]+polygon->vertex[i][1])) < ymin) ymin = y;
      }
   *lx = xmin; *rx = xmax; *ty = ymin; *by = ymax;
}




/*  #######################    SPLINE	 ###################  */
/*  at most 8 vertices */

void
spline_create(obj,lx,ly,vis,color,vertices,vertexX,vertexY,width,style)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   int	     vertices;
   double    vertexX[];
   double    vertexY[];
   double    width;
   double    style;
{
   TANGO_SPLINE_PTR   spline;
   TANGO_IMAGE_PTR    image;
   int		      i;

   spline = (TANGO_SPLINE_PTR) WORMaccess(TANGO__type_spline,obj);
   spline->color = color;
   spline->vertices = vertices;
   for (i=0; i<7; ++i)
     { spline->vertex[i][0] = vertexX[i];
       spline->vertex[i][1] = vertexY[i];
     }
   spline->width = width;
   spline->style = style;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
spline_draw(spline,obj,timage,dx,dy)
   TANGO_SPLINE_PTR spline;
   WORM_OBJ	    obj;
   TANGO_IMAGE timage;
   WIN_COORD	    dx,dy;
{
   int		   i;
   TANGO_IMAGE_PTR image;
   WIN_COORD	   x0, y0, xr,yr;
   int		   pixX[9], pixY[9];
   ASH_LINE_STYLE  lstyle,old;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   x0 = image->loc[0] + dx;
   y0 = image->loc[1] + dy;
   pixX[0] = XPIXEL(xr,x0);
   pixY[0] = YPIXEL(yr,y0);
   for (i=1; i<spline->vertices; ++i)
      { pixX[i] = XPIXEL( xr,x0 + spline->vertex[i-1][0] );
	pixY[i] = YPIXEL( yr,y0 + spline->vertex[i-1][1] );
      }

   ASHanim_color(TANGO__data->window,spline->color);
   lstyle = get_linestyle(spline->width,spline->style);

   if (lstyle != ASH_STYLE_SOLID)
      old = ASHline_style(TANGO__data->window,lstyle);
   ASHspline(TANGO__data->window,spline->vertices,pixX,pixY);
   if (lstyle != ASH_STYLE_SOLID)
      ASHline_style(TANGO__data->window,old);

   add_erase(SPLINE,timage,lstyle,0,0,0,0,0,NULL,spline->vertices,pixX,pixY);
}



void
spline_trans(spline,obj,trans_type,dx,dy)
   TANGO_SPLINE_PTR	spline;
   WORM_OBJ		obj;
   TANGO_TRANS_TYPE	trans_type;
   WIN_COORD		dx,dy;
{
   int i,d;

   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 spline->width += dx;
	 if (spline->width < 0.0)
	    spline->width = 0.0;
	 else if (spline->width > 1.0)
	    spline->width = 1.0;
	 spline->style += dy;
	 if (spline->style < 0.0)
	    spline->style = 0.0;
	 else if (spline->style > 1.0)
	    spline->style = 1.0;
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 spline->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
   if ((d=decode_trans(trans_type)) >= 0) /* RESIZE or GRAB */
      { if (d >= 11)  /* GRABx */
	   { d = d%10;
	     spline->vertex[d-1][0] += dx;
	     spline->vertex[d-1][1] += dy;
	   }
	else if (d >= 1)  /* RESIZEx */
	   { for (i=d; i<spline->vertices; ++i)
		{ spline->vertex[i-1][0] += dx;
		  spline->vertex[i-1][1] += dy;
		}
	   }
      }
}


TANGO_LOC
spline_loc(spline,obj,part)
   TANGO_SPLINE_PTR	spline;
   WORM_OBJ		obj;
   TANGO_PART_TYPE	part;
{
   WIN_COORD	   xmin,ymin,xmax,ymax;

   spline_bb(obj,&xmin,&ymax,&xmax,&ymin);

   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((xmin + xmax)/2.0,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(xmin,ymin) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((xmin+ xmax)/2.0,ymin) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(xmax,ymin) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(xmax,(ymin + ymax)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(xmax,ymax) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((xmin + xmax)/2.0,ymax) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(xmin,ymax) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(xmin,(ymin + ymax)/2.0) );
   }
}





void
spline_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_SPLINE_PTR spline;
   WIN_COORD x,y,xmin,xmax,ymin,ymax;
   int i;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   spline = (TANGO_SPLINE_PTR) WORMaccess(TANGO__type_spline,obj);

   xmin = xmax = image->loc[0];
   ymin = ymax = image->loc[1];   /* Find the bounding box */
   for (i=0; i<(spline->vertices - 1); ++i)
      { if ((x = (image->loc[0]+spline->vertex[i][0])) > xmax) xmax = x;
	if ((x = (image->loc[0]+spline->vertex[i][0])) < xmin) xmin = x;
	if ((y = (image->loc[1]+spline->vertex[i][1])) > ymax) ymax = y;
	if ((y = (image->loc[1]+spline->vertex[i][1])) < ymin) ymin = y;
      }
   *lx = xmin; *rx = xmax; *ty = ymin; *by = ymax;
}




/*  #######################	TEXT	  ###################  */

void
text_create(obj,lx,ly,vis,color,fontname,string,orient)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   int	     color;
   char     *fontname;
   char     *string;
   int	     orient;
{
   TANGO_TEXT_PTR  text;
   TANGO_IMAGE_PTR image;
   int		   ex,ey,ox,oy;
   WIN_COORD	   dx,dy;

   text = (TANGO_TEXT_PTR) WORMaccess(TANGO__type_text,obj);
   text->color = color;

   if (fontname == NULL)
      { text->fontid = ASHinq_font(TANGO__data->window);
	text->font_name[0] =  0;
      }
   else
      { text->fontid = ASHloadfont(fontname);
	strcpy(text->font_name,fontname);
      }
   strcpy(text->text,string);
   text->orient = orient;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   ASHinq_text_extent(text->fontid,string,&(text->xext),&(text->yext));
   ASHinq_text_offset(text->fontid,string,&(text->xoff),&(text->yoff));

   image->loc[0] = lx;
   image->loc[1] = ly;
   image->visible = vis;
}


void
text_draw(text,obj,timage,dx,dy)
   TANGO_TEXT_PTR text;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   WIN_COORD x0,y0,x1,y1, xr,yr;
   int oldfont;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   if (text->orient == 0) /* lower left */
      { x0 = image->loc[0] + dx;
	y0 = image->loc[1] + dy;
      }
   else /* centered */
      { x0 = image->loc[0] - ((text->xext/2.0 - text->xoff) / TANGO__data->width) + dx;
	y0 = image->loc[1] + ((text->yext/2.0 - text->yoff) / TANGO__data->height) + dy;
      }

   xr = TANGO__data->width / (TANGO__data->rx - TANGO__data->lx);
   yr = TANGO__data->height / (TANGO__data->by - TANGO__data->ty);

   oldfont = ASHfont(TANGO__data->window,text->fontid);
   ASHanim_color(TANGO__data->window,text->color);
   ASHtext(TANGO__data->window,XPIXEL(xr,x0),YPIXEL(yr,y0),text->text);
   ASHfont(TANGO__data->window,oldfont);

   add_erase(TEXT,timage,ASH_STYLE_SOLID,XPIXEL(xr,x0),YPIXEL(yr,y0),0,0,text->fontid,text->text);
}



void
text_trans(text,obj,trans_type,dx,dy)
   TANGO_TEXT_PTR     text;
   WORM_OBJ	      obj;
   TANGO_TRANS_TYPE   trans_type;
   WIN_COORD	      dx,dy;
{
   switch( trans_type )
   {
      case TANGO_TRANS_TYPE_FILL:
	 break;
      case TANGO_TRANS_TYPE_RESIZE:
	 break;
      case TANGO_TRANS_TYPE_COLOR:
	 text->color = inquire_pathcolor(dx,dy);
	 break;
      default:
	 break;
   }
}


TANGO_LOC
text_loc(text,obj,part)
   TANGO_TEXT_PTR  text;
   WORM_OBJ	   obj;
   TANGO_PART_TYPE part;
{
   WIN_COORD	   lx,by,rx,ty;

   text_bb(obj,&lx,&by,&rx,&ty);

   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((lx + rx)/2.0,(by + ty)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(lx,ty) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((lx+ rx)/2.0,ty) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(rx,ty) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(rx,(ty + by)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(rx,by) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((lx + rx)/2.0,by) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(lx,by) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(lx,(by + ty)/2.0) );
   }

}




void
text_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_TEXT_PTR text;
   WIN_COORD ex,ox,ey,oy;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   text = (TANGO_TEXT_PTR) WORMaccess(TANGO__type_text,obj);

   ex = (double) text->xext / (double) TANGO__data->width;
   ox = (double) text->xoff / (double) TANGO__data->width;
   ey = (double) text->yext / (double) TANGO__data->height;
   oy = (double) text->yoff / (double) TANGO__data->height;
   if (text->orient == 0) /* lower left */
      { *lx = image->loc[0] - ox;
	*rx = image->loc[0] + ex - ox;
	*by = image->loc[1] + oy;
	*ty = image->loc[1] - ey + oy;
      }
   else
      { *lx = image->loc[0] - ex/2.0;
	*rx = image->loc[0] + ex/2.0;
	*by = image->loc[1] + ey/2.0;
	*ty = image->loc[1] - ey/2.0 - oy;  /* seems to need that little extra */
      }
}




/*  #######################  COMPOSITE	  ###################  */

void
composite_create(obj,lx,ly,vis,subimages)
   WORM_OBJ  obj;
   WIN_COORD lx,ly;
   int	     vis;
   TANGO_IMAGE_COMPONENT subimages[];
{
   TANGO_COMPOSITE_PTR	 composite;
   TANGO_IMAGE_PTR	 image;

   composite = (TANGO_COMPOSITE_PTR) WORMaccess(TANGO__type_composite,obj);
   composite->image_list = read_composite(subimages);

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   image->visible = vis;
   image->loc[0] = lx;
   image->loc[1] = ly;
}


void
composite_draw(composite,obj,timage,dx,dy)
   TANGO_COMPOSITE_PTR composite;
   WORM_OBJ obj;
   TANGO_IMAGE timage;
   WIN_COORD dx,dy;
{
   TANGO_IMAGE_PTR image;
   TANGO_IMAGE	   sub;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);

   if (!image->visible)
      return;

   for (sub=composite->image_list; sub; sub=sub->nexti)
      WORMapply(TANGO__opid_draw,sub->worm_obj,timage,image->loc[0],image->loc[1]);
   /* don't check vis here, but add absol pos of compos image to the draw */
}



void
composite_trans(composite,obj,trans_type,dx,dy)
   TANGO_COMPOSITE_PTR composite;
   WORM_OBJ	       obj;
   TANGO_TRANS_TYPE    trans_type;
   WIN_COORD	       dx,dy;
{
   TANGO_IMAGE	sub;

   for (sub=composite->image_list; sub; sub=sub->nexti)
      WORMapply(TANGO__opid_trans,sub->worm_obj,trans_type,dx,dy);

   /* probably want to have special handlers here rather than do it
      like a GROUP image, but for now, OK			    */
}


TANGO_LOC
composite_loc(composite,obj,part)
   TANGO_COMPOSITE_PTR	 composite;
   WORM_OBJ	   obj;
   TANGO_PART_TYPE  part;
{
   WIN_COORD lx,by,rx,ty;

   composite_bb(obj,&lx,&by,&rx,&ty);
   switch (part)
   {
      case TANGO_PART_TYPE_C :
	 return( TANGOloc_create((lx + rx)/2.0,(by + ty)/2.0) );
      case TANGO_PART_TYPE_NW :
	 return( TANGOloc_create(lx,ty) );
      case TANGO_PART_TYPE_N :
	 return( TANGOloc_create((lx+ rx)/2.0,ty) );
      case TANGO_PART_TYPE_NE :
	 return( TANGOloc_create(rx,ty) );
      case TANGO_PART_TYPE_E :
	 return( TANGOloc_create(rx,(ty + by)/2.0) );
      case TANGO_PART_TYPE_SE :
	 return( TANGOloc_create(rx,by) );
      case TANGO_PART_TYPE_S :
	 return( TANGOloc_create((lx + rx)/2.0,by) );
      case TANGO_PART_TYPE_SW :
	 return( TANGOloc_create(lx,by) );
      case TANGO_PART_TYPE_W :
	 return( TANGOloc_create(lx,(by + ty)/2.0) );
   }
}




void
composite_bb(obj,lx,by,rx,ty)
   WORM_OBJ obj;
   WIN_COORD *lx,*by,*rx,*ty;
{
   TANGO_IMAGE_PTR image;
   TANGO_COMPOSITE_PTR composite;
   TANGO_IMAGE	sub;
   WIN_COORD x0,y0,x1,y1;

   image = (TANGO_IMAGE_PTR) WORMaccess(TANGO__type_image,obj);
   composite = (TANGO_COMPOSITE_PTR) WORMaccess(TANGO__type_composite,obj);

   TANGO_bounding_box(composite->image_list->worm_obj,lx,by,rx,ty);  /* get first */
   for (sub=composite->image_list->nexti; sub; sub=sub->nexti)
      { switch (sub->type)
	   { case TANGO_IMAGE_TYPE_LINE:
		line_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_RECTANGLE:
		rectangle_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_CIRCLE:
		circle_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_ELLIPSE:
		ellipse_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_POLYLINE:
		polyline_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_POLYGON:
		polygon_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_SPLINE:
		spline_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     case TANGO_IMAGE_TYPE_TEXT:
		text_bb(sub->worm_obj,&x0,&y1,&x1,&y0);
		break;
	     default:
		;
	    }
	if (x0 < *lx)
	   *lx = x0;
	if (x1 > *rx)
	   *rx = x1;
	if (y0 < *ty)
	   *ty = y0;
	if (y1 > *by)
	   *by = y1;
      }
   *lx += image->loc[0];  /* composite so add in overall loc to locals */
   *rx += image->loc[0];
   *by += image->loc[1];
   *ty += image->loc[1];
}




/*  #########################################################  */
/*   read_composite - read a composite TANGO_IMAGE_TYPE from   */
/*	a structure, create the sub-images, and return the     */
/*	list of them.					       */
/*  #########################################################  */

TANGO_IMAGE
read_composite(images)
   TANGO_IMAGE_COMPONENT images[];
{
   TANGO_IMAGE_TYPE  type;
   char 	    *args,*p;
   int		     num,i;
   TANGO_IMAGE	     image,head,tail;
   WIN_COORD	     lx,ly,sx,sy,rad,vx[7],vy[7];
   int		     color,vis,arrow,orient,vertices;
   double	     wid,sty,fill;
   char 	     colorstr[STRINGLENGTH];
   char 	     fontname[STRINGLENGTH];
   char 	    *fname;
   char 	     text[STRINGLENGTH];

   head = tail = NULL;
   num = 0;
   do
      { type = images[num].type;
	args = images[num].args;
	if (type == TANGO_IMAGE_TYPE_COMPOSITE) break;

	switch (type)
	{
	   case TANGO_IMAGE_TYPE_LINE:
	      sscanf(args,"%lf %lf %s %lf %lf %lf %lf %d",&lx,&ly,colorstr,&sx,&sy,&wid,&sty,&arrow);
	      color = get_color(colorstr);
	      image = image_create(TANGO_IMAGE_TYPE_LINE,lx,ly,1,color,sx,sy,wid,sty,arrow);
	      break;
	   case TANGO_IMAGE_TYPE_RECTANGLE:
	      sscanf(args,"%lf %lf %s %lf %lf %lf",&lx,&ly,colorstr,&sx,&sy,&fill);
	      color = get_color(colorstr);
	      image = image_create(TANGO_IMAGE_TYPE_RECTANGLE,lx,ly,1,color,sx,sy,fill);
	      break;
	   case TANGO_IMAGE_TYPE_CIRCLE:
	      sscanf(args,"%lf %lf %s %lf %lf",&lx,&ly,colorstr,&rad,&fill);
	      color = get_color(colorstr);
	      image = image_create(TANGO_IMAGE_TYPE_CIRCLE,lx,ly,1,color,rad,fill);
	      break;
	   case TANGO_IMAGE_TYPE_ELLIPSE:
	      sscanf(args,"%lf %lf %s %lf %lf %lf",&lx,&ly,colorstr,&sx,&sy,&fill);
	      color = get_color(colorstr);
	      image = image_create(TANGO_IMAGE_TYPE_ELLIPSE,lx,ly,1,color,sx,sy,fill);
	      break;
	   case TANGO_IMAGE_TYPE_POLYLINE:
	      sscanf(args,"%lf %lf %s %lf %lf %d %d",&lx,&ly,colorstr,&wid,&sty,&arrow,&vertices);
	      color = get_color(colorstr);
	      p = args;
	      while (*p == ' ') p++;
	      for (i=1; i<=6; ++i)	   /* get past prelim args */
		 { while (*p != ' ') p++;
		   while (*p == ' ') p++;
		 }
	      for (i=0; i<vertices-1; ++i)
		 { sscanf(p,"%lf",&(vx[i]));
		   while (*p != ' ') p++;
		   while (*p == ' ') p++;
		   sscanf(p,"%lf",&(vy[i]));
		   while (*p != ' ')
		      { if (*p == '\0') break;
			p++;
		      }
		   if (*p != '\0')
		      while (*p == ' ') p++;
		 }
	      image = image_create(TANGO_IMAGE_TYPE_POLYLINE,lx,ly,1,color,vertices,
				     vx,vy,wid,sty,arrow);
	      break;
	   case TANGO_IMAGE_TYPE_POLYGON:
	      sscanf(args,"%lf %lf %s %lf %d",&lx,&ly,colorstr,&fill,&vertices);
	      color = get_color(colorstr);
	      p = args;
	      while (*p == ' ') p++;
	      for (i=1; i<=5; ++i)	   /* get past prelim args */
		 { while (*p != ' ') p++;
		   while (*p == ' ') p++;
		 }
	      for (i=0; i<vertices-1; ++i)
		 { sscanf(p,"%lf",&(vx[i]));
		   while (*p != ' ') p++;
		   while (*p == ' ') p++;
		   sscanf(p,"%lf",&(vy[i]));
		   while (*p != ' ')
		      { if (*p == '\0') break;
			p++;
		      }
		   if (*p != '\0')
		      while (*p == ' ') p++;
		 }
	      image = image_create(TANGO_IMAGE_TYPE_POLYGON,lx,ly,1,color,vertices,
				     vx,vy,fill);
	      break;
	   case TANGO_IMAGE_TYPE_SPLINE:
	      sscanf(args,"%lf %lf %s %lf %lf %d",&lx,&ly,colorstr,&wid,&sty,&vertices);
	      color = get_color(colorstr);
	      p = args;
	      while (*p == ' ') p++;
	      for (i=1; i<=6; ++i)	   /* get past prelim args */
		 { while (*p != ' ') p++;
		   while (*p == ' ') p++;
		 }
	      for (i=0; i<vertices-1; ++i)
		 { sscanf(p,"%lf",&(vx[i]));
		   while (*p != ' ') p++;
		   while (*p == ' ') p++;
		   sscanf(p,"%lf",&(vy[i]));
		   while (*p != ' ')
		      { if (*p == '\0') break;
			p++;
		      }
		   if (*p != '\0')
		      while (*p == ' ') p++;
		 }
	      image = image_create(TANGO_IMAGE_TYPE_SPLINE,lx,ly,1,color,vertices,
				     vx,vy,wid,sty);
	      break;
	    case TANGO_IMAGE_TYPE_TEXT:
	      sscanf(args,"%lf %lf %s %s %s %d",&lx,&ly,colorstr,fontname,text,&orient);
	      color = get_color(colorstr);
	      if ((strcmp(fontname,"NULL") == 0) || (strcmp(fontname,"0") == 0))
		 fname = NULL;
	      else
		 fname = fontname;
	      image = image_create(TANGO_IMAGE_TYPE_TEXT,lx,ly,1,color,fname,text,orient);
	      break;
	   default :
	      break;
	}
	++num;

	image->nexti = NULL;
	if (!head)
	   head = image;
	else
	   tail->nexti = image;
	tail = image;
      } while(1==1);

   return(head);
}



/*  #########################################################  */
/*   compute_arrow_off - figure out what the offsets should    */
/*	be for the points of an arrow.			       */
/*	I haven't got a clue how this works, but it just does. */
/*	Many thanks to Eric Golin for code.		       */
/*  #########################################################  */

#define   PI   ((double)3.14159)
#define   ARC_HEAD_LENGTH   ((double)0.025)
#define   ARC_HEAD_ANGLE    PI/2.5

void
compute_arrow_off(dx,dy,adx1,ady1,adx2,ady2)
   double dx,dy;
   double *adx1,*ady1,*adx2,*ady2;
{
   double arclen,len,theta,alpha,beta;

   if ((dx == 0.0) && (dy == 0.0))
      { *adx1 = *adx2 = *ady1 = *ady2 = 0.0;
	return;
      }

   arclen = ARC_HEAD_LENGTH;
   len = sqrt((dx*dx) + (dy*dy));
   if (len < arclen)
      arclen = len;

   theta = atan2(dx,dy);
   alpha = -(theta + ARC_HEAD_ANGLE);
   beta = PI - (theta - ARC_HEAD_ANGLE);
   *adx1 = arclen * cos(alpha);
   *ady1 = arclen * sin(alpha);
   *adx2 = arclen * cos(beta);
   *ady2 = arclen * sin(beta);
}



/*  #########################################################  */
/*   arrow_poly_direction - determine what direction to        */
/*	point the arrowheads on a polyline.  Tricky because    */
/*	offsets at the two endpoints may be 0,0.  We want the  */
/*	last non-negative pair. 			       */
/*  #########################################################  */

void
arrow_poly_direction(num,vert,fx,fy,bx,by)
   int num;
   WIN_COORD vert[7][2],*fx,*fy,*bx,*by;
{
   WIN_COORD tempx[8],tempy[8];
   int	     i;

   for (i=1; i<num; ++i)   /* need a starting position too */
      { tempx[i] = vert[i-1][0];
	tempy[i] = vert[i-1][1];
      }
   tempx[0] = tempy[0] = 0.0;

   for (i=num-1; i>=1; --i)
      { *fx = tempx[i] - tempx[i-1];
	*fy = tempy[i] - tempy[i-1];
	if ((*fx != 0.0) || (*fy != 0.0))
	   break;
      }

   for (i=1; i<num; ++i)
      { *bx = tempx[i-1] - tempx[i];
	*by = tempy[i-1] - tempy[i];
	if ((*bx != 0.0) || (*by != 0.0))
	   break;
      }
}



/*  #########################################################  */
/*   decode_trans - give a number representation of the        */
/*	transition type.  Anything but RESIZEs and GRABs is    */
/*	-1.  RESIZE run in the single digits, and GRABs in     */
/*	the tens.					       */
/*  #########################################################  */

int
decode_trans(t)
   TANGO_TRANS_TYPE t;
{
   switch (t)
   { case TANGO_TRANS_TYPE_RESIZE:
	return(0);
     case TANGO_TRANS_TYPE_RESIZE1:
	return(1);
     case TANGO_TRANS_TYPE_RESIZE2:
	return(2);
     case TANGO_TRANS_TYPE_RESIZE3:
	return(3);
     case TANGO_TRANS_TYPE_RESIZE4:
	return(4);
     case TANGO_TRANS_TYPE_RESIZE5:
	return(5);
     case TANGO_TRANS_TYPE_RESIZE6:
	return(6);
     case TANGO_TRANS_TYPE_RESIZE7:
	return(7);
     case TANGO_TRANS_TYPE_GRAB1:
	return(11);
     case TANGO_TRANS_TYPE_GRAB2:
	return(12);
     case TANGO_TRANS_TYPE_GRAB3:
	return(13);
     case TANGO_TRANS_TYPE_GRAB4:
	return(14);
     case TANGO_TRANS_TYPE_GRAB5:
	return(15);
     case TANGO_TRANS_TYPE_GRAB6:
	return(16);
     case TANGO_TRANS_TYPE_GRAB7:
	return(17);
     default:
	return(-1);
   }
}



/***************************************************************/
/*							       */
/*   inquire_pathcolor - given the dx and dy offsets of a      */
/*	transition unit, figure out which color to change to.  */
/*							       */
/***************************************************************/

int
inquire_pathcolor(dx,dy)
   WIN_COORD dx,dy;
{
   if (dx > 0.0)
      { if (dy > 0.0)
	   { if (ABS(dx) >= ABS(dy))
		return(TANGO_COLOR_MAROON);
	     else
		return(TANGO_COLOR_BLUE);
	   }
	else
	   { if (ABS(dx) > ABS(dy))
		return(TANGO_COLOR_WHITE);
	     else
		return(TANGO_COLOR_BLACK);
	   }
      }
   else if (dx < 0.0)
      { if (dy < 0.0)
	   { if (ABS(dx) >= ABS(dy))
		return(TANGO_COLOR_ORANGE);
	     else
		return(TANGO_COLOR_RED);
	   }
	else
	   { if (ABS(dx) > ABS(dy))
		return(TANGO_COLOR_YELLOW);
	     else
		return(TANGO_COLOR_GREEN);
	   }
      }
   else /* dx == 0.0 */
      { if (dy < 0.0)
	   return(TANGO_COLOR_RED);
	else if (dy > 0.0)
	   return(TANGO_COLOR_BLUE);
	else
	   return(TANGO_COLOR_WHITE);
      }
}





/***************************************************************/
/*							       */
/*   get_color - given a string of a TANGO_COLOR, return its   */
/*	integer value.					       */
/*							       */
/***************************************************************/

int
get_color(str)
   char *str;
{
   if (strcmp("TANGO_COLOR_WHITE",str) == 0)
      return( TANGO_COLOR_WHITE );
   else if (strcmp("TANGO_COLOR_BLACK",str) == 0)
      return( TANGO_COLOR_BLACK );
   else if (strcmp("TANGO_COLOR_RED",str) == 0)
      return( TANGO_COLOR_RED );
   else if (strcmp("TANGO_COLOR_ORANGE",str) == 0)
      return( TANGO_COLOR_ORANGE );
   else if (strcmp("TANGO_COLOR_YELLOW",str) == 0)
      return( TANGO_COLOR_YELLOW );
   else if (strcmp("TANGO_COLOR_GREEN",str) == 0)
      return( TANGO_COLOR_GREEN );
   else if (strcmp("TANGO_COLOR_BLUE",str) == 0)
      return( TANGO_COLOR_BLUE );
   else if (strcmp("TANGO_COLOR_MAROON",str) == 0)
      return( TANGO_COLOR_MAROON );
   else
      return( TANGO_COLOR_BLACK );
}




/***************************************************************/
/*							       */
/*   get_fillstyle - given a fill value between 0.0 and 1.0    */
/*	return the ASH fill style it corresponds to.	       */
/*							       */
/***************************************************************/

int
get_fillstyle(val)
   double val;
{
   if (val < 0.111111)
      return(TANGO_BASE_FILL);
   if (val < 0.222222)
      return(TANGO_BASE_FILL + 1);
   else if (val < 0.333333)
      return(TANGO_BASE_FILL + 2);
   else if (val < 0.444444)
      return(TANGO_BASE_FILL + 3);
   else if (val < 0.555555)
      return(TANGO_BASE_FILL + 4);
   else if (val < 0.666666)
      return(TANGO_BASE_FILL + 5);
   else if (val < 0.777777)
      return(TANGO_BASE_FILL + 6);
   else if (val < 0.888888)
      return(TANGO_BASE_FILL + 7);
   else
      return(TANGO_BASE_FILL + 8);
}



/***************************************************************/
/*							       */
/*   get_linestyle - given a width and a style, return an      */
/*	ASH_FILL_STYLE. 				       */
/*							       */
/***************************************************************/

ASH_LINE_STYLE
get_linestyle(width,style)
   double width,style;
{
   if (width > 0.666667)
      { if (style < 0.333333)
	   return(ASH_STYLE_THICKER_DOTTED);
	else if (style < 0.666667)
	   return(ASH_STYLE_THICKER_DASHED);
	else
	   return(ASH_STYLE_THICKER);
      }
   else if (width > 0.333333)
      { if (style < 0.333333)
	   return(ASH_STYLE_THICK_DOTTED);
	else if (style < 0.666667)
	   return(ASH_STYLE_THICK_DASHED);
	else
	   return(ASH_STYLE_THICK);
      }
   else
      { if (style < 0.333333)
	   return(ASH_STYLE_DOTTED);
	else if (style < 0.666667)
	   return(ASH_STYLE_DASHED);
	else
	   return(ASH_STYLE_SOLID);
      }
}




/***************************************************************/
/*							       */
/*   TANGO_bounding_box - return the bounding box coordinates  */
/*	for the given image.				       */
/*							       */
/***************************************************************/

void
TANGO_bounding_box(im,lx,by,rx,ty)
   TANGO_IMAGE im;
   WIN_COORD *lx,*by,*rx,*ty;
{
   switch (im->type)
   { case TANGO_IMAGE_TYPE_LINE:
	line_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_RECTANGLE:
	rectangle_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_CIRCLE:
	circle_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_ELLIPSE:
	ellipse_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_POLYLINE:
	polyline_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_POLYGON:
	polygon_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_SPLINE:
	spline_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_TEXT:
	text_bb(im->worm_obj,lx,by,rx,ty);
	break;
     case TANGO_IMAGE_TYPE_COMPOSITE:
	composite_bb(im->worm_obj,lx,by,rx,ty);
	break;
     default:
	*lx = *by = *rx = *ty = 0.0;
    }
}




/* @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ */
/* routines used to help speed up animations (should be removed later) */
void
TANGO_clear_screen()
{
   ERASER_PTR e,f,prev;
   int oldfont;
   ASH_FILL_STYLE oldfill;

   if (TANGO__data->motionblur) return;  /* used for window dumps; best to
					    set the var through the debugger */
   if (!TANGO__data->blting)
      TANGO__data->framenum = (TANGO__data->framenum + 1) % 2;
   if (TANGO__data->numimages > TANGO__data->imageclear)
      { ASHclear_box(TANGO__data->window,0,TANGO__data->height,TANGO__data->width,0);
	e = TANGO__data->erase[TANGO__data->framenum];
	while (e)
	   { f = e;
	     e = e->next;
	     free(f);
	   }
	TANGO__data->erase[TANGO__data->framenum] = NULL;
      }
   else
      { ASHanim_color(TANGO__data->window,TANGO_COLOR_WHITE);  /* background */
	oldfill = ASHfill(TANGO__data->window,ASH_FILL_SOLID);
	prev = NULL;
	e = TANGO__data->erase[TANGO__data->framenum];
	while (e)
	   { if ((TANGO__data->mode == 0) || (e->image->status > 0))  /* update it */
		{ switch(e->itype)
		     { case LINE:
			  ASHline_style(TANGO__data->window,e->thick);
			  ASHline(TANGO__data->window,e->x0,e->y0,e->x1,e->y1);
			  if (e->arrow & 0x1)
			     { ASHline(TANGO__data->window,e->x1,e->y1,e->arrowx[0],e->arrowy[0]);
			       ASHline(TANGO__data->window,e->x1,e->y1,e->arrowx[1],e->arrowy[1]);
			     }
			  if (e->arrow & 0x2)
			     { ASHline(TANGO__data->window,e->x0,e->y0,e->arrowx[2],e->arrowy[2]);
			       ASHline(TANGO__data->window,e->x0,e->y0,e->arrowx[3],e->arrowy[3]);
			     }
			  ASHline_style(TANGO__data->window,ASH_STYLE_SOLID);
			  break;
		       case RECT:
			  ASHbox(TANGO__data->window,e->x0,e->y0,e->x1,e->y1);
			  break;
		       case FILLRECT:
			  ASHrectangle(TANGO__data->window,e->x0,e->y0,e->x1,e->y1);
			  break;
		       case CIRCLE:
			  ASHcircle(TANGO__data->window,e->x0,e->y0,e->x1);
			  break;
		       case FILLCIRCLE:
			  ASHfilled_circle(TANGO__data->window,e->x0,e->y0,e->x1);
			  break;
		       case ELLIPSE:
			  ASHellipse(TANGO__data->window,e->x0,e->y0,e->x1,e->y1);
			  break;
		       case FILLELLIPSE:
			  ASHfilled_ellipse(TANGO__data->window,e->x0,e->y0,e->x1,e->y1);
			  break;
		       case POLYLINE:
			  ASHline_style(TANGO__data->window,e->thick);
			  ASHpolyline(TANGO__data->window,e->sides,e->xs,e->ys);
			  if (e->arrow & 0x1)
			     { ASHline(TANGO__data->window,e->xs[e->sides-1],e->ys[e->sides-1],e->arrowx[0],e->arrowy[0]);
			       ASHline(TANGO__data->window,e->xs[e->sides-1],e->ys[e->sides-1],e->arrowx[1],e->arrowy[1]);
			     }
			  if (e->arrow & 0x2)
			     { ASHline(TANGO__data->window,e->xs[0],e->ys[0],e->arrowx[2],e->arrowy[2]);
			       ASHline(TANGO__data->window,e->xs[0],e->ys[0],e->arrowx[3],e->arrowy[3]);
			     }
			  ASHline_style(TANGO__data->window,ASH_STYLE_SOLID);
			  break;
		       case POLYGON:
			  ASHpolyline(TANGO__data->window,e->sides,e->xs,e->ys);
			  break;
		       case FILLPOLYGON:
			  ASHpolygon(TANGO__data->window,e->sides,e->xs,e->ys);
			  break;
		       case SPLINE:
			  ASHline_style(TANGO__data->window,e->thick);
			  ASHspline(TANGO__data->window,e->sides,e->xs,e->ys);
			  ASHline_style(TANGO__data->window,ASH_STYLE_SOLID);
			  break;
		       case TEXT:
			  oldfont = ASHfont(TANGO__data->window,e->fontid);
			  ASHtext(TANGO__data->window,e->x0,e->y0,e->text);
			  ASHfont(TANGO__data->window,oldfont);
			  break;
		     };  /* switch */
		  f = e;
		  if (prev)
		     prev->next = e->next;
		  else
		     TANGO__data->erase[TANGO__data->framenum] = e->next;
		  e = e->next;
		  free(f);
		} /* if */
	     else
		{ prev = e;
		  e = e->next;
		}
	   } /* while */
	ASHfill(TANGO__data->window,oldfill);
      }
}


void
add_erase(itype,im,thick,x0,y0,x1,y1,fontid,text,sides,xs,ys,arrow,ax,ay)
   ITYPE itype;
   TANGO_IMAGE im;
   ASH_LINE_STYLE thick;
   int x0,y0,x1,y1,fontid;
   char *text;
   int sides,xs[],ys[], arrow,ax[4],ay[4];
{
   ERASER_PTR ep;
   int i;

   if (TANGO__data->numimages > TANGO__data->imageclear) return;

   ep = (ERASER_PTR) malloc( sizeof( struct ERASER));
   ep->itype = itype;
   ep->image = im;
   ep->thick = thick;
   ep->x0 = x0;
   ep->y0 = y0;
   ep->x1 = x1;
   ep->y1 = y1;
   if ((itype == POLYGON) || (itype == FILLPOLYGON) ||
       (itype == POLYLINE) || (itype == SPLINE))
      { ep->sides = sides;
	for (i=0; i<sides; ++i)
	   { ep->xs[i] = xs[i];
	     ep->ys[i] = ys[i];
	   }
      }
   if ((itype == LINE) || (itype == POLYLINE))
      { ep->arrow = arrow;
	for (i=0; i<4; ++i)
	   { ep->arrowx[i] = ax[i];
	     ep->arrowy[i] = ay[i];
	   }
      }
   if (itype == TEXT)
      { ep->fontid = fontid;
	strcpy(ep->text,text);
      }
   ep->next = TANGO__data->erase[TANGO__data->framenum];
   TANGO__data->erase[TANGO__data->framenum] = ep;
}


