/********************************************************************************/
/*										*/
/*		eventtrace.c							*/
/*										*/
/*	System to capture, record and playback events				*/
/*										*/
/********************************************************************************/


#include "/pro/field/include/datatypes.h"
#include "/pro/field/include/msg.h"




/********************************************************************************/
/*										*/
/*	This command will either record (-r file) or playback (-p file) a	*/
/*	series of events from a tango demonstration.  It listens for a		*/
/*	the IE messages and records them.  It also listens for DEBUG START	*/
/*	and DEBUG STOP messages to know when to start and finish recording.	*/
/*	If the -n # option is given, then file.1, ... file.# will be		*/
/*	created for subsequent program runs.					*/
/*										*/
/********************************************************************************/



/********************************************************************************/
/*										*/
/*	Parameters								*/
/*										*/
/********************************************************************************/


#define START_MSG	"DEBUG START %s"
#define STOP_MSG	"DEBUG STOP %s %s"
#define EXIT_MSG	"DEBUG FINISH %s %s"

#define EVENT_MSG	"IE %1r"




/********************************************************************************/
/*										*/
/*	Local variables 							*/
/*										*/
/********************************************************************************/


static	FILE *		out_file;
static	Integer 	out_max;
static	Integer 	out_count;
static	String		out_base;

static	FILE *		in_file;






/********************************************************************************/
/*										*/
/*	Forward definitions							*/
/*										*/
/********************************************************************************/


static	void		record();
static	void		start_handler();
static	void		stop_handler();
static	void		exit_handler();
static	void		event_handler();
static	void		play();





/********************************************************************************/
/*										*/
/*	main -- main program							*/
/*										*/
/********************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   String infile;
   String outfile;
   Integer count;
   Integer i;
   Boolean err;

   infile = NULL;
   outfile = NULL;
   count = -1;
   err = FALSE;

   for (i = 1; i < argc; ++i) {
      if (argv[i][0] == '-') {
	 switch (argv[i][1]) {
	    case 'r' :
	       if (outfile == NULL) outfile = argv[++i];
	       else err = TRUE;
	       break;
	    case 'p' :
	       if (infile == NULL) infile = argv[++i];
	       else err = TRUE;
	       break;
	    case 'n' :
	       if (count >= 0) err = TRUE;
	       if (isdigit(argv[i+1][0])) count = atol(argv[++i]);
	       else count = 0;
	       break;
	    default :
	       err = TRUE;
	       break;
	  };
       }
      else err = TRUE;
   };

   if (outfile == NULL && infile == NULL) err = TRUE;
   if (count >= 0 && outfile == NULL) err = TRUE;
   if (infile != NULL && outfile != NULL) err = TRUE;

   if (err) {
      fprintf(stderr,"eventtrace -r output_file -n [count]\n");
      fprintf(stderr,"eventtrace -p input_file\n");
      exit(1);
    };

   MSGinit();

   if (outfile != NULL) {
      record(outfile,count);
    }
   else if (infile != NULL) {
      play(infile);
    };

   exit(0);
};






/********************************************************************************/
/*										*/
/*	record -- record messages						*/
/*										*/
/********************************************************************************/


static void
record(file,ct)
   String file;
   Integer ct;
{
   out_file = NULL;
   out_max = ct;
   out_count = 0;
   out_base = file;

   MSGregister(START_MSG,start_handler,0,NULL);
   MSGregister(STOP_MSG,stop_handler,0,NULL);
   MSGregister(EXIT_MSG,exit_handler,0,NULL);
   MSGregister(EVENT_MSG,event_handler,1,NULL);

   while (out_count >= 0) {
      CMPXselect(0);
    };
};





static void
start_handler(rid)
   Integer rid;
{
   Character buf[1024];

   if (rid >= 0) MSGreply(rid,NULL);

   if (out_file != NULL) {
      fclose(out_file);
      out_file = NULL;
    };

   if (out_max >= 0) {
      sprintf(buf,"%s.%d",out_base,out_count++);
    }
   else sprintf(buf,"%s",out_base);

   out_file = fopen(buf,"w");
   if (out_file == NULL) {
      fprintf(stderr,"Couldn't open file %s for output\n",buf);
      exit(2);
    };
};





static void
stop_handler(rid)
   Integer rid;
{
   if (rid >= 0) MSGreply(rid,NULL);

   if (out_file == NULL) return;

   fclose(out_file);
   out_file = NULL;

   if (out_max < 0) out_count = -1;
   else if (out_max > 0 && out_count >= out_max) out_count = -1;
};





static void
exit_handler(rid)
   Integer rid;
{
   if (rid >= 0) MSGreply(rid,NULL);

   if (out_file != NULL) {
      fclose(out_file);
      out_file = NULL;
    };

   out_count = -1;
};





static void
event_handler(rid,msg)
   Integer rid;
   String msg;
{
   Character fg;

   if (rid >= 0) MSGreply(rid,NULL);

   if (out_file == NULL) return;

   fg = (rid >= 0 ? '+' : ' ');

   fprintf(out_file,"%d %s\n",fg,msg);
};





/********************************************************************************/
/*										*/
/*	play -- playback an event file						*/
/*										*/
/********************************************************************************/


static void
play(file)
   String file;
{
   FILE * inf;
   Character buf[10240];
   Integer i;
   Boolean fg;

   inf = fopen(file,"r");
   if (inf == NULL) {
      fprintf(stderr,"Couldn't open %s for input\n",file);
      exit(3);
    };

   while (fgets(buf,10240,inf) != NULL) {
      i = strlen(buf);
      if (buf[i-1] == '\n') buf[i-1] = 0;
      fg = (buf[0] != ' ');
      if (fg) MSGcalla("IE %s",&buf[1]);
      else MSGsenda("IE %s",&buf[1]);
    };

   fclose(inf);
};





/* end of eventtrace.c */
