/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrspsyn.y						*/
/*									*/
/*	Syntax definitions for pascal xref scanning			*/
/*									*/
/************************************************************************/


#include "xrsc_local.h"


/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


#define MAX_ARGS	128


static	Character	cur_filename[256];
static	Integer 	fct_line;
static	Integer 	fct_argct;
static	String		fct_args[MAX_ARGS];
static	String		fct_name;
static	Boolean 	fct_params;
static	String		last_name;
static	String *	all_fcts = NULL;
static	Integer 	num_fcts;
static	Integer 	max_fcts;

static	String		type_name;
static	Character	array_buf[128];


typedef struct _FILE_DATA *	FILE_DATA;
typedef struct _FILE_DATA {
   Integer line;
   Character name[256];
   FILE * file;
   FILE_DATA previous;
} FILE_DATA_INFO;

static	FILE_DATA	file_stack;



typedef struct _FCT_DATA *	FCT_DATA;
typedef struct _FCT_DATA {
   Integer line;
   Integer argct;
   String args[MAX_ARGS];
   String name;
   Boolean params;
   FCT_DATA previous;
} FCT_DATA_INFO;

static	FCT_DATA	fct_stack;






/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/

static	String		makeid();
static	void		push_function();
static	void		pop_function();
static	void		add_function();
static	void		add_param();



%}



%union {
   String	string;
   Integer	intval;
}


%token		LX_ARRAY LX_BEGIN LX_CASE
%token		LX_CONST LX_DO
%token		LX_DOWNTO LX_ELSE LX_END LX_FILE
%token		LX_FOR LX_FORWARD LX_FUNCTION LX_GOTO
%token		LX_IF
%token		LX_LABEL
%token		LX_OF LX_PACKED LX_NIL
%token		LX_PROCEDURE LX_PROGRAM LX_RECORD LX_REPEAT
%token		LX_SET LX_THEN LX_TO
%token		LX_TYPE LX_UNTIL LX_VAR LX_WHILE
%token		LX_WITH LX_OCT LX_HEX
%token		LX_OTHERWISE LX_EXTERNAL

%token		LX_OTHER

%token		LX_DOTDOT

%token	<string>	LX_ID LX_FCTID
%token			LX_STRING LX_NUM

%type	<string>	fctid varid


%left		'<' '=' '>' LX_IN
%left		'+' '-' LX_OR '|'
%left		UNARYSIGN
%left		'*' '/' LX_DIV LX_MOD LX_AND '&'
%left		LX_NOT

%%



/************************************************************************/
/*									*/
/*	Top level scanning						*/
/*									*/
/************************************************************************/


start	:
			{ fct_name = SALLOC("PROGRAM");
			  fct_argct = 0;
			  fct_line = yylineno;
			  type_name = NULL;
			  XRSC_decl_fct(cur_filename,fct_line,fct_name,fct_argct,
					   fct_args);
			  XRSC_decl_begin(SCOPE_EXTERN,cur_filename,yylineno);
			}
		program
			{ XRSC_decl_end(cur_filename,yylineno); }
	;



program :	prog_head declblock '.'
	|	decl_seq
	;


prog_head :	LX_PROGRAM LX_ID '(' file_id_list ')' ';'
			{ SFREE($2); }
	|	LX_PROGRAM LX_ID
			{ SFREE($2); }
	|	LX_PROGRAM error
	;


file_id_list :	LX_ID
			{ SFREE($1); }
	|	file_id_list ',' LX_ID
			{ SFREE($3); }
	;


declblock :
			{ XRSC_decl_begin(SCOPE_INTERN,cur_filename,yylineno); }
		    decl_seq
			{ XRSC_decl_statements(); }
		    block
			{ XRSC_decl_end(cur_filename,yylineno); }
	;


decl_seq :	/* empty */
	|	decl_seq decl
	|	decl_seq error
	;


decl	:	label_decls
	|	const_decls
	|	type_decls
	|	var_decls
	|	proc_decls
	;


label_decls :	LX_LABEL label_ids
	;


label_ids :	LX_NUM
	|	label_ids ',' LX_NUM
	;



const_decls :	LX_CONST const_decl
	|	const_decls const_decl
	|	LX_CONST error
	;


const_decl :	LX_ID '='
			{ XRSC_decl_set_type("const");
			  XRSC_decl_set_id($1);
			  XRSC_decl_set_sclass(SCLASS_CONST);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,yylineno,fct_name);
			  SFREE($1);
			}
		    expr ';'
	;


type_decls :	LX_TYPE type_decl
	|	type_decls type_decl
	|	LX_TYPE error
	;


type_decl :	LX_ID
			{ type_name = $1; }
		    '=' type ';'
			{ XRSC_decl_set_id($1);
			  XRSC_decl_set_sclass(SCLASS_TYPEDEF);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,yylineno,fct_name);
			  SFREE($1);
			  type_name = NULL;
			}
	;


var_decls :	LX_VAR var_decl ';'
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	|	var_decls var_decl ';'
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	|	LX_VAR error
			{ XRSC_decl_finish(FINISH_BAD,cur_filename,yylineno,fct_name); }
	;


var_decl :	varid
			{ $<intval>$ = yylineno; }
		    ':' type
			{ XRSC_decl_set_id($1);
			  add_param($1);
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,$<intval>2,fct_name);
			  SFREE($1);
			}
	|	varid
			{ $<intval>$ = yylineno; }
		    ',' var_decl
			{ XRSC_decl_set_id($1);
			  add_param($1);
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,$<intval>2,fct_name);
			  SFREE($1);
			}
	;


varid	:	LX_ID
	|	LX_FCTID
	;



proc_decls :	proc_decls1
			{ XRSC_decl_end(cur_filename,yylineno);
			  pop_function();
			}
	;


proc_decls1 :	phead LX_FORWARD ';'
			{ XRSC_decl_set_id(fct_name);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,fct_line,
					      fct_stack->name);
			}
	|	phead LX_EXTERNAL ';'
			{ XRSC_decl_set_id(fct_name);
			  XRSC_decl_set_sclass(SCLASS_EXTERN);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,fct_line,
					      fct_stack->name);
			}
	|	phead LX_EXTERNAL LX_ID ';'
			{ XRSC_decl_set_id(fct_name);
			  XRSC_decl_set_sclass(SCLASS_EXTERN);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,fct_line,
					      fct_stack->name);
			  SFREE($3);
			}
	|	phead
			{ XRSC_decl_set_id(fct_name);
			  XRSC_decl_set_fclass();
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,fct_line,
					      fct_stack->name);
			  XRSC_decl_fct(cur_filename,fct_line,fct_name,fct_argct,
					   fct_args);
			}
		    declblock ';'
	;


phead	:	LX_PROCEDURE fctid
			{ push_function();
			  fct_name = $2;
			  fct_argct = 0;
			  fct_line = yylineno;
			}
		    params ftype ';'
	|	LX_FUNCTION fctid
			{ push_function();
			  fct_name = $2;
			  fct_argct = 0;
			  fct_line = yylineno;
			  fct_params = FALSE;
			}
		    params ftype ';'
	;


fctid	:	LX_ID
			{ add_function($1);
			  $$ = $1;
			}
	|	LX_FCTID
	;


params	:
			{ XRSC_decl_begin(SCOPE_ARGS,cur_filename,yylineno);
			  fct_params = TRUE;
			}
		    params1
			{ fct_params = FALSE; }
	;

params1 :	/* empty */
	|	'(' param_list ')'
	;


param_list :	param
	|	param_list ';' param
	;


param	:	var_decl
	|	LX_VAR var_decl
	|	LX_FUNCTION param3
	|	LX_PROCEDURE param3
	;


param3	:	fctid params ftype
			{ XRSC_decl_set_id($1);
			  add_param($1);
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_name);
			  SFREE($1);
			}
	|	fctid ',' param3
			{ XRSC_decl_set_id($1);
			  add_param($1);
			  XRSC_decl_finish(FINISH_COMMA,cur_filename,yylineno,fct_name);
			  SFREE($1);
			}
	;


ftype	:	/* empty */
			{ XRSC_decl_set_type("void ()"); }
	|	':' type
			{ XRSC_decl_add_type(" ()"); }
	;


const	:	LX_STRING
	|	number
	|	'+' number
	|	'-' number
	;

number	:	LX_ID
			{ SFREE($1); }
	|	LX_NUM
	;


const_list :	const
	|	const_list ',' const
	;


type	:	simple_type
	|	'^' LX_ID
			{ XRSC_decl_set_type($2);
			  XRSC_decl_add_type("*");
			  SFREE($2);
			}
	|	struct_type
	|	LX_PACKED struct_type
	;


simple_type :	LX_ID
			{ XRSC_decl_set_type($1);
			  SFREE($1);
			}
	|	'(' enum_id_list ')'
			{ XRSC_decl_set_type("enum"); }
	|	const LX_DOTDOT const
			{ XRSC_decl_set_type("range"); }
	;


enum_id_list :	LX_ID
			{ XRSC_decl_enum(cur_filename,yylineno,$1);
			  SFREE($1);
			}
	|	enum_id_list ',' LX_ID
			{ XRSC_decl_enum(cur_filename,yylineno,$3);
			  SFREE($3);
			}
       ;


struct_type :	LX_ARRAY '['
			{ XRSC_decl_begin(SCOPE_NONE,cur_filename,yylineno); }
		    simple_type_list ']'
			{ XRSC_decl_end(cur_filename,yylineno); }
		    LX_OF type
			{ XRSC_decl_add_type(array_buf); }
	|	LX_FILE LX_OF type
			{ XRSC_decl_add_type(" <file>"); }
	|	LX_SET LX_OF type
			{ XRSC_decl_add_type("{}"); }
	|	LX_RECORD field_list LX_END
			{ XRSC_decl_set_type("record"); }
	;


simple_type_list : simple_type
			{ strcpy(array_buf,"[]"); }
	|	simple_type_list ',' simple_type
			{ strcat(array_buf,"[]"); }
	;


field_list :
			{ XRSC_decl_begin(SCOPE_SUE,cur_filename,yylineno); }
		    fixed_part variant_part
			{ XRSC_decl_end(cur_filename,yylineno); }
	;


fixed_part :	field
	|	fixed_part ';' field
	|	fixed_part error
	;


variant_part :	/* empty */
	|	LX_CASE LX_ID LX_OF variant_list
			{ SFREE($2); }
	|	LX_CASE LX_ID ':' LX_ID
			{ XRSC_decl_set_type($4);
			  XRSC_decl_set_id($2);
			  XRSC_decl_finish(FINISH_SEMI,cur_filename,yylineno,fct_name);
			  SFREE($2);
			  SFREE($4);
			}
		    LX_OF variant_list
	;


variant_list :	variant
	|	variant_list ';' variant
	|	variant_list error
	;


variant :	/* empty */
	|	const_list ':' '(' field_list ')'
	;


field	:	/* empty */
	|	var_decl
	;


block	:	LX_BEGIN stmt_list LX_END
	;


stmt_list :	stmt
	|	stmt_list ';' stmt
	;


cstmt_list :	cstmt
	|	cstmt_list ';' cstmt
	;


cstmt	:	const_list ':' stmt
	|	LX_OTHERWISE stmt
	|	/* empty */
	;


stmt	:	/* empty */
	|	LX_NUM ':' stmt
	|	LX_ID
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
	|	LX_FCTID
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
	|	LX_ID '('
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
		    wexpr_list ')'
	|	LX_FCTID '('
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
		    wexpr_list ')'
	|	assign
	|	LX_BEGIN stmt_list LX_END
	|	LX_CASE expr LX_OF cstmt_list LX_END
	|	LX_WITH var_list LX_DO stmt
	|	LX_WHILE expr LX_DO stmt
	|	LX_REPEAT stmt_list LX_UNTIL expr
	|	LX_FOR assign LX_TO expr LX_DO stmt
	|	LX_FOR assign LX_DOWNTO expr LX_DO stmt
	|	LX_GOTO LX_NUM
	|	LX_IF expr LX_THEN stmt
	|	LX_IF expr LX_THEN stmt LX_ELSE stmt
	|	error
	;


assign	:	variable ':' '='
			{ XRSC_expr_assignop(); }
		    expr
	;


expr	:	expr1
			{ XRSC_expr_op(); }
	;

expr1	:	error
	|	expr relop expr 	%prec '<'
	|	'+' expr                %prec UNARYSIGN
	|	'-' expr                %prec UNARYSIGN
	|	expr addop expr 	%prec '+'
	|	expr divop expr 	%prec '*'
	|	LX_NIL
	|	LX_STRING
	|	LX_NUM
	|	variable
	|	LX_ID '('
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
		    wexpr_list ')'
	|	LX_FCTID '('
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
		    wexpr_list ')'
	|	LX_FCTID
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  XRSC_expr_call(fct_name);
			  SFREE($1);
			}
	|	'(' expr ')'
	|	negop expr		%prec LX_NOT
	|	'[' element_list ']'
	|	'[' ']'
	;


element_list :	element
	|	element_list ',' element
	;


element :	expr
	|	expr LX_DOTDOT expr
	;


variable :	LX_ID
			{ XRSC_expr_id(cur_filename,yylineno,$1);
			  SFREE($1);
			}
	|	variable '['
			{ XRSC_expr_push(); }
		    expr_list ']'
			{ XRSC_expr_pop(); }
	|	variable '.' LX_ID
			{ XRSC_expr_id(cur_filename,yylineno,$3);
			  SFREE($3);
			}
	|	variable '.' LX_FCTID
			{ XRSC_expr_id(cur_filename,yylineno,$3);
			  SFREE($3);
			}
	|	variable '^'
	;


wexpr	:	expr
	|	expr ':' expr
	|	expr ':' expr ':' expr
	|	expr octhex
	|	expr ':' expr octhex
	;


octhex	:	LX_OCT
	|	LX_HEX
	;


expr_list :	expr
	|	expr_list ',' expr
	;


wexpr_list :	wexpr
	|	wexpr_list ',' wexpr
	;


relop	:	'=' | '<' | '>' | '<' '>' | '<' '=' | '>' '=' | LX_IN
	;


addop	:	'+' | '-' | LX_OR | '|'
	;


divop	:	'*' | '/' | LX_DIV | LX_MOD | LX_AND | '&'
	;


negop	:	LX_NOT | '~'
	;


var_list :	variable
	|	var_list ',' variable
	;


%%


#include "xrsplex.c"





/************************************************************************/
/*									*/
/*	XRSC_parse -- parse a file					*/
/*									*/
/************************************************************************/


void
XRSC_parse(file,name,inf)
   String file;
   String name;
   FILE * inf;
{
   file_stack = NULL;
   fct_stack = NULL;
   if (all_fcts == NULL) {
      max_fcts = 128;
      all_fcts = (String *) calloc(sizeof(String),max_fcts);
    };
   num_fcts = 0;

   if (file == NULL) {
      yyin = inf;
      file = name;
    }
   else yyin = fopen(file,"r");

   if (yyin != NULL) {
      XRSC_find_file_line(file);
      XRSC_set_file(file,1);
      yyparse();
    }
   else {
      if (XRSC__interact) fprintf(stderr,"Can't open file \"%s\"\n",file);
    };
};




/************************************************************************/
/*									*/
/*	yyerror -- handle errors					*/
/*									*/
/************************************************************************/


yyerror()
{
   if (XRSC__interact)
      fprintf(stderr,"syntax error at %d in %s\n",yylineno,cur_filename);
};



/************************************************************************/
/*									*/
/*	XRSC_set_file -- set file and line				*/
/*									*/
/************************************************************************/


static void
XRSC_set_file(file,line)
   String file;
   Integer line;
{
   strcpy(cur_filename,file);
   yylineno = line;
};





/************************************************************************/
/*									*/
/*	XRSC_push_file -- push a new file onto the stack		*/
/*									*/
/************************************************************************/


static void
XRSC_push_file(file,glbl)
   String file;
   Boolean glbl;
{
   FILE * newf;
   Character nfil[256];
   FILE_DATA fd;

   if (XRSC_find_file(file,glbl,cur_filename,nfil)) return;

   XRSC_out_header(nfil);

   newf = fopen(nfil,"r");
   if (newf == NULL) {
      if (XRSC__interact) fprintf(stderr,"Can't open include file %s\n",nfil);
      return;
    };

   fd = PALLOC(FILE_DATA_INFO);
   strcpy(fd->name,cur_filename);
   fd->line = yylineno;
   fd->file = yyin;
   fd->previous = file_stack;
   file_stack = fd;

   strcpy(cur_filename,nfil);
   yylineno = 1;
   yyin = newf;
};





yywrap()
{
   FILE_DATA fd;

   if (file_stack == NULL) return 1;

   fclose(yyin);

   fd = file_stack;
   file_stack = fd->previous;
   yyin = fd->file;
   yylineno = fd->line;
   strcpy(cur_filename,fd->name);

   free(fd);

   return 0;
};





/************************************************************************/
/*									*/
/*	makeid -- make a new id from two old ones			*/
/*									*/
/************************************************************************/


static String
makeid(s1,s2)
   String s1;
   String s2;
{
   Character buf[256];

   sprintf(buf,"%s %s",s1,s2);

   return SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	push_function -- handle nested function definitions		*/
/*									*/
/************************************************************************/


static void
push_function()
{
   FCT_DATA fd;
   Integer i;

   fd = PALLOC(FCT_DATA_INFO);

   fd->line = fct_line;
   fd->argct = fct_argct;
   for (i = 0; i < fd->argct; ++i) fd->args[i] = fct_args[i];
   fd->name = fct_name;
   fd->params = fct_params;
   fd->previous = fct_stack;
   fct_stack = fd;
};





static void
pop_function()
{
   FCT_DATA fd;
   Integer i;

   fd = fct_stack;
   if (fd == NULL) {
      fct_line = 0;
      fct_argct = 0;
      fct_name = NULL;
      fct_params = FALSE;
    }
   else {
      fct_stack = fd->previous;
      fct_line = fd->line;
      fct_argct = fd->argct;
      fct_params = fd->params;
      for (i = 0; i < fd->argct; ++i) fct_args[i] = fd->args[i];
      fct_name = fd->name;
      free(fd);
    };
};





/************************************************************************/
/*									*/
/*	add_function -- add function to list of functions		*/
/*									*/
/************************************************************************/


static void
add_function(nm)
   String nm;
{
   Integer i;

   if (num_fcts >= max_fcts) {
      max_fcts *= 2;
      all_fcts = (String *) realloc(all_fcts,sizeof(String)*max_fcts);
    };

   all_fcts[num_fcts++] = SALLOC(nm);
};





/************************************************************************/
/*									*/
/*	add_param -- add parameter to current function			*/
/*									*/
/************************************************************************/


static void
add_param(nm)
   String nm;
{
   if (!fct_params) return;

   if (fct_argct > MAX_ARGS) return;

   fct_args[fct_argct++] = SALLOC(nm);
};






/* end of xrspsyn.y */


