/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrsplex.l						*/
/*									*/
/*	Lexical definitions for PASCAL cross reference scanner		*/
/*									*/
/************************************************************************/




#undef YYLMAX
#define YYLMAX		20480


static	Integer 	scan_id();
static	Integer 	handle_num();
static	Integer 	handle_string();
static	void		set_file_line();
static	void		include_file();


static	String		cur_macro = NULL;


typedef struct _KEY_ID {
   String name;
   Integer id;
} KEY_ID;

static	KEY_ID	keywords[] = {
   { "and",          LX_AND     },
   { "array",        LX_ARRAY   },
   { "begin",        LX_BEGIN   },
   { "case",         LX_CASE    },
   { "const",        LX_CONST   },
   { "div",          LX_DIV     },
   { "do",           LX_DO      },
   { "downto",       LX_DOWNTO  },
   { "else",         LX_ELSE    },
   { "end",          LX_END     },
   { "external",     LX_EXTERNAL },
   { "file",         LX_FILE    },
   { "for",          LX_FOR     },
   { "forward",      LX_FORWARD  },
   { "function",     LX_FUNCTION },
   { "goto",         LX_GOTO    },
   { "hex",          LX_HEX     },
   { "if",           LX_IF      },
   { "in",           LX_IN      },
   { "label",        LX_LABEL   },
   { "mod",          LX_MOD     },
   { "nil",          LX_NIL     },
   { "not",          LX_NOT     },
   { "oct",          LX_OCT     },
   { "of",           LX_OF      },
   { "or",           LX_OR      },
   { "otherwise",    LX_OTHERWISE },
   { "packed",       LX_PACKED  },
   { "procedure",    LX_PROCEDURE },
   { "program",      LX_PROGRAM },
   { "record",       LX_RECORD  },
   { "repeat",       LX_REPEAT  },
   { "set",          LX_SET     },
   { "then",         LX_THEN    },
   { "to",           LX_TO      },
   { "type",         LX_TYPE    },
   { "until",        LX_UNTIL   },
   { "var",          LX_VAR     },
   { "while",        LX_WHILE   },
   { "with",         LX_WITH    },
   { 0, 0 }
};



%}

%a 10240
%o 10240
%p 10240

%START PREP PRED PREX PREI PREL IGN CMMT BMMT


WHITE	([\ \t\f])+
WHITE0	([\ \t\f])*

INT	([0-9]+)
OCTINT	([0-7]+[bB])

STRING	(\'([^\n']|(\'\'))*\')

ID	([a-zA-Z_][a-zA-Z_0-9]*)
ASGID	({ID}/({WHITE0}":"{WHITE0}"="))

FILE	([-0-9/.a-zA-Z_$+]+)

FLT0	([0-9]+\.[0-9]+)
EXP	([Ee][-+]?[0-9]+)
FLT1	({INT}{EXP})
FLT2	({FLT0}{EXP})
FLT	({FLT0}|{FLT1}|{FLT2})

CMMT	("(*"((("*"+[^*)])|[^*])*)("*"+")"))
BCMMT	("{"([^}]*)"}")

SINGLE	([-<=>+|*/&.();,:^\[\]~])




%%

^"#"{WHITE}/[1-9]               { BEGIN PREL; }
^"#"{WHITE0}                    { BEGIN PREX; }
<PREX>"define"{WHITE}           { BEGIN PRED; }
<PREX>"undef"{WHITE}            { BEGIN PRED; }
<PREX>"include"{WHITE}          { BEGIN PREI; }
<PREX>"line"{WHITE}             { BEGIN PREL; }
<PREX>{ID}			{ BEGIN PREP; }
<PREX>{ASGID}			{ BEGIN PREP; }
<PREX>. 			;
<PRED>{ID}			{ XRSC_out_decl(cur_filename,yylineno,
						   yytext,"",0,SCLASS_MACRO,NULL);
				  cur_macro = SALLOC(yytext);
				  BEGIN PREP;
				}
<PRED>{ASGID}			{ XRSC_out_decl(cur_filename,yylineno,
						   yytext,"",0,SCLASS_MACRO,NULL);
				  cur_macro = SALLOC(yytext);
				  BEGIN PREP;
				}
<PREP>{ID}			{ XRSC_out_reference(yytext,cur_filename,
							yylineno,cur_macro);
				}
<PREP>{ASGID}			{ XRSC_out_reference(yytext,cur_filename,
							yylineno,cur_macro);
				}
<PREP>{STRING}			{ handle_string(TRUE); }
<PREP>{INT}			{ handle_num(TRUE); }
<PREP>{OCTINT}			{ handle_num(TRUE); }
<PREP>{FLT}			{ handle_num(TRUE); }
<PREP>{WHITE}			;
<PREP>".."                      ;
<PREP,PREX,PRED,PREL,PREI>"\\\n" ;
<PREP,PREX,PRED,PREL,PREI>"\n"  { if (cur_macro != NULL) {
				      SFREE(cur_macro);
				      cur_macro = NULL;
				    };
				   BEGIN 0;
				}
<PREP>. 			;
<PREL>{INT}{WHITE}\"{FILE}\"    { BEGIN PREP;
				  set_file_line(yytext);
				}
<PREL>. 			{ BEGIN PREP; }
<PREI>\<{FILE}\>.*\n		{ include_file(yytext); BEGIN 0; }
<PREI>\"{FILE}\".*\n            { include_file(yytext); BEGIN 0; }
<PREI>. 			{ BEGIN PREP; }

<IGN>^"#"{WHITE0}"line"{WHITE}/.*[\n]  { BEGIN PREL; }
<IGN>^"#"{WHITE}/[1-9].*[\n]    { BEGIN PREL; }
<IGN>.*[\n]			;
<IGN>.				;

{SINGLE}			{ return yytext[0]; }
".."                            { return LX_DOTDOT; }

{ID}				{ return scan_id(FALSE); }
{ASGID} 			{ return scan_id(TRUE); }
{STRING}			{ return handle_string(FALSE); }
{INT}				{ return handle_num(FALSE); }
{OCTINT}			{ return handle_num(FALSE); }
{FLT}				{ return handle_num(FALSE); }

[\n]				;
{WHITE} 			;
{CMMT}				;
{BCMMT} 			;
.				{ return LX_OTHER; }





%%

static Integer
scan_id(asg)
   Boolean asg;
{
   Integer i;
   Character buf[128];
   String s,t;

   t = buf;
   for (s = yytext; *s != 0; ++s) {
      if (isupper(*s)) *t++ = tolower(*s);
      else *t++ = *s;
    };
   *t = 0;

   for (i = 0; keywords[i].name != 0; ++i) {
      if (STREQL(buf,keywords[i].name)) return keywords[i].id;
    };

   yylval.string = SALLOC(yytext);

   XRSC_out_reference(yytext,cur_filename,yylineno,fct_name);

   if (!asg) {
      for (i = 0; i < num_fcts; ++i) {
	 if (STREQL(yytext,all_fcts[i])) return LX_FCTID;
       };
    };

   return LX_ID;
};




static int
handle_num(mac)
   Boolean mac;
{
   if (USE_NUMBERS && STRNEQ(yytext,"0") && STRNEQ(yytext,"1")) {
      XRSC_out_reference(yytext,cur_filename,yylineno,(mac ? cur_macro : fct_name));
    };

   return LX_NUM;
};





static int
handle_string(mac)
   Boolean mac;
{
   if (USE_STRINGS) {
      XRSC_out_reference(yytext,cur_filename,yylineno,(mac ? cur_macro : fct_name));
    };

   return LX_STRING;
};




static void
set_file_line(buf)
   String buf;
{
   Integer line;
   String file,s;

   line = atol(buf);
   file = index(buf,'"');
   if (file != NULL && line != 0) {
      ++file;
      s = index(file,'"');
      if (s != NULL) {
	 *s = 0;
	 XRSC_set_file(file,line-1);
       };
    };
};





static void
include_file(name)
   String name;
{
   String s;
   Boolean gbl;

   gbl = (name[0] == '<');
   ++name;
   if (gbl) s = index(name,'>');
   else s = index(name,'"');

   *s = 0;

   XRSC_push_file(name,gbl);
};





/* end of xrsplex.l */
