/************************************************************************/
/*									*/
/*		xrscout.c						*/
/*									*/
/*	Output routines for c cross-reference program			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xrsc_local.h"





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


#define WD_SIZE 	1024

static	Character	curwd[WD_SIZE];

	Boolean 	XRSC__ignore;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		real_file_name();
static	String		real_fct_name();
static	String		check_ignore();





/************************************************************************/
/*									*/
/*	XRSC_out_init -- module initialization				*/
/*									*/
/************************************************************************/


void
XRSC_out_init()
{
   XRSC__ignore = FALSE;

   getwd(curwd);
};





/************************************************************************/
/*									*/
/*	XRSC_out_header -- output starting information			*/
/*									*/
/************************************************************************/


void
XRSC_out_header(file)
   String file;
{
   file = real_file_name(file);

   printf(FILE_FORMAT,file,curwd,XRSC__level);

   if (XRSC__level == 0 && XRSC__scan_all) XRSC__level = 2;
};





/************************************************************************/
/*									*/
/*	XRSC_out_reference -- output a reference			*/
/*									*/
/************************************************************************/


void
XRSC_out_reference(item,file,line,fct)
   String item;
   String file;
   Integer line;
   String fct;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(REF_FORMAT,file,line,real_fct_name(fct),item);
};





/************************************************************************/
/*									*/
/*	XRSC_out_scope -- output a scope				*/
/*									*/
/************************************************************************/


void
XRSC_out_scope(id,typ,file,start,end,inid)
   Integer id;
   XRSC_SCOPE_TYPE typ;
   String file;
   Integer start;
   Integer end;
   Integer inid;
{
   String t;

   file = check_ignore(file);
   if (XRSC__ignore) return;

   switch (typ) {
      case SCOPE_EXTERN :
	 t = "EXTERN";
	 break;
      case SCOPE_ARGS :
	 t = "ARGDEF";
	 break;
      case SCOPE_INTERN :
	 t = "INTERN";
	 break;
      case SCOPE_SUE :
	 t = "STRUCT";
	 break;
    };

   printf(SCOPE_FORMAT,file,start,end,id,inid,t);
};





/************************************************************************/
/*									*/
/*	XRSC_out_decl -- output declaration information 		*/
/*									*/
/************************************************************************/


void
XRSC_out_decl(file,line,name,type,scope,class,fct)
   String file;
   Integer line;
   String name;
   String type;
   Integer scope;
   XRSC_SCLASS class;
   String fct;
{
   String c;

   file = check_ignore(file);
   if (XRSC__ignore) return;

   switch (class) {
      case SCLASS_STATIC :
	 c = "STATIC";
	 break;
      case SCLASS_EXTERN :
	 c = "EXTERN";
	 break;
      case SCLASS_AUTO :
	 c = "AUTO";
	 break;
      case SCLASS_REGISTER :
	 c = "REGISTER";
	 break;
      case SCLASS_TYPEDEF :
	 c = "TYPEDEF";
	 break;
      case SCLASS_EXTDEF :
	 c = "EXTDEF";
	 break;
      case SCLASS_PARAM :
	 c = "PARAM";
	 break;
      case SCLASS_FIELD :
	 c = "FIELD";
	 break;
      case SCLASS_SFUNCTION :
	 c = "SFUNCTION";
	 break;
      case SCLASS_EFUNCTION :
	 c = "EFUNCTION";
	 break;
      case SCLASS_STRUCTID :
	 c = "STRUCTID";
	 break;
      case SCLASS_UNIONID :
	 c = "UNIONID";
	 break;
      case SCLASS_ENUMID :
	 c = "ENUMID";
	 break;
      case SCLASS_CONST :
	 c = "CONST";
	 break;
      case SCLASS_MACRO :
	 c = "MACRO";
	 break;
      case SCLASS_FRIEND :
	 c = "FRIEND";
	 break;
      case SCLASS_CLASSID :
	 c = "CLASSID";
	 break;
      default :
	 c = "UNKNOWN";
	 break;
    };

   printf(DECL_FORMAT,file,line,name,scope,c,real_fct_name(fct),type);
};





/************************************************************************/
/*									*/
/*	XRSC_out_proc -- output procedure information			*/
/*									*/
/************************************************************************/


void
XRSC_out_proc(file,line,name,scope,act,args)
   String file;
   Integer line;
   String name;
   Integer act;
   String args;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(PROC_FORMAT,file,line,name,scope,act,args);
};





/************************************************************************/
/*									*/
/*	XRSC_out_assign -- output assignment information		*/
/*									*/
/************************************************************************/


void
XRSC_out_assign(file,line,name)
   String file;
   Integer line;
   String name;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(ASSIGN_FORMAT,file,line,name);
};





/************************************************************************/
/*									*/
/*	XRSC_out_call -- output call information			*/
/*									*/
/************************************************************************/


void
XRSC_out_call(file,line,to,from)
   String file;
   Integer line;
   String to;
   String from;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(CALL_FORMAT,file,line,to,from);
};





/************************************************************************/
/*									*/
/*	XRSC_out_hier -- output hierarchy information			*/
/*									*/
/************************************************************************/


void
XRSC_out_hier(file,line,cls,par,pub,virt,frnd)
   String file;
   Integer line;
   String cls;
   String par;
   Boolean pub;
   Boolean virt;
   Boolean frnd;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(HIER_FORMAT,file,line,cls,par,pub,virt,frnd);
};





/************************************************************************/
/*									*/
/*	XRSC_out_memb -- ouptut class member information		*/
/*									*/
/************************************************************************/


void
XRSC_out_memb(file,line,cls,mem,prot,isdata,isinline,isfrnd,isvirt,isstatic,ispure,iscon)
   String file;
   Integer line;
   String cls;
   String mem;
   String prot;
   Boolean isdata;
   Boolean isinline;
   Boolean isfrnd;
   Boolean isvirt;
   Boolean isstatic;
   Boolean ispure;
   Boolean iscon;
{
   file = check_ignore(file);
   if (XRSC__ignore) return;

   printf(MEMB_FORMAT,file,line,cls,mem,prot,isdata,
	     isinline,isfrnd,isvirt,isstatic,ispure,iscon);
};





/************************************************************************/
/*									*/
/*	XRSC_out_map -- output name mapping information 		*/
/*									*/
/************************************************************************/


void
XRSC_out_map(from,to)
   String from;
   String to;
{
   printf(MAP_FORMAT,from,to);
};





/************************************************************************/
/*									*/
/*	real_file_name -- get full file name				*/
/*									*/
/************************************************************************/


static String
real_file_name(f)
   String f;
{
   static Character buf[1024],last[1024];
   Character wd[1024];
   String s,w;

   if (f[0] == '/') return f;

   if (STREQL(last,f)) return buf;
   strcpy(last,f);

   w = curwd;
   while (*f != 0) {
      if (STREQL(f,".")) ++f;
      else if (strncmp(f,"./",2) == 0) f += 2;
      else if (f[0] == '.' && f[1] == '.' && (f[2] == 0 || f[2] == '/')) {
	 f += 2;
	 if (*f == '/') ++f;
	 if (w == curwd) {
	    strcpy(wd,curwd);
	    w = wd;
	  };
	 s = rindex(w,'/');
	 if (s == NULL) break;
	 *s = 0;
       }
      else break;
    };

   sprintf(buf,"%s/%s",w,f);

   return buf;
};






/************************************************************************/
/*									*/
/*	real_fct_name -- get function name for output			*/
/*									*/
/************************************************************************/


static String
real_fct_name(f)
   String f;
{
   if (f == NULL || f[0] == 0) return "*TOP*";

   return f;
};





/************************************************************************/
/*									*/
/*	check_ignore -- check if we should ignore this entry		*/
/*		     -- based on file name; return real file name	*/
/*									*/
/************************************************************************/


static String
check_ignore(f)
   String f;
{
   static Character last[1024];
   Boolean fg;

   if (f != NULL) {
      f = real_file_name(f);

      if (STRNEQ(f,last)) {
	 strcpy(last,f);
	 fg = XRSC_find_file_line(f);
	 if (!XRSC__scan_all) XRSC__ignore = fg;
       };
    };

   return f;
};







/* end of xrscout.c */
