/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		xrdbsyn.y						*/
/*									*/
/*	Syntax definitions for database query language			*/
/*									*/
/************************************************************************/


#include "xrdb_local.h"
#include "xrdb_defs.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TAB_CHAR	'\177'





/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef Integer 	QOP;
#define QOP_EQL 	1
#define QOP_NEQ 	2
#define QOP_GEQ 	3
#define QOP_GTR 	4
#define QOP_LEQ 	5
#define QOP_LSS 	6
#define QOP_MATCH	7

#define QOP_SINGLE	20
#define QOP_AND 	21
#define QOP_OR		22



typedef struct _QREL {
   String name;
   Integer relation;
   Integer count;
} QREL;



typedef struct _QITEM {
   Integer qrel;
   Integer relation;
   Integer offset;
   Integer value;
   FIELD_TYPE type;
} QITEM;




typedef struct _QCOMP {
   QOP	op;
   Integer lhs;
   Integer rhs;
} QCOMP;



typedef struct _QEXPR {
   QOP op;
   Integer lhs;
   Integer rhs;
} QEXPR;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/

#define MAX_REL 	16
#define MAX_OUTPUT	32
#define MAX_ITEM	64
#define MAX_SELECT	48
#define MAX_EXPR	64


static	Integer 	num_rel;
static	QREL		rels[MAX_REL];

static	Integer 	num_output;
static	Integer 	outputs[MAX_OUTPUT];

static	Integer 	num_item;
static	QITEM		items[MAX_ITEM];

static	Integer 	num_select;
static	QCOMP		selects[MAX_SELECT];

static	Integer 	num_expr;
static	QEXPR		exprs[MAX_EXPR];

static	Boolean 	err_flag;
static	Boolean 	sort_flag;
static	FIELD_TYPE	last_type;
static	QOP		last_op;
static	String		last_re;
static	Boolean 	inquery;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		prompt();
static	void		begin_query();
static	Integer 	create_field();
static	Integer 	find_qrel();
static	Integer 	create_const();
static	Integer 	new_select();
static	Integer 	new_expr();
static	void		process_query();
static	void		do_query();
static	Boolean 	eval_expr();
static	Boolean 	eval_select();
static	void		item_value();
static	void		do_output();
static	void		print_value();
static	void		sortflags();
static	void		list_systems();
static	Boolean 	item_match();
static	String		item_string();



%}



%union {
   String	string;
   Integer	intval;
}


%left			'&' '|'

%token	<string>	LX_ITEM LX_CONST
%token	<intval>	LX_OP

%type	<intval>	field operator expr selector selector_expr
%type	<intval>	selectors




%%

input	:	/* empty */
			{ prompt(); }
	|	input command
			{ prompt(); }
	;


command :	sys_command
	|	query
	|	error ';'
			{ if (inquery) {
			     if (!XRDB__interact) printf("%s\n",END_QUERY_MARKER);
			     inquery = FALSE;
			   };
			}
	;


sys_command :	'#' '@' LX_ITEM ';'
			{ XRDB_file_setup($3,FALSE); }
	|	'#' '+' LX_ITEM ';'
			{ XRDB_file_include($3);
			  SFREE($3);
			}
	|	'#' '+' ';'
			{ XRDB_file_include(NULL); }
	|	'#' '?' ';'
			{ list_systems(); }
	;


query	:
			{ begin_query(); }
		   output_list selectors ';'
			{ process_query($3); }
	;


output_list :	'(' output_fields ')'
	|	'(' '*' output_fields ')'
			{ sort_flag = FALSE; }
	;


output_fields : field
			{ outputs[num_output++] = $1; }
	|	output_fields ',' field
			{ outputs[num_output++] = $3; }
	;


selectors :	/* empty */
			{ $$ = -1; }
	|	selector_expr
	;


selector_expr : '(' selector_expr ')'
			{ $$ = $2; }
	|	selector
			{ $$ = new_expr(QOP_SINGLE,$1,-1); }
	|	selector_expr '&' selector_expr
			{ $$ = new_expr(QOP_AND,$1,$3); }
	|	selector_expr '|' selector_expr
			{ $$ = new_expr(QOP_OR,$1,$3); }
	;


selector :	field operator expr
			{ $$ = new_select($2,$1,$3); }
	;


field	:	LX_ITEM
			{ $$ = create_field($1);
			  SFREE($1);
			}
	;


expr	:	field
	|	LX_CONST
			{ $$ = create_const($1);
			  SFREE($1);
			}
	;


operator :	LX_OP
			{ last_op = $1;
			  $$ = $1;
			}
	;



%%

#include "xrdblex.c"





/************************************************************************/
/*									*/
/*	XRDB_syn_init -- module initialization				*/
/*									*/
/************************************************************************/


void
XRDB_syn_init()
{
   inquery = FALSE;
};






/************************************************************************/
/*									*/
/*	yyerror -- handle errors					*/
/*	yywrap -- lex routine						*/
/*									*/
/************************************************************************/


yyerror()
{
   fprintf(stderr,"XRDB+ syntax error\n");
};




yywrap()
{
   return 1;
};




/************************************************************************/
/*									*/
/*	prompt -- output prompt 					*/
/*									*/
/************************************************************************/


static void
prompt()
{
   if (XRDB__interact) printf("XRDB: ");
   fflush(stdout);
};





/************************************************************************/
/*									*/
/*	begin_query -- set up to accumulate a query			*/
/*									*/
/************************************************************************/


static void
begin_query()
{
   err_flag = FALSE;
   sort_flag = TRUE;
   last_type = FIELD_TYPE_NONE;
   last_op = 0;
   last_re = NULL;

   num_rel = 0;
   num_output = 0;
   num_item = 0;
   num_select = 0;
   num_expr = 0;

   inquery = TRUE;
};





/************************************************************************/
/*									*/
/*	create_field -- generate a field item for variable string	*/
/*									*/
/************************************************************************/


static Integer
create_field(buf)
   String buf;
{
   String s;
   Integer q,it;
   FIELD fp;

   s = index(buf,'.');
   if (s == NULL) {
      if (XRDB__interact) fprintf(stderr,"XRDB+ Bad field descriptor %s\n",buf);
      err_flag = TRUE;
      return -1;
    };

   *s++ = 0;

   q = find_qrel(buf);
   if (q < 0) return -1;

   for (fp = relations[rels[q].relation].field; fp->name != NULL; ++fp) {
      if (STREQL(fp->name,s)) break;
    }
   if (fp->name == NULL) {
      if (XRDB__interact) fprintf(stderr,"XRDB+ Bad field name %s\n",s);
      err_flag = TRUE;
      return -1;
    };

   it = num_item++;
   items[it].qrel = q;
   items[it].relation = rels[q].relation;
   items[it].offset = fp->offset;
   items[it].value = 0;
   items[it].type = fp->type;

   last_type = fp->type;

   return it;
};






/************************************************************************/
/*									*/
/*	find_qrel -- find variable for relation 			*/
/*									*/
/************************************************************************/


static Integer
find_qrel(name)
   String name;
{
   Integer q;
   Integer r;

   for (q = 0; q < num_rel; ++q) {
      if (STREQL(name,rels[q].name)) return q;
    };

   for (r = 0; relations[r].name != NULL; ++r) {
      if (relations[r].id == name[0]) break;
    };

   if (relations[r].name == NULL) {
      if (XRDB__interact) fprintf(stderr,"XRDB+ Bad variable name %s\n",name);
      err_flag = TRUE;
      return -1;
    };

   q = num_rel++;
   rels[q].name = SALLOC(name);
   rels[q].relation = r;

   return q;
};





/************************************************************************/
/*									*/
/*	create_const -- create constant item from string		*/
/*									*/
/************************************************************************/


static Integer
create_const(buf)
   String buf;
{
   Integer i,v,it;
   FIELD_TYPE typ;

   if (last_op == QOP_MATCH) typ = FIELD_TYPE_STRING;
   else typ = last_type;

   switch (typ) {
      default :
      case FIELD_TYPE_NONE :
      case FIELD_TYPE_SCOPE :
      case FIELD_TYPE_DECL :
	 if (XRDB__interact) fprintf(stderr,"XRDB+ Bad type for constant %s\n",buf);
	 err_flag = TRUE;
	 return -1;

      case FIELD_TYPE_INT :
	 v = atol(buf);
	 break;

      case FIELD_TYPE_STRING :
	 v = (Integer) SALLOC(buf);
	 break;

      case FIELD_TYPE_SCOPE_CLASS :
	 for (i = 0; scope_class_table[i].name != NULL; ++i) {
	    if (STREQL(buf,scope_class_table[i].name)) break;
	  };
	 if (scope_class_table[i].name == 0) v = -1;
	 else v = scope_class_table[i].value;
	 break;

      case FIELD_TYPE_FILE :
	 v = XRDB_db_find_file(buf);
	 break;

      case FIELD_TYPE_NAME :
	 v = XRDB_db_find_name(buf);
	 break;

      case FIELD_TYPE_TYPE :
	 v = XRDB_db_find_type(buf);
	 break;

      case FIELD_TYPE_BOOL :
	 if (buf[0] == 'T' || buf[0] == 't' || buf[0] == '1') v = 1;
	 else v = 0;
	 break;

      case FIELD_TYPE_DECL_CLASS :
	 for (i = 0; decl_class_table[i].name != NULL; ++i) {
	    if (STREQL(buf,decl_class_table[i].name)) break;
	  };
	 if (decl_class_table[i].name == 0) v = -1;
	 else v = decl_class_table[i].value;
	 break;

      case FIELD_TYPE_PROT_CLASS :
	 for (i = 0; prot_class_table[i].name != NULL; ++i) {
	    if (STREQL(buf,prot_class_table[i].name)) break;
	  };
	 if (prot_class_table[i].name == 0) v = -1;
	 else v = prot_class_table[i].value;
	 break;
    };

   if (v < 0) {
      if (XRDB__interact) fprintf(stderr,"XRDB+ Bad constant value %s\n",buf);
      err_flag = TRUE;
      return -1;
    };

   it = num_item++;
   items[it].qrel = -1;
   items[it].relation = -1;
   items[it].offset = 0;
   items[it].value = v;
   items[it].type = typ;

   return it;
};





/************************************************************************/
/*									*/
/*	new_select -- create a new selector				*/
/*									*/
/************************************************************************/


static Integer
new_select(op,lhs,rhs)
   Integer op;
   Integer lhs;
   Integer rhs;
{
   Integer s;

   s = num_select++;
   selects[s].op = op;
   selects[s].lhs = lhs;
   selects[s].rhs = rhs;

   if (op != QOP_MATCH && items[lhs].type != items[rhs].type) {
      if (XRDB__interact) fprintf(stderr,"XRDB+ Mixed types in expression\n");
      err_flag = TRUE;
      return -1;
    };

   return s;
};





/************************************************************************/
/*									*/
/*	new_expr -- create a new expression				*/
/*									*/
/************************************************************************/


static Integer
new_expr(op,lhs,rhs)
   Integer op;
   Integer lhs;
   Integer rhs;
{
   Integer s;

   s = num_expr++;
   exprs[s].op = op;
   exprs[s].lhs = lhs;
   exprs[s].rhs = rhs;

   return s;
};





/************************************************************************/
/*									*/
/*	process_query -- do query processing				*/
/*									*/
/************************************************************************/


static void
process_query(e)
   Integer e;
{
   Character buf[64],cmd[256],fgs[128];
   FILE * fp;
   String s;

   inquery = FALSE;

   if (err_flag || num_rel == 0) {
      if (!XRDB__interact) printf("%s\n",END_QUERY_MARKER);
      return;
    };

   if (sort_flag) {
      sprintf(buf,"/tmp/xrefdbXXXXXX");
      mktemp(buf);
      fp = fopen(buf,"w");
    }
   else fp = stdout;

   do_query(e,num_rel-1,fp);

   if (sort_flag) fclose(fp);

   if (sort_flag) {
      sortflags(fgs);
      sprintf(cmd,"sort -t%c -u %s %s",TAB_CHAR,fgs,buf);

      if (XRDB__interact) {
	 sprintf(fgs,"| tr '\\%o' '\\40'",TAB_CHAR);
	 strcat(cmd,fgs);
       };

      system(cmd);
    };

   if (sort_flag) unlink(buf);

   if (!XRDB__interact) printf("%s\n",END_QUERY_MARKER);
}




static void
do_query(e,i,fp)
   Integer e;
   Integer i;
   FILE * fp;
{
   Integer j;

   for (rels[i].count = 0;
	rels[i].count < *relations[rels[i].relation].count;
	++rels[i].count) {

      for (j = 0; j < num_item; ++j) item_value(j,i);

      if (!eval_expr(e,i)) continue;

      if (i == 0)  do_output(fp);
      else do_query(e,i-1,fp);
    };
};





static Boolean
eval_expr(e,i)
   Integer e;
   Integer i;
{
   Boolean v;

   if (e < 0) return TRUE;

   switch (exprs[e].op) {
      case QOP_SINGLE :
	 v = eval_select(exprs[e].lhs,i);
	 break;
      case QOP_AND :
	 v = eval_expr(exprs[e].lhs,i) && eval_expr(exprs[e].rhs,i);
	 break;
      case QOP_OR :
	 v = eval_expr(exprs[e].lhs,i) || eval_expr(exprs[e].rhs,i);
	 break;
      default :
	 v = FALSE;
	 break;
    };

   return v;
};





static Boolean
eval_select(e,i)
   Integer e;
   Integer i;
{
   Integer v1,v2;
   Boolean v;
   FIELD_TYPE typ;
   QITEM * q;

   q = &items[selects[e].lhs];
   if (q->qrel < i && q->qrel >= 0) return TRUE;
   v1 = q->value;
   q = &items[selects[e].rhs];
   if (q->qrel < i && q->qrel >= 0) return TRUE;
   v2 = q->value;

   switch (selects[e].op) {
      case QOP_EQL :
	 v = (v1 == v2);
	 break;
      case QOP_NEQ :
	 v = (v1 != v2);
	 break;
      case QOP_GEQ :
	 v = (v1 >= v2);
	 break;
      case QOP_GTR :
	 v = (v1 > v2);
	 break;
      case QOP_LEQ :
	 v = (v1 <= v2);
	 break;
      case QOP_LSS :
	 v = (v1 < v2);
	 break;
      case QOP_MATCH :
	 v = item_match(selects[e].lhs,selects[e].rhs);
	 break;
      default :
	 v = FALSE;
	 break;
    };

   return v;
};





static void
item_value(i,q)
   Integer i;
   Integer q;
{
   Integer * vp;
   Integer r,c;

   if (items[i].qrel == q) {
      r = rels[q].relation;
      vp = *relations[r].base;
      c = relations[r].unit_size * rels[q].count + items[i].offset;
      items[i].value = vp[c];
    };
};





static void
do_output(fp)
   FILE * fp;
{
   Integer i;

   for (i = 0; i < num_output; ++i) {
      print_value(fp,outputs[i]);
    }

   fprintf(fp,"\n");
};




static void
print_value(fp,i)
   FILE * fp;
   Integer i;
{
   Integer v;
   String s;
   Integer tab;

   v = items[i].value;
   s = NULL;

   switch (items[i].type) {
      default :
      case FIELD_TYPE_NONE :
	 return;

      case FIELD_TYPE_DECL :
	 break;

      case FIELD_TYPE_SCOPE :
	 break;

      case FIELD_TYPE_INT :
	 break;

      case FIELD_TYPE_STRING :
	 s = (String) v;
	 break;

      case FIELD_TYPE_SCOPE_CLASS :
	 for (i = 0; scope_class_table[i].name != NULL; ++i) {
	    if (v == scope_class_table[i].value) break;
	  };
	 if (scope_class_table[i].name == 0) v = -1;
	 else s = scope_class_table[i].name;
	 break;

      case FIELD_TYPE_FILE :
	 s = XRDB_db_file_string(v);
	 break;

      case FIELD_TYPE_NAME :
	 s = XRDB_db_name_string(v);
	 break;

      case FIELD_TYPE_TYPE :
	 s = XRDB_db_type_string(v);
	 break;

      case FIELD_TYPE_BOOL :
	 s = (v ? "True" : "False");
	 break;

      case FIELD_TYPE_DECL_CLASS :
	 for (i = 0; decl_class_table[i].name != NULL; ++i) {
	    if (v == decl_class_table[i].value) break;
	  };
	 if (decl_class_table[i].name == 0) v = -1;
	 else s = decl_class_table[i].name;
	 break;

      case FIELD_TYPE_PROT_CLASS :
	 for (i = 0; prot_class_table[i].name != NULL; ++i) {
	    if (v == prot_class_table[i].value) break;
	  };
	 if (prot_class_table[i].name == 0) v = -1;
	 else s = prot_class_table[i].name;
	 break;
    };

   if (sort_flag || !XRDB__interact) tab = TAB_CHAR;
   else tab = ' ';

   if (s != NULL) fprintf(fp,"%s%c",s,tab);
   else if (v >= 0) fprintf(fp,"%d%c",v,tab);
};





static void
sortflags(buf)
   String buf;
{
   Integer i,ct;
   Character tmp[16];

   buf[0] = 0;
   ct = MIN(5,num_output);

   for (i = 0; i < ct; ++i) {
      sprintf(tmp," +%d",i);
      switch (items[outputs[i]].type) {
	 case FIELD_TYPE_INT :
	    strcat(tmp,"n");
	    break;
       };
      strcat(buf,tmp);
    };
};





/************************************************************************/
/*									*/
/*	list_systems -- output a list of systems			*/
/*									*/
/************************************************************************/


static void
list_systems()
{
   Sequence l;
   String s;

   forin (s,String,l,XRDB__all_systems) {
      printf("%s\n",s);
    };

   if (!XRDB__interact) printf("%s\n",END_QUERY_MARKER);
};





/************************************************************************/
/*									*/
/*	item_match -- do complex item matching				*/
/*									*/
/************************************************************************/


static Boolean
item_match(i1,i2)
   Integer i1,i2;
{
   String s1,s2,s;
   Integer l1,l2;

   s1 = item_string(i1);
   s2 = item_string(i2);

   if (s1 == NULL || s2 == NULL) {
      if (s1 == NULL && s2 == NULL) return (items[i1].value == items[i2].value);
      return FALSE;
    };

   if (last_re != s2) {
      last_re = s2;
      if (re_comp(s2) != NULL) return FALSE;
    };

   if (re_exec(s1) == 1) return TRUE;

   return FALSE;
};





/************************************************************************/
/*									*/
/*	item_string -- get string for item				*/
/*									*/
/************************************************************************/


static String
item_string(itm)
   Integer itm;
{
   String s;
   Integer i,v;

   v = items[itm].value;

   switch (items[itm].type) {
      default :
      case FIELD_TYPE_NONE :
      case FIELD_TYPE_SCOPE :
      case FIELD_TYPE_DECL :
      case FIELD_TYPE_INT :
	 s = NULL;
	 break;

      case FIELD_TYPE_STRING :
	 s = (String) v;
	 break;

      case FIELD_TYPE_SCOPE_CLASS :
	 for (i = 0; scope_class_table[i].name != NULL; ++i) {
	    if (v == scope_class_table[i].value) break;
	  };
	 s = scope_class_table[i].name;
	 break;

      case FIELD_TYPE_FILE :
	 s = XRDB_db_file_string(v);
	 break;

      case FIELD_TYPE_NAME :
	 s = XRDB_db_name_string(v);
	 break;

      case FIELD_TYPE_TYPE :
	 s = XRDB_db_type_string(v);
	 break;

      case FIELD_TYPE_BOOL :
	 s = (v ? "True" : "False");
	 break;

      case FIELD_TYPE_DECL_CLASS :
	 for (i = 0; decl_class_table[i].name != NULL; ++i) {
	    if (v == decl_class_table[i].value) break;
	  };
	 s = decl_class_table[i].name;
	 break;

      case FIELD_TYPE_PROT_CLASS :
	 for (i = 0; prot_class_table[i].name != NULL; ++i) {
	    if (v == prot_class_table[i].value) break;
	  };
	 s = prot_class_table[i].name;
	 break;
    };

   return s;
};





/* end of xrdbsyn.y */

