/************************************************************************/
/*									*/
/*		xrdbdata.c						*/
/*									*/
/*	Data base routines for xref database				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "xrdb_local.h"
#include <hashsearch.h>




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	XRDB_SYS	cur_system;

	Sequence	XRDB__all_systems;

	XRDB_FILE	XRDB__db_file;
	Integer 	XRDB__db_file_max;
	Integer 	XRDB__db_file_ct;
	XRDB_FILE_ID	XRDB__db_file_id;

	XRDB_SCOPE	XRDB__db_scope;
	Integer 	XRDB__db_scope_max;
	Integer 	XRDB__db_scope_ct;
	XRDB_SCOPE_ID	XRDB__db_scope_id;

	XRDB_NAME	XRDB__db_name;
	Integer 	XRDB__db_name_max;
	Integer 	XRDB__db_name_ct;
	XRDB_NAME_ID	XRDB__db_name_id;

	XRDB_TYPE	XRDB__db_type;
	Integer 	XRDB__db_type_max;
	Integer 	XRDB__db_type_ct;
	XRDB_TYPE_ID	XRDB__db_type_id;

	XRDB_REF	XRDB__db_ref;
	Integer 	XRDB__db_ref_max;
	Integer 	XRDB__db_ref_ct;

	XRDB_DECL	XRDB__db_decl;
	Integer 	XRDB__db_decl_max;
	Integer 	XRDB__db_decl_ct;
	XRDB_DECL_ID	XRDB__db_decl_id;

	XRDB_CALL	XRDB__db_call;
	Integer 	XRDB__db_call_max;
	Integer 	XRDB__db_call_ct;

	XRDB_FCT	XRDB__db_fct;
	Integer 	XRDB__db_fct_max;
	Integer 	XRDB__db_fct_ct;

	XRDB_HIER	XRDB__db_hier;
	Integer 	XRDB__db_hier_max;
	Integer 	XRDB__db_hier_ct;

	XRDB_MEMB	XRDB__db_memb;
	Integer 	XRDB__db_memb_max;
	Integer 	XRDB__db_memb_ct;

	XRDB_MDEF	XRDB__db_mdef;
	Integer 	XRDB__db_mdef_max;
	Integer 	XRDB__db_mdef_ct;

static	Boolean 	name_hash_valid;
static	Boolean 	name_ids_ok;
static	Integer 	match_length;
static	Boolean 	do_elim;
static	XRDB_FILE_ID	cur_file;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


extern	ENTRY * 	hsearch();

static	int		memname_comp();
static	Integer 	start_index();

static	void		free_database();
static	void		fix_name_ids();

static	void		eliminate_duplicates();
static	int		scope_compare();
static	int		fields_compare();

static	void		check_dates();
static	void		mark_valid();
static	void		free_file_info();





/************************************************************************/
/*									*/
/*	XRDB_data_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
XRDB_data_init()
{
   cur_system = PALLOC(XRDB_SYS_INFO);
   cur_system->name = NULL;
   cur_system->directory = NULL;
   name_hash_valid = FALSE;
   name_ids_ok = FALSE;
   do_elim = FALSE;

   XRDB__all_systems = NULL;

   XRDB__db_file_max = 32;
   XRDB__db_file_ct = 0;
   XRDB__db_file = (XRDB_FILE) calloc(XRDB__db_file_max,sizeof(XRDB_FILE_INFO));
   XRDB__db_file_id = 0;

   XRDB__db_scope_max = 64;
   XRDB__db_scope_ct = 0;
   XRDB__db_scope = (XRDB_SCOPE) calloc(XRDB__db_scope_max,sizeof(XRDB_SCOPE_INFO));
   XRDB__db_scope_id = 0;

   XRDB__db_name_max = 128;
   XRDB__db_name_ct = 0;
   XRDB__db_name = (XRDB_NAME) calloc(XRDB__db_name_max,sizeof(XRDB_NAME_INFO));
   XRDB__db_name_id = 0;

   XRDB__db_type_max = 32;
   XRDB__db_type_ct = 0;
   XRDB__db_type = (XRDB_TYPE) calloc(XRDB__db_type_max,sizeof(XRDB_TYPE_INFO));
   XRDB__db_type_id = 0;

   XRDB__db_ref_max = 1024;
   XRDB__db_ref_ct = 0;
   XRDB__db_ref = (XRDB_REF) calloc(XRDB__db_ref_max,sizeof(XRDB_REF_INFO));

   XRDB__db_decl_max = 32;
   XRDB__db_decl_ct = 0;
   XRDB__db_decl = (XRDB_DECL) calloc(XRDB__db_decl_max,sizeof(XRDB_DECL_INFO));
   XRDB__db_decl_id = 0;

   XRDB__db_call_max = 128;
   XRDB__db_call_ct = 0;
   XRDB__db_call = (XRDB_CALL) calloc(XRDB__db_call_max,sizeof(XRDB_CALL_INFO));

   XRDB__db_fct_max = 32;
   XRDB__db_fct_ct = 0;
   XRDB__db_fct = (XRDB_FCT) calloc(XRDB__db_fct_max,sizeof(XRDB_FCT_INFO));

   XRDB__db_hier_max = 2;
   XRDB__db_hier_ct = 0;
   XRDB__db_hier = (XRDB_HIER) calloc(XRDB__db_hier_max,sizeof(XRDB_HIER_INFO));

   XRDB__db_memb_max = 32;
   XRDB__db_memb_ct = 0;
   XRDB__db_memb = (XRDB_MEMB) calloc(XRDB__db_memb_max,sizeof(XRDB_MEMB_INFO));

   XRDB__db_mdef_max = 32;
   XRDB__db_mdef_ct = 0;
   XRDB__db_mdef = (XRDB_MDEF) calloc(XRDB__db_mdef_max,sizeof(XRDB_MDEF_INFO));
};





/************************************************************************/
/*									*/
/*	XRDB_db_store -- save database					*/
/*	XRDB_db_load -- reload saved database				*/
/*									*/
/************************************************************************/


void
XRDB_db_store(file)
   String file;
{
   FILE * fp;
   Integer i;
   Sequence l;
   String s;
   Character buf[256];

   if (XRDB__db_file_ct == 0 || XRDB__db_fct_ct == 0) {
      unlink(file);
      return;
    };

   fp = fopen(file,"w");
   if (fp == NULL) return;

   fprintf(fp,"XRDB VERSION %d\n",XRDB_VERSION);
   fprintf(fp,"%s %s\n",cur_system->name,cur_system->directory);

   fprintf(fp,"SYSTEMS %d\n",LENGTH(XRDB__all_systems));
   forin (s,String,l,XRDB__all_systems) {
      fprintf(fp,"%s\n",s);
    };

   fprintf(fp,"NAMES %d\n",XRDB__db_name_ct);
   for (i = 0; i < XRDB__db_name_ct; ++i) {
      fprintf(fp,"%d %s\n",XRDB__db_name[i].id,XRDB__db_name[i].name);
    };

   fprintf(fp,"FILES %d\n",XRDB__db_file_ct);
   for (i = 0; i < XRDB__db_file_ct; ++i) {
      fprintf(fp,"%d %d %d %d\n",XRDB__db_file[i].name,XRDB__db_file[i].tail,
		 XRDB__db_file[i].id,XRDB__db_file[i].usedby);
    };

   fprintf(fp,"SCOPES %d\n",XRDB__db_scope_ct);
   for (i = 0; i < XRDB__db_scope_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d %d\n",XRDB__db_scope[i].class,XRDB__db_scope[i].id,
		 XRDB__db_scope[i].start_line,XRDB__db_scope[i].end_line,
		 XRDB__db_scope[i].file,XRDB__db_scope[i].inside);
    };

   fprintf(fp,"TYPES %d\n",XRDB__db_type_ct);
   for (i = 0; i < XRDB__db_type_ct; ++i) {
      fprintf(fp,"%d %d\n",XRDB__db_type[i].id,XRDB__db_type[i].type);
    };

   fprintf(fp,"REFS %d\n",XRDB__db_ref_ct);
   for (i = 0; i < XRDB__db_ref_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d\n",XRDB__db_ref[i].name,XRDB__db_ref[i].file,
		 XRDB__db_ref[i].line,XRDB__db_ref[i].assign,
		 XRDB__db_ref[i].function);
    };

   fprintf(fp,"DECLS %d\n",XRDB__db_decl_ct);
   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d %d %d %d\n",XRDB__db_decl[i].name,XRDB__db_decl[i].scope,
		 XRDB__db_decl[i].file,XRDB__db_decl[i].type,XRDB__db_decl[i].class,
		 XRDB__db_decl[i].line,XRDB__db_decl[i].id,XRDB__db_decl[i].function);
    };

   fprintf(fp,"CALLS %d\n",XRDB__db_call_ct);
   for (i = 0; i < XRDB__db_call_ct; ++i) {
      fprintf(fp,"%d %d %d %d\n",XRDB__db_call[i].from,XRDB__db_call[i].to,
		 XRDB__db_call[i].file,XRDB__db_call[i].line);
    };

   fprintf(fp,"FCTS %d\n",XRDB__db_fct_ct);
   for (i = 0; i < XRDB__db_fct_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d %s\n",XRDB__db_fct[i].name,XRDB__db_fct[i].file,
		 XRDB__db_fct[i].line,XRDB__db_fct[i].scope,XRDB__db_fct[i].numargs,
		 XRDB__db_fct[i].args);
    };

   fprintf(fp,"HIER %d\n",XRDB__db_hier_ct);
   for (i = 0; i < XRDB__db_hier_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d %d %d\n",XRDB__db_hier[i].class,
		 XRDB__db_hier[i].parent,XRDB__db_hier[i].file,
		 XRDB__db_hier[i].line,XRDB__db_hier[i].public,
		 XRDB__db_hier[i].virtual,XRDB__db_hier[i].friend);
    };

   fprintf(fp,"MEMB %d\n",XRDB__db_memb_ct);
   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      fprintf(fp,"%d %d %d %d %d %d %d %d %d %d %d %d\n",XRDB__db_memb[i].class,
		 XRDB__db_memb[i].member,
		 XRDB__db_memb[i].file,XRDB__db_memb[i].line,
		 XRDB__db_memb[i].protect,XRDB__db_memb[i].isdata,
		 XRDB__db_memb[i].inline,
		 XRDB__db_memb[i].friend,XRDB__db_memb[i].virtual,
		 XRDB__db_memb[i].isstatic,XRDB__db_memb[i].ispure,
		 XRDB__db_memb[i].isconst);
    };

   fclose(fp);
   if (XRDB__compress) {
      sprintf(buf,"compress -f %s &",file);
      system(buf);
    };
};





Integer
XRDB_db_load(file)
   String file;
{
   FILE * fp;
   Character buf[1024],lbuf[1024],mbuf[1024];
   Integer i,j,k;
   time_t time;
   struct stat sinfo;
   Boolean compress;

   sprintf(buf,"%s.Z",file);
   if (stat(file,&sinfo) >= 0) {
      compress = FALSE;
      fp = fopen(file,"r");
    }
   else if (stat(buf,&sinfo) >= 0) {
      file = buf;
      compress = TRUE;
      XRDB__compress = TRUE;
      sprintf(mbuf,"zcat %s",buf);
      fp = popen(mbuf,"r");
    }
   else return 0;

   if (fp == NULL) return 0;

   time = sinfo.st_mtime;

   if (name_hash_valid) {
      hdestroy();
      name_hash_valid = FALSE;
    };

   name_ids_ok = TRUE;
   do_elim = FALSE;

   fgets(lbuf,1024,fp);
   if (sscanf(lbuf,"XRDB VERSION %d",&i) != 1 || i != XRDB_VERSION) {
      if (compress) pclose(fp);
      else fclose(fp);
      return 0;
    };

   fprintf(stderr,"XREFDB: Loading %s\n",file);

   fgets(lbuf,1024,fp);

   if (fscanf(fp,"SYSTEMS %d\n",&i) == 1) {
      fgets(lbuf,1024,fp);
      i = strlen(lbuf);
      if (lbuf[i-1] == '\n') lbuf[i-1] = 0;
      XRDB_db_add_system(lbuf);
    };

   fscanf(fp,"NAMES %d\n",&XRDB__db_name_ct);
   if (XRDB__db_name_ct > XRDB__db_name_max) {
      XRDB__db_name_max = XRDB__db_name_ct*2;
      XRDB__db_name = (XRDB_NAME) realloc(XRDB__db_name,XRDB__db_name_max*sizeof(XRDB_NAME_INFO));
    };
   XRDB__db_name_id = XRDB__db_name_ct;
   k = 0;
   for (i = 0; i < XRDB__db_name_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %s\n",&XRDB__db_name[k].id,buf);
      if (j != 2)
	 { printf("Xref error on NAMES: %s\n",lbuf); continue; }
      if (XRDB__db_name[k].id >= XRDB__db_name_id) XRDB__db_name_id = XRDB__db_name[k].id+1;
      XRDB__db_name[k].name = SALLOC(buf);
      if (XRDB__db_name[k].id != k+1) name_ids_ok = FALSE;
      ++k;
    };
   XRDB__db_name_ct = k;

   fscanf(fp,"FILES %d\n",&XRDB__db_file_ct);
   if (XRDB__db_file_ct > XRDB__db_file_max) {
      XRDB__db_file_max = XRDB__db_file_ct*2;
      XRDB__db_file = (XRDB_FILE) realloc(XRDB__db_file,XRDB__db_file_max*sizeof(XRDB_FILE_INFO));
    };
   XRDB__db_file_id = XRDB__db_file_ct;
   k = 0;
   for (i = 0; i < XRDB__db_file_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d\n",&XRDB__db_file[k].name,&XRDB__db_file[k].tail,
		    &XRDB__db_file[k].id,&XRDB__db_file[k].usedby);
      if (j != 4)
	 { printf("Xref error on FILES: %s\n",lbuf); continue; }
      if (XRDB__db_file[k].id >= XRDB__db_file_id) XRDB__db_file_id = XRDB__db_file[k].id+1;
      ++k;
    };
   XRDB__db_file_ct = k;

   fscanf(fp,"SCOPES %d\n",&XRDB__db_scope_ct);
   if (XRDB__db_scope_ct > XRDB__db_scope_max) {
      XRDB__db_scope_max = XRDB__db_scope_ct*2;
      XRDB__db_scope = (XRDB_SCOPE) realloc(XRDB__db_scope,XRDB__db_scope_max*sizeof(XRDB_SCOPE_INFO));
    };
   XRDB__db_scope_id = XRDB__db_scope_ct;
   k = 0;
   for (i = 0; i < XRDB__db_scope_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d %d %d\n",&XRDB__db_scope[k].class,&XRDB__db_scope[k].id,
		 &XRDB__db_scope[k].start_line,&XRDB__db_scope[k].end_line,
		 &XRDB__db_scope[k].file,&XRDB__db_scope[k].inside);
      if (j != 6)
	 { printf("Xref error on SCOPES: %s\n",lbuf); continue; }
      if (XRDB__db_scope[k].id >= XRDB__db_scope_id) XRDB__db_scope_id = XRDB__db_scope[k].id+1;
      ++k;
    };
   XRDB__db_scope_ct = k;

   fscanf(fp,"TYPES %d\n",&XRDB__db_type_ct);
   if (XRDB__db_type_ct > XRDB__db_type_max) {
      XRDB__db_type_max = XRDB__db_type_ct*2;
      XRDB__db_type = (XRDB_TYPE) realloc(XRDB__db_type,XRDB__db_type_max*sizeof(XRDB_TYPE_INFO));
    };
   XRDB__db_type_id = XRDB__db_type_ct;
   k = 0;
   for (i = 0; i < XRDB__db_type_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d\n",&XRDB__db_type[k].id,&XRDB__db_type[k].type);
      if (j != 2)
	 { printf("Xref error on TYPES: %s\n",lbuf); continue; }
      if (XRDB__db_type[k].id >= XRDB__db_type_id) XRDB__db_type_id = XRDB__db_type[k].id+1;
      ++k;
    };
   XRDB__db_type_ct = k;

   fscanf(fp,"REFS %d\n",&XRDB__db_ref_ct);
   if (XRDB__db_ref_ct > XRDB__db_ref_max) {
      XRDB__db_ref_max = XRDB__db_ref_ct*2;
      XRDB__db_ref = (XRDB_REF) realloc(XRDB__db_ref,XRDB__db_ref_max*sizeof(XRDB_REF_INFO));
    };
   k = 0;
   for (i = 0; i < XRDB__db_ref_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d %d\n",&XRDB__db_ref[k].name,&XRDB__db_ref[k].file,
		    &XRDB__db_ref[k].line,&XRDB__db_ref[k].assign,
		    &XRDB__db_ref[k].function);
      if (j != 5)
	 { printf("Xref error on REFS: %s\n",lbuf); continue; }
      ++k;
    };
   XRDB__db_ref_ct = k;

   fscanf(fp,"DECLS %d\n",&XRDB__db_decl_ct);
   if (XRDB__db_decl_ct > XRDB__db_decl_max) {
      XRDB__db_decl_max = XRDB__db_decl_ct*2;
      XRDB__db_decl = (XRDB_DECL) realloc(XRDB__db_decl,XRDB__db_decl_max*sizeof(XRDB_DECL_INFO));
    };
   XRDB__db_decl_id = XRDB__db_decl_ct;
   k = 0;
   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d %d %d %d %d\n",&XRDB__db_decl[k].name,&XRDB__db_decl[k].scope,
		    &XRDB__db_decl[k].file,&XRDB__db_decl[k].type,&XRDB__db_decl[k].class,
		    &XRDB__db_decl[k].line,&XRDB__db_decl[k].id,&XRDB__db_decl[k].function);
      if (j != 8)
	 { printf("Xref error on DECLS: %s\n",lbuf); continue; }
      if (XRDB__db_decl[k].id >= XRDB__db_decl_id) XRDB__db_decl_id = XRDB__db_decl[k].id+1;
      ++k;
    };
   XRDB__db_decl_ct = k;

   fscanf(fp,"CALLS %d\n",&XRDB__db_call_ct);
   if (XRDB__db_call_ct > XRDB__db_call_max) {
      XRDB__db_call_max = XRDB__db_call_ct*2;
      XRDB__db_call = (XRDB_CALL) realloc(XRDB__db_call,XRDB__db_call_max*sizeof(XRDB_CALL_INFO));
    };
   k = 0;
   for (i = 0; i < XRDB__db_call_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d\n",&XRDB__db_call[k].from,&XRDB__db_call[k].to,
		    &XRDB__db_call[k].file,&XRDB__db_call[k].line);
      if (j != 4)
	 { printf("Xref error on CALLS: %s\n",lbuf); continue; }
      ++k;
    };
   XRDB__db_call_ct = k;

   fscanf(fp,"FCTS %d\n",&XRDB__db_fct_ct);
   if (XRDB__db_fct_ct > XRDB__db_fct_max) {
      XRDB__db_fct_max = XRDB__db_fct_ct*2;
      XRDB__db_fct = (XRDB_FCT) realloc(XRDB__db_fct,XRDB__db_fct_max*sizeof(XRDB_FCT_INFO));
    };
   k = 0;
   for (i = 0; i < XRDB__db_fct_ct; ++i) {
      fgets(lbuf,1024,fp);
      j = XRDB_scan_string(lbuf,"%d %d %d %d %d %s\n",&XRDB__db_fct[k].name,&XRDB__db_fct[k].file,
		    &XRDB__db_fct[k].line,&XRDB__db_fct[k].scope,&XRDB__db_fct[k].numargs,
		    buf);
      if (j < 5)
	 { printf("Xref error on FCTS: %s\n",lbuf); continue; }
      if (j == 6) XRDB__db_fct[k].args = SALLOC(buf);
      else XRDB__db_fct[k].args = SALLOC("");
      ++k;
    };
   XRDB__db_fct_ct = k;

   i = 0;
   if (fscanf(fp,"HIER %d\n",&i) == 1) {
      XRDB__db_hier_ct = i;
      if (XRDB__db_hier_ct > XRDB__db_hier_max) {
	 XRDB__db_hier_max = XRDB__db_hier_ct*2;
	 XRDB__db_hier = (XRDB_HIER) realloc(XRDB__db_hier,XRDB__db_hier_max*sizeof(XRDB_HIER_INFO));
       };
      k = 0;
      for (i = 0; i < XRDB__db_hier_ct; ++i) {
	 fgets(lbuf,1024,fp);
	 j = XRDB_scan_string(lbuf,"%d %d %d %d %d %d %d\n",&XRDB__db_hier[k].class,
		       &XRDB__db_hier[k].parent,&XRDB__db_hier[k].file,
		       &XRDB__db_hier[k].line,&XRDB__db_hier[k].public,
		       &XRDB__db_hier[k].virtual,&XRDB__db_hier[k].friend);
	 if (j != 7)
	    { printf("Xref error on HIER: %s\n",lbuf); continue; }
	 ++k;
       };
      XRDB__db_hier_ct = k;
    };

   i = 0;
   if (fscanf(fp,"MEMB %d\n",&i) == 1) {
      XRDB__db_memb_ct = i;
      if (XRDB__db_memb_ct > XRDB__db_memb_max) {
	 XRDB__db_memb_max = XRDB__db_memb_ct*2;
	 XRDB__db_memb = (XRDB_MEMB) realloc(XRDB__db_memb,XRDB__db_memb_max*sizeof(XRDB_MEMB_INFO));
       };
      k = 0;
      for (i = 0; i < XRDB__db_memb_ct; ++i) {
	 fgets(lbuf,1024,fp);
	 j = XRDB_scan_string(lbuf,"%d %d %d %d %d %d %d %d %d %d %d %d\n",&XRDB__db_memb[k].class,
		       &XRDB__db_memb[k].member,
		       &XRDB__db_memb[k].file,&XRDB__db_memb[k].line,
		       &XRDB__db_memb[k].protect,&XRDB__db_memb[k].isdata,
		       &XRDB__db_memb[k].inline,
		       &XRDB__db_memb[k].friend,&XRDB__db_memb[k].virtual,
		       &XRDB__db_memb[k].isstatic,&XRDB__db_memb[k].ispure,
		       &XRDB__db_memb[k].isconst);
	 if (j != 12)
	    { printf("Xref error on MEMB: %s\n",lbuf); continue; }
	 ++k;
       };
      XRDB__db_memb_ct = k;
    };

   XRDB__db_mdef_ct = 0;

   if (compress) pclose(fp);
   else fclose(fp);

   if (XRDB__db_file_ct == 0 || XRDB__db_fct_ct == 0) return FALSE;

   check_dates(time);

   if (XRDB__db_file_ct == 0 || XRDB__db_fct_ct == 0) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	XRDB_db_system -- define system information			*/
/*	XRDB_db_add_system -- note that system is part of database	*/
/*	XRDB_db_systems -- return list of systems			*/
/*									*/
/************************************************************************/


void
XRDB_db_system(name,dir)
   String name;
   String dir;
{
   if (cur_system->name != NULL) {
      free_database();
    };

   cur_system->name = SALLOC(name);
   cur_system->directory = SALLOC(dir);
};





void
XRDB_db_add_system(name)
   String name;
{
   String s;
   Sequence l;

   s = rindex(name,'/');
   if (s != NULL) name = ++s;

   forin (s,String,l,XRDB__all_systems) {
      if (STREQL(s,name)) return;
    }

   XRDB__all_systems = APPEND(SALLOC(name),XRDB__all_systems);
};






/************************************************************************/
/*									*/
/*	XRDB_db_file_scan -- prepare for file scanning			*/
/*	XRDB_db_check_scan -- check if we should rescan a file		*/
/*	XRDB_db_new_file -- define a new file				*/
/*	XRDB_db_find_file -- find file given name			*/
/*									*/
/************************************************************************/


void
XRDB_db_file_scan()
{
   cur_file = 0;
};





Boolean
XRDB_db_check_scan(file)
   String file;
{
   Integer i;
   XRDB_NAME_ID nid;
   XRDB_FILE_ID fid;
   String s,t;
   Boolean fg;

   if (access(file,4) < 0) return TRUE; 	/* don't scan if it isn't there */

   if (file[0] == '/') fg = FALSE;
   else {
      s = rindex(file,'/');
      if (s != NULL) file = &s[1];
      fg = TRUE;
    };

   nid = XRDB_db_find_name(file);

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (fg) {
	 if (nid == XRDB__db_file[i].tail) break;
       }
      else {
	 if (nid == XRDB__db_file[i].name) break;
       };
    };

   if (i >= XRDB__db_file_ct) return FALSE;

   return TRUE;
};





XRDB_FILE_ID
XRDB_db_new_file(file,dir,scan,lvl)
   String file;
   String dir;
   Boolean scan;
   Integer lvl;
{
   Character buf[1024];
   struct stat sbuf;
   Integer i;
   XRDB_NAME_ID name;
   XRDB_FILE_ID fid;
   String s;

   if (file[0] != '/') sprintf(buf,"%s/%s",dir,file);
   else strcpy(buf,file);

   name = XRDB_db_find_name(buf);
   fid = 0;

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (name == XRDB__db_file[i].name) {
	 if (fid == 0) fid = XRDB__db_file[i].id;
	 if (cur_file == XRDB__db_file[i].usedby) break;
	 if (cur_file == fid && XRDB__db_file[i].usedby == 0) break;
       };
    };

   if (fid == 0) fid = ++XRDB__db_file_id;

   if (i >= XRDB__db_file_ct) {
      if (XRDB__db_file_ct >= XRDB__db_file_max) {
	 XRDB__db_file_max *= 2;
	 XRDB__db_file = (XRDB_FILE) realloc(XRDB__db_file,
					  XRDB__db_file_max*sizeof(XRDB_FILE_INFO));
       };
      i = XRDB__db_file_ct++;

      s = rindex(buf,'/');
      if (s == NULL) s = buf;
      else ++s;

      XRDB__db_file[i].name = name;
      XRDB__db_file[i].tail = XRDB_db_find_name(s);
      XRDB__db_file[i].id = fid;
      XRDB__db_file[i].usedby = cur_file;
    };

   do_elim = TRUE;

   if (cur_file == 0) cur_file = fid;

   if (scan) XRDB_file_check(buf);

   return XRDB__db_file[i].id;
};





XRDB_FILE_ID
XRDB_db_find_file(file)
   String file;
{
   Integer i;
   XRDB_NAME_ID nid;
   XRDB_FILE_ID fid;
   String s,t;
   Boolean fg;

   if (file[0] == '/') fg = FALSE;
   else {
      s = rindex(file,'/');
      if (s != NULL) file = &s[1];
      fg = TRUE;
    };

   nid = XRDB_db_find_name(file);

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (fg) {
	 if (nid == XRDB__db_file[i].tail) break;
       }
      else {
	 if (nid == XRDB__db_file[i].name) break;
       };
    };

   if (i >= XRDB__db_file_ct) {
      fid = XRDB_db_new_file(file,"",FALSE,1);
    }
   else fid = XRDB__db_file[i].id;

   return fid;
};





/************************************************************************/
/*									*/
/*	XRDB_db_new_scope -- get id for a new scope			*/
/*	XRDB_db_set_scope -- set properties of a scope			*/
/*									*/
/************************************************************************/


XRDB_SCOPE_ID
XRDB_db_new_scope()
{
   return ++XRDB__db_scope_id;
};





XRDB_SCOPE_ID
XRDB_db_set_scope(id,file,from,to,class,inid)
   XRDB_SCOPE_ID id;
   XRDB_FILE_ID file;
   XRDB_LINE from;
   XRDB_LINE to;
   XRDB_SCOPE_CLASS class;
   XRDB_SCOPE_ID inid;
{
   Integer i;

   if (XRDB__db_scope_ct >= XRDB__db_scope_max) {
      XRDB__db_scope_max *= 2;
      XRDB__db_scope = (XRDB_SCOPE) realloc(XRDB__db_scope,
					 XRDB__db_scope_max*sizeof(XRDB_SCOPE_INFO));
    };

   i = XRDB__db_scope_ct++;

   if (id == 0) id = XRDB_db_new_scope();

   XRDB__db_scope[i].class = class;
   XRDB__db_scope[i].id = id;
   XRDB__db_scope[i].start_line = from;
   XRDB__db_scope[i].end_line = to;
   XRDB__db_scope[i].file = file;
   XRDB__db_scope[i].inside = inid;

   return XRDB__db_scope[i].id;
};





/************************************************************************/
/*									*/
/*	XRDB_db_find_name -- find name id given name			*/
/*									*/
/************************************************************************/


XRDB_NAME_ID
XRDB_db_find_name(name)
   String name;
{
   Integer i;
   ENTRY item;
   ENTRY * fnd;

   if (!name_hash_valid) {
      hcreate(XRDB__db_ref_max);
      for (i = 0; i < XRDB__db_name_ct; ++i) {
	 item.key = XRDB__db_name[i].name;
	 item.data = (char *) XRDB__db_name[i].id;
	 hsearch(item,ENTER);
       };
      name_hash_valid = TRUE;
    };

   item.key = name;
   item.data = NULL;
   fnd = hsearch(item,FIND);
   if (fnd != NULL) return (XRDB_NAME_ID) fnd->data;

   if (XRDB__db_name_ct >= XRDB__db_name_max) {
      XRDB__db_name_max *= 2;
      XRDB__db_name = (XRDB_NAME) realloc(XRDB__db_name,
					     XRDB__db_name_max*sizeof(XRDB_NAME_INFO));
      if (XRDB__db_name == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   i = XRDB__db_name_ct++;
   XRDB__db_name[i].name = SALLOC(name);
   XRDB__db_name[i].id = ++XRDB__db_name_id;

   if (XRDB__db_name[i].id != i+1) name_ids_ok = FALSE;

   item.key = XRDB__db_name[i].name;
   item.data = (char *) XRDB__db_name[i].id;
   fnd = hsearch(item,ENTER);
   if (fnd == NULL) {
      hdestroy();
      name_hash_valid = FALSE;
    };

   return XRDB__db_name[i].id;
};





/************************************************************************/
/*									*/
/*	XRDB_db_find_type -- find type id given type string		*/
/*									*/
/************************************************************************/


XRDB_TYPE_ID
XRDB_db_find_type(type)
   String type;
{
   Integer i;
   XRDB_NAME_ID nid;

   if (type == NULL || *type == 0) type = "*UNKNOWN*";
   nid = XRDB_db_find_name(type);

   for (i = 0; i < XRDB__db_type_ct; ++i) {
      if (nid == XRDB__db_type[i].type) break;
    };

   if (i >= XRDB__db_type_ct) {
      if (XRDB__db_type_ct >= XRDB__db_type_max) {
	 XRDB__db_type_max *= 2;
	 XRDB__db_type = (XRDB_TYPE) realloc(XRDB__db_type,
					  XRDB__db_type_max*sizeof(XRDB_TYPE_INFO));
	 if (XRDB__db_type == NULL) {
	    fprintf(stderr,"XREFDB:  Out of memory\n");
	    exit(1);
	  };
       };
      i = XRDB__db_type_ct++;
      XRDB__db_type[i].type = nid;
      XRDB__db_type[i].id = ++XRDB__db_type_id;
    };

   return XRDB__db_type[i].id;
};






/************************************************************************/
/*									*/
/*	XRDB_db_ref -- add a reference to the database			*/
/*	XRDB_db_ref_assign -- make reference into assignment		*/
/*									*/
/************************************************************************/


void
XRDB_db_ref(file,line,name,assign,fct)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID name;
   Boolean assign;
   XRDB_NAME_ID fct;
{
   if (XRDB__db_ref_ct >= XRDB__db_ref_max) {
      XRDB__db_ref_max *= 2;
      XRDB__db_ref = (XRDB_REF) realloc(XRDB__db_ref,
				     XRDB__db_ref_max*sizeof(XRDB_REF_INFO));
      if (XRDB__db_ref == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_ref[XRDB__db_ref_ct].name = name;
   XRDB__db_ref[XRDB__db_ref_ct].file = file;
   XRDB__db_ref[XRDB__db_ref_ct].line = line;
   XRDB__db_ref[XRDB__db_ref_ct].assign = assign;
   XRDB__db_ref[XRDB__db_ref_ct].function = fct;
   ++XRDB__db_ref_ct;
};





void
XRDB_db_ref_assign(file,line,name,fct)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID name;
   XRDB_NAME_ID fct;
{
   Integer i;

   for (i = XRDB__db_ref_ct-1; i >= 0; --i) {
      if (XRDB__db_ref[i].name == name &&
	     XRDB__db_ref[i].file == file &&
	     XRDB__db_ref[i].line == line) break;
    };

   if (i < 0) {
      XRDB_db_ref(file,line,name,TRUE,fct);
    }
   else {
      XRDB__db_ref[i].assign = TRUE;
    };
};





/************************************************************************/
/*									*/
/*	XRDB_db_decl -- add declaration to the database 		*/
/*									*/
/************************************************************************/


void
XRDB_db_decl(file,line,name,scope,class,type,fct)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID name;
   XRDB_SCOPE_ID scope;
   XRDB_TYPE_ID type;
   XRDB_NAME_ID fct;
{
   if (XRDB__db_decl_ct >= XRDB__db_decl_max) {
      XRDB__db_decl_max *= 2;
      XRDB__db_decl = (XRDB_DECL) realloc(XRDB__db_decl,
				       XRDB__db_decl_max*sizeof(XRDB_DECL_INFO));
      if (XRDB__db_decl == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_decl[XRDB__db_decl_ct].name = name;
   XRDB__db_decl[XRDB__db_decl_ct].scope = scope;
   XRDB__db_decl[XRDB__db_decl_ct].file = file;
   XRDB__db_decl[XRDB__db_decl_ct].type = type;
   XRDB__db_decl[XRDB__db_decl_ct].class = class;
   XRDB__db_decl[XRDB__db_decl_ct].line = line;
   XRDB__db_decl[XRDB__db_decl_ct].id = ++XRDB__db_decl_id;
   XRDB__db_decl[XRDB__db_decl_ct].function = fct;
   ++XRDB__db_decl_ct;
};





/************************************************************************/
/*									*/
/*	XRDB_db_call -- add call to the database			*/
/*									*/
/************************************************************************/


void
XRDB_db_call(file,line,from,to)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID from;
   XRDB_NAME_ID to;
{
   if (XRDB__db_call_ct >= XRDB__db_call_max) {
      XRDB__db_call_max *= 2;
      XRDB__db_call = (XRDB_CALL) realloc(XRDB__db_call,
				       XRDB__db_call_max*sizeof(XRDB_CALL_INFO));
      if (XRDB__db_call == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_call[XRDB__db_call_ct].from = from;
   XRDB__db_call[XRDB__db_call_ct].to = to;
   XRDB__db_call[XRDB__db_call_ct].file = file;
   XRDB__db_call[XRDB__db_call_ct].line = line;
   ++XRDB__db_call_ct;
};





/************************************************************************/
/*									*/
/*	XRDB_db_fct -- add function information to the database 	*/
/*									*/
/************************************************************************/


void
XRDB_db_fct(file,line,name,scope,narg,args)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID name;
   XRDB_SCOPE_ID scope;
   Integer narg;
   String args;
{
   if (XRDB__db_fct_ct >= XRDB__db_fct_max) {
      XRDB__db_fct_max *= 2;
      XRDB__db_fct = (XRDB_FCT) realloc(XRDB__db_fct,
				     XRDB__db_fct_max*sizeof(XRDB_FCT_INFO));
      if (XRDB__db_fct == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_fct[XRDB__db_fct_ct].name = name;
   XRDB__db_fct[XRDB__db_fct_ct].file = file;
   XRDB__db_fct[XRDB__db_fct_ct].line = line;
   XRDB__db_fct[XRDB__db_fct_ct].scope = scope;
   XRDB__db_fct[XRDB__db_fct_ct].numargs = narg;
   XRDB__db_fct[XRDB__db_fct_ct].args = SALLOC(args);
   ++XRDB__db_fct_ct;
};





/************************************************************************/
/*									*/
/*	XRDB_db_hier -- add hierarchy information to the database	*/
/*									*/
/************************************************************************/


void
XRDB_db_hier(file,line,cls,parent,pub,virt,frnd)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID cls;
   XRDB_NAME_ID parent;
   Boolean frnd;
{
   if (XRDB__db_hier_ct >= XRDB__db_hier_max) {
      XRDB__db_hier_max *= 2;
      XRDB__db_hier = (XRDB_HIER) realloc(XRDB__db_hier,
				     XRDB__db_hier_max*sizeof(XRDB_HIER_INFO));
      if (XRDB__db_hier == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_hier[XRDB__db_hier_ct].class = cls;
   XRDB__db_hier[XRDB__db_hier_ct].parent = parent;
   XRDB__db_hier[XRDB__db_hier_ct].file = file;
   XRDB__db_hier[XRDB__db_hier_ct].line = line;
   XRDB__db_hier[XRDB__db_hier_ct].public = pub;
   XRDB__db_hier[XRDB__db_hier_ct].virtual = virt;
   XRDB__db_hier[XRDB__db_hier_ct].friend = frnd;
   ++XRDB__db_hier_ct;
};





/************************************************************************/
/*									*/
/*	XRDB_db_memb -- add class-member information to the database	*/
/*									*/
/************************************************************************/


void
XRDB_db_memb(file,line,cls,member,protect,isdata,inln,frnd,virt,stat,pur,con)
   XRDB_FILE_ID file;
   XRDB_LINE line;
   XRDB_NAME_ID cls;
   XRDB_NAME_ID member;
   XRDB_PROT_CLASS protect;
   Boolean isdata;
   Boolean inln;
   Boolean frnd;
   Boolean virt;
   Boolean stat;
   Boolean pur;
   Boolean con;
{
   if (XRDB__db_memb_ct >= XRDB__db_memb_max) {
      XRDB__db_memb_max *= 2;
      XRDB__db_memb = (XRDB_MEMB) realloc(XRDB__db_memb,
				     XRDB__db_memb_max*sizeof(XRDB_MEMB_INFO));
      if (XRDB__db_memb == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   XRDB__db_memb[XRDB__db_memb_ct].class = cls;
   XRDB__db_memb[XRDB__db_memb_ct].member = member;
   XRDB__db_memb[XRDB__db_memb_ct].file = file;
   XRDB__db_memb[XRDB__db_memb_ct].line = line;
   XRDB__db_memb[XRDB__db_memb_ct].protect = protect;
   XRDB__db_memb[XRDB__db_memb_ct].isdata = isdata;
   XRDB__db_memb[XRDB__db_memb_ct].inline = inln;
   XRDB__db_memb[XRDB__db_memb_ct].friend = frnd;
   XRDB__db_memb[XRDB__db_memb_ct].virtual = virt;
   XRDB__db_memb[XRDB__db_memb_ct].isstatic = stat;
   XRDB__db_memb[XRDB__db_memb_ct].ispure = pur;
   XRDB__db_memb[XRDB__db_memb_ct].isconst = con;
   ++XRDB__db_memb_ct;
};





/************************************************************************/
/*									*/
/*	XRDB_db_file_string -- find string given file value		*/
/*	XRDB_db_type_string -- find string given type value		*/
/*	XRDB_db_name_string -- find string given name value		*/
/*									*/
/************************************************************************/


String
XRDB_db_file_string(fid)
   XRDB_FILE_ID fid;
{
   Integer i;

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (XRDB__db_file[i].id == fid) break;
    };

   if (i >= XRDB__db_file_ct) return "*UNKNOWN*";

   return XRDB_db_name_string(XRDB__db_file[i].name);
};





String
XRDB_db_type_string(tid)
   XRDB_TYPE_ID tid;
{
   Integer i;

   for (i = 0; i < XRDB__db_type_ct; ++i) {
      if (XRDB__db_type[i].id == tid) break;
    };

   if (i >= XRDB__db_type_ct) return "*UNKNOWN*";

   return XRDB_db_name_string(XRDB__db_type[i].type);
};





String
XRDB_db_name_string(nid)
   XRDB_NAME_ID nid;
{
   Integer i;

   if (!name_ids_ok) fix_name_ids();

   return XRDB__db_name[nid-1].name;
};






/************************************************************************/
/*									*/
/*	XRDB_db_check_files -- eliminate unused files			*/
/*									*/
/************************************************************************/


void
XRDB_db_check_files(flst)
   Sequence flst;
{
};





/************************************************************************/
/*									*/
/*	XRDB_db_map_name -- map a single name				*/
/*									*/
/************************************************************************/


#define DOMAP1(x)	if (x == f) x = t;

void
XRDB_db_map_name(f,t)
   XRDB_NAME_ID f;
   XRDB_NAME_ID t;
{
   Integer i;

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      DOMAP1(XRDB__db_file[i].name);
      DOMAP1(XRDB__db_file[i].tail);
    };

   for (i = 0; i < XRDB__db_type_ct; ++i) {
      DOMAP1(XRDB__db_type[i].type);
    };

   for (i = 0; i < XRDB__db_ref_ct; ++i) {
      DOMAP1(XRDB__db_ref[i].name);
      DOMAP1(XRDB__db_ref[i].function);
    };

   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      DOMAP1(XRDB__db_decl[i].name);
      DOMAP1(XRDB__db_decl[i].function);
    };

   for (i = 0; i < XRDB__db_call_ct; ++i) {
      DOMAP1(XRDB__db_call[i].from);
      DOMAP1(XRDB__db_call[i].to);
    };

   for (i = 0; i < XRDB__db_fct_ct; ++i) {
      DOMAP1(XRDB__db_fct[i].name);
    };

   for (i = 0; i < XRDB__db_hier_ct; ++i) {
      DOMAP1(XRDB__db_hier[i].class);
      DOMAP1(XRDB__db_hier[i].parent);
    };

   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      DOMAP1(XRDB__db_memb[i].class);
      DOMAP1(XRDB__db_memb[i].member);
    };

   for (i = 0; i < XRDB__db_mdef_ct; ++i) {
      DOMAP1(XRDB__db_mdef[i].class);
      DOMAP1(XRDB__db_mdef[i].member);
      DOMAP1(XRDB__db_mdef[i].name);
    };
};






/************************************************************************/
/*									*/
/*	XRDB_db_fixup_member_defs -- compute mdef relation		*/
/*									*/
/************************************************************************/


void
XRDB_db_fixup_member_defs()
{
   Integer i,j,k,l,j0;
   String s,p,q;
   XRDB_NAME_ID n1,n2;
   Character buf[1024];
   Integer * sorted;

   if (do_elim) eliminate_duplicates();
   if (!name_ids_ok) fix_name_ids();

   if (XRDB__db_memb_ct == 0 || XRDB__db_mdef_ct != 0) {
      if (name_hash_valid) {
	 hdestroy();
	 name_hash_valid = FALSE;
       };
      return;
    };

   if (XRDB__db_memb_ct > 0) fprintf(stderr,"XREFDB: Computing member definitions\n");

   if (XRDB__db_mdef_max < XRDB__db_memb_ct) {
      while (XRDB__db_mdef_max < XRDB__db_memb_ct) XRDB__db_mdef_max *= 2;
      XRDB__db_mdef = (XRDB_MDEF) realloc(XRDB__db_mdef,
					     XRDB__db_mdef_max*sizeof(XRDB_MDEF_INFO));
      if (XRDB__db_mdef == NULL) {
	 fprintf(stderr,"XREFDB:  Out of memory\n");
	 exit(1);
       };
    };

   sorted = (Integer *) alloca((XRDB__db_memb_ct+4)*sizeof(Integer));
   for (i = 0; i < XRDB__db_memb_ct; ++i) sorted[i] = i;
   qsort(sorted,XRDB__db_memb_ct,sizeof(Integer),memname_comp);

   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      XRDB__db_mdef[i].class = XRDB__db_memb[i].class;
      XRDB__db_mdef[i].member = XRDB__db_memb[i].member;
      XRDB__db_mdef[i].file = 0;
      XRDB__db_mdef[i].line = 0;
      XRDB__db_mdef[i].type = 0;
      XRDB__db_mdef[i].name = 0;
    };

   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      if (XRDB__db_decl[i].class == XRDB_DECL_FIELD ||
	     XRDB__db_decl[i].class == XRDB_DECL_EXTDEF) {
	 s = XRDB_db_name_string(XRDB__db_decl[i].name);
	 p = buf;
	 q = NULL;
	 while (*s != 0) {
	    if (*s == ':' && s[1] == ':') {
	       *p++ = 0;
	       q = p;
	       s += 2;
	     }
	    else *p++ = *s++;
	  };
	 *p = 0;
	 if (q == NULL) continue;
	 n1 = XRDB_db_find_name(buf);
	 n2 = XRDB_db_find_name(q);
	 for (j0 = start_index(sorted,n2); j0 < XRDB__db_memb_ct; ++j0) {
	    j = sorted[j0];
	    if (XRDB__db_memb[j].member != n2) break;
	    else if (XRDB__db_mdef[j].class == n1) {
	       XRDB__db_mdef[j].file = XRDB__db_decl[i].file;
	       XRDB__db_mdef[j].line = XRDB__db_decl[i].line;
	       XRDB__db_mdef[j].type = XRDB__db_decl[i].type;
	       XRDB__db_mdef[j].name = XRDB__db_decl[i].name;
	       break;
	     };
	  };
       }
      else if (XRDB__db_decl[i].class == XRDB_DECL_EFUNCTION ||
		  XRDB__db_decl[i].class == XRDB_DECL_SFUNCTION) {
	 s = XRDB_db_name_string(XRDB__db_decl[i].name);
	 p = buf;
	 q = NULL;
	 while (*s != 0) {
	    if (*s == ':' && s[1] == ':') {
	       *p++ = 0;
	       q = p;
	       s += 2;
	     }
	    else if (*s == '(') break;
	    else *p++ = *s++;
	  };
	 *p = 0;
	 if (*s != '(') continue;
	 k = 2;
	 l = -1;
	 if (q != NULL) {
	    n1 = XRDB_db_find_name(buf);
	    n2 = XRDB_db_find_name(q);
	    for (j0 = start_index(sorted,n2); j0 < XRDB__db_memb_ct; ++j0) {
	       j = sorted[j0];
	       if (XRDB__db_memb[j].member != n2) break;
	       else if (XRDB__db_mdef[j].class == n1 &&
		      XRDB__db_memb[j].file == XRDB__db_decl[i].file &&
		      XRDB__db_decl[i].line >= XRDB__db_memb[j].line &&
		      XRDB__db_decl[i].line - XRDB__db_memb[j].line < k) {
		  k = XRDB__db_decl[i].line - XRDB__db_memb[j].line;
		  l = j;
		};
	     };
	    if (l >= 0) {
	       XRDB__db_mdef[l].type = XRDB__db_decl[i].type;
	       XRDB__db_mdef[l].name = XRDB__db_decl[i].name;
	     };
	  }
	 else {
	    n2 = XRDB_db_find_name(buf);
	    for (j0 = start_index(sorted,n2); j0 < XRDB__db_memb_ct; ++j0) {
	       j = sorted[j0];
	       if (XRDB__db_memb[j].member != n2) break;
	       else if (XRDB__db_memb[j].friend) {
		  if (XRDB__db_memb[j].file == XRDB__db_decl[i].file &&
		      XRDB__db_decl[i].line >= XRDB__db_memb[j].line &&
		      XRDB__db_decl[i].line - XRDB__db_memb[j].line < k) {
		     k = XRDB__db_decl[i].line - XRDB__db_memb[j].line;
		     XRDB__db_mdef[j].type = XRDB__db_decl[i].type;
		     XRDB__db_mdef[j].name = XRDB__db_decl[i].name;
		   }
		  else if (XRDB__db_mdef[j].name == 0) {
		     XRDB__db_mdef[j].type = XRDB__db_decl[i].type;
		     XRDB__db_mdef[j].name = XRDB__db_decl[i].name;
		   }
		};
	     };
	  };
       };
    };

   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      if (XRDB__db_mdef[i].name > 0 && XRDB__db_mdef[i].file == 0) {
	 for (j = 0; j < XRDB__db_fct_ct; ++j) {
	    if (XRDB__db_fct[j].name == XRDB__db_mdef[i].name) {
	       XRDB__db_mdef[i].file = XRDB__db_fct[j].file;
	       XRDB__db_mdef[i].line = XRDB__db_fct[j].line;
	       break;
	     };
	  };
       };
    };

   j = 0;
   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      if (XRDB__db_mdef[i].file == 0 || XRDB__db_mdef[i].line == 0) {
	 if (XRDB__db_mdef[i].name != 0) {
	    XRDB__db_mdef[i].file = XRDB__db_memb[i].file;
	    XRDB__db_mdef[i].line = XRDB__db_memb[i].line;
	  };
       };
      if (XRDB__db_mdef[i].file == 0 || XRDB__db_mdef[i].line == 0 ||
	     XRDB__db_mdef[i].type == 0 || XRDB__db_mdef[i].name == 0) ++j;
      else if (j > 0) XRDB__db_mdef[i-j] = XRDB__db_mdef[i];
    };
   XRDB__db_mdef_ct = XRDB__db_memb_ct - j;

   if (name_hash_valid) {
      hdestroy();
      name_hash_valid = FALSE;
    };
};





static int
memname_comp(ip,jp)
   Integer * ip;
   Integer * jp;
{
   return XRDB__db_memb[*ip].member - XRDB__db_memb[*jp].member;
};





static Integer
start_index(sorted,n)
   Integer * sorted;
   Integer n;
{
   Integer hi,lo,md,n1;

   lo = 0;
   hi = XRDB__db_memb_ct-1;

   while (hi > lo) {
      md = (hi+lo)/2;
      n1 = XRDB__db_memb[sorted[md]].member;
      if (n < n1) hi = md-1;
      else if (n > n1) lo = md+1;
      else break;
      md = lo;
    };

   while (XRDB__db_memb[sorted[md]].member == n && md >= 0) --md;
   ++md;

   return md;
};





/************************************************************************/
/*									*/
/*	free_database -- free the whole kit and kaboodle		*/
/*									*/
/************************************************************************/


static void
free_database()
{
   Sequence l;
   String s;

   cur_system->name = NULL;
   cur_system->directory = NULL;

   forin (s,String,l,XRDB__all_systems) {
      SFREE(s);
    };
   LFREE(XRDB__all_systems);
   XRDB__all_systems = NULL;

   XRDB__db_file_ct = 0;
   XRDB__db_file_id = 0;

   XRDB__db_scope_ct = 0;
   XRDB__db_scope_id = 0;

   XRDB__db_name_ct = 0;
   XRDB__db_name_id = 0;

   XRDB__db_type_ct = 0;
   XRDB__db_type_id = 0;

   XRDB__db_ref_ct = 0;

   XRDB__db_decl_ct = 0;
   XRDB__db_decl_id = 0;

   XRDB__db_call_ct = 0;

   XRDB__db_fct_ct = 0;

   XRDB__db_hier_ct = 0;

   XRDB__db_memb_ct = 0;

   XRDB__db_mdef_ct = 0;
};





/************************************************************************/
/*									*/
/*	fix_name_ids							*/
/*									*/
/************************************************************************/


#define DOMAP(x) x = map[x]

static void
fix_name_ids()
{
   Integer * map;
   Integer i;

   if (name_ids_ok) return;

   map = (Integer *) alloca(sizeof(Integer)*(XRDB__db_name_id+1));

   for (i = 0; i <= XRDB__db_name_id; ++i) map[i] = 0;

   for (i = 0; i < XRDB__db_name_ct; ++i) {
      map[XRDB__db_name[i].id] = i+1;
    };

   for (i = 0; i < XRDB__db_name_ct; ++i) {
      DOMAP(XRDB__db_name[i].id);
    };

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      DOMAP(XRDB__db_file[i].name);
      DOMAP(XRDB__db_file[i].tail);
    };

   for (i = 0; i < XRDB__db_type_ct; ++i) {
      DOMAP(XRDB__db_type[i].type);
    };

   for (i = 0; i < XRDB__db_ref_ct; ++i) {
      DOMAP(XRDB__db_ref[i].name);
      DOMAP(XRDB__db_ref[i].function);
    };

   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      DOMAP(XRDB__db_decl[i].name);
      DOMAP(XRDB__db_decl[i].function);
    };

   for (i = 0; i < XRDB__db_call_ct; ++i) {
      DOMAP(XRDB__db_call[i].from);
      DOMAP(XRDB__db_call[i].to);
    };

   for (i = 0; i < XRDB__db_fct_ct; ++i) {
      DOMAP(XRDB__db_fct[i].name);
    };

   for (i = 0; i < XRDB__db_hier_ct; ++i) {
      DOMAP(XRDB__db_hier[i].class);
      DOMAP(XRDB__db_hier[i].parent);
    };

   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      DOMAP(XRDB__db_memb[i].class);
      DOMAP(XRDB__db_memb[i].member);
    };

   for (i = 0; i < XRDB__db_mdef_ct; ++i) {
      DOMAP(XRDB__db_mdef[i].class);
      DOMAP(XRDB__db_mdef[i].member);
      DOMAP(XRDB__db_mdef[i].name);
    };

   name_ids_ok = TRUE;
};






/************************************************************************/
/*									*/
/*	eliminate_duplicates -- remove duplicate tuples from database	*/
/*									*/
/************************************************************************/


static void
eliminate_duplicates()
{
   Integer i,j,k;
   XRDB_SCOPE s1,s2;

   fprintf(stderr,"XREFDB: Eliminating Duplicate Tuples\n");

   qsort(XRDB__db_scope,XRDB__db_scope_ct,sizeof(XRDB_SCOPE_INFO),scope_compare);
   j = 0;
   for (i = 1; i < XRDB__db_scope_ct; ++i) {
      s1 = &XRDB__db_scope[i-1];
      s2 = &XRDB__db_scope[i];
      if (s1->class == s2->class && s1->start_line == s2->start_line &&
	     s1->end_line == s2->end_line && s1->file == s2->file) {
	 ++j;
	 for (k = 0; k < XRDB__db_decl_ct; ++k) {
	    if (XRDB__db_decl[k].scope == s2->id) XRDB__db_decl[k].scope = s1->id;
	  };
	 for (k = 0; k < XRDB__db_fct_ct; ++k) {
	    if (XRDB__db_fct[k].scope == s2->id) XRDB__db_fct[k].scope = s1->id;
	  };
	 for (k = 0; k < XRDB__db_scope_ct; ++k) {
	    if (XRDB__db_scope[k].inside == s2->id) XRDB__db_scope[k].inside = s1->id;
	  };
	 s2->id = s1->id;
       }
      else if (j > 0) {
	 XRDB__db_scope[i-j] = XRDB__db_scope[j];
       };
    };
   XRDB__db_scope_ct -= j;

   match_length = sizeof(XRDB_REF_INFO)/sizeof(int);
   qsort(XRDB__db_ref,XRDB__db_ref_ct,sizeof(XRDB_REF_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_ref_ct; ++i) {
      if (fields_compare(&XRDB__db_ref[i-1],&XRDB__db_ref[i]) == 0) ++j;
      else if (j > 0) XRDB__db_ref[i-j] = XRDB__db_ref[i];
    };
   XRDB__db_ref_ct -= j;

   match_length = sizeof(XRDB_DECL_INFO)/sizeof(int);
   qsort(XRDB__db_decl,XRDB__db_decl_ct,sizeof(XRDB_DECL_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_decl_ct; ++i) {
      if (fields_compare(&XRDB__db_decl[i-1],&XRDB__db_decl[i]) == 0) ++j;
      else if (j > 0) XRDB__db_decl[i-j] = XRDB__db_decl[i];
    };
   XRDB__db_decl_ct -= j;

   match_length = sizeof(XRDB_CALL_INFO)/sizeof(int);
   qsort(XRDB__db_call,XRDB__db_call_ct,sizeof(XRDB_CALL_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_call_ct; ++i) {
      if (fields_compare(&XRDB__db_call[i-1],&XRDB__db_call[i]) == 0) ++j;
      else if (j > 0) XRDB__db_call[i-j] = XRDB__db_call[i];
    };
   XRDB__db_call_ct -= j;

   match_length = sizeof(XRDB_FCT_INFO)/sizeof(int);
   qsort(XRDB__db_fct,XRDB__db_fct_ct,sizeof(XRDB_FCT_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_fct_ct; ++i) {
      if (fields_compare(&XRDB__db_fct[i-1],&XRDB__db_fct[i]) == 0) ++j;
      else if (j > 0) XRDB__db_fct[i-j] = XRDB__db_fct[i];
    };
   XRDB__db_fct_ct -= j;

   match_length = sizeof(XRDB_HIER_INFO)/sizeof(int);
   qsort(XRDB__db_hier,XRDB__db_hier_ct,sizeof(XRDB_HIER_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_hier_ct; ++i) {
      if (fields_compare(&XRDB__db_hier[i-1],&XRDB__db_hier[i]) == 0) ++j;
      else if (j > 0) XRDB__db_hier[i-j] = XRDB__db_hier[i];
    };
   XRDB__db_hier_ct -= j;

   match_length = sizeof(XRDB_MEMB_INFO)/sizeof(int);
   qsort(XRDB__db_memb,XRDB__db_memb_ct,sizeof(XRDB_MEMB_INFO),fields_compare);
   j = 0;
   for (i = 1; i < XRDB__db_memb_ct; ++i) {
      if (fields_compare(&XRDB__db_memb[i-1],&XRDB__db_memb[i]) == 0) ++j;
      else if (j > 0) XRDB__db_memb[i-j] = XRDB__db_memb[i];
    };
   XRDB__db_memb_ct -= j;

   do_elim = FALSE;
};





static int
scope_compare(s1,s2)
   XRDB_SCOPE s1;
   XRDB_SCOPE s2;
{
   Integer i;

   i = (s1->class - s2->class);
   if (i == 0) {
      i = s1->start_line - s2->start_line;
      if (i == 0) {
	 i = s1->end_line - s2->end_line;
	 if (i == 0) {
	    i = s1->file - s2->file;
	    if (i == 0) i = s1->id - s2->id;
	  };
       };
    };
};





static int
fields_compare(s1,s2)
   Integer * s1;
   Integer * s2;
{
   Integer i,j;

   i = 0;
   for (j = 0; j < match_length; ++j) {
      i = s1[j] - s2[j];
      if (i != 0) break;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	check_dates -- check dates of files after load			*/
/*	mark_valid -- handle validation marking for check_dates 	*/
/*									*/
/************************************************************************/


static void
check_dates(time)
   time_t time;
{
   Integer i,j;
   struct stat sinfo;
   Boolean * flgs;
   Boolean * valid;
   String s;
   Boolean fg;

   flgs = (Boolean *) alloca(XRDB__db_file_id+1);
   for (i = 0; i <= XRDB__db_file_id; ++i) flgs[i] = FALSE;

   valid = (Boolean *) alloca(XRDB__db_file_ct);
   for (i = 0; i < XRDB__db_file_ct; ++i) valid[i] = 2;

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (valid[i] != 2) continue;
      s = XRDB_db_name_string(XRDB__db_file[i].name);
      if (stat(s,&sinfo) < 0) fg = FALSE;
      else if (sinfo.st_mtime > time) fg = FALSE;
      else fg = TRUE;
      mark_valid(XRDB__db_file[i].id,fg,valid,flgs);
    };

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (valid[i] && XRDB__db_file[i].usedby == 0) break;
    };

   if (i >= XRDB__db_file_ct) {
      free_database();
      return;
    };

   free_file_info(flgs);
};





static void
mark_valid(id,fg,valid,flgs)
   XRDB_FILE_ID id;
   Boolean fg;
   Boolean * valid;
   Boolean * flgs;
{
   Integer i;

   flgs[id] = fg;

   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (valid[i] == 0) continue;
      if (XRDB__db_file[i].id == id) {
	 valid[i] = fg;
	 if (!fg && XRDB__db_file[i].usedby != 0) {
	    mark_valid(XRDB__db_file[i].usedby,fg,valid,flgs);
	  };
       };
      if (!fg && XRDB__db_file[i].usedby == id) valid[i] = 0;
    };
};




/************************************************************************/
/*									*/
/*	free_file_info -- free information associated with old files	*/
/*									*/
/************************************************************************/


static void
free_file_info(flgs)
   Boolean * flgs;
{
   Integer i,j;

   j = 0;
   for (i = 0; i < XRDB__db_file_ct; ++i) {
      if (!flgs[XRDB__db_file[i].id]) ++j;
      else if (j != 0) {
	 XRDB__db_file[i-j] = XRDB__db_file[i];
       };
    };
   XRDB__db_file_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_scope_ct; ++i) {
      if (!flgs[XRDB__db_scope[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_scope[i-j] = XRDB__db_scope[i];
       };
    };
   XRDB__db_scope_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_ref_ct; ++i) {
      if (!flgs[XRDB__db_ref[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_ref[i-j] = XRDB__db_ref[i];
       };
    };
   XRDB__db_ref_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_decl_ct; ++i) {
      if (!flgs[XRDB__db_decl[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_decl[i-j] = XRDB__db_decl[i];
       };
    };
   XRDB__db_decl_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_call_ct; ++i) {
      if (!flgs[XRDB__db_call[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_call[i-j] = XRDB__db_call[i];
       };
    };
   XRDB__db_call_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_fct_ct; ++i) {
      if (!flgs[XRDB__db_fct[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_fct[i-j] = XRDB__db_fct[i];
       };
    };
   XRDB__db_fct_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_hier_ct; ++i) {
      if (!flgs[XRDB__db_hier[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_hier[i-j] = XRDB__db_hier[i];
       };
    };
   XRDB__db_hier_ct -= j;

   j = 0;
   for (i = 0; i < XRDB__db_memb_ct; ++i) {
      if (!flgs[XRDB__db_memb[i].file]) ++j;
      else if (j != 0) {
	 XRDB__db_memb[i-j] = XRDB__db_memb[i];
       };
    };
   XRDB__db_memb_ct -= j;

   XRDB__db_mdef_ct = 0;
};





/* end of xrdbdata.c */




