/************************************************************************/
/*									*/
/*		msgserver.c						*/
/*									*/
/*	Server for message handling					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "msg_local.h"

#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <pwd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/file.h>
#include <netdb.h>

#ifdef mips
#include <unistd.h>
#endif
#ifdef vax
#include <unistd.h>
#endif




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_FILES	64
#define MSG_READ_SIZE	1024




/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _MSG_INFO *	MSG_INFO;
typedef struct _MSG_QUEUE *	MSG_QUEUE;
typedef struct _MSG_CALL *	   MSG_CALL;



typedef struct _MSG_INFO {
   String text;
   PMAT_PAT pattern;
   Integer socket;
   MSG_INFO next;
   Integer count;
} MSG_INFO_INFO;




typedef struct _MSG_QUEUE {
   String text;
   Integer socket;
   MSG_QUEUE next;
} MSG_QUEUE_INFO;




typedef struct _MSG_CALL {
   Character callid;
   Character ourid;
   Integer fd;
   String rslt;
   Boolean done;
   fd_set who;
   Integer count;
} MSG_CALL_INFO;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	socket_fid;
static	Integer 	lock_fid;

static	fd_set		readfds;
static	fd_set		writefds;
static	Integer 	maxfile;

static	MSG_INFO	all_pats;
static	MSG_QUEUE	all_queued;

static	Integer 	numfiles;
static	Short		files[MAX_FILES];
static	FILE *		filed[MAX_FILES];
static	String		filemsg[MAX_FILES];
static	Integer 	filemln[MAX_FILES];

static	Boolean 	debugfg;
static	FILE *		debugf;

extern	int		errno;

static	Sequence	call_msgs;
static	Integer 	call_id;

static	String		lock_file;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	create_server();
static	void		accept_client();
static	void		get_request();
static	void		process_request();
static	void		register_msg();
static	void		free_msg();
static	void		handle_msg();
static	Boolean 	send_msg();
static	void		check_write();
static	void		fix_write_fds();
static	void		setup_call();
static	void		handle_reply();
static	void		send_reply();
static	void		check_close();
static	void		get_lock_name();



/************************************************************************/
/*									*/
/*	main -- main program for server 				*/
/*									*/
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   fd_set rfds,wfds;
   Integer i,j;
   Boolean force;

   debugfg = FALSE;
   lock_file = NULL;
   force = FALSE;

   signal(SIGPIPE,SIG_IGN);

   for (i = 1; i < argc; ++i) {
      if (STREQL(argv[i],"-D")) {
	 debugf = fopen("msg.trace","w");
	 if (debugf == NULL) debugf = stderr;
	 if (debugf != NULL) debugfg = TRUE;
	 setbuf(debugf,NULL);
       }
      else if (strncmp(argv[i],"-msg",2) == 0 && i+1 < argc) {
	 lock_file = argv[++i];
       }
      else if (strncmp(argv[i],"-force",2) == 0) {
	 force = TRUE;
       }
      else {
	 fprintf(stderr,"MSG/S: bad parameter list\n");
	 exit(1);
       };
    };

   for (i = 0; i < MAX_FILES; ++i) {
      files[i] = -1;
      filed[i] = NULL;
      filemsg[i] = NULL;
      filemln[i] = 0;
    };

   if (!create_server(force)) {
      exit(1);
    };

   call_id = 0;
   call_msgs = NULL;

   FD_ZERO(&readfds);
   FD_ZERO(&writefds);

   FD_SET(socket_fid,&readfds);
   numfiles = -1;
   maxfile = socket_fid;

   while (numfiles != 0) {
      if (debugfg) fprintf(debugf,"MSG/S: Select\n");
      rfds = readfds;
      wfds = writefds;
      i = select(maxfile+1,&rfds,&wfds,NULL,NULL);
      if (i == 0) continue;
      if (i < 0) {
	 if (errno == EINTR) continue;
	 fprintf(stderr,"MSG/S: Select problem -- %d\n",errno);
	 exit(1);
       };

      if (FD_ISSET(socket_fid,&rfds)) {
	 accept_client();
	 --i;
       };

      for (j = 0; i > 0 && j < numfiles; ++j) {
	 if (FD_ISSET(files[j],&rfds)) {
	    get_request(j);
	    --i;
	  };
	 if (j >= numfiles) continue;
	 if (FD_ISSET(files[j],&wfds)) {
	    check_write(files[j]);
	    --i;
	  };
       };
    };

   exit(0);

   fgets(0,0,0);	/* force fgets to load */
};





/************************************************************************/
/*									*/
/*	create_server -- open a new server socket and lock		*/
/*									*/
/************************************************************************/


static Boolean
create_server(force)
   Boolean force;
{
   Integer s;
   struct sockaddr_in name;
   int namelen,i,fd;
   int v;
   Character buf[128],hbuf[64];
   struct hostent *he;
   struct protoent *pe;

   get_lock_name(buf);
   s = socket(AF_INET,SOCK_STREAM,0);
   if (s < 0) {
      fprintf(stderr,"MSG/S: Can't open socket -- %d\n",errno);
      return FALSE;
    };
   bzero(&name,sizeof(name));
   name.sin_family = AF_INET;
   if (bind(s,&name,sizeof(name)) < 0) {
      fprintf(stderr,"MSG/S: Can't bind socket -- %d\n",errno);
      close(s);
      return FALSE;
    };
   namelen = sizeof(name);
   if (getsockname(s,&name,&namelen) < 0) {
      fprintf(stderr,"MSG/S: Can't get socket name -- %d\n",errno);
      close(s);
      return FALSE;
    };

   gethostname(hbuf,64);
   he = gethostbyname(hbuf);
   name.sin_addr.s_addr = * ((int *) he->h_addr);

   if (listen(s,5) < 0) {
      fprintf(stderr,"MSG/S: Can't listen on socket -- %d\n",errno);
      close(s);
      return FALSE;
    };

   i = fcntl(s,F_GETFL,0);
   i |= FNDELAY;
   if (fcntl(s,F_SETFL,i) == -1) {
      fprintf(stderr,"MSG/S: Problem with fcntl -- %d\n",errno);
      close(s);
      return FALSE;
    };

   fd = open(buf,O_WRONLY|O_CREAT,0777);
   if (fd < 0) {
      fprintf(stderr,"MSG/S: Problem with file open -- %d\n",errno);
      close(s);
      return FALSE;
    };

   for (i = 0; i < 5; ++i) {
      if (lockf(fd,F_TLOCK,0) == 0) break;
      if (errno != EWOULDBLOCK && errno != EAGAIN && errno != EACCES) {
	 fprintf(stderr,"MSG/S: Problem with file lock -- %d\n",errno);
	 close(fd);
	 close(s);
	 return FALSE;
       };
      sleep(1);
    };

   if (i >= 5 && !force) {
      fprintf(stderr,"MSG/S: Server already running\n");
      close(fd);
      close(s);
      return FALSE;
    };

   sprintf(buf,"%s %d\n",hbuf,name.sin_port);
   write(fd,buf,strlen(buf));

   if (debugfg) fprintf(debugf,"MSG/S: Server created\n");

   pe = getprotobyname("TCP");
   if (pe != NULL) {
      v = 1;
      i = setsockopt(s,IPPROTO_TCP,TCP_NODELAY,&v,4);
      if (i < 0) fprintf(stderr,"MSG/S: setsockopt failed -- %d\n",errno);
    };

   socket_fid = s;
   lock_fid = fd;

   return TRUE;
};




/************************************************************************/
/*									*/
/*	accept_client -- accept a client onto the server		*/
/*									*/
/************************************************************************/


static void
accept_client()
{
   struct sockaddr addr;
   int addrlen;
   Integer ns,i;

   addrlen = sizeof(addr);
   ns = accept(socket_fid,&addr,&addrlen);

   if (ns < 0 && errno != EWOULDBLOCK) {
      fprintf(stderr,"MSG/S: Problem with accept -- %d\n",errno);
      return;
    };

   i = fcntl(ns,F_GETFL,0);
   i |= FNDELAY;
   if (fcntl(ns,F_SETFL,i) == -1) {
      fprintf(stderr,"MSG/S: Problem with fcntl -- %d\n",errno);
      close(ns);
      return;
    };

   if (numfiles < 0) numfiles = 0;
   files[numfiles] = ns;
   filed[numfiles] = fdopen(ns,"r");
   filemln[numfiles] = 1000;
   filemsg[numfiles] = (String) malloc(filemln[numfiles]+2);
   filemsg[numfiles][0] = 0;
   ++numfiles;

   if (debugfg) fprintf(debugf,"MSG/S: Accept client %d\n",ns);

   if (ns > maxfile) maxfile = ns;

   FD_SET(ns,&readfds);
};





/************************************************************************/
/*									*/
/*	get_request -- handle write from client to server		*/
/*									*/
/************************************************************************/


static void
get_request(j)
   Integer j;
{
   String msgbuf;
   Integer ch;
   FILE * fl;
   Integer fd;
   String mpt;
   Integer ln,ct;

   fl = filed[j];
   fd = files[j];

   ln = strlen(filemsg[j]);
   msgbuf = (String) alloca(MSG_READ_SIZE + ln + 10);

   for ( ; ; ) {
      mpt = msgbuf;
      if (filemsg[j][0] != 0) {
	 strcpy(msgbuf,filemsg[j]);
	 filemsg[j][0] = 0;
	 mpt = &msgbuf[strlen(msgbuf)];
       };

      ct = 0;
      for ( ; ; ) {
	 ch = getc(fl);
	 if (ch == EOM_CHAR || ch == ABORT_CHAR) break;
	 if (ch < 0) {
	    if (ferror(fl)) {
	       if (errno == EWOULDBLOCK) break;
	       if (errno == EINTR) continue;
	     }
	    check_close(fd);
	    return;
	  };
	 if (++ct > MSG_READ_SIZE) {
	    *mpt = 0;
	    ln = strlen(msgbuf);
	    mpt = msgbuf;
	    msgbuf = (String) alloca(MSG_READ_SIZE + ln + 10);
	    strcpy(msgbuf,mpt);
	    ct = 0;
	    mpt = &msgbuf[ln];
	  };
	 *mpt++ = ch;
       };

      if (ch == EOM_CHAR) {
	 *mpt = 0;
	 process_request(msgbuf,fd);
       }
      else if (ch == ABORT_CHAR) ;
      else if (mpt != msgbuf) {
	 *mpt = 0;
	 ln = strlen(msgbuf);
	 if (ln > filemln[j]) {
	    while (ln > filemln[j]) filemln[j] *= 2;
	    filemsg[j] = (String) realloc(filemsg[j],filemln[j]+2);
	  };
	 strcpy(filemsg[j],msgbuf);
	 break;
       }
      else break;
    };

   clearerr(fl);
};





/************************************************************************/
/*									*/
/*	process_request -- process request from client			*/
/*									*/
/************************************************************************/


static void
process_request(msg,fd)
   String msg;
   Integer fd;
{
   switch (msg[0]) {
      case REGISTER_CHAR :
	 register_msg(&msg[1],fd);
	 break;
      case FREE_CHAR :
	 free_msg(&msg[1],fd);
	 break;
      case CALL_CHAR :
	 setup_call(msg[1],&msg[2],fd);
	 break;
      case REPLY_CHAR :
	 handle_reply(msg[1],&msg[2],fd);
	 break;
      default :
	 handle_msg(msg);
	 break;
    };
};





/************************************************************************/
/*									*/
/*	register_msg -- register a message to reply to			*/
/*									*/
/************************************************************************/


static void
register_msg(txt,fd)
   String txt;
   Integer fd;
{
   MSG_INFO msg,m;
   Integer ct;
   String s;

   for (m = all_pats; m != NULL; m = m->next) {
      if (STREQL(txt,m->text) && m->socket == fd) {
	 ++m->count;
	 return;
       };
    };

   ct = 0;
   for (s = txt; *s != 0; ++s) if (*s == '%') ++ct;

   msg = PALLOC(MSG_INFO_INFO);

   msg->text = SALLOC(txt);
   msg->pattern = PMATmake_pattern(txt,ct,NULL);
   msg->socket = fd;
   msg->next = NULL;
   msg->count = 1;

   if (all_pats == NULL) all_pats = msg;
   else {
      for (m = all_pats; m->next != NULL; m = m->next);
      m->next = msg;
    };

   if (debugfg) fprintf(debugf,"MSG/S: %d Register msg: %s\n",fd,txt);
};





/************************************************************************/
/*									*/
/*	free_msg -- remove message reply				*/
/*									*/
/************************************************************************/


static void
free_msg(txt,fd)
   String txt;
   Integer fd;
{
   MSG_INFO m,ml,mn;

   ml = NULL;
   for (m = all_pats; m != NULL; m = mn) {
      mn = m->next;
      if (m->socket == fd && (txt == NULL || STREQL(txt,m->pattern))) {
	 if (txt != NULL && m->count-- > 1) continue;
	 SFREE(m->text);
	 PMATfree_pattern(m->pattern);
	 if (ml == NULL) all_pats = m->next;
	 else ml->next = m->next;
	 free(m);
       }
      else ml = m;
    };

   if (debugfg) {
      if (txt != NULL)
	 fprintf(debugf,"MSG/S: %d Free msg: %s\n",fd,txt);
      else
	 fprintf(debugf,"MSG/S: %d Free all messages\n",fd);
    };
};





/************************************************************************/
/*									*/
/*	handle_msg -- send message to perspective clients		*/
/*									*/
/************************************************************************/


static void
handle_msg(txt)
   String txt;
{
   MSG_INFO m;
   Boolean sent[256];
   Integer i;

   if (debugfg) fprintf(debugf,"MSG/S: Handle msg: %s\n",txt);

   for (i = 0; i <= maxfile; ++i) sent[i] = FALSE;

   for (m = all_pats; m != NULL; m = m->next) {
      if (!sent[m->socket] && PMATmatch(txt,m->pattern,NULL) >= 0) {
	 send_msg(m->socket,txt,NULL);
	 sent[m->socket] = TRUE;
       };
    };
};





/************************************************************************/
/*									*/
/*	send_msg -- send message to client				*/
/*									*/
/************************************************************************/


static Boolean
send_msg(fd,umsg,onq)
   Integer fd;
   String umsg;
   MSG_QUEUE onq;
{
   Integer i,ln;
   MSG_QUEUE q;
   String msg;

   if (debugfg && umsg[0] != CALL_CHAR && umsg[0] != REPLY_CHAR)
      fprintf(debugf,"MSG/S: %d Send msg: %s\n",fd,umsg);

   msg = (String) alloca(strlen(umsg)+10);

   sprintf(msg,"%s%c",umsg,EOM_CHAR);
   if (msg[0] == ABORT_CHAR) msg[1] = 0;

   ln = strlen(msg);
   i = send(fd,msg,ln,0);

   if (i < 0) {
      if (errno != EWOULDBLOCK && errno != EBADF && errno != EINTR) {
	 fprintf(stderr,"MSG/S: problem with send -- %d\n",errno);
	 exit(1);
       };
      if (errno == EBADF) check_close(fd);
    }
   else if (i != ln) {
      msg[ln-1] = 0;
      return send_msg(fd,&msg[i],onq);
    }

   if (i >= 0 && onq != NULL) {
      if (all_queued == onq) all_queued = onq->next;
      else {
	 for (q = all_queued; q->next != NULL && q->next != onq; q = q->next);
	 if (q->next == onq) q->next = onq->next;
       };
      free(onq->text);
      free(onq);
      fix_write_fds();
    }
   else if (i < 0 && onq == NULL) {
      onq = PALLOC(MSG_QUEUE_INFO);
      onq->text = SALLOC(msg);
      onq->socket = fd;
      onq->next = NULL;
      if (all_queued == NULL) all_queued = onq;
      else {
	 for (q = all_queued; q->next != NULL; q = q->next);
	 q->next = onq;
       };
      fix_write_fds();
    };

   return (i >= 0);
};





/************************************************************************/
/*									*/
/*	check_write -- try to write to socket				*/
/*									*/
/************************************************************************/


static void
check_write(fd)
   Integer fd;
{
   MSG_QUEUE q,qn;

   for (q = all_queued; q != NULL; q = qn) {
      qn = q->next;
      if (q->socket == fd) {
	 if (!send_msg(q->socket,q->text,q)) break;
       };
    };
};





/************************************************************************/
/*									*/
/*	fix_write_fds -- set up set of writefds correctly		*/
/*									*/
/************************************************************************/


static void
fix_write_fds()
{
   MSG_QUEUE q;

   FD_ZERO(&writefds);

   for (q = all_queued; q != NULL; q = q->next) {
      FD_SET(q->socket,&writefds);
    };
};





/************************************************************************/
/*									*/
/*	setup_call -- setup and process a CALL message			*/
/*									*/
/************************************************************************/


static void
setup_call(id,msg,fd)
   Integer id;
   String msg;
   Integer fd;
{
   MSG_INFO m;
   MSG_CALL mc,mca;
   Sequence l;
   String buf;
   Boolean sent[256];
   Integer i;

   if (debugfg) fprintf(debugf,"MSG/S: Handle call %d: %s\n",id,msg);

   for (i = 0; i <= maxfile; ++i) sent[i] = FALSE;

   mc = PALLOC(MSG_CALL_INFO);
   mc->callid = id;
   for ( ; ; ) {
      call_id = (call_id + 1) % 32;
      forin (mca,MSG_CALL,l,call_msgs) {
	 if (mca->ourid == call_id + '@') break;
       };
      if (EMPTY(l)) break;
    };
   mc->ourid = call_id + '@';
   mc->fd = fd;
   FD_ZERO(&mc->who);
   mc->count = 0;
   mc->done = FALSE;
   mc->rslt = NULL;
   call_msgs = CONS(mc,call_msgs);

   buf = (String) alloca(strlen(msg)+10);
   sprintf(buf,"%c%c%s",CALL_CHAR,mc->ourid,msg);

   for (m = all_pats; m != NULL; m = m->next) {
      if (PMATmatch(msg,m->pattern,NULL) >= 0) {
	 mc->count += m->count;
	 if (!sent[m->socket]) {
	    FD_SET(m->socket,&mc->who);
	    if (debugfg) fprintf(debugf,"MSG/S: Call %d (%d) to %d: %s\n",
				    id,mc->ourid,m->socket,msg);
	    send_msg(m->socket,buf,NULL);
	    sent[m->socket] = TRUE;
	  };
       };
    };

   if (mc->count == 0) send_reply(mc);
};





/************************************************************************/
/*									*/
/*	handle_reply -- get reply from a client 			*/
/*									*/
/************************************************************************/


static void
handle_reply(id,msg,fd)
   Integer id;
   String msg;
   Integer fd;
{
   MSG_CALL mc;
   Sequence l;

   if (debugfg) {
      fprintf(debugf,"MSG/S: Reply from %d for %d: %s\n",
		 fd,id,(msg == NULL ? "*NONE*" : msg));
    };

   forin (mc,MSG_CALL,l,call_msgs) {
      if (mc->ourid == id) break;
    };
   if (mc == NULL) return;

   if ((msg != NULL && msg[0] != 0) || mc->count == 1) {
      if (msg != NULL && *msg != 0) mc->rslt = SALLOC(msg);
      send_reply(mc);
    }
   else {
      FD_CLR(fd,&mc->who);
      mc->count -= 1;
    };
};





/************************************************************************/
/*									*/
/*	send_reply -- reply to a call message				*/
/*									*/
/************************************************************************/


static void
send_reply(mc)
   MSG_CALL mc;
{
   Character rbuf[128];
   String buf;

   if (mc->rslt == NULL) buf = rbuf;
   else buf = (String) alloca(strlen(mc->rslt)+10);

   buf[0] = REPLY_CHAR;
   buf[1] = mc->callid;
   if (mc->rslt != NULL) strcpy(&buf[2],mc->rslt);
   else buf[2] = 0;

   call_msgs = REMOB(mc,call_msgs);

   if (debugfg) {
      fprintf(debugf,"MSG/S: Reply to %d for %d: %s\n",
		 mc->fd,mc->callid,&buf[2]);
    };

   send_msg(mc->fd,buf,NULL);

   if (mc->rslt != NULL) SFREE(mc->rslt);
   free(mc);
};





/************************************************************************/
/*									*/
/*	check_close -- something has happened to some file		*/
/*									*/
/************************************************************************/


static void
check_close(fd)
   Integer fd;
{
   Integer i,j;
   MSG_QUEUE q,lq,qn;
   Sequence l;
   MSG_CALL mc;

   /* should double check the file */

   if (debugfg) fprintf(debugf,"MSG/S: Close for %d\n",fd);

   for (i = 0; i < numfiles; ++i) {
      if (files[i] == fd) break;
    };
   if (i >= numfiles) return;

   fclose(filed[i]);
   free(filemsg[i]);
   filemsg[i] = NULL;
   filemln[i] = 0;

   for (j = i+1; j < numfiles; ++j) {
      files[j-1] = files[j];
      filed[j-1] = filed[j];
      filemln[j-1] = filemln[j];
      filemsg[j-1] = filemsg[j];
    };
   --numfiles;

   files[numfiles] = -1;
   filed[numfiles] = NULL;
   filemln[numfiles] = 0;
   filemsg[numfiles] = NULL;

   FD_CLR(fd,&readfds);

   free_msg(NULL,fd);

   lq = NULL;
   for (q = all_queued; q != NULL; q = qn) {
      qn = q->next;
      if (q->socket == fd) {
	 if (lq == NULL) all_queued = q->next;
	 else lq->next = q->next;
	 free(q->text);
	 free(q);
       }
      else lq = q;
    };
   fix_write_fds();

   for (l = call_msgs; l != NULL; l = (l == NULL ? NULL : CDR(l))) {
      mc = CAR(MSG_CALL,l);
      if (mc->fd == fd) {
	 call_msgs = REMOB(mc,call_msgs);
	 free(mc);
	 l = call_msgs;
       }
      else if (FD_ISSET(fd,&mc->who)) {
	 mc->count--;
	 if (mc->count == 0) {
	    send_reply(mc);
	    l = call_msgs;
	  };
       };
    };

   close(fd);
};





/************************************************************************/
/*									*/
/*	get_lock_name -- get name for lock file 			*/
/*									*/
/************************************************************************/


static void
get_lock_name(buf)
   Character buf[];
{
   Character nbuf[32];
   struct passwd * pwd;
   String s;

   if (lock_file != NULL) {
      strcpy(buf,lock_file);
      return;
    };

   gethostname(nbuf,32);
   pwd = getpwuid(getuid());
   if (pwd == NULL) s = (String) getenv("USER");
   else s = pwd->pw_name;
   if (s == NULL) s = "XXX";

   sprintf(buf,"/tmp/msg.%s.%s.addr",nbuf,s);
};





/* end of msgserver.c */
