/************************************************************************/
/*									*/
/*		msgfile.c						*/
/*									*/
/*	General file routines						*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "msg_local.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define WD_SIZE 1024
#define WD_MSG	"MSG FILE_WD %1s"
#define INQ_MSG "MSG FILE_INQ_WD"





/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _FILE_KEY * FILE_KEY;


typedef struct _FILE_KEY {
   String name1;
   String name2;
   Boolean same;
   Boolean valid;
   FILE_KEY next;
} FILE_KEY_INFO;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Character	curwd[WD_SIZE];
static	FILE_KEY	all_keys;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		set_wd_rqst();
static	void		inq_wd_rqst();
static	void		invalidate_wd();




/************************************************************************/
/*									*/
/*	MSG_file_init -- module initialization				*/
/*									*/
/************************************************************************/


void
MSG_file_init()
{
   String s;

   getwd(curwd);
   s = MSGcall(INQ_MSG);
   if (s != NULL && *s != 0 && STRNEQ(s,curwd)) {
      chdir(s);
      strcpy(curwd,s);
    };

   all_keys = NULL;

   MSGregister(WD_MSG,set_wd_rqst,1,NULL);
   MSGregister(INQ_MSG,inq_wd_rqst,0,NULL);
};





/************************************************************************/
/*									*/
/*	MSGinq_wd -- return current working directory			*/
/*									*/
/************************************************************************/


String
MSGinq_wd()
{
   MSGinit(FALSE);

   return curwd;
};





/************************************************************************/
/*									*/
/*	MSGset_wd -- set current working directory			*/
/*									*/
/************************************************************************/


Boolean
MSGset_wd(new)
   String new;
{
   MSGinit(FALSE);

   if (chdir(new) != 0) return FALSE;

   getwd(curwd);

   invalidate_wd();

   MSGsenda("MSG FILE_WD %s",new);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	MSGfile_compare -- compare two filenames as being same file	*/
/*									*/
/************************************************************************/


Boolean
MSGfile_compare(f1,f2)
   String f1;
   String f2;
{
   String s1,s2;
   Integer i,i1,i2;
   FILE_KEY k;
   struct stat sbuf;

   MSGinit(FALSE);

   if (f1 == NULL && f2 == NULL) return TRUE;
   else if (f1 == NULL || f2 == NULL) return FALSE;

   i = strcmp(f1,f2);
   if (i == 0) return TRUE;

   s1 = rindex(f1,'/');
   if (s1 == NULL) s1 = f1;
   else ++s1;
   s2 = rindex(f2,'/');
   if (s2 == NULL) s2 = f2;
   else ++s2;

   if (STRNEQ(s1,s2)) return FALSE;

   if (i >= 0) {		/* insure f1 < f2 */
      s1 = f1;
      f1 = f2;
      f2 = s1;
    };

   for (k = all_keys; k != NULL; k = k->next) {
      if (STREQL(f1,k->name1) && STREQL(f2,k->name2)) break;
    };

   if (k == NULL) {
      k = PALLOC(FILE_KEY_INFO);
      k->name1 = SALLOC(f1);
      k->name2 = SALLOC(f2);
      k->next = all_keys;
      k->valid = FALSE;
      all_keys = k;
    };

   if (!k->valid) {
      if (stat(f1,&sbuf) < 0) i1 = 0;
      else i1 = sbuf.st_ino;
      if (stat(f2,&sbuf) < 0) i2 = 0;
      else i2 = sbuf.st_ino;
      if (i1 == 0 && i2 == 0) k->same = FALSE;
      else k->same = (i1 == i2);
      if (i1 != 0 && i2 != 0) k->valid = TRUE;
    };

   return k->same;
};





/************************************************************************/
/*									*/
/*	MSGfile_check -- check if file exists, try to get it otherwise	*/
/*									*/
/************************************************************************/


Boolean
MSGfile_check(file)
   String file;
{
   String s;

   if (access(file,R_OK) >= 0) return TRUE;

   s = MSGcalla("MSG FILE_GET %s",file);

   if (s != NULL) {
      if (access(file,R_OK) >= 0) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	MSGrundata_name -- get rundata file name			*/
/*	MSGcommand_name -- get command file name			*/
/*									*/
/************************************************************************/


String
MSGrundata_name(name,buf)
   String name;
   String buf;
{
   Boolean experiment;

   experiment = FALSE;
   if (getenv("FIELD_DEBUG") != NULL) experiment = TRUE;

   sprintf(buf,"%s/lib/field/rundata/%s",BWElocal_project(),name);
   if (experiment || access(buf,4) < 0) {
      sprintf(buf,"%s/lib/field/rundata/%s/%s",BWElocal_project(),BWEarch(),name);
      if (experiment || access(buf,4) < 0) {
	 sprintf(buf,"%s/field/rundata/%s/%s",BWElocal_project(),BWEarch(),name);
       };
    };

   return buf;
};






String
MSGcommand_name(name,buf)
   String name;
   String buf;
{
   Boolean experiment;

   experiment = FALSE;
   if (getenv("FIELD_DEBUG") != NULL) experiment = TRUE;

   if (name[0] == '/') {
      strcpy(buf,name);
    }
   else {
      sprintf(buf,"%s/bin/field/%s",BWElocal_project(),name);
      if (experiment || access(buf,1) < 0) {
	 sprintf(buf,"%s/bin/field/%s/%s",BWElocal_project(),BWEarch(),name);
	 if (experiment || access(buf,1) < 0) {
	    sprintf(buf,"%s/field/cmd/%s/%s",BWElocal_project(),BWEarch(),name);
	  };
       };
   };

   return buf;
};





/************************************************************************/
/*									*/
/*	set_wd_rqst -- handle SET WD message				*/
/*	inq_wd_rqst -- handle INQ WD message				*/
/*									*/
/************************************************************************/



static void
set_wd_rqst(wd)
   String wd;
{
   if (STREQL(wd,curwd)) return;

   chdir(wd);
   strcpy(curwd,wd);
   invalidate_wd();
};






static void
inq_wd_rqst(rid)
   Integer rid;
{
   MSGreply(rid,curwd);
};






/************************************************************************/
/*									*/
/*	invalidate_wd -- invalidate by wd				*/
/*									*/
/************************************************************************/


static void
invalidate_wd()
{
   FILE_KEY k;

   for (k = all_keys; k != NULL; k = k->next) {
      if (k->name1[0] != '/' || k->name2[0] != '/') k->valid = FALSE;
    };
};





/* end of msgfile.c */
