/************************************************************************/
/*									*/
/*		formproj.c						*/
/*									*/
/*	Project management routines for FORM				*/
/*									*/
/************************************************************************/


#include "form_local.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_projects;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		fix_path_name();
static	Boolean 	get_real_path_name();
static	void		adjust_path_name();
static	void		load_rc_file();




/************************************************************************/
/*									*/
/*	FORM_proj_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FORM_proj_init()
{
   all_projects = NULL;
};





/************************************************************************/
/*									*/
/*	FORM_proj_setup -- setup project for window			*/
/*									*/
/************************************************************************/


void
FORM_proj_setup(fw)
   FORM_WIN fw;
{
   fw->root = FORM_proj_find(fw->project);
   fw->selection = NULL;

   if (fw->root != NULL) {
      FORM_item_free_contents(fw->root);
      fw->dflt = FORM_back_scan(fw->root);
      if (fw->dflt != NULL) {
	 FORM_item_attr_set(fw->root,"DEFAULT",FORM_attr_value_string(fw->dflt->name));
       };
    };
};






/************************************************************************/
/*									*/
/*	FORM_proj_update -- update dependencies for a project/item	*/
/*									*/
/************************************************************************/


void
FORM_proj_update(itm)
   FORM_ITEM itm;
{
   FORM_ITEM dflt;

   dflt = FORM_back_scan(itm);
   if (itm->is_project && dflt != NULL) {
      FORM_item_attr_set(itm,"DEFAULT",FORM_attr_value_string(dflt->name));
    };
};






/************************************************************************/
/*									*/
/*	FORM_proj_find -- find/create a project for a directory 	*/
/*									*/
/************************************************************************/


FORM_ITEM
FORM_proj_find(name)
   String name;
{
   Character buf[1024];
   FORM_ITEM itm;
   Sequence l;
   String s;

   if (!get_real_path_name(name,buf)) return NULL;

   forin (itm,FORM_ITEM,l,all_projects) {
      s = FORM_item_attr_get(itm,"PATHNAME").string_value;
      if (s == NULL) continue;
      if (STREQL(s,buf)) break;
    };

   if (itm == NULL) {
      itm = FORM_item_define(NULL,name);
      FORM_item_attr_set(itm,"PATHNAME",FORM_attr_value_string(buf));
      load_rc_file(itm,buf);
    };

   return itm;
};






/************************************************************************/
/*									*/
/*	FORM_proj_find_file -- find item given file pathname		*/
/*									*/
/************************************************************************/


FORM_ITEM
FORM_proj_find_file(fnm)
   String fnm;
{
   Character buf[1024],dir[1024];
   String s,t;
   FORM_ITEM proj,itm;
   Sequence l,la;

   if (fnm[0] == '/') strcpy(buf,fnm);
   else sprintf(buf,"%s/%s",MSGinq_wd(),fnm);

   if (get_real_path_name(buf,buf)) return FORM_proj_find(buf);

   adjust_path_name(buf);

   strcpy(dir,buf);
   s = rindex(dir,'/');
   if (s != NULL) *s = 0;
   else strcpy(dir,".");
   get_real_path_name(dir,dir);

   s = rindex(buf,'/');
   if (s == NULL) s = buf;
   else ++s;

   forin (proj,FORM_ITEM,l,all_projects) {
      t = FORM_item_attr_get(proj,"PATHNAME").string_value;
      if (t == NULL) continue;
      if (STREQL(t,dir)) {
	 forin (itm,FORM_ITEM,la,proj->items) {
	    if (STREQL(itm->name,s)) return itm;
	  };
       };
    };

   strcat(dir,"/");
   strcat(dir,s);

   forin (proj,FORM_ITEM,l,all_projects) {
      t = FORM_item_attr_get(proj,"PATHNAME").string_value;
      if (t == NULL) continue;
      forin (itm,FORM_ITEM,la,proj->items) {
	 if (strncmp(itm->name,"../",3) == 0) {
	    sprintf(buf,"%s/%s",t,itm->name);
	    adjust_path_name(buf);
	    if (STREQL(dir,buf)) return itm;
	  };
       };
    };

   s = rindex(dir,'/');
   *s++ = 0;

   proj = FORM_proj_find(dir);
   if (proj->items == NULL) FORM_proj_update(proj);

   itm = FORM_item_define(proj,s);

   return itm;
};





/************************************************************************/
/*									*/
/*	FORM_proj_execute -- execute an item for a given project	*/
/*									*/
/************************************************************************/



Boolean
FORM_proj_execute(proj,what)
   FORM_ITEM proj;
   String what;
{
   Boolean fg;

   fg = FORM_back_exec_start(proj,what);

   return fg;
};






/************************************************************************/
/*									*/
/*	FORM_proj_register -- note that an item is a project		*/
/*									*/
/************************************************************************/


void
FORM_proj_register(proj)
   FORM_ITEM proj;
{
   all_projects = APPEND(proj,all_projects);

   fix_path_name(proj);
};





/************************************************************************/
/*									*/
/*	FORM_proj_remove -- remove project				*/
/*									*/
/************************************************************************/


void
FORM_proj_remove(proj)
   FORM_ITEM proj;
{
   all_projects = REMOB(proj,all_projects);
};






/************************************************************************/
/*									*/
/*	FORM_proj_inq_name -- return array of all project names 	*/
/*									*/
/************************************************************************/


Integer
FORM_proj_inq_names(mx,nams)
   Integer mx;
   String nams[];
{
   Integer ct;
   Sequence l;
   FORM_ITEM itm;

   ct = 0;

   forin (itm,FORM_ITEM,l,all_projects) {
      if (itm->project == NULL) {
	 nams[ct++] = itm->name;
	 if (ct >= mx) break;
       };
    };

   if (ct < mx) {
      forin (itm,FORM_ITEM,l,all_projects) {
	 if (itm->project != NULL) {
	    nams[ct++] = itm->name;
	    if (ct >= mx) break;
	  };
       };
    };

   return ct;
};







/************************************************************************/
/*									*/
/*	fix_path_name -- set up proper path name for a project		*/
/*									*/
/************************************************************************/


static void
fix_path_name(proj)
   FORM_ITEM proj;
{
   String s,t;
   Character buf[1024];

   s = FORM_item_attr_get(proj,"PATHNAME").string_value;
   if (s == NULL) s = proj->name;

   if (s[0] != '/') {
      if (proj->project == NULL) {
	 sprintf(buf,"%s/%s",MSGinq_wd(),s);
       }
      else {
	 t = FORM_item_attr_get(proj->project,"PATHNAME").string_value;
	 sprintf(buf,"%s/%s",t,s);
       };
    }
   else strcpy(buf,s);

   if (!get_real_path_name(buf,buf)) {
      FORM_error("Can't access directory %s",buf);
      return;
    };

   FORM_item_attr_set(proj,"PATHNAME",FORM_attr_value_string(buf));
};






/************************************************************************/
/*									*/
/*	get_real_path_name -- get real path name for a name		*/
/*									*/
/************************************************************************/


static Boolean
get_real_path_name(nm,buf)
   String nm;
   String buf;
{
   Character old[1024];
   struct stat sinfo;

   if (access(nm,5) < 0 || stat(nm,&sinfo) < 0 || (sinfo.st_mode & S_IFDIR) == 0) {
      if (nm != buf) strcpy(buf,nm);
      return FALSE;
    };

   strcpy(old,MSGinq_wd());

   if (chdir(nm) != 0) {
      if (nm != buf) strcpy(buf,nm);
      return FALSE;
    };

   getwd(buf);
   chdir(old);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	adjust_path_name -- handle oddities in file names		*/
/*									*/
/************************************************************************/


static void
adjust_path_name(buf)
   String buf;
{
   String s,t;

   s = buf;
   t = buf;
   while (*s != 0) {
      if (*s != '/') *t++ = *s++;
      else {
	 if (strncmp(s,"/./",3) == 0) s += 2;
	 else if (strncmp(s,"/../",4) == 0) {
	    if (t != buf) while (*t != '/') --t;
	    if (t != buf) --t;
	    s += 3;
	  }
	 else *t++ = *s++;
       };
    };
   *t = 0;
};







/************************************************************************/
/*									*/
/*	load_rc_file -- load definition file for project		*/
/*									*/
/************************************************************************/


static void
load_rc_file(proj,dir)
   FORM_ITEM proj;
   String dir;
{
   Character buf[1024],val[1024],attr[1024];
   FILE * inf;
   FORM_ATTR_ID fa;
   Boolean fg;
   Integer i;
   Float f;

   sprintf(buf,"%s/.formrc",dir);
   inf = fopen(buf,"r");
   if (inf == NULL) return;

   while (fgets(buf,1024,inf) != NULL) {
      if (sscanf(buf,"%s = %s",attr,val) == 2) {
	 fa = FORM_attr_find_id(attr);
	 if (fa != NULL) {
	    switch (fa->type->aclass) {
	       case FORM_ATTR_CLASS_BOOLEAN :
		  fg = (index("tT1",val[0]) != NULL);
		  FORM_item_attr_set(proj,attr,FORM_attr_value_boolean(fg));
		  break;
	       case FORM_ATTR_CLASS_INT :
		  i = atol(val);
		  FORM_item_attr_set(proj,attr,FORM_attr_value_int(i));
		  break;
	       case FORM_ATTR_CLASS_STRING :
		  FORM_item_attr_set(proj,attr,FORM_attr_value_string(val));
		  break;
	       case FORM_ATTR_CLASS_FLOAT :
		  f = atof(val);
		  FORM_item_attr_set(proj,attr,FORM_attr_value_float(f));
		  break;
	       case FORM_ATTR_CLASS_ENUM :
		  FORM_item_attr_set(proj,attr,FORM_attr_value_enum(fa->type,val));
		  break;
	     };
	  };
       }
    };

   fclose(inf);
};





/* end of formproj.c */
