/************************************************************************/
/*									*/
/*		formmenu.c						*/
/*									*/
/*	Menu and button routines for FORM configuration management	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "form_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_INFO_ARC		32
#define MAX_ITEMS		256





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	int		form_menu_form();
static	int		form_menu_display();
static	int		form_menu_layout();
static	int		form_menu_transwin();
static	int		form_menu_make();
static	int		str_compare();
static	Integer 	form_make_wait_fct();
static	int		form_menu_select();
static	int		form_btn_handler();
static	int		form_item_handler();
static	int		form_link_handler();
static	void		form_reset_btns();
static	Boolean 	show_item_info();
static	int		item_compare();
static	void		form_display_finish();





/************************************************************************/
/*									*/
/*	Tables -- menu definitions					*/
/*									*/
/************************************************************************/


static STEM_PDM_DATA	form_menus[] = {
   { STEM_PSTATE_MENU,	"Form",         NULL },
      { STEM_PSTATE_BTN,   "Transcript",  form_menu_transwin },
      { STEM_PSTATE_BTN,   "Restart",      form_menu_form },
      { STEM_PSTATE_BTN,   "Reset",        form_menu_form },
      { STEM_PSTATE_BTN,   "Update",       form_menu_form },
      { STEM_PSTATE_BTN,   "Set Project",   form_menu_form },
      { STEM_PSTATE_BTN,   "Quit",         form_menu_form },

   { STEM_PSTATE_MENU,	"Make",         NULL },
      { STEM_PSTATE_BTN,   "Make Current", form_menu_make },
      { STEM_PSTATE_BTN,   "Make Default", form_menu_make },
      { STEM_PSTATE_BTN,   "Make ...",     form_menu_make },

   { STEM_PSTATE_MENU, "Selection",      NULL },
      { STEM_PSTATE_BTN,   "Set Selection", form_menu_select },
      { STEM_PSTATE_BTN,   "Clear Selection",  form_menu_select },
      { STEM_PSTATE_BTN,   "Select Item",   form_menu_select },
      { STEM_PSTATE_BTN,   "Item Patterns", form_menu_select },

   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX,  "Display",      NULL },
      { STEM_PSTATE_BTN,   "Options",      form_menu_display },
      { STEM_PSTATE_BTN,   "Show All",     form_menu_display },
      { STEM_PSTATE_BTN,   "Show Dependencies", form_menu_display },
      { STEM_PSTATE_BTN,   "Show Uses",    form_menu_display },
      { STEM_PSTATE_BTN,   "Layout",       form_menu_layout },

   { STEM_PSTATE_END }
};






/************************************************************************/
/*									*/
/*	FORM_menu_init -- module initialization 		       */
/*									*/
/************************************************************************/


void
FORM_menu_init()
{
};





/************************************************************************/
/*									*/
/*	FORM_menu_setup -- setup buttons				*/
/*									*/
/************************************************************************/


void
FORM_menu_setup(fw)
   FORM_WIN fw;
{
   if (fw->menu_win != NULL) {
      STEMpdm_define(fw->menu_win,fw,form_menus);
    };

   fw->region = RIPdefine_region(fw->disp_win,0,0,0,0,RIP_ALL_CHARS,
				    RIP_BTN_ANY_EITHER,
				    form_btn_handler,
				    ASH_SENSE_NO_CHANGE);
   RIPuse_local_window(fw->region);
   RIPset_data(fw->region,fw);

   form_reset_btns(fw);
};






/************************************************************************/
/*									*/
/*	FORM_request_project -- request project name from user		*/
/*									*/
/************************************************************************/


String
FORM_request_project(w)
   ASH_WINDOW w;
{
   String projs[32];
   register Integer i,ct,sct;
   Character btn[32][64];
   Character nbuf[64];
   String menu[40];
   Integer val;
   String name;

   sct = FORM_proj_inq_names(32,projs);
   if (sct == 1) return projs[0];

   ct = 0;
   menu[ct++] = "%CFORM Configuration Management\n";

   menu[ct++] = "Project: %1.32.63t\n";

   for (i = 0; i < sct; ++i) {
      sprintf(btn[i],"    %%0.%do %s",i,projs[i]);
      menu[ct++] = btn[i];
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   val = 0;
   nbuf[0] = 0;

   if (!STEMdialog(w,menu,&val,nbuf)) return NULL;

   if (nbuf[0] != 0 && nbuf[0] != ' ') {
      name = SALLOC(nbuf);
    }
   else if (val < sct) name = projs[val];
   else name = NULL;

   return name;
};





/************************************************************************/
/*									*/
/*	form_menu_form -- handle form menu options			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_form(fw,menu,btn)
   FORM_WIN fw;
   String menu;
   String btn;
{
   String cls;
   Character buf[1024];

   if (STREQL(btn,"Restart")) {
      FORM_disp_reset(fw,TRUE);
    }
   else if (STREQL(btn,"Reset")) {
      FORM_disp_reset(fw,FALSE);
    }
   else if (STREQL(btn,"Update")) {
      ASHinput_lock_makelong(fw->window);
      FORM_proj_update(fw->root);
      FORM_disp_reset(fw,TRUE);
    }
   else if (STREQL(btn,"Set Project")) {
      cls = FORM_request_project(fw->window);
      if (cls == NULL || cls[0] == 0 || STREQL(cls,"*")) return FALSE;
      fw->project = SALLOC(cls);
      ASHinput_lock_makelong(fw->window);
      sprintf(buf,"formview: %s",cls);
      ASHset_window_name(fw->window,buf);
      FORM_disp_setup(fw);
    }
   else if (STREQL(btn,"Quit")) {
      ASHremove(fw->window);
      return TRUE;
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	form_menu_display -- handle display commands			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_display(fw,mnm,bnm)
   FORM_WIN fw;
   String mnm;
   String bnm;
{
   String menu[10240],buf[1024];
   Integer v0,v1,v2,v3,v4,v5,v6,v7;

   if (STREQL(bnm,"Show All")) {
      fw->display_all = !fw->display_all;
    }
   else if (STREQL(bnm,"Show Dependencies")) {
      fw->display_inlinks = !fw->display_inlinks;
    }
   else if (STREQL(bnm,"Show Uses")) {
      fw->display_outlinks = !fw->display_outlinks;
    }
   else {
      strcpy(menu,"%CFORM Display Options\n\n");

      strcat(menu,"%0.1o Show complete dependency graph\n");
      strcat(menu,"%0.0o Show graph around selection only\n\n");
      v0 = fw->display_all;

      strcat(menu,"%1o Force redisplay on selection\n\n");
      v1 = fw->display_force;

      strcat(menu,"%2o Show items current depends upon\n");
      strcat(menu,"%3o Show items depending on current\n");
      v2 = fw->display_inlinks;
      v3 = fw->display_outlinks;

      sprintf(buf,"      Levels: %%4.%uo All   %%4.0o None   %%4.1o Single   %%4.6d\n\n",-1);
      strcat(menu,buf);
      v4 = fw->display_levels;

      strcat(menu,"Show selection:\n");

      strcat(menu,"   %5o Expand size to show text\n\n");
      v5 = fw->display_fixcur;

      strcat(menu, "  Zoom selection:  %6.0o Normal   %6.2o Emphasize   Factor: %6.6d\n\n");
      v6 = fw->display_zoom;

      strcat(menu,"   %a%M   %c");

      if (!STEMdialog1(fw->window,menu,&v0,&v1,&v2,&v3,&v4,&v5,&v6,&v7))
	 return FALSE;

      if (v6 <= 0) v6 = 1;

      fw->display_all = v0;
      fw->display_force = v1;
      fw->display_inlinks = v2;
      fw->display_outlinks = v3;
      fw->display_levels = v4;
      fw->display_fixcur = v5;
      fw->display_zoom = v6;
    };

   FORM_disp_show(fw);

   form_reset_btns(fw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	form_menu_layout -- handle display layout menu			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_layout(fw,mnm,btn)
   FORM_WIN fw;
   String mnm;
   String btn;
{
   Integer fix,std,cen,mthd,cmthd,white;

   fix = fw->fixed;
   std = fw->standard;
   cen = fw->centered;

   if (!GELOrequest_layout_methods(fw->window,&fw->method,&fw->connmethod,
				      &fix,&std,&cen,&fw->whitespace))
      return FALSE;

   fw->fixed = fix;
   fw->standard = std;
   fw->centered = cen;

   FORM_disp_show(fw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	form_menu_transwin -- handle requests for information window	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_transwin(fw,mnm,bnm)
   FORM_WIN fw;
   String mnm;
   String bnm;
{
   ASH_WINDOW w;

   if (fw->text_win == NULL) {
      FORM_trans_setup(fw,TRUE);
    }
   else {
      for (w = fw->text_win; ASHinq_parent(w) != NULL; w = ASHinq_parent(w));
      ASHvisible(w,TRUE);
      ASHpop(w);
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	form_menu_make -- handle Make menu options			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_make(fw,mnm,bnm)
   FORM_WIN fw;
   String mnm;
   String bnm;
{
   FORM_ITEM itm;
   String itms[MAX_ITEMS+1];
   Integer ct,idx,dflt;
   STEM_DIALOG_LIST sdl;
   Character menu[10240],buf[1024];
   String s,t;
   Sequence l;
   Integer wd,ht;

   if (fw->root == NULL) return FALSE;
   else if (STREQL(bnm,"Make Current")) {
      itm = fw->selection;
      if (itm == NULL) itm = fw->root;
    }
   else if (STREQL(bnm,"Make Default")) {
      itm = fw->root;
    }
   else {
      itm = fw->root;
      s = FORM_item_attr_get(fw->root,"DEFAULT").string_value;
      ct = 0;
      idx = -1;
      forin (itm,FORM_ITEM,l,fw->root->items) {
	 if (itm->in_links != NULL) {
	    if (s != NULL && STREQL(itm->name,s)) idx = ct;
	    itms[ct++] = itm->name;
	    if (ct >= MAX_ITEMS-1) break;
	  }
       }

      itms[ct] = NULL;

      qsort(itms,ct,sizeof(String),str_compare);

      strcpy(menu,"%CMake ...\n\n");
      if (s != NULL) {
	 sprintf(buf,"%%0o %s (Default)\n\n",s);
	 strcat(menu,buf);
       };
      strcat(menu,"Target: %2.40.256t\n\n");

      if (ct > 0) {
	  if (ct <= 4) {
	    wd = 32;
	    ht = ct*2+1;
	  }
	 else if (ct <= 8) {
	    wd = 48;
	    ht = 9;
	  }
	 else if (ct <= 16) {
	    wd = 48;
	    ht = (ct+1)/2;
	    ht = ht*2+1;
	  }
	 else {
	    wd = 48;
	    ht = 16;
	  };
	 sprintf(buf,"    %%1.%d.%dl   \n\n",wd,ht);
	 strcat(menu,buf);
      };

      strcat(menu,"   %a%M   %c");
      sdl.btns = itms;
      sdl.choice = idx;
      dflt = FALSE;
      buf[0] = 0;
      if (!STEMdialog1(fw->window,menu,&dflt,&sdl,buf)) return FALSE;
      idx = sdl.choice;
      for (s = buf; isspace(*s); ++s);
      if (*s != 0) itm = FORM_item_define(fw->root,s);
      else if (idx < 0 || dflt) itm = fw->root;
      else itm = FORM_item_define(fw->root,itms[idx]);
    };

   ASHinput_lock_makelong(fw->window);

   ASHfix_damage(fw->window);
   ASHfix_damage(fw->menu_win);
   ASHfix_damage(fw->disp_win);

   if (itm->is_project) {
      sprintf(menu,"\n Making <DEFAULT> in %s \n\n ",itm->name);
      sprintf(buf,"BUILD COMMAND %s *",itm->name);
      t = "<DEFAULT>";
    }
   else {
      sprintf(menu,"\n Making %s in %s \n\n ",itm->name,itm->project->name);
      sprintf(buf,"BUILD COMMAND %s %s",itm->project->name,itm->name);
      t = itm->name;
    };

   fw->exec_btn = buf;
   STEMdialog1_wait(fw->window,form_make_wait_fct,menu);
   s = fw->exec_btn;
   fw->exec_btn = NULL;

   FORM_proj_update(itm);

   form_display_finish(fw,t,s);

   FORM_disp_show(fw);

   return TRUE;
};






static int
str_compare(s1,s2)
   String * s1;
   String * s2;
{
   return strcmp(*s1,*s2);
};





static Integer
form_make_wait_fct(w)
   ASH_WINDOW w;
{
   FORM_WIN fw;

   fw = FORM_find_window(w);
   if (fw == NULL || fw->exec_btn == NULL) return -1;

   fw->exec_btn = MSGcall(fw->exec_btn);

   return 1;
};





/************************************************************************/
/*									*/
/*	form_menu_select -- handle selection menu buttons		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_menu_select(fw,mnm,bnm)
   FORM_WIN fw;
   String mnm;
   String bnm;
{
   Character menu[1024],bufa[1024],bufb[1024];
   FORM_ITEM itm,proj;
   Integer exfg,infg;

   if (STREQL(bnm,"Set Selection")) {
      strcpy(menu,"%CFORM Dependency Selection\n\n");
      strcat(menu,"Show Item: %0.48t\n\n");
      strcat(menu,"Project:   %1.48t\n\n");
      strcat(menu,"   %a%M   %c");

      if (fw->selection == NULL) {
	 bufa[0] = 0;
	 if (fw->root != NULL) strcpy(bufb,fw->root->name);
	 else bufb[0] = 0;
       }
      else if (fw->selection->project != NULL) {
	 strcpy(bufa,fw->selection->name);
	 strcpy(bufb,fw->selection->project->name);
       }
      else {
	 bufa[0] = 0;
	 strcpy(bufb,fw->selection->name);
       };

      itm = NULL;
      while (itm == NULL) {
	 if (!STEMdialog1(fw->window,menu,bufa,bufb)) return FALSE;
	 if (bufb[0] == NULL) proj = fw->root;
	 else proj = FORM_proj_find(bufb);
	 if (bufa[0] == 0) itm = proj;
	 else itm = FORM_item_define(proj,bufa);
       };
      FORM_disp_select(fw,itm);
    }
   else if (STREQL(bnm,"Clear Selection")) {
      FORM_disp_select(fw,NULL);
    }
   else if (STREQL(bnm,"Select Item")) {
      proj_item_select(fw,fw->root,0);
      FORM_disp_show(fw);
    }
   else if (STREQL(bnm,"Item Patterns")) {
      strcpy(menu,"%CRegular Expression Item Selection\n\n");
      strcat(menu,"Include items: %0.48.128t\n");
      strcat(menu,"   %2o Exclude all others\n\n");
      strcat(menu,"Exclude items: %1.48.128t\n");
      strcat(menu,"   %3o Include all others\n\n");
      strcat(menu,"   %a%M   %c");

      bufa[0] = 0;
      bufb[0] = 0;
      exfg = FALSE;
      infg = FALSE;
      if (!STEMdialog1(fw->window,menu,bufa,bufb,&exfg,&infg)) return FALSE;

      FORM_disp_regex(fw,bufa,exfg,bufb,infg);
      FORM_disp_show(fw);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	form_btn_handler -- handle hits in form window			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
form_btn_handler(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   FORM_WIN fw;
   GELO_OBJECT go;
   Boolean fg;
   static Integer xtrinfo;
   static GELO_OBJECT downobj = NULL;

   fw = (FORM_WIN) RIPinq_data(rgn);
   if (fw == NULL || fw->window == NULL) return FALSE;

   if (btn & RIP_BTN_NONE) {
      xtrinfo = 0;
      downobj = NULL;
      switch (ch) {
	 case 'r' :
	    FORM_disp_reset(fw,FALSE);
	    break;
	 case 'R' :
	    FORM_disp_reset(fw,TRUE);
	    break;
	 case 'a' :
	 case 'A' :
	    fw->display_all = !fw->display_all;
	    FORM_disp_show(fw);
	    break;
	 case 'c' :
	 case 'C' :
	    FORM_disp_select(fw,NULL);
	    break;
	 default :
	    return FALSE;
       };

      return TRUE;
    };

   if (btn & RIP_BTN_TAP) {
      xtrinfo = 0;
      downobj = NULL;
    }
   else if (btn & RIP_BTN_DOWN) {
      downobj = GELOcorrelate(fw->disp_win,x,y);
      xtrinfo = btn & RIP_BTN_EXTRA;
      return FALSE;
    }
   else if (btn & RIP_BTN_UP) {
      btn |= xtrinfo;
    };

   go = GELOcorrelate(fw->disp_win,x,y);

   if (go == NULL || (downobj != NULL && downobj != go)) return FALSE;

   if (GELOinq_contents(go) != 0) {
      fg = form_link_handler(fw,go,btn);
    }
   else {
      fg = form_item_handler(fw,go,btn);
    };

   return fg;
};






/************************************************************************/
/*									*/
/*	form_item_handler -- handle hits on item			*/
/*									*/
/************************************************************************/


static int
form_item_handler(fw,go,btn)
   FORM_WIN fw;
   GELO_OBJECT go;
   Integer btn;
{
   FORM_ITEM itm;
   Boolean fg;

   itm = (FORM_ITEM) GELOinq_user_structure(go);
   if (itm == NULL) return FALSE;

   fg = (fw->selection == itm);

   if ((btn & RIP_BTN_EXTRA) != 0 && (btn & RIP_BTN_LEFT) != 0) {
      FORM_disp_ignore(fw,itm);
      FORM_disp_show(fw);
    }
   else if (fg && (btn & RIP_BTN_EXTRA) != 0 && (btn & RIP_BTN_RIGHT) != 0) {
    }
   else {
      if ((btn & RIP_BTN_MID) != 0) {
	 show_item_info(fw,itm);
       }
      else if (fg) {
	 if (btn & RIP_BTN_LEFT) {
	    /* display_node(fw,fn);	*/
	  }
	 else {
	    FORM_disp_select(fw,NULL);
	  };
       }
      else if (btn & RIP_BTN_LEFT) {
	 ASHinput_lock_makelong(fw->window);
	 FORM_win_select(fw,itm,TRUE);
       }
      else {
	 FORM_win_select(fw,itm,FALSE);
       };
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	form_link_handler -- handle hits on links			*/
/*									*/
/************************************************************************/


static int
form_link_handler(fw,go,btn)
   FORM_WIN fw;
   GELO_OBJECT go;
   Integer btn;
{
   FORM_LINK lnk;

   lnk = (FORM_LINK) GELOinq_user_structure(go);
   if (lnk == NULL) return FALSE;

   if (btn & RIP_BTN_MID) {
      /* arc_info(fw,lnk); */
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	form_reset_btns -- handle setting conditions on menu btns	*/
/*									*/
/************************************************************************/


static void
form_reset_btns(fw)
   FORM_WIN fw;
{
   STEMpdm_btn_select(fw->menu_win,"Display","Show All",fw->display_all);
   STEMpdm_btn_select(fw->menu_win,"Display","Show Dependencies",fw->display_inlinks);
   STEMpdm_btn_select(fw->menu_win,"Display","Show Uses",fw->display_outlinks);
};





/************************************************************************/
/*									*/
/*	show_item_info -- show information about an item		*/
/*									*/
/************************************************************************/


static Boolean
show_item_info(fw,itm)
   FORM_WIN fw;
   FORM_ITEM itm;
{
   Character menu[20480];
   Character buf[1024];
   String s;
   FORM_LINK lnk;
   FORM_ATTR attr;
   Sequence l;
   FORM_ITEM sel;
   Integer i;

   if (itm == NULL) return FALSE;

   if (itm->is_project) {
      sprintf(menu,"%%CProject %s\n\n",itm->name);
    }
   else if (itm->in_links == NULL) {
      sprintf(menu,"%%CSource %s\n\n",itm->name);
    }
   else if (itm->out_links == NULL) {
      sprintf(menu,"%%CTarget %s\n\n",itm->name);
    }
   else {
      sprintf(menu,"%%CItem %s\n\n",itm->name);
    };

   if (itm->project != NULL) {
      sprintf(buf,"Project:   %s  \n",itm->project->name);
      strcat(menu,buf);
      s = FORM_item_attr_get(itm->project,"PATHNAME").string_value;
      if (s != NULL && STRNEQ(s,itm->project->name)) {
	 sprintf(buf,"Directory: %s  \n",s);
       };
      strcat(menu,"\n");
    };

   if (itm->in_links != NULL) {
      strcat(menu,"Dependent On:");
      i = 0;
      forin (lnk,FORM_LINK,l,itm->in_links) {
	 if (i == 0) strcat(menu,"\n   ");
	 else strcat(menu,"%M");
	 s = (FORM_link_attr_get(lnk,"OUTOFDATE").bool_value ? "+" : "");
	 sprintf(buf,"%%0.%d\"%s\"%sb",lnk->from,lnk->from->name,s);
	 strcat(menu,buf);
	 i = (i+1)%3;
       };
      strcat(menu,"\n\n");
    };

   if (itm->out_links != NULL) {
      strcat(menu,"Used by:");
      i = 0;
      forin (lnk,FORM_LINK,l,itm->out_links) {
	 if (i == 0) strcat(menu,"\n   ");
	 else strcat(menu,"%M");
	 s = (FORM_link_attr_get(lnk,"OUTOFDATE").bool_value ? "+" : "");
	 sprintf(buf,"%%0.%d\"%s\"%sb",lnk->to,lnk->to->name,s);
	 strcat(menu,buf);
	 i = (i+1)%3;
       };
      strcat(menu,"\n\n");
    };

   /* display attrs */

   strcat(menu,"\n      %a%M%M   %c");

   sel = NULL;
   if (!STEMdialog1(fw->window,menu,&sel)) return FALSE;

   if (sel != NULL) FORM_disp_select(fw,sel);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	proj_item_select -- select items in project			*/
/*									*/
/************************************************************************/


#define MAX_SELECT		32



static Integer
proj_item_select(fw,proj,idx)
   FORM_WIN fw;
   FORM_ITEM proj;
{
   Character menu[20480],buf[1024];
   Integer i,j,k,st,ct,oct;
   FORM_ITEM itm,fi;
   Integer ignfg[(MAX_SELECT+16)/16],cur,show;
   FORM_ITEM * itms,* oths;
   Sequence l,others;
   FORM_LINK lnk;

   ct = LENGTH(proj->items);
   itms = (FORM_ITEM *) alloca(sizeof(FORM_ITEM)*(ct+2));
   ct = 0;
   forin (itm,FORM_ITEM,l,proj->items) itms[ct++] = itm;
   qsort(itms,ct,sizeof(FORM_ITEM),item_compare);

   st = idx;
   if (st < 0) st = 0;

   if (ct == 0) return 0;

   strcpy(menu,"%CItem Selection\n\n");
   sprintf(buf,"Project %s\n\n",proj->name);
   strcat(menu,buf);
   strcat(menu,"Ignore   Item%MIgnore   Item");

   for (i = 0; i < (MAX_SELECT+16)/16; ++i) ignfg[i] = 0;
   cur = -1;

   others = NULL;
   for (i = 0; i < ct; ++i) {
      itm = itms[i];
      if (itm->is_project) {
	 if (!MEMQ(itm,others)) others = CONS(itm,others);
	 continue;
       };
      forin (lnk,FORM_LINK,l,itm->in_links) {
	 fi = lnk->from;
	 if (fi->project != itm->project) {
	    if (!MEMQ(fi->project,others)) others = CONS(fi->project,others);
	  };
       };
      forin (lnk,FORM_LINK,l,itm->out_links) {
	 fi = lnk->to;
	 if (fi->project != itm->project) {
	    if (!MEMQ(fi->project,others)) others = CONS(fi->project,others);
	  };
       };
      if (i < st || i-st >= MAX_SELECT) continue;

      j = (i-st) % 16;
      k = (i-st) / 16;
      if (i & 1) strcat(menu,"%M  ");
      else strcat(menu,"\n  ");
      sprintf(buf,"%%%d.%df   %%0.%d\"%s\"b",k+2,(1 << j),i,itm->name);
      strcat(menu,buf);
      if (MEMQ(itm,fw->ignore))ignfg[k] |= (1 << j);
      if (fw->selection == itm) cur = i;
    };

   strcat(menu,"\n\n");
   show = 0;
   if (st != 0 || i < ct) {
      if (st != 0) strcat(menu,"   %1.1\"Previous Items\"a");
      if (i < ct) strcat(menu,"%M%1.2\"More Items\"a");
      strcat(menu,"\n\n");
    };

   if (others != NULL) {
      oths = (FORM_ITEM *) alloca((LENGTH(others)+2)*sizeof(FORM_ITEM));
      oct = 0;
      forin (fi,FORM_ITEM,l,others) oths[oct++] = fi;
      qsort(oths,oct,sizeof(FORM_ITEM),item_compare);
      LFREE(others);
      strcat(menu,"Related Projects:\n");
      for (i = 0; i < oct; ++i) {
	 sprintf(buf,"   %%1.%d\"%s\"a\n",i+3,oths[i]->name);
	 strcat(menu,buf);
       };
      strcat(menu,"\n");
    };

   strcat(menu,"      %a%M   %c");

   if (!STEMdialog1(fw->window,menu,&cur,&show,&ignfg[0],&ignfg[1],&ignfg[2],&ignfg[3]))
      return 0;

   for (i = st; i < ct && i-st < MAX_SELECT; ++i) {
      itm = itms[i];
      j = (i-st) % 16;
      k = (i-st) / 16;
      if ((ignfg[k] & (1 << j)) != 0) FORM_disp_ignore(fw,itm);
      else FORM_disp_unignore(fw,itm);
    };
   if (cur >= 0) fw->selection = itms[cur];

   if (show != 0) {
      itm = proj;
      if (show == 1) {
	 st -= MAX_SELECT;
	 if (st < 0) st = 0;
       }
      else if (show == 2) {
	 st += MAX_SELECT;
	 if (st >= ct) st = ct-2;
       }
      else {
	 itm = oths[show-3];
	 st = 0;
       };

      i = proj_item_select(fw,itm,st);
      while (i < 0) {
	 i = proj_item_select(fw,proj,idx);
       };

      if (i > 0) return 1;
    };

   return 1;
};





static int
item_compare(ita,itb)
   FORM_ITEM * ita, * itb;
{
   return strcmp((*ita)->name,(*itb)->name);
};





/********************************************************************************/
/*										*/
/*	form_display_finish -- show that a command has finished 		*/
/*										*/
/********************************************************************************/


static void
form_display_finish(fw,what,rslt)
   FORM_WIN fw;
   String what;
   String rslt;
{
   String menu[10];
   Integer ct,inf,ln,lct,cct;
   Character buf[256],fbuf[10241],ffbf[10240];
   String s,t;

   ct = 0;
   if (rslt == NULL || *rslt == 0 || STREQL(rslt,"*") || (inf = open(rslt,0)) < 0) {
      sprintf(buf," Make of %s has finished ",what);
      menu[ct++] = buf;
    }
   else {
      sprintf(buf,"%%CResult of making %s\n",what);
      menu[ct++] = buf;
      ln = read(inf,fbuf,10240);
      fbuf[ln] = 0;
      s = fbuf;
      t = ffbf;
      lct = 0;
      cct = 0;
      while (*s != 0) {
	 if (cct < 80 || *s == '\n') {
	    *t++ = *s;
	    if (*s == '%') *t++ = '%';
	    ++cct;
	  };
	 if (*s == '\n') {
	    cct = 0;
	    if (++lct >= 10) break;
	  };
	 ++s;
       };
      *t = 0;
      if (*s != 0) strcat(ffbf," ...\n");
      menu[ct++] = ffbf;
      close(inf);
      unlink(rslt);
    };

   menu[ct++] = "\n%M   %a";
   menu[ct] = 0;

   STEMdialog(fw->window,menu);
};





/* end of formmenu.c */
