/************************************************************************/
/*									*/
/*		formake.c						*/
/*									*/
/*	FORM interface to GNUMAKE, Sun make, Dec3100 make		*/
/*									*/
/************************************************************************/


#include "form_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define GNUMAKE_PATH	"%s/bin/gnumake"
#define SUNMAKE_PATH	"/bin/make"
#define DECMAKE_PATH	"/bin/make"




/************************************************************************/
/*									*/
/*	Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _MAKE_DATA *	MAKE_DATA;
typedef struct _MAKE_STACK *	MAKE_STACK;
typedef struct _MAKE_PSTACK *	MAKE_PSTACK;


typedef struct _MAKE_PAT {
   String text;
   voidFunction_Ptr rtn;
   String param;
   PMAT_PAT pattern;
} MAKE_PAT;





typedef struct _MAKE_DATA {
   FORM_ITEM project;
   Boolean handle_rule;
   Boolean notarget;
   Boolean infiles;
   FORM_ITEM last;
   FORM_ITEM lastdef;
   MAKE_STACK stack;
   MAKE_PSTACK pstack;
   Integer level;
   Integer nspace;
   MAKE_PAT * pat_table;
} MAKE_DATA_INFO;




typedef struct _MAKE_STACK {
   FORM_ITEM item;
   MAKE_STACK next;
   Boolean implicit;
} MAKE_STACK_INFO;




typedef struct _MAKE_PSTACK {
   FORM_ITEM proj;
   FORM_ITEM last;
   MAKE_PSTACK next;
} MAKE_PSTACK_INFO;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		make_scan();
static	void		make_scan_line();

static	void		gnuscan_def();
static	void		gnuscan_notarget();
static	void		gnuscan_defflags();
static	void		gnuscan_setrules();
static	void		gnuscan_files();
static	void		gnuscan_consider();
static	void		gnuscan_done();
static	void		gnuscan_flags();
static	void		gnuscan_dep();
static	void		gnuscan_proj();

static	void		makescan_def();
static	void		sunscan_consider();
static	void		sunscan_done();
static	void		sunscan_flags();
static	void		sunscan_dep();

static	void		decscan_def();
static	void		decscan_consider();
static	void		decscan_done();
static	void		decscan_flags();
static	void		decscan_dep();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MAX_ARGS	3


static	MAKE_PAT	 gnu_pattern_table[] = {
   { "Considering target file `%1s'",                   gnuscan_consider, NULL, NULL },
   { "Found an implicit rule for `%1s'",                gnuscan_flags, "I", NULL },
   { "Finished dependencies of target file `%1s'",      gnuscan_done, NULL, NULL },
   { "File `%1s' was considered already",               gnuscan_done, NULL, NULL },
   { "Dependency `%1s' is older than dependent `%2s'",  gnuscan_dep, "O", NULL },
   { "Dependency `%1s' is newer than dependent `%2s'",  gnuscan_dep, "N", NULL },
   { "Must remake target `%1s'",                        gnuscan_flags, "N", NULL },
   { "No need to remake target `%1s'",                  gnuscan_flags, "O", NULL },

   { "# Files",                                         gnuscan_files, "1", NULL },
   { "# Not a target:",                                 gnuscan_notarget, NULL, NULL },
   { "#  commands to execute",                          gnuscan_setrules, NULL, NULL },
   { "#  Phony target",                                 gnuscan_defflags, "X", NULL },
   { "#  Precious file",                                gnuscan_defflags, "P", NULL },

   { "# VPATH Search Paths",                            gnuscan_files, "0", NULL },

   { "# gnumake: Entering directory `%1s'",             gnuscan_proj, "E0", NULL },
   { "# gnumake: Leaving directory `%1s'",              gnuscan_proj, "L0", NULL },
   { "# gnumake[%d]: Entering directory `%1s'",         gnuscan_proj, "E1", NULL },
   { "# gnumake[%d]: Leaving directory `%1s'",          gnuscan_proj, "L1", NULL },

   { "# %r",                                            NULL, NULL, NULL },
   { "%1s = %2r",                                       NULL, NULL, NULL },
   { "%1s: %2r",                                        gnuscan_def, NULL, NULL },
   { "%1s:",                                            gnuscan_def, NULL, NULL },

   { NULL, NULL, NULL, NULL }
};




static	MAKE_PAT	sun_pattern_table[] = {
   { "doname(%1s)",                                     sunscan_consider, NULL, NULL },
   { "Found %1s",                                       sunscan_flags, "I", NULL },
   { "Building %2s because it is out of date relative to %1s",
							sunscan_dep, "N", NULL },
   { "Building %2s using suffix rule for %s because it is out of date relative to %1s",
							sunscan_dep, "N", NULL },
   { "@= %1s",                                          sunscan_flags, "N", NULL },

   { "Date(%s)=",                                       NULL, NULL, NULL },
   { "MAKEFLAGS value:",                                NULL, NULL, NULL },
   { "%1s =",                                           NULL, NULL, NULL },
   { "%1s= %2r",                                        NULL, NULL, NULL },

   { "%1s:\t%2r",                                       makescan_def, NULL, NULL },
   { "%1s:",                                            makescan_def, NULL, NULL },

   { NULL, NULL, NULL, NULL }
};




static	MAKE_PAT	dec_pattern_table[] = {
   { "doname(%1s,%2d)",                                 decscan_consider, NULL, NULL },
   { "setvar: @ = %1s noreset",                         decscan_flags, "@", NULL },
   { "setvar: ? = %1r",                                 decscan_flags, "?", NULL },
   { "look for explicit deps. %1d",                     decscan_flags, "E", NULL },
   { "look for implicit rules. %1d",                    decscan_flags, "I", NULL },

   { "Warning: %s changed",                             NULL, NULL, NULL },
   { "setvar: %1s =",                                   NULL, NULL, NULL },
   { "%1s = %2r",                                       NULL, NULL, NULL },
   { "Rules:",                                          NULL, NULL, NULL },

   { "%1s: %2r",                                        makescan_def, NULL, NULL },
   { "%1s:",                                            makescan_def, NULL, NULL },

   { NULL, NULL, NULL, NULL }
};




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	String		gnumake_path;
static	String		sunmake_path;
static	String		decmake_path;





/************************************************************************/
/*									*/
/*	FORM_gnumake_init -- gnu make initialization			*/
/*	FORM_gnumake_exec_build -- execute command in a makefile	*/
/*	FORM_gnumake_exec_scan -- start scanning makefile info		*/
/*	FORM_gnumake_scan -- scan a line of input			*/
/*									*/
/************************************************************************/


void
FORM_gnumake_init(db)
   AUXD db;
{
   String s;
   Character buf[1024];
   Integer i;

   s = getenv("GNUMAKE");
   if (s == NULL) s = AUXDget_info(db,"PATH");
   if (s == NULL) s = GNUMAKE_PATH;
   sprintf(buf,s,BWElocal_project(),BWEarch());
   gnumake_path = SALLOC(buf);

   for (i = 0; gnu_pattern_table[i].text != NULL; ++i) {
      gnu_pattern_table[i].pattern = PMATmake_pattern(gnu_pattern_table[i].text,MAX_ARGS,NULL);
    };
};





void
FORM_gnumake_exec_build(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "gnumake";

   if (dir != NULL) {
      argv[argc++] = "-C";
      argv[argc++] = dir;
    };

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   execv(gnumake_path,argv);
};





void
FORM_gnumake_exec_scan(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "gnumake";

   if (dir != NULL) {
      argv[argc++] = "-C";
      argv[argc++] = dir;
    };

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   if (itm == NULL) argv[argc++] = "-p";
   argv[argc++] = "-d";

   argv[argc++] = "-n";
   argv[argc++] = "-k";
   argv[argc++] = "-w";

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   execv(gnumake_path,argv);
};





void
FORM_gnumake_scan(fs,text)
   FORM_SCAN fs;
   String text;
{
   make_scan(fs,text,gnu_pattern_table);
};





/************************************************************************/
/*									*/
/*	FORM_sunmake_init -- sun make initialization			*/
/*	FORM_sunmake_exec_build -- execute command in a makefile	*/
/*	FORM_sunmake_exec_scan -- start scanning makefile info		*/
/*	FORM_sunmake_scan -- scan a line of input			*/
/*									*/
/************************************************************************/


void
FORM_sunmake_init(db)
   AUXD db;
{
   String s;
   Character buf[1024];
   Integer i;

   s = getenv("MAKE");
   if (s == NULL) s = AUXDget_info(db,"PATH");
   if (s == NULL) s = SUNMAKE_PATH;
   sprintf(buf,s,BWElocal_project(),BWEarch());
   sunmake_path = SALLOC(buf);

   for (i = 0; sun_pattern_table[i].text != NULL; ++i) {
      sun_pattern_table[i].pattern = PMATmake_pattern(sun_pattern_table[i].text,MAX_ARGS,NULL);
    };
};





void
FORM_sunmake_exec_build(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "make";

   if (dir != NULL) chdir(dir);

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   execv(sunmake_path,argv);
};





void
FORM_sunmake_exec_scan(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "make";

   if (dir != NULL) chdir(dir);

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   if (itm == NULL) argv[argc++] = "-D";
   argv[argc++] = "-dd";

   argv[argc++] = "-n";
   argv[argc++] = "-k";

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   execv(sunmake_path,argv);
};





void
FORM_sunmake_scan(fs,text)
   FORM_SCAN fs;
   String text;
{
   MAKE_DATA md;

   if (text == NULL && fs->data != NULL) {
      md = (MAKE_DATA) fs->data;
      while (md->level > 0) sunscan_done(md);
    };

   make_scan(fs,text,sun_pattern_table);
};





/************************************************************************/
/*									*/
/*	FORM_decmake_init -- dec make initialization			*/
/*	FORM_decmake_exec_build -- execute command in a makefile	*/
/*	FORM_decmake_exec_scan -- start scanning makefile info		*/
/*	FORM_decmake_scan -- scan a line of input			*/
/*									*/
/************************************************************************/


void
FORM_decmake_init(db)
   AUXD db;
{
   String s;
   Character buf[1024];
   Integer i;

   s = getenv("MAKE");
   if (s == NULL) s = AUXDget_info(db,"PATH");
   if (s == NULL) s = SUNMAKE_PATH;
   sprintf(buf,s,BWElocal_project(),BWEarch());
   decmake_path = SALLOC(buf);

   for (i = 0; dec_pattern_table[i].text != NULL; ++i) {
      dec_pattern_table[i].pattern = PMATmake_pattern(dec_pattern_table[i].text,MAX_ARGS,NULL);
    };
};





void
FORM_decmake_exec_build(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "make";

   if (dir != NULL) chdir(dir);

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   execv(decmake_path,argv);
};





void
FORM_decmake_exec_scan(fb,dir,proj,itm)
   FORM_BACKEND fb;
   String dir;
   FORM_ITEM proj;
   FORM_ITEM itm;
{
   Integer argc,nul;
   String argv[32];
   String s;

   argc = 0;
   argv[argc++] = "make";

   if (dir != NULL) chdir(dir);

   s = FORM_item_attr_get(proj,"MAKEFILE").string_value;
   if (s != NULL) {
      argv[argc++] = "-f";
      argv[argc++] = s;
    };

   argv[argc++] = "-d";

   argv[argc++] = "-n";
   argv[argc++] = "-k";

   if (itm != NULL) {
      argv[argc++] = itm->name;
    };

   argv[argc] = NULL;

   nul = open("/dev/null",1);
   dup2(nul,2);
   close(nul);

   execv(decmake_path,argv);
};





void
FORM_decmake_scan(fs,text)
   FORM_SCAN fs;
   String text;
{
   MAKE_DATA md;

   if (text == NULL && fs->data != NULL) {
      md = (MAKE_DATA) fs->data;
      while (md->level > 0) decscan_done(md);
    };

   make_scan(fs,text,dec_pattern_table);
};





/************************************************************************/
/*									*/
/*	make_scan -- setup scanning					*/
/*									*/
/************************************************************************/


static void
make_scan(fs,text,pats)
   FORM_SCAN fs;
   String text;
   MAKE_PAT * pats;
{
   MAKE_DATA md;

   if (text == NULL && fs->data == NULL) {
      md = PALLOC(MAKE_DATA_INFO);
      fs->data = (Universal) md;
      md->project = fs->project;
      md->last = NULL;
      md->stack = NULL;
      md->handle_rule = FALSE;
      md->notarget = FALSE;
      md->infiles = FALSE;
      md->pstack = NULL;
      md->lastdef = NULL;
      md->pat_table = pats;
      md->level = 0;
    }
   else if (text == NULL) {
      md = (MAKE_DATA) fs->data;
      fs->item = md->last;
      free(md);
      fs->data = NULL;
    }
   else {
      make_scan_line(fs->data,text);
    };
};





/************************************************************************/
/*									*/
/*	make_scan_line -- scan a line of debug output from make 	*/
/*									*/
/************************************************************************/


static void
make_scan_line(md,text)
   MAKE_DATA md;
   String text;
{
   Integer spct;
   Universal args[MAX_ARGS];
   Integer i;

   spct = 0;
   while (isspace(*text)) {
      ++spct;
      ++text;
    };

   if (md->handle_rule && spct == 8) {
      return;
    };

   md->handle_rule = FALSE;
   md->nspace = spct;

   if (*text == 0) {			/* blank line			*/
      md->notarget = FALSE;
      return;
    };

   for (i = 0; md->pat_table[i].text != NULL; ++i) {
      if (md->pat_table[i].text[0] == text[0] ||
	     (spct == 0 && md->pat_table[i].text[0] == '%')) {
	 if (PMATmatch(text,md->pat_table[i].pattern,args) >= 0) {
	    if (md->pat_table[i].rtn != NULL) {
	       (*md->pat_table[i].rtn)(md,args,md->pat_table[i].param);
	     };
	    PMATfree_args(md->pat_table[i].pattern,args);
	    break;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	GNU Make database scanning routines				*/
/*									*/
/************************************************************************/


static void
gnuscan_def(md,args)
   MAKE_DATA md;
   String args[];
{
   FORM_ITEM itm,ritm;
   String rhs,s;

   md->lastdef = NULL;

   if (!md->infiles || md->notarget) return;

   if (STREQL(args[0],".SUFFIXES")) return;
   if (STREQL(args[0],".PHONY")) return;
   if (STREQL(args[0],".DEFAULT")) return;

   rhs = args[1];

   itm = FORM_item_define(md->project,args[0]);
   FORM_item_attr_set(itm,"PHONY",FORM_attr_value_boolean(FALSE));
   FORM_item_attr_set(itm,"PRECIOUS",FORM_attr_value_boolean(FALSE));
   md->lastdef = itm;

   while (rhs != NULL && rhs[0] != 0) {
      s = index(rhs,' ');
      if (s != NULL) *s++ = 0;
      else s = NULL;
      ritm = FORM_item_define(md->project,rhs);
      FORM_link_define(ritm,itm);
      rhs = s;
    };
};





static void
gnuscan_notarget(md)
   MAKE_DATA md;
{
   md->notarget = TRUE;
};






/* ARGSUSED */

static void
gnuscan_defflags(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_ITEM itm;
   Boolean fg;

   if (md->lastdef == NULL) return;

   switch (flag[0]) {
      case 'X' :
	 FORM_item_attr_set(md->lastdef,"PHONY",FORM_attr_value_boolean(TRUE));
	 break;
      case 'P' :
	 FORM_item_attr_set(md->lastdef,"PRECIOUS",FORM_attr_value_boolean(TRUE));
	 break;
    };

};




static void
gnuscan_setrules(md)
   MAKE_DATA md;
{
   md->handle_rule = TRUE;
};






static void
gnuscan_files(md,args,v)
   MAKE_DATA md;
   String args[];
   String v;
{
   md->infiles = (v[0] == '1' ? TRUE : FALSE);
};





/************************************************************************/
/*									*/
/*	GNU Make debug output scanning routines 			*/
/*									*/
/************************************************************************/


static void
gnuscan_consider(md,args)
   MAKE_DATA md;
   String args[];
{
   FORM_ITEM itm;
   MAKE_STACK stk;
   FORM_LINK lnk;

   itm = FORM_item_define(md->project,args[0]);
   FORM_item_attr_set(itm,"CHECKED",FORM_attr_value_boolean(TRUE));
   FORM_item_attr_set(itm,"CHANGED",FORM_attr_value_boolean(FALSE));
   FORM_item_attr_set(itm,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
   FORM_item_attr_set(itm,"DISPITEM",FORM_attr_value_int(0));

   stk = PALLOC(MAKE_STACK_INFO);
   stk->item = itm;
   stk->next = md->stack;
   stk->implicit = FALSE;

   if (md->stack != NULL) {
      lnk = FORM_link_define(stk->item,md->stack->item);
      FORM_link_attr_set(lnk,"IMPLICIT",FORM_attr_value_boolean(md->stack->implicit));
      FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
      FORM_link_attr_set(lnk,"DISPLINK",FORM_attr_value_int(0));
      md->stack->implicit = FALSE;
    };

   md->stack = stk;
};





static void
gnuscan_done(md)
   MAKE_DATA md;
{
   MAKE_STACK ms;

   ms = md->stack;

   if (ms != NULL) {
      md->last = ms->item;
      md->stack = ms->next;
      free(ms);
    };
};





static void
gnuscan_flags(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_LINK lnk;
   Boolean fg;

   switch (flag[0]) {
      case 'I' :
	 md->stack->implicit = TRUE;
	 break;
      case 'O' :
      case 'N' :
	 if (md->last == NULL) break;
	 fg = (flag[0] == 'N' ? TRUE : FALSE);
	 FORM_item_attr_set(md->last,"OUTOFDATE",FORM_attr_value_boolean(fg));
	 if (fg && md->stack != NULL && md->stack->item != NULL) {
	    lnk = FORM_link_define(md->last,md->stack->item);
	    FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(TRUE));
	  };
	 break;
    };

};




static void
gnuscan_dep(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_ITEM frm,to;
   FORM_LINK lnk;
   Boolean fg;

   frm = FORM_item_define(md->project,args[0]);
   to = FORM_item_define(md->project,args[1]);
   lnk = FORM_link_define(frm,to);

   fg = ((flag[0] == 'o' || flag[0] == 'O') ? FALSE : TRUE);
   FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(fg));

   if (fg) {
      if (!FORM_item_attr_get(frm,"OUTOFDATE").bool_value) {
	 FORM_item_attr_set(frm,"CHANGED",FORM_attr_value_boolean(TRUE));
       };
    };
};





/************************************************************************/
/*									*/
/*	GNU Make project scanning routines				*/
/*									*/
/************************************************************************/


static void
gnuscan_proj(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_ITEM proj;
   MAKE_PSTACK ps;
   FORM_LINK lnk;

   proj = FORM_proj_find(args[0]);

   if (STREQL(flag,"E0")) {
      md->project = proj;
    }
   else if (STREQL(flag,"L0")) {
    }
   else if (STREQL(flag,"E1")) {
      ps = PALLOC(MAKE_PSTACK_INFO);
      ps->proj = md->project;
      ps->last = md->last;
      ps->next = md->pstack;
      md->pstack = ps;
      md->project = proj;
    }
   else if (STREQL(flag,"L1")) {
      ps = md->pstack;
      if (ps != NULL) {
	 lnk = FORM_link_define(md->last,ps->last);
	 FORM_link_attr_set(lnk,"RECURSE",FORM_attr_value_boolean(TRUE));
	 md->project = ps->proj;
	 md->last = ps->last;
	 md->pstack = ps->next;
	 free(ps);
       };
    };
};





/************************************************************************/
/*									*/
/*	Generic Make scanning routines					*/
/*									*/
/************************************************************************/


static void
makescan_def(md,args)
   MAKE_DATA md;
   String args[];
{
   FORM_ITEM itm,ritm;
   String rhs,s;
   Boolean pfg;

   if (md->nspace != 0) return;

   md->lastdef = NULL;

   pfg = FALSE;
   if (args[0][0] == '.') {
      itm = NULL;
      if (STREQL(args[0],".PRECIOUS")) pfg = TRUE;
      else return;
    }
   else {
      itm = FORM_item_define(md->project,args[0]);
      md->lastdef = itm;
    };

   rhs = args[1];

   while (rhs != NULL && rhs[0] != 0) {
      s = index(rhs,' ');
      if (s != NULL) {
	 while (isspace(*s)) *s++ = 0;
	 if (*s == 0) s = NULL;
       }
      else s = NULL;
      ritm = FORM_item_define(md->project,rhs);
      if (itm != NULL) FORM_link_define(ritm,itm);
      if (pfg) FORM_item_attr_set(itm,"PRECIOUS",FORM_attr_value_boolean(TRUE));
      rhs = s;
    };
};





/************************************************************************/
/*									*/
/*	Sun Make debug output scanning routines 			*/
/*									*/
/************************************************************************/


static void
sunscan_consider(md,args)
   MAKE_DATA md;
   String args[];
{
   FORM_ITEM itm;
   MAKE_STACK stk;
   FORM_LINK lnk;
   Boolean ign;

   while (md->nspace < md->level) {
      sunscan_done(md);
    };

   ign = FALSE;
   itm = NULL;

   if (md->stack == NULL) {
      if (STREQL(args[0],".INIT") ||
	     STREQL(args[0],".DONE") ||
	     STREQL(args[0],"default.mk") ||
	     STREQL(args[0],"/usr/include/make/default.mk")) {
	 ign = TRUE;
       };
    };

   if (!ign) {
      itm = FORM_item_define(md->project,args[0]);
      FORM_item_attr_set(itm,"CHECKED",FORM_attr_value_boolean(TRUE));
      FORM_item_attr_set(itm,"CHANGED",FORM_attr_value_boolean(FALSE));
      FORM_item_attr_set(itm,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
      FORM_item_attr_set(itm,"DISPITEM",FORM_attr_value_int(0));
    };

   stk = PALLOC(MAKE_STACK_INFO);
   stk->item = itm;
   stk->next = md->stack;
   stk->implicit = FALSE;

   if (md->stack != NULL && md->stack->item != NULL) {
      lnk = FORM_link_define(stk->item,md->stack->item);
      FORM_link_attr_set(lnk,"IMPLICIT",FORM_attr_value_boolean(md->stack->implicit));
      FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
      FORM_link_attr_set(lnk,"DISPLINK",FORM_attr_value_int(0));
      md->stack->implicit = FALSE;
    };

   md->stack = stk;
   ++md->level;
};





static void
sunscan_done(md)
   MAKE_DATA md;
{
   MAKE_STACK ms;

   ms = md->stack;

   if (ms != NULL) {
      md->last = ms->item;
      md->stack = ms->next;
      free(ms);
    };

   --md->level;
};





static void
sunscan_flags(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_LINK lnk;
   FORM_ITEM itm;
   Boolean fg;

   switch (flag[0]) {
      case 'I' :
	 while (md->nspace < md->level) sunscan_done(md);
	 if (md->last != NULL) {
	    itm = FORM_item_define(md->project,args[0]);
	    lnk = FORM_link_define(itm,md->last);
	    FORM_link_attr_set(lnk,"IMPLICIT",FORM_attr_value_boolean(TRUE));
	  };
	 break;
      case 'O' :
      case 'N' :
	 itm = FORM_item_define(md->project,args[0]);
	 while (md->last != NULL && md->last != itm) sunscan_done(md);
	 if (md->last == NULL) break;
	 fg = (flag[0] == 'N' ? TRUE : FALSE);
	 FORM_item_attr_set(md->last,"OUTOFDATE",FORM_attr_value_boolean(fg));
	 if (fg && md->stack != NULL && md->stack->item != NULL) {
	    lnk = FORM_link_define(md->last,md->stack->item);
	    FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(TRUE));
	  };
	 break;
    };

};




static void
sunscan_dep(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_ITEM frm,to;
   FORM_LINK lnk;
   Boolean fg;

   while (md->nspace < md->level) {
      sunscan_done(md);
    };

   if (flag[0] == 'L') return;

   frm = FORM_item_define(md->project,args[0]);
   to = FORM_item_define(md->project,args[1]);
   lnk = FORM_link_define(frm,to);

   fg = (flag[0] == 'O' ? FALSE : TRUE);
   FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(fg));

   if (fg) {
      if (!FORM_item_attr_get(frm,"OUTOFDATE").bool_value) {
	 FORM_item_attr_set(frm,"CHANGED",FORM_attr_value_boolean(TRUE));
       };
    };
};





/************************************************************************/
/*									*/
/*	Dec Make debug output scanning routines 			*/
/*									*/
/************************************************************************/


static void
decscan_consider(md,args)
   MAKE_DATA md;
   String args[];
{
   FORM_ITEM itm;
   MAKE_STACK stk;
   FORM_LINK lnk;
   Boolean ign;
   Integer dolvl;

   dolvl = ((Integer) args[1]);

   while (dolvl < md->level) {
      decscan_done(md);
    };

   itm = NULL;

   itm = FORM_item_define(md->project,args[0]);
   FORM_item_attr_set(itm,"CHECKED",FORM_attr_value_boolean(TRUE));
   FORM_item_attr_set(itm,"CHANGED",FORM_attr_value_boolean(FALSE));
   FORM_item_attr_set(itm,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
   FORM_item_attr_set(itm,"DISPITEM",FORM_attr_value_int(0));

   stk = PALLOC(MAKE_STACK_INFO);
   stk->item = itm;
   stk->next = md->stack;
   stk->implicit = FALSE;

   if (md->stack != NULL && md->stack->item != NULL) {
      lnk = FORM_link_define(stk->item,md->stack->item);
      FORM_link_attr_set(lnk,"IMPLICIT",FORM_attr_value_boolean(md->stack->implicit));
      FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(FALSE));
      FORM_link_attr_set(lnk,"DISPLINK",FORM_attr_value_int(0));
      md->stack->implicit = FALSE;
    };

   md->stack = stk;
   ++md->level;
};





static void
decscan_done(md)
   MAKE_DATA md;
{
   MAKE_STACK ms;

   ms = md->stack;

   if (ms != NULL) {
      md->last = ms->item;
      md->stack = ms->next;
      free(ms);
    };

   --md->level;
};





static void
decscan_flags(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_LINK lnk;
   FORM_ITEM itm,ritm;
   Boolean fg;
   String s,rhs;
   Integer lvl;

   switch (flag[0]) {
      case 'I' :
      case 'E' :
	 lvl = ((int) args[0]) + 1;
	 while (lvl < md->level) decscan_done(md);
	 if (md->stack != NULL) md->stack->implicit = (flag[0] == 'I');
	 break;

      case '@' :
	 if (args[0] != NULL && args[0][0] != 0 && args[0][0] != ' ') {
	    itm = FORM_item_define(md->project,args[0]);
	    while (md->stack != NULL && md->stack->item != itm) decscan_done(md);
	  };
	 break;
      case '?' :
	 if (md->stack == NULL || md->stack->item == NULL) break;
	 rhs = args[0];
	 while (rhs != NULL && rhs[0] != 0) {
	    if (strncmp(rhs,"noreset =",9) == 0) break;
	    s = index(rhs,' ');
	    if (s != NULL) *s++ = 0;
	    else s = NULL;
	    ritm = FORM_item_define(md->project,rhs);
	    lnk = FORM_link_define(ritm,md->stack->item);
	    FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(TRUE));
	    FORM_item_attr_set(md->stack->item,"OUTOFDATE",FORM_attr_value_boolean(TRUE));
	    if (!FORM_item_attr_get(ritm,"OUTOFDATE").bool_value) {
	       FORM_item_attr_set(ritm,"CHANGED",FORM_attr_value_boolean(TRUE));
	     };
	    rhs = s;
	  };
	 break;
    };
};




static void
decscan_dep(md,args,flag)
   MAKE_DATA md;
   String args[];
   String flag;
{
   FORM_ITEM frm,to;
   FORM_LINK lnk;
   Boolean fg;
   Integer i;

   i = md->nspace/3;
   while (i < md->level) decscan_done(md);

   frm = FORM_item_define(md->project,args[0]);
   to = FORM_item_define(md->project,args[1]);
   lnk = FORM_link_define(frm,to);

   if (flag[0] == 'L') {
      FORM_link_attr_set(lnk,"IMPLICIT",FORM_attr_value_boolean(TRUE));
    }
   else {
      fg = (flag[0] == 'O' ? FALSE : TRUE);
      FORM_link_attr_set(lnk,"OUTOFDATE",FORM_attr_value_boolean(fg));

      if (fg) {
	 if (!FORM_item_attr_get(frm,"OUTOFDATE").bool_value) {
	    FORM_item_attr_set(frm,"CHANGED",FORM_attr_value_boolean(TRUE));
	  };
       };
    };
};





/* end of formmake.c */
