/************************************************************************/
/*									*/
/*		formdisp.c						*/
/*									*/
/*	Display interface for form windows				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "form_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/

#define DEFAULT_METHOD		GELO_METHOD_LEVEL|GELO_METHOD_BIT_HOR
#define DEFAULT_CONNMETHOD	GELO_CONN_METHOD_DIRECT_RECT
#define DEFAULT_FIXED		FALSE
#define DEFAULT_STANDARD	FALSE
#define DEFAULT_CENTERED	TRUE
#define DEFAULT_ARC_STYLE	ASH_STYLE_SOLID
#define DEFAULT_ARROW		GELO_ARROW_SINGLE_ALL
#define PERT_ARROW		GELO_ARROW_SINGLE
#define DEFAULT_WHITESPACE	0

#define SELECT_STYLE		1
#define OUTOFDATE_STYLE 	4
#define CHANGED_STYLE		2

#define EXPLICIT_ARC_STYLE	ASH_STYLE_THICK
#define IMPLICIT_ARC_STYLE	ASH_STYLE_SOLID
#define USES_ARC_STYLE		ASH_STYLE_THICK_DOTTED

#define DEFAULT_DISPLAY_ALL	FALSE
#define DEFAULT_DISPLAY_FORCE	TRUE
#define DEFAULT_DISPLAY_FIXCUR	FALSE
#define DEFAULT_DISPLAY_LEVELS	(-1)
#define DEFAULT_DISPLAY_ZOOM	0
#define DEFAULT_DISPLAY_INLINKS TRUE
#define DEFAULT_DISPLAY_OUTLINKS TRUE



#define MAX_LEVEL		100000000




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	GELO_METHOD		default_method;
static	GELO_CONN_METHOD	default_connmethod;
static	Boolean 		default_fixed;
static	Boolean 		default_standard;
static	Boolean 		default_centered;
static	Integer 		default_whitespace;
static	ASH_LINE_STYLE		arc_style;
static	GELO_ARC_ARROW		arrow_style;
static	GELO_ARC_ARROW		pert_arrow_style;
static	Integer 		hilite_style[FORM_HILITE_STYLE_COUNT];
static	Boolean 		auto_update;

static	Boolean 		default_display_all;
static	Boolean 		default_display_force;
static	Boolean 		default_display_fixcur;
static	Boolean 		default_display_inlinks;
static	Boolean 		default_display_outlinks;
static	Integer 		default_display_levels;
static	Integer 		default_display_zoom;

static	Sequence		default_item_exc;
static	Sequence		default_item_inc;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		clear_display();
static	void		form_display();
static	void		form_hilites();
static	GELO_OBJECT	show_item();
static	void		set_item_props();
static	void		set_link_props();
static	void		hilite_object();
static	Sequence	form_auxd_list();





/************************************************************************/
/*									*/
/*	FORM_disp_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FORM_disp_init()
{
   AUXD hdl;
   String msg;

   hdl = AUXDget_handle(NULL,"FORM");

   if (!AUXDget_defined(hdl,"METHOD")) default_method = DEFAULT_METHOD;
   else default_method = (GELO_METHOD) AUXDget_info_int(hdl,"METHOD");

   if (!AUXDget_defined(hdl,"CONNMETHOD")) default_connmethod = DEFAULT_CONNMETHOD;
   else default_connmethod = (GELO_CONN_METHOD) AUXDget_info_int(hdl,"CONNMETHOD");

   if (!AUXDget_defined(hdl,"FIXED")) default_fixed = DEFAULT_FIXED;
   else default_fixed = AUXDget_info_int(hdl,"FIXED");

   if (!AUXDget_defined(hdl,"STANDARD")) default_standard = DEFAULT_STANDARD;
   else default_standard = AUXDget_info_int(hdl,"STANDARD");

   if (!AUXDget_defined(hdl,"CENTERED")) default_centered = DEFAULT_CENTERED;
   else default_centered = AUXDget_info_int(hdl,"CENTERED");

   if (!AUXDget_defined(hdl,"WHITE_SPACE")) default_whitespace = DEFAULT_WHITESPACE;
   else default_whitespace = AUXDget_info_int(hdl,"WHITE_SPACE");

   if (!AUXDget_defined(hdl,"ARC_STYLE")) arc_style = DEFAULT_ARC_STYLE;
   else arc_style = (ASH_LINE_STYLE) AUXDget_info_int(hdl,"ARC_STYLE");

   if (!AUXDget_defined(hdl,"ARROW_STYLE")) arrow_style = DEFAULT_ARROW;
   else arrow_style = (GELO_ARC_ARROW) AUXDget_info_int(hdl,"ARROW_STYLE");

   if (!AUXDget_defined(hdl,"PERT_ARROW_STYLE")) pert_arrow_style = PERT_ARROW;
   else pert_arrow_style = (GELO_ARC_ARROW) AUXDget_info_int(hdl,"PERT_ARROW_STYLE");

   auto_update = AUXDget_defined(hdl,"AUTO_UPDATE");

   if (!AUXDget_defined(hdl,"DISPLAY_ALL")) default_display_all = DEFAULT_DISPLAY_ALL;
   else default_display_all = AUXDget_info_int(hdl,"DISPLAY_ALL");

   if (!AUXDget_defined(hdl,"DISPLAY_FORCE")) default_display_force = DEFAULT_DISPLAY_FORCE;
   else default_display_force = AUXDget_info_int(hdl,"DISPLAY_FORCE");

   if (!AUXDget_defined(hdl,"DISPLAY_FIXCUR")) default_display_fixcur = DEFAULT_DISPLAY_FIXCUR;
   else default_display_fixcur = AUXDget_info_int(hdl,"DISPLAY_FIXCUR");

   if (!AUXDget_defined(hdl,"DISPLAY_LEVELS")) default_display_levels = DEFAULT_DISPLAY_LEVELS;
   else default_display_levels = AUXDget_info_int(hdl,"DISPLAY_LEVELS");

   if (!AUXDget_defined(hdl,"DISPLAY_ZOOM")) default_display_zoom = DEFAULT_DISPLAY_ZOOM;
   else default_display_zoom = AUXDget_info_int(hdl,"DISPLAY_ZOOM");

   if (!AUXDget_defined(hdl,"DISPLAY_INLINKS")) default_display_inlinks = DEFAULT_DISPLAY_INLINKS;
   else default_display_inlinks = AUXDget_info_int(hdl,"DISPLAY_INLINKS");

   if (!AUXDget_defined(hdl,"DISPLAY_OUTLINKS")) default_display_outlinks = DEFAULT_DISPLAY_OUTLINKS;
   else default_display_outlinks = AUXDget_info_int(hdl,"DISPLAY_OUTLINKS");

   default_item_exc = form_auxd_list(hdl,"ITEM_EXCLUDE");
   default_item_inc = form_auxd_list(hdl,"ITEM_INCLUDE");

   hilite_style[FORM_HILITE_NONE] = 0;
   hilite_style[FORM_HILITE_SELECT] = SELECT_STYLE;
   hilite_style[FORM_HILITE_OUTOFDATE] = OUTOFDATE_STYLE;
   hilite_style[FORM_HILITE_CHANGED] = CHANGED_STYLE;
};





/************************************************************************/
/*									*/
/*	FORM_disp_setup -- setup display in window			*/
/*									*/
/************************************************************************/


void
FORM_disp_setup(fw)
   FORM_WIN fw;
{
   Integer i;

   fw->gid = NULL;
   if (fw->disp_win == NULL) return;

   GELOwindow_open(fw->disp_win);

   fw->method = default_method;
   fw->connmethod = default_connmethod;
   fw->fixed = default_fixed;
   fw->standard = default_standard;
   fw->centered = default_centered;
   fw->whitespace = default_whitespace;

   fw->selection = NULL;
   for (i = 0; i < FORM_HILITE_STYLE_COUNT; ++i) fw->hilite[i] = NULL;
   fw->dispproj = NULL;
   fw->dispitem = NULL;
   fw->ignore = NULL;

   fw->display_all = default_display_all;
   fw->display_force = default_display_force;
   fw->display_fixcur = default_display_fixcur;
   fw->display_inlinks = default_display_inlinks;
   fw->display_outlinks = default_display_outlinks;
   fw->display_levels = default_display_levels;
   fw->display_zoom = default_display_zoom;

   FORM_disp_reset(fw,TRUE);
};





/************************************************************************/
/*									*/
/*	FORM_disp_reset -- set up initial nodes to display		*/
/*									*/
/************************************************************************/


void
FORM_disp_reset(fw,clr)
   FORM_WIN fw;
   Boolean clr;
{
   Integer ct,lct;
   String s;

   if (fw->root == NULL) return;

   fw->selection = NULL;

   if (clr && fw->root->is_project) {
      s = FORM_item_attr_get(fw->root,"DEFAULT").string_value;
      if (s != NULL) {
/*	 fw->selection = FORM_item_define(fw->root,s);		*/
       };
    };

   if (clr) {
      if (fw->ignore != NULL) {
	 LFREE(fw->ignore);
	 fw->ignore = NULL;
       };
    };

   fw->useregex = clr;
   FORM_disp_show(fw);
   fw->useregex = FALSE;
};





/************************************************************************/
/*									*/
/*	FORM_disp_show -- handle displaying the tree			*/
/*									*/
/************************************************************************/


void
FORM_disp_show(fw)
   FORM_WIN fw;
{
   Sequence l,la;

   ASHinput_lock_makelong(fw->window);

   clear_display(fw);

   fw->dispproj = NULL;
   fw->nodes = HASHcreate(100);

   fw->gid = GELOdefine_layout();
   GELOset_owner(fw->gid,fw);
   GELOdefine_layout_method(fw->gid,fw->method);
   GELOdefine_layout_conn_method(fw->gid,fw->connmethod);
   GELOdefine_layout_fixed(fw->gid,fw->fixed);
   GELOdefine_layout_standard(fw->gid,fw->standard);
   GELOdefine_layout_centered(fw->gid,fw->centered);
   GELOdefine_layout_white_space(fw->gid,fw->whitespace);
   GELOwindow_draw(fw->disp_win,fw->gid);
   GELOwindow_resize_object(fw->disp_win,fw->gid);

   form_display(fw);

   form_hilites(fw);

   GELOwindow_redraw(fw->disp_win);
};





/************************************************************************/
/*									*/
/*	FORM_disp_free -- free current display				*/
/*									*/
/************************************************************************/


void
FORM_disp_free(fw)
   FORM_WIN fw;
{
   if (fw->gid != NULL) {
      GELOwindow_free(fw->disp_win,fw->gid);
      fw->gid = NULL;
    };
};





/************************************************************************/
/*									*/
/*	FORM_disp_select -- set selection				*/
/*									*/
/************************************************************************/


void
FORM_disp_select(fw,fi)
   FORM_WIN fw;
   FORM_ITEM fi;
{
   Boolean redo;
   GELO_OBJECT go;
   Character buf[1024];
   ENTRY look,* fnd;

   if (fw->selection == fi) return;

   if (fi != NULL) FORM_disp_unignore(fw,fi);

   redo = fw->display_force;
   if (fw->display_all && !fw->display_fixcur && fw->display_zoom <= 1) redo = FALSE;
   if (fw->nodes == NULL) redo = TRUE;

   if (redo) {
      fw->selection = fi;
      FORM_disp_show(fw);
    }
   else {
      if (fw->selection != NULL && fw->hilite[FORM_HILITE_SELECT] != NULL) {
	 go = CAR(GELO_OBJECT,fw->hilite[FORM_HILITE_SELECT]);
	 if (go != NULL) {
	    hilite_object(fw,go,FORM_HILITE_SELECT,FALSE,TRUE);
	  };
       };

      fw->selection = fi;

      if  (fi != NULL) {
	 sprintf(buf,"%x",fi);
	 look.key = buf;
	 look.data = NULL;
	 fnd = HASHsearch(fw->nodes,look,ENTER);
	 if (fnd->data != NULL) {
	    go = (GELO_OBJECT) fnd->data;
	    hilite_object(fw,go,FORM_HILITE_SELECT,TRUE,TRUE);
	  }
	 else FORM_disp_show(fw);
       };
   };
};





/************************************************************************/
/*									*/
/*	FORM_disp_ignore -- ignore an item				*/
/*	FORM_disp_unignore -- use (no ignore) an item			*/
/*									*/
/************************************************************************/


void
FORM_disp_ignore(fw,fi)
   FORM_WIN fw;
   FORM_ITEM fi;
{
   if (MEMQ(fi,fw->ignore)) return;

   fw->ignore = CONS(fi,fw->ignore);

   if (fw->selection == fi) fw->selection = NULL;
};






void
FORM_disp_unignore(fw,fi)
   FORM_WIN fw;
   FORM_ITEM fi;
{
   if (fi == NULL) return;

   fw->ignore = REMOB(fi,fw->ignore);
};





/************************************************************************/
/*									*/
/*	FORM_disp_regex -- regular expression elimination of items	*/
/*									*/
/************************************************************************/


void
FORM_disp_regex(fw,inc,exfg,exc,infg)
   FORM_WIN fw;
   String inc;
   Boolean exfg;
   String exc;
   Boolean infg;
{
   FORM_ITEM fi;
   Sequence l;

   if (inc != NULL && inc[0] != 0 && re_comp(inc) == NULL) {
      forin (fi,FORM_ITEM,l,fw->dispitem) {
	 if (re_exec(fi->name)) FORM_disp_unignore(fw,fi);
	 else if (exfg) FORM_disp_ignore(fw,fi);
       };
    };

   if (exc != NULL && exc[0] != 0 && re_comp(exc) == NULL) {
      forin (fi,FORM_ITEM,l,fw->dispitem) {
	 if (re_exec(fi->name)) {
	    FORM_disp_ignore(fw,fi);
	  }
	 else if (infg) FORM_disp_unignore(fw,fi);
       };
    };

   FORM_disp_show(fw);
};







/************************************************************************/
/*									*/
/*	clear_display -- clear displays for a tree			*/
/*									*/
/************************************************************************/


static void
clear_display(fw)
   FORM_WIN fw;
{
   Integer i;

   if (fw->gid != NULL) {
      GELOwindow_free(fw->disp_win,fw->gid);
      fw->gid = NULL;
    };

   if (fw->nodes != NULL) {
      HASHdestroy(fw->nodes);
      fw->nodes = NULL;
    };

   for (i = 0; i < FORM_HILITE_STYLE_COUNT; ++i) {
      if (fw->hilite[i] != NULL) {
	 LFREE(fw->hilite[i]);
	 fw->hilite[i] = NULL;
       };
    };

   if (fw->dispproj != NULL) {
      LFREE(fw->dispproj);
      fw->dispproj = NULL;
    };

   if (fw->dispitem != NULL) {
      LFREE(fw->dispitem);
      fw->dispitem = NULL;
    };
};






/************************************************************************/
/*									*/
/*	form_display -- set up items on the display			*/
/*									*/
/************************************************************************/


static void
form_display(fw)
   FORM_WIN fw;
{
   FORM_ITEM itm;
   Sequence l;

   if (fw->display_all || fw->selection == NULL) {
      if (fw->root->items == NULL) {
	 show_item(fw,fw->root,-1,TRUE,TRUE);
       }
      else {
	 forin (itm,FORM_ITEM,l,fw->root->items) {
	    show_item(fw,itm,-1,TRUE,TRUE);
	  };
       }
    }
   else {
      show_item(fw,fw->selection,0,TRUE,TRUE);
    };
};





/************************************************************************/
/*									*/
/*	form_hilites -- handle GELO selections				*/
/*									*/
/************************************************************************/


static void
form_hilites(fw)
   FORM_WIN fw;
{
   Integer i;
   GELO_OBJECT go;
   Sequence l;

   for (i = FORM_HILITE_STYLE_COUNT-1; i >= 0; --i) {
      if (fw->hilite[i] != NULL && hilite_style[i] != 0) {
	 forin (go,GELO_OBJECT,l,fw->hilite[i]) {
	    GELOwindow_select(fw->disp_win,go,hilite_style[i],TRUE);
	  };
       };
    };
};






/************************************************************************/
/*									*/
/*	show_item -- add item and its dependents to the display 	*/
/*									*/
/************************************************************************/


static GELO_OBJECT
show_item(fw,itm,lvl,fwd,bwd)
   FORM_WIN fw;
   FORM_ITEM itm;
   Integer lvl;
   Boolean fwd;
   Boolean bwd;
{
   Character buf[1024];
   ENTRY look,* fnd;
   GELO_OBJECT go,tgt,arc;
   GELO_CONNECT gc;
   FORM_LINK lnk;
   Sequence l;
   Float f;
   String s;
   Boolean fg;

   if (MEMQ(itm,fw->ignore)) return NULL;

   sprintf(buf,"%x",itm);
   look.key = buf;
   look.data = NULL;
   fnd = HASHsearch(fw->nodes,look,ENTER);
   if (fnd->data != NULL) return (GELO_OBJECT) fnd->data;

   if (fw->useregex) {
      forin (s,String,l,default_item_exc) {
	 if (re_comp(s) == NULL && re_exec(itm->name)) FORM_disp_ignore(fw,itm);
       };
      fg = (default_item_exc == NULL);
      forin (s,String,l,default_item_inc) {
	 if (re_comp(s) == NULL) {
	    if (re_exec(itm->name)) FORM_disp_unignore(fw,itm);
	    else if (fg) FORM_disp_ignore(fw,itm);
	  };
       };
      if (MEMQ(itm,fw->ignore)) return NULL;
    };

   if (itm->project != NULL && !MEMQ(itm->project,fw->dispproj))
      fw->dispproj = CONS(itm->project,fw->dispproj);
   fw->dispitem = CONS(itm,fw->dispitem);

   fnd->key = SALLOC(buf);

   go = GELOdefine_data();
   fnd->data = (char *) go;

   GELOset_owner(go,fw);
   GELOset_contents(go,0);
   GELOset_user_structure(go,itm);
   GELOdefine_layout_component(fw->gid,go);

   set_item_props(fw,itm,go);

   if (fw->selection == itm) {
      if (fw->display_fixcur) {
	 GELOuse_default_x(go,TRUE);
	 GELOuse_default_y(go,TRUE);
       };
      if (fw->display_zoom > 0) {
	 f = fw->display_zoom;
	 GELOset_priority_x(go,f);
	 GELOset_priority_y(go,f);
       };
    };

   if (fw->display_levels < 0 || lvl < 0 || lvl < fw->display_levels) {
      if (fw->display_inlinks && fwd) {
	 forin (lnk,FORM_LINK,l,itm->in_links) {
	    tgt = show_item(fw,lnk->from,(lvl < 0 ? lvl : lvl+1),TRUE,FALSE);
	    if (tgt == NULL) continue;
	    arc = GELOdefine_arc();
	    gc = GELOnew_connect(tgt,GELO_PORT_ANY,go,GELO_PORT_ANY);
	    GELOdefine_arc_connect(arc,gc);
	    set_link_props(fw,lnk,arc,gc);
	    GELOset_contents(arc,1);
	    GELOset_owner(arc,fw);
	    GELOset_user_structure(arc,lnk);
	    GELOdefine_layout_arc(fw->gid,arc);
	  };
       };
      if (fw->display_outlinks && bwd) {
	 forin (lnk,FORM_LINK,l,itm->out_links) {
	    tgt = show_item(fw,lnk->to,(lvl < 0 ? lvl : lvl+1),FALSE,TRUE);
	    if (tgt == NULL) continue;
	    arc = GELOdefine_arc();
	    gc = GELOnew_connect(go,GELO_PORT_ANY,tgt,GELO_PORT_ANY);
	    GELOdefine_arc_connect(arc,gc);
	    set_link_props(fw,lnk,arc,gc);
	    GELOset_contents(arc,1);
	    GELOset_owner(arc,fw);
	    GELOset_user_structure(arc,lnk);
	    GELOdefine_layout_arc(fw->gid,arc);
	  };
       };
    };

   return go;
};






/************************************************************************/
/*									*/
/*	set_item_props -- get features of item				*/
/*									*/
/************************************************************************/


static void
set_item_props(fw,itm,go)
   FORM_WIN fw;
   FORM_ITEM itm;
   GELO_OBJECT go;
{
   String s;
   Integer i;
   Character buf[1024];
   GELO_SHAPE shp;
   Integer cnt;

   s = FORM_item_attr_get(itm,"DISPNAME").string_value;

   if (s == NULL) {
      s = itm->name;

      if (itm->is_project) {
	 s = FORM_item_attr_get(itm,"PATHNAME").string_value;
	 if (s == NULL) s = itm->name;
	 else {
	    i = strlen(s);
	    cnt = 0;
	    for (i = strlen(s)-1; i >= 0; --i) {
	       if (s[i] == '/') {
		  ++cnt;
		  if (cnt == 3) break;
		};
	     };
	    if (i > 0) {
	       sprintf(buf,"...%s",&s[i]);
	       s = SALLOC(buf);
	     };
	    FORM_item_attr_set(itm,"DISPNAME",FORM_attr_value_string(s));
	  };
       };
    };
   GELOdefine_data_text(go,s);

   i = FORM_item_attr_get(itm,"DISPITEM").enum_value;
   if (i == 0) {
      if (itm->is_project) i = 1;
      else if (itm->out_links == NULL) i = 2;
      else if (itm->in_links == NULL) i = 5;
      else {
	 s = rindex(itm->name,".");
	 if ( s == NULL || index(s,'/') != NULL) i = 3;
	 else i = 4;
       };
      FORM_item_attr_set(itm,"DISPITEM",FORM_attr_value_int(i));
    };

   switch (i) {
      case 0 :					/* NONE 		*/
	 shp = GELO_SHAPE_RECTANGLE;
	 break;
      case 1 :					/* PROJECT		*/
	 shp = GELO_SHAPE_PENTAGON;
	 break;
      case 2 :					/* COMMAND		*/
	 shp = GELO_SHAPE_DIAMOND;
	 break;
      case 3 :					/* SYSTEM		*/
	 shp = GELO_SHAPE_HEXAGON;
	 break;
      case 4 :					/* INTERMEDIATE 	*/
	 shp = GELO_SHAPE_ROUND_RECTANGLE;
	 break;
      case 5 :					/* SOURCE		*/
	 shp = GELO_SHAPE_ELLIPSE;
	 break;
    };
   GELOdefine_data_shape(go,shp);

   if (itm == fw->selection) {
      hilite_object(fw,go,FORM_HILITE_SELECT,TRUE,FALSE);
    };

   if (FORM_item_attr_get(itm,"OUTOFDATE").bool_value) {
      hilite_object(fw,go,FORM_HILITE_OUTOFDATE,TRUE,FALSE);
    }
   else if (FORM_item_attr_get(itm,"CHANGED").bool_value) {
      hilite_object(fw,go,FORM_HILITE_CHANGED,TRUE,FALSE);
    };
};





/************************************************************************/
/*									*/
/*	set_link_props -- get features of link				*/
/*									*/
/************************************************************************/


static void
set_link_props(fw,lnk,go,gc)
   FORM_WIN fw;
   FORM_LINK lnk;
   GELO_OBJECT go;
   GELO_CONNECT gc;
{
   Integer i;

   i = FORM_link_attr_get(lnk,"DISPLINK").enum_value;
   if (i == 0) {
      if (FORM_link_attr_get(lnk,"IMPLICIT").bool_value) i = 2;
      else if (FORM_link_attr_get(lnk,"RECURSE").bool_value) i = 3;
      else i = 1;
      FORM_item_attr_set(lnk,"DISPLINK",FORM_attr_value_int(i));
    };

   switch (i) {
      case 0 :					/* NONE 		*/
	 i = 0;
	 break;

      case 1 :					/* EXPLICIT		*/
	 i = EXPLICIT_ARC_STYLE;
	 break;

      case 2 :					/* IMPLICIT		*/
	 i = IMPLICIT_ARC_STYLE;
	 break;

      case 3 :					/* USES 		*/
	 i = USES_ARC_STYLE;
	 break;
    };
   GELOconnect_arc_style(gc,i,GELO_ARROW_SINGLE);

   if (FORM_link_attr_get(lnk,"OUTOFDATE").bool_value) {
      hilite_object(fw,go,FORM_HILITE_OUTOFDATE,TRUE,FALSE);
    };
};





/************************************************************************/
/*									*/
/*	hilite_object -- change/set hiliting for item/link		*/
/*									*/
/************************************************************************/


static void
hilite_object(fw,go,sty,onoff,immed)
   FORM_WIN fw;
   GELO_OBJECT go;
   FORM_HILITE_STYLE sty;
   Boolean onoff;
   Boolean immed;
{
   Boolean fnd;

   fnd = MEMQ(go,fw->hilite[sty]);
   if (fnd && onoff) return;
   if (!fnd && !onoff) return;

   if (onoff) fw->hilite[sty] = APPEND(go,fw->hilite[sty]);
   else fw->hilite[sty] = REMOB(go,fw->hilite[sty]);

   if (immed) GELOwindow_select(fw->disp_win,go,hilite_style[sty],onoff);
};






/************************************************************************/
/*									*/
/*	form_auxd_list -- get list of AUXD strings as Sequence		*/
/*									*/
/************************************************************************/


static Sequence
form_auxd_list(hdl,id)
   AUXD hdl;
   String id;
{
   String txt[1024];
   Integer ct,i;
   Sequence l;

   if (!AUXDget_defined(hdl,id)) return NULL;

   ct = AUXDget_info_list(hdl,id,1024,txt);

   l = NULL;
   for (i = ct-1; i >= 0; --i) l = CONS(txt[i],l);

   return l;
};





/* end of formdisp.c */


