/************************************************************************/
/*									*/
/*		flowwin.c						*/
/*									*/
/*	Window manipulation routines for flow graphs			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "flow_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_windows;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_window();
static	void		check_windows();
static	int		flow_control();
static	void		flow_remove();
static	String		request_system();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define DISP_WIN	2



static	LEAF_DATA	flow_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DISP_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	FLOW_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
FLOW_win_init()
{
   all_windows = NULL;
};





/************************************************************************/
/*									*/
/*	FLOW_create -- create a new flow graph window			*/
/*									*/
/************************************************************************/


FLOW_WIN
FLOW_create(w,name)
   ASH_WINDOW w;
   String name;
{
   register FLOW_WIN fw;
   Character buf[256];

   if (name == NULL) name = request_system(w);
   else name = SALLOC(name);

   if (name == NULL) return NULL;
		
   sprintf(buf,"flowview: %s",name);
   ASHset_window_name(w,buf);
   ASHset_window_id(w,"FLOW");
   ASHset_window_defaults(w);

   fw = PALLOC(FLOW_WIN_INFO);
   fw->window = w;
   fw->system = name;

   PROTECT;
   all_windows = CONS(fw,all_windows);
   UNPROTECT;

   if (!setup_window(fw)) return NULL;

   if (ASHinput_lock(fw->window,TRUE) != ASH_LOCK_GRANTED) return NULL;

   fw->root = NULL;
   fw->topdisp = NULL;
   fw->selection = NULL;
   fw->hashtable = NULL;

   FLOW_disp_setup(fw);
   FLOW_menu_setup(fw);

   FLOW_node_setup(fw);

   if (fw->window != NULL) {
      FLOW_disp_reset(fw,TRUE);
      if (fw->disp_win != NULL) ASHvisible(fw->disp_win,TRUE);
    };

   ASHinput_unlock(fw->window);

   if (fw->window == NULL || fw->disp_win == NULL) return NULL;
   if (fw->root == NULL || fw->root->son == NULL) return NULL;

   return fw;
};





/************************************************************************/
/*									*/
/*	FLOW_find_window -- find flow window given ash window		*/
/*	FLOW_find_system -- find flow window given system name		*/
/*									*/
/************************************************************************/


FLOW_WIN
FLOW_find_window(w)
   ASH_WINDOW w;
{
   Sequence l;
   FLOW_WIN fw;

   forin (fw,FLOW_WIN,l,all_windows) {
      if (fw->window == w) break;
      else if (fw->disp_win == w) break;
      else if (fw->menu_win == w) break;
      else if (fw->text_win == w) break;
    };

   return fw;
};





FLOW_WIN
FLOW_find_system(sys)
   String sys;
{
   Sequence l;
   FLOW_WIN fw;

   forin (fw,FLOW_WIN,l,all_windows) {
      if (FIELDsystem_match(fw->system,sys)) break;
    };

   return fw;
};





/************************************************************************/
/*									*/
/*	setup_window -- define the portions of an flow window		*/
/*									*/
/************************************************************************/


static Boolean
setup_window(fw)
   FLOW_WIN fw;
{
   LEAFsetup_window(fw->window,flow_leaf,fw);

   LEAFset_control(fw->window,flow_control);
   LEAFredraw(fw->window);

   fw->disp_win = LEAFinq_window(fw->window,DISP_WIN,0);
   if (fw->disp_win == NULL) {
      fprintf(stderr,"FLOW: Window too small to run flow graph viewer\n");
      ASHremove(fw->window);
      return FALSE;
   };

   fw->menu_win = LEAFinq_window(fw->window,MENU_WIN,0);
   fw->text_win = NULL;
   fw->edit_info = NULL;

   LEAFset_refresh(fw->window,check_windows);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_windows -- check for user doing a stupid resize		*/
/*									*/
/************************************************************************/


static void
check_windows(w)
   ASH_WINDOW w;
{
   FLOW_WIN fw;
   ASH_WINDOW dw;

   fw = FLOW_find_window(w);

   if (fw == NULL) return;

   dw = LEAFinq_window(w,DISP_WIN,0);

   if (dw != fw->disp_win) {
      if (ASHinq_valid_window(w)) ASHremove(w);
    };
};





/************************************************************************/
/*									*/
/*	flow_control -- handle ASH and other control messages		*/
/*									*/
/************************************************************************/


static int
flow_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register FLOW_WIN fw;

   fw = FLOW_find_window(w);

   if (fw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      flow_remove(fw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	flow_remove -- remove given flow				*/
/*									*/
/************************************************************************/


static void
flow_remove(fw)
   FLOW_WIN fw;
{
   PROTECT;

   all_windows = REMOB(fw,all_windows);

   UNPROTECT;

   fw->window = NULL;
   if (fw->text_win != NULL && ASHinq_valid_window(fw->text_win))
      ASHremove(fw->text_win);
};





/************************************************************************/
/*									*/
/*	request_system -- request system name from user 		*/
/*									*/
/************************************************************************/


static String
request_system(w)
   ASH_WINDOW w;
{
   String systems[32];
   register Integer i,ct,sct;
   Character btn[32][64];
   Character nbuf[64];
   String menu[40];
   Integer val;
   String name;

   sct = FIELDinq_systems(32,systems);
   if (sct == 1) return systems[0];

   ct = 0;
   menu[ct++] = "%CFlow Graph Viewer\n";

   menu[ct++] = "System: %1.32t\n";

   for (i = 0; i < sct; ++i) {
      sprintf(btn[i],"    %%0.%do %s",i,systems[i]);
      menu[ct++] = btn[i];
    };

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   val = 0;
   nbuf[0] = 0;

   if (!STEMdialog(w,menu,&val,nbuf)) return NULL;

   if (nbuf[0] != 0 && nbuf[0] != ' ') {
      name = SALLOC(nbuf);
    }
   else if (val < sct) name = systems[val];
   else name = NULL;

   return name;
};





/* end of flowwin.c */
