/************************************************************************/
/*									*/
/*		flowinfo.c						*/
/*									*/
/*	Routines to manage information window for flow graph viewer	*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "flow_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_COMPONENTS		256





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		directory_info();
static	void		file_info();
static	void		function_info();
static	int		arc_compare();
static	int		flow_info_control();
static	void		flow_info_mouse();
static	Boolean 	prefix();
static	void		clear_info_line();
static	void		select_node();
static	FLOW_NODE	find_info_function();
static	FLOW_NODE	find_info_fct();
static	FLOW_ARC	find_call();





/************************************************************************/
/*									*/
/*	FLOW_info_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
FLOW_info_init()
{
};





/************************************************************************/
/*									*/
/*	FLOW_info_setup -- setup information window for browser 	*/
/*									*/
/************************************************************************/


void
FLOW_info_setup(fw)
   FLOW_WIN fw;
{
   ASH_WINDOW w;

   w = ASHroot_window(ASHinq_display_name(fw->window),"flow_info");
   w = WINDsetup(w);

 if (w == NULL) return;

   fw->text_win = w;
   ASHset_control(fw->text_win,flow_info_control);

   fw->edit_info = EDTedit_output(fw->text_win,fw);
   if (fw->edit_info == NULL) {
      fw->text_win = NULL;
      if (ASHinq_valid_window(w)) ASHremove(w);
      return;
    };

   EDTset_mouse_rtn(fw->edit_info,flow_info_mouse);
   EDTenable(fw->edit_info,FALSE);
   EDTprohibit_close(fw->edit_info);

   ASHvisible(w,TRUE);

   if (fw->selection != NULL) FLOW_info_node(fw,fw->selection);
};





/************************************************************************/
/*									*/
/*	FLOW_info_node -- give information on given node		*/
/*									*/
/************************************************************************/


void
FLOW_info_node(fw,fn)
   FLOW_WIN fw;
   FLOW_NODE fn;
{
   Character info[40960];
   Integer i;

   if (fw->text_win == NULL || fw->edit_info == NULL) return;
   if (fn == NULL) return;
   if (fn->infoline > 0) {
      EDTgoto_top(fw->edit_info,fn->infoline);
      return;
    };

   switch (fn->type) {
      case FLOW_TYPE_DIRECTORY :
	 directory_info(fw,fn,info);
	 break;

      case FLOW_TYPE_FILE :
	 file_info(fw,fn,info);
	 break;

      case FLOW_TYPE_FUNCTION :
	 function_info(fw,fn,info);
	 break;
    };

   strcat(info,"\n\n------------------------------------------------------\n");

   i = EDTlast_line(fw->edit_info);
   fn->infoline = i;
   EDTenable(fw->edit_info,TRUE);
   EDTinsert_line(fw->edit_info,i,info);
   EDTgoto_top(fw->edit_info,i);
   EDTenable(fw->edit_info,FALSE);
};





/************************************************************************/
/*									*/
/*	directory_info -- generate information for a directory		*/
/*									*/
/************************************************************************/


static void
directory_info(fw,fn,info)
   FLOW_WIN fw;
   FLOW_NODE fn;
   String info;
{
   Character buf[2048];
   Integer i;
   FLOW_NODE sf;

   sprintf(info,"Directory: %s\n",fn->name);

   if (fn->parent != NULL) {
      sprintf(buf,"\n   Contained in Directory: %s\n",fn->parent->name);
      strcat(info,buf);
    };

   i = 0;
   for (sf = fn->son; sf != NULL; sf = sf->brother) {
      if (i >= MAX_COMPONENTS) {
	 strcat(info,"      ...\n");
	 break;
       };
      if (i == 0) strcat(info,"\n   Includes:\n");
      if (sf->type == FLOW_TYPE_DIRECTORY)
	 sprintf(buf,"      Directory:  %s\n",sf->name);
      else
	 sprintf(buf,"      File:       %s\n",sf->name);
      strcat(info,buf);
      ++i;
    };
};





/************************************************************************/
/*									*/
/*	file_info -- create information display for a file		*/
/*									*/
/************************************************************************/


static void
file_info(fw,fn,info)
   FLOW_WIN fw;
   FLOW_NODE fn;
   String info;
{
   Character buf[2048];
   Integer i;
   FLOW_NODE sf;

   sprintf(info,"File: %s\n",fn->name);

   if (fn->parent != NULL) {
      sprintf(buf,"\n   Contained in Directory: %s\n",fn->parent->name);
      strcat(info,buf);
    };

   i = 0;
   for (sf = fn->son; sf != NULL; sf = sf->brother) {
      if (i >= MAX_COMPONENTS) {
	 strcat(info,"      ...\n");
	 break;
       };
      if (i == 0) strcat(info,"\n   Includes:\n");
      sprintf(buf,"      Routine:  %s\n",sf->name);
      strcat(info,buf);
      ++i;
    };
};





/************************************************************************/
/*									*/
/*	function_info -- create information display for a function	*/
/*									*/
/************************************************************************/


static void
function_info(fw,fn,info)
   FLOW_WIN fw;
   FLOW_NODE fn;
   String info;
{
   Character buf[2048];
   Integer i,ct;
   FLOW_ARC fa;
   FLOW_ARC arcs[MAX_COMPONENTS];

   sprintf(info,"Routine: %s\n",fn->name);

   if (fn->parent != NULL) {
      sprintf(buf,"\n   Defined at line %d in file ",fn->line);
      strcat(info,buf);
      FLOW_node_file(fn->parent,buf);
      strcat(info,buf);
      strcat(info,"\n");
    };

   ct = 0;
   for (i = 0; i < fw->numconn; ++i) {
      fa = &fw->conns[i];
      if (fa->from == fn) arcs[ct++] = fa;
      if (ct >= MAX_COMPONENTS) break;
    };

   if (ct > 0) {
      qsort(arcs,ct,sizeof(FLOW_ARC),arc_compare);
      strcat(info,"\n   Calls:\n");
      for (i = 0; i < ct; ++i) {
	 sprintf(buf,"      %s   at line %d\n",arcs[i]->to->name,arcs[i]->line);
	 strcat(info,buf);
       };
      if (ct >= MAX_COMPONENTS) strcat(info,"      ...\n");
      strcat(info,"\n");
    };

   ct = 0;
   for (i = 0; i < fw->numconn; ++i) {
      fa = &fw->conns[i];
      if (fa->to == fn) arcs[ct++] = fa;
      if (ct >= MAX_COMPONENTS) break;
    };

   if (ct > 0) {
      qsort(arcs,ct,sizeof(FLOW_ARC),arc_compare);
      strcat(info,"\n   Called By:\n");
      for (i = 0; i < ct; ++i) {
	 sprintf(buf,"      %s\n         at line %d in file  ",arcs[i]->from->name,arcs[i]->line);
	 strcat(info,buf);
	 FLOW_node_file(arcs[i]->from->parent,buf);
	 strcat(info,buf);
	 strcat(info,"\n");
       };
      if (ct >= MAX_COMPONENTS) strcat(info,"      ...\n");
      strcat(info,"\n");
    };
};



static int
arc_compare(fa1,fa2)
   FLOW_ARC * fa1;
   FLOW_ARC * fa2;
{
   Integer i;

   i = strcmp((*fa1)->from->name,(*fa2)->from->name);
   if (i == 0) {
      i = strcmp((*fa1)->to->name,(*fa2)->to->name);
    };

   if (i == 0) {
      i = (*fa1)->line - (*fa2)->line;
      if (i < 0) i = -1;
      else if (i > 0) i = 1;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	flow_info_control -- handle control messages on window		*/
/*									*/
/************************************************************************/


static int
flow_info_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   FLOW_WIN fw;
   FLOW_NODE fn;

   fw = FLOW_find_window(w);
   if (fw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      fw->text_win = NULL;
      fw->edit_info = NULL;
      for (fn = fw->root; fn != NULL; fn = fn->brother) clear_info_line(fn);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	flow_info_mouse -- handle mouse hits in information window	*/
/*									*/
/************************************************************************/


static void
flow_info_mouse(fw,btn,l,c,fl,fc,tl,tc)
   FLOW_WIN fw;
   Integer btn;
   Integer l,c;
   Integer fl,fc;
   Integer tl,tc;
{
   String b;
   Character lbuf[10240];
   String s,t;
   Integer spct,lno,i;
   Character fbuf[1024];
   FLOW_NODE fn;
   FLOW_ARC fa;

   if (btn & RIP_BTN_LEFT) b = "LEFT";
   else if (btn & RIP_BTN_MID) b = "MID";
   else if (btn & RIP_BTN_RIGHT) b = "RIGHT";
   else b = "NONE";

   if (l == 0) {
      l = EDTcur_line(fw->edit_info);
      c = EDTcur_char(fw->edit_info);
    };

   EDTinq_line_text(fw->edit_info,l,lbuf);
   i = strlen(lbuf);
   if (lbuf[i-1] == '\n') lbuf[--i] = 0;

   spct = 0;
   for (s = lbuf; isspace(*s); ++s) ++spct;

   if (*s == 0) return;

   if (prefix(s,"Defined at line")) {
      if (sscanf(s,"Defined at line %d in file %s",&lno,fbuf) == 2) {
	 MSGsenda("FLOW SET %s %d %s",fbuf,lno,b);
       };
      fn = FLOW_node_find_function(fw,fbuf,NULL);
      select_node(fw,fn,NULL);
    }
   else if (prefix(s,"Directory: ")) {
      if (sscanf(s,"Directory: %s",fbuf) == 1) {
	 fn = FLOW_node_find_function(fw,fbuf,NULL);
	 select_node(fw,fn,b);
       };
    }
   else if (prefix(s,"File: ")) {
      if (sscanf(s,"File: %s",fbuf) == 1) {
	 fn = FLOW_node_find_function(fw,fbuf,NULL);
	 select_node(fw,fn,b);
       };
    }
   else if (prefix(s,"Routine: ")) {
      if (sscanf(s,"Routine: %s",fbuf) == 1) {
	 fn = find_info_function(fw,l);
	 select_node(fw,fn,b);
       };
    }
   else if (prefix(s,"Includes:") || prefix(s,"Calls:") || prefix(s,"Called By:")) ;
   else if (prefix(s,"at line ")) {
      if (sscanf(s,"at line %d in file %s",&lno,fbuf) == 2) {
	 MSGsenda("FLOW SET %s %d %s",fbuf,lno,b);
       };
    }
   else {
      t = s;
      while (*t != 0 && !isspace(*t)) ++t;
      lno = 0;
      if (*t != 0) {
	 *t++ = 0;
	 while (isspace(*t)) ++t;
	 sscanf(t,"at line %d",&lno);
       };

      fn = find_info_function(fw,l);

      if (*t == 0) {
	 fa = find_call(fw,fn,s,FALSE,lno);
	 select_node(fw,fa->from,b);
       }
      else {
	 fa = find_call(fw,fn,s,TRUE,lno);
	 if (fa == NULL) ;
	 else if (c > spct + strlen(s) + 2) {
	    FLOW_node_file(fa->from->parent,fbuf);
	    MSGsenda("FLOW SET %s %d %s",fbuf,fa->line,b);
	    select_node(fw,fa->to,NULL);
	  }
	 else select_node(fw,fa->to,b);
       };
    };
};






/************************************************************************/
/*									*/
/*	prefix -- check for string prefix				*/
/*									*/
/************************************************************************/


static Boolean
prefix(s,pfx)
   String s;
   String pfx;
{
   Integer i;

   i = strlen(pfx);

   return (strncmp(s,pfx,i) == 0);
};





/************************************************************************/
/*									*/
/*	clear_info_line -- note information window removed		*/
/*									*/
/************************************************************************/


static void
clear_info_line(fn)
   FLOW_NODE fn;
{
   FLOW_NODE sf;

   fn->infoline = -1;

   for (sf = fn->son; sf != NULL; sf = sf->brother) clear_info_line(sf);
};





/************************************************************************/
/*									*/
/*	select_node -- handle user selection of a node			*/
/*									*/
/************************************************************************/


static void
select_node(fw,fn,xref)
   FLOW_WIN fw;
   FLOW_NODE fn;
   String xref;
{
   Character buf[1024];

   FLOW_disp_select(fw,fn);

   if (xref != NULL) {
      FLOW_node_file(fn,buf);
      MSGsenda("FLOW SET %s %d %s",buf,fn->line,xref);
    };
};





/************************************************************************/
/*									*/
/*	find_info_function -- find function we are getting info on	*/
/*									*/
/************************************************************************/


static FLOW_NODE
find_info_function(fw,l)
   FLOW_WIN fw;
   Integer l;
{
   FLOW_NODE fn,sf;

   sf = NULL;
   for (fn = fw->root; fn != NULL; fn = fn->brother) {
      sf = find_info_fct(fn,l,sf);
    };

   return sf;
};





static FLOW_NODE
find_info_fct(fn,l,sf)
   FLOW_NODE fn;
   Integer l;
   FLOW_NODE sf;
{
   FLOW_NODE fn1;
   Integer i;

   if (fn->son != NULL) {
      for (fn1 = fn->son; fn1 != NULL; fn1 = fn1->brother) {
	 sf = find_info_fct(fn1,l,sf);
       };
    }
   else {
      i = (sf == NULL ? 0 : sf->infoline);
      if (fn->infoline > 0 && fn->infoline > i && fn->infoline <= l) sf = fn;
    }

   return sf;
};





/************************************************************************/
/*									*/
/*	find_call -- find call reference by text line			*/
/*									*/
/************************************************************************/


static FLOW_ARC
find_call(fw,fn,s,from,line)
   FLOW_WIN fw;
   FLOW_NODE fn;
   String s;
   Boolean from;
   Integer line;
{
   Integer i;
   FLOW_NODE fn1;
   FLOW_ARC fa;

   for (i = 0; i < fw->numconn; ++i) {
      fa = &fw->conns[i];
      if (from) {
	 if (fa->from != fn) continue;
	 fn1 = fa->to;
       }
      else {
	 if (fa->to != fn) continue;
	 fn1 = fa->from;
       };
      if (line != 0 && fa->line != line) continue;
      if (STRNEQ(fn1->name,s)) continue;
      return fa;
    };

   return NULL;
};





/* end of flowinfo.c */
