/************************************************************************/
/*									*/
/*		dispddt.c						*/
/*									*/
/*	DDT interface for data structure display routines		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "disp_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Types							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	DISP_CACHE	cache;
static	Integer 	counter;
static	Integer 	last_addr;
static	Integer 	last_value;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		debug_query();





/************************************************************************/
/*									*/
/*	DISP_ddt_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DISP_ddt_init()
{
   cache = NULL;
   counter = 0;
   last_addr = 0;
};





/************************************************************************/
/*									*/
/*	DISP_var_info -- get information on variable			*/
/*									*/
/************************************************************************/


Boolean
DISP_var_info(dw,file,func,name)
   DISP_WIN dw;
   String file;
   String func;
   String name;
{
   String s,t;
   Character buf[1024];

   s = debug_query("VARINFO",dw->system,file,func,name);

   if (s == NULL || *s == 0 || *s == '"' ||
	  strncmp(s,"no module, ",11) == 0 ||
	  strncmp(s,"syntax error on ",16) == 0) {
      if (s == NULL || *s == 0) s = "variable not defined";
      sprintf(buf,"%%CProblem displaying %s:\n\n%s\n\n%%M   %%a",name,s);
      if (dw->window != NULL) STEMdialog1(dw->window,buf);
      return FALSE;
    };

   t = index(s,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;
   if (dw->file == NULL || STRNEQ(s,dw->file)) {
      if (dw->file != NULL) SFREE(dw->file);
      dw->file = SALLOC(s);
    };
   s = t;

   t = index(s,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;
   if (dw->func == NULL || STRNEQ(s,dw->func)) {
      if (dw->func != NULL) SFREE(dw->func);
      dw->func = SALLOC(s);
    };
   s = t;

   t = index(s,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;
   if (dw->variable == NULL || STRNEQ(s,dw->variable)) {
      if (dw->variable != NULL) SFREE(dw->variable);
      dw->variable = SALLOC(s);
    };
   s = t;

   t = index(s,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;
   s = t;

   t = index(s,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;
   dw->var_addr = atol(s);
   if (dw->var_addr == 0) {
      ++counter;
      dw->var_addr = UNDEF_ADDR(counter);
    };
   s = t;

   if (strncmp(s,"record ",7) == 0) dw->var_type = NULL;
   else dw->var_type = DISP_find_type(dw,dw->file,dw->func,s);

   if (dw->var_type == NULL) {
      if (dw->variable != NULL) SFREE(dw->variable);
      dw->variable = NULL;
      sprintf(buf,"%%CProblem displaying %s:\n\nType can't be displayed\n\n%%M   %%a",
		 name);
      if (dw->window != NULL) STEMdialog1(dw->window,buf);
      return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DISP_var_addr -- get address of variable			*/
/*									*/
/************************************************************************/


Integer
DISP_var_addr(addr)
   Integer addr;
{
   DISP_WIN dw;
   String s,t;
   Integer i;

   if (!TEST_UNDEF(addr)) return addr;
   if (last_addr == addr) return last_value;

   dw = DISP_find_window_by_addr(addr);
   if (dw == NULL) return 0;

   s = debug_query("VARINFO",dw->system,dw->file,dw->func,dw->variable);

   if (s == NULL || *s == 0) return 0;

   t = index(s,' ');
   if (t == NULL) return 0;
   *t++ = 0;
   s = t;

   t = index(s,' ');
   if (t == NULL) return 0;
   *t++ = 0;
   s = t;

   t = index(s,' ');
   if (t == NULL) return 0;
   *t++ = 0;
   s = t;

   t = index(s,' ');
   if (t == NULL) return 0;
   *t++ = 0;
   s = t;

   t = index(s,' ');
   if (t == NULL) return 0;
   *t++ = 0;
   i = atol(s);

   last_addr = addr;
   last_value = i;

   return i;
};





/************************************************************************/
/*									*/
/*	DISP_find_type -- find type for item				*/
/*									*/
/************************************************************************/


DISP_TYPE
DISP_find_type(dw,file,func,name)
   DISP_WIN dw;
   String file;
   String func;
   String name;
{
   DISP_TYPE dt;
   DISP_FIELD df;
   String s,t;
   Character buf[10240];
   Integer i;

   dt = DISP_find_defined_type(dw,name);

   if (dt != NULL && dt->class != DISP_TYPE_UNKNOWN) return dt;

   if (dt == NULL) {
      dt = PALLOC(DISP_TYPE_INFO);
      dt->system = (dw->system == NULL ? NULL : SALLOC(dw->system));
      dt->name = SALLOC(name);
      dt->file = (file == NULL ? NULL : SALLOC(file));
      dt->func = (func == NULL ? NULL : SALLOC(func));
      dt->drawinfo = NULL;
      dt->nested = dw->nested;
      dt->usedb = dw->usedb;
      dt->user = (dw->show_type != DISP_SHOW_PEAR);
      dt->usecount = 1;
      dt->enumbase = NULL;
      dw->all_types = CONS(dt,dw->all_types);
    };

   s = debug_query("TYPEINFO",dw->system,file,func,name);

   if (s == NULL || *s == 0 || index(s,' ') == NULL) {
      dt->class = DISP_TYPE_UNKNOWN;
      return dt;
    };

   strcpy(buf,s);
   s = buf;

   t = index(s,' ');
   *t++ = 0;

   if (STREQL(s,"PRIMITIVE")) {
      dt->class = DISP_TYPE_PRIMITIVE;
    }
   else if (STREQL(s,"SAME")) {
      dt->class = DISP_TYPE_SAME;
      dt->base = DISP_find_type(dw,file,func,t);
    }
   else if (STREQL(s,"ENUM")) {
      dt->class = DISP_TYPE_ENUM;
      dt->count = atol(t);
      if (dt->count > 0) {
	 t = index(t,' ');
	 if (t != NULL) {
	    ++t;
	    dt->enumbase = SALLOC(t);
	  };
       };
    }
   else if (STREQL(s,"ARRAY")) {
      dt->class = DISP_TYPE_ARRAY;
      dt->count0 = atol(t);
      t = index(t,' ');
      ++t;
      dt->count = atol(t);
      t = index(t,' ');
      ++t;
      if (STREQL(t,"char") && dt->count > 5) {
	 dt->class = DISP_TYPE_PRIMITIVE;
       }
      else dt->base = DISP_find_type(dw,file,func,t);
    }
   else if (STREQL(s,"POINTER")) {
      dt->class = DISP_TYPE_POINTER;
      dt->base = DISP_find_type(dw,file,func,t);
    }
   else if (STREQL(s,"RECORD")) {
      dt->class = DISP_TYPE_RECORD;
      dt->count = atol(t);
    }
   else if (STREQL(s,"UNION")) {
      dt->class = DISP_TYPE_UNION;
      dt->count = atol(t);
    }
   else if (STREQL(s,"VARREC")) {
      dt->class = DISP_TYPE_VARREC;
    }
   else if (STREQL(s,"SET")) {
      dt->class = DISP_TYPE_SET;
    }
   else if (STREQL(s,"FILE")) {
      dt->class = DISP_TYPE_FILE;
    }
   else {
      dt->class = DISP_TYPE_UNKNOWN;
    };

   if (dt->class == DISP_TYPE_UNION || dt->class == DISP_TYPE_RECORD) {
      t = index(t,'\n');
      if (t != NULL) ++t;
      while (t != NULL) {
	 while (isspace(*t)) ++t;
	 s = t;
	 t = index(t,':');
	 if (t == NULL) break;
	 *t++ = 0;
	 df = PALLOC(DISP_FIELD_INFO);
	 df->name = SALLOC(s);
	 df->base = dt;
	 s = t;
	 t = index(t,'\n');
	 if (t != NULL) *t++ = 0;
	 i = strlen(s);
	 while (i > 0 && isspace(s[i-1])) s[--i] = 0;
	 dt->fields = APPEND(df,dt->fields);
	 df->type = DISP_find_type(dw,file,func,s);
       };
    };

   return dt;
};





/************************************************************************/
/*									*/
/*	DISP_get_value -- get expression value				*/
/*									*/
/************************************************************************/


void
DISP_get_value(sys,expr,buf)
   String sys;
   String expr;
   String buf;
{
   String s,t;
   Integer i;
   DISP_CACHE dc;

   if (sys == NULL || sys[0] == 0 || STREQL(sys,"*")) sys = NULL;

   for (dc = cache; dc != NULL; dc = dc->next) {
      if (sys == dc->system ||
	     (sys != NULL && dc->system != NULL && STREQL(sys,dc->system))) {
	 if (STREQL(expr,dc->expr)) {
	    if (dc->value == NULL) buf[0] = 0;
	    else strcpy(buf,dc->value);
	    return;
	  };
       };
    };

   t = (sys == NULL ? "*" : sys);
   s = MSGcalla("DDTR EVAL %s %c%s%c",t,LIT_STRING,expr,LIT_STRING);

   if (s == NULL) buf[0] = 0;
   else {
      t = s;
      i = strlen(t);
      if (t[i-1] == '\n') t[i-1] = 0;
      if (STREQL(t,"reference through nil pointer")) t = "<< UNDEFINED >>";
      strcpy(buf,t);
      SFREE(s);
    };

   if (buf[0] == '\'') {
      s = buf;
      i = strlen(s);
      if (i > 4 && s[i-1] == '\'') {
	 i -= 1;
	 while (i > 0 && isspace(s[i-1])) --i;
	 s[i] = '\'';
	 s[i+1] = 0;
       };
    };

   dc = PALLOC(DISP_CACHE_INFO);
   dc->system = (sys == NULL ? NULL : SALLOC(sys));
   dc->expr = SALLOC(expr);
   dc->value = (buf[0] == 0 ? NULL : SALLOC(buf));

   PROTECT;
   dc->next = cache;
   cache = dc;
   UNPROTECT;

};





/************************************************************************/
/*									*/
/*	DISP_set_value -- do an assignment				*/
/*									*/
/************************************************************************/


Boolean
DISP_set_value(sys,lhs,rhs)
   String sys;
   String lhs;
   String rhs;
{
   String s;

   if (sys == NULL || sys[0] == 0) sys = "*";
   s = MSGcalla("DDTR ASSIGN %s %c%s%c %c%s%c",sys,
		   LIT_STRING,lhs,LIT_STRING,
		   LIT_STRING,rhs,LIT_STRING);

   if (s != NULL && *s != 0) {
      SFREE(s);
      return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DISP_flush_cache -- flush cached values 			*/
/*									*/
/************************************************************************/


void
DISP_flush_cache(dw)
   DISP_WIN dw;
{
   DISP_CACHE ce,nce,lce;
   String sys;

   if (dw == NULL) sys = NULL;
   else sys = dw->system;

   nce = cache;
   cache = NULL;
   lce = NULL;
   last_addr = 0;

   PROTECT;
   for (ce = nce; ce != NULL; ce = nce) {
      nce = ce->next;
      if (ce->system != NULL && sys != NULL && STRNEQ(ce->system,sys)) {
	 if (lce == NULL) cache = ce;
	 else lce->next = ce;
	 lce = ce;
	 ce->next = NULL;
       }
      else {
	 if (ce->system != NULL) SFREE(ce->system);
	 if (ce->expr != NULL) SFREE(ce->expr);
	 if (ce->value != NULL) SFREE(ce->value);
	 free(ce);
       };
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	debug_query -- ask question of debugger 			*/
/*									*/
/************************************************************************/


static String
debug_query(qt,sys,file,func,name)
   String qt;
   String sys,file,func,name;
{
   String s;
   Integer ln;

   if (file == NULL || *file == 0) file = "*";
   if (func == NULL || *func == 0) func = "*";
   if (sys == NULL || *sys == 0) sys = "*";

   s = MSGcalla("DDTR SYMINFO %s %s %s %s 0 %c%s%c",sys,qt,file,func,
		   LIT_STRING,name,LIT_STRING);

   if (*s == 0) s = NULL;
   if (s != NULL) {
      ln = strlen(s);
      if (s[ln-1] == '\n') s[--ln] = 0;
      while (ln > 0 && s[ln-1] == ' ') s[--ln] = 0;
      if (ln == 0) s = NULL;
    };

   return s;
};





/* end of dispddt.c */
