/************************************************************************/
/*									*/
/*		ddttrace.c						*/
/*									*/
/*	Maintain run time state information for ddt interface		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_STACK_SIZE	32
#define MAX_SMALL_GLOBALS	32





/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _FRAME * 	FRAME;
typedef struct _FRAME_VAR *	FRAME_VAR;

typedef struct _FRAME {
   Integer level;
   String file;
   String func;
   Integer line;
   Integer addr;
   String args;
   FRAME_VAR vars;
   Integer displevel;
   Boolean ignore;
} FRAME_INFO;




typedef struct _FRAME_VAR {
   FRAME_VAR next;
   String name;
   String value;
} FRAME_VAR_INFO;




typedef enum {
   WHERE_NONE,
   WHERE_CALL,
   WHERE_DUMP,
} WHERE_STATE;



typedef enum {
   GBLS_NONE,
   GBLS_SMALL,
   GBLS_SMALL_EXT,
   GBLS_ALL,
   GBLS_ALL_EXT,
} GBLS_STATE;




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	FRAME_INFO *	cur_stack;
static	Integer 	stack_len;
static	Integer 	stack_max;

static	Integer 	auto_top;
static	Integer 	auto_bot;
static	Boolean 	auto_dump;
static	GBLS_STATE	auto_global;
static	Boolean 	ignore_temps;

static	Boolean 	update_on_stop;
static	Boolean 	no_main;

static	WHERE_STATE	status;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		check_where_msgs();
static	void		set_where_focus();
static	void		get_current();
static	void		print_frame();
static	void		msg_frame();
static	Boolean 	is_temp();





/************************************************************************/
/*									*/
/*	DDT_trace_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
DDT_trace_init()
{
   stack_max = DEFAULT_STACK_SIZE;
   cur_stack = (FRAME) calloc(stack_max,sizeof(FRAME_INFO));
   stack_len = 0;

   auto_top = 0;
   auto_bot = 0;
   auto_dump = FALSE;
   auto_global = GBLS_SMALL;
   ignore_temps = TRUE;

   update_on_stop = FALSE;
   no_main = FALSE;

   status = WHERE_NONE;
};





/************************************************************************/
/*									*/
/*	DDT_trace_set_where -- someone wants WHERE messages sent out	*/
/*									*/
/************************************************************************/


Boolean
DDT_trace_set_where(what,arg)
   String what;
   String arg;
{
   Integer val;

   if (STREQL(what,"STACK_TOP")) auto_top = atol(arg);
   else if (STREQL(what,"STACK_BOTTOM")) auto_bot = atol(arg);
   else if (STREQL(what,"STACK_DUMP")) auto_dump = atol(arg);
   else if (STREQL(what,"STACK_GLOBAL")) auto_global = (GBLS_STATE) atol(arg);
   else if (STREQL(what,"STACK_SHOW")) {
      val = atol(arg);
      if ((val & 4) != 0) {
	 if (auto_global == GBLS_SMALL) auto_global = GBLS_SMALL_EXT;
	 else if (auto_global == GBLS_ALL) auto_global = GBLS_ALL_EXT;
       };
      val &= 3;
      if (val == 0) auto_top = 0;
      else if (auto_top == 0) auto_top = 1024;
      if (val > 1) auto_dump = TRUE;
      check_where_msgs();
    }
   else if (STREQL(what,"STOP_UPDATE")) update_on_stop = atol(arg);
   else if (STREQL(what,"STACK_NO_MAIN")) no_main = atol(arg);
   else if (STREQL(what,"IGNORE_TEMPS")) ignore_temps = atol(arg);
   else return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_trace_print_where -- print current stack			*/
/*									*/
/************************************************************************/


void
DDT_trace_print_where(top,bot,dump)
   Integer top;
   Integer bot;
   Boolean dump;
{
   register Integer i;

   get_current(dump,auto_global);

   if (dump && top != 0 && top >= stack_len) top = stack_len-1;

   if (top > stack_len || top == 0) top = stack_len;
   i = stack_len - top;
   if (bot > i) bot = i;

   for (i = 0; i < top; ++i) {
      print_frame(&cur_stack[i],dump);
    };

   for (i = 0; i < bot; ++i) {
      print_frame(&cur_stack[stack_len-i-1],dump);
    };
};





/************************************************************************/
/*									*/
/*	DDT_trace_where -- set a portion of the call stack		*/
/*									*/
/************************************************************************/


void
DDT_trace_where(level,file,func,line,addr,args)
   Integer level;
   String file;
   String func;
   Integer line;
   Integer addr;
   String args;
{
   FRAME f;

   while (level > stack_max) {
      stack_max *= 2;
      cur_stack = (FRAME_INFO *) realloc(cur_stack,stack_max*sizeof(FRAME_INFO));
    };

   if (level > stack_len) stack_len = level;

   f = &cur_stack[level-1];

   f->level = level;
   f->file = SALLOC(file);
   f->func = SALLOC(func);
   f->line = line;
   f->addr = addr;
   f->args = SALLOC(args);
   f->vars = NULL;
   f->ignore = FALSE;

   if (level != 1 && no_main && STREQL(func,"main") && line == 0 &&
	  (file == NULL || file[0] == 0 || STREQL(file,"*")))
      f->ignore = TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_trace_where_top -- set current top of stack 		*/
/*									*/
/************************************************************************/


void
DDT_trace_where_top(lvl)
   Integer lvl;
{
   stack_len = lvl;
};





/************************************************************************/
/*									*/
/*	DDT_trace_dump -- set local variable for stack level		*/
/*									*/
/************************************************************************/


void
DDT_trace_dump(level,var,value)
   Integer level;
   String var;
   String value;
{
   FRAME f;
   FRAME_VAR v,lv,nv;

   f = &cur_stack[level-1];

   v = PALLOC(FRAME_VAR_INFO);
   v->name = SALLOC(var);
   v->value = SALLOC(value);

   lv = NULL;
   for (nv = f->vars; nv != NULL; nv = nv->next) {
      if (STRLSS(var,nv->name)) break;
      lv = nv;
    };

   v->next = nv;
   if (lv == NULL) f->vars = v;
   else lv->next = v;
};





/************************************************************************/
/*									*/
/*	DDT_trace_process -- do trace update at stop point		*/
/*									*/
/************************************************************************/


void
DDT_trace_process()
{
   LOCATION loc;

   status = WHERE_NONE;
   DDT_x_stop();
   DDT_x_set_view(NULL,NULL,0);
   check_where_msgs();
   set_where_focus();

   if (update_on_stop) {
      DDT_model_inq_location(&loc);
      DDT_msg_update(loc.file,loc.line);
    };
};





/********************************************************************************/
/*										*/
/*	DDT_trace_check_if_in -- check if inside given function 		*/
/*										*/
/********************************************************************************/


Boolean
DDT_trace_check_if_in(fct,nest)
   String fct;
   Boolean nest;
{
   Integer i,mx;

   get_current(FALSE,GBLS_NONE);

   mx = (nest ? stack_len : 1);

   for (i = 0; i < mx; ++i) {
      if (cur_stack[i].func != NULL && STREQL(cur_stack[i].func,fct))
	 return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	check_where_msgs -- check for and send stack messages		*/
/*									*/
/************************************************************************/


static void
check_where_msgs()
{
   register Integer i,j;
   Integer top,bot;

   if (auto_top == 0) return;

   DDT_msg_where_begin();

   get_current(auto_dump,auto_global);

   j = 0;
   for (i = 0; i < stack_len; ++i) {
      cur_stack[i].displevel = cur_stack[i].level - j;
      if (cur_stack[i].ignore) ++j;
    };

   top = auto_top;
   bot = auto_bot;
   if (top > stack_len) top = stack_len;
   i = stack_len - top;
   if (bot > i) bot = i;

   for (i = 0; i < top; ++i) {
      msg_frame(&cur_stack[i],auto_dump);
    };

   for (i = 0; i < bot; ++i) {
      msg_frame(&cur_stack[stack_len-i-1],auto_dump);
    };

   DDT_msg_where_end(stack_len+1-j);
};





/************************************************************************/
/*									*/
/*	set_where_focus -- set focus from stack info			*/
/*									*/
/************************************************************************/


static void
set_where_focus()
{
   Integer i;
   LOCATION loc;

   get_current(FALSE,GBLS_NONE);

   for (i = 0; i < stack_len; ++i) {
      if (cur_stack[i].file != NULL && cur_stack[i].line > 0) {
	 DDT_x_set_view(cur_stack[i].file,cur_stack[i].func,
			   cur_stack[i].line);
	 DDT_model_inq_location(&loc);
	 DDT_msg_src_at_line(&loc);
	 break;
       };
    };
};





/************************************************************************/
/*									*/
/*	get_current -- get current stack				*/
/*									*/
/************************************************************************/


static void
get_current(dump,gbls)
   Boolean dump;
   GBLS_STATE gbls;
{
   Boolean glbl;

   if (gbls == GBLS_NONE) glbl = FALSE;
   else if (gbls == GBLS_ALL || gbls == GBLS_ALL_EXT) glbl = TRUE;
   else if (DDT_symbol_variable_count() > MAX_SMALL_GLOBALS) glbl = FALSE;
   else glbl = TRUE;

   if (dump && status != WHERE_DUMP) {
      DDT_x_where(TRUE,glbl);
      status = WHERE_DUMP;
    }
   else if (!dump && status == WHERE_NONE) {
      DDT_x_where(FALSE,FALSE);
      status = WHERE_CALL;
    };
};





/************************************************************************/
/*									*/
/*	print_frame -- print given frame				*/
/*									*/
/************************************************************************/


static void
print_frame(f,dump)
   FRAME f;
   Boolean dump;
{
   FRAME_VAR v;

   if (f->ignore) return;

   if (f->line != 0) {
      DDT_mprint("%s(%s) in line %d of %s\n",f->func,f->args,f->line,f->file);
    }
   else if (f->addr != 0) {
      DDT_mprint("%s(%s) at 0x%x\n",f->func,f->args,f->addr);
    }
   else if (f->func != NULL && STRNEQ(f->func,"*")) {
      DDT_mprint("%s(%s)\n",f->func,f->args);
    }
   else {
      if (dump && auto_global != GBLS_NONE) DDT_mprint("GLOBALS:\n");
      else dump = FALSE;
    };

   if (dump) {
      for (v = f->vars; v != NULL; v = v->next) {
	 if (!ignore_temps || !is_temp(v->name)) {
	    DDT_mprint("\t%s\t = %s\n",v->name,v->value);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	msg_frame -- print message for given frame			*/
/*									*/
/************************************************************************/


static void
msg_frame(f,dump)
   FRAME f;
   Boolean dump;
{
   FRAME_VAR v;
   Integer ct;

   if (f->ignore) return;

   if (f->line == 0 && f->addr == 0 && (f->func == NULL || STREQL(f->func,"*"))) {
      if (auto_global == GBLS_NONE || auto_global == GBLS_SMALL || auto_global == GBLS_ALL)
	 return;
      if (auto_global == GBLS_SMALL_EXT && DDT_symbol_variable_count() > MAX_SMALL_GLOBALS)
	 return;
    };

   DDT_msg_where(f->displevel,f->file,f->func,f->line,f->addr,f->args);

   if (dump) {
      ct = 0;
      for (v = f->vars; v != NULL; v = v->next) {
	 if (!ignore_temps || !is_temp(v->name)) {
	    ++ct;
	    DDT_msg_where_dump(f->level,v->name,v->value,ct);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	is_temp -- check if variable name should be ignored temporary	*/
/*									*/
/************************************************************************/


static Boolean
is_temp(name)
   String name;
{
   if (DDT__cplusplus == CPLUSPLUS20) {
      if (strncmp(name,"__V",3) == 0) {
	 name += 3;
	 while (isdigit(*name)) ++name;
	 if (*name == 0) return TRUE;
       }
      else if (strncmp(name,"__X",3) == 0) {
	 name += 3;
	 while (*name != 0) {
	    if (*name == '0' && name[1] == '0') return TRUE;
	    ++name;
	  };
       };
    };

   return FALSE;
};





/* end of ddttrace.c */

