/************************************************************************/
/*									*/
/*		ddtsyms.c						*/
/*									*/
/*	Symbol handling routines for the ddt debugger interface 	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"
#include "ddt_names.h"

#include <hashsearch.h>
#include <fsym.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#ifdef mips
#define NO_UNDERSCORE
#endif




/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	FILE_INFO *	sys_files;
static	Integer 	num_files;
static	FUN_INFO *	sys_functions;
static	Integer 	num_functions;
static	VAR_INFO *	sys_variables;
static	Integer 	num_variables;
static	TYPE_INFO *	sys_types;
static	Integer 	num_types;
static	Sequence	directorys;

static	Integer 	main_prog;

static	time_t		system_time;

static	HASH_TABLE	function_hash;
static	HASH_TABLE	type_hash;
static	HASH_TABLE	variable_hash;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	get_symbol_info();
static	void		get_sym_dir();
static	void		get_sym_so();
static	void		get_sym_tso();
static	void		get_sym_fun();
static	void		get_sym_line();
static	void		get_sym_sym();
static	void		get_sym_addr();
static	Integer 	add_function();
static	Integer 	add_variable();
static	Integer 	add_type();
static	Boolean 	check_file_date();
static	Boolean 	c_typedef_info();
static	Boolean 	c_struct_info();
static	Boolean 	c_enum_info();
static	Boolean 	isidchar();
static	void		fix_pascal_type();
static	void		pas_struct_info();
static	void		get_pascal_dims();
static	Boolean 	prefix();
static	String		find_file();
static	String		filetail();
static	void		add_directory();
static	Boolean 	filename_match();
static	void		fix_filename();





/************************************************************************/
/*									*/
/*	DDT_symbol_init -- module initialization			*/
/*									*/
/************************************************************************/


void
DDT_symbol_init()
{
   num_files = 0;
   sys_files = NULL;
   num_functions = 0;
   sys_functions = NULL;
   num_variables = 0;
   sys_variables = NULL;
   num_types = 0;
   sys_types = NULL;
   directorys = NULL;
   main_prog = -1;
   system_time = 0;
   function_hash = NULL;
   type_hash = NULL;
   variable_hash = NULL;
};






/************************************************************************/
/*									*/
/*	DDT_symbol_setup -- get system information			*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_setup(sys)
   String sys;
{
   Boolean fg;

   DDT_symbol_free_info();
   fg = get_symbol_info(sys);
   check_file_date(-1);

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_free_info -- free symbol information 		*/
/*									*/
/************************************************************************/


void
DDT_symbol_free_info()
{
   Integer i;
   Sequence l;
   String d;

   if (sys_files != NULL) {
      for (i = 0; i < num_files; ++i) {
	 if (sys_files[i].pname != sys_files[i].name) SFREE(sys_files[i].pname);
	 SFREE(sys_files[i].name);
       };
      free(sys_files);
      sys_files = NULL;
      num_files = 0;
    };

   if (sys_functions != NULL) {
      for (i = 0; i < num_functions; ++i) {
	 if (sys_functions[i].pname != sys_functions[i].name)
	    SFREE(sys_functions[i].pname);
	 SFREE(sys_functions[i].name);
       };
      free(sys_functions);
      sys_functions = NULL;
      num_functions = 0;
    };

   if (sys_variables != NULL) {
      for (i = 0; i < num_variables; ++i) SFREE(sys_variables[i].name);
      free(sys_variables);
      sys_variables = NULL;
      num_variables = 0;
    };

   if (sys_types != NULL) {
      for (i = 0; i < num_types; ++i) {
	 if (sys_types[i].pname != sys_types[i].name) SFREE(sys_types[i].pname);
	 SFREE(sys_types[i].name);
       };
      free(sys_types);
      sys_types = NULL;
      num_types = 0;
    };

   forin (d,String,l,directorys) SFREE(d);
   LFREE(directorys);
   directorys = NULL;

   if (function_hash != NULL) {
      HASHdestroy(function_hash);
      function_hash = NULL;
    };
   if (type_hash != NULL) {
      HASHdestroy(type_hash);
      type_hash = NULL;
    };
   if (variable_hash != NULL) {
      HASHdestroy(variable_hash);
      variable_hash = NULL;
    };
};





/************************************************************************/
/*									*/
/*	DDT_symbol_funct_info -- find file/line given function name	*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_funct_info(func,filp,linp,elinp)
   String func;
   String * filp;
   Integer * linp;
   Integer * elinp;
{
   Integer i;

   *filp = NULL;
   *linp = 0;
   *elinp = 0;

   for (i = 0; i < num_functions; ++i) {
      if (STREQL(func,sys_functions[i].name) || STREQL(func,sys_functions[i].pname)) {
	 if (sys_functions[i].file >= 0) {
	    if (linp != NULL) *linp = sys_functions[i].sline;
	    if (filp != NULL) *filp = sys_files[sys_functions[i].file].pname;
	    if (elinp != NULL) *elinp = sys_functions[i].eline;
	  };
	 return TRUE;
       };
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_funct_table -- return function table and count	*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_funct_table(ftbl)
   FUN_INFO ** ftbl;
{
   *ftbl = sys_functions;

   return num_functions;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_main_program -- fill in location of main program	*/
/*									*/
/************************************************************************/


void
DDT_symbol_main_program(loc,begin)
   LOCATION * loc;
   Boolean begin;
{
   if (main_prog < 0) {
      loc->file = NULL;
      loc->func = NULL;
      loc->line = 0;
      return;
    };

   if (sys_functions[main_prog].file < 0) loc->file = NULL;
   else loc->file = sys_files[sys_functions[main_prog].file].dname;

   loc->func = sys_functions[main_prog].pname;

   if (begin) loc->line = sys_functions[main_prog].sline;
   else loc->line = sys_functions[main_prog].eline;
};





/********************************************************************************/
/*										*/
/*	DDT_symbol_check_dates -- check dates on files before a run		*/
/*										*/
/********************************************************************************/


Boolean
DDT_symbol_check_dates()
{
   Boolean fg;
   Integer i;

   fg = TRUE;

   if (!check_file_date(-1)) {
      DDT_mprint("Binary %s has been rebuilt\n",DDT__system_name);
      DDT_x_reinit(NULL,NULL);
      DDT_model_clear(FALSE);
      CMPXselect(-2);
      while (DDT_command_process(FALSE));
      DDT_mprint("Debugger has been reinitialized\n");
      check_file_date(-1);
      fg = FALSE;
    };

   for (i = 0; i < num_files; ++i) {
      if (!check_file_date(i)) {
	 DDT_mprint("Source %s is newer than system %s\n",
		       sys_files[i].pname,DDT__system_name);
	 fg = FALSE;
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_files -- return list of files			*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_files(mx,fnms)
   Integer mx;
   String fnms[];
{
   Integer i;

   for (i = 0; i < num_files; ++i) {
      if (i >= mx) break;
      fnms[i] = sys_files[i].pname;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_functions -- return list of functions		*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_functions(mx,fnms,active)
   Integer mx;
   String fnms[];
   Boolean active;
{
   Integer i,ct;

   ct = 0;
   for (i = 0; i < num_functions; ++i) {
      if (ct >= mx) break;
      if (active) {
	 if (sys_functions[i].sline <= 0) continue;
	 fnms[ct++] = sys_functions[i].name;
       }
      else {
/*	 if (sys_functions[i].pname[0] == '#') continue;        */
	 fnms[ct++] = sys_functions[i].pname;
       };
    };

   return ct;
};






/************************************************************************/
/*									*/
/*	DDT_symbol_funnames -- return list of fcts matching prefix	*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_funnames(pfx,mx,fnms)
   String pfx;
   Integer mx;
   String fnms[];
{
   Integer i,ct,ln;

   if (pfx == NULL) ln = 0;
   else ln = strlen(pfx);

   ct = 0;
   for (i = 0; i < num_functions; ++i) {
      if (ln <= 0 || strncmp(pfx,sys_functions[i].name,ln) == 0) {
	 fnms[ct++] = sys_functions[i].name;
	 if (ct >= mx) break;
       };
    };

   return ct;
};






/************************************************************************/
/*									*/
/*	DDT_symbol_variables -- return list of variables		*/
/*	DDT_symbol_variable_count -- return count of variables		*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_variables(mx,vnms)
   Integer mx;
   String vnms[];
{
   Integer i;

   for (i = 0; i < num_variables; ++i) {
      if (i >= mx) break;
      vnms[i] = sys_variables[i].name;
    };

   return i;
};





Integer
DDT_symbol_variable_count()
{
   return num_variables;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_types -- return list of types			*/
/*									*/
/************************************************************************/


Integer
DDT_symbol_types(mx,vnms)
   Integer mx;
   String vnms[];
{
   Integer i;

   for (i = 0; i < num_types; ++i) {
      if (i >= mx) break;
      vnms[i] = sys_types[i].pname;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_inq_type_name -- check if name identifies a type	*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_inq_type_name(nm)
   String nm;
{
   Integer i;

   for (i = 0; i < num_types; ++i) {
      if (STREQL(nm,sys_types[i].pname)) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_fix_file_name -- get our file name from user name	*/
/*									*/
/************************************************************************/


String
DDT_symbol_fix_file_name(file)
   String file;
{
   String s,t;
   Integer i,j;
   Character buf[1024];

   if (file == NULL) return NULL;

   if (index(file,',') != NULL && index(file,'"') != NULL) {
      s = index(file,'"');
      ++s;
      strcpy(buf,s);
      s = index(buf,'"');
      if (s != NULL) *s = 0;
      file = buf;
    };

   s = rindex(file,'/');
   if (s == NULL) s = file;
   else s = ++s;
   for (i = 0; i < num_files; ++i) {
      t = sys_files[i].tail;
      if (STREQL(s,t) /* && MSGfile_compare(file,sys_files[i].name) */) break;
    };

   if (i < num_files) return sys_files[i].pname;

   t = file;

   if (DDT__cplusplus != 0) {
      if (access(t,4) < 0) {
	 j = strlen(t);
	 if (j > 3 && t[j-1] == 'c' && t[j-2] == '.' && t[j-3] == '.' &&
		DDT__cplusplus == CPLUSPLUS12) {
	    if (t != buf) {
	       strcpy(buf,t);
	       t = buf;
	     }
	    t[j-2] = 'C';
	    t[j-1] = 0;
	    if (access(t,4) < 0) {
	       t[j-2] = 'c';
	       if (access(t,4) < 0) t[j-2] = 'C';
	     };
	  }
	 else if (j > 2 && t[j-1] == 'c' && t[j-2] == '.' &&
		     DDT__cplusplus == CPLUSPLUS20) {
	    if (t != buf) {
	       strcpy(buf,t);
	       t = buf;
	     }
	    t[j-1] = 'C';
	  };
       };

      if (t == buf) t = SALLOC(t);
    };


   return t;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_fct_from_file -- find function given file/line	*/
/*									*/
/************************************************************************/


String
DDT_symbol_fct_from_file(file,line)
   String file;
   Integer line;
{
   String s,t;
   Integer i,j,k;
   Character buf[1024];

   if (file == NULL) return NULL;

   if (index(file,',') != NULL && index(file,'"') != NULL) {
      s = index(file,'"');
      ++s;
      strcpy(buf,s);
      s = index(buf,'"');
      if (s != NULL) *s = 0;
      file = buf;
    };

   s = rindex(file,'/');
   if (s == NULL) s = file;
   else s = ++s;
   for (i = 0; i < num_files; ++i) {
      t = rindex(sys_files[i].name,'/');
      if (t == NULL) t = sys_files[i].name;
      else ++t;
      if (STREQL(s,t) /* && MSGfile_compare(file,sys_files[i].name) */) break;
    };
   if (i >= num_files) return NULL;

   k = -1;
   for (j = 0; j < num_functions; ++j) {
      if (sys_functions[j].file == i &&
	     line >= sys_functions[j].sline && line <= sys_functions[j].eline) {
	 if (k < 0 || sys_functions[j].sline > sys_functions[k].sline) k = j;
       };
    };

   if (k < 0) return NULL;

   return sys_functions[k].name;
};




/************************************************************************/
/*									*/
/*	DDT_symbol_find_function -- find function name given tail	*/
/*									*/
/************************************************************************/


String
DDT_symbol_find_function(f)
   String f;
{
   Integer i,j,k,l;

   j = strlen(f);
   k = -1;
   for (i = 0; i < num_functions; ++i) {
      l = strlen(sys_functions[i].name);
      if (l >= j && STREQL(f,&sys_functions[i].name[l-j]) &&
	     (l == j || sys_functions[i].name[l-j-1] == '_')) {
	 if (l == j || k < 0) k = i;
	 if (l == j) break;
       };
    };

   if (k < 0) return NULL;

   return sys_functions[k].name;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_inq_sue_name -- check for sue name			*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_inq_sue_name(s,ln)
   String s;
   Integer ln;
{
   Integer i;

   if (ln == 0) ln = strlen(s);

   for (i = 0; i < num_types; ++i) {
      if (sys_types[i].sue && strlen(sys_types[i].name) == ln &&
	     strncmp(sys_types[i].name,s,ln) == 0)
	 return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_find_variable -- find variable given tail,file	*/
/*									*/
/************************************************************************/


String
DDT_symbol_find_variable(var,file,afile)
   String var;
   String file;
   String afile;
{
   Integer i,l,j,k;
   Boolean fg;

   j = strlen(var);
   k = -1;
   for (i = 0; i < num_variables; ++i) {
      l = strlen(sys_variables[i].name);
      if (l >= j && STREQL(var,&sys_variables[i].name[l-j]) &&
	     (l == j || sys_variables[i].name[l-j-1] == '_')) {
	 fg = sys_variables[i].ext;
	 if (!fg && file != NULL && sys_variables[i].file >= 0 &&
		filename_match(file,sys_variables[i].file)) fg = TRUE;
	 if (!fg && afile != NULL && sys_variables[i].file >= 0 &&
		filename_match(afile,sys_variables[i].file)) fg = TRUE;
	 if (fg) {
	    if (l == j || k < 0) k = i;
	    if (l == j) break;
	  }
       };
    };

   if (k < 0) return NULL;

   return sys_variables[k].name;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_find_fct_local -- find local for given function	*/
/*									*/
/************************************************************************/


String
DDT_symbol_find_fct_local(fct,var,buf)
   String fct;
   String var;
   String buf;
{
   Integer i;
   String s;
   Sequence lcls;

   if (DDT__cplusplus == 0) return var;

   s = rindex(fct,'`');
   if (s != NULL) ++s;
   else s = fct;

   lcls = NULL;
   for (i = 0; i < num_functions; ++i) {
      if (STREQL(s,sys_functions[i].name)) {
	 lcls = sys_functions[i].locals;
	 break;
       };
    };

   var = DDT_map_find_fct_local(var,lcls,buf);

   return var;
};




/************************************************************************/
/*									*/
/*	DDT_symbol_type_data -- return information about a type 	*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_type_data(file,func,line,name)
   String file;
   String func;
   Integer line;
   String name;
{
   String s,t,u;
   Character rslt[10240];
   Character abuf[1024],bbuf[1024],cbuf[1024],fbuf[1024];
   Integer i,ln;
   String fname;

   if (prefix(name,"file of ")) {
      DDT_mprint("FILE %s\n",&name[8]);
      return TRUE;
    }
   else if (isdigit(*name) && STREQL(&name[1]," byte real")) {
      DDT_mprint("PRIMITIVE %s\n",name);
      return TRUE;
    }
   else if (prefix(name,"array[")) {
      s = &name[6];
      t = abuf;
      while (*s != ']' && *s != 0) *t++ = *s++;
      if (*s == 0) return FALSE;
      *t = 0;
      get_pascal_dims(abuf,bbuf);
      s += 5;
      DDT_mprint("ARRAY %s %s\n",bbuf,s);
      return TRUE;
    }
   else if (prefix(name,"record ")) {
      pas_struct_info(name);
      return TRUE;
    }
   else if (name[0] == '^') {
      s = &name[1];
      while (isspace(*s)) ++s;
      if (*s == 0) return FALSE;
      DDT_mprint("POINTER %s\n",s);
       return TRUE;
    };

   t = cbuf;
   s = name;
   while (*s != 0 && (isidchar(*s) || *s == ' '))
      *t++ = *s++;
   *t = 0;

   if (*s == 0) {
      DDT_x_var_info(file,func,line,name,TRUE,FALSE,FALSE,rslt) ;
      s = rslt;
      fbuf[0] = 0;
      u = NULL;
      while (s != NULL && *s != 0) {
	 if (prefix(s,"type:")) {
	    t = index(s,'`');
	    if (fbuf[0] == 0) {
	       u = fbuf;
	       while (t != NULL && *t != 0 && *t != '\n') *u++ = *t++;
	       *u = 0;
	       u = fbuf;
	     }
	    else u = NULL;
	  };
	 s = index(s,'\n');
	 if (s != NULL) ++s;
       };
      if (fbuf[0] != 0 && u != NULL) fname = &fbuf[1];
      else fname = name;

      DDT_x_var_info(file,func,line,fname,FALSE,FALSE,TRUE,rslt);
      s = rslt;

      if (s[0] == '"' || STREQL(s,"no such type")) return FALSE;
      else {
	 i = strlen(s);
	 if (s[i-1] == ';') s[i-1] = 0;
	 else if (!DDT__use_gdb) return FALSE;
       };

      if (STREQL(s,name)) {
	 DDT_mprint("PRIMITIVE %s\n",name);
	 return TRUE;
       };

      i = strlen(s);

      if (prefix(s,"typedef ") || s[i-1] == '*') {
	 if (!c_typedef_info(s,name,abuf)) return FALSE;
	 DDT_mprint("SAME %s\n",abuf);
       }
      else if (prefix(s,"type ")) {
	 fix_pascal_type(s,abuf);
	 return DDT_symbol_type_data(file,func,line,abuf);
	 /* DDT_mprint("SAME %s\n",abuf);       */
       }
      else if (prefix(s,"struct ")) c_struct_info(s,"RECORD");
      else if (prefix(s,"union ")) c_struct_info(s,"UNION");
      else if (prefix(s,"enum ")) c_enum_info(&s[5]);
      else return FALSE;
    }
   else if (*s == '(' && cbuf[0] == 0) {
      c_enum_info(s);
    }
   else if (*s == '.' || ((*s == '(' || *s == '-') && cbuf[0] == 0)) {
      DDT_mprint("ENUM 0\n");
    }
   else {
      if (cbuf[0] != 0) {
	 *t++ = ' ';
	 *t = 0;
       };
      ln = strlen(s);
      if (s[0] == '(' && s[1] == '*') {
	 i = 0;
	 t = abuf;
	 s += 2;
	 while (*s != ')' && i == 0) {
	    if (*s == '(') ++i;
	    else if (*s == ')') --i;
	    *t++ = *s++;
	  };
	 ++s;
	 DDT_mprint("POINTER %s%s%s\n",cbuf,abuf,s);
       }
      else if (s[ln-1] == ')' && s[ln-2] == '(') {
	 s[ln-2] = 0;
	 DDT_mprint("FUNCTION %s%s\n",cbuf,s);
       }
      else if (s[ln-1] == ']') {
	 t = abuf;
	 while (*s != '[') *t++ = *s++;
	 *t = 0;
	 t = bbuf;
	 ++s;
	 while (*s != ']') *t++ = *s++;
	 ++s;
	 if (bbuf[0] == 0) *t++ = '*';
	 *t = 0;
	 while (*s == 0 && strlen(abuf) > 0 && abuf[strlen(abuf)-1] == 0)
	    abuf[strlen(abuf)-1] = 0;
	 DDT_mprint("ARRAY 0 %s %s%s%s\n",bbuf,cbuf,abuf,s);
       }
      else if (*s == '*') {
	 ++s;
	 if (STREQL(cbuf,"struct  ") || STREQL(cbuf,"union  ") || STREQL(cbuf,"enum  ")) {
	    DDT_mprint("PRIMITIVE int\n");
	  }
	 else DDT_mprint("POINTER %s%s\n",cbuf,s);
       }
      else return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_var_data -- get data for display of a variable	*/
/*									*/
/************************************************************************/


Boolean
DDT_symbol_var_data(file,func,line,name)
   String file;
   String func;
   Integer line;
   String name;
{
   String s;
   Boolean fg;
   Character flb[256],fnb[256],nmb[256],full[4096],typ[4096],exb[4096];
   Character buf[4096],buf1[4096];
   Integer i,addr;
   String t,u,class;
   Boolean varpar,expr;

   fg = TRUE;
   varpar = FALSE;
   expr = FALSE;

   DDT_x_var_info(file,func,line,name,FALSE,TRUE,FALSE,exb);
   s = exb;
   if (strncmp(s,"syntax error on ",16) == 0) { s = name; expr = TRUE; }
   else if (s == NULL || s[0] != '`') fg = FALSE;

   if (fg) {
      i = 0;
      if (s[0] == '`') ++s;
      strcpy(full,s);
      for (t = s; *t != 0; ++t) if (*t == '`') ++i;
      if (i != 0) {
	 t = flb;
	 while (*s != '`') *t++ = *s++;
	 *t = 0;
	 ++s;
       }
      else flb[0] = 0;
      if (i > 1) {
	 t = fnb;
	 u = rindex(s,'`');
	 while (s != u) *t++ = *s++;
	 *t = 0;
	 ++s;
       }
      else fnb[0] = 0;

      t = nmb;
      while (*s != 0) {
	 if (*s != ' ') *t++ = *s;
	 ++s;
       };
      *t = 0;

      if (flb[0] != 0) {
	 t = find_file(flb);
	 if (t == NULL) fg = FALSE;
	 else strcpy(flb,t);
       };
    };

   if (fg && !expr) {
      DDT_x_var_info(file,func,line,nmb,TRUE,FALSE,FALSE,buf);
      s = buf;
      i = strlen(full);
      while (s != NULL && *s != 0) {
	 if (prefix(s,"variable:")) {
	    t = index(s,'`');
	    if (t != NULL) ++t;
	    if (prefix(t,full) && (t[i] == '\n' || t[i] == 0)) break;
	  }
	 else if (prefix(s,"conf value:") || prefix(s,"ref:")) {
	    t = index(s,'`');
	    if (t != NULL) ++t;
	    if (prefix(t,full) && (t[i] == '\n' || t[i] == 0)) {
	       varpar = TRUE;
	       break;
	     };
	  }
	 s = index(s,'\n');
	 if (s != NULL) ++s;
       };
      if (fnb[0] == 0 && (s == NULL || *s == 0)) fg = FALSE;
    };

   if (fg) {
      DDT_x_var_info(file,func,line,nmb,FALSE,FALSE,TRUE,buf);
      s = buf;
      if (prefix(s,"register ")) {
	 class = "register";
	 s = &s[9];
       }
      else if (prefix(s,"static ")) {
	 class = "static";
	 s = &s[7];
       }
      else if (fnb[0] == 0) class = "extern";
      else class = "auto";

      if (prefix(s,"(var parameter)")) varpar = TRUE;

      i = strlen(nmb);
      if ((prefix(s,"var ") || prefix(s,"(field) ") ||
	      prefix(s,"(parameter)") || prefix(s,"(var parameter)")) &&
	     index(s,':') != NULL) {
	 s = index(s,':');
	 ++s;
	 ++s;
	 strcpy(typ,s);
	 i = strlen(typ);
	 if (typ[i-1] == ';') typ[i-1] = 0;
       }
      else if (prefix(s,nmb) && s[i] == ' ' && prefix(&s[i+1],nmb) && s[2*i+1] == ';') {
	 strcpy(typ,nmb);
       }
      else {
	 fg = c_typedef_info(s,nmb,typ);
       };
    };

   if (fg) {
      s = full;
      for ( ; ; ) {
	 if (varpar) sprintf(buf1,"*((integer *)(& %s))",s);
	 else sprintf(buf1,"& %s",s);
	 DDT_x_eval(buf1,buf,1024);
	 if (!prefix(buf,"syntax error") && buf[0] != '"') break;
	 s = index(s,'`');
	 if (s == NULL) break;
	 ++s;
       };
      if (varpar) addr = atol(buf);
      else {
	 s = buf;
	 while (*s != 0 && (*s != '0' || s[1] != 'x')) ++s;
	 if (sscanf(s,"0x%x",&addr) != 1) addr = 0;
       };
      if (addr > 0 && addr < 2048) addr = 0;
    };

   if (fg) {
      if (flb[0] == 0) strcpy(flb,"*");
      if (fnb[0] == 0) strcpy(fnb,"*");
      DDT_mprint("%s %s %s %s %d %s\n",flb,fnb,nmb,class,addr,typ);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_symbol_dump_globals -- dump global symbol values		*/
/*									*/
/************************************************************************/


void
DDT_symbol_dump_globals(lvl)
   Integer lvl;
{
   Integer i;
   Character rslt[2050],expr[256];

   for (i = 0; i < num_variables; ++i) {
      if (DDT__use_gdb)
	 sprintf(expr,"%s",sys_variables[i].name);
      else if (sys_variables[i].ext)
	 sprintf(expr,"`%s",sys_variables[i].name);
      else
	 sprintf(expr,"`%s`%s",filetail(sys_files[sys_variables[i].file].name),
		    sys_variables[i].name);
      DDT_x_eval(expr,rslt,2048);
      if (strncmp(rslt,"no module or function named",27) == 0) {
	 sprintf(expr,"%s",sys_variables[i].name);
	 DDT_x_eval(expr,rslt,2048);
       };
      DDT_trace_dump(lvl,sys_variables[i].name,rslt);
    };
};





/************************************************************************/
/*									*/
/*	DDT_symbol_dynamic_type -- get dynamic type of expression	*/
/*									*/
/************************************************************************/


String
DDT_symbol_dynamic_type(e,blen,buf)
   String e;
   Integer blen;
   String buf;
{
   String s;
   Character rslt[2050],expr[2050];
   Integer i,j,addr;

   if (strlen(e) > 2000) return NULL;

   buf[0] = 0;
   if (DDT__cplusplus > 0) {
      sprintf(expr,"(%s).__vptr",e);
      if (DDT_x_eval(expr,rslt,2048) && sscanf(rslt,"%i",&addr) == 1 && addr > 0) {
	 for (i = 0; i < num_variables; ++i) {
	    if (sys_variables[i].addr == addr &&
		   strncmp(sys_variables[i].name,"__vtbl__",8) == 0) {
	       s = &sys_variables[i].name[8];
	       j = 0;
	       while (isdigit(*s)) j = j*10 + *s++ - '0';
	       if (j >= blen) j = blen-1;
	       strncpy(buf,s,j);
	       buf[j] = 0;
	       break;
	     };
	  };
       };
    };

   if (buf[0] == 0) return NULL;

   return buf;
};





/************************************************************************/
/*									*/
/*	get_symbol_info -- read info from symbol table			*/
/*	get_sym_dir -- handle directory name				*/
/*	get_sym_so -- handle source name				*/
/*	get_sym_tso -- handle source name in text region		*/
/*	get_sym_fun -- handle function name				*/
/*	get_sym_line -- handle source line				*/
/*	get_sym_sym -- handle debugging info on symbol			*/
/*	add_function -- add function to function table			*/
/*									*/
/************************************************************************/


typedef struct _SYM_READ_INFO {
   Integer mxfil,mxfct,mxvar,mxtyp;
   Integer cur_file,cur_fct;
   String directory;
} SYM_READ_INFO;


static	FSYM_CALL	ddt_fsym[] = {
   { FSYM_TYPE_DIR,		get_sym_dir },
   { FSYM_TYPE_SO,		get_sym_so  },
   { FSYM_TYPE_SOL,		get_sym_so  },
   { FSYM_TYPE_TEXT_SO, 	get_sym_tso },
   { FSYM_TYPE_TEXT,		get_sym_fun },
   { FSYM_TYPE_TEXT_EXT,	get_sym_fun },
   { FSYM_TYPE_FUN,		get_sym_fun },
   { FSYM_TYPE_SLINE,		get_sym_line },
   { FSYM_TYPE_GLOBAL,		get_sym_sym },
   { FSYM_TYPE_LOCAL,		get_sym_sym },
   { FSYM_TYPE_AUTO,		get_sym_sym },
   { FSYM_TYPE_TYPE,		get_sym_sym },
   { FSYM_TYPE_STRUCT,		get_sym_sym },
   { FSYM_TYPE_ENUM,		get_sym_sym },
   { FSYM_TYPE_UNION,		get_sym_sym },
   { FSYM_TYPE_DATA_EXT,	get_sym_addr },
   { FSYM_TYPE_END, NULL }
};




static Boolean
get_symbol_info(sys)
   String sys;
{
   SYM_READ_INFO sri;
   FSYM_ID fsym;
   Integer i;
   Character buf[1024];
   String s;

   num_files = 0;
   num_functions = 0;
   num_variables = 0;
   num_types = 0;
   main_prog = -1;
   directorys = NULL;

   fsym = FSYMopen(sys,TRUE);
   if (fsym == NULL) return FALSE;

   sri.mxfil = 16;
   sri.mxfct = 128;
   sri.mxvar = 128;
   sri.mxtyp = 32;

   sys_files = (FILE_INFO *) calloc(sri.mxfil,sizeof(FILE_INFO));
   sys_functions = (FUN_INFO *) calloc(sri.mxfct,sizeof(FUN_INFO));
   sys_variables = (VAR_INFO *) calloc(sri.mxvar,sizeof(FUN_INFO));
   sys_types = (TYPE_INFO *) calloc(sri.mxtyp,sizeof(TYPE_INFO));

   function_hash = HASHcreate(1024);
   type_hash = HASHcreate(1024);
   variable_hash = HASHcreate(512);

   sri.cur_file = -1;
   sri.cur_fct = -1;
   sri.directory = NULL;

   FSYMsymbols(fsym,ddt_fsym,&sri);

   if (DDT__cplusplus != 0) {
      for (i = 0; i < num_functions; ++i) {
	 strcpy(buf,sys_functions[i].pname);
	 s = DDT_map_unfix_function(buf);
	 if (s != buf) {
	    sys_functions[i].pname = SALLOC(s);
	  };
       };
      for (i = 0; i < num_files; ++i) {
	 fix_filename(i);
       };
    };

   FSYMclose(fsym);

   HASHdestroy(function_hash);
   function_hash = NULL;
   HASHdestroy(type_hash);
   type_hash = NULL;
   HASHdestroy(variable_hash);
   variable_hash = NULL;

   return TRUE;
};






/*ARGSUSED*/

static void
get_sym_dir(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   String s;

   for (s = name; s != NULL; s = index(&s[1],'/')) {
      if (strncmp(s,"/tmp/CC.",8) == 0) {
	 DDT__cplusplus = CPLUSPLUS20;
	 return;
       };
    };

   sri->directory = SALLOC(name);
   add_directory(sri->directory);
};






/*ARGSUSED*/

static void
get_sym_so(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   Integer j;
   Character nbuf[1024];
   String name0;
   String s;

   if (STREQL(name,"libg.s")) return;

   if (num_files >= sri->mxfil) {
      sri->mxfil *= 2;
      sys_files = (FILE_INFO *) realloc(sys_files,sri->mxfil*sizeof(FILE_INFO));
    };

   j = strlen(name);
   if ((name[j-3] == '.' && name[j-2] == '.' && name[j-1] == 'c' && DDT__cplusplus == 0) ||
	  (name[j-2] == '.' && name[j-1] == 'C' && DDT__cplusplus == 0)) {
      if (getenv("CPLUS20") != NULL) DDT__cplusplus = CPLUSPLUS20;
      else if (getenv("CPLUS12") != NULL) DDT__cplusplus = CPLUSPLUS12;
      else DDT__cplusplus = CPLUSPLUS20;
    };


   name0 = name;
   if (name[0] != '/' && sri->directory != NULL) {
      sprintf(nbuf,"%s%s",sri->directory,name);
      name = nbuf;
    };

   if (ft == FSYM_TYPE_SOL && sri->cur_file >= 0)
      sys_files[sri->cur_file].altfile = num_files;

   sri->cur_file = num_files++;
   sys_files[sri->cur_file].name = SALLOC(name);
   sys_files[sri->cur_file].pname = sys_files[sri->cur_file].name;
   if (name == name0) sys_files[sri->cur_file].dname = sys_files[sri->cur_file].name;
   else sys_files[sri->cur_file].dname = SALLOC(name0);
   sys_files[sri->cur_file].mtime = 0;
   sys_files[sri->cur_file].altfile = -1;

   s = rindex(sys_files[sri->cur_file].name,'/');
   if (s == NULL) sys_files[sri->cur_file].tail = sys_files[sri->cur_file].name;
   else sys_files[sri->cur_file].tail = ++s;
};






/*ARGSUSED*/

static void
get_sym_tso(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   sri->cur_file = -1;
};





/*ARGSUSED*/

static void
get_sym_fun(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   String s;

   if (num_functions >= sri->mxfct) {
      sri->mxfct *= 2;
      sys_functions = (FUN_INFO *) realloc(sys_functions,sri->mxfct*sizeof(FUN_INFO));
    };

   s = index(name,':');
   if (s != NULL) *s = 0;

#ifndef NO_UNDERSCORE
   if (ft != FSYM_TYPE_FUN && name[0] == '_') s = &name[1];
   else if (ft != FSYM_TYPE_FUN) return;
   else s = name;
#else
   s = name;
#endif

   sri->cur_fct = add_function(s,sri->cur_file);
};






/*ARGSUSED*/

static void
get_sym_line(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   if (sri->cur_fct >= 0 && desc > 0) {
      if (sys_functions[sri->cur_fct].sline == 0) {
	 sys_functions[sri->cur_fct].file = sri->cur_file;
	 sys_functions[sri->cur_fct].sline = desc;
       }
      sys_functions[sri->cur_fct].eline = desc;
      sys_functions[sri->cur_fct].eaddr = val;
    };
};





/*ARGSUSED*/

static void
get_sym_sym(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   Integer tfg;
   String s;
   SUE sue;

   switch (ft) {
      case FSYM_TYPE_GLOBAL :
      case FSYM_TYPE_LOCAL :
	 tfg = 0;
	 break;
      case FSYM_TYPE_TYPE :
	 tfg = 1;
	 sue = TYPE_NAME;
	 break;
      case FSYM_TYPE_STRUCT :
	 tfg = 1;
	 sue = TYPE_STRUCT;
	 break;
      case FSYM_TYPE_UNION :
	 tfg = 1;
	 sue = TYPE_UNION;
	 break;
      case FSYM_TYPE_ENUM :
	 tfg = 1;
	 sue = TYPE_ENUM;
	 break;
      case FSYM_TYPE_AUTO :
	 tfg = -1;
	 break;
    };

   s = index(name,':');
   if (s != NULL) *s++ = 0;

   if (sri->cur_fct >= 0) {
      if (DDT_map_test_save_local(name)) {
	 sys_functions[sri->cur_fct].locals = CONS(SALLOC(name),
						      sys_functions[sri->cur_fct].locals);
       };
    };

   if (tfg == 0) {
      if (num_variables >= sri->mxvar) {
	 sri->mxvar *= 2;
	 sys_variables = (VAR_INFO *) realloc(sys_variables,sri->mxvar*sizeof(VAR_INFO));
       };
      add_variable(name,sri->cur_file,(ft == FSYM_TYPE_GLOBAL),val);
    }
   else if (tfg == 1) {
      if (num_types >= sri->mxtyp) {
	 sri->mxtyp *= 2;
	 sys_types = (TYPE_INFO *) realloc(sys_types,sri->mxtyp*sizeof(TYPE_INFO));
       };
      add_type(name,sri->cur_file,(ft == FSYM_TYPE_GLOBAL),sue);
    }
};





/*ARGSUSED*/

static void
get_sym_addr(name,desc,val,ft,sri)
   String name;
   Integer desc;
   Integer val;
   FSYM_TYPE ft;
   SYM_READ_INFO * sri;
{
   Integer i;
   ENTRY ent;
   ENTRY *fnd;

#ifndef NO_UNDERSCORE
   if (name[0] == '_') ++name;
#endif

   ent.key = name;
   ent.data = (char *) -1;

   fnd = HASHsearch(variable_hash,ent,FIND);

   if (fnd == NULL) return;

   for (i = (Integer) fnd->data; i >= 0; i = sys_variables[i].next) {
      if (val > 0 && sys_variables[i].addr == 0) sys_variables[i].addr = val;
    };
};





static Integer
add_function(name,cur_file)
   String name;
   Integer cur_file;
{
   Integer i,cur_fct;
   ENTRY ent;
   ENTRY *fnd;

   ent.key = name;
   ent.data = (char *) -1;

   fnd = HASHsearch(function_hash,ent,ENTER);

   i = (Integer) fnd->data;

   if (i >= 0) {
      for ( ; ; ) {
	 if (cur_file >= 0 && sys_functions[i].file < 0) {
	    sys_functions[i].file = cur_file;
	    return i;
	  }
	 else if (cur_file < 0 || sys_functions[i].file < 0 ||
		     cur_file == sys_functions[i].file) {
	    return i;
	  }
	 if (sys_functions[i].next < 0) {
	    sys_functions[i].next = num_functions;
	    break;
	  }
	 i = sys_functions[i].next;
       };
    }
   else fnd->data = (char *) num_functions;

   if (main_prog < 0 && (STREQL(name,"main") || STREQL(name,"program")))
      main_prog = num_functions;

   if (index(name,'.') != NULL || strncmp(name,"-l",2) == 0) return -1;

   sys_functions[num_functions].name = SALLOC(name);
   sys_functions[num_functions].pname = sys_functions[num_functions].name;
   sys_functions[num_functions].file = cur_file;
   sys_functions[num_functions].sline = 0;
   sys_functions[num_functions].eline = 0;
   sys_functions[num_functions].eaddr = 0;
   sys_functions[num_functions].locals = NULL;
   sys_functions[num_functions].mangle = NULL;
   sys_functions[num_functions].next = -1;
   cur_fct = num_functions;
   ++num_functions;

   return cur_fct;
};





static Integer
add_variable(name,cur_file,ext,addr)
   String name;
   Integer cur_file;
   Boolean ext;
   Integer addr;
{
   Integer i,cur_var;
   ENTRY ent;
   ENTRY *fnd;

   if (cur_file < 0) return -1;

   ent.key = name;
   ent.data = (char *) -1;

   fnd = HASHsearch(variable_hash,ent,ENTER);

   i = (Integer) fnd->data;

   if (i >= 0) {
      for ( ; ; ) {
	 if (!ext && sys_variables[i].file >= 0 &&
		sys_variables[i].file != cur_file) {
	    if (sys_variables[i].next < 0) {
	       sys_variables[i].next = num_variables;
	       break;
	     }
	    else i = sys_variables[i].next;
	  }
	 else {
	    if (addr > 0 && sys_variables[i].addr == 0) sys_variables[i].addr = addr;
	    return i;
	  };
       }
    }
   else fnd->data = (char *) num_variables;

   sys_variables[num_variables].name = SALLOC(name);
   sys_variables[num_variables].file = cur_file;
   sys_variables[num_variables].ext = ext;
   sys_variables[num_variables].mangle = NULL;
   sys_variables[num_variables].addr = addr;
   sys_variables[num_variables].next = -1;
   cur_var = num_variables;
   ++num_variables;

   return cur_var;
};





static Integer
add_type(name,cur_file,ext,sue)
   String name;
   Integer cur_file;
   Boolean ext;
   SUE sue;
{
   Integer i,cur_typ;
   Character buf[1024];
   ENTRY ent;
   ENTRY *fnd;

   if (cur_file < 0) return -1;

   ent.key = name;
   ent.data = (char *) -1;

   fnd = HASHsearch(type_hash,ent,ENTER);

   i = (Integer) fnd->data;

   if (i >= 0) {
      for ( ; ; ) {
	 if (!ext && sys_types[i].file >= 0 && sys_types[i].sue != sue &&
		sys_types[i].file != cur_file) {
	    if (sys_types[i].next < 0) {
	       sys_types[i].next = num_types;
	       break;
	     }
	    else i = sys_types[i].next;
	  }
	 else return i;
       }
    }
   else fnd->data = (char *) num_types;

   sys_types[num_types].name = SALLOC(name);
   sys_types[num_types].file = cur_file;
   sys_types[num_types].ext = ext;
   sys_types[num_types].sue = sue;
   sys_types[num_types].next = -1;

   if (sue == TYPE_NAME) sys_types[num_types].pname = sys_types[num_types].name;
   else {
      switch (sue) {
	 case TYPE_STRUCT :
	    sprintf(buf,"struct %s",name);
	    break;
	 case TYPE_UNION :
	    sprintf(buf,"union %s",name);
	    break;
	 case TYPE_ENUM :
	    sprintf(buf,"enum %s",name);
	    break;
       };
      sys_types[num_types].pname = SALLOC(buf);
    };

   cur_typ = num_types;
   ++num_types;

   return cur_typ;
};





/********************************************************************************/
/*										*/
/*	check_file_date -- check date for a file				*/
/*										*/
/********************************************************************************/


static Boolean
check_file_date(i)
   Integer i;
{
   String s;
   struct stat buf;
   time_t t;
   Boolean fg;

   if (i < 0) s = DDT__system_name;
   else s = sys_files[i].name;

   if (stat(s,&buf) < 0) t = 0;
   else t = buf.st_mtime;

   fg = TRUE;
   if (i < 0) {
      if (system_time != 0 && t != 0 && t != system_time) fg = FALSE;
      system_time = t;
    }
   else {
      if (t != 0 && t != sys_files[i].mtime) {
	 sys_files[i].mtime = t;
	 if (t > system_time) fg = FALSE;
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	c_typedef_info -- convert typedef output to type name		*/
/*	c_struct_info -- return info on struct/union			*/
/*	c_enum_info -- return info on enum				*/
/*									*/
/************************************************************************/


static Boolean
c_typedef_info(tstr,nam,rslt)
   String tstr;
   String nam;
   String rslt;
{
   String s,t;
   Boolean afg;
   Integer i,j;

   if (prefix(tstr,"typedef ")) tstr += 8;
   else if (prefix(tstr,"type = ")) tstr += 7;

   s = tstr;
   t = rslt;

   if (prefix(tstr,"record\n")) {
      while (*s != 0) {
	 if (*s == '\n') *t++ = ' ';
	 else *t++ = *s;
	 ++s;
       };
      if (t[-1] == ';') --t;
      *t = 0;
      return TRUE;
    };

   afg = FALSE;
   i = strlen(nam);
   while (*s != 0 && *s != ';') {
      if (!afg && isidchar(*s) && nam != NULL && prefix(s,nam) && !isidchar(s[i])) {
	 s = &s[i];
       }
      else if (!afg && prefix(s,"struct ")) {
	 for (j = 0; j < 7; ++j) *t++ = *s++;
	 afg = TRUE;
       }
      else if (!afg && prefix(s,"union ")) {
	 for (j = 0; j < 6; ++j) *t++ = *s++;
	 afg = TRUE;
       }
      else if (!afg && prefix(s,"enum ")) {
	 for (j = 0; j < 5; ++j) *t++ = *s++;
	 afg = TRUE;
       }
      else if (*s == '{') {
	 j = 0;
	 while (*s != 0) {
	    if (*s == '{') ++j;
	    else if (*s == '}' && --j == 0) break;
	    ++s;
	  };
	 if (*s == '}') ++s;
	 while (isspace(*s)) ++s;
	 afg = FALSE;
       }
      else if (isspace(*s)) {
	 afg = (isidchar(*s));
	 *t++ = ' ';
	 while (isspace(*s)) ++s;
       }
      else {
	 afg = (isidchar(*s));
	 *t++ = *s++;
       };
    };

   *t = 0;

   i = strlen(rslt);
   while (i > 0 && rslt[i-1] == ' ') --i;
   rslt[i] = 0;

   return TRUE;
};





static Boolean
c_struct_info(typ,cls)
   String typ;
   String cls;
{
   String s,t,u;
   Integer i;
   Boolean afg;
   Character nbuf[1024],fbuf[1024],tbuf[1024];
   Character pbuf[1024];

   s = typ;
   t = pbuf;
   while (isalnum(*s)) *t++ = *s++;
   while (isspace(*s)) *t++ = *s++;
   while (isalnum(*s) || *s == '_' || *s == '$') *t++ = *s++;
   *t = 0;

   while (*s != '{' && *s != 0) ++s;
   if (*s == 0) return FALSE;
   ++s;

   t = s;
   i = 0;
   while (*t != 0) if (*t++ == ';') ++i;

   DDT_mprint("%s %d\n",cls,i);

   while (*s != 0) {
      while (isspace(*s)) ++s;
      if (*s == '}') break;
      t = fbuf;
      u = nbuf;
      afg = FALSE;
      while (*s != ';' && *s != 0) {
	 if (isidchar(*s)) {
	    if (!afg) u = nbuf;
	    *u++ = *s;
	    afg = TRUE;
	  }
	 else afg = FALSE;
	 *t++ = *s++;
       };
      if (*s == 0) break;
      else ++s;
      *t = 0;
      *u = 0;

      if (!c_typedef_info(fbuf,nbuf,tbuf)) break;

      if (STREQL(tbuf,"struct *") || STREQL(tbuf,"union *") || STREQL(tbuf,"enum *")) {
	 strcpy(tbuf,"int");
       };

      t = DDT_map_unfix_field(nbuf,pbuf);

      DDT_mprint(" %s:%s\n",t,tbuf);
    };

   return TRUE;
};





static Boolean
c_enum_info(typ)
   String typ;
{
   String s,t;
   Integer i;
   Character fbuf[1024];
   Character pbuf[1024];

   s = typ;
   t = pbuf;
   while (isalnum(*s) || *s == '_' || *s == '$') *t++ = *s++;
   *t = 0;

   while (*s != '{' && *s != '(' && *s != 0) ++s;
   if (*s == 0) {
      DDT_mprint("ENUM 0\n");
      return TRUE;
    };
   ++s;

   t = s;
   i = 1;
   while (*t != 0) if (*t++ == ',') ++i;

   DDT_mprint("ENUM %d",i);

   while (*s != 0) {
      while (isspace(*s)) ++s;
      if (*s == '}' || *s == ')') break;
      t = fbuf;
      while (*s != ',' && *s != '}' && *s != ')' && *s != 0) {
	 if (isidchar(*s)) *t++ = *s++;
	 else ++s;
       };
      if (*s == ',') ++s;
      *t = 0;
      if (fbuf[0] != 0) DDT_mprint(" %s",fbuf);
    };

   DDT_mprint("\n");

   return TRUE;
};





/************************************************************************/
/*									*/
/*	isidchar -- check for character of id string			*/
/*									*/
/************************************************************************/


static Boolean
isidchar(c)
   Integer c;
{
   return (isalnum(c) || c == '$' || c == '_');
};





/************************************************************************/
/*									*/
/*	fix_pascal_type -- fix up a pascal type string			*/
/*	pas_struct_info -- handle pascal structure definitions		*/
/*	get_pascal_dims -- get dimension information for pascal array	*/
/*									*/
/************************************************************************/


static void
fix_pascal_type(s,buf)
   String s;
   String buf;
{
   Boolean sfg;

   if (prefix(s,"type ")) {
      s = index(s,'=');
      if (s == NULL) {
	 buf[0] = 0;
	 return;
       };
      s += 2;
    };

   sfg = FALSE;

   while (*s != 0) {
      if (isspace(*s)) {
	 if (!sfg) *buf++ = ' ';
	 sfg = TRUE;
       }
      else {
	 *buf++ = *s;
	 sfg = FALSE;
       };
      ++s;
    };

   if (sfg) --buf;
   *buf = 0;
};




static void
pas_struct_info(typ)
   String typ;
{
   String s,t,nm;
   Integer ct;
   Boolean vfg;
   Character buf[10240];
   Integer cct;

   ct = 0;
   vfg = FALSE;
   cct = 0;

   for (s = typ; *s != 0; ++s) {
      if (prefix(s," case ")) { vfg = TRUE; cct = ct; }
      if (*s == ':') ++ct;
    };

   if (vfg) {			/* IGNORE VARIANT PART NOW *****************/
      vfg = FALSE;
      ct = cct;
    };

   DDT_mprint("%s %d\n",(vfg ? "VARREC" : "RECORD"),ct);

   if (prefix(typ,"record ")) typ = &typ[7];
   strcpy(buf,typ);

   for (s = buf; s != NULL; ) {
      while (isspace(*s)) ++s;
      t = index(s,' ');
      if (t == NULL) break;
      nm = s;
      *t++ = 0;
      if (*t++ != ':') break;
      while (isspace(*t)) ++t;
      s = t;
      t = index(s,';');
      if (t == NULL) break;
      *t++ = 0;
      DDT_mprint(" %s:%s\n",nm,s);
      s = t;
    };
};





static void
get_pascal_dims(s,buf)
   String s;
   String buf;
{
   Integer i,j;
   String t;
   Character abuf[1024],buf1[1024];
   Boolean fg;

   fg = TRUE;
   if (isdigit(*s)) {
      i = atol(s);
      s = index(s,'.');
      s += 2;
      j = atol(s)-i+1;
    }
   else if (*s == '(') {
      i = 0;
      j = 1;
      while (*s != 0) {
	 if(*s++ == ',') ++j;
       };
    }
   else {
      t = abuf;
      while (*s != ']' && *s != 0) *t++ = *s++;
      *t = 0;
      t = DDT_debug_command("whatis %s\n",abuf);
      if (t == NULL || !prefix(t,"type ")) fg = FALSE;
      else if ((t = index(t,'=')) == NULL) fg = FALSE;
      else {
	 t += 2;
	 strcpy(buf1,t);
	 get_pascal_dims(buf1,buf);
	 return;
       };
    };

   if (fg == FALSE) strcpy(buf,"* *");
   else sprintf(buf,"%d %d",i,j);
};




/************************************************************************/
/*									*/
/*	prefix -- check for a valid prefix				*/
/*									*/
/************************************************************************/


static Boolean
prefix(s,pfx)
   String s;
   String pfx;
{
   Integer ln;

   ln = strlen(pfx);

   return (strncmp(s,pfx,ln) == 0);
};





/************************************************************************/
/*									*/
/*	find_file -- find file matching simple name			*/
/*									*/
/************************************************************************/


static String
find_file(s)
   String s;
{
   Integer i;

   for (i = 0; i < num_files; ++i) {
      if (STREQL(s,filetail(sys_files[i].name))) break;
    };

   if (i >= num_files) return NULL;

   return sys_files[i].name;
};





/************************************************************************/
/*									*/
/*	filetail -- return filename for dbx symbol usage		*/
/*									*/
/************************************************************************/


static String
filetail(s)
   String s;
{
   static Character buf[64];
   String p;

   p = rindex(s,'/');
   if (p == NULL) p = s;
   else ++p;

   strcpy(buf,p);
   p = rindex(buf,'.');
   if (p != NULL && STREQL(&p[-1],"..c")) --p;
   if (p != NULL) *p = 0;

   return buf;
};





/************************************************************************/
/*									*/
/*	add_directory -- add directory to list				*/
/*									*/
/************************************************************************/


static void
add_directory(d)
   String d;
{
   Sequence l;
   String dx;

   forin (dx,String,l,directorys) {
      if (STREQL(d,dx)) break;
    };

   if (dx == NULL) {
      DDT_model_use(d);
      directorys = CONS(SALLOC(d),directorys);
    };
};






/************************************************************************/
/*									*/
/*	filename_match -- compare file names				*/
/*									*/
/************************************************************************/


static Boolean
filename_match(f1,idx)
   String f1;
   Integer idx;
{
   String s;
   String f2;

   s = rindex(f1,'/');
   if (s != NULL) f1 = ++s;

   f2 = sys_files[idx].name;
   s = rindex(f2,'/');
   if (s != NULL) f2 = ++s;
   if (STREQL(f1,f2)) return TRUE;

   f2 = sys_files[idx].dname;
   s = rindex(f2,'/');
   if (s != NULL) f2 = ++s;
   if (STREQL(f1,f2)) return TRUE;

   idx = sys_files[idx].altfile;
   if (idx >= 0) return filename_match(f1,idx);

   return FALSE;
};





/************************************************************************/
/*									*/
/*	fix_filename -- fix up filename for internal use		*/
/*									*/
/************************************************************************/


static void
fix_filename(idx)
   Integer idx;
{
   String t;
   Integer j;
   Character buf[1024];

   t = sys_files[idx].dname;
   if (t == NULL) t = sys_files[idx].name;

   if (DDT__cplusplus != 0) {
      if (access(t,4) < 0) {
	 j = strlen(t);
	 if (j > 3 && t[j-1] == 'c' && t[j-2] == '.' && t[j-3] == '.' &&
		DDT__cplusplus == CPLUSPLUS12) {
	    if (t != buf) {
	       strcpy(buf,t);
	       t = buf;
	     }
	    t[j-2] = 'C';
	    t[j-1] = 0;
	    if (access(t,4) < 0) {
	       t[j-2] = 'c';
	       if (access(t,4) < 0) t[j-2] = 'C';
	     };
	  }
	 else if (j > 2 && t[j-1] == 'c' && t[j-2] == '.' &&
		     DDT__cplusplus == CPLUSPLUS20) {
	    if (t != buf) {
	       strcpy(buf,t);
	       t = buf;
	     }
	    t[j-1] = 'C';
	  };
       };
    };

   if (STRNEQ(t,sys_files[idx].pname)) sys_files[idx].pname = SALLOC(t);
};





/* end of ddtsyms.c */
