/************************************************************************/
/*									*/
/*		ddtrun.c						*/
/*									*/
/*	Routines for running debugger for ddt debugger interface	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sgtty.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/errno.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define OUTLINE_SIZE		20480


/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


	String		DDT__system_name;

static	Integer 	ddt_tty;	/* tty master for debuger in/out*/
static	Integer 	usr_tty;	/* tty master for user in/out	*/
static	Character	usr_name[64];	/* user tty name		*/
static	Character	ddt_name[64];	/* ddt	tty name		*/
static	Integer 	ddt_process;	/* pid of debugger		*/
static	Integer 	usr_input;	/* user input pid		*/
static	Character	sys_name[128];	/* real system name		*/
static	Character	outline[OUTLINE_SIZE+1];  /* current output line*/
static	Integer 	outcnt;
static	Character	intr_ch;
static	Character	eof_ch;

static	FILE *		trace_file;






/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		run_debug();
static	Integer 	get_ptty();
static	Integer 	ddt_from_debug();
static	Integer 	ddt_from_user();
static	Boolean 	send_input();
static	void		hup_handler();
static	void		abort_handler();





/************************************************************************/
/*									*/
/*	DDT_run_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_run_init()
{
   ddt_tty = -1;
   usr_tty = -1;
   usr_name[0] = 0;
   ddt_name[0] = 0;
   ddt_process = 0;
   usr_input = -1;
   sys_name[0] = 0;
   outline[0] = 0;
   outcnt = 0;
   trace_file = NULL;

   DDT__system_name = NULL;

   signal(SIGHUP,hup_handler);
};





/************************************************************************/
/*									*/
/*	DDT_start_debug -- start up debugger				*/
/*									*/
/************************************************************************/


Boolean
DDT_start_debug(system,core)
   String system;
   String core;
{
   Character ttynm[64],utynm[64];
   Integer child;
   Integer msk[2];
   struct tchars tc;
   Integer tty;

   if (getenv("DDT_DEBUG") != NULL && trace_file == NULL) {
      trace_file = fopen("ddt.trace","w");
      if (trace_file == NULL) trace_file = stderr;
    };

   tty = open("/dev/tty",2);
   if (tty <= 1) tty = 1;
   ioctl(tty, TIOCGETC, (char *)&tc);
   if (tty > 1) close(tty);
   intr_ch = tc.t_intrc;
   eof_ch = tc.t_eofc;

   if (ddt_tty < 0) {
      ddt_tty = get_ptty(ttynm);
      strcpy(ddt_name,ttynm);
      fcntl(ddt_tty,F_SETFL,FNDELAY);
      msk[0] = 1 << ddt_tty;
      CMPXregister(ddt_tty+1,msk,NULL,NULL,ddt_from_debug);
    }
   else {
      strcpy(ttynm,ddt_name);
    };

   if (usr_tty < 0) {
      usr_tty = get_ptty(utynm);
      strcpy(usr_name,utynm);
      usr_input = -1;
      fcntl(usr_tty,F_SETFL,FNDELAY);
      msk[0] = 1 << usr_tty;
      CMPXregister(usr_tty+1,msk,NULL,NULL,ddt_from_user);
    }
   else {
      strcpy(utynm,usr_name);
    };

   if (ddt_process != 0) {
      kill(ddt_process,SIGKILL);
      ddt_process = 0;
    };

   fflush(stdout);
   fflush(stderr);

   child = vfork();
   if (child < 0) return FALSE;

   if (child == 0) {
      run_debug(system,core,ttynm,utynm);

      printf("Debugger aborted\n");
      _exit(0);
    };

   ddt_process = child;

   utynm[strlen("/dev/")] = 't';
   DDT_model_user_tty(utynm);
   usr_input = usr_tty;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_stop_debug -- stop debugger 				*/
/*									*/
/************************************************************************/


Boolean
DDT_stop_debug(sts)
   Integer sts;
{
   Integer pid;

   pid = ddt_process;
   ddt_process = 0;

   if (pid != 0) {
      if (sts == 0) DDT_x_quit();

      DDT_state_abort();

      if (ddt_process > 0) {
	 kill(ddt_process,SIGKILL);
	 ddt_process = 0;
       };

      if (ddt_tty >= 0) {
	 close(ddt_tty);
	 ddt_tty = -1;
       };
      if (usr_tty >= 0) {
	 close(usr_tty);
	 usr_tty = -1;
       };
    };

   DDT_msg_exit();

   exit(sts);
};




/************************************************************************/
/*									*/
/*	DDT_printf -- Printf input to the debugger			*/
/*	DDT_ddt_input -- Send input to the debugger			*/
/*	DDT_send_interrupt -- send interrupt to the debugger		*/
/*	DDT_send_eof -- send eof to the debugger			*/
/*									*/
/************************************************************************/


/*VARARGS1*/

Integer
DDT_printf(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   Integer fg;
   Character buf[2048];

   sprintf(buf,pat,a1,a2,a3,a4,a5,a6,a7,a8);

   if (trace_file != NULL) {
      fprintf(trace_file,"%s",buf);
      fflush(trace_file);
    };

   /* should handle line too long by dividing here */

   fg = send_input(ddt_tty,buf);

   return fg;
};





Integer
DDT_ddt_input(st)
   String st;
{
   Integer fg;

   if (trace_file != NULL) {
      fprintf(trace_file,"%s",st);
      fflush(trace_file);
    };

   fg = send_input(ddt_tty,st);

   return fg;
};





void
DDT_send_interrupt()
{
   Character buf[2];

   if (ddt_process <= 0) return;

   buf[0] = intr_ch;
   buf[1] = 0;
   send_input(ddt_tty,buf);
};





void
DDT_send_eof()
{
   Character buf[2];

   if (ddt_process <= 0) return;

   buf[0] = eof_ch;
   buf[1] = 0;
   send_input(ddt_tty,buf);
};





/************************************************************************/
/*									*/
/*	DDT_user_input -- Send input to the user process		*/
/*	DDT_user_eof -- send eof to user process			*/
/*									*/
/************************************************************************/


Integer
DDT_user_input(st)
   String st;
{
   Integer fg,tty;

   if (ddt_process <= 0) return FALSE;

   tty = usr_input;
   if (tty < 0) tty = ddt_tty;

   if (trace_file != NULL) {
      fprintf(trace_file,"%s",st);
      fflush(trace_file);
    };

   fg = send_input(tty,st);

   return fg;
};





void
DDT_user_eof()
{
   Character buf[2];
   Integer tty;

   if (ddt_process <= 0) return;

   tty = usr_input;
   if (tty < 0) tty = ddt_tty;

   buf[0] = eof_ch;
   buf[1] = 0;
   send_input(tty,buf);
};





/************************************************************************/
/*									*/
/*	DDT_wait_debugger -- wait for debugger to respond		*/
/*									*/
/************************************************************************/


void
DDT_wait_debugger()
{
   Integer rmask[2];
   struct timeval to;
   Integer i;

   if (ddt_tty < 0) return;

   rmask[0] = 1 << ddt_tty;

   to.tv_usec = 0;
   to.tv_sec = 0;

   i = select(ddt_tty+1,rmask,NULL,NULL,NULL);

   if (i > 0) ddt_from_debug(rmask,0,0);
};






/************************************************************************/
/*									*/
/*	run_debug -- run the debugger in its own process		*/
/*									*/
/************************************************************************/


static void
run_debug(sys,cor,ttynm,utynm)
   String sys;
   String cor;
   Character ttynm[];
   Character utynm[];
{
   Integer dbslv,usslv;
   struct  sgttyb b;
   struct  tchars tc;
   struct  ltchars lc;
   int	   lb;
   int	   l,i,j;
   int	   pgrp;
   Integer tty;
   extern int errno;

   if (sys == NULL) sys = "a.out";
   strcpy(sys_name,sys);
   DDT__system_name = sys_name;

   i = open("/dev/tty",2);
   tty = (i > 0 ? i : 1);
   ioctl(tty, TIOCGETP, (char *)&b);
   ioctl(tty, TIOCGETC, (char *)&tc);
   ioctl(tty, TIOCGETD, (char *)&l);
   ioctl(tty, TIOCGLTC, (char *)&lc);
   ioctl(tty, TIOCLGET, (char *)&lb);
   if (i > 2) {
      ioctl(i,TIOCNOTTY,0);
      close(i);
    };

   ttynm[strlen("/dev/")] = 't';
   dbslv = open(ttynm, 2);
   if (dbslv < 0) {
      printf("failed to open debug slave tty %s\n",ttynm);
      return;
    };

   utynm[strlen("/dev/")] = 't';
   usslv = open(utynm, 2);
   if (usslv < 0) {
      printf("failed to open user slave tty %s\n",ttynm);
      return;
    };

/* b.sg_flags &= ~CRMOD;	*/
   b.sg_flags &= ~ECHO;
   b.sg_flags &= ~XTABS;
   lc.t_suspc = -1;
   lc.t_dsuspc = -1;

   pgrp = getpid();
   setpgrp(0,pgrp);
   ioctl(dbslv,TIOCSPGRP,&pgrp);
   ioctl(usslv,TIOCSPGRP,&pgrp);

   ioctl(dbslv, TIOCSETP, (char *)&b);
   ioctl(dbslv, TIOCSETC, (char *)&tc);
   ioctl(dbslv, TIOCSLTC, (char *)&lc);
   ioctl(dbslv, TIOCLSET, (char *)&lb);
   ioctl(dbslv, TIOCSETD, (char *)&l);

   ioctl(usslv, TIOCSETP, (char *)&b);
   ioctl(usslv, TIOCSETC, (char *)&tc);
   ioctl(usslv, TIOCSLTC, (char *)&lc);
   ioctl(usslv, TIOCLSET, (char *)&lb);
   ioctl(usslv, TIOCSETD, (char *)&l);

   dup2(dbslv,0);
   dup2(dbslv,1);
   dup2(dbslv,2);		/* usslv??? */

   j = getdtablesize();
   for (i = 3; i < j; ++i) {
      if (i != usslv) close(i);
    };

   execl(DDT_x_debugger_path(),DDT_x_debugger_name(),sys,cor,0);
}





/************************************************************************/
/*									*/
/*	get_ptty -- get pseudo tty					*/
/*									*/
/************************************************************************/


static Integer
get_ptty(ttynm)
   Character ttynm[];
{
   Character c;
   struct stat stb;
   Integer i,master,slave;

   strcpy(ttynm,"/dev/ptyXX");

   for (c = 'p'; c <= 's'; c++) {
      ttynm[strlen("/dev/pty")] = c;
      ttynm[strlen("/dev/ptyp")] = '0';
      if (stat(ttynm, &stb) < 0)
	 break;
      for (i = 0; i < 16; i++) {
	 ttynm[strlen("/dev/ptyp")] = "0123456789abcdef"[i];
	 master = open(ttynm, 2);
	 if (master < 0) continue;
	 ttynm[strlen("/dev/")] = 't';
	 slave = open(ttynm,2);
	 close(slave);
	 ttynm[strlen("/dev/")] = 'p';
	 if (slave >= 0) return master;
	 close(master);
       };
    };

   printf("Out of pty's\n");
   exit(1);
   /*NOTREACHED*/
};





/************************************************************************/
/*									*/
/*	ddt_from_debug -- handle output on tty from debugger		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
ddt_from_debug(rfg,wfg,xfg)
   Integer rfg[];
   Integer wfg[];
   Integer xfg[];
{
   Integer ct,i;
   Character buf[1025];
   String s;
   extern int errno;

   errno = 0;
   while ((ct = read(ddt_tty,buf,1024)) > 0) {
      s = buf;
      for (i = 0; i < ct; ++i) {
	 if (buf[i] != 0) *s++ = buf[i];
       };
      *s = 0;
      if (trace_file != NULL) {
	 fprintf(trace_file,"%s",buf);
	 fflush(trace_file);
       };

      for (s = buf; *s != 0; ++s) {
	 if (*s == '\r' && (s[1] == '\n' || s[1] == 0)) ;
	 else if (*s != '\n') {
	    if (outcnt < OUTLINE_SIZE-1) outline[outcnt++] = *s;
	  }
	 else {
	    if (outcnt > 0) {
	       outline[outcnt] = 0;
	       DDT_state_msg(outline);
	     };
	    outcnt = 0;
	  };
       };
    };

   if (ct < 0 && errno != EWOULDBLOCK) {
      abort_handler();
    };

   outline[outcnt] = 0;
   if (outcnt != 0 && DDT_state_prompt(outline)) outcnt = 0;
};





/************************************************************************/
/*									*/
/*	ddt_from_user -- handle output on tty from user 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
ddt_from_user(rfg,wfg,xfg)
   Integer rfg[];
   Integer wfg[];
   Integer xfg[];
{
   Integer ct,i;
   Character buf[1025];
   String s;

   while ((ct = read(usr_tty,buf,1024)) > 0) {
      s = buf;
      for (i = 0; i < ct; ++i) {
	 if (buf[i] != 0) *s++ = buf[i];
       };
      *s = 0;
      if (trace_file != NULL) {
	 fprintf(trace_file,"%s",buf);
	 fflush(trace_file);
       };
      DDT_msg_user_out(buf);
    };

   if (ct < 0 && errno != EWOULDBLOCK) {
      abort_handler();
    };
};





/************************************************************************/
/*									*/
/*	send_input -- send input to a tty				*/
/*									*/
/************************************************************************/


static Boolean
send_input(tty,st)
   Integer tty;
   String st;
{
   Integer ln,i;

   if (tty < 0) return FALSE;

   if (st != NULL) {
      ln = strlen(st);
      i = write(tty,st,ln);
      if (i < 0 && errno != EWOULDBLOCK) {
	 abort_handler();
       };
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	hup_handler -- handle hang up by cleanly terminating dbx	*/
/*									*/
/************************************************************************/


static void
hup_handler()
{
   if (ddt_process > 0) {
      DDT_send_interrupt();
    };

   DDT_stop_debug(0);

   exit(0);
};





/************************************************************************/
/*									*/
/*	abort_handler -- handle debuger abort				*/
/*									*/
/************************************************************************/


static void
abort_handler()
{
   if (ddt_process > 0) {
      printf("DDT: System debugger aborted.  Sorry...\n");
      DDT_stop_debug(1);
    }
   else DDT_state_abort();
};





/* end of ddtrun.c */
