/************************************************************************/
/*									*/
/*		ddtpascal.c						*/
/*									*/
/*	Pascal mapping routines for ddt debugger interface		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"
#include "ddt_names.h"





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		make_lowercase();
static	String		pascal_disp_format();




/************************************************************************/
/*									*/
/*	DDT_pascal_init -- module initialization			*/
/*									*/
/************************************************************************/


void
DDT_pascal_init()
{
};






/************************************************************************/
/*									*/
/*	DDT_pascal_fix_function -- convert user function name		*/
/*									*/
/************************************************************************/


String
DDT_pascal_fix_function(f,buf)
   String f;
   String buf;
{
   if (!DDT_symbol_funct_info(f,NULL,NULL,NULL)) {
      make_lowercase(f,buf);
      if (DDT_symbol_funct_info(buf,NULL,NULL,NULL)) f = buf;
    };

   return f;
};





/************************************************************************/
/*									*/
/*	DDT_pascal_fix_variable -- map user var to pascal name		*/
/*									*/
/************************************************************************/


String
DDT_pascal_fix_variable(v,buf)
   String v;
   String buf;
{
   Integer i;

   if (DDT_x_valid_name(v)) return v;

   strcpy(buf,v);
   i = strlen(buf)-1;

   while (i >= 0) {
      if (isupper(buf[i])) buf[i] = tolower(buf[i]);
      else if (buf[i] == '`' && DDT_x_valid_name(buf)) return buf;
      --i;
    };

   if (DDT_x_valid_name(buf)) v = buf;

   return v;
};




/************************************************************************/
/*									*/
/*	DDT_pascal_fix_field -- convert user field name to pascal name	*/
/*									*/
/************************************************************************/


String
DDT_pascal_fix_field(fld,lhs,buf)
   String fld;
   String lhs;
   String buf;
{
   Character xbuf[1024];

   sprintf(xbuf,"((%s).%s)",lhs,fld);
   if (!DDT_x_valid_name(xbuf)) {
      make_lowercase(fld,buf);
      sprintf(xbuf,"((%s).%s)",lhs,buf);
      if (DDT_x_valid_name(xbuf)) fld = buf;
    };

   return fld;
};





/************************************************************************/
/*									*/
/*	DDT_pascal_fix_outexpr -- format pascal expression		*/
/*									*/
/************************************************************************/


String
DDT_pascal_fix_outexpr(expr,calfg,ebuf)
   String expr;
   Boolean calfg;
   String ebuf;
{
   String s;

   s = pascal_disp_format(expr,ebuf);

   return s;
};







/************************************************************************/
/*									*/
/*	make_lowercase -- convert name to all lowercase 		*/
/*									*/
/************************************************************************/


static String
make_lowercase(nm,buf)
   String nm;
   String buf;
{
   String s;

   for (s = buf; *nm != 0; ++nm) {
      if (isupper(*nm)) *s++ = tolower(*nm);
      else *s++ = *nm;
    };

   *s = 0;

   return buf;
};






/************************************************************************/
/*									*/
/*	pascal_disp_format -- handle reformating pascal output		*/
/*									*/
/************************************************************************/


static String
pascal_disp_format(txt,buf)
   String txt;
   String buf;
{
   String s,t;
   Integer lvl,i;
   Boolean fg,lfg;

   s = index(txt,'\n');
   if (s == NULL) return txt;

   lvl = 0;
   t = buf;
   s = txt;
   lfg = FALSE;

   while (*s != 0) {
      fg = TRUE;
      if (strncmp(s,"record\n",7) == 0)  ++lvl;
      else if (strncmp(s,"(\n",2) == 0) ++lvl;
      else if (strncmp(s,"end\n",4) == 0) --lvl;
      else if (strncmp(s,")\n",2) == 0) --lvl;
      else if (lvl == 1 && STREQL(s,"end") || STREQL(s,")")) {
	 *t++ = '\n';
	 while (*s != 0) *t++ = *s++;
	 break;
       }
      else fg = FALSE;

      if (fg) {
	 if (lfg && (*s == 'r' || *s == '(')) fg = FALSE;
	 while (*s != '\n' && *s != 0) *t++ = *s++;
	 if (*s == 0) break;
       };

      if (isalnum(*s)) while (isalnum(*s) || *s == '_') *t++ = *s++;

      if (*s == '\n') {
	 if (fg || lvl == 0) {
	    *t++= '\n';
	    for (i = 1; i < lvl; ++i) {
	       *t++ = ' ';
	       *t++ = ' ';
	     };
	  };
	 *t++ = ' ';
	 *t++ = ' ';
	 ++s;
	 while (isspace(*s)) ++s;
	 lfg = fg;
       }
      else if (*s == '\t') {
	 ++s;
	 *t++ = ' ';
	 *t++ = ' ';
       }
      else {
	 if (*s != 0) *t++ = *s++;
	 lfg = FALSE;
       };
    };

   *t = 0;

   return buf;
};





/* end of ddtpascal.c */
