/************************************************************************/
/*									*/
/*		ddtmsg.c						*/
/*									*/
/*	Output routines for DDT interface				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_OUTPUT	32




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


typedef struct _OUT_TBL {
   Integer qid;
   FILE * fid;
} OUT_TBL;


static	OUT_TBL 	output_table[MAX_OUTPUT];
static	Integer 	num_output;
static	Integer 	cur_qid;
static	FILE *		cur_output;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/






/************************************************************************/
/*									*/
/*	DDT_msg_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_msg_init()
{
   num_output = 0;
   cur_qid = 0;
   cur_output = stdout;
};





/************************************************************************/
/*									*/
/*	DDT_output_define -- define qid/file pair for output		*/
/*									*/
/************************************************************************/


void
DDT_output_define(name)
   String name;
{
   Integer i;

   if (cur_qid == 0) return;

   for (i = 0; i < num_output; ++i) {
      if (output_table[i].qid == cur_qid) break;
    };

   if (i >= num_output) i = num_output++;
   else {
      if (output_table[i].fid != NULL && output_table[i].fid != stdout)
	 fclose(output_table[i].fid);
      output_table[i].fid = NULL;
    };

   output_table[i].qid = cur_qid;
   if (name == NULL) {
      output_table[i].fid = NULL;
    }
   else {
      output_table[i].fid = fopen(name,"w");
    };

   if (output_table[i].fid == NULL) cur_output = stdout;
   else cur_output = output_table[i].fid;
};





/************************************************************************/
/*									*/
/*	DDT_output_set -- set output qid for command			*/
/*									*/
/************************************************************************/


void
DDT_output_set(qid)
   Integer qid;
{
   Integer i;

   cur_qid = qid;

   if (qid == 0) cur_output = stdout;
   else {
      for (i = 0; i < num_output; ++i) {
	 if (output_table[i].qid == qid) break;
       }
      if (i >= num_output || output_table[i].fid == NULL)
	 cur_output = stdout;
      else
	 cur_output = output_table[i].fid;
    };
};





/************************************************************************/
/*									*/
/*	DDT_msg_user_out -- handle output from user			*/
/*									*/
/************************************************************************/


void
DDT_msg_user_out(text)
   String text;
{
   printf("%s",text);
   fflush(stdout);
};





/************************************************************************/
/*									*/
/*	DDT_msg_debug_out -- handle output from DDT to client		*/
/*	DDT_mprint -- print and format output to client 		*/
/*									*/
/************************************************************************/


void
DDT_msg_debug_out(text)
   String text;
{
   fprintf(cur_output,"%s\n",text);
};





/*VARARGS1*/

void
DDT_mprint(pat,a1,a2,a3,a4,a5,a6,a7,a8)
   String pat;
   Universal a1,a2,a3,a4,a5,a6,a7,a8;
{
   if (DDT_state_user_reply(pat,a1,a2,a3,a4,a5,a6,a7,a8)) return;

   fprintf(cur_output,pat,a1,a2,a3,a4,a5,a6,a7,a8);
};





/************************************************************************/
/*									*/
/*	DDT_msg_focus -- broadcast location in source for current focus */
/*									*/
/************************************************************************/


void
DDT_msg_focus(locp)
   LOCATION * locp;
{
   String file,func;

   if (DDT__standalone || locp->file == NULL) return;
   else file = DDT_model_real_name(locp->file);

   if (locp->func == NULL) func = "*";
   else func = locp->func;

   MSGsenda("DEBUG FOCUS %s %s %S %d",DDT__system_name,file,func,locp->line);
};





/************************************************************************/
/*									*/
/*	DDT_msg_at_line -- broadcast current execution location 	*/
/*									*/
/************************************************************************/


void
DDT_msg_at_line(locp)
   LOCATION * locp;
{
   String file,func;

   if (DDT__standalone) return;

   file = locp->file;
   if (locp->line > 0) {
      if (file == NULL) file = DDT_model_default_file();
      if (file == NULL) return;
      file = DDT_model_real_name(file);
    }
   else {
      if (file == NULL) file = "*";
      else file = DDT_model_real_name(file);
    };

   if (locp->func == NULL) func = "*";
   else func = locp->func;

   MSGsenda("DEBUG AT %s %s %S %d",DDT__system_name,file,func,locp->line);
};





/************************************************************************/
/*									*/
/*	DDT_msg_src_at_line -- broadcast current execution loc in source*/
/*									*/
/************************************************************************/


void
DDT_msg_src_at_line(locp)
   LOCATION * locp;
{
   String file,func;

   if (DDT__standalone) return;

   file = locp->file;
   if (locp->line > 0) {
      if (file == NULL) file = DDT_model_default_file();
      if (file == NULL) return;
      file = DDT_model_real_name(file);
    }
   else {
      if (file == NULL) file = "*";
      else file = DDT_model_real_name(file);
    };

   if (locp->func == NULL) func = "*";
   else func = locp->func;

   MSGsenda("DEBUG ATSOURCE %s %s %S %d",DDT__system_name,file,func,locp->line);
};





/************************************************************************/
/*									*/
/*	DDT_msg_event -- broadcast add/delete of an event		*/
/*									*/
/************************************************************************/


void
DDT_msg_event(add,id,typ,file,line,text)
   Boolean add;
   Integer id;
   EVENT_ACTION typ;
   String file;
   Integer line;
   String text;
{
   Character buf[1024];

   if (DDT__standalone) return;

   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   sprintf(buf,"EVENT %s %s %d %s %s %d %c%s%c",
	      (add ? "ADD" : "REMOVE"),DDT__system_name,
	      id,DDT_event_name(typ),file,line,
	      LIT_STRING,text,LIT_STRING);

   MSGsend(buf);
};





/************************************************************************/
/*									*/
/*	DDT_msg_value -- broadcast value change message 		*/
/*									*/
/************************************************************************/


void
DDT_msg_value(file,line,expr,val)
   String file;
   Integer line;
   String expr;
   String val;
{
   if (DDT__standalone) return;

   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   MSGsenda("DEBUG VALUE %s %s %d %c%s%c %c%s%c",DDT__system_name,file,line,
	       LIT_STRING,expr,LIT_STRING,LIT_STRING,val,LIT_STRING);
};





/************************************************************************/
/*									*/
/*	DDT_msg_call -- broadcast call message				*/
/*	DDT_msg_return -- broadcast return message			*/
/*									*/
/************************************************************************/


void
DDT_msg_call(file,func,line,args,from)
   String file;
   String func;
   Integer line;
   String args;
   String from;
{
   if (DDT__standalone) return;

   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   MSGsenda("DEBUG ENTER %s %s %S %d %S %S",DDT__system_name,file,func,
	       line,args,from);
};





void
DDT_msg_return(file,func,line,rval)
   String file;
   String func;
   Integer line;
   String rval;
{
   if (DDT__standalone) return;

   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);
   if (rval == NULL) rval = "";

   MSGsenda("DEBUG EXIT %s %s %S %d %S",DDT__system_name,file,func,line,rval);
};





/************************************************************************/
/*									*/
/*	DDT_msg_ie -- broadcast interesting event message		*/
/*									*/
/************************************************************************/


void
DDT_msg_ie(trig,file,line,typ,val)
   Boolean trig;
   String file;
   Integer line;
   String typ;
   String val;
{
   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);
   if (val == NULL) val = "";

   if (trig) {
      MSGcalla("IE %s %s %s %d %s",typ,DDT__system_name,file,line,val);
    }
   else {
      MSGsenda("IE %s %s %s %d %s",typ,DDT__system_name,file,line,val);
    };
};





/************************************************************************/
/*									*/
/*	DDT_msg_update -- broadcast update message			*/
/*	DDT_msg_when -- broadcast when message				*/
/*									*/
/************************************************************************/


void
DDT_msg_update(file,line)
   String file;
   Integer line;
{
   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   MSGcalla("UPDATE %s %s %d",DDT__system_name,file,line);
};





void
DDT_msg_when(file,line,id)
   String file;
   Integer line;
   String id;
{
   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   MSGcalla("WHEN %s %s %d %s",DDT__system_name,file,line,id);
};





/************************************************************************/
/*									*/
/*	DDT_msg_clear -- broadcast clear message			*/
/*	DDT_msg_reset -- broadcast reset message			*/
/*									*/
/************************************************************************/


void
DDT_msg_clear()
{
   if (DDT__standalone) return;

   MSGsenda("DEBUG CLEAR %s",DDT__system_name);
};





void
DDT_msg_reset()
{
   if (DDT__standalone) return;

   MSGcalla("DEBUG RESET %s",DDT__system_name);
};





/********************************************************************************/
/*										*/
/*	DDT_msg_start_run -- start run message					*/
/*	DDT_msg_stop_run -- end run message					*/
/*										*/
/********************************************************************************/


void
DDT_msg_start_run()
{
   if (DDT__standalone) return;

   MSGcalla("DEBUG START %s",DDT__system_name);
};





void
DDT_msg_continue_run()
{
   if (DDT__standalone) return;

   MSGsenda("DEBUG CONTINUE %s",DDT__system_name);      /* CANT BE CALL */
};





void
DDT_msg_stop_run(sts,sig)
   Integer sts;
   String sig;
{
   if (DDT__standalone) return;

   if (sig == NULL) sig = "OK";

   MSGcalla("DEBUG STOP %s %d %s",DDT__system_name,sts,sig);
};





/************************************************************************/
/*									*/
/*	DDT_msg_stop_error -- indicate stopped for a signal		*/
/*									*/
/************************************************************************/


void
DDT_msg_stop_error(locp,sig)
   LOCATION * locp;
   Integer sig;
{
   String file,func,s;
   Character buf[1024],bufa[1024];

   if (DDT__standalone) return;

   file = locp->file;
   if (locp->line > 0) {
      if (file == NULL) file = DDT_model_default_file();
      if (file == NULL) return;
      file = DDT_model_real_name(file);
    }
   else {
      if (file == NULL) file = "*";
      else file = DDT_model_real_name(file);
    };

   if (locp->func == NULL) func = "*";
   else func = locp->func;

   if (file == NULL) s = NULL;
   else {
      s = rindex(file,'/');
      if (s == NULL) s = file;
      else ++s;
    };

   DDT_position(bufa,s,func,locp->line,0);
   sprintf(buf,"%s %s",DDT__signal_table[sig].desc,bufa);

   MSGsenda("DEBUG STOP_ERROR %s %s %S %d %d %S",DDT__system_name,file,func,locp->line,
	       sig,buf);
};






/************************************************************************/
/*									*/
/*	DDT_msg_where -- broadcast where line message			*/
/*	DDT_msg_where_dump -- broadcast where dump message		*/
/*	DDT_msg_where_begin -- start broadcast of where messages	*/
/*	DDT_msg_where_end -- broadcast end of where message		*/
/*									*/
/************************************************************************/


void
DDT_msg_where(lvl,file,func,line,addr,args)
   Integer lvl;
   String file;
   String func;
   Integer line;
   Integer addr;
   String args;
{
   Character mode;

   if (DDT__standalone) return;

   if (file == NULL) file = "*";
   else file = DDT_model_real_name(file);

   if (line != 0) mode = 'L';
   else if (addr != 0) mode = 'A';
   else if (STRNEQ(func,"*") || STRNEQ(args,"*")) mode = 'L';
   else mode = 'G';

   MSGsenda("WHERE %s %d %s %S %d %d %S %c",DDT__system_name,
	       lvl,file,func,line,addr,args,mode);
};





void
DDT_msg_where_dump(lvl,name,val,ct)
   Integer lvl;
   String name;
   String val;
   Integer ct;
{
   Float f;
   Character buf[16];
   Character vbf[256];
   String s;

   if (DDT__standalone) return;

   f = ct;
   f /= 1000.0;
   f += lvl;
   sprintf(buf,"%f",f);

   if (strlen(val) > 150) {
      strncpy(vbf,val,150);
      vbf[150] = 0;
    }
   else {
      strcpy(vbf,val);
    };

   val = vbf;
   for (s = val; *s != 0; ++s) if (*s == '\n') *s = ' ';

   MSGsenda("WHERE_DUMP %s %s %S %S",DDT__system_name,buf,name,val);
};





void
DDT_msg_where_begin()
{
   if (DDT__standalone) return;

   MSGsenda("WHERE_BEGIN %s",DDT__system_name);
};





void
DDT_msg_where_end(lvl)
   Integer lvl;
{
   if (DDT__standalone) return;

   MSGsenda("WHERE_END %s %d",DDT__system_name,lvl);
};





/************************************************************************/
/*									*/
/*	DDT_msg_system -- identify system being debugged		*/
/*	DDT_msg_new_system -- identify system change			*/
/*									*/
/************************************************************************/


void
DDT_msg_system(fg)
   Boolean fg;
{
   if (DDT__standalone) return;

   if (fg) MSGsenda("DEBUG SYSTEM %s",DDT__system_name);
   else MSGsenda("DEBUG NO SYSTEM %s",DDT__system_name);
};




void
DDT_msg_new_system(new)
   String new;
{
   if (DDT__standalone) return;

   MSGsenda("DEBUG NEWSYS %s %s",DDT__system_name,new);
};





/************************************************************************/
/*									*/
/*	DDT_msg_exit -- handle exit message				*/
/*									*/
/************************************************************************/


void
DDT_msg_exit()
{
   if (DDT__standalone) return;

   MSGsenda("DEBUG FINISH %s",DDT__system_name);
};




/* end of ddtmsg.c */
