/************************************************************************/
/*									*/
/*		ddtmonitor.c						*/
/*									*/
/*	Monitoring routines for dbx maintenance 			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "ddt_local.h"
#include <setjmp.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_ARGS	16
#define MAX_FCTS	10240



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


typedef struct _COMMAND *	COMMAND;

typedef struct _COMMAND {
   String contents;
   Integer rid;
} COMMAND_INFO;




static	Sequence	user_queue;
static	Sequence	pgm_queue;
static	jmp_buf 	cmd_env;
static	Boolean 	env_ok;

static	Boolean 	force_run;



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		ddt_handler();
static	void		wd_handler();
static	void		stop_handler();
static	void		handle_action();
static	void		handle_assign();
static	void		handle_call();
static	void		handle_dump();
static	void		handle_eval();
static	void		handle_event_add();
static	void		handle_event_show();
static	void		handle_event_remove();
static	void		handle_event_change();
static	void		handle_run();
static	void		handle_set();
static	void		handle_show();
static	void		handle_stack();
static	void		handle_step();
static	void		handle_syminfo();
static	void		handle_view();
static	Boolean 	check_system();
static	Boolean 	match_system();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


typedef struct _MSG_TABLE {
   String auxd_name;
   voidFunction_Ptr handler;
   Integer numargs;
   PMAT_PAT pattern;
} MSG_TABLE;

static MSG_TABLE msg_table[] = {
   { "ACTION",          handle_action,          2,      NULL },
   { "ASSIGN",          handle_assign,          3,      NULL },
   { "CALL",            handle_call,            3,      NULL },
   { "DUMP",            handle_dump,            5,      NULL },
   { "EVAL",            handle_eval,            2,      NULL },
   { "EVENT_ADD",       handle_event_add,       9,      NULL },
   { "EVENT_SHOW",      handle_event_show,      9,      NULL },
   { "EVENT_REMOVE",    handle_event_remove,    9,      NULL },
   { "EVENT_CHANGE",    handle_event_change,    9,      NULL },
   { "RUN",             handle_run,             5,      NULL },
   { "SET",             handle_set,             3,      NULL },
   { "SHOW",            handle_show,            2,      NULL },
   { "STACK",           handle_stack,           4,      NULL },
   { "STEP",            handle_step,            4,      NULL },
   { "SYMINFO",         handle_syminfo,         6,      NULL },
   { "VIEW",            handle_view,            6,      NULL },

   { NULL, NULL, 0 }
};

#define WD_MSG		"MSG FILE_WD %1s"





/************************************************************************/
/*									*/
/*	DDT_monitor_init -- module initialization			*/
/*									*/
/************************************************************************/


void
DDT_monitor_init()
{
   register AUXD hd0,hd;
   register String s;
   register Integer i,j,ct;
   Universal args[MAX_ARGS];
   Character buf[1024];

   user_queue = NULL;
   pgm_queue = NULL;
   env_ok = FALSE;
   force_run = TRUE;

   if (DDT__norun) return;

   for (i = 0; i < MAX_ARGS; ++i) args[i] = 0;

   hd0 = AUXDget_handle(NULL,"DDT");
   hd = AUXDget_handle(hd0,"REQUESTS");

   args[0] = NULL;
   args[1] = 1;
   MSGregister("DDT %1r",ddt_handler,2,args);
   args[1] = 0;
   MSGregister("DDTR %1r",ddt_handler,2,args);

   s = AUXDget_info(hd0,"STOP");
   if (s != NULL) {
      args[0] = NULL;
      args[1] = NULL;
      args[2] = 1;
      sprintf(buf,"DDT %s",s);
      MSGregister(buf,stop_handler,3,args);
      args[2] = 0;
      sprintf(buf,"DDTR %s",s);
      MSGregister(buf,stop_handler,3,args);
    };

   MSGregister(WD_MSG,wd_handler,1,NULL);

   for (i = 0; msg_table[i].auxd_name != NULL; ++i) {
      s = AUXDget_info(hd,msg_table[i].auxd_name);
      if (s != NULL) {
	 ct = msg_table[i].numargs;
	 for (j = 0; j < ct; ++j) args[j] = NULL;
	 args[ct] = 0;
	 msg_table[i].pattern = PMATmake_pattern(s,ct+1,args);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_command_process -- process next command			*/
/*									*/
/************************************************************************/


Boolean
DDT_command_process(user)
   Boolean user;
{
   Universal args[MAX_ARGS];
   Integer i,j;
   COMMAND c;

   if (user) {
      if (user_queue == NULL) return FALSE;
      c = CAR(COMMAND,user_queue);
      user_queue = CDRF(user_queue);
    }
   else {
      if (pgm_queue == NULL) return FALSE;
      c = CAR(COMMAND,pgm_queue);
      pgm_queue = CDRF(pgm_queue);
    };

   if (setjmp(cmd_env) == 0) {
      env_ok = TRUE;
      for (i = 0; msg_table[i].auxd_name != NULL; ++i) {
	 if (msg_table[i].pattern != NULL) {
	    if (PMATmatch(c->contents,msg_table[i].pattern,args) >= 0) {
	       j = msg_table[i].numargs;
	       args[j] = (Universal) (user ? 0 : 1);
	       args[j+1] = (Universal) c->rid;
	       (*msg_table[i].handler)(args[0],args[1],args[2],args[3],
					  args[4],args[5],args[6],args[7],
					  args[8],args[9],args[10],args[11],
					  args[12],args[13],args[14],args[15]);
	       PMATfree_args(msg_table[i].pattern,args);
	       break;
	     };
	  };
       };
      if (msg_table[i].auxd_name == NULL && c->rid >= 0) {
	 j = (user ? 0 : 1);
	 DDT_state_cmd_begin(j,c->rid,0,NULL);
	 DDT_mprint("Illegal command\n");
	 DDT_state_cmd_end(j,c->rid);
       };
    }
   else {
/*    DDT_state_cmd_abort((user ? 0 : 1),c->rid);	*/
    };

   env_ok = FALSE;
   SFREE(c->contents);
   free(c);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_command_abort -- abort current command if active		*/
/*									*/
/************************************************************************/


void
DDT_command_abort()
{
   if (env_ok) {
      longjmp(cmd_env,1);
    };
};





/************************************************************************/
/*									*/
/*	ddt_handler -- handle DDT style messages			*/
/*	wd_handler -- handle set WD message				*/
/*									*/
/************************************************************************/


static void
ddt_handler(body,user,rid)
   String body;
   Boolean user;
   Integer rid;
{
   COMMAND c;

   c = PALLOC(COMMAND_INFO);
   c->contents = SALLOC(body);
   c->rid = rid;

   if (user) {
      user_queue = APPEND(c,user_queue);
    }
   else {
      pgm_queue = APPEND(c,pgm_queue);
    };
};





static void
wd_handler(file,rid)
   String file;
   Integer rid;
{
   COMMAND c;
   Character buf[1024];

   c = PALLOC(COMMAND_INFO);
   sprintf(buf,"SET * WD %s",file);
   c->contents = SALLOC(buf);
   c->rid = rid;

   pgm_queue = APPEND(c,pgm_queue);
};





static void
stop_handler(sys,user,rid)
   String sys;
   Boolean user;
   Integer rid;
{
   if (match_system(sys,NULL,NULL,NULL)) {
      DDT_state_stop_run();
    };

   if (rid >= 0) MSGreply(rid,NULL);
};





/************************************************************************/
/*									*/
/*	command_routines -- handle_<cmd>				*/
/*									*/
/************************************************************************/


static void
handle_action(sys,act,quiet,rid)
   String sys;
   String act;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   DDT_state_stop_run();

   if (STREQL(act,"INIT")) {
      DDT_event_clear(TRUE);
      DDT_x_reinit(NULL,NULL);
      DDT_model_clear(TRUE);
      DDT_mprint("\nDebugger reinitialized...\n");
    }
   else if (STREQL(act,"RESET")) {
      DDT_x_reinit(NULL,NULL);
      DDT_model_clear(FALSE);
      DDT_mprint("\nDebugger reset...\n");
    }
   else if (STREQL(act,"QUIT")) {
      DDT_mprint("Exiting debugger...\n");
      DDT_stop_debug(0);
    }
   else if (STREQL(act,"KILL")) {
      DDT_x_kill(FALSE);
      DDT_mprint("Program terminated\n");
    }
   else if (STREQL(act,"STOP")) {
      DDT_mprint("Program stopped");
      DDT_print_current_position();
      DDT_mprint("\n");
    };

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_assign(sys,var,expr,quiet,rid)
   String sys;
   String var;
   String expr;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   DDT_x_assign(var,expr);

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_call(sys,rtn,args,quiet,rid)
   String sys;
   String rtn;
   String args;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   if (!DDT_valid_string(args)) args = "";

   DDT_x_call(rtn,args);

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_dump(sys,from,to,len,fmt,quiet,rid)
   String sys;
   String from;
   String to;
   Integer len;
   String fmt;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   if (!DDT_valid_string(to)) to = NULL;
   if (!DDT_valid_string(fmt)) fmt = "";

   DDT_x_dump(from,to,len,fmt);

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_eval(sys,expr,quiet,rid)
   String sys;
   String expr;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   DDT_x_print(expr);

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_event_add(sys,file,func,line,expr,cond,addr,sact,fgs,quiet,rid)
   String sys;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   String sact;
   Integer fgs;
   Integer quiet;
   Integer rid;
{
   EVENT_ACTION act;
   String fct[MAX_FCTS];
   Integer i,ct;
   String s,nfile;
   Integer eline;

   if (!check_system(sys,file,quiet,rid)) return;

   act = DDT_event_type(sact);
   if ((fgs & DDT_EVENT_NAME) != 0 || act == EVENT_NONE) {
      if ((fgs & DDT_EVENT_TRIGGER) != 0) act = EVENT_TRIGGER;
      else act = EVENT_EVENT;
    }
   else {
      sact = NULL;
    };

   if (act != EVENT_NONE) {
      if (!DDT_valid_string(file)) file = NULL;
      if (!DDT_valid_string(func)) func = NULL;
      if (!DDT_valid_string(expr)) expr = NULL;
      if (!DDT_valid_string(cond)) cond = NULL;
      if ((fgs & (DDT_EVENT_INTERNAL|DDT_EVENT_EXTERNAL)) == 0)
	 fgs |= DDT_EVENT_EXTERNAL;

      if ((fgs & DDT_EVENT_PATTERN) == 0) {
	 DDT_x_add_event(act,file,func,line,expr,cond,addr,sact,
			    ((fgs&DDT_EVENT_INTERNAL) != 0),
			    ((fgs&DDT_EVENT_EXTERNAL) != 0));
       }
      else {
	 ct = DDT_symbol_functions(MAX_FCTS,fct,FALSE);

	 if (func != NULL && (func[0] == '"' || func[0] == '\'')) {
	    i = strlen(func)-1;
	    if (func[i] == func[0]) func[i] = 0;
	    ++func;
	    if (func[0] == 0) func = NULL;
	  };

	 if (file != NULL) file = DDT_symbol_fix_file_name(file);

	 if (func != NULL) {
	    s = re_comp(func);
	    if (s != NULL) {
	       DDT_mprint("%s\n",s);
	       ct = 0;
	     };
	  };

	 for (i = 0; i < ct; ++i) {
	    if (STREQL(fct[i],"main") || STREQL(fct[i],"program")) continue;
	    if (fct[i][0] == '#') continue;
	    if (func == NULL || re_exec(fct[i])) {
	       if (!DDT_symbol_funct_info(fct[i],&nfile,&line,&eline)) continue;
	       if (file != NULL && STRNEQ(file,nfile)) continue;
	       if (act == EVENT_STOPEXIT) line = eline;
	       if (line == 0) continue;
	       DDT_x_add_event(act,nfile,fct[i],line,expr,cond,addr,sact,
				  ((fgs&DDT_EVENT_INTERNAL) != 0),
				  ((fgs&DDT_EVENT_EXTERNAL) != 0));
	     };
	  };
       };
    };

   DDT_state_cmd_end(quiet,rid);
};






static void
handle_event_show(sys,file,func,line,expr,cond,addr,sact,id,quiet,rid)
   String sys;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   String sact;
   Integer id;
   Integer quiet;
   Integer rid;
{
   EVENT_ACTION act;

   if (!check_system(sys,file,quiet,rid)) return;

   act = DDT_event_type(sact);

   DDT_for_each_event(file,func,line,expr,cond,addr,act,id,DDT_event_show,TRUE);

   DDT_state_cmd_end(quiet,rid);
};






static void
handle_event_remove(sys,file,func,line,expr,cond,addr,sact,id,quiet,rid)
   String sys;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   String sact;
   Integer id;
   Integer quiet;
   Integer rid;
{
   EVENT_ACTION act;

   if (!check_system(sys,file,quiet,rid)) return;

   act = DDT_event_type(sact);

   DDT_for_each_event(file,func,line,expr,cond,addr,act,id,DDT_event_remove,TRUE);

   DDT_state_cmd_end(quiet,rid);
};






static void
handle_event_change(sys,file,func,line,expr,cond,addr,sact,id,quiet,rid)
   String sys;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   String sact;
   Integer id;
   Integer quiet;
   Integer rid;
{
   EVENT_ACTION act;

   if (!check_system(sys,file,quiet,rid)) return;

   act = DDT_event_type(sact);

   DDT_for_each_event(file,func,0,expr,cond,addr,act,id,DDT_event_change,line);

   DDT_state_cmd_end(quiet,rid);
};






static void
handle_run(sys,args,in,out,new,quiet,rid)
   String sys;
   String args;
   String in;
   String out;
   Boolean new;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   if (!DDT_valid_string(args)) args = NULL;
   if (!DDT_valid_string(in)) in = NULL;
   if (!DDT_valid_string(out)) out = NULL;

   if (new || in != NULL || out != NULL) {
      DDT_model_in_file(in);
      DDT_model_out_file(out);
    };

   args = DDT_model_run_args(args,new);
   in = DDT_model_inq_in_file(in,new);
   out = DDT_model_inq_out_file(out,new);

   if (!DDT_symbol_check_dates() && !force_run) {
      DDT_state_cmd_end(quiet,rid);
      return;
    };

   DDT_model_show_run();

   if (!DDT_x_run(args,in,out)) {
      DDT_state_cmd_end(quiet,rid);
    };
};





static void
handle_set(sys,what,arg,quiet,rid)
   String sys;
   String what;
   String arg;
   Integer quiet;
   Integer rid;
{
   Boolean fg;
   LOCATION loc;

   if (!check_system(sys,NULL,quiet,rid)) return;

   if (!DDT_valid_string(arg)) arg = NULL;

   if (STREQL(what,"INFILE")) DDT_model_in_file(arg);
   else if (STREQL(what,"OUTFILE")) DDT_model_out_file(arg);
   else if (STREQL(what,"USER_TTY")) DDT_model_default_tty(arg);
   else if (STREQL(what,"CATCH")) DDT_model_sigset(arg,SIGNAL_CATCH);
   else if (STREQL(what,"IGNORE")) DDT_model_sigset(arg,SIGNAL_IGNORE);
   else if (STREQL(what,"PRINTIGN")) DDT_model_sigset(arg,SIGNAL_PRINTIGN);
   else if (STREQL(what,"PRINTUSE")) DDT_model_sigset(arg,SIGNAL_PRINTUSE);
   else if (STREQL(what,"USE")) DDT_model_use(arg);
   else if (STREQL(what,"NEWSYS")) DDT_model_newsys(arg);
   else if (STREQL(what,"PROG")) DDT_state_user_level(TRUE);
   else if (STREQL(what,"ENDPROG")) DDT_state_user_level(FALSE);
   else if (STREQL(what,"DDTOUT")) DDT_output_define(arg);
   else if (STREQL(what,"FORCE_RUN")) force_run = atol(arg);
   else if (STREQL(what,"RUN_ARGS")) DDT_model_run_args(arg,TRUE);
   else if (STREQL(what,"ENV")) DDT_model_set_env(arg);
   else if (STREQL(what,"CPLUSPLUS")) DDT__cplusplus = atol(arg);
   else if (STREQL(what,"PICKLEVEL")) DDT__picklevel = atol(arg);
   else if (STREQL(what,"SEARCH") || STREQL(what,"BSEARCH")) {
      fg = DDT_x_search((what[0] != 'B'),arg);
      if (fg) {
	 DDT_x_current_line();
	 DDT_model_inq_location(&loc);
	 DDT_mprint("Focus is");
	 DDT_print_position(loc.file,loc.func,loc.line,0);
	 DDT_mprint("\n");
       };
    }
   else if (DDT_trace_set_where(what,arg)) ;
   else DDT_x_set_option(what,arg);

   DDT_state_cmd_end(quiet,rid);
};




static void
handle_show(sys,what,quiet,rid)
   String sys;
   String what;
   Integer quiet;
   Integer rid;
{
   LOCATION loc;
   Integer addr;

   if (!check_system(sys,NULL,quiet,rid)) return;

   if (STREQL(what,"SIGNAL")) DDT_model_sig_status();
   else if (STREQL(what,"RUN")) DDT_model_show_run();
   else if (STREQL(what,"USE")) DDT_model_show_use();
   else if (STREQL(what,"ENV")) DDT_model_show_env(NULL);
   else if (STREQL(what,"WD")) DDT_x_show("WD",NULL);
   else if (STREQL(what,"SYSTEM")) {
      DDT_model_show_system();
    }
   else if (STREQL(what,"LOCATION")) {
      DDT_model_inq_exec_location(&loc,&addr);
      DDT_msg_at_line(&loc);
      DDT_mprint("Currently");
      DDT_print_position(loc.file,loc.func,loc.line,addr);
      DDT_mprint("\n");
    }
   else if (STREQL(what,"FOCUS")) {
      DDT_model_inq_location(&loc);
      DDT_mprint("Focus is");
      DDT_print_position(loc.file,loc.func,loc.line,0);
      DDT_msg_focus(&loc);
      DDT_mprint("\n");
    };

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_stack(sys,from,to,dump,quiet,rid)
   String sys;
   Integer from;
   Integer to;
   Boolean dump;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,NULL,quiet,rid)) return;

   DDT_trace_print_where(from,to,dump);

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_step(sys,count,sig,type,quiet,rid)
   String sys;
   Integer count;
   String sig;
   String type;
   Integer quiet;
   Integer rid;
{
   Boolean over,inst,fg,retn;
   Integer sign;

   if (!check_system(sys,NULL,quiet,rid)) return;

   if (!DDT_valid_string(sig)) sig = NULL;
   if (sig == NULL) sign = 0;
   else if (isdigit(sig[0])) sign = atol(sig);
   else sign = DDT_model_signal_name(sig);

   if (STREQL(type,"CONT") || sign != 0) {
      if (sign >= 0) fg = DDT_x_continue(count,sign);
      else fg = FALSE;
    }
   else {
      if (STREQL(type,"NEXT") || STREQL(type,"NEXTI")) over = TRUE;
      else over = FALSE;
      if (STREQL(type,"STEPI") || STREQL(type,"NEXTI")) inst = TRUE;
      else inst = FALSE;
      if (STREQL(type,"RETURN")) retn = TRUE;
      else retn = FALSE;
      fg = DDT_x_step(count,over,inst,retn,TRUE);
    };

   if (!fg) {
      DDT_state_cmd_end(quiet,rid);
    };
};





static void
handle_syminfo(sys,what,file,func,line,name,quiet,rid)
   String sys;
   String what;
   String file;
   String func;
   Integer line;
   String name;
   Integer quiet;
   Integer rid;
{
   if (!check_system(sys,file,quiet,rid)) return;

   if (STREQL(what,"WHAT")) DDT_x_var_info(file,func,line,name,FALSE,FALSE,TRUE,NULL);
   else if (STREQL(what,"WHERE")) DDT_x_var_info(file,func,line,name,TRUE,FALSE,FALSE,NULL);
   else if (STREQL(what,"WHICH")) DDT_x_var_info(file,func,line,name,FALSE,TRUE,FALSE,NULL);
   else if (STREQL(what,"VARINFO")) {
      DDT_symbol_var_data(file,func,line,name);
    }
   else if (STREQL(what,"TYPEINFO")) {
      DDT_symbol_type_data(file,func,line,name);
    }
   else if (STREQL(what,"LIST_FILES")) {
      if (!DDT_valid_string(name)) name = NULL;
      DDT_model_list_files(name);
    }
   else if (STREQL(what,"LIST_FCTS")) {
      if (!DDT_valid_string(name)) name = NULL;
      DDT_model_list_functions(name);
    }
   else if (STREQL(what,"LIST_VARS")) {
      if (!DDT_valid_string(name)) name = NULL;
      DDT_model_list_variables(name);
    }
   else if (STREQL(what,"LIST_TYPS")) {
      if (!DDT_valid_string(name)) name = NULL;
      DDT_model_list_types(name);
    }
   else if (STREQL(what,"ENV")) {
      if (!DDT_valid_string(name)) name = NULL;
      DDT_model_show_env(name);
    }
   else if (STREQL(what,"DYNAMIC_TYPE")) {
      Character buf[256];
      String s;

      s = DDT_symbol_dynamic_type(name,256,buf);
      if (s == NULL) DDT_x_var_info(file,func,line,name,FALSE,FALSE,TRUE,NULL);
      else DDT_mprint("%s\n",buf);
    };

   DDT_state_cmd_end(quiet,rid);
};





static void
handle_view(sys,file,func,line,count,stkdelta,quiet,rid)
   String sys;
   String file;
   String func;
   Integer line;
   Integer count;
   Integer stkdelta;
   Integer quiet;
   Integer rid;
{
   register Boolean fg;
   Integer f,t;
   LOCATION loc;

   if (!check_system(sys,file,quiet,rid)) return;

   if (stkdelta != 0) {
      fg = DDT_x_updown(stkdelta);
    }
   else {
      fg = DDT_x_set_view(file,func,line);
    };

   if (fg && count != 0) {
      DDT_model_inq_location(&loc);
      if (count > 0) {
	 f = loc.line;
	 t = f + count -1;
       }
      else {
	 f = loc.line + count;
	 t = f - count - count;
	 if (f < 1) f = 1;
       };
      DDT_x_list(f,t);
    }
   else if (fg) {
      DDT_model_inq_location(&loc);
      DDT_mprint("Focus is");
      DDT_print_position(loc.file,loc.func,loc.line,0);
      DDT_mprint("\n");
    };

   DDT_state_cmd_end(quiet,rid);
};





/************************************************************************/
/*									*/
/*	check_system -- check if we are the correct ddt server		*/
/*									*/
/************************************************************************/


static Boolean
check_system(sys,fil,quiet,rid)
   String sys;
   String fil;
   Boolean quiet;
   Integer rid;
{
   register Boolean fg;
   Character buf[128];
   String ctx;
   Integer qid;

   fg = match_system(sys,fil,&qid,buf);
   if (buf[0] != 0) ctx = buf;
   else ctx = NULL;

   if (!fg) {
      if (rid > 0) {
	 DDT_state_cmd_begin(quiet,rid,qid,ctx);
	 DDT_state_cmd_abort(quiet,rid);
       };
    }
   else {
      DDT_state_cmd_begin(quiet,rid,qid,ctx);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	match_system -- match given system name to ours 		*/
/*									*/
/************************************************************************/


static Boolean
match_system(sys,fil,qidp,bufp)
   String sys;
   String fil;
   Integer * qidp;
   String bufp;
{
   register Boolean fg;
   Character buf[128];
   String s,p,ctx;
   Integer qid,pid;

   if (!DDT_valid_string(sys)) sys = NULL;
   if (!DDT_valid_string(fil)) fil = NULL;

   fg = TRUE;
   qid = 0;
   ctx = NULL;

   if (sys != NULL) {
      s = sys;
      p = buf;
      while (*s != 0 && *s != '^') *p++ = *s++;
      *p = 0;
      if (STRNEQ(buf,DDT__system_name)) fg = FALSE;
      if (fg && *s == '^') {
	 ++s;
	 p = buf;
	 while (*s != 0 && *s != '^') *p++ = *s++;
	 *p = 0;
	 if (isdigit(buf[0])) {
	    pid = atol(buf);
	    if (pid != 0 && pid != getpid()) fg = FALSE;
	  };
       };
      if (fg && *s == '^') {
	 ++s;
	 p = buf;
	 while (*s != 0 && *s != '^') *p++ = *s++;
	 *p = 0;
	 qid = atol(buf);
       };
      if (fg && *s == '^') {
	 ++s;
	 p = buf;
	 while (*s != 0 && *s != '^') *p++ = *s++;
	 *p = 0;
	 ctx = buf;
       };
    }
   else if (fil != NULL && !DDT_model_check_file(fil)) fg = FALSE;

   if (bufp != NULL) {
      if (ctx == NULL) *bufp = 0;
      else strcpy(bufp,ctx);
    };

   if (qidp != NULL) {
      *qidp = qid;
    };

   return fg;
};





/* end of ddtmonitor.c */


