/************************************************************************/
/*									*/
/*		ddtmodel.c						*/
/*									*/
/*	Run time and debugger state model facilities for ddt interface	*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_FILES	512
#define MAX_FCTS	10240





/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	String		user_tty;
static	String		default_tty;
static	String		in_file;
static	String		out_file;
static	String		run_args;
static	String		last_in_file;
static	String		last_out_file;
static	String		core_file;

static	String		use_dirs;

static	LOCATION	current_focus;
static	LOCATION	current_exec;
static	Integer 	exec_address;

static	String		all_files[MAX_FILES];
static	String		real_names[MAX_FILES];
static	Integer 	num_files;

static	Boolean 	sys_valid;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		get_real_filename();



/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	DDT_model_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
DDT_model_init()
{
   user_tty = NULL;
   default_tty = NULL;
   in_file = NULL;
   last_in_file = NULL;
   out_file = NULL;
   last_out_file = NULL;
   run_args = NULL;
   core_file = NULL;

   use_dirs = SALLOC(".");

   current_focus.file = NULL;
   current_focus.func = NULL;
   current_focus.line = 0;
   current_exec.file = NULL;
   current_exec.func = NULL;
   current_exec.line = 0;
   exec_address = 0;

   num_files = 0;

   sys_valid = FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_model_clear -- clear model on reinitialization		*/
/*									*/
/************************************************************************/


void
DDT_model_clear(new)
   Boolean new;
{
   register Integer i;
   Boolean fg;

   num_files = 0;

   current_focus.file = NULL;
   current_focus.func = NULL;
   current_focus.line = 0;
   current_exec.file = NULL;
   current_exec.func = NULL;
   current_exec.line = 0;
   exec_address = 0;

   for (i = 0; i < MAX_FILES; ++i) real_names[i] = NULL;

   DDT_msg_clear();

   DDT_x_use(use_dirs);

   fg = DDT_x_system_info();
   if (fg) {
      num_files = DDT_symbol_files(MAX_FILES,all_files);
      for (i = 0; i < MAX_FILES; ++i) real_names[i] = NULL;
      DDT_x_stop();
      DDT_x_set_view(NULL,NULL,0);
      DDT_x_sigset(-1,0);
    };

   DDT_event_clear(new);

   DDT_msg_system(fg);

   DDT_msg_reset();

   sys_valid = fg;
};





/************************************************************************/
/*									*/
/*	DDT_model_newsys -- change system				*/
/*									*/
/************************************************************************/


void
DDT_model_newsys(name)
   String name;
{
   String s;
   Character buf[1024];

   strcpy(buf,name);
   s = index(buf,' ');
   if (s != NULL) {
      name = buf;
      *s++ = 0;
      while (isspace(*s)) ++s;
      if (*name == 0) name = NULL;
      if (*s == 0) s = NULL;
    };

   if (name != NULL && STRNEQ(name,"*")) {
      if (DDT__system_name != NULL & STRNEQ(DDT__system_name,name)) DDT_msg_new_system(name);
      DDT__system_name = SALLOC(name);
    };
   if (s != NULL) core_file = SALLOC(s);
   else if (name != NULL) core_file = NULL;

   DDT_event_clear(TRUE);
   DDT_x_reinit(name,s);
   DDT_model_clear(TRUE);
   DDT_mprint("\nDebugger reinitialized for %s\n",DDT__system_name);
   DDT_x_set_view(NULL,NULL,0);
};





/************************************************************************/
/*									*/
/*	DDT_model_show_system -- show current system			*/
/*									*/
/************************************************************************/


void
DDT_model_show_system()
{
   DDT_mprint("System: %s",DDT__system_name);
   if (core_file != NULL) {
      DDT_mprint(", core: %s",core_file);
    };

   DDT_mprint(", debug pid: %d\n",getpid());
};





/************************************************************************/
/*									*/
/*	DDT_model_user_tty -- specify tty for use input 		*/
/*	DDT_model_default_tty -- specify default tty for running	*/
/*	DDT_model_in_file -- set input file				*/
/*	DDT_model_out_file -- set output file				*/
/*	DDT_model_run_args -- set and return run arguments		*/
/*	DDT_model_inq_in_file -- return input file to use		*/
/*	DDT_model_inq_out_file -- return output file to use		*/
/*	DDT_model_show_run -- show current run string			*/
/*									*/
/************************************************************************/


void
DDT_model_user_tty(nm)
   String nm;
{
   if (user_tty != NULL) SFREE(user_tty);
   if (nm == NULL) user_tty = NULL;
   else user_tty = SALLOC(nm);

   if (in_file == NULL) DDT_model_in_file(NULL);
   if (out_file == NULL) DDT_model_out_file(NULL);
};





void
DDT_model_default_tty(nm)
   String nm;
{
   if (default_tty != NULL) SFREE(default_tty);
   if (nm == NULL) default_tty = NULL;
   else default_tty = SALLOC(nm);

   if (in_file == NULL || (user_tty != NULL && STREQL(in_file,user_tty)))
      DDT_model_in_file(NULL);
   if (out_file == NULL || (user_tty != NULL && STREQL(out_file,user_tty)))
      DDT_model_out_file(NULL);
};





void
DDT_model_in_file(name)
   String name;
{
   if (last_in_file != NULL && in_file != NULL && STREQL(last_in_file,in_file))
      last_in_file = NULL;

   if (in_file != NULL) SFREE(in_file);

   if (name == NULL) name = default_tty;
   if (name == NULL) name = user_tty;

   if (name == NULL) in_file = NULL;
   else in_file = SALLOC(name);
};





void
DDT_model_out_file(name)
   String name;
{
   if (last_out_file != NULL && out_file != NULL && STREQL(last_out_file,out_file))
      last_out_file = NULL;

   if (out_file != NULL) SFREE(out_file);

   if (name == NULL) name = default_tty;
   if (name == NULL) name = user_tty;

   if (name == NULL) out_file = NULL;
   else out_file = SALLOC(name);
};





String
DDT_model_run_args(args,newfg)
   String args;
   Boolean newfg;
{
   if (args != NULL || newfg) {
      if (args == NULL) args = "";
      if (run_args != NULL) SFREE(run_args);
      while (*args == ' ') ++args;
      run_args = SALLOC(args);
    };

   return run_args;
};





String
DDT_model_inq_in_file(in,newfg)
   String in;
   Boolean newfg;
{
   if (in != NULL || newfg) {
      if (in == NULL) in = in_file;
      if (last_in_file != NULL) SFREE(last_in_file);
      last_in_file = SALLOC(in);
    }
   else if (last_in_file == NULL && in_file != NULL) {
      last_in_file = SALLOC(in_file);
    };

   return last_in_file;
};





String
DDT_model_inq_out_file(out,newfg)
   String out;
   Boolean newfg;
{
   if (out != NULL || newfg) {
      if (out == NULL) out = out_file;
      if (last_out_file != NULL) SFREE(last_out_file);
      last_out_file = SALLOC(out);
    }
   else if (last_out_file == NULL && out_file != NULL) {
      last_out_file = SALLOC(out_file);
    };

   return last_out_file;
};





String
DDT_model_show_in_file()
{
   if (last_in_file == NULL ||
	  (default_tty != NULL && STREQL(last_in_file,default_tty)) ||
	  (user_tty != NULL && STREQL(last_in_file,user_tty))) return NULL;

   return last_in_file;
};




String
DDT_model_show_out_file()
{
   if (last_out_file == NULL ||
	  (default_tty != NULL && STREQL(last_in_file,default_tty)) ||
	  (user_tty != NULL && STREQL(last_out_file,user_tty))) return NULL;

   return last_out_file;
};





void
DDT_model_show_run()
{

   DDT_mprint("Running: %s",DDT__system_name);

   if (run_args != NULL) DDT_mprint(" %s",run_args);

   if (last_in_file != NULL &&
	  (user_tty == NULL || STRNEQ(last_in_file,user_tty)) &&
	  (default_tty == NULL || STRNEQ(last_in_file,default_tty)))
      DDT_mprint(" < %s",last_in_file);

   if (last_out_file != NULL &&
	  (user_tty == NULL || STRNEQ(last_out_file,user_tty)) &&
	  (default_tty == NULL || STRNEQ(last_out_file,default_tty)))
      DDT_mprint(" > %s",last_out_file);

   DDT_mprint("\n");
};





/************************************************************************/
/*									*/
/*	DDT_model_show_env -- show environment				*/
/*									*/
/************************************************************************/


void
DDT_model_show_env(what)
   String what;
{
   Integer ln;
   String *env,s;
   extern char **environ;
   Character buf[1024];

   if (what != NULL) {
      sprintf(buf,"%s=",what);
      ln = strlen(buf);
    };

   env = environ;
   while (*env != NULL) {
      s = *env++;
      if (what != NULL) {
	 if (strncmp(s,buf,ln) != 0) continue;
	 else s += ln;
       };
      DDT_mprint("%s\n",s);
    };
};





void
DDT_model_set_env(s)
   String s;
{
   Character buf[1024];
   Character buf1[1024];
   String p,q;

   while (isspace(*s)) ++s;
   if (*s == 0 || *s == '=') return;

   strcpy(buf,s);

   p = index(buf,'=');
   if (p == NULL) return;
   q = p;
   --q;
   *p++ = 0;
   while (isspace(*q)) *q-- = 0;

   while (isspace(*p)) ++p;

   sprintf(buf1,"%s=%s",buf,p);
   putenv(buf1);

   DDT_x_set_env(buf,p);
};





/************************************************************************/
/*									*/
/*	DDT_model_signal_name -- return signal number for name		*/
/*									*/
/************************************************************************/


Integer
DDT_model_signal_name(name)
   String name;
{
   register Integer i,j;
   register String s,t;
   Character buf[64];
   Boolean fg;

   if (strncmp(name,"SIG",3) == 0) s = &name[3];
   else s = name;

   fg = TRUE;
   t = buf;
   while (*s != 0) {
      if (!isdigit(*s)) fg = FALSE;
      if (islower(*s)) *t++ = *s++ - 'a' + 'A';
      else *t++ = *s++;
    };
   *t = 0;

   if (fg && buf[0] != 0) {
      i = atol(buf);
      for (j = 0; DDT__signal_table[j].name != NULL; ++j);
      if (i >= j) i = j;
    }
   else {
      for (i = 0; DDT__signal_table[i].name != NULL; ++i) {
	 if (STREQL(buf,DDT__signal_table[i].name)) break;
       };
    };

   if (DDT__signal_table[i].name == NULL) {
      DDT_mprint("Illegal signal name %s\n",name);
      i = -1;
    };

   return i;
};






/************************************************************************/
/*									*/
/*	DDT_model_sigset -- turn signal on/off				*/
/*	DDT_model_sig_status -- display signal status			*/
/*									*/
/************************************************************************/


static struct { int flags; String id; String id1; } sig_status_info[] = {
   { SIGNAL_CATCH,  "Catch", "caught" },
   { SIGNAL_PRINTIGN,  "Print and discard", "printed and discarded" },
   { SIGNAL_IGNORE, "Ignore", "ignored" },
   { SIGNAL_PRINTUSE, "Print and pass", "printed and passed" },
   { 0, 0 }
};


void
DDT_model_sigset(name,fgs)
   String name;
   Integer fgs;
{
   register Integer i,j;

   i = DDT_model_signal_name(name);

   if (i > 0) {
      DDT_x_sigset(i,fgs);

      for (j = 0; sig_status_info[j].flags != fgs; ++j);
      DDT_mprint("Signal %s will be %s\n",DDT__signal_table[i].name,sig_status_info[j].id1);
    };
};






void
DDT_model_sig_status()
{
   register Integer i,j,k,j0;

   j0 = 0;
   for (k = 0; sig_status_info[k].id != NULL; ++k) {
      j = 0;
      for (i = 0; DDT__signal_table[i].name != NULL; ++i) {
	 if (DDT__signal_table[i].flags == sig_status_info[k].flags) {
	    if (j == 0) {
	       if (j0++ != 0) DDT_mprint("\n");
	       DDT_mprint("%s:",sig_status_info[k].id);
	     };
	    if ((++j % 5) == 0) DDT_mprint("\n    ");
	    else DDT_mprint(" ");
	    DDT_mprint(DDT__signal_table[i].name);
	  };
       };
    };

   DDT_mprint("\n");
};





/************************************************************************/
/*									*/
/*	DDT_model_use -- add to use string				*/
/*	DDT_model_show_use -- display current use string		*/
/*									*/
/************************************************************************/


void
DDT_model_use(dir)
   String dir;
{
   Character buf[10240];

   if (dir == NULL) sprintf(buf,".");
   else sprintf(buf,"%s %s",use_dirs,dir);

   if (use_dirs != NULL) SFREE(use_dirs);
   use_dirs = SALLOC(buf);

   DDT_x_use(use_dirs);
};





void
DDT_model_show_use()
{
   DDT_mprint("USE: %s\n",use_dirs);
};





/************************************************************************/
/*									*/
/*	DDT_model_real_name -- return real file name for debugger one	*/
/*									*/
/************************************************************************/


String
DDT_model_real_name(fn)
   String fn;
{
   register Integer i,j,k;

   for (i = 0; i < num_files; ++i) {
      if (STREQL(fn,all_files[i])) break;
    };

   if (i >= num_files) {
      j = strlen(fn);
      for (i = 0; i < num_files; ++i) {
	 k = strlen(all_files[i]);
	 if (j < k && STREQL(&all_files[i][k-j],fn) && all_files[i][k-j-1] == '/') break;
       };
    };

   if (i < num_files) {
      if (real_names[i] == NULL) get_real_filename(i);
      if (real_names[i] != NULL) fn = real_names[i];
      else fn = "*";
    };

   return fn;
};





/************************************************************************/
/*									*/
/*	DDT_model_default_file -- return default file name		*/
/*									*/
/************************************************************************/


String
DDT_model_default_file()
{
   return current_focus.file;
};





/************************************************************************/
/*									*/
/*	DDT_model_set_location -- set focus location			*/
/*	DDT_model_inq_location -- return focus location 		 */
/*									*/
/************************************************************************/


void
DDT_model_set_location(locp)
   LOCATION * locp;
{
   Boolean new;

   new = FALSE;

   if (locp->file != NULL) {
      if (current_focus.file == NULL || STRNEQ(locp->file,current_focus.file)) {
	 if (current_focus.file != NULL) SFREE(current_focus.file);
	 current_focus.file = SALLOC(locp->file);
	 new = TRUE;
       };
    }
   else {
      if (current_focus.file != NULL) {
	 SFREE(current_focus.file);
	 new = TRUE;
       };
      current_focus.file = NULL;
    };

   if (locp->func != NULL) {
      if (current_focus.func == NULL || STRNEQ(current_focus.func,locp->func)) {
	 if (current_focus.func != NULL) SFREE(current_focus.func);
	 current_focus.func = SALLOC(locp->func);
	 new = TRUE;
       };
    }
   else {
      if (current_focus.func != NULL) {
	 SFREE(current_focus.func);
	 new = TRUE;
       };
      current_focus.func = NULL;
    };

   if (locp->line != current_focus.line) {
      current_focus.line = locp->line;
      new = TRUE;
    };

   if (new) {
      DDT_msg_focus(&current_focus);
    };
};





void
DDT_model_inq_location(locp)
   LOCATION * locp;
{
   *locp = current_focus;
};





/************************************************************************/
/*									*/
/*	DDT_model_set_exec_location -- set execution location		*/
/*	DDT_model_inq_exec_location -- return execution location	*/
/*									*/
/************************************************************************/


void
DDT_model_set_exec_location(locp,addr)
   LOCATION * locp;
   Integer addr;
{
   if (current_exec.file != NULL) SFREE(current_exec.file);
   current_exec.file = SALLOC(locp->file);

   if (current_exec.func != NULL) SFREE(current_exec.func);
   current_exec.func = SALLOC(locp->func);

   current_exec.line = locp->line;
   if (current_exec.line < 0)
      current_exec.line = 0;

   if (addr != -1) exec_address = addr;

   DDT_msg_at_line(&current_exec);
};





void
DDT_model_inq_exec_location(locp,addrp)
   LOCATION * locp;
   Integer * addrp;
{
   *locp = current_exec;
   *addrp = exec_address;
};






/************************************************************************/
/*									*/
/*	DDT_valid_string -- check for valid string			*/
/*									*/
/************************************************************************/


Boolean
DDT_valid_string(s)
   String s;
{
   if (s != NULL && *s != 0 && (*s != '*' || s[1] != 0)) return TRUE;

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_print_current_position -- print current position		*/
/*	DDT_print_exec_position -- print execution position		*/
/*	DDT_print_position -- print a location				*/
/*	DDT_position -- dump location into buffer			*/
/*									*/
/************************************************************************/


void
DDT_print_current_position()
{
   DDT_print_position(current_focus.file,current_focus.func,current_focus.line,0);
};





void
DDT_print_exec_position()
{
   DDT_print_position(current_exec.file,current_exec.func,current_exec.line,0);
};





void
DDT_print_position(file,func,line,addr)
   String file;
   String func;
   Integer line;
   Integer addr;
{
   Character buf[1024];

   DDT_position(buf,file,func,line,addr);

   DDT_mprint("%s",buf);
};





void
DDT_position(buf,file,func,line,addr)
   String buf;
   String file;
   String func;
   Integer line;
   Integer addr;
{
   if (addr != 0 && line > 0)
      sprintf(buf," at address 0x%x of line %d",addr,line);
   else if (addr != 0)
      sprintf(buf," at address 0x%x",addr);
   else if (line > 0)
      sprintf(buf," at line %d",line);

   if (DDT_valid_string(func)) {
      if (line > 0 || addr > 0) {
	 strcat(buf," in ");
	 strcat(buf,func);
       }
      else sprintf(buf," in routine %s",func);
    }
   else if (addr == 0 && line <= 0) sprintf(buf," at <unknown>");

   if (DDT_valid_string(file)) {
      strcat(buf," of file ");
      strcat(buf,file);
    };
};





/************************************************************************/
/*									*/
/*	DDT_model_check_file -- check if file is part of system 	*/
/*									*/
/************************************************************************/


Boolean
DDT_model_check_file(nam)
   String nam;
{
   register Integer i;
   String s;

   for (i = 0; i < num_files; ++i) {
      if (real_names[i] == NULL) get_real_filename(i);
      if (real_names[i] != NULL &&
	     MSGfile_compare(nam,real_names[i])) return TRUE;
      if (nam[0] != '/') {
	 s = rindex(real_names[i],'/');
	 if (s != NULL) {
	    ++s;
	    if (STREQL(s,nam)) return TRUE;
	  };
       };
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_model_list_files -- list files matching pattern		*/
/*									*/
/************************************************************************/


void
DDT_model_list_files(pat)
   String pat;
{
   String s;
   Integer i;

   if (pat != NULL && (pat[0] == '"' || pat[0] == '\'')) {
      i = strlen(pat)-1;
      if (pat[i] == pat[0]) pat[i] = 0;
      ++pat;
      if (pat[0] == 0) pat = NULL;
    };

   if (pat != NULL) {
      s = re_comp(pat);
      if (s != NULL) {
	 DDT_mprint("%s\n",s);
	 return;
       };
    };

   for (i = 0; i < num_files; ++i) {
      if (real_names[i] == NULL) get_real_filename(i);
      s = (real_names[i] == NULL ? all_files[i] : real_names[i]);
      if (s == NULL) continue;
      if (pat == NULL ||
	     (all_files[i] != NULL && re_exec(all_files[i])) ||
	     (real_names[i] != NULL && re_exec(real_names[i]))) {
	 DDT_mprint("%s\n",s);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_model_list_functions -- list functions matching pattern	*/
/*									*/
/************************************************************************/


void
DDT_model_list_functions(pat)
   String pat;
{
   String s;
   String fct[MAX_FCTS];
   Integer i,ct;

   ct = DDT_symbol_functions(MAX_FCTS,fct,FALSE);

   if (pat != NULL && (pat[0] == '"' || pat[0] == '\'')) {
      i = strlen(pat)-1;
      if (pat[i] == pat[0]) pat[i] = 0;
      ++pat;
      if (pat[0] == 0) pat = NULL;
    };

   if (pat != NULL) {
      s = re_comp(pat);
      if (s != NULL) {
	 DDT_mprint("%s\n",s);
	 return;
       };
    };

   for (i = 0; i < ct; ++i) {
      if (pat == NULL || re_exec(fct[i])) {
	 DDT_mprint("%s\n",fct[i]);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_model_list_variables -- list variables matching pattern	*/
/*									*/
/************************************************************************/


void
DDT_model_list_variables(pat)
   String pat;
{
   String s;
   String var[MAX_FCTS];
   Integer i,ct;

   ct = DDT_symbol_variables(MAX_FCTS,var);

   if (pat != NULL && (pat[0] == '"' || pat[0] == '\'')) {
      i = strlen(pat)-1;
      if (pat[i] == pat[0]) pat[i] = 0;
      ++pat;
      if (pat[0] == 0) pat = NULL;
    };

   if (pat != NULL) {
      s = re_comp(pat);
      if (s != NULL) {
	 DDT_mprint("%s\n",s);
	 return;
       };
    };

   for (i = 0; i < ct; ++i) {
      if (pat == NULL || re_exec(var[i])) {
	 DDT_mprint("%s\n",var[i]);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_model_list_types -- list types matching pattern		*/
/*									*/
/************************************************************************/


void
DDT_model_list_types(pat)
   String pat;
{
   String s;
   String typ[MAX_FCTS];
   Integer i,ct;

   ct = DDT_symbol_types(MAX_FCTS,typ);

   if (pat != NULL && (pat[0] == '"' || pat[0] == '\'')) {
      i = strlen(pat)-1;
      if (pat[i] == pat[0]) pat[i] = 0;
      ++pat;
      if (pat[0] == 0) pat = NULL;
    };

   if (pat != NULL) {
      s = re_comp(pat);
      if (s != NULL) {
	 DDT_mprint("%s\n",s);
	 return;
       };
    };

   for (i = 0; i < ct; ++i) {
      if (pat == NULL || re_exec(typ[i])) {
	 DDT_mprint("%s\n",typ[i]);
       };
    };
};





/************************************************************************/
/*									*/
/*	get_real_filename -- get the real filename for a given name	*/
/*									*/
/************************************************************************/


static void
get_real_filename(idx)
   Integer idx;
{
   register String s,t;
   String nam;
   Character buf[256];

   nam = all_files[idx];

   if (nam[0] == '/') strcpy(buf,nam);
   else {
      t = buf;
      for (s = use_dirs; ; ++s) {
	 if (*s != ' ' && *s != 0) *t++ = *s;
	 else {
	    *t++ = '/';
	    *t = 0;
	    strcat(buf,nam);
	    if (access(buf,0) >= 0) break;
	    t = buf;
	    if (*s == 0) {
	       buf[0] = 0;
	       break;
	     };
	  };
       };
    };

   if (buf[0] != 0) {
      real_names[idx] = SALLOC(buf);
    };
};




/* end of ddtmodel.c */
