/************************************************************************/
/*									*/
/*		ddtexpr.c						*/
/*									*/
/*	System expression routines					*/
/*									*/
/*	This file is included in ddtexprsyn.[yc]			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	String		curptr;
static	String		result;
static	Boolean 	doing_parse;





/************************************************************************/
/*									*/
/*	Parse tables							*/
/*									*/
/************************************************************************/


#define SINGLE_CHARS	"|^&=+-*/%!~()[].,<>"

typedef struct _MULT_CHAR {
   String item;
   Integer result;
} MULT_CHAR;


static MULT_CHAR	mult_chars[] = {
   { "==",      LX_EQL },
   { "<>",      LX_NEQ },
   { "!=",      LX_NEQ },
   { "<=",      LX_LEQ },
   { ">=",      LX_GEQ },
   { ">>",      LX_RSH },
   { "<<",      LX_LSH },
   { "..",      LX_DOTDOT },
   { "||",      LX_OR },
   { "&&",      LX_AND },
   { "->",      LX_PTS },
   { "^.",      LX_PTS },
   { NULL, 0 },
};


static	MULT_CHAR	id_names[] = {
   { "or",      LX_OR },
   { "and",     LX_AND },
   { "mod",     LX_MOD },
   { "div",     LX_DIV },
   { "sizeof",  LX_SIZEOF },
   { "struct",  LX_STRUCT },
   { "union",   LX_UNION },
   { "enum",    LX_ENUM },
   { "unsigned", LX_UNSIGNED },
   { NULL, 0},
};






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 		scan_string();
static	Integer 		scan_id();
static	void			set_result();
static	String			expr_build();
static	String			expr_map_var();
static	String			expr_map_field();
static	String			expr_map_member_fct();




/************************************************************************/
/*									*/
/*	DDT_expr_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_expr_init()
{
   doing_parse = FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_expr_parse -- handle parsing				*/
/*									*/
/************************************************************************/



String
DDT_expr_parse(str,buf)
   String str;
   String buf;
{
   Integer ln;

   if (doing_parse) return str;
   doing_parse = TRUE;

   ln = strlen(str);

   curptr = (String) alloca(ln+10);
   strcpy(curptr,str);
   result = NULL;

   if (DDT_expr_yyparse() == 0) {
      if (result != NULL) {
	 strcpy(buf,result);
	 str = buf;
	 SFREE(result);
       };
    }

   doing_parse = FALSE;

   return str;
};





/************************************************************************/
/*									*/
/*	DDT_expr_yyerror -- handle errors				*/
/*									*/
/************************************************************************/


void
DDT_expr_yyerror(msg)
   String msg;
{
   return;
};






/************************************************************************/
/*									*/
/*	DDT_expr_yylex -- return next token				*/
/*									*/
/************************************************************************/


Integer
DDT_expr_yylex()
{
   Integer i,j;

   DDT_expr_yylval.strval = 0;

   while (isspace(*curptr)) ++curptr;

   if (*curptr == 0) return -1;

   if (index(SINGLE_CHARS,*curptr) != NULL) {
      for (i = 0; mult_chars[i].item != NULL; ++i) {
	 j = strlen(mult_chars[i].item);
	 if (strncmp(curptr,mult_chars[i].item,j) == 0) {
	    curptr += j;
	    return mult_chars[i].result;
	  };
       };
      return *curptr++;
    };

   if (*curptr == '"' || *curptr == '\'') {
      i = scan_string();
    }
   else {
      i = scan_id();
    };

   return i;
};





/************************************************************************/
/*									*/
/*	scan_string -- string scanning routine				*/
/*									*/
/************************************************************************/


static Integer
scan_string()
{
   String buf;
   Character sbuf[1024];
   Character term;
   Integer ln;

   buf = sbuf;

   term = *curptr;
   *buf++ = *curptr++;
   ln = 0;

   while (*curptr != 0 && *curptr != term) {
      if (*curptr == '\\') {
	 *buf++ = *curptr++;
	 if (isdigit(*curptr) && isdigit(curptr[1])) *buf++ = *curptr++;
	 if (isdigit(*curptr) && isdigit(curptr[1])) *buf++ = *curptr++;
       };
      *buf++ = *curptr++;
      ++ln;
    };

   if (*curptr == 0) return LX_ERROR;

   *buf++ = term;
   *buf = 0;
   ++curptr;

   if (term == '\'' && ln > 2 && DDT__cplusplus != 0) {
      --buf;
      *buf = 0;
      DDT_expr_yylval.strval = SALLOC(&sbuf[1]);
      return LX_ID;
    };

   DDT_expr_yylval.strval = SALLOC(sbuf);

   return LX_STRING;
};






/************************************************************************/
/*									*/
/*	scan_id -- scan identifier or numeric constant			*/
/*									*/
/************************************************************************/


static Integer
scan_id()
{
   String s;
   Integer i,j;
   Character buf[1024],xbuf[1024],ybuf[1024];
   double fval;

   s = buf;
   j = 0;
   for ( ; ; ) {
      if (*curptr == 0 || isspace(*curptr)) break;
      else if (*curptr == '.' && j == 0) ;
      else if (*curptr == '-' && j == 2) ;
      else if (index(SINGLE_CHARS,*curptr) != NULL) break;

      if (isdigit(*curptr)) ;
      else if (j == 0 && *curptr == '.') j = 1;
      else if (j <= 1 && (*curptr == 'e' || *curptr == 'E')) j = 2;
      else if (*curptr == '-') j = 3;
      else j = 4;

      *s++ = *curptr++;
    };

   *s = 0;

   if (j == 0) {
      DDT_expr_yylval.strval = SALLOC(buf);
      return LX_INT;
    }
   else if (j < 4) {
      sprintf(ybuf,"%s HELLO THERE",buf);
      i = sscanf(ybuf,"%lf HELLO %s",&fval,xbuf);
      if (i != 2 || STRNEQ(xbuf,"THERE")) j = 4;
      else {
	 DDT_expr_yylval.strval = SALLOC(buf);
	 return LX_FLT;
       };
    };

   if (buf[0] == '0') {
      DDT_expr_yylval.strval = SALLOC(buf);
      return LX_INT;
    };

   for (i = 0; id_names[i].item != NULL; ++i) {
      if (STREQL(buf,id_names[i].item)) {
	 return id_names[i].result;
       };
    };

   i = LX_ID;
   s = buf;
   if (DDT_symbol_inq_type_name(s)) i = LX_TYPEID;

   DDT_expr_yylval.strval = SALLOC(s);

   return i;
};





/************************************************************************/
/*									*/
/*	set_result -- store result					*/
/*									*/
/************************************************************************/


static void
set_result(s)
   String s;
{
   result = s;
};





/************************************************************************/
/*									*/
/*	expr_build -- construct expression				*/
/*									*/
/************************************************************************/


static String
expr_build(s1,f1,s2,f2,s3,f3,s4,f4)
   String s1,s2,s3,s4;
   Boolean f1,f2,f3,f4;
{
   Character buf[10240];

   if (s1 != NULL) {
      strcpy(buf,s1);
      if (f1) SFREE(s1);
    }
   else buf[0] = 0;

   if (s2 != NULL) {
      strcat(buf," ");
      strcat(buf,s2);
      if (f2) SFREE(s2);
    };

   if (s3 != NULL) {
      strcat(buf," ");
      strcat(buf,s3);
      if (f3) SFREE(s3);
    };
   if (s4 != NULL) {
      strcat(buf," ");
      strcat(buf,s4);
      if (f4) SFREE(s4);
    };

   return SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	expr_map_var -- handle mapping					*/
/*									*/
/************************************************************************/


static String
expr_map_var(v)
   String v;
{
   Character xbuf[1024];
   String s;

   s = DDT_map_fix_variable(v,xbuf);

   if (s == v) {
      s = DDT_map_fix_function(s,xbuf);
    };

   if (s != v) {
      SFREE(v);
      v = SALLOC(s);
    };

   return v;
};





/************************************************************************/
/*									*/
/*	expr_map_field -- handle field mapping				*/
/*									*/
/************************************************************************/


static String
expr_map_field(name,lhs)
   String name;
   String lhs;
{
   Character xbuf[1024];
   String s;

   s = DDT_map_fix_field(name,lhs,xbuf);

   if (s != name) {
      SFREE(name);
      name = SALLOC(s);
    };

   return name;
};





/************************************************************************/
/*									*/
/*	expr_map_member_function -- handle member functions		*/
/*									*/
/************************************************************************/


static String
expr_map_member_fct(lhs,name,args,ptrfg)
   String lhs;
   String name;
   String args;
   Boolean ptrfg;
{
   String th,fnm;
   Character buf[10240],thbuf[10240];

   if (ptrfg) {
      sprintf(thbuf,"*(%s)",lhs);
      th = thbuf;
    }
   else th = lhs;

   fnm = DDT_map_member_function(name,th,args,buf);

   if (ptrfg) {
      if (args != NULL) sprintf(thbuf,"%s(%s,%s)",fnm,lhs,args);
      else sprintf(thbuf,"%s(%s)",fnm,lhs);
    }
   else {
      if (args != NULL) sprintf(thbuf,"%s(&(%s),%s)",fnm,lhs,args);
      else sprintf(thbuf,"%s(&(%s))",fnm,lhs);
    };

   SFREE(lhs);
   SFREE(name);
   if (args != NULL) SFREE(args);

   return SALLOC(thbuf);
};





/* end of ddtexpr.c */
