/************************************************************************/
/*									*/
/*		ddtevent.c						*/
/*									*/
/*	Run time and debugger event processing routines 		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_EVENTS	512




/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _EVENT * 	EVENT;

typedef struct _EVENT {
   Integer id;
   Integer intern_id;
   String file;
   String func;
   Integer line;
   Integer newline;
   EVENT_ACTION action;
   String expr;
   String cond;
   Integer addr;
   Boolean active;
   String event_info;
   Boolean intr;
   Boolean extr;
   Boolean disabled;
   String text;
} EVENT_INFO;







/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_events;
static	Integer 	event_id;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	valid_event();
static	Boolean 	valid_step_event();
static	void		handle_trace();
static	void		handle_call_trace();
static	void		handle_break();
static	void		handle_event();
static	void		handle_step();
static	void		process_monitor();
static	void		get_event_data();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


typedef struct _EVENT_TABLE {
   String name;
   EVENT_ACTION action;
} EVENT_TABLE;


static EVENT_TABLE event_table[] = {
   { "TRACE",   EVENT_TRACE     },
   { "BREAK",   EVENT_BREAK     },
   { "TBREAK",  EVENT_TBREAK     },
   { "WATCH",   EVENT_WATCH     },
   { "CALL",    EVENT_CALL      },
   { "EVENT",   EVENT_EVENT     },
   { "TRIGGER", EVENT_TRIGGER   },
   { "STEPSTEP",EVENT_STEPSTEP  },
   { "NEXTNEXT",EVENT_NEXTNEXT  },
   { "MONITOR", EVENT_MONITOR   },
   { "UPDATE",  EVENT_UPDATE    },
   { "WHEN",    EVENT_WHEN      },
   { "DISPLAY", EVENT_DISPLAY   },
   { "STOPEXIT",EVENT_STOPEXIT  },

   { NULL, EVENT_NONE }
};



/************************************************************************/
/*									*/
/*	DDT_event_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
DDT_event_init()
{
   all_events = NULL;
   event_id = 1;
};





/************************************************************************/
/*									*/
/*	DDT_event_clear -- clear model on reinitialization		*/
/*									*/
/************************************************************************/


void
DDT_event_clear(new)
   Boolean new;
{
   EVENT use[MAX_EVENTS];
   EVENT ev;
   Integer i,ct;
   Sequence l;

   ct = 0;
   forin (ev,EVENT,l,all_events) {
      ev->intern_id = 0;
      use[ct++] = ev;
    };

   if (new) {
      for (i = 0; i < ct; ++i) {
	 DDT_event_remove(use[i],TRUE);
       };
    }
   else {
      for (i = 0; i < ct; ++i) {
	 ev = use[i];
	 DDT_x_add_event(ev->action,ev->file,ev->func,ev->newline,ev->expr,
			    ev->cond,ev->addr,ev->event_info,
			    ev->intr,ev->extr);
       };
      for (i = 0; i < ct; ++i) {
	 if (use[i]->intern_id == 0) DDT_event_remove(use[i],TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_event_add -- add event to event table			*/
/*									*/
/************************************************************************/


Boolean
DDT_event_add(act,name,file,func,line,expr,cond,addr,did,intr,extr)
   EVENT_ACTION act;
   String name;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   Integer did;
   Boolean intr;
   Boolean extr;
{
   EVENT ev;
   Boolean found;
   Character buf[1024],bufa[1024];
   Sequence l;

   if (!DDT_valid_string(file)) file = NULL;
   if (!DDT_valid_string(func)) func = NULL;
   if (addr > 0 && line > 0 && line == addr) line = 0;

   found = FALSE;
   forin (ev,EVENT,l,all_events) {
      if (valid_event(ev,file,func,line,expr,cond,addr,act,0)) {
	 if ((ev->event_info != NULL && name != NULL && STREQL(name,ev->event_info)) ||
		(ev->event_info == NULL && name == NULL)) {
	    if (ev->file == NULL && file != NULL) continue;
	    if (ev->func == NULL && func != NULL) continue;
	    if (ev->line == 0 && line > 0) continue;
	    if (ev->expr == NULL && expr != NULL) continue;
	    if (ev->cond == NULL && cond != NULL) continue;
	    if (ev->addr == 0 && addr != 0) continue;
	    if (ev->intr == intr && ev->extr == extr) {
	       found = TRUE;
	       break;
	     };
	  };
       };
    };

   if (found) {
      if (ev->intern_id == 0) {
	 ev->intern_id = did;
	 return TRUE;
       };
      return FALSE;
    };

   ev = PALLOC(EVENT_INFO);

   ev->id = event_id++;
   ev->intern_id = did;
   ev->file = SALLOC(file);
   ev->func = SALLOC(func);
   ev->line = line;
   ev->newline = line;
   ev->action = act;
   ev->expr = (DDT_valid_string(expr) ? SALLOC(expr) : NULL);
   ev->cond = (DDT_valid_string(cond) ? SALLOC(cond) : NULL);
   ev->addr = addr;
   ev->active = FALSE;
   ev->event_info = (name == NULL ? NULL : SALLOC(name));
   ev->intr = intr;
   ev->extr = extr;
   ev->disabled = FALSE;

   sprintf(buf,"[%d] %s",ev->id,DDT_event_name(ev->action));
   if (ev->addr != 0) strcat(buf,"I");
   if (ev->event_info != NULL) {
      strcat(buf," (");
      strcat(buf,ev->event_info);
      strcat(buf,")");
    };
   if (DDT_valid_string(ev->expr)) {
      strcat(buf," ");
      strcat(buf,ev->expr);
    };
   if (ev->func != NULL || ev->line > 0 || ev->addr > 0) {
      DDT_position(bufa,ev->file,ev->func,ev->line,ev->addr);
      strcat(buf,bufa);
    };
   if (DDT_valid_string(ev->cond)) {
      strcat(buf," if ");
      strcat(buf,ev->cond);
    };
   ev->text = SALLOC(buf);

   all_events = APPEND(ev,all_events);

   DDT_event_show(ev,TRUE);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	DDT_for_each_event -- call routine for each matching event	*/
/*									*/
/************************************************************************/


void
DDT_for_each_event(file,func,line,expr,cond,addr,act,id,rtn,arg)
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   EVENT_ACTION act;
   Integer id;
   Function_Ptr rtn;
   Universal arg;
{
   EVENT use[MAX_EVENTS];
   EVENT ev;
   Integer i,ct;
   Sequence l;

   ct = 0;
   forin (ev,EVENT,l,all_events) {
      if (valid_event(ev,file,func,line,expr,cond,addr,act,id)) {
	 use[ct++] = ev;
       };
    };

   for (i = 0; i < ct; ++i) {
      (*rtn)(use[i],arg);
    };
};





/************************************************************************/
/*									*/
/*	DDT_event_remove -- remove an event				*/
/*									*/
/************************************************************************/


void
DDT_event_remove(ev,show)
   EVENT ev;
   Boolean show;
{
   Boolean fg;

   all_events = REMOB(ev,all_events);

   fg = TRUE;
   if (ev->intern_id != 0) fg = DDT_x_remove_event(ev->intern_id);

   if (fg && show) DDT_mprint("Delete [%d]\n",ev->id);

   if (fg) DDT_msg_event(FALSE,ev->id,ev->action,ev->file,ev->line,ev->text);

   if (ev->file != NULL) SFREE(ev->file);
   if (ev->func != NULL) SFREE(ev->func);
   if (ev->expr != NULL) SFREE(ev->expr);
   if (ev->cond != NULL) SFREE(ev->cond);

   free(ev);
};





/************************************************************************/
/*									*/
/*	DDT_event_change -- change line number for an event		*/
/*									*/
/************************************************************************/


void
DDT_event_change(ev,line)
   EVENT ev;
   Integer line;
{
   if (ev->line != 0) ev->newline = line;
};





/************************************************************************/
/*									*/
/*	DDT_event_disable -- disable an event				*/
/*									*/
/************************************************************************/


void
DDT_event_disable(ev,show)
   EVENT ev;
   Boolean show;
{
   Boolean fg;

   fg = TRUE;
   if (ev->intern_id != 0) {
      fg = DDT_x_remove_event(ev->intern_id);
      ev->intern_id = 0;
    };

   if (fg && show) DDT_mprint("Disable [%d]\n",ev->id);

/* if (fg) DDT_msg_event(FALSE,ev->id,ev->action,ev->file,ev->line,ev->text);	*/

   ev->disabled = TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_event_show -- display information about an event		*/
/*									*/
/************************************************************************/


void
DDT_event_show(ev,fg)
   EVENT ev;
   Boolean fg;
{
   DDT_msg_event(TRUE,ev->id,ev->action,ev->file,ev->line,ev->text);

   if (fg) DDT_mprint("%s\n",ev->text);
};





/************************************************************************/
/*									*/
/*	DDT_event_handle -- handle trace event occurring		*/
/*									*/
/************************************************************************/


void
DDT_event_handle(file,func,line,addr,expr,exval,id,trfg,clfg)
   String file;
   String func;
   Integer line;
   Integer addr;
   String expr;
   String exval;
   Integer id;
   Boolean trfg;
   Integer clfg;
{
   Sequence l;
   EVENT ev;
   Integer ct;

   ct = 0;
   forin (ev,EVENT,l,all_events) {
      if (clfg != 0) {
	 if (valid_event(ev,file,func,line,NULL,"?",addr,EVENT_CALL,id)) {
	    handle_call_trace(ev,func,expr,exval,(clfg == 1));
	    ++ct;
	  }
       }
      else if (trfg) {
	 if (ev->action == EVENT_TRACE) {
	    if (valid_event(ev,file,func,line,expr,"?",addr,EVENT_NONE,id)) {
	       handle_trace(ev,exval,line);
	       ++ct;
	     }
	  }
	 else if (ev->action == EVENT_WATCH) {
	    if (valid_event(ev,file,func,line,expr,"?",addr,EVENT_WATCH,id)) {
	       handle_trace(ev,exval,line);
	       ++ct;
	     }
	  };
       }
      else {
	 DDT_state_stop_expected();
	 switch (ev->action) {
	    case EVENT_BREAK :
	    case EVENT_TBREAK :
	    case EVENT_STOPEXIT :
	       if (valid_event(ev,file,func,line,expr,"?",addr,EVENT_NONE,id)) {
		  DDT_state_stop_action(STOP_STOP);
		  handle_break(ev,exval,line);
		  ev->active = TRUE;
		  ++ct;
		};
	       break;
	    case EVENT_EVENT :
	    case EVENT_TRIGGER :
	    case EVENT_UPDATE :
	    case EVENT_WHEN :
	       if (valid_event(ev,file,func,line,NULL,"?",addr,EVENT_NONE,id)) {
		  DDT_state_stop_action(STOP_CONTINUE);
		  handle_event(ev,line);
		  ++ct;
		};
	       break;
	    case EVENT_STEPSTEP :
	       handle_step(ev);
	       ++ct;
	       break;
	    case EVENT_NEXTNEXT :
	       if (!DDT_valid_string(func) || ev->func == NULL || STREQL(func,ev->func)) {
		  handle_step(ev);
		  ++ct;
		};
	       break;
	    case EVENT_MONITOR :
	       if (valid_event(ev,file,func,line,NULL,"?",addr,EVENT_NONE,id)) {
		  DDT_state_stop_action(STOP_CONTINUE);
		  ++ct;
		  ev->active = TRUE;
		};
	       break;
	    default :
	       break;
	  };
       };
    };

   if (ct == 0 && !trfg && !clfg) DDT_state_stop_action(STOP_STOP);
};





/************************************************************************/
/*									*/
/*	DDT_event_name -- return external name for event type		*/
/*	DDT_event_type -- return event type for external name		*/
/*									*/
/************************************************************************/


String
DDT_event_name(act)
   EVENT_ACTION act;
{
   Integer i;

   for (i = 0; event_table[i].name != NULL; ++i) {
      if (act == event_table[i].action) break;
    };

   return event_table[i].name;
};





EVENT_ACTION
DDT_event_type(typ)
   String typ;
{
   Integer i;

   for (i = 0; event_table[i].name != NULL; ++i) {
      if (STREQL(typ,event_table[i].name)) break;
    };

   return event_table[i].action;
};





/************************************************************************/
/*									*/
/*	DDT_event_process -- process active events			*/
/*									*/
/************************************************************************/


Boolean
DDT_event_process()
{
   Integer ct;
   LOCATION loc;
   EVENT ev;
   Sequence l;
   Character buf[40960];
   EVENT delev[16];
   Integer i,delct,addr;

   DDT_model_inq_exec_location(&loc,&addr);
   ct = 0;
   delct = 0;

   forin (ev,EVENT,l,all_events) {
      if (ev->active) {
	 ++ct;
	 DDT_state_cmd_event(TRUE);
	 ev->active = FALSE;
	 switch (ev->action) {
	    case EVENT_TRIGGER :
	       get_event_data(ev,buf);
	       DDT_msg_ie(TRUE,loc.file,loc.line,ev->event_info,buf);
	       break;
	    case EVENT_UPDATE :
	       DDT_event_stop(&loc);
	       DDT_msg_update(loc.file,loc.line);
	       break;
	    case EVENT_WHEN :
	       DDT_msg_when(loc.file,loc.line,ev->expr);
	       break;
	    case EVENT_EVENT :
	       get_event_data(ev,buf);
	       DDT_msg_ie(FALSE,loc.file,loc.line,ev->event_info,buf);
	       break;
	    case EVENT_MONITOR :
	       process_monitor(ev,&loc);
	       break;
	    case EVENT_STEPSTEP :
	       if (valid_step_event(ev)) DDT_state_stop_action(STOP_STEP);
	       else DDT_state_stop_action(STOP_CONTINUE);
	       break;
	    case EVENT_NEXTNEXT :
	       if (valid_step_event(ev)) DDT_state_stop_action(STOP_NEXT);
	       else DDT_state_stop_action(STOP_CONTINUE);
	       break;
	    case EVENT_TBREAK :
	       delev[delct++] = ev;
	       break;
	    default :
	       break;
	  };
	 DDT_state_cmd_event(FALSE);
       };
    };

   for (i = 0; i < delct; ++i) {
      DDT_event_remove(delev[i],FALSE);
    };

   return (ct > 0);
};





/************************************************************************/
/*									*/
/*	DDT_event_stop -- handle events when program stops		*/
/*									*/
/************************************************************************/


void
DDT_event_stop(loc)
   LOCATION * loc;
{
   Sequence l;
   EVENT ev;
   Character buf[10240];
   Boolean fg;

   forin (ev,EVENT,l,all_events) {
      if (ev->action == EVENT_DISPLAY) {
	 if (ev->expr != NULL) {
	    fg = DDT_x_eval(ev->expr,buf,10240);
	    if (!fg) continue;
	    if (ev->extr) {
	       DDT_mprint("Display [%d]",ev->id);
	       DDT_mprint(": %s = %s",ev->expr,buf);
	       DDT_mprint("\n");
	     };
	    if (ev->intr) {
	       DDT_msg_value(loc->file,loc->line,ev->expr,buf);
	     };
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	valid_event -- check if event is chosen 			*/
/*									*/
/************************************************************************/


static Boolean
valid_event(ev,file,func,line,expr,cond,addr,act,id)
   EVENT ev;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   EVENT_ACTION act;
   Integer id;
{
   Boolean fg;

   if (act != EVENT_NONE && act != ev->action) fg = FALSE;
   else if (id > 0 && ev->id > 0 && id == ev->id) fg = TRUE;
   else if (DDT_valid_string(file) && ev->file != NULL &&
	  !MSGfile_compare(file,ev->file)) fg = FALSE;
   else if (DDT_valid_string(func) && ev->func != NULL &&
	       STRNEQ(func,ev->func)) fg = FALSE;
   else if (line > 0 && ev->line > 0 && line != ev->line) fg = FALSE;
   else if (DDT_valid_string(expr) && STRNEQ(expr,"?") &&
	       (ev->expr == NULL || STRNEQ(expr,ev->expr))) fg = FALSE;
   else if (DDT_valid_string(cond) && STRNEQ(cond,"?") &&
	       (ev->cond == NULL || STRNEQ(cond,ev->cond))) fg = FALSE;
   else if (addr != 0 && ev->addr != 0 && addr != ev->addr) fg = FALSE;
   else if (addr != 0 && line == 0 && ev->line > 0) fg = FALSE;
   else if (id > 0 && id != ev->id) fg = FALSE;
   else fg = TRUE;

   return fg;
};





/************************************************************************/
/*									*/
/*	valid_step_event -- check if STEPSTEP/NEXTNEXT event is chosen	*/
/*									*/
/************************************************************************/


static Boolean
valid_step_event(ev)
   EVENT ev;
{
   Boolean fg;
   Boolean nfg;

   nfg = (ev->action == EVENT_STEPSTEP ? TRUE : FALSE);

   if (ev->func != NULL && !DDT_trace_check_if_in(ev->func,nfg)) fg = FALSE;
   else fg = TRUE;

   return fg;
};





/************************************************************************/
/*									*/
/*	handle_trace -- handle a trace event occurring			*/
/*									*/
/************************************************************************/


static void
handle_trace(ev,val,line)
   EVENT ev;
   String val;
   Integer line;
{
   if (ev->extr) {
      DDT_mprint("Trace [%d]",ev->id);
      if (ev->expr != NULL) DDT_mprint(": %s = %s",ev->expr,val);
      DDT_print_position(ev->file,ev->func,line,ev->addr);
      DDT_mprint("\n");
    };

   if (ev->intr) {
      if (ev->expr != NULL) {
	 DDT_msg_value(ev->file,line,ev->expr,val);
       };
    };
};






/************************************************************************/
/*									*/
/*	handle_call_trace -- handle a call event occurring		*/
/*									*/
/************************************************************************/


static void
handle_call_trace(ev,func,args,fret,clfg)
   EVENT ev;
   String func;
   String args;
   String fret;
   Boolean clfg;
{
   String buf;

   if (ev->extr) {
      if (args != NULL && DDT__cplusplus != 0) {
	 buf = (String) alloca(strlen(func)+strlen(args)+200);
	 args = DDT_map_fix_outexpr(args,TRUE,buf);
       };
      if (clfg && args == NULL) args = "";
      if (clfg && fret != NULL) DDT_mprint("Enter %s(%s) from %s\n",func,args,fret);
      else if (clfg) DDT_mprint("Enter %s(%s)\n",func,args);
      else if (fret == NULL || *fret == 0) DDT_mprint("Exit %s\n",func);
      else DDT_mprint("Exit %s returning %s\n",func,fret);
    };

   if (ev->intr) {
      if (clfg) DDT_msg_call(ev->file,func,ev->line,args,fret);
      else DDT_msg_return(ev->file,func,ev->line,fret);
    };
};






/************************************************************************/
/*									*/
/*	handle_break -- handle a break event occurring			*/
/*									*/
/************************************************************************/


static void
handle_break(ev,val,line)
   EVENT ev;
   String val;
   Integer line;
{
   if (ev->extr) {
      if (ev->action == EVENT_TBREAK) DDT_mprint("Breakpoint [%d]",ev->id);
      else DDT_mprint("Stoppoint [%d]",ev->id);
      if (ev->expr != NULL) DDT_mprint(": %s = %s",ev->expr,val);
      DDT_mprint(": ");
    };

   if (ev->intr) {
      if (ev->expr != NULL) {
	 DDT_msg_value(ev->file,line,ev->expr,val);
       };
    };
};






/************************************************************************/
/*									*/
/*	handle_event -- handle a event event occurring			*/
/*									*/
/************************************************************************/


static void
handle_event(ev,line)
   EVENT ev;
   Integer line;
{
   if (ev->extr) {
      if (ev->action == EVENT_UPDATE) {
	 DDT_mprint("Trace [%d]",ev->id);
       }
      else if (ev->action == EVENT_WHEN) {
	 DDT_mprint("When [%d]",ev->id);
       }
      else {
	 DDT_mprint("[%d] Event %s",ev->id,ev->event_info);
       };
      DDT_print_position(ev->file,ev->func,line,ev->addr);
      DDT_mprint("\n");
    };

   if (ev->intr) {
      ev->active = TRUE;
    };
};






/************************************************************************/
/*									*/
/*	handle_step -- handle a stepstep/nextnext event occurring	*/
/*									*/
/************************************************************************/


static void
handle_step(ev)
   EVENT ev;
{
   LOCATION loc;
   Integer addr;

   if (ev->extr) {
      DDT_model_inq_exec_location(&loc,&addr);
      DDT_mprint("Trace [%d]",ev->id);
      DDT_print_position(loc.file,loc.func,loc.line,addr);
      DDT_mprint("\n");
    };

   ev->active = TRUE;
};






/************************************************************************/
/*									*/
/*	process_monitor -- process a monitor event			*/
/*									*/
/************************************************************************/


static void
process_monitor(ev,locp)
   EVENT ev;
   LOCATION * locp;
{
   Character buf[10240];
   Boolean fg;

   if (ev->expr == NULL) return;

   fg = DDT_x_eval(ev->expr,buf,10240);
   if (!fg) return;

   if (ev->extr) {
      DDT_mprint("[%d] Monitor %s = %s\n",ev->id,ev->expr,buf);
    };

   if (ev->intr) {
      DDT_msg_value(locp->file,locp->line,ev->expr,buf);
    };
};




/************************************************************************/
/*									*/
/*	get_event_data -- get data for event				*/
/*									*/
/************************************************************************/


static void
get_event_data(ev,buf)
   EVENT ev;
   String buf;
{
   Character expr[1024],line[1024];
   String s,t,u;
   Integer ch;

   if (ev->expr == NULL) {
      strcpy(buf,"*");
      return;
    };

   strcpy(expr,ev->expr);
   s = expr;

   buf[0] = 0;
   while (s != NULL && *s != 0) {
      t = index(s,'@');
      if (t != NULL) *t++ = 0;
      if (buf[0] != 0) strcat(buf," ");

      if (s[0] == '"' || s[0] == '\'' || isdigit(s[0])) strcpy(line,s);
      else if (!DDT_x_eval(s,line,1024)) strcpy(line,"*");

      if (STREQL(line,"(nil)")) strcpy(line,"0");

      u = index(line,'"');
      if (u != NULL) ch = '"';
      else {
	 u = index(line,'\'');
	 if (u != NULL) ch = '\'';
       };
      if (u != NULL) {
	 ++u;
	 line[0] = LIT_STRING;
	 if (u != &line[1]) strcpy(&line[1],u);
	 u = rindex(line,ch);
	 if (ch == '\'') {
	    while (u[-1] == ' ') --u;
	  };
	 *u++ = LIT_STRING;
	 *u = 0;
       };

      strcat(buf,line);

      s = t;
    };

   if (buf[0] == 0) strcpy(buf,"*");
};





/* end of ddtevent.c */
