/************************************************************************/
/*									*/
/*		ddtdbx.c						*/
/*									*/
/*	DBX-specific routines for the ddt debugger interface		*/
/*	(Currently implemented for SUN3/SUN4 systems)			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "ddt_local.h"
#include <a.out.h>
#include <stab.h>
#include <sys/types.h>
#include <sys/stat.h>



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DBX_PATH	"/usr/ucb/dbx"
#define GDB_PATH	"/cs/bin/gnu/gdb"


#define LINEADDR_PAT	"%1s:% "

#ifndef mips
#define EOF_PATTERN	"\"%s\" has only %1d lines"
#define WHERE_FILE	", line %1d in \"%2s\""
#define WHERE_ADDR	" at 0x%1x"
#define EVENT_OK_CHAR	'('
#define NAME_SEP_CHAR	'`'
#else
#define EOF_PATTERN	"line %1d beyond end of file"
#define WHERE_FILE	" [\"%2s\":%1d, 0x%x]"
#define WHERE_ADDR	" [0x%1x]"
#define EVENT_OK_CHAR	'['
#define NAME_SEP_CHAR	'.'
#endif

#define EVENT_ARGS	9
#define EV_IDX_FILE		0
#define EV_IDX_FUNCTION 	1
#define EV_IDX_LINE		2
#define EV_IDX_EXPRESSION	3
#define EV_IDX_CONDITION	4
#define EV_IDX_ADDRESS		5
#define EV_IDX_ID		6
#define EV_IDX_NEW		7
#define EV_IDX_SIGNAL		8

#define EP_ARGS 	6
#define EP_IDX_ID		0
#define EP_IDX_FILE		1
#define EP_IDX_LINE		2
#define EP_IDX_FUNC		3
#define EP_IDX_EXPR		4





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	event_pending;
static	Universal	pending_args[EVENT_ARGS+1];
static	PMAT_PAT	lineaddr_pattern;
static	PMAT_PAT	eof_pattern;
static	PMAT_PAT	where_file_pat;
static	PMAT_PAT	where_addr_pat;
static	Integer 	exit_status;
static	String		exit_signal;






/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#ifndef mips

SIGNAL_TABLE DDT__signal_table[] = {
  { "NONE",     "no signal",            SIGNAL_IGNORE },
  { "HUP",      "hangup",               SIGNAL_IGNORE },
  { "INT",      "interrupt",            SIGNAL_CATCH  },
  { "QUIT",     "quit",                 SIGNAL_CATCH  },
  { "ILL",      "illegal instruction",  SIGNAL_CATCH  },
  { "TRAP",     "trace trap",           SIGNAL_CATCH  },
  { "ABRT",     "IOT trap",             SIGNAL_CATCH  },
  { "EMT",      "EMT trap",             SIGNAL_IGNORE },
  { "FPE",      "arithmetic exception", SIGNAL_CATCH  },
  { "KILL",     "kill",                 SIGNAL_IGNORE },
  { "BUS",      "bus error",            SIGNAL_CATCH  },
  { "SEGV",     "segmentation violation", SIGNAL_CATCH  },
  { "SYS",      "bad argument to system call", SIGNAL_CATCH  },
  { "PIPE",     "broken pipe",          SIGNAL_CATCH  },
  { "ALRM",     "alarm clock",          SIGNAL_IGNORE },
  { "TERM",     "software termination", SIGNAL_CATCH  },
  { "URG",      "urgent condition",     SIGNAL_CATCH  },
  { "STOP",     "stop",                 SIGNAL_CATCH  },
  { "TSTP",     "keyboard stop",        SIGNAL_IGNORE },
  { "CONT",     "continue",             SIGNAL_IGNORE },
  { "CHLD",     "child status",         SIGNAL_IGNORE },
  { "TTIN",     "background read",      SIGNAL_CATCH  },
  { "TTOU",     "background write",     SIGNAL_CATCH  },
  { "IO",       "I/O possible",         SIGNAL_CATCH  },
  { "XCPU",     "cpu time exceeded",    SIGNAL_CATCH  },
  { "XFSZ",     "file size exceeded",   SIGNAL_CATCH  },
  { "VTALRM",   "virtual time alarm",   SIGNAL_CATCH  },
  { "PROF",     "profiling timer",      SIGNAL_CATCH  },
  { "WINCH",    "window changed",       SIGNAL_IGNORE },
  { "LOST",     "resource lost",        SIGNAL_CATCH  },
  { "USR1",     "user_defined 1",       SIGNAL_CATCH  },
  { "USR2",     "user_defined 2",       SIGNAL_CATCH  },
  { NULL,0,0 }
};

#else

SIGNAL_TABLE DDT__signal_table[] = {
  { "NONE",     "no signal",            SIGNAL_IGNORE },
  { "HUP",      "Hangup",               SIGNAL_IGNORE },
  { "INT",      "Interrupt",            SIGNAL_CATCH  },
  { "QUIT",     "Quit",                 SIGNAL_CATCH  },
  { "ILL",      "Illegal instruction",  SIGNAL_CATCH  },
  { "TRAP",     "Trace/BPT trap",       SIGNAL_CATCH  },
  { "IOT",      "IOT trap",             SIGNAL_CATCH  },
  { "EMT",      "EMT trap",             SIGNAL_IGNORE },
  { "FPE",      "Floating point exception", SIGNAL_IGNORE },
  { "KILL",     "Killed",               SIGNAL_IGNORE },
  { "BUS",      "Bus error",            SIGNAL_CATCH  },
  { "SEGV",     "Segmentation fault",   SIGNAL_CATCH  },
  { "SYS",      "Bad system call",      SIGNAL_CATCH  },
  { "PIPE",     "Broken pipe",          SIGNAL_CATCH  },
  { "ALRM",     "Alarm clock",          SIGNAL_IGNORE },
  { "TERM",     "Terminated",           SIGNAL_CATCH  },
  { "URG",      "Urgent I/O condition", SIGNAL_CATCH  },
  { "STOP",     "Stopped (signal)",     SIGNAL_CATCH  },
  { "TSTP",     "Stopped",              SIGNAL_IGNORE },
  { "CONT",     "Continued",            SIGNAL_IGNORE },
  { "CHLD",     "Child exited",         SIGNAL_IGNORE },
  { "TTIN",     "Stopped (tty input)",  SIGNAL_CATCH  },
  { "TTOU",     "Stopped (tty output)", SIGNAL_CATCH  },
  { "IO",       "I/O possible",         SIGNAL_CATCH  },
  { "XCPU",     "Cputime limit exceeded", SIGNAL_CATCH  },
  { "XFSZ",     "Filesize limit exceeded", SIGNAL_CATCH  },
  { "VTALRM",   "Virtual timer expired", SIGNAL_CATCH  },
  { "PROF",     "Profiling timer expired", SIGNAL_CATCH  },
  { "WINCH",    "Window size changes",  SIGNAL_IGNORE },
  { "LOST",     "Signal 29",            SIGNAL_CATCH  },
  { "USR1",     "User signal 1",        SIGNAL_CATCH  },
  { "USR2",     "User signal 2",        SIGNAL_CATCH  },
  { NULL,0,0 }
};

#endif





/************************************************************************/
/*									*/
/*	Local tables							*/
/*									*/
/************************************************************************/


typedef struct _RUN_PAT {
   String text;
   Boolean trace;
   Character call;		/* 0 -- no; 1 -- call; 2 -- return	*/
   Character sig;		/* 0 -- no; # -- # */
   Boolean line;
   Boolean ignore;
   Boolean forcestop;
   Boolean exit;
   Boolean printit;
   PMAT_PAT pattern;
} RUN_PAT;


#ifndef mips

static RUN_PAT run_events[] = {
   { "stopped in %2s at line %3d in file \"%1s\"",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "stopped in %2s at line %3d",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "trace:% %3d",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "at line %3d in file \"%1s\": %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "signal %9s (%s) in %2s at line %3d in file \"%1s\"",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "signal %9s (%s) in %2s at line %3d",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "signal %9s (%s) in %2s at 0x%6x",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "interrupt in %2s at line %3d in file \"%1s\"",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  TRUE,  NULL },
   { "interrupt in %2s at line %3d",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "interrupt in %2s at 0x%6x",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "Unexpected SIG%9s in %2s at 0x%6x",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "Unexpected SIG%9s in %2s at line %3d",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "Unexpected SIG%9s in %2s at line %3d in file \"%1s\"",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, TRUE,  TRUE , NULL },
   { "after line %3d:% %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "initially (at line %3d):% %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "trace:  couldn't read",
	TRUE,  0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "calling %2s(%4s) from function %8r",
	TRUE,  1, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "calling %2s(%4s) from program %8r",
	TRUE,  1, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning %8s from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning 0x%s %8q from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "stopped in %2s at 0x%6x",
	FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "Running: ",
	FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "program exited with %9d",
	FALSE, 0, 0, FALSE, TRUE,  TRUE,  TRUE,  FALSE, NULL },
   { "program terminated by signal %9s",
	FALSE, 0, 0, FALSE, TRUE,  TRUE,  TRUE,  FALSE, NULL },
   { "warning: object file read error: text address not found",
	FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "Current function is %s",
	FALSE, 0, 0, TRUE,  TRUE,  FALSE, FALSE, FALSE, NULL },
   { NULL, FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL }
};

#else

static RUN_PAT run_events[] = {
   { "[%7d] stopped at [%2s:%3d ,0x%6x]",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "[%7d] stopped at [%2s:%3d +0x%x,0x%6x]",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "trace:% %3d",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "trace:% \"%1s\":%3d",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "at line %3d in file \"%1s\": %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "Hangup [%2s:%3d ,0x%6x]",
	 FALSE, 0, 1, TRUE,  FALSE, FALSE, TRUE,  FALSE, NULL },
   { "%9s[%2s:%3d ,0x%6x]",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  FALSE, NULL },
   { "%9s[%2s:%3d +0x%x,0x%6x]",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  FALSE, NULL },
   { "after line %3d:% %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "initially (at line %3d):% %4s = %8r",
	 TRUE,	0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "trace:  couldn't read",
	TRUE,  0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "call: >% %d %2s(%4s)",
	TRUE,  1, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning %8s from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "returning 0x%s %8q from %2s",
	TRUE,  2, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "stopped in %2s at 0x%6x",
	FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "Running: ",
	FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "program exited with %9d",
	FALSE, 0, 0, FALSE, TRUE,  TRUE,  TRUE,  FALSE, NULL },
   { "program terminated by signal %9s",
	FALSE, 0, 0, FALSE, TRUE,  TRUE,  TRUE,  FALSE, NULL },
   { "warning: object file read error: text address not found",
	FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { NULL, FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL }
};

#endif




static RUN_PAT gdb_run_events[] = {
   { "Bpt %7d, %2s (%s) (%1s line %3d)",
	 FALSE, 0, 0, TRUE,  FALSE, FALSE, FALSE, FALSE, NULL },
   { "Bpt %7d, %6x in %2s (%s)",
	 FALSE, 0, 0, FALSE, FALSE, FALSE, FALSE, FALSE, NULL },
   { "%s (%s) (%s line %d)",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "%d\t",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "0x%x\t%d\t",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "Starting program: ",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "Continuing.",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "Continuing with signal",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "(More stack frames follow...)",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "ptrace: I/O error",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },
   { "Program received signal %9d, %s",
	 FALSE, 0, 2, TRUE,  FALSE, FALSE, TRUE,  FALSE, NULL },
   { "The program being debugged has been started already.",
	 FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL },

   { NULL, FALSE, 0, 0, FALSE, TRUE,  FALSE, FALSE, FALSE, NULL }
};







typedef struct _EVENT_PAT {
   String text;
   PMAT_PAT pattern;
} EVENT_PAT;


#ifndef mips

static EVENT_PAT event_pats[] = {
   { "(%1d) stop% at \"%2s\":%3d",      NULL },
   { "(%1d) stop% at %3d",              NULL },
   { "(%1d) stop% in %4s",              NULL },
   { "(%1d) stop% %5s",                 NULL },
   { "(%1d) stop% if",                  NULL },
   { "(%1d) trace% at \"%2s\":%3d",     NULL },
   { "(%1d) trace% at %3d",             NULL },
   { "(%1d) trace %5s at \"%2s\":%3d",  NULL },
   { "(%1d) trace %5s at %3d",          NULL },
   { "(%1d) trace %4s",                 NULL },
   { "(%1d) tracei% at \"%2s\":%3d",     NULL },
   { "(%1d) tracei% at %3d",             NULL },
   { "(%1d) tracei %5s at \"%2s\":%3d",  NULL },
   { "(%1d) tracei %5s at %3d",          NULL },
   { "(%1d) tracei %4s",                 NULL },
   { NULL,				NULL },
};

#else

static EVENT_PAT event_pats[] = {
   { "[%1d] stop% at \"%2s\":%3d",      NULL },
   { "[%1d] stop% at %3d",              NULL },
   { "[%1d] stop% in %4s",              NULL },
   { "[%1d] stop% %5s",                 NULL },
   { "[%1d] stop% if",                  NULL },
   { "[%1d] trace% at \"%2s\":%3d",     NULL },
   { "[%1d] trace% at %3d",             NULL },
   { "[%1d] trace %5s at \"%2s\":%3d",  NULL },
   { "[%1d] trace %5s at %3d",          NULL },
   { "[%1d] trace %4s",                 NULL },
   { NULL,				NULL },
};

#endif


static EVENT_PAT gdb_event_pats[] = {
   { "Breakpoint %1d at %s: file %2s, line %3d.",       NULL },
   { "Breakpoint %1d at %s",                            NULL },
   { NULL,						NULL },
};





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	set_location();
static	Boolean 	set_line_loc();
static	Boolean 	report_location();
static	String		inq_file();
static	String		filetail();
static	void		manage_event();
static	void		trace_all_calls();
static	void		event_all_calls();
static	Boolean 	scan_where();
static	Boolean 	scan_where_gdb();
static	Boolean 	prefix();
static	String		set_function();
static	String		inq_function();
static	Integer 	get_line_number();
static	Integer 	find_signal();
static	String		fix_name_separator();
static	String		unfix_name_separator();
static	void		stack_bottom();
static	String		scan_dump_var();
static	Integer 	gdb_call_trace();





/************************************************************************/
/*									*/
/*	DDT_x_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DDT_x_init()
{
   Integer i;
   Universal args[EVENT_ARGS+1];

   args[EV_IDX_FILE] = NULL;
   args[EV_IDX_FUNCTION] = NULL;
   args[EV_IDX_LINE] = 0;
   args[EV_IDX_EXPRESSION] = NULL;
   args[EV_IDX_CONDITION] = NULL;
   args[EV_IDX_ADDRESS] = 0;
   args[EV_IDX_ID] = 0;
   args[EV_IDX_NEW] = NULL;
   args[EV_IDX_SIGNAL] = NULL;
   args[EVENT_ARGS] = 0;

   for (i = 0; run_events[i].text != NULL; ++i) {
      run_events[i].pattern = PMATmake_pattern(run_events[i].text,EVENT_ARGS,args);
    };
   for (i = 0; gdb_run_events[i].text != NULL; ++i) {
      gdb_run_events[i].pattern = PMATmake_pattern(gdb_run_events[i].text,EVENT_ARGS,args);
    };

   args[EP_IDX_ID] = 0;
   args[EP_IDX_FILE] = NULL;
   args[EP_IDX_LINE] = 0;
   args[EP_IDX_FUNC] = NULL;
   args[EP_IDX_EXPR] = NULL;

   for (i = 0; event_pats[i].text != NULL; ++i) {
      event_pats[i].pattern = PMATmake_pattern(event_pats[i].text,EP_ARGS,args);
    };
   for (i = 0; gdb_event_pats[i].text != NULL; ++i) {
      gdb_event_pats[i].pattern = PMATmake_pattern(gdb_event_pats[i].text,EP_ARGS,args);
    };

   lineaddr_pattern = PMATmake_pattern(LINEADDR_PAT,1,NULL);
   eof_pattern = PMATmake_pattern(EOF_PATTERN,1,NULL);
   where_file_pat = PMATmake_pattern(WHERE_FILE,2,NULL);
   where_addr_pat = PMATmake_pattern(WHERE_ADDR,1,NULL);

   event_pending = -1;
};






/************************************************************************/
/*									*/
/*	DDT_x_debugger_name -- return name of the debugger		*/
/*	DDT_x_debugger_path -- return path name for debugger		*/
/*									*/
/************************************************************************/


String
DDT_x_debugger_name()
{
   return (DDT__use_gdb ? "gdb" : "dbx");
};




String
DDT_x_debugger_path()
{
   AUXD hdl;
   String s;

   hdl = AUXDget_handle(NULL,"DDT");

   s = AUXDget_info(hdl,"DEBUG_PATH");
   if (s == NULL) {
      if (DDT__use_gdb) {
	 s = getenv("GDB");
	 if (s == NULL) s = GDB_PATH;
       }
      else {
	 s = getenv("DBX");
	 if (s == NULL) s = DBX_PATH;
       };
    };

   return s;
};





/************************************************************************/
/*									*/
/*	DDT_x_check_prompt -- check string to see if its a prompt	*/
/*	DDT_x_check_query -- check string to see if its a query 	*/
/*									*/
/************************************************************************/


Boolean
DDT_x_check_prompt(str)
   String str;
{
   Integer i;
   String p;

   p = (DDT__use_gdb ? "(gdb) " : "(dbx) ");

   i = strlen(str);
   if (i < 6) return FALSE;
   str = &str[i-6];

   return STREQL(str,p);
};





Boolean
DDT_x_check_query(str)
   String str;
{
   Integer i,ln;
   String s;

   if (DDT__use_gdb) {
      i = strlen(str);
      s = "(y or n) ";
      ln = strlen(s);
      if (i >= ln && STREQL(&str[i-ln],s)) {
	 DDT_printf("y\n");
	 return TRUE;
       };

      s = "---Type <return> to continue---";
      ln = strlen(s);
      if (i >= ln && STREQL(&str[i-ln],s)) {
	 DDT_printf("y\n");
	 return TRUE;
       };

      if (STREQL(s,"--Type Return to print more--")) {
	 DDT_printf("\n");
	 return TRUE;
       };
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DDT_x_startup -- handle commands at start			*/
/*	DDT_x_system_info -- get system information			*/
/*									*/
/************************************************************************/


void
DDT_x_startup()
{
   if (DDT__use_gdb) {
      DDT_debug_command("define dumpvars\ninfo args\ninfo locals\nend\n");
      DDT_debug_command("set screensize 0\n");
    }
   else {
#ifdef mips
      DDT_debug_command("set $casesense = 1\n");
      DDT_debug_command("set $page = 0\n");
      DDT_debug_command("set $printdata = 1\n");
      DDT_debug_command("set $readtextfile = 0\n");
      DDT_debug_command("set $repeatmode = 0\n");
      if (DDT__stdpascal) {
	 DDT_debug_command("set $casesense = 0\n");
       };
#endif
#ifdef sun
      DDT_debug_command("alias return next\n");
#endif
    };
};




Boolean
DDT_x_system_info()
{
   Boolean fg;
   struct stat buf;

   if (stat(DDT__system_name,&buf) < 0) return FALSE;
   if ((buf.st_mode & 0111) == 0) return FALSE;

   fg = DDT_symbol_setup(DDT__system_name);

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_run_monitor -- catch dbx messages while running		*/
/*									*/
/************************************************************************/


Boolean
DDT_x_run_monitor(text)
   String text;
{
   Universal args[EVENT_ARGS+1];
   Boolean match;
   Integer i,j;
   RUN_PAT * evts;
   String s1,s2;

   match = FALSE;

   if (event_pending >= 0) {
      j = get_line_number(text);
#ifdef mips
      if (j <= 0 && !DDT__use_gdb && (prefix(text,"        ") || text[0] == '\t'))
	 j = pending_args[EV_IDX_LINE];
#endif
      if (j > 0) {
	 pending_args[EV_IDX_LINE] = j;
	 manage_event(event_pending,pending_args);
	 match = TRUE;
	 event_pending = -1;
       }
      else if (DDT__use_gdb) {
	 if (prefix(text,"0x")) {
	    if (scan_where_gdb(text,&pending_args[EV_IDX_LINE],
				  &pending_args[EV_IDX_FILE],
				  &pending_args[EV_IDX_ADDRESS],
				  &pending_args[EV_IDX_FUNCTION],
				  &s1)) {
		if (args[EV_IDX_LINE] > 0) return TRUE;
	      };
	  };
	 manage_event(event_pending,pending_args);
	 match = TRUE;
	 event_pending = -1;
       }
      else if (PMATmatch(text,lineaddr_pattern,args) >= 0) {
	 s1 = (String) args[0];
	 s2 = index(s1,'+');
	 if (s2 != NULL) *s2 = 0;
	 if (STREQL(s1,pending_args[EV_IDX_FUNCTION])) {
	    manage_event(event_pending,pending_args);
	    match = TRUE;
	    event_pending = -1;
	  };
	 PMATfree_args(lineaddr_pattern,args);
       };
    };

   if (!match) {
      event_pending = -1;
      evts = (DDT__use_gdb ? gdb_run_events : run_events);
      for (i = 0; evts[i].pattern != NULL; ++i) {
	 if (PMATmatch(text,evts[i].pattern,args) >= 0) {
	    if (!evts[i].printit) match = TRUE;
	    if (evts[i].line) {
	       event_pending = i;
	       for (j = 0; j <= EVENT_ARGS; ++j) pending_args[j] = args[j];
	     }
	    else {
	       manage_event(i,args);
	     };
	    break;
	  };
       };
    };

   return match;
};





/************************************************************************/
/*									*/
/*	DDT_x_add_event -- create an event				*/
/*	DDT_x_remove_event -- remove an event				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_add_event(act,file,func,line,expr,cond,addr,name,itr,etr)
   EVENT_ACTION act;
   String file;
   String func;
   Integer line;
   String expr;
   String cond;
   Integer addr;
   String name;
   Boolean itr;
   Boolean etr;
{
   Character buf[1024],ebuf[1024],cbuf[1024],fbuf[1024];
   String s,t;
   Character ftxt[128],ltxt[128];
   Boolean fg,done;
   String inst;
   LOCATION oloc;
   Integer id;
   Universal args[EP_ARGS];
   Integer i;
   String uexpr;
   Integer eline;
   EVENT_PAT * ept;

   DDT_model_inq_location(&oloc);

   uexpr = expr;
   if (act == EVENT_EVENT || act == EVENT_TRIGGER || act == EVENT_MONITOR ||
	  act == EVENT_UPDATE || act == EVENT_DISPLAY || act == EVENT_WHEN)
      expr = NULL;

   if ((act == EVENT_BREAK || act == EVENT_EVENT || act == EVENT_TRIGGER ||
	   act == EVENT_TBREAK || act == EVENT_UPDATE || act == EVENT_STEPSTEP ||
	   act == EVENT_STOPEXIT ||
	   act == EVENT_NEXTNEXT || act == EVENT_WHEN) &&
	  file != NULL && line > 0 && func == NULL) {
      func = DDT_symbol_fct_from_file(file,line);
    };

   if (func != NULL) {
      s = DDT_map_fix_function(func,fbuf);
      if (s != func) {
	 func = DDT_map_unfix_function(s);
       };
    };

   if (!set_location(file,func,line,TRUE,FALSE)) return FALSE;

   if ((act == EVENT_BREAK || act == EVENT_EVENT || act == EVENT_TRIGGER ||
	   act == EVENT_TBREAK || act == EVENT_UPDATE || act == EVENT_STEPSTEP ||
	   act == EVENT_STOPEXIT ||
	   act == EVENT_NEXTNEXT || act == EVENT_WHEN) &&
	  expr == NULL && line == 0 && func != NULL && !DDT__use_gdb) {
      s = DDT_map_fix_function(func,buf);
      DDT_symbol_funct_info(s,&file,&line,&eline);
      if (act == EVENT_STOPEXIT) line = eline;
    };

   if (addr > 0) {
      inst = (DDT__use_gdb ? "*" : "i");
      line = addr;
    }
   else inst = "";

   if (file != NULL) {
      if (line > 0) {
	 if (DDT__use_gdb) sprintf(ltxt,"%s:%d",filetail(file),line);
#ifndef mips
	 else sprintf(ltxt,"\"%s\":%d",file,line);
#else
	 else sprintf(ltxt,"\"%s\":%d",DDT_symbol_fix_file_name(file),line);
#endif
       }
      if (func != NULL) {
	 s = DDT_map_fix_function(func,buf);
	 if (DDT__use_gdb) sprintf(ftxt,"%s:%s",filetail(file),s);
	 else sprintf(ftxt,"%c%s%c%s",NAME_SEP_CHAR,filetail(file),NAME_SEP_CHAR,s);
       };
    }
   else {
      file = inq_file();
      if (line > 0) sprintf(ltxt,"%d",line);
      if (func != NULL) {
	 s = DDT_map_fix_function(func,buf);
	 sprintf(ftxt,"%s",s);
	 s = set_function(func);
	 if (s != NULL && prefix(s,"warning:")) file = NULL;
	 else file = inq_file();
       };
    };

   fg = TRUE;
   done = FALSE;

   if (!DDT__use_gdb && act == EVENT_WATCH &&
	  func == NULL && expr != NULL && line == 0) {
      s = DDT_debug_command("which %s\n",expr);
      if (index(s,NAME_SEP_CHAR) != NULL) {
	 t = s;
	 if (*t == NAME_SEP_CHAR) ++t;
	 i = 0;
	 for ( ; *t != 0; ++t) if (*t == NAME_SEP_CHAR) ++i;
	 if (i > 1) {
	    strcpy(ftxt,s);
	    t = rindex(ftxt,NAME_SEP_CHAR);
	    *t = 0;
	    func = ftxt;
	  };
       };
    };

   if (expr != NULL) {
      expr = DDT_expr_parse(expr,ebuf);
      expr = fix_name_separator(expr);
      uexpr = expr;
    };
   if (cond != NULL) {
      cond = DDT_expr_parse(cond,cbuf);
      cond = fix_name_separator(cond);
    };

   if (DDT__use_gdb) {
      if (act == EVENT_TRACE && expr != NULL) act = EVENT_MONITOR;
      else if (act == EVENT_TRACE && line <= 0) {
	 DDT_mprint("Continuous tracing not supported by gdb\n");
	 return FALSE;
       }
    };

   switch (act) {
      case EVENT_TRACE :
      case EVENT_WATCH :
	 if (expr != NULL) {
	    if (line > 0) sprintf(buf,"trace%s %s at %s",inst,expr,ltxt);
	    else if (func != NULL) sprintf(buf,"trace %s in %s",expr,ftxt);
	    else sprintf(buf,"trace %s",expr);
	  }
	 else if (line > 0) {
	    if (DDT__use_gdb) sprintf(buf,"break %s%s",inst,ltxt);
	    else {
#ifndef mips
	       sprintf(buf,"trace%s %s",inst,ltxt);
#else
	       sprintf(buf,"when%s at %s { printf \"trace: %s\\n\" }",inst,ltxt,ltxt);
	       cond = NULL;
#endif
	     };
	  }
	 else if (func != NULL) {
	    sprintf(buf,"trace in %s",ftxt);
	  }
	 else sprintf(buf,"trace");
	 break;

      case EVENT_CALL :
	 if (func != NULL) {
	    if (DDT__use_gdb) {
	       id = gdb_call_trace(file,func);
	       if (id == 0) return FALSE;
	       if (!DDT_event_add(act,name,file,func,line,uexpr,cond,addr,id,itr,etr)) {
		  DDT_mprint("Duplicate event ignored\n");
		  DDT_x_remove_event(id);
		};
	       done = TRUE;
	     }
	    else {
#ifndef mips
	       sprintf(buf,"trace %s",ftxt);
#else
	       sprintf(buf,"when in %s { printf \"call: \"; where 1 }",ftxt,ltxt);
#endif
	     };
	  }
	 else {
	    trace_all_calls(itr,etr);
	    done = TRUE;
	  };
	 break;

      case EVENT_EVENT :
      case EVENT_TRIGGER :
      case EVENT_UPDATE :
      case EVENT_STEPSTEP :
      case EVENT_NEXTNEXT :
      case EVENT_BREAK :
      case EVENT_TBREAK :
      case EVENT_WHEN :
      case EVENT_STOPEXIT :
	 if (act == EVENT_UPDATE && line == 0 && func == NULL && expr == NULL) {
	    DDT_event_add(act,name,file,func,line,uexpr,cond,addr,0,itr,etr);
	    done = TRUE;
	  }
	 else if (line == 0 && func == NULL && expr == NULL) {
	    event_all_calls(act,uexpr,cond,name,itr,etr);
	    done = TRUE;
	  }
	 else if (DDT__use_gdb) {
	    if (line > 0) sprintf(buf,"break %s%s",inst,ltxt);
	    else if (func != NULL) sprintf(buf,"break %s",ftxt);
	    else fg = FALSE;
	  }
	 else {
	    if (expr != NULL) sprintf(buf,"stop %s",expr);
	    else if (line > 0) sprintf(buf,"stop%s at %s",inst,ltxt);
	    else if (func != NULL) sprintf(buf,"stop in %s",ftxt);
	    else fg = FALSE;
	  };
	 break;

      case EVENT_MONITOR :
	 if (line > 0) {
	    if (DDT__use_gdb) sprintf(buf,"break %s",ltxt);
	    else sprintf(buf,"stop at %s",ltxt);
	  }
	 else {
	    DDT_event_add(act,name,file,func,line,uexpr,cond,addr,0,itr,etr);
	    done = TRUE;
	  };
	 break;

      case EVENT_DISPLAY :
	 if (line == 0 && func == NULL && uexpr == NULL) {
	    event_all_calls(EVENT_UPDATE,uexpr,cond,name,itr,etr);
	  }
	 else {
	    DDT_event_add(act,name,file,func,line,uexpr,cond,addr,0,itr,etr);
	  };
	 done = TRUE;
	 break;

      default :
	 fg = FALSE;
	 break;
    };

   if (!done && fg && cond != NULL) {
      strcat(buf," if ");
      strcat(buf,cond);
    };

   if (!done) {
      if (fg) s = DDT_debug_command("%s\n",buf);
      else s = "Illegal trace/break command";
      if (s == NULL) s = "Problem with trace/break command";

      if (prefix(s,"calling ") && index(s,'\n') != NULL) {
	 s = index(s,'\n');
	 ++s;
       };

      if (DDT__use_gdb && prefix(s,"Note: breakpoint")) {
	 s = index(s,'\n');
	 ++s;
       };

      if (DDT__use_gdb && !prefix(s,"Breakpoint")) fg = FALSE;
      else if (!DDT__use_gdb && s[0] != EVENT_OK_CHAR) fg = FALSE;

      if (!fg) {
	 if (s == NULL || *s == 0) s = "Problem with trace/break command";
	 DDT_mprint("%s\n",s);
	 fg = FALSE;
       }
      else {
	 ept = (DDT__use_gdb ? gdb_event_pats : event_pats);

	 for (i = 0; ept[i].text != NULL; ++i) {
	    if (PMATmatch(s,ept[i].pattern,args) > 0) break;
	  };

	 if (ept[i].text != NULL) {
	    if (args[EP_IDX_EXPR] == NULL && args[EP_IDX_FUNC] != NULL &&
		   act == EVENT_WATCH) {
	       args[EP_IDX_EXPR] = args[EP_IDX_FUNC];
	       args[EP_IDX_FUNC] = NULL;
	     };

	    id = args[EP_IDX_ID];
	    if (args[EP_IDX_FILE] != NULL) {
	       file = DDT_symbol_fix_file_name((String) args[EP_IDX_FILE]);
	     };
	    if (args[EP_IDX_LINE] != 0) line = (Integer) args[EP_IDX_LINE];
	    if (args[EP_IDX_FUNC] != 0) {
	       func = DDT_map_unfix_function((String) args[EP_IDX_FUNC]);
	     };
	    if (args[EP_IDX_EXPR] != 0) uexpr = (String) args[EP_IDX_EXPR];
	  };

	 if (DDT__use_gdb && expr != NULL) {	/* handle expression casing */
	  };

	 if (!DDT_event_add(act,name,file,func,line,uexpr,cond,addr,id,itr,etr)) {
	    DDT_mprint("Duplicate event ignored\n");
	    DDT_debug_command("delete %d\n",id);
	  };
       };
    };

   set_location(oloc.file,oloc.func,oloc.line,FALSE,TRUE);

   return fg;
};





Boolean
DDT_x_remove_event(id)
   Integer id;
{
   String s;
   Boolean fg;
   Integer id0,id1;

   if (id < 0) {
      id = -id;
      id0 = id >> 16;
      id1 = id & 0xffff;
      for (id = id0; id <= id1; ++id) {
	 s = DDT_debug_command("delete %d\n",id);
       };
      s = NULL;
    }
   else if (id > 0) {
      s = DDT_debug_command("delete %d\n",id);
    }
   else s = NULL;

   if (s == NULL) fg = TRUE;
   else fg = FALSE;

   return fg;
};




/************************************************************************/
/*									*/
/*	DDT_x_assign -- assign to a variable				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_assign(var,expr)
   String var;
   String expr;
{
   String s,c;

   c = (DDT__use_gdb ? "set variable" : "assign");

   s = DDT_debug_call_command("%s %s = %s\n",c,var,expr);

   if (s != NULL) {
      DDT_mprint("%s\n",s);
      return FALSE;
    }

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_call -- call routine with arguments			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_call(rtn,args)
   String rtn;
   String args;
{
   String s,expr;
   Character buf[10240];

   sprintf(buf,"%s(%s)",rtn,args);
   expr = DDT_expr_parse(buf,buf);
   expr = fix_name_separator(expr);

   if (DDT__use_gdb) {
      s = DDT_debug_call_command("set %s\n",expr);
    }
   else {
#ifndef mips
      s = DDT_debug_call_command("call %s\n",expr);
#else
      s = DDT_debug_call_command("printf \"\", %s\n",expr);
#endif
    };

   if (s != NULL) {
      DDT_mprint("%s\n",s);
      return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_continue -- issue continue command			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
DDT_x_continue(cnt,sig)
   Integer cnt;
   Integer sig;
{
   exit_status = 0;
   exit_signal = NULL;

   DDT_msg_continue_run();

   if (DDT__use_gdb) {
      if (sig == 0) DDT_state_run_command("cont\n");
      else DDT_state_run_command("signal %d\n",sig);
    }
   else if (sig == 0) {
#ifndef mips
      DDT_state_run_command("cont\n");
#else
      if (cnt <= 1 || DDT__use_gdb) DDT_state_run_command("cont\n");
      else DDT_state_run_command("%d cont\n",cnt);
#endif
    }
   else {
#ifndef mips
      DDT_state_run_command("cont sig %d\n",sig);
#else
      if (cnt <= 1) DDT_state_run_command("cont %d\n",sig);
      else DDT_state_run_command("%d cont %d\n",cnt,sig);
#endif
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_current_line -- list current line 			*/
/*	DDT_x_current_addr -- list current line of execution		*/
/*									*/
/************************************************************************/


Boolean
DDT_x_current_line()
{
   LOCATION loc;
   String s;

   DDT_model_inq_location(&loc);

   if (loc.line <= 0) return FALSE;

   if (DDT__use_gdb) {
      s = DDT_debug_command("list %d,%d\n",loc.line,loc.line);
    }
   else {
      s = DDT_debug_command("list %d\n",loc.line);
    };

   DDT_mprint("%s\n",s);

   set_line_loc(loc.line);

   return TRUE;
};






Boolean
DDT_x_current_addr(force)
   Boolean force;
{
   LOCATION floc,eloc;
   Integer addr;
   String s;

   DDT_model_inq_location(&floc);
   DDT_model_inq_exec_location(&eloc,&addr);

   if (((floc.file == NULL && eloc.file == NULL) ||
	   (floc.file != NULL && eloc.file != NULL && STREQL(floc.file,eloc.file))) &&
	  floc.func == NULL && eloc.func != NULL &&
	  floc.line == eloc.line && floc.line > 0) {
      floc.func = eloc.func;
    };

   if (!force && ((floc.file == NULL && eloc.file == NULL) ||
		     (floc.file != NULL && eloc.file != NULL && STREQL(floc.file,eloc.file))) &&
	  ((floc.func == NULL && eloc.func == NULL) ||
	      (floc.func != NULL && eloc.func != NULL && STREQL(floc.func,eloc.func))) &&
	  floc.line == eloc.line && floc.line > 0) {
      if (addr == 0) return FALSE;
      else eloc.line = 0;
    };

   if (eloc.line > 0) {
      if (DDT__use_gdb) {
	 s = DDT_debug_command("list %d,%d\n",eloc.line,eloc.line);
       }
      else {
	 s = DDT_debug_command("list %d\n",eloc.line);
       };
      DDT_mprint("%s\n",s);
      set_location(floc.file,floc.func,floc.line,FALSE,TRUE);
    }
   else {
      if (DDT__use_gdb) {
	 s = DDT_debug_command("x/i $pc\n");
       }
      else {
	 s = DDT_debug_command("&$pc/i\n");
       };
      DDT_mprint("%s\n",s);
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	DDT_x_dump -- dump addresses in given format			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_dump(from,to,len,fmt)
   String from;
   String to;
   Integer len;
   String fmt;
{
   String s;
   Boolean fg;

   if (DDT__use_gdb) {
      if (to == NULL) {
	 s = DDT_debug_command("x/%d%s %s\n",len,fmt,from);
       }
      else {
	 s = DDT_debug_command("printf (\"%d\n\",((%s)-(%s))\n",to,from);
	 len = atol(s);
	 s = DDT_debug_command("x/%d%s %s\n",len,fmt,from);
       };
    }
   else {
      if (to == NULL) {
	 s = DDT_debug_command("%s/%d%s\n",from,len,fmt);
       }
      else {
	 s = DDT_debug_command("%s,%s/%s",from,to,fmt);
       };
    };

   DDT_mprint("%s\n",s);

   fg = isdigit(s[0]);

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_kill -- kill running process				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_kill(initfg)
   Boolean initfg;
{
   String s;

   s = DDT_debug_command("kill\n");

   if (s != NULL && DDT__use_gdb && STREQL(s,"The program is not being run."))
      s = NULL;

   if (!initfg) DDT_msg_stop_run(256,NULL);

   if (s != NULL) DDT_mprint("%s\n",s);

   return (s == NULL);
};





/************************************************************************/
/*									*/
/*	DDT_x_list -- list some of source				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_list(from,to)
   Integer from;
   Integer to;
{
   String s;

#ifndef mips
   s = DDT_debug_command("list %d,%d\n",from,to);
#else
   if (DDT__use_gdb) s = DDT_debug_command("list %d,%d\n",from,to);
   else s = DDT_debug_command("list %d:%d\n",from,to-from);
#endif

   DDT_mprint("%s\n",s);

   report_location();

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_print --	print expression value				*/
/*	DDT_x_eval -- evaluate expression value 			*/
/*									*/
/************************************************************************/


#define MAX_EXPR_LEN	32768

Boolean
DDT_x_print(expr)
   String expr;
{
   Character buf[MAX_EXPR_LEN];
   Boolean fg;

   fg = DDT_x_eval(expr,buf,MAX_EXPR_LEN);

   DDT_mprint("%s",buf);
   if (strlen(buf) >= MAX_EXPR_LEN-1) DDT_mprint("...");
   DDT_mprint("\n");

   return fg;
};





Boolean
DDT_x_eval(expr,buf,len)
   String expr;
   String buf;
   Integer len;
{
   String s,t,bf;
   Integer i,j;
   Boolean fg,pfxfg;
   Character ebuf[10240];

   expr = DDT_expr_parse(expr,ebuf);
   if (expr != ebuf) {
      strcpy(ebuf,expr);
      expr = ebuf;
    };
   expr = fix_name_separator(expr);

   s = DDT_debug_call_command("print %s\n",expr);

   if (s == NULL) return FALSE;

   i = strlen(s);
   j = len*4/3;
   if (DDT__cplusplus != 0 && i > j) {
      t = &s[j];
      while (*t != 0 && *t != '\n') ++t;
      *t = 0;
      i = strlen(s);
    };

   j = strlen(expr);
   if (i < 10200-j) bf = ebuf;
   else bf = (String) alloca(i+200+j);

   pfxfg = TRUE;

#ifdef mips
   if (DDT__cplusplus != 0) {
      sprintf(bf,"%s = %s",expr,s);
      s = bf;
    }
   else pfxfg = FALSE;
#endif

   s = DDT_map_fix_outexpr(s,FALSE,bf);

   if (pfxfg) {
      for (i = 0; s[i] != 0; ++i) {
	 if (s[i] == ' ' && s[i+1] == '=' && s[i+2] == ' ') break;
       };

      while (s[i] == ' ') ++i;
      if (s[i] == '=') ++i;
    }
   else i = 0;

   while (s[i] == ' ') ++i;

   if (s[i] != 0) {
      s = &s[i];
      fg = TRUE;
    }
   else if (isdigit(s[0])) fg = TRUE;
   else fg = FALSE;

   if (strlen(s) >= len) s[len-1] = 0;

   strcpy(buf,s);

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_quit -- quit debugger					*/
/*									*/
/************************************************************************/


Boolean
DDT_x_quit()
{
   if (DDT__use_gdb) DDT_debug_command("quit\ny\n");
   else DDT_debug_command("quit\n");

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_reinit -- reinitialize debugger				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_reinit(sys,cor)
   String sys;
   String cor;
{
   String s,c;

   DDT_symbol_free_info();

   if (DDT__use_gdb) {
      s = DDT_debug_command("delete\n");
      if (sys == NULL) sys = DDT__system_name;
      s = DDT_debug_command("exec-file %s\n",sys);
      if (s != NULL) DDT_mprint("%s\n",s);
      s = DDT_debug_command("symbol-file %s\n",sys);
      if (s != NULL) DDT_mprint("%s\n",s);
      if (cor != NULL) {
	 s = DDT_debug_command("core-file %s\n",cor);
	 if (s != NULL) DDT_mprint("%s\n",s);
       };
    }
   else {
      s = DDT_debug_command("delete all\n");
      c = (cor == NULL ? "" : cor);
#ifndef mips
      if (sys == NULL) s = DDT_debug_command("debug %s %s\n",DDT__system_name,c);
      else s = DDT_debug_command("debug %s %s\n",sys,c);
      if (s != NULL) DDT_mprint("%s\n",s);
#else
      if (sys == NULL) sys = DDT__system_name;
      if (sys != NULL) DDT_start_debug(sys,cor);
#endif
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_run -- start program running				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_run(args,in,out)
   String args;
   String in;
   String out;
{
   String s;

   DDT_msg_start_run();

   exit_status = 0;
   exit_signal = NULL;

#ifndef mips
   s = (DDT__use_gdb ? "run" : "rerun");
#else
   s = "run";
#endif

   if (args == NULL) args = "";
   if (*args == 0 && DDT__use_gdb) DDT_debug_command("set args\n");

   if (in == NULL && out == NULL) {
      DDT_state_run_command("%s %s\n",s,args);
    }
   else if (in == NULL) {
      DDT_state_run_command("%s %s > %s\n",s,args,out);
    }
   else if (out == NULL) {
      DDT_state_run_command("%s %s < %s\n",s,args,in);
    }
   else {
      DDT_state_run_command("%s %s < %s > %s\n",s,args,in,out);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_search -- handle search commands				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_search(dir,ctx)
   Boolean dir;
   String ctx;
{
   String s;
   Boolean fg;
   Integer ln;

   if (dir) {
      s = DDT_debug_command("/%s/\n",ctx);
    }
   else {
      s = DDT_debug_command("?%s?\n",ctx);
    };

   ln = get_line_number(s);
   if (ln > 0) fg = TRUE;
   else {
      DDT_mprint("%s\n",s);
      fg = FALSE;
    };

   if (fg) fg = report_location();

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_set_context -- set context for commands			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
DDT_x_set_context(qid,ctx)
   Integer qid;
   String ctx;
{
   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_set_env -- set environment				*/
/*									*/
/************************************************************************/


void
DDT_x_set_env(var,val)
   String var;
   String val;
{
   String s;

#ifndef mips
   s = DDT_debug_command("setenv %s \"%s\"\n",var,val);

   if (s != NULL) DDT_mprint("%s\n",s);
#endif
};





/************************************************************************/
/*									*/
/*	DDT_x_set_option -- set options from user command		*/
/*									*/
/************************************************************************/


void
DDT_x_set_option(what,val)
   String what;
   String val;
{
   String s;

   if (STREQL(what,"QUOTE")) {
      s = DDT_debug_command("%s\n",val);
      if (s != NULL) DDT_mprint("%s\n",s);
    }
   else if (STREQL(what,"WD")) {
      s = DDT_debug_command("cd %s\n",val);
      if (s != NULL) DDT_mprint("%s\n",s);
    }
   else if (STREQL(what,"SHCMD")) {
      s = DDT_debug_command("sh %s\n",val);
      if (s != NULL) DDT_mprint("%s\n",s);
    }
   else if (STREQL(what,"STDPASCAL")) {
      DDT__stdpascal = atol(val);
      if (DDT__stdpascal) {
#ifdef mips
	 s = DDT_debug_command("set $casesense = 0\n");
#endif
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_x_set_view -- set current view location			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_set_view(file,func,line)
   String file;
   String func;
   Integer line;
{
   Boolean fg;

   fg = set_location(file,func,line,TRUE,FALSE);

   if (fg) fg = report_location();

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_show -- show information					*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

void
DDT_x_show(what,arg)
   String what;
   String arg;
{
   String s;

   if (STREQL(what,"WD")) {
      s = DDT_debug_command("pwd\n");
    }
   else s = NULL;

   if (s != NULL) DDT_mprint("%s\n",s);
};





/************************************************************************/
/*									*/
/*	DDT_x_sigset -- turn on/off signals				*/
/*									*/
/************************************************************************/


Boolean
DDT_x_sigset(sig,fgs)
   Integer sig;
   Integer fgs;
{
   String s;
   Integer i;

   if (sig < 0) {
      for (i = 0; DDT__signal_table[i].name != NULL; ++i) {
	 DDT_x_sigset(i,DDT__signal_table[i].flags);
       };
      return FALSE;
    };

   if (sig == 0) s= NULL;
   else if (fgs == SIGNAL_CATCH) {
      if (DDT__use_gdb) s = DDT_debug_command("handle %d stop\n",sig);
      else s = DDT_debug_command("catch %d\n",sig);
    }
   else if (fgs == SIGNAL_IGNORE) {
      if (DDT__use_gdb) s = DDT_debug_command("handle %d pass\n",sig);
      else s = DDT_debug_command("ignore %d\n",sig);
    }
   else if (fgs == SIGNAL_PRINTUSE) {
      if (DDT__use_gdb) s = DDT_debug_command("handle %d pass print\n",sig);
      else s = DDT_debug_command("catch %d\n",sig);
    }
   else if (fgs == SIGNAL_PRINTIGN) {
      if (DDT__use_gdb) s = DDT_debug_command("handle %d nopass print\n",sig);
      else s = DDT_debug_command("catch %d\n",sig);
    }
   else s = "Signal mode not supported";

   if (DDT__use_gdb && s != NULL && prefix(s,"Number")) s = NULL;
   if (DDT__use_gdb && s != NULL && prefix(s,"Signal")) s = NULL;

   if (s != NULL) {
      DDT_mprint("%s\n",s);
    }
   else {
      DDT__signal_table[sig].flags = fgs;
    };

   return (s == NULL);
};





/************************************************************************/
/*									*/
/*	DDT_x_step -- single step program				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
DDT_x_step(count,over,inst,retn,user)
   Integer count;
   Boolean over;
   Boolean inst;
   Boolean retn;
   Boolean user;
{
   String s,in;

   DDT_msg_continue_run();

   if (retn) {
      if (DDT__use_gdb) DDT_state_run_command("finish\n");
#ifndef sun
      else DDT_state_run_command("return\n");
#else
#ifdef sparc
      else {
	 Character buf[1024];
	 Integer i,j;

	 if (DDT_x_eval("$fp[-9]+8",buf,1020)) i = atol(buf);
	 else i = 0;
	 if (DDT_x_eval("(int) $fp",buf,1020)) j = atol(buf);
	 else j = 0;
	 sprintf(buf,"((int) $fp) > %d",j);
	 if (i > 0 && j != 0 &&
		DDT_x_add_event(EVENT_TBREAK,NULL,NULL,0,NULL,buf,i,NULL,TRUE,TRUE)) {
	    DDT_state_run_command("cont\n");
	  }
	 else {
	    DDT_mprint("return command not supported by sun's dbx\n");
	    return FALSE;
	  };
       };
#else
      else {
	 DDT_mprint("return command not supported by sun's dbx\n");
	 return FALSE;
       };
#endif
#endif
      return TRUE;
    };

   s = (over ? "next" : "step");
   in = (inst ? "i" : "");

   if (count <= 0) count = 1;

   exit_status = 0;
   exit_signal = NULL;

   if (user) DDT_state_step_command("%s%s %d\n",s,in,count);
   else DDT_state_run_command("%s%s %d\n",s,in,count);


   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_stop -- handle stop					*/
/*									*/
/************************************************************************/


void
DDT_x_stop()
{
   String s;
   Integer line,addr;
   String file,rtn,args;
   Character fileb[256],funcb[256];
   LOCATION loc;
   Boolean fg;

   if (!DDT__use_gdb) stack_bottom();
   s = DDT_debug_command("where 1\n");

   addr = 0;

   if (s == NULL || STREQL(s,"No active stack frames")) {
      DDT_symbol_main_program(&loc,TRUE);
      if (loc.line > 0 || loc.func != NULL) fg = TRUE;
      else fg = FALSE;
    }
   else if (STREQL(s,"program is not active")) {
      DDT_symbol_main_program(&loc,FALSE);
      DDT_msg_stop_run(exit_status,exit_signal);
      if (loc.line > 0 || loc.func != NULL) fg = TRUE;
      else fg = FALSE;
    }
   else if (scan_where(s,&line,&file,&addr,&rtn,&args)) {
      if (file != NULL) {
	 strcpy(fileb,file);
	 file = DDT_symbol_fix_file_name(fileb);
       };
      if (rtn != NULL) {
	 strcpy(funcb,rtn);
	 rtn = funcb;
       };
      loc.file = file;
      loc.func = rtn;
      loc.line = line;
      if (line > 0) addr = -1;
      fg = TRUE;
    }
   else fg = FALSE;

   if (fg) {
      DDT_model_set_exec_location(&loc,addr);
      if (loc.line != 0) {
	 set_location(loc.file,loc.func,loc.line,FALSE,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	DDT_x_updown -- move up or down the stack			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_updown(delta)
   Integer delta;
{
   Boolean fg;
   String s;

   if (delta > 0) {
      s = DDT_debug_command("up %d\n",delta);
    }
   else if (delta < 0) {
      s = DDT_debug_command("down %d\n",-delta);
    };

   if (s == NULL) fg = TRUE;
   else if (DDT__use_gdb) {
      fg = (s[0] == '#');
    }
   else {
#ifndef mips
      fg = (s[0] == 'C');
#else
      fg = !prefix(s,"not that many levels");
#endif
    };

   if (!fg && s != NULL) DDT_mprint("%s\n",s);

   fg = report_location();

   return fg;
};





/************************************************************************/
/*									*/
/*	DDT_x_use -- set use string					*/
/*									*/
/************************************************************************/


Boolean
DDT_x_use(str)
   String str;
{
   Character buf[1024];
   String t;

#ifdef sun
#ifndef NEW_DBX
   { Character obuf[1024];

     if (!DDT__use_gdb) {
	sprintf(obuf,"use ");
	while (str != NULL && *str != 0) {
	   t = buf;
	   while (*str == ' ') ++str;
	   while (*str != ' ' && *str != 0) *t++ = *str++;
	   while (*str == ' ') ++str;
	   *t = 0;
	   strcat(obuf,buf);
	   strcat(obuf," ");
	   if (strlen(obuf) > 256) break;
	 };
	if (strlen(obuf) > 250) {
	   t = &obuf[250];
	   while (!isspace(*t)) --t;
	   *t = 0;
	 };
	DDT_debug_command("%s\n",obuf);
	return TRUE;
      };
   };
#endif
#endif

   if (DDT__use_gdb) DDT_debug_command("directory\n");
#ifndef mips
   else DDT_debug_command("use \\\n");
#else
   else DDT_printf("use \\\n");
#endif

   while (str != NULL && *str != 0) {
      t = buf;
      while (*str == ' ') ++str;
      while (*str != ' ' && *str != 0) *t++ = *str++;
      while (*str == ' ') ++str;
      *t = 0;
      if (DDT__use_gdb) {
	 if (buf[0] != 0 && STRNEQ(buf,".")) DDT_debug_command("directory %s\n",buf);
       }
#ifndef mips
     else DDT_debug_command("%s \\\n",buf);
#else
     else DDT_printf("%s \\\n",buf);
#endif
    };

   if (!DDT__use_gdb) DDT_debug_command("\n");

   return TRUE;
};




/************************************************************************/
/*									*/
/*	DDT_x_valid_name -- check if name is valid			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_valid_name(txt)
   String txt;
{
   String s;

   s = DDT_debug_command("whatis %s\n",txt);
   if (s == NULL || s[0] == '"' || prefix(s,"no module") ||
	  prefix(s,"Invalid token") || prefix(s,"No symbol")) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_var_info -- get info on given variable			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_var_info(file,func,line,name,where,which,what,buf)
   String file;
   String func;
   Integer line;
   String name;
   Boolean where;
   Boolean which;
   Boolean what;
   String buf;
{
   String s,t;
   LOCATION oloc;
   Character xbuf[1024],ybuf[1024],zbuf[1024];

   DDT_model_inq_location(&oloc);

   if (buf != NULL) buf[0] = 0;

   if (!set_location(file,func,line,TRUE,FALSE)) return FALSE;

   name = DDT_expr_parse(name,xbuf);
   if (name != xbuf) {
      strcpy(xbuf,name);
      name = xbuf;
    };
   name = fix_name_separator(name);

   if (where) { if (DDT__use_gdb) {
	 sprintf(ybuf,"variable: `%s",name);
	 s = ybuf;
       }
      else {
	 s = DDT_debug_command("whereis %s\n",name);
	 s = unfix_name_separator(s);
       };
      if (buf == NULL) {
	 s = DDT_map_fix_outexpr(s,TRUE,zbuf);
	 DDT_mprint("%s\n",s);
       }
      else strcat(buf,s);
    };

   if (which) {
      if (DDT__use_gdb) {
	 sprintf(ybuf,"`%s",name);
	 s = ybuf;
       }
      else {
	 s = DDT_debug_command("which %s\n",name);
       };
      s = unfix_name_separator(s);
      if (buf == NULL) {
	 s = DDT_map_fix_outexpr(s,TRUE,zbuf);
	 DDT_mprint("%s\n",s);
       }
      else {
	 if (buf[0] != 0) strcat(buf,"\n");
	 strcat(buf,s);
       };
    };

   if (what) {
      s = DDT_debug_command("whatis (%s)\n",name);
      if (DDT__use_gdb) {
	 if (STREQL(s,"Invalid syntax in expression."))
	    s = DDT_debug_command("ptype %s\n",name);
       }
      else {
#ifndef mips
	 if (prefix(s,"syntax error on"))
	    s = DDT_debug_command("whatis %s\n",name);
#else
	 if (prefix(s,"whatis (") && index(s,'^') != 0)
	    s = DDT_debug_command("whatis %s\n",name);
#endif
	 if (s[0] == '"' && name[0] == '*' && name[1] == ' ' && name[2] == '`') {
	    t = index(&s[1],'"');
	    if (STREQL(t,"\" is not a pointer")) {
	       s = DDT_debug_command("whatis (%s)\n",&name[1]);
	     };
	  };
       };
      if (buf == NULL) {
	 s = unfix_name_separator(s);
	 s = DDT_map_fix_outexpr(s,TRUE,zbuf);
	 DDT_mprint("%s\n",s);
       }
      else {
	 if (s[0] == '"') s = "";
	 if (buf[0] != 0) strcat(buf,"\n");
	 strcat(buf,s);
       };
    };

   if (file != NULL || func != NULL || line > 0) {
      set_location(oloc.file,oloc.func,oloc.line,FALSE,TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DDT_x_where -- set up the current stack 			*/
/*									*/
/************************************************************************/


Boolean
DDT_x_where(dumpfg,glblfg)
   Boolean dumpfg;
   Boolean glblfg;
{
   String s,t;
   String  rtn,args,file;
   Integer line,addr,lvl;
   Character rslt[20480];
   Integer i;

   s = DDT_debug_command("where\n");

   lvl = 1;

   while (s != NULL) {
      t = s;
      s = index(s,'\n');
      if (s != NULL) *s++ = 0;
      if (!scan_where(t,&line,&file,&addr,&rtn,&args)) break;
      args = DDT_map_fix_outexpr(args,TRUE,rslt);
      DDT_trace_where(lvl,file,rtn,line,addr,args);
      lvl++;
    };

   if (dumpfg && glblfg) {
      DDT_trace_where(lvl,"*","*",0,0,"*");
      DDT_trace_where_top(lvl);
     }
   else {
      DDT_trace_where_top(lvl-1);
    };

   if (dumpfg) {
      stack_bottom();
      for (i = 1; i < lvl; ++i) {
	 if (DDT__use_gdb) s = DDT_debug_command("dumpvars\n");
	 else {
	    s = DDT_debug_command("dump\n");
#ifdef mips
	    s = index(s,'(');
	    if (s == NULL) break;
	    ++s;
	    if (*s == ')') {
	       s = index(s,'\n');
	       if (s == NULL) break;
	       ++s;
	     };
#endif
	  };

	 while (s != NULL) {
	    s = scan_dump_var(i,s);
	  };

	 s = DDT_debug_command("up\n");
       };
      if (glblfg) DDT_symbol_dump_globals(lvl);
      stack_bottom();
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	set_location -- change current location 			*/
/*	set_line_loc -- change current line only			*/
/*									*/
/************************************************************************/


static Boolean
set_location(file,func,line,errfg,force)
   String file;
   String func;
   Integer line;
   Boolean errfg;
{
   String s,t;
   Boolean fg;
   LOCATION oloc,mloc;

   DDT_model_inq_location(&oloc);

   if (file != NULL && STRNEQ(file,"*")) {
      t = DDT_symbol_fix_file_name(file);
      if (DDT__use_gdb) {
	 s = DDT_debug_command("list %s:1,1\n",t);
	 if (s == NULL) s = "Bad source file";
	 if (s[0] == '1') s = NULL;
       }
      else {
	 s = DDT_debug_command("file %s\n",t);
	 if (s != NULL && prefix(s,"warning: ")) {
	    if (errfg) DDT_mprint("%s\n",s);
	    s = NULL;
	  };
       };
      if (s != NULL) {
	 if (errfg) DDT_mprint("%s\n",s);
	 return FALSE;
       };
    };

   fg = TRUE;

   if (func != NULL && STRNEQ(func,"*")) {
      s = set_function(func);
      if (s != NULL) {
	 if (errfg) {
	    if (index(func,':') != NULL && prefix(s,"no module, procedure or file")) {
	       DDT_mprint("%s is not a valid function name\n",func);
	     }
	    else DDT_mprint("%s\n",s);
	  };
	 if (!prefix(s,"warning:")) fg = FALSE;
       };
      if (!fg && force && index(func,'.') != NULL) {
	 DDT_symbol_main_program(&mloc,TRUE);
	 s = set_function(mloc.func);
	 line = mloc.line;
       };
    };

   if (fg && line > 0) {
      fg = set_line_loc(line);
      if (!fg && errfg) DDT_mprint("Bad line number %d\n",line);
    };

   if (!fg && !force) {
      set_location(oloc.file,oloc.func,oloc.line,FALSE,TRUE);
    };

   return fg;
};





static Boolean
set_line_loc(line)
   Integer line;
{
   Boolean fg;
   String s;

   if (line <= 0) line = 1;

   if (DDT__use_gdb) {
      s = DDT_debug_command("list %d,+0\n",line);
    }
   else if (line == 1) {
      s = DDT_debug_command("list 1; ?.?\n");
    }
   else {
      s = DDT_debug_command("list %d\n",line-1);
    };

   if (s == NULL || (s[0] != ' ' && !isdigit(s[0]))) fg = FALSE;
   else fg = TRUE;

   return fg;
};






/************************************************************************/
/*									*/
/*	report_location -- current location changed			*/
/*	inq_file -- return current file 				*/
/*									*/
/************************************************************************/


static Boolean
report_location()
{
   String s,t;
   LOCATION loc,cloc;
   Universal args[2];
   Boolean fg;
   Integer j;

   DDT_model_inq_location(&cloc);

   if (DDT__use_gdb) {
      s = DDT_debug_command("info line\n");
      if (STREQL(s,"No source file specified.")) {
	 DDT_debug_command("list -\n");
	 s = DDT_debug_command("info line\n");
       };
      if (prefix(s,"Line number ")) s += 12;
      else if (prefix(s,"Line ")) s += 5;
      else return FALSE;
      loc.line = atol(s);
      if (loc.line == 0) loc.line = 1;
      loc.file = NULL;
      loc.func = NULL;

      s = index(s,'"');
      if (s != NULL) {
	 ++s;
	 t = s;
	 s = index(s,'"');
	 if (s != NULL) *s = 0;
	 loc.file = SALLOC(t);
       };

      if (loc.file != NULL && loc.line > 0 && loc.func == NULL) {
	 loc.func = DDT_symbol_fct_from_file(loc.file,loc.line);
       };
    }
   else {
      loc.file = DDT_debug_command("file\n");
      if (loc.file != NULL &&
	     STREQL(loc.file,"No current source file")) return FALSE;
      if (loc.file != NULL) loc.file = DDT_symbol_fix_file_name(loc.file);
      if (loc.file != NULL) loc.file = SALLOC(loc.file);

      if (cloc.file != NULL && loc.file != NULL) fg = STREQL(loc.file,cloc.file);
      else fg = FALSE;

      loc.func = inq_function();

      if (loc.func != NULL) loc.func = SALLOC(loc.func);

      if (fg && cloc.func != NULL && loc.func != NULL)
	 fg = STREQL(loc.func,cloc.func);
      else fg = FALSE;

      s = DDT_debug_command("list\n");
      if (s != NULL && PMATmatch(s,eof_pattern,args) >= 0) {
	 loc.line = args[0];
#ifdef mips
	 loc.line -= 1;
#endif
       }
      else if (s != NULL && (j = get_line_number(s)) > 0) {
	 loc.line = j;
	 set_line_loc(loc.line);
       }
#ifdef sun
      else if (s != NULL && STREQL(s,"warning: file is not compiled with the -g option")) {
	 loc.file = NULL;
	 loc.line = 0;
       }
#endif
      else if (fg) loc.line = cloc.line;
      else loc.line = 0;
    };

   DDT_model_set_location(&loc);

   return TRUE;
};





static String
inq_file()
{
   String s,t;

   if (DDT__use_gdb) {
      s = DDT_debug_command("info line\n");
      if (prefix(s,"Line ")) {
	 s = index(s,'"');
	 if (s != NULL) {
	    ++s;
	    t = index(s,'"');
	    if (t != NULL) *t = 0;
	  };
       };
    }
   else s = DDT_debug_command("file\n");

   if (s == NULL) s = "";
   else s = DDT_symbol_fix_file_name(s);

   return SALLOC(s);
};





/************************************************************************/
/*									*/
/*	filetail -- return filename for dbx symbol usage		*/
/*									*/
/************************************************************************/


static String
filetail(s)
   String s;
{
   static Character buf[64];
   String p;

   p = rindex(s,'/');
   if (p == NULL) p = s;
   else ++p;

   strcpy(buf,p);
   p = rindex(buf,'.');
   if (p != NULL && STREQL(&p[-1],"..c")) --p;
   if (p != NULL) *p = 0;

   return buf;
};





/************************************************************************/
/*									*/
/*	manage_event -- handle an event found in debugger text		*/
/*									*/
/************************************************************************/


static void
manage_event(ev,args)
   Integer ev;
   Universal args[EVENT_ARGS+1];
{
   LOCATION loc;
   Integer i;
   String file,fct;
   RUN_PAT * evts;
   Integer addr;

   if (args[EV_IDX_FILE] != NULL) file = DDT_symbol_fix_file_name((String) args[EV_IDX_FILE]);
   else file = NULL;
   if (args[EV_IDX_FUNCTION] != NULL) fct = DDT_map_unfix_function(args[EV_IDX_FUNCTION]);
   else fct = NULL;

   evts = (DDT__use_gdb ? gdb_run_events : run_events);

   if (args[EV_IDX_LINE] > 0) {
      loc.file = file;
      loc.func = fct;
      loc.line = (Integer) args[EV_IDX_LINE];
      addr = (Integer) args[EV_IDX_ADDRESS];
      DDT_model_set_exec_location(&loc,addr);
    }
   else if (args[EV_IDX_ADDRESS] != 0 && ((int) args[EV_IDX_ADDRESS]) != -1) {
      loc.file = file;
      loc.func = fct;
      loc.line = 0;
      addr = (Integer) args[EV_IDX_ADDRESS];
      DDT_model_set_exec_location(&loc,addr);
    }
   else if (evts[ev].exit) {
      loc.file = NULL;
      loc.func = NULL;
      loc.line = 0;
      DDT_symbol_main_program(&loc,FALSE);
      if (loc.func != NULL) {
	 file = loc.file;
	 DDT_model_set_exec_location(&loc,0);
	 set_location(loc.file,loc.func,loc.line,FALSE,FALSE);
       };
      exit_status = args[EV_IDX_SIGNAL];
      exit_signal = NULL;
      if (exit_status > 256) {
	 exit_signal = (String) args[EV_IDX_SIGNAL];
	 i = find_signal(exit_signal);
	 DDT_mprint("Program terminated by signal %s (%s)\n",
		       DDT__signal_table[i].name,
		       DDT__signal_table[i].desc);
	 exit_status = i<<8;
       };
      args[EV_IDX_SIGNAL] = 0;
    };

   if (args[EV_IDX_SIGNAL] > 0 || evts[ev].sig > 0) {
      i = evts[ev].sig;
      if (i <= 0) {
	 i = find_signal(args[EV_IDX_SIGNAL]);
       };
      if (DDT__signal_table[i].flags == SIGNAL_PRINTIGN ||
	     DDT__signal_table[i].flags == SIGNAL_PRINTUSE) {
	 if (!evts[ev].printit) {
	    DDT_mprint("Signal %s (%s) ",DDT__signal_table[i].name,DDT__signal_table[i].desc);
	    DDT_print_exec_position();
	    DDT_mprint("\n");
	  };
       };
      if (DDT__signal_table[i].flags == SIGNAL_PRINTIGN) {
	 DDT_state_stop_action(STOP_CONTINUE);
       }
      else if (DDT__signal_table[i].flags == SIGNAL_PRINTUSE) {
	 DDT_state_stop_signal(i);
       }
      else {
	 DDT_model_inq_exec_location(&loc,&addr);
	 if (i > 0) DDT_msg_stop_error(&loc,i);
	 DDT_state_stop_action(STOP_STOP);
       };
    }
   else if (evts[ev].ignore) ;
   else {
      DDT_event_handle(file,fct,
			  args[EV_IDX_LINE],
			  args[EV_IDX_ADDRESS],
			  args[EV_IDX_EXPRESSION],
			  args[EV_IDX_NEW],
			  args[EV_IDX_ID],
			  evts[ev].trace,
			  evts[ev].call);
    };

   if (evts[ev].forcestop) DDT_state_stop_action(STOP_STOP);
};





/************************************************************************/
/*									*/
/*	trace_all_calls -- setup to trace all calls			*/
/*	event_all_calls -- add an event for all calls			*/
/*									*/
/************************************************************************/


static void
trace_all_calls(itr,etr)
   Boolean itr;
   Boolean etr;
{
   Character buf[1024];
   String s;
   Integer id0,id1,i,ct,id;
   String fcts[10240];

   id0 = 0;

   ct = DDT_symbol_functions(10240,fcts,TRUE);

   for (i = 0; i < ct; ++i) {
      if (STREQL(fcts[i],"main") || STREQL(fcts[i],"program")) continue;
      if (DDT__use_gdb) {
	 id = gdb_call_trace(NULL,fcts[i]);
	 if (id0 == 0) {
	    if (id < 0) id0 = (-id)>>16;
	    else id0 = id;
	  };
	 if (id < 0) id1 = (-id) & 0xffff;
	 else id1 = id;
       }
      else {
	 sprintf(buf,"trace %s",fcts[i]);
	 s = DDT_debug_command("%s\n",buf);

	 if (s[0] == '(') {
	    sscanf(s,"(%d)",&id1);
	    if (id0 == 0) id0 = id1;
	  };
       };
    };

   if (id0 == 0) return;

   id0 = (id0<<16) + id1;
   id0 = -id0;

   DDT_event_add(EVENT_CALL,NULL,NULL,NULL,0,NULL,NULL,0,id0,itr,etr);
};




static void
event_all_calls(evt,expr,cnd,name,itr,etr)
   EVENT_ACTION evt;
   String expr;
   String cnd;
   String name;
   Boolean itr;
   Boolean etr;
{
   Integer i,ct;
   String fcts[10240];
   String file;
   Integer line,eline;

   ct = DDT_symbol_functions(10240,fcts,TRUE);

   for (i = 0; i < ct; ++i) {
      if (STREQL(fcts[i],"main") || STREQL(fcts[i],"program")) continue;
      if (!DDT_symbol_funct_info(fcts[i],&file,&line,&eline)) continue;
      if (evt == EVENT_STOPEXIT) line = eline;
      if (line == 0) continue;

      DDT_x_add_event(evt,file,fcts[i],line,expr,cnd,0,name,itr,etr);
    };
};




/************************************************************************/
/*									*/
/*	scan_where -- scan a WHERE line 				*/
/*									*/
/************************************************************************/


static Boolean
scan_where(s,linep,filep,addrp,rtnp,argp)
   String s;
   Integer * linep;
   String * filep;
   Integer * addrp;
   String * rtnp;
   String * argp;
{
   String t;
   Integer fg;
   Universal marg[2];

   t = s;
   s = index(s,'\n');
   if (s != NULL) *s++ = 0;

   if (DDT__use_gdb) {
      fg = scan_where_gdb(t,linep,filep,addrp,rtnp,argp);
      if (!fg) return FALSE;
    }
   else {
#ifdef mips
      if (*t == '>') ++t;
      while (isspace(*t)) ++t;
      while (isdigit(*t)) ++t;
      while (isspace(*t)) ++t;
#endif

      *rtnp = t;
      t = index(t,'(');
      if (t == NULL) return FALSE;
      *t++ = 0;
      *argp = t;
      fg = 0;
      t = rindex(t,')');
      if (t == NULL) return FALSE;
#ifndef mips
      if (prefix(t,"), line")) fg = 1;
      else if (prefix(t,") at ")) fg = 2;
      else if (t[1] == 0) fg = 3;
      else ++t;
#else
      if (prefix(t,") [\"")) fg = 1;
      else if (prefix(t,") [")) fg = 2;
      else if (t[1] == 0) fg = 3;
      else ++t;
#endif
      if (fg == 0) return FALSE;
      *t++ = 0;
      if (fg == 1 && PMATmatch(t,where_file_pat,marg) >= 0) {
	 *linep = (Integer) marg[0];
	 *filep = (String) marg[1];
	 *addrp = 0;
       }
      else if (fg == 3) {
	 *linep = 0;
	 *addrp = 0;
	 *filep = NULL;
       }
      else if (PMATmatch(t,where_addr_pat,marg) >= 0) {
	 *addrp = (Integer) marg[0];
	 *linep = 0;
	 *filep = NULL;
       }
      else return FALSE;
    };

   *rtnp = DDT_map_unfix_function(*rtnp);

   return TRUE;
};





static Boolean
scan_where_gdb(s,linep,filep,addrp,rtnp,argp)
   String s;
   Integer * linep;
   String * filep;
   Integer * addrp;
   String * rtnp;
   String * argp;
{
   String t;

   *addrp = 0;
   *linep = 0;
   *filep = NULL;

   t = s;
   if (*t != '#') return FALSE;
   ++t;
   while (isdigit(*t)) ++t;
   while (isspace(*t)) ++t;
   if (*t == '0' && t[1] == 'x') {
      t += 2;
      while (!isspace(*t) && *t != 0) {
	 if (isdigit(*t)) *addrp = *addrp * 16 + *t - '0';
	 else if (islower(*t)) *addrp = *addrp * 16 + *t - 'a' + 10;
	 ++t;
       };
      if (prefix(t," in ")) t += 4;
      else return FALSE;
    };

   *rtnp = t;
   t = index(t,' ');
   if (t == NULL) return FALSE;
   *t++ = 0;

   ++t;
   *argp = t;

   while (*t != 0) {
      if (t[0] == ')' && ((t[1] == ' ' && t[2] == '(') || t[1] == 0)) break;
      ++t;
    };

   if (*t == 0) return FALSE;

   *t++ = 0;
   while (isspace(*t)) ++t;

   if (*t == '(') {
      ++t;
      *filep = t;
      while (!isspace(*t) && *t != 0) ++t;
      if (*t != 0) *t++ = 0;
      if (prefix(t,"line ")) {
	 t += 5;
	 *linep = atol(t);
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	prefix -- check for a valid prefix				*/
/*									*/
/************************************************************************/


static Boolean
prefix(s,pfx)
   String s;
   String pfx;
{
   Integer ln;

   ln = strlen(pfx);

   return (strncmp(s,pfx,ln) == 0);
};





/************************************************************************/
/*									*/
/*	set_function -- set current function				*/
/*	inq_function -- return current function 			*/
/*									*/
/************************************************************************/


static String
set_function(f)
   String f;
{
   String s;
   Character buf[1024];
   Integer i;		/* used in mips code */

   f = DDT_map_fix_function(f,buf);
   if (f != buf) {
      strcpy(buf,f);
      f = buf;
    };
   f = fix_name_separator(f);

   if (DDT__use_gdb) {
      s = index(f,':');
      if (s != NULL && s[1] == ':') s[1] = ' ';
      s = DDT_debug_command("list %s,+0\n",f);
      if (s != NULL && isdigit(s[0])) s = NULL;
    }
   else {
      s = DDT_debug_command("func %s\n",f);
#ifdef mips
      i = strlen(f);
      if (s != NULL && prefix(s,f) && s[i] == ':') s = NULL;
#endif
    };

   return s;
};






static String
inq_function()
{
   String s;
   String file,rtn,arg;
   Integer line,addr;

   if (DDT__use_gdb) {
      s = DDT_debug_command("frame\n");
      if (!scan_where_gdb(s,&line,&file,&addr,&rtn,&arg)) s = NULL;
      s = rtn;
    }
   else {
      s = DDT_debug_command("func\n");
    };

   if (s != NULL) s = DDT_map_unfix_function(s);

   return s;
};





/************************************************************************/
/*									*/
/*	get_line_number -- get line number for list output		*/
/*									*/
/************************************************************************/


static Integer
get_line_number(txt)
   String txt;
{
   Integer v;

   if (prefix(txt,"warning: File")) {
      txt = index(txt,'\n');
      if (txt == NULL) return 0;
      ++txt;
    };

#ifdef mips
   if (txt[0] == ' ' && txt[1] == '*') txt += 2;
#endif

   while (isspace(*txt)) ++txt;
   if (!isdigit(*txt)) return 0;

   v = 0;
   while (isdigit(*txt)) v = v*10 + (*txt++) - '0';
   if (!isspace(*txt)) v = 0;

   return v;
};





/************************************************************************/
/*									*/
/*	find_signal -- find signal given name				*/
/*									*/
/************************************************************************/


static Integer
find_signal(s)
   String s;
{
   Integer i;
   Character buf[256];

   strcpy(buf,s);
   i = strlen(buf);
   while (i > 0 && buf[i-1] == ' ') --i;
   buf[i] = 0;
   if (i == 0) buf[1] = 0;

   for (i = 1; DDT__signal_table[i].name != NULL; ++i) {
      if (STREQL(DDT__signal_table[i].name,buf)) break;
      if (STREQL(&DDT__signal_table[i].desc[1],&buf[1]) &&
	     (DDT__signal_table[i].desc[0] == buf[0] ||
		 DDT__signal_table[i].desc[0] == tolower(buf[0])))
	 break;
    };

   if (DDT__signal_table[i].name == NULL) {
      if (isdigit(buf[0])) i = atol(buf);
    };

   return i;
};





/************************************************************************/
/*									*/
/*	fix_name_separator -- convert name separator if necessary	*/
/*	unfix_name_separator -- convert name for us from dbx		*/
/*									*/
/************************************************************************/


static String
fix_name_separator(s)
   String s;
{
   String t;
   String b;
   Integer ln;

   if (DDT__use_gdb) {
      if (s[0] == '`') ++s;
      if (s != NULL && index(s,'`') != NULL) {
	 ln = strlen(s);
	 b = (String) alloca(ln*4/3+10);
	 strcpy(b,s);
	 t = s;
	 for ( ; *b != 0; ++b) {
	    if (*b == '`') {
	       if (b[-1] != ' ') {
		  *t++ = ':';
		  *t++ = ':';
		};
	     }
	    else *t++ = *b;
	  };
	 *t = 0;
       };
    }
   else if (s != NULL && NAME_SEP_CHAR != '`') {
      while ((t = index(s,'`')) != NULL) *t = NAME_SEP_CHAR;
    };

   return s;
};






static String
unfix_name_separator(s)
   String s;
{
   String t,p;

   if (DDT__use_gdb) {
      if (s != NULL) {
	 p = s;
	 for (t = s; *t != 0; ++t) {
	    if (*t == ':' && t[1] == ':') {
	       *p++ = '`';
	       ++t;
	     }
	    else *p++ = *t;
	  };
	 *p = 0;
       };
    }
   else if (s != NULL && NAME_SEP_CHAR != '`') {
      while ((t = index(s,NAME_SEP_CHAR)) != NULL) *t = '`';
    };

   return s;
};






/************************************************************************/
/*									*/
/*	stack_bottom -- go to stack bottom				*/
/*									*/
/************************************************************************/


static void
stack_bottom()
{
   String s;

   if (DDT__use_gdb) {
      s = DDT_debug_command("frame 0\n");
    }
   else {
#ifndef mips
      s = DDT_debug_command("down 1000000\n");
#else
      for ( ; ; ) {
	 s = DDT_debug_command("down\n");
	 if (STREQL(s,"not that many levels")) break;
	 if (STREQL(s,"program is not active")) break;
       };
#endif
    };
};






/************************************************************************/
/*									*/
/*	scan_dump_var -- scan variable from dump information		*/
/*									*/
/************************************************************************/


static String
scan_dump_var(lvl,s)
   Integer lvl;
   String s;
{
   String t,rtn;
   Integer qt,ct,ch;

   if (s == NULL) return NULL;

   t = s;
   rtn = t;

   qt = 0;
   ct = 0;
   while (*s != 0) {
      if (*s == '\n') break;
      if (*s == qt) qt = 0;
      else if (qt == 0) {
	 if (*s == '"' || *s == '\'') qt = *s;
	 if (*s == '(') ++ct;
	 else if (*s == ',') break;
	 else if (*s == ')' && ct > 0) --ct;
	 else if (*s == ')') break;
       };
      ++s;
    };

   ch = *s;
   if (*s != 0) *s++ = 0;
   if (ch == ')') s = index(s,'\n');
   if (s != NULL) while (isspace(*s)) ++s;
   if (s != NULL && *s == 0) s = NULL;

   while (*t != ' ' && *t != 0) ++t;
   if (*t == ' ') *t++ = 0;
   if (*t++ == '=' && *t++ == ' ') {
      rtn = DDT_map_unfix_variable(rtn);
      DDT_trace_dump(lvl,rtn,t);
    };

   return s;
};





/************************************************************************/
/*									*/
/*	gdb_call_trace -- do a call trace for given function		*/
/*									*/
/************************************************************************/


static Integer
gdb_call_trace(fil,fct)
   String fil;
   String fct;
{
   Integer id0,id1;
   Character ftxt[1024],ltxt[1024];
   String s,file;
   Integer line,eline;

   if (fil == NULL) strcpy(ftxt,fct);
   else sprintf(ftxt,"%s:%s",filetail(fil),fct);

   s = DDT_debug_command("break %s\n",ftxt);
   if (sscanf(s,"Breakpoint %d",&id0) != 1) return 0;

   if (DDT_symbol_funct_info(fct,&file,&line,&eline)) {
      if (fil == NULL) fil = file;
      if (fil == NULL) sprintf(ltxt,"%d",eline);
      else sprintf(ltxt,"%s:%d",filetail(fil),eline);
      s = DDT_debug_command("info line %s\n",ltxt);
      if (prefix(s,"Line ")) {
	 while (*s != 0 && !prefix(s,"ends at ")) ++s;
	 if (*s != 0 && sscanf(s,"ends at 0x%x",&eline) == 1) {
#ifdef sparc
	    sprintf(ltxt,"* 0x%x",eline-4);
#endif
#ifdef mips
	    sprintf(ltxt,"* 0x%x",eline-8);
#endif
	  };
       };
      s = DDT_debug_command("break %s\n",ltxt);
      if (sscanf(s,"Breakpoint %d",&id1) != 1) id1 = 0;
    };

   DDT_debug_command("commands %d\nsilent\nprintf \"Call: \"\nwhere 1\ncont\nend\n",
			id0);

   if (id1 != 0) {
      DDT_debug_command("commands %d\nsilent\nprintf \"Return from %s\\n\"\ncont\nend\n",
			id1,fct);
      id0 = (id0 << 16) + id1;
      id0 = -id0;
    };

   return id0;
};





/* end of ddtdbx.c */
