/************************************************************************/
/*									*/
/*		dbgbtn.c					       */
/*									*/
/*	Button handling routines for the debugger interface		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "dbg_local.h"
#include <sgtty.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


	Integer 	DBG__button_line_size;

static	DBG_BTN 	dflt_btns;
static	DBG_BTN 	user_btns;
static	Integer 	button_font;
static	Integer 	interrupt;



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	menu_add_btn();
static	Boolean 	menu_remove_btn();
static	Boolean 	menu_modify_btn();
static	int		dbg_btn_handler();
static	DBG_BTN 	add_button();
static	void		draw_buttons();
static	void		fix_output();
static	void		reset_buttons();
static	Boolean 	save_buttons();





/************************************************************************/
/*									*/
/*	DBG_btn_init -- module initialization				*/
/*									*/
/************************************************************************/


void
DBG_btn_init()
{
   AUXD db;
   String nm,out,col,s;
   struct  tchars tc;
   Integer tty;
   Integer x,y;
   Character buf[10240];
   FILE * inf;
   DBG_BTN_MODE mode;

   tty = open("/dev/tty",2);
   if (tty < 0) tty = 1;

   if (ioctl(tty, TIOCGETC, (char *)&tc) >= 0) {
      interrupt = tc.t_intrc;
    }
   else tc.t_intrc = '\3';
   if (tty > 1) close(tty);

   db = AUXDget_handle(NULL,"DBG");
   nm = AUXDget_info(db,"BUTTON_FONT");
   if (nm == NULL) nm = ASHinq_resource("dbg.font");
   if (nm == NULL) button_font = ASH_DEFAULT_FONT;
   else button_font = ASHloadfont(nm);

   STEMbtn_size(button_font,"_Xp",&x,&y);
   DBG__button_line_size = y+4;

   dflt_btns = NULL;
   for (db = AUXDget_handle(db,"BUTTON"); db != NULL; db = AUXDnext_handle(db)) {
      nm = AUXDget_info(db,"NAME");
      if (nm == NULL) continue;
      out = AUXDget_info(db,"OUTPUT");
      if (out == NULL) continue;
      col = AUXDget_info(db,"COLOR");
      if (AUXDget_defined(db,"RUN")) mode = DBG_MODE_RUN;
      else if (AUXDget_defined(db,"CONTINUE")) mode = DBG_MODE_CONTINUE;
      else mode = DBG_MODE_NONE;
      dflt_btns = add_button(NULL,dflt_btns,nm,out,col,mode,NULL,TRUE);
    };

   sprintf(buf,"%s",BUTTON_FILE_NAME);
   if (access(buf,4) < 0) sprintf(buf,"%s/%s",getenv("HOME"),BUTTON_FILE_NAME);
   inf = fopen(buf,"r");
   if (inf != NULL) {
      user_btns = NULL;
      while (fgets(buf,10240,inf) != NULL) {
	 x = strlen(buf);
	 if (buf[x-1] == '\n') buf[x-1] = 0;
	 nm = buf;
	 out = index(nm,BUTTON_FILE_SEP);
	 if (out == NULL) continue;
	 *out++ = 0;
	 col = index(out,BUTTON_FILE_SEP);
	 if (col != NULL) {
	    *col++ = 0;
	    if (*col == 0) col = NULL;
	  };
	 mode = DBG_MODE_NONE;
	 if (col == NULL) s = NULL;
	 else s = index(col,BUTTON_FILE_SEP);
	 if (s != NULL) {
	    *s++ = 0;
	    mode = (DBG_BTN_MODE) atol(s);
	  };
	 if (col != NULL && *col == 0) col = NULL;
	 user_btns = add_button(NULL,user_btns,nm,out,col,mode,NULL,TRUE);
       };
      fclose(inf);
    }
   else user_btns = dflt_btns;
};




/************************************************************************/
/*									*/
/*	DBG_btn_setup -- setup buttons for window			*/
/*									*/
/************************************************************************/


void
DBG_btn_setup(dw)
   DBG_WIN dw;
{
   DBG_BTN db;

   dw->buttons = NULL;

   for (db = user_btns; db != NULL; db = db->next) {
      dw->buttons = add_button(dw,dw->buttons,db->name,db->output,db->color,
				  db->mode,NULL,TRUE);
    };

   ASHset_user_data(dw->btn_win,dw);
   ASHfont(dw->btn_win,button_font);
   STEMpanel_define(dw->btn_win,NULL,dbg_btn_handler,dw,0);

   BIOnew_input_window(dw->btn_win);
   BIOset_cursor_standard(dw->btn_win,ASH_CURSOR_X);

   draw_buttons(dw);
};





/************************************************************************/
/*									*/
/*	DBG_btn_menu -- handle hits on button menu			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
DBG_btn_menu(dw,mnm,btn)
   DBG_WIN dw;
   String mnm;
   String btn;
{
   register Boolean fg;

   fg = TRUE;

   if (STREQL(btn,"Add")) {
      fg = menu_add_btn(dw);
    }
   else if (STREQL(btn,"Remove")) {
      fg = menu_remove_btn(dw);
    }
   else if (STREQL(btn,"Modify")) {
      fg = menu_modify_btn(dw);
    }
   else if (STREQL(btn,"Reload")) {
      reset_buttons(dw,user_btns);
    }
   else if (STREQL(btn,"Default")) {
      reset_buttons(dw,dflt_btns);
    }
   else if (STREQL(btn,"Save Local")) {
      fg = save_buttons(dw,".");
    }
   else if (STREQL(btn,"Save Global")) {
      fg = save_buttons(dw,getenv("HOME"));
    }
   else fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	DBG_btn_mode -- handle mode set for buttons			*/
/*									*/
/************************************************************************/


void
DBG_btn_mode(dw,mode)
   DBG_WIN dw;
   DBG_BTN_MODE mode;
{
   DBG_BTN db;

   if (mode == DBG_MODE_CONTINUE) {
      for (db = dw->buttons; db != NULL; db = db->next) {
	 if (db->mode == DBG_MODE_CONTINUE) break;
       };
      if (db == NULL) mode = DBG_MODE_RUN;
    };


   for (db = dw->buttons; db != NULL; db = db->next) {
      if (db->mode != DBG_MODE_NONE) {
	 STEMpanel_btn_select(dw->btn_win,db->name,(db->mode == mode));
       };
    };
};





/************************************************************************/
/*									*/
/*	menu_add_btn -- handle ADD button on menu			*/
/*									*/
/************************************************************************/


static Boolean
menu_add_btn(dw)
   DBG_WIN dw;
{
   String menu[16];
   Integer ct;
   Character nm[32],out[72],af[64],col[32];
   String co;
   Integer pos;
   Integer mode;
   DBG_BTN btn;

   ct = 0;
   menu[ct++] = "%CNew Button Definition";

   menu[ct++] = "Name:   %0.16t";
   menu[ct++] = "Output: %1.48t";
   if (ASHinq_configuration_depth(dw->window) > 1) menu[ct++] = "Color: %4.32t";
   menu[ct++] = "\n%5.0o No highlighting";
   menu[ct++] = "%5.2o Highlight during run";
   menu[ct++] = "%5.3o Highlight during continue";
   menu[ct++] = "\nPosition:";
   menu[ct++] = "%2.1o First%M%2.2o Last";
   menu[ct++] = "%2.0o After button %3.32t";
   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   nm[0] = 0;
   out[0] = 0;
   af[0] = 0;
   pos = 2;
   col[0] = 0;
   mode = 0;

   if (!STEMdialog(dw->window,menu,nm,out,&pos,af,col,&mode)) return FALSE;

   if (nm[0] == 0) return FALSE;

   if (pos == 0 && af[0] == 0) pos = 1;

   if (af[0] != 0) {
      for (btn = dw->buttons; btn != NULL; btn = btn->next) {
	 if (STREQL(btn->name,af)) break;
       };
      if (btn == NULL) pos = 2;
    }
   else btn = NULL;

   if (btn != NULL && pos != 0) pos = 0;
   else if (btn == NULL && pos == 0) pos = 2;

   co = (col[0] == 0 ? NULL : col);

   add_button(dw,dw->buttons,nm,out,co,mode,btn,(pos == 2));

   draw_buttons(dw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	menu_remove_btn -- handle REMOVE menu option			*/
/*									*/
/************************************************************************/


static Boolean
menu_remove_btn(dw)
   DBG_WIN dw;
{
   String menu[64];
   Character buf[48][48];
   Integer ct,bct,bfct;
   DBG_BTN btn,lbtn;
   Integer bnm;

   ct = 0;
   menu[ct++] = "%CRemove Button\n";

   bct = 0;
   bfct = 0;
   for (btn = dw->buttons; btn != NULL; btn = btn->next) {
      if (btn->next == NULL) {
	 sprintf(buf[bfct],"%%0.%do %s",bct,btn->name);
	 bct++;
       }
      else {
	 sprintf(buf[bfct],"%%0.%do %s%%M%%0.%do %s",
		    bct,btn->name,bct+1,btn->next->name);
	 btn = btn->next;
	 bct += 2;
       };
      menu[ct++] = buf[bfct++];
    };

   if (bct == 0) return FALSE;

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   bnm = -1;

   if (!STEMdialog(dw->window,menu,&bnm)) return FALSE;
   if (bnm < 0) return FALSE;

   bct = 0;
   lbtn = NULL;
   for (btn = dw->buttons; btn != NULL; btn = btn->next) {
      if (bct++ == bnm) break;
      lbtn = btn;
    };

   if (btn != NULL) {
      if (lbtn == NULL) dw->buttons = btn->next;
      else lbtn->next = btn->next;
      SFREE(btn->name);
      SFREE(btn->output);
      if (btn->color != NULL) SFREE(btn->color);
      free(btn);
    };

   draw_buttons(dw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	menu_modify_btn -- handle MODIFY menu option			*/
/*									*/
/************************************************************************/


static Boolean
menu_modify_btn(dw)
   DBG_WIN dw;
{
   String menu[64];
   Character buf[48][48];
   Character nm[32],out[64],col[32];
   Character aft[64];
   Integer ct,bct,bfct;
   DBG_BTN btn,bta,btb;
   Integer bnm;
   Integer mode,pos;

   bfct = 0;
   ct = 0;
   bnm = -1;
   menu[ct++] = "%CModify Button\n";

   bct = 0;
   for (btn = dw->buttons; btn != NULL; btn = btn->next) {
      if (btn->next == NULL) {
	 sprintf(buf[bfct],"%%0.%do %s",bct,btn->name);
	 bct++;
       }
      else {
	 sprintf(buf[bfct],"%%0.%do %s%%M%%0.%do %s",
		    bct,btn->name,bct+1,btn->next->name);
	 btn = btn->next;
	 bct += 2;
       };
      menu[ct++] = buf[bfct++];
    };

   if (bct == 0) return FALSE;

   menu[ct++] = "\nNew Name:   %1.16t";
   menu[ct++] = "New Output: %2.48t";
   menu[ct++] = "\n%4.0o No highlighting";
   menu[ct++] = "%4.2o Highlight during run";
   menu[ct++] = "%4.3o Highlight during continue";
   if (ASHinq_configuration_depth(dw->window) > 1) menu[ct++] = "New Color: %3.16t";

   menu[ct++] = "\n%5.1o Make First%M%5.2o Make Last";
   menu[ct++] = "\n%5.0o Move after button %6.32t";

   menu[ct++] = "\n   %a%M   %c";
   menu[ct] = 0;

   if (bnm < 0) {
      nm[0] = 0;
      out[0] = 0;
      col[0] = 0;
      mode = 0;
      pos = 0;
      aft[0] = 0;
    };

   if (!STEMdialog(dw->window,menu,&bnm,nm,out,col,&mode,&pos,aft)) return FALSE;
   if (bnm < 0) return FALSE;

   bct = 0;
   bta = NULL;
   for (btn = dw->buttons; btn != NULL; btn = btn->next) {
      if (bct++ == bnm) break;
      bta = btn;
    };

   if (btn != NULL) {
      if (nm[0] != 0) {
	 SFREE(btn->name);
	 btn->name = SALLOC(nm);
       };
      if (out[0] != 0) {
	 SFREE(btn->output);
	 btn->output = SALLOC(out);
       };
      if (col[0] != 0) {
	 if (btn->color != NULL) SFREE(btn->color);
	 btn->color = SALLOC(col);
       }
      else if (btn->color != NULL) {
	 SFREE(btn->color);
	 btn->color = NULL;
       };
      btn->mode = (DBG_BTN_MODE) mode;

      if (pos != 0 || aft[0] != 0) {
	 if (aft[0] != 0) {
	    for (btb = dw->buttons; btb != NULL; btb = btb->next) {
	       if (STREQL(btb->name,aft)) break;
	     };
	    if (btb == NULL) pos = 2;
	  };
	 if (bta == NULL) dw->buttons = btn->next;
	 else bta->next = btn->next;
	 if (pos == 1) {
	    btn->next = dw->buttons;
	    dw->buttons = btn;
	  }
	 else if (pos == 2) {
	    btn->next = NULL;
	    if (dw->buttons == NULL) dw->buttons = btn;
	    else {
	       for (btb = dw->buttons; btb->next != NULL; btb = btb->next);
	       btb->next = btn;
	     };
	  }
	 else {
	    btn->next = btb->next;
	    btb->next = btn;
	  };
       };
    };

   draw_buttons(dw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	dbg_btn_handler -- handle button presses			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
dbg_btn_handler(db,dw,w,txt)
   DBG_BTN db;
   DBG_WIN dw;
   ASH_WINDOW w;
   String txt;
{
   Character buf[1024];

   if (db == NULL) return FALSE;
   if (dw == NULL) return FALSE;

   fix_output(dw,buf,db);
   EDTexec_input(dw->exec_info,buf);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	add_button -- add button to list of buttons			*/
/*									*/
/************************************************************************/


static DBG_BTN
add_button(dw,fst,nm,out,color,mode,aft,endfg)
   DBG_WIN dw;
   DBG_BTN fst;
   String nm;
   String out;
   String color;
   DBG_BTN_MODE mode;
   DBG_BTN aft;
   Boolean endfg;
{
   DBG_BTN new;

   new = PALLOC(DBG_BTN_INFO);
   new->name = SALLOC(nm);
   new->output = SALLOC(out);
   new->color = SALLOC(color);
   new->mode = mode;
   new->win = dw;

   PROTECT;
   if (endfg && aft == NULL && fst != NULL) {
      for (aft = fst; aft->next != NULL; aft = aft->next) {
	 if (STREQL(aft->name,nm)) {
	    UNPROTECT;
	    free(new);
	    return fst;
	  };
       };
    };

   if (aft == NULL) {
      new->next = fst;
      fst = new;
    }
   else {
      new->next = aft->next;
      aft->next = new;
    };
   UNPROTECT;

   return fst;
};





/************************************************************************/
/*									*/
/*	draw_buttons -- draw buttons for a debugger instance		*/
/*									*/
/************************************************************************/


static void
draw_buttons(dw)
   DBG_WIN dw;
{
   Integer ct;
   DBG_BTN db;
   STEM_PANEL_BTN btns[64];

   ct = 0;
   for (db = dw->buttons;db != NULL; db = db->next) {
      btns[ct].text = db->name;
      btns[ct].color = db->color;
      btns[ct].data = (int) db;
      ++ct;
    };
   btns[ct].text = NULL;

   STEMpanel_reset(dw->btn_win,btns);
};






/************************************************************************/
/*									*/
/*	fix_output -- convert user output string to real string 	*/
/*									*/
/************************************************************************/


static void
fix_output(dw,buf,db)
   DBG_WIN dw;
   String buf;
   DBG_BTN db;
{
   Boolean fg;
   String txt,s,v,buf0;
   Character menu[4096],argn[64],tbuf[64],mbuf[128];
   Character arg[6][64];
   Integer act,c;
   Integer qfg;

   buf0 = buf;
   fg = FALSE;
   txt = db->output;
   act = 0;
   sprintf(menu,"Arguments for %s\n\n",db->name);
   qfg = 0;

   while (*txt != 0) {
      if (*txt == '%' && txt[1] != 0) {
	 ++txt;
	 if (*txt == '\'') {
	    ++txt;
	    s = argn;
	    while (*txt != '\'' && *txt != 0) {
	       *s++ = *txt++;
	     };
	    if (*txt == 0) break;
	    ++txt;
	    *s++ = 0;
	    s = argn;
	  }
	 else s = NULL;

	 switch (*txt++) {
	    case 'Q' :
	       s = NULL;
	       qfg |= 1;
	       break;
	    case 'S' :
	       if (s == NULL) s = "Text";
	       v = EDTinq_selection(NULL);
	       c = 48;
	       break;
	    case 'l' :
	       if (s == NULL) s = "Line";
	       if (dw->line > 0) {
		  sprintf(tbuf,"%d",dw->line+1);
		  v = tbuf;
		}
	       else v = NULL;
	       c = 16;
	       break;
	    case 'L' :
	       if (s == NULL) s = "Line";
	       if (dw->line > 0) {
		  sprintf(tbuf,"%d",dw->line);
		  v = tbuf;
		}
	       else v = NULL;
	       c = 16;
	       break;
	    default :
	    case 's' :
	       if (s == NULL) s = "Argument";
	       v = NULL;
	       c = 48;
	       break;
	    case 'F' :
	       if (s == NULL) s = "Function";
	       v = dw->function;
	       c = 32;
	       break;
	    case 'd' :
	       if (s == NULL) s = "Count";
	       v = NULL;
	       c = 16;
	       break;
	   };

	 if (s != NULL) {
	    sprintf(mbuf,"%s: %%%d.%dt\n",s,act,c);
	    strcat(menu,mbuf);
	    if (v == NULL) qfg = 2;
	    if (v == NULL) arg[act][0] = 0;
	    else strcpy(arg[act],v);
	    ++act;
	    *buf++ = '%';
	    *buf++ = 's';
	  };
	 fg = TRUE;
       }
      else if (*txt == '\\' && txt[1] != 0) {
	 ++txt;
	 switch (*txt) {
	    default :
	       *buf++ = *txt;
	       fg = TRUE;
	       break;
	    case 'n' :
	       *buf++ = '\n';
	       fg = FALSE;
	       break;
	    case 't' :
	       *buf++ = '\t';
	       fg = TRUE;
	       break;
	    case 'b' :
	       *buf++ = '\b';
	       fg = TRUE;
	       break;
	    case 'r' :
	       *buf++ = '\r';
	       fg = FALSE;
	       break;
	  };
	 ++txt;
       }
      else if (*txt == '^' && isupper(txt[1])) {
	 ++txt;
	 *buf++ = *txt++ - 0x40;
       }
      else if (*txt == '^' && islower(txt[1])) {
	 ++txt;
	 *buf++ = *txt++ - 0x60;
       }
      else if (*txt == '^' && txt[1] == '?') {
	 txt += 2;
	 *buf++ = interrupt;
       }
      else {
	 *buf++ = *txt++;
	 fg = TRUE;
       };
    };

   if (fg) *buf++ = '\n';

   *buf = 0;

   if (act != 0) {
      strcat(menu,"\n   %a%M   %c");
      if (qfg != 1 &&
	     !STEMdialog1(dw->window,menu,arg[0],arg[1],arg[2],arg[3],arg[4],arg[5])) {
	 buf[0] = 0;
	 return;
       };
      strcpy(menu,buf0);
      sprintf(buf0,menu,arg[0],arg[1],arg[2],arg[3],arg[4],arg[5]);
    };
};






/************************************************************************/
/*									*/
/*	reset_buttons -- reset all buttons on a window			*/
/*									*/
/************************************************************************/


static void
reset_buttons(dw,btns)
   DBG_WIN dw;
   DBG_BTN btns;
{
   DBG_BTN db,ndb;

   for (db = dw->buttons; db != NULL; db = ndb) {
      ndb = db->next;
      SFREE(db->name);
      SFREE(db->output);
      if (db->color != NULL) SFREE(db->color);
      free(db);
    };
   dw->buttons = NULL;

   for (db = btns; db != NULL; db = db->next) {
      dw->buttons = add_button(dw,dw->buttons,db->name,db->output,db->color,
				  db->mode,NULL,TRUE);
    };

   draw_buttons(dw);
};





/************************************************************************/
/*									*/
/*	save_buttons -- write current button definitions to a file	*/
/*									*/
/************************************************************************/


static Boolean
save_buttons(dw,dir)
   DBG_WIN dw;
   String dir;
{
   DBG_BTN db;
   Character buf[128];
   FILE * otf;

   sprintf(buf,"%s/%s",dir,BUTTON_FILE_NAME);
   otf = fopen(buf,"w");
   if (otf == NULL) {
      STEMdialog1(dw->window,"Can't open file to save button information\n\n%M   %a");
      return FALSE;
    };

   for (db = dw->buttons; db != NULL; db = db->next) {
      fprintf(otf,"%s%c%s",db->name,BUTTON_FILE_SEP,db->output);
      if (db->color != NULL) fprintf(otf,"%c%s",BUTTON_FILE_SEP,db->color);
      if (db->mode != DBG_MODE_NONE) {
	 if (db->color == NULL) fprintf(otf,"%c",BUTTON_FILE_SEP);
	 fprintf(otf,"%c%d",BUTTON_FILE_SEP,db->mode);
       };
      fprintf(otf,"\n");
    };

   fclose(otf);

   return TRUE;
};





/* end of dbgmsg.c */

