/************************************************************************/
/*									*/
/*		cbrowclass.c						*/
/*									*/
/*	Routines for managing cbrow class data				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "cbrow_local.h"




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	split_line();
static	Boolean 	boolv();
static	CBROW_CLASS	make_class();
static	CBROW_HIER	add_hierarchy();
static	CBROW_MEMB	add_member();
static	void		add_def_info();
static	String		fix_string();
static	void		free_class_data();
static	void		add_class_types();
static	void		add_inherited();
static	void		add_inherited_membs();
static	void		fix_print_names();
static	String		fix_type_name();






/************************************************************************/
/*									*/
/*	CBROW_class_init -- module initialization			*/
/*									*/
/************************************************************************/


void
CBROW_class_init()
{
};





/************************************************************************/
/*									*/
/*	CBROW_class_setup -- setup classes for a new system		*/
/*									*/
/************************************************************************/


void
CBROW_class_setup(cw)
   CBROW_WIN cw;
{
   String filnm;
   String flds[16];
   FILE * inf;
   Character lbuf[10240];

   CBROW_disp_free(cw);

   if (cw->classes != NULL) {
      free_class_data(cw);
    };

   cw->classes = NULL;

   filnm = MSGcalla("XREF QUERY %s *D.name,D.file,D.line D.class == `CLASSID",cw->system);
   inf = fopen(filnm,"r");
   if (inf == NULL) {
      fprintf(stderr,"CBROW: Problem with XREF\n");
      ASHremove(cw->window);
      return;
    };

   while (fgets(lbuf,10240,inf) != NULL) {
      if (split_line(lbuf,16,flds) == 3) {
	 make_class(cw,flds[0],flds[1],atol(flds[2]));
       };
    };

   fclose(inf);
   unlink(filnm);
   SFREE(filnm);

   filnm = MSGcalla("XREF QUERY %s *H.class,H.parent,H.file,H.line,H.public,H.virtual,H.friend ",cw->system);

   inf = fopen(filnm,"r");
   while (fgets(lbuf,10240,inf) != NULL) {
      if (split_line(lbuf,16,flds) == 7) {
	 add_hierarchy(cw,flds[0],flds[1],flds[2],atol(flds[3]),
			  boolv(flds[4]),boolv(flds[5]),boolv(flds[6]));
       };
    };

   fclose(inf);
   unlink(filnm);
   SFREE(filnm);

   filnm = MSGcalla("XREF QUERY %s *M.class,M.member,M.file,M.line,M.protect,M.isdata,M.inline,M.friend,M.virtual,M.static,M.pure,M.const ",
		       cw->system);

   inf = fopen(filnm,"r");
   while (fgets(lbuf,10240,inf) != NULL) {
      if (split_line(lbuf,16,flds) == 12) {
	 add_member(cw,flds[0],flds[1],flds[2],atol(flds[3]),flds[4],
		       boolv(flds[5]),boolv(flds[6]),boolv(flds[7]),boolv(flds[8]),
		       boolv(flds[9]),boolv(flds[10]),boolv(flds[11]),NULL);
       };
    };

   fclose(inf);
   unlink(filnm);
   SFREE(filnm);

   filnm = MSGcalla("XREF QUERY %s *m.class,m.member,m.file,m.line,m.type,m.name ",
		       cw->system);

   inf = fopen(filnm,"r");
   while (fgets(lbuf,10240,inf) != NULL) {
      if (split_line(lbuf,16,flds) == 6) {
	 add_def_info(cw,flds[0],flds[1],flds[2],atol(flds[3]),flds[4],flds[5]);
       };
    };

   fclose(inf);
   unlink(filnm);
   SFREE(filnm);

   add_class_types(cw);
   add_inherited(cw);
   fix_print_names(cw);
};





/************************************************************************/
/*									*/
/*	CBROW_class_find -- find class by name				*/
/*									*/
/************************************************************************/


CBROW_CLASS
CBROW_class_find(cw,nam)
   CBROW_WIN cw;
   String nam;
{
   CBROW_CLASS cc;

   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      if (STRGEQ(cc->name,nam)) break;
    };

   if (cc != NULL && STRNEQ(cc->name,nam)) cc = NULL;

   return cc;
};





/************************************************************************/
/*									*/
/*	CBROW_class_find_member -- find member given name from FIELD	*/
/*									*/
/************************************************************************/


CBROW_MEMB
CBROW_class_find_member(cw,nam)
   CBROW_WIN cw;
   String nam;
{
   CBROW_CLASS cc;
   CBROW_MEMB cm;
   Character buf[1024];
   String s,t;

   t = buf;
   for (s = nam; *s != 0 && *s != ':'; ++s) *t++ = *s;

   if (*s == 0) {
      for (cc = cw->classes; cc != NULL; cc = cc->next) {
	 for (cm = cc->members; cm != NULL; cm = cm->next) {
	    if (cm->cls == cm->real_cls) {
	       if (STREQL(cm->name,nam)) return cm;
	       if (cm->defname != NULL && STREQL(cm->defname,nam)) return cm;
	     };
	  };
       };
    }
   else {
      *t = 0;
      cc = CBROW_class_find(cw,buf);
      if (cc == NULL) return NULL;
      for (cm = cc->members; cm != NULL; cm = cm->next) {
	 if (STREQL(cm->name,nam)) return cm;
	 if (cm->defname != NULL && STREQL(cm->defname,nam)) return cm;
       };
    };

   return NULL;
};





/************************************************************************/
/*									*/
/*	CBROW_class_test_super -- check if A is supertype of B		*/
/*									*/
/************************************************************************/


Boolean
CBROW_class_test_super(ca,cb)
   CBROW_CLASS ca;
   CBROW_CLASS cb;
{
   CBROW_HIER ch;

   for (ch = cb->supers; ch != NULL; ch = ch->next_super) {
      if (ch->isfriend) continue;
      if (ch->super == ca) return TRUE;
      if (CBROW_class_test_super(ca,ch->super)) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	CBROW_class_compare -- sort order for two classes		*/
/*	CBROW_member_compare -- sort order for two members		*/
/*									*/
/************************************************************************/


int
CBROW_class_compare(c1p,c2p)
   CBROW_CLASS * c1p;
   CBROW_CLASS * c2p;
{
   CBROW_CLASS cc1,cc2;

   cc1 = *c1p;
   cc2 = *c2p;

   return strcmp(cc1->name,cc2->name);
};






int
CBROW_member_compare(m1p,m2p)
   CBROW_MEMB * m1p;
   CBROW_MEMB * m2p;
{
   CBROW_MEMB cm1,cm2;

   cm1 = *m1p;
   cm2 = *m2p;

   if (cm1->isprivate != cm2->isprivate || cm1->isprotected != cm2->isprotected ||
	  cm1->isfriend != cm2->isfriend) {
      if (cm1->isfriend && !cm2->isfriend) return 1;
      else if (cm2->isfriend && !cm1->isfriend) return -1;
      if (cm1->isprivate && !cm2->isprivate) return -1;
      else if (cm2->isprivate && !cm1->isprivate) return 1;
      if (cm1->isprotected) return -1;
      else return 1;
    };

   if (cm1->isdata && !cm2->isdata) return -1;
   else if (!cm1->isdata && cm2->isdata) return 1;

   return strcmp(cm1->name,cm2->name);
};






/************************************************************************/
/*									*/
/*	CBROW_find_string -- get string for display purposes		*/
/*									*/
/************************************************************************/


String
CBROW_find_string(cw,s)
   CBROW_WIN cw;
   String s;
{
   return fix_string(cw,s);
};





/************************************************************************/
/*									*/
/*	split_line -- split line into strings				*/
/*									*/
/************************************************************************/


static Integer
split_line(lbuf,max,ptrs)
   String lbuf;
   String ptrs[];
{
   Integer ct;
   String s;

   ct = 0;
   ptrs[ct++] = lbuf;

   for (s = lbuf; *s != 0 && *s != '\n'; ++s) {
      if (*s == '\177') {
	 *s++ = 0;
	 if (*s == 0 || *s == '\n') break;
	 if (ct >= max) return -1;
	 ptrs[ct++] = s;
       };
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	boolv -- return boolean value from string			*/
/*									*/
/************************************************************************/


static Boolean
boolv(s)
   String s;
{
   return (s[0] == 'T' || s[0] == 't');
};





/************************************************************************/
/*									*/
/*	make_class -- create a new class				*/
/*									*/
/************************************************************************/


static CBROW_CLASS
make_class(cw,name,file,lno)
   CBROW_WIN cw;
   String name;
   String file;
   Integer lno;
{
   CBROW_CLASS cc,lcc;

   lcc = NULL;
   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      if (STRGEQ(cc->name,name)) break;
      lcc = cc;
    };

   if (cc == NULL || STRNEQ(cc->name,name)) {
      cc = PALLOC(CBROW_CLASS_INFO);
      if (lcc == NULL) {
	 cc->next = cw->classes;
	 cw->classes = cc;
       }
      else {
	 cc->next = lcc->next;
	 lcc->next = cc;
       };
      cc->name = SALLOC(name);
      cc->file = NULL;
      cc->line = 0;
      cc->supers = NULL;
      cc->subs = NULL;
      cc->members = NULL;
      cc->ignore = FALSE;
      cc->infoline = -1;
      cc->isabstract = FALSE;
    };

   if (file != NULL && cc->file == NULL) cc->file = fix_string(cw,file);

   if (lno > 0 && cc->line == 0) cc->line = lno;

   return cc;
};





/************************************************************************/
/*									*/
/*	add_hierarchy -- add hierarchy node				*/
/*									*/
/************************************************************************/


static CBROW_HIER
add_hierarchy(cw,cls,par,file,lno,pub,vir,fnd)
   CBROW_WIN cw;
   String cls;
   String par;
   String file;
   Integer lno;
   Boolean pub;
   Boolean vir;
   Boolean fnd;
{
   CBROW_CLASS cc,cp;
   CBROW_HIER ch,lch,nch;

   cc = make_class(cw,cls,file,lno);
   cp = make_class(cw,par,NULL,0);

   lch = NULL;
   for (ch = cc->supers; ch != NULL; ch = ch->next_super) {
      if (ch->super == cp) {
	 if (ch->isfriend && fnd) break;
	 if (!ch->isfriend && !fnd) break;
       };
      lch = ch;
    };

   if (ch == NULL) {
      ch = PALLOC(CBROW_HIER_INFO);
      ch->super = cp;
      ch->sub = cc;
      ch->file = NULL;
      ch->line = 0;
      ch->ispublic = FALSE;
      ch->isvirtual = FALSE;
      ch->isfriend = FALSE;
      ch->next_super = NULL;
      if (lch == NULL) cc->supers = ch;
      else lch->next_super = ch;
      lch = NULL;
      for (nch = cp->subs; nch != NULL; nch = nch->next_sub) lch = nch;
      if (lch == NULL) cp->subs = ch;
      else lch->next_sub = ch;
    };

   if (file != NULL && ch->file == NULL) ch->file = fix_string(cw,file);
   if (lno > 0 && ch->line == 0) ch->line = lno;
   ch->ispublic |= pub;
   ch->isvirtual |= vir;
   ch->isfriend |= fnd;

   return ch;
};





/************************************************************************/
/*									*/
/*	add_member -- add member to class				*/
/*									*/
/************************************************************************/


static CBROW_MEMB
add_member(cw,cls,mem,file,lno,prot,isd,inl,frn,vir,stt,pur,con,frm)
   CBROW_WIN cw;
   String cls;
   String mem;
   String file;
   Integer lno;
   String prot;
   Boolean isd;
   Boolean inl;
   Boolean frn;
   Boolean vir;
   Boolean stt;
   Boolean pur;
   Boolean con;
   CBROW_MEMB frm;
{
   CBROW_CLASS cc;
   CBROW_MEMB cm,lcm;
   Boolean prv,prt;
   String s,t,sig,nsig;
   Character buf[1024];

   cc = make_class(cw,cls,NULL,0);

   if (frm == NULL || frm->defname == NULL) sig = NULL;
   else {
      sig = index(frm->defname,':');
      if (sig != NULL) sig += 2;
    };

   lcm = NULL;
   for (cm = cc->members; cm != NULL; cm = cm->next) {
      if (STREQL(cm->name,mem)) {
	 if (frm == NULL) {
	    if (file == NULL || cm->file == NULL) break;
	    if (lno == 0 || cm->line == 0) break;
	    if (STREQL(file,cm->file) && lno == cm->line) break;
	  }
	 else {
	    if (sig == NULL || cm->defname == NULL) nsig = NULL;
	    else {
	       nsig = index(cm->defname,':');
	       if (nsig != NULL) nsig += 2;
	     };
	    if (nsig == NULL || STREQL(sig,nsig)) break;
	  };
       };
      lcm = cm;
    };

   if (cm == NULL) {
      cm = PALLOC(CBROW_MEMB_INFO);
      cm->cls = cc;
      cm->real_cls = cc;
      cm->next = NULL;
      if (lcm == NULL) cc->members = cm;
      else lcm->next = cm;

      t = buf;
      for (s = mem; *s != 0 && *s != ':'; ++s) *t++ = *s++;
      *t = 0;
      if (*s == ':' && s[1] == ':') {
	 cm->real_cls = make_class(cw,buf,NULL,0);
	 s += 2;
	 mem = s;
       };

      cm->name = fix_string(cw,mem);
      if (cm->cls == cm->real_cls) {
	 cm->pname = cm->name;
	 cm->dname = cm->name;
       }
      else {
	 sprintf(buf,"%s::%s",cm->real_cls->name,cm->name);
	 cm->pname = fix_string(cw,buf);
	 sprintf(buf,"::%s",cm->name);
	 cm->dname = fix_string(cw,buf);
       };
      cm->file = NULL;
      cm->line = 0;
      cm->isprivate = FALSE;
      cm->isprotected = FALSE;
      cm->isdata = FALSE;
      cm->isinline = FALSE;
      cm->isfriend = FALSE;
      cm->isvirtual = FALSE;
      cm->isstatic = FALSE;
      cm->ispure = FALSE;
      cm->isconst = FALSE;
      cm->deffile = NULL;
      cm->defline = 0;
      cm->defname = NULL;
      cm->deftype = NULL;
      cm->defcls = NULL;
      cm->ignore = FALSE;
    };

   if (file == NULL) return cm;

   if (prot == NULL) prot == "";
   prt = STREQL(prot,"PROTECTED");
   prv = STREQL(prot,"PRIVATE");

   if (cm->file == NULL && file != NULL) cm->file = fix_string(cw,file);
   if (cm->line == 0 && lno > 0) cm->line = lno;

   cm->isprivate |= prv;
   cm->isprotected |= prt;
   cm->isdata |= isd;
   cm->isinline |= inl;
   cm->isfriend |= frn;
   cm->isvirtual |= vir;
   cm->isstatic |= stt;
   cm->ispure |= pur;
   cm->isconst |= con;

   if (cm->ispure) cm->cls->isabstract = TRUE;

   return cm;
};





/************************************************************************/
/*									*/
/*	add_def_info -- add definition information to a member/class	*/
/*									*/
/************************************************************************/


static void
add_def_info(cw,cls,mem,dfile,dline,dtyp,dname)
   CBROW_WIN cw;
   String cls;
   String mem;
   String dfile;
   Integer dline;
   String dtyp;
   String dname;
{
   CBROW_CLASS cc;
   CBROW_MEMB cm;
   Integer i;

   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      if (STRGEQ(cc->name,cls)) break;
    };
   if (cc == NULL) return;

   for (cm = cc->members; cm != NULL; cm = cm->next) {
      if (cm->deffile == NULL && STREQL(cm->name,mem)) break;
    };
   if (cm == NULL) return;

   if (!cm->isdata) {
      i = strlen(dtyp);
      if (dtyp[i-1] == ')' && dtyp[i-2] == '(') {
	 if (dtyp[i-3] == ' ') dtyp[i-3] = 0;
	 else dtyp[i-2] = 0;
       };
    };

   dtyp = fix_type_name(cw,dtyp);
   cm->deffile = fix_string(cw,dfile);
   cm->defline = dline;
   cm->deftype = fix_string(cw,dtyp);
   cm->defname = fix_string(cw,dname);
};





/************************************************************************/
/*									*/
/*	fix_string -- get standardized string				*/
/*									*/
/************************************************************************/


static String
fix_string(cw,fnm)
   CBROW_WIN cw;
   String fnm;
{
   String s;
   Sequence l;

   if (fnm == NULL) return NULL;

   forin (s,String,l,cw->strings) {
      if (STREQL(s,fnm)) return s;
    };

   s = SALLOC(fnm);
   cw->strings = CONS(s,cw->strings);

   return s;
};





/************************************************************************/
/*									*/
/*	free_class_data -- free all data for window			*/
/*									*/
/************************************************************************/


static void
free_class_data(cw)
   CBROW_WIN cw;
{
   CBROW_CLASS cc,ncc;
   CBROW_HIER ch,nch;
   CBROW_MEMB cm,ncm;
   Sequence strs,l;
   String s;

   PROTECT;
   ncc = cw->classes;
   cw->classes = NULL;
   strs = cw->strings;
   cw->strings = NULL;
   cw->display = NULL;
   cw->curmemb = NULL;
   cw->cureval = NULL;
   UNPROTECT;

   for (cc = ncc; cc != NULL; cc = ncc) {
      ncc = cc->next;
      for (ch = cc->supers; ch != NULL; ch = nch) {
	 nch = ch->next_super;
	 free(ch);
       };
      for (cm = cc->members; cm != NULL; cm = ncm) {
	 ncm = cm->next;
	 free(cm);
       };
      free(cc);
    };
   cw->classes = NULL;

   forin (s,String,l,strs) SFREE(s);
   LFREE(strs);
};





/************************************************************************/
/*									*/
/*	add_class_types -- add class types for members			*/
/*									*/
/************************************************************************/


static void
add_class_types(cw)
   CBROW_WIN cw;
{
   CBROW_CLASS cc,cc1;
   CBROW_MEMB cm;
   Character buf[1024];
   String s,p;

   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      for (cm = cc->members; cm != NULL; cm = cm->next) {
	 if (cm->deftype != NULL) {
	    p = buf;
	    for (s = cm->deftype; isidchar(*s); ++s) *p++ = *s;
	    *p = 0;
	    if (STREQL(buf,"struct") || STREQL(buf,"union") || STREQL(buf,"class")) {
	       while (isspace(*s)) ++s;
	       p = buf;
	       while (isidchar(*s)) *p++ = *s++;
	       *p = 0;
	     };
	    if (buf[0] != 0) {
	       for (cc1 = cw->classes; cc1 != NULL; cc1 = cc1->next) {
		  if (STREQL(cc1->name,buf)) break;
		};
	       cm->defcls = cc1;
	     };
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	add_inherited -- add inherited members to all classes		*/
/*	add_inherited -- add inherited members for one class/superclass */
/*									*/
/************************************************************************/


static void
add_inherited(cw)
   CBROW_WIN cw;
{
   CBROW_CLASS cc;
   CBROW_HIER ch;

   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      for (ch = cc->supers; ch != NULL; ch = ch->next_super) {
	 add_inherited_membs(cw,cc,ch);
       };
    };
};





static void
add_inherited_membs(cw,cc,ch)
   CBROW_WIN cw;
   CBROW_CLASS cc;
   CBROW_HIER ch;
{
   CBROW_MEMB cm,sm;
   CBROW_CLASS sup;
   Character buf[1024];

   if (ch->isfriend) return;

   sup = ch->super;

   for (cm = sup->members; cm != NULL; cm = cm->next) {
      if (STREQL(cm->name,cm->cls->name)) continue;
      if (cm->name[0] == '~') continue;
      if (!cm->isprivate && !cm->isfriend) {
	 sm = add_member(cw,cc->name,cm->name,NULL,0,NULL,FALSE,FALSE,FALSE,FALSE,FALSE,
			    FALSE,FALSE,cm);
	 if (sm->file == NULL) {
	    sm->real_cls = cm->real_cls;
	    if (cm->cls != cm->real_cls) {
	       sm->pname = cm->pname;
	       sm->dname = cm->dname;
	     }
	    else {
	       sprintf(buf,"%s::%s",sm->real_cls->name,sm->name);
	       sm->pname = fix_string(cw,buf);
	       sprintf(buf,"::%s",sm->name);
	       sm->dname = fix_string(cw,buf);
	     };
	    sm->file = cm->file;
	    sm->line = cm->line;
	    sm->deffile = cm->deffile;
	    sm->defline = cm->defline;
	    sm->defname = cm->defname;
	    sm->deftype = cm->deftype;
	    sm->defcls = cm->defcls;
	    if (ch->ispublic) {
	       sm->isprivate = cm->isprivate;
	       sm->isprotected = cm->isprotected;
	     }
	    else {
	       sm->isprivate = TRUE;
	       sm->isprotected = FALSE;
	     };
	    sm->isdata = cm->isdata;
	    sm->isinline = cm->isinline;
	    sm->isfriend = cm->isfriend;
	    sm->isvirtual = cm->isvirtual;
	    sm->isstatic = cm->isstatic;
	    sm->ispure = cm->ispure;
	    sm->isconst = cm->isconst;
	    if (sm->ispure) sm->cls->isabstract = TRUE;
	  };
       };
    };

   for (ch = sup->supers; ch != NULL; ch = ch->next_super) {
      add_inherited_membs(cw,cc,ch);
    };
};





/************************************************************************/
/*									*/
/*	fix_print_names -- fix print names for members			*/
/*									*/
/************************************************************************/


static void
fix_print_names(cw)
   CBROW_WIN cw;
{
   CBROW_CLASS cc;
   CBROW_MEMB cm;
   String s;

   for (cc = cw->classes; cc != NULL; cc = cc->next) {
      for (cm = cc->members; cm != NULL; cm = cm->next) {
	 cm->defpname = cm->defname;
	 if (cm->defname != NULL && cm->cls == cm->real_cls) {
	    s = index(cm->defpname,':');
	    if (s != NULL && s[1] == ':') cm->defpname = &s[2];
	  }
	 else if (cm->defpname == NULL) cm->defpname = cm->pname;
       };
    };
};





/************************************************************************/
/*									*/
/*	fix_type_name -- fix type name for display			*/
/*									*/
/************************************************************************/


static String
fix_type_name(cw,t)
   CBROW_WIN cw;
   String t;
{
   String s,p;
   CBROW_CLASS cc;
   Character buf[1024];

   if (t == NULL) return NULL;

   s = t;
   while (isspace(*s)) ++s;

   if (strncmp(s,"struct ",7) == 0 || strncmp(s,"class ",6) == 0) {
      s = index(s,' ');
      while (isspace(*s)) ++s;
      strcpy(buf,s);
      for (p = buf; isidchar(*p); ++p);
      if (p != buf) {
	 *p = 0;
	 cc = CBROW_class_find(cw,buf);
	 if (cc != NULL) t = s;
       };
    };

   return t;
};





/* end of cbrowclass.c */

