/************************************************************************/
/*									*/
/*		buildmon.c						*/
/*									*/
/*	BUILD message monitor/sending routines				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "build_local.h"




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		handle_system_msg();
static	void		handle_make_msg();
static	void		handle_compile_msg();
static	void		handle_compilego_msg();
static	void		handle_command_msg();
static	void		handle_wd_msg();
static	void		fix_msg_name();
static	void		check_new_file();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


typedef struct _ERR_INFO {
   String text;
   String type;
   String format;
   PMAT_PAT pattern;
} ERR_INFO;


static	ERR_INFO	err_table[] = {
   { "\"%1s\", line %2d: warning: %3s\n", "WARNING", NULL, NULL },
   { "\"%1s\", line %2d: %3s\n", "ERROR", NULL, NULL },
   { "%1s: %2d: %3s\n", "ERROR", NULL, NULL },
   { "%s %s% %d %d:%d %d  %1s:", NULL, NULL, NULL },
   { "% %2d  ", NULL, NULL, NULL },
   { "w ------%s %3s\n", "WARNING", NULL, NULL },
   { "e ------%s %3s\n", "ERROR", NULL, NULL },
   { "E ------%s %3s\n", "ERROR", NULL, NULL },
   { "w %2d - %3s\n", "WARNING", NULL, NULL },
   { "e %2d - %3s\n", "ERROR", NULL, NULL },
   { "E %2d - %3s\n", "ERROR", NULL, NULL },
   { "s %2d - %3s\n", "WARNING", NULL, NULL },
   { "  E - %5s undefined on lines %2d", "ERROR", "%5 undefined in %4", NULL },
   { "In procedure %4s:", NULL, NULL, NULL },
   { "In program %4s:", NULL, NULL, NULL },
   { "  w - %3s\n", "WARNING", "%3 in %4", NULL },
   { NULL, 0, NULL, NULL },
};





/************************************************************************/
/*									*/
/*	BUILD_mon_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
BUILD_mon_init()
{
   Integer i;

   for (i = 0; err_table[i].text != NULL; ++i) {
      err_table[i].pattern = PMATmake_pattern(err_table[i].text,6,NULL);
    };
};





/************************************************************************/
/*									*/
/*	BUILD_mon_setup -- setup to handle messages			*/
/*									*/
/************************************************************************/


void
BUILD_mon_setup()
{
   AUXD hdl;
   String msg;

   hdl = AUXDget_handle(NULL,"BUILD");

   msg = AUXDget_info(hdl,"SYSTEM_MSG");
   if (msg != NULL) MSGregister(msg,handle_system_msg,1,NULL);

   msg = AUXDget_info(hdl,"MAKE_MSG");
   if (msg != NULL) MSGregister(msg,handle_make_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMPILE_MSG");
   if (msg != NULL) MSGregister(msg,handle_compile_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMPILEGO_MSG");
   if (msg != NULL) MSGregister(msg,handle_compilego_msg,1,NULL);

   msg = AUXDget_info(hdl,"COMMAND_MSG");
   if (msg != NULL) MSGregister(msg,handle_command_msg,2,NULL);

   msg = AUXDget_info(hdl,"FILEWD_MSG");
   if (msg != NULL) MSGregister(msg,handle_wd_msg,0,NULL);
};





/************************************************************************/
/*									*/
/*	BUILD_check_output -- check output for message possibilities	*/
/*									*/
/************************************************************************/


void
BUILD_check_output(bp,msg)
   BUILD_PROJ bp;
   String msg;
{
   Integer i,j;
   Universal args[10];
   Character buf[1024],buf1[1024];
   String err;
   String s,t,p;

   for (i = 0; err_table[i].pattern != NULL; ++i) {
      if (PMATmatch(msg,err_table[i].pattern,args) >= 0) {
	 if (err_table[i].format == NULL) {
	    if (args[0] != NULL) {
	       if (bp->last_file != NULL) SFREE(bp->last_file);
	       bp->last_file = SALLOC(args[0]);
	       bp->last_line = 0;
	     };
	    if (args[3] != NULL) {
	       if (bp->last_proc != NULL) SFREE(bp->last_proc);
	       bp->last_proc = SALLOC(args[3]);
	       bp->last_line = 0;
	     };
	    if (((Integer) args[1]) > 0) bp->last_line = (Integer) args[1];
	    err = (String) args[2];
	  }
	 else {
	    s = buf;
	    for (t = err_table[i].format; *t != 0; ++t) {
	       if (*t != '%') *s++ = *t;
	       else {
		  ++t;
		  if (*t < '1' || *t > '6') *s++ = *t;
		  else {
		     j = *t - '1';
		     p = (String) args[j];
		     if (p == NULL) {
			switch (j) {
			   case 0:
			      p = bp->last_file;
			      break;
			   case 1 :
			      sprintf(buf1,"%d",bp->last_line);
			      p = buf1;
			      break;
			   case 3 :
			      p = bp->last_proc;
			      break;
			 };
		      };
		     if (p != NULL) while (*p != 0) *s++ = *p++;
		   };
		};
	     };
	    *s = 0;
	    err = buf;
	  };

	 if (err_table[i].type != NULL) {
	    if (bp->last_file != NULL && bp->last_line > 0 && err != NULL) {
	       MSGsenda("BUILD %s %s %d %S",
			   err_table[i].type,bp->last_file,bp->last_line,err);
	     }
	    else if (bp->last_file != NULL && err != NULL) {
	       MSGsenda("BUILD %s %s %d %S",
			   err_table[i].type,bp->last_file,9999999,err);
	     };
	  };
	 break;
       };
    };
};






/************************************************************************/
/*									*/
/*	handle_system_msg -- handle system message			*/
/*									*/
/************************************************************************/


static void
handle_system_msg(sys)
   String sys;
{
   BUILD_find_project(sys);
};





/************************************************************************/
/*									*/
/*	handle_make_msg -- handle MAKE message				*/
/*									*/
/************************************************************************/


static void
handle_make_msg(name,rid)
   String name;
   Integer rid;
{
   Character buf[1024];

   fix_msg_name(name,buf);
   check_new_file(buf);

   BUILD_cmd_make(NULL,buf,rid);
};





/************************************************************************/
/*									*/
/*	handle_compile_msg -- handle COMPILE message			*/
/*									*/
/************************************************************************/


static void
handle_compile_msg(name,rid)
   String name;
   Integer rid;
{
   Character buf[1024];

   fix_msg_name(name,buf);
   check_new_file(buf);

   BUILD_cmd_compile(NULL,buf,rid);
};





/************************************************************************/
/*									*/
/*	handle_compilego_msg -- handle COMPGO message			*/
/*									*/
/************************************************************************/


static void
handle_compilego_msg(name,rid)
   String name;
   Integer rid;
{
   Character buf[1024];

   fix_msg_name(name,buf);
   check_new_file(buf);

   BUILD_cmd_compile_go(NULL,buf,rid);
};





/************************************************************************/
/*									*/
/*	handle_command_msg -- handle COMMAND message			*/
/*									*/
/************************************************************************/


static void
handle_command_msg(sys,command,rid)
   String sys;
   String command;
   Integer rid;
{
   BUILD_cmd_make(BUILD_find_project(sys),command,rid);
};





/********************************************************************************/
/*										*/
/*	handle_wd_msg -- handle WD change message (from MSG)			*/
/*										*/
/********************************************************************************/


static void
handle_wd_msg()
{
   BUILD_data_new_wd();

   BUILD_find_project(MSGinq_wd());
};





/************************************************************************/
/*									*/
/*	fix_msg_name -- fix file name from a message request		*/
/*									*/
/************************************************************************/


static void
fix_msg_name(name,buf)
   String name;
   String buf;
{
   String s;

   s = rindex(name,'/');
   if (s == NULL) s = name;
   else ++s;

   strcpy(buf,s);
};





/************************************************************************/
/*									*/
/*	check_new_file -- check for new files from messages		*/
/*									*/
/************************************************************************/


static void
check_new_file(nm)
   String nm;
{
   BUILD_FILE bf;
   BUILD_PROJ bp;
   BUILD_WIN bw;

   bf = BUILD_find_file(nm,BUILD_TYPE_UNKNOWN,NULL);
   if (bf == NULL) {
      if (access(nm,0) >= 0) {
	 bp = BUILD_find_project(MSGinq_wd());
	 if (bp == NULL) return;
	 BUILD_data_remove(bp);
	 BUILD_file_setup(bp);
	 bw = BUILD_find_window_by_project(bp);
	 if (bw != NULL) {
	    bw->project = NULL;
	    BUILD_set_project(bw,bp);
	  };
       };
     };
};






/* end of buildmon.c */



