/************************************************************************/
/*									*/
/*		buildfile.c						*/
/*									*/
/*	File management routines					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "build_local.h"
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		get_dir_files();
static	void		gen_suffix();
static	void		add_suffix();
static	void		gen_macros();
static	void		gen_project();
static	void		gen_files();
static	void		gen_depend();
static	void		add_auto_depends();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String	makefiles[] = {
   "Projfile",
   "makefile",
   "Makefile",
   0
};



static	String	ignorefiles[] = {
   "bBACKUP",
   "bCONTROL",
   "bBUFFERS",
   "core",
   0
};





/************************************************************************/
/*									*/
/*	BUILD_file_init -- module initialization			*/
/*									*/
/************************************************************************/


void
BUILD_file_init()
{
};





/************************************************************************/
/*									*/
/*	BUILD_file_setup -- setup by reading files			*/
/*									*/
/************************************************************************/


void
BUILD_file_setup(bp)
   BUILD_PROJ bp;
{
   String s;
   Character buf[1024];
   Integer i;

   if (bp->make_name == NULL) {
      s = (String) getenv("BUILD_RULES");
      if (s == NULL) {
	 s = MSGrundata_name("build.rules",buf);
       };
      BUILD_read_file(bp,s,TRUE);
      for (i = 0; makefiles[i] != NULL; ++i) {
	 sprintf(buf,"%s/%s",bp->system_dir,makefiles[i]);
	 if (access(buf,4) >= 0) break;
	 buf[0] = 0;
       };
      if (buf[0] != 0) {
	 BUILD_read_file(bp,buf,FALSE);
       }
      else {
	 sprintf(buf,"%s/%s",bp->system_dir,makefiles[0]);
       };
      bp->make_name = SALLOC(buf);
      get_dir_files(bp,bp->system_dir);
      get_dir_files(bp,bp->source_dir);
      BUILD_data_cleanup(bp);
    };
};





/************************************************************************/
/*									*/
/*	BUILD_file_gen_makefile -- write out new makefile		*/
/*									*/
/************************************************************************/


void
BUILD_file_gen_makefile(bp,save)
   BUILD_PROJ bp;
   Boolean save;
{
   FILE * f;
   BUILD_WIN bw;

   if (!save) f = fopen(bp->temp_make,"w");
   else f = fopen(bp->make_name,"w");

   if (f == NULL) {
      bw = BUILD_find_window_by_project(bp);
      if (save) BUILD_error(bw,"Couldn't open new file %s",bp->make_name);
      else BUILD_error(bw,"Couldn't open new file %s",bp->temp_make);
      return;
    };

   gen_suffix(bp,f,save);
   gen_macros(bp,f,save);
   gen_project(bp,f,save);
   gen_files(bp,f,save);
   gen_depend(bp,f,save);

   fclose(f);
};





/************************************************************************/
/*									*/
/*	BUILD_file_depends -- do dependency analysis			*/
/*									*/
/************************************************************************/


void
BUILD_file_depends(bp)
   BUILD_PROJ bp;
{
   BUILD_FILE bf;
   BUILD_DEPEND bd;
   Sequence l,la,lb;

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->autodepend) {
	 lb = NULL;
	 forin (bd,BUILD_DEPEND,la,bf->depends) {
	    if (!bd->automatic || bd->implicit) lb = APPEND(bd,lb);
	  };
	 LFREE(bf->depends);
	 bf->depends = lb;
	 if (bp->autodepend) bf->autodepend = FALSE;
       };
    };

   BUILD_cmd_depends(bp);

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->autodepend || bp->autodepend) {
	 lb = LCOPY(bf->depends);
	 forin (bd,BUILD_DEPEND,la,lb) {
	    add_auto_depends(bp,bf,bd);
	  };
	 LFREE(lb);
       };
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->type & BUILD_TYPE_DEPEND) {
	 unlink(bf->name);
       };
    };
};





/************************************************************************/
/*									*/
/*	get_dir_files -- get all files from source directory		*/
/*									*/
/************************************************************************/


static void
get_dir_files(bp,dir)
   BUILD_PROJ bp;
   String dir;
{
   struct direct * dp;
   DIR * df;
   Integer i;
   BUILD_FILE bf;
   Character buf[1024];
   struct stat info;

   if (dir == NULL) return;

   df = opendir(dir);
   if (df == NULL) return;

   while ((dp = readdir(df)) != NULL) {
      if (dp->d_name[0] == '.') continue;
      if (dp->d_name[0] == '#') continue;
      for (i = 0; ignorefiles[i] != NULL; ++i) {
	 if (STREQL(dp->d_name,ignorefiles[i])) break;
       };
      if (ignorefiles[i] != NULL) continue;
      sprintf(buf,"%s/%s",dir,dp->d_name);
      if (stat(buf,&info) < 0) continue;
      bf = BUILD_data_file(bp,dp->d_name,FALSE);
      if (info.st_mode & S_IFDIR) bf->type = BUILD_TYPE_DIRECTORY;
    };

   closedir(df);
};





/************************************************************************/
/*									*/
/*	gen_suffix -- generate .SUFFIXES line				*/
/*	add_suffix -- add a suffix to string if not already there	*/
/*									*/
/************************************************************************/


static void
gen_suffix(bp,f,save)
   BUILD_PROJ bp;
   FILE * f;
   Boolean save;
{
   Character buf[10240],lbuf[64];
   BUILD_FILE bf;
   Sequence l;
   String s;

   if (bp->nosuffix) {
      fprintf(f,".SUFFIXES:\n\n");
      return;
    };

   strcpy(buf," ");

   forin (bf,BUILD_FILE,l,bp->defaults) {
      if ((bf->type & BUILD_TYPE_DEFAULT) == 0) continue;
      if (save && bf->library) continue;
      strcpy(lbuf,bf->name);
      s = index(&lbuf[1],'.');
      if (s != NULL) {
	 add_suffix(buf,s);
	 *s = 0;
       };
      add_suffix(buf,lbuf);
    };

   if (buf[1] != 0) {
      fprintf(f,".SUFFIXES: %s\n\n",buf);
    };
};





static void
add_suffix(buf,suf)
   String buf;
   String suf;
{
   Integer i;
   String s;

   i = strlen(suf);

   for (s = buf; *s != 0; ++s) {
      if (s[0] == ' ' && strncmp(&s[1],suf,i) == 0 && s[i+1] == ' ')
	 return;
    };

   strcat(buf,suf);
   strcat(buf," ");
};





/************************************************************************/
/*									*/
/*	gen_macros -- generate macros for make file			*/
/*									*/
/************************************************************************/


static void
gen_macros(bp,f,save)
   BUILD_PROJ bp;
   FILE * f;
   Boolean save;
{
   Sequence l;
   BUILD_MACRO bm;
   String s;

   forin (bm,BUILD_MACRO,l,bp->macros) {
      if (save && bm->library) continue;
      fprintf(f,"%s= ",bm->name);
      if (bm->body != NULL) {
	 for (s = bm->body; *s != 0; ++s) {
	    if (*s == '\n') putc('\\',f);
	    putc(*s,f);
	  };
       };
      putc('\n',f);
      if (bm->option_name != NULL || bm->bit_value != 0 || bm->dflt_type != 0) {
	 fprintf(f,"#MACRO %s %d %d \"%s\"\n",bm->name,bm->dflt_type,
		    bm->bit_value,bm->option_name);
       };
    };
};





/************************************************************************/
/*									*/
/*	gen_project -- output project specific stuff			 */
/*									*/
/************************************************************************/


/* ARGSUSED */

static void
gen_project(bp,f,save)
   BUILD_PROJ bp;
   FILE * f;
   Boolean save;
{
   fprintf(f,"#PROJECT %s %d %d %d\n",bp->system_name,
	      bp->continue_opt,bp->ignore_opt,bp->autodepend);

   if (bp->source_dir != NULL) {
      fprintf(f,"VPATH= %s\n",bp->source_dir);
    };

   fprintf(f,"\n# -----------------------------\n\n");

   if (bp->system_name != NULL) {
      fprintf(f,".BUILD_DEFAULT: %s\n\n",bp->system_name);
    };
};





/************************************************************************/
/*									*/
/*	gen_files -- generate make rules for files			*/
/*									*/
/************************************************************************/


static void
gen_files(bp,f,save)
   BUILD_PROJ bp;
   FILE * f;
   Boolean save;
{
   BUILD_FILE bf,bff;
   Sequence l,la;
   BUILD_DEPEND bd;
   String s;

   forin (bf,BUILD_FILE,l,bp->files) {
      if (save && bf->library) continue;
      if (bf->type & BUILD_TYPE_SPECIAL) continue;
      if (bp->nosuffix && bf->type & BUILD_TYPE_DEFAULT) continue;
      bff = NULL;
      forin (bd,BUILD_DEPEND,la,bf->depends) {
	 if (bd->automatic || bd->macro) continue;
	 if (bd->file == bf->from) bff = bd->file;
	 else fprintf(f,"%s: %s\n",bf->name,bd->file->name);
       };
      if (bf->silent) fprintf(f,"%s: .SILENT",bf->name);
      if (bf->ignore) fprintf(f,"%s: .IGNORE",bf->name);
      if (bf->precious) fprintf(f,"%s: .PRECIOUS",bf->name);
      if (bf->build != NULL) {
	 if (bff == NULL) fprintf(f,"%s:\n\t",bf->name);
	 else fprintf(f,"%s: %s\n\t",bf->name,bff->name);
	 for (s = bf->build; *s != 0; ++s) {
	    if (*s == '\\' && s[1] != 0) {
	       putc(*s++,f);
	       putc(*s,f);
	     }
	    else {
	       putc(*s,f);
	       if (*s == '\n' && s[1] != 0) putc('\t',f);
	     };
	  };
	 fprintf(f,"\n\n");
       };
      fprintf(f,"#FILE %s %d %d %d\n",bf->name,bf->type,bf->match_bits,
		 bf->autodepend);
    };
};





/************************************************************************/
/*									*/
/*	gen_depend -- generate automatic dependency stuff		*/
/*									*/
/************************************************************************/


static void
gen_depend(bp,f,save)
   BUILD_PROJ bp;
   FILE * f;
   Boolean save;
{
   BUILD_FILE bf;
   Sequence l,la;
   BUILD_DEPEND bd;

   fprintf(f,"\n#AUTODEPEND\n\n");

   forin (bf,BUILD_FILE,l,bp->files) {
      if (save && bf->library) continue;
      if (bf->type & BUILD_TYPE_SPECIAL) continue;
      forin (bd,BUILD_DEPEND,la,bf->depends) {
	 if (!bd->automatic || bd->implicit) continue;
	 fprintf(f,"%s: %s\n",bf->name,bd->file->name);
       };
    };
};




/************************************************************************/
/*									*/
/*	add_auto_depends -- add automatic dependencies for file 	*/
/*									*/
/************************************************************************/


static void
add_auto_depends(bp,bf,bd)
   BUILD_PROJ bp;
   BUILD_FILE bf;
   BUILD_DEPEND bd;
{
   BUILD_FILE dep,bff;
   BUILD_DEPEND dbd;
   Sequence l,la;
   FILE * bfp;
   Character buf[1024];
   String s;

   forin (dep,BUILD_FILE,l,bp->files) {
      if (dep->type & BUILD_TYPE_DEPEND) {
	 forin (dbd,BUILD_DEPEND,la,dep->depends) {
	    if (dbd->file == bd->file) break;
	  };
	 if (dbd != NULL) break;
       };
    };

   if (dep == NULL) return;

   bfp = fopen(dep->name,"r");
   if (bfp != NULL) {
      while (fgets(buf,1024,bfp) != NULL) {
	 s = index(buf,'\n');
	 if (s != NULL) *s = 0;
	 s = buf;
	 if (s[0] == '.' && s[1] == '/') s = &s[2];
	 bff = BUILD_data_file(bp,s,FALSE);
	 BUILD_data_depend(bp,bf,bff,TRUE);
       };
      fclose(bfp);
    };
};





/* end of buildfile.c */
